#!/usr/bin/env bash
# SPDX-License-Identifier: GPL-2.0-or-later

# This script is part of the official build environment, see WIKI page for details.
# https://wiki.blender.org/wiki/Building_Blender/Other/CentOS7ReleaseEnvironment

set -e

if [ `id -u` -ne 0 ]; then
   echo "This script must be run as root"
   exit 1
fi

# yum-config-manager does not come in the default minimal install,
# so make sure it is installed and available.
yum -y update
yum -y install yum-utils

# Install all the packages needed for a new toolchain.
#
# NOTE: Keep this separate from the packages install, since otherwise
# older toolchain will be installed.
yum -y update
yum -y install epel-release
yum -y install centos-release-scl
yum -y install devtoolset-9

# Install packages needed for Blender's dependencies.
PACKAGES_FOR_LIBS=(
    # Used to checkout Blender's code.
    git
    # Used to checkout Blender's `../lib/` directory.
    subversion
    # Used to extract packages.
    bzip2
    # Used to extract packages.
    tar
    # Blender and some dependencies use `cmake`.
    cmake3
    # Apply patches from Blender's: `./build_files/build_environment/patches`
    patch
    # Use by `cmake` and `autoconf`.
    make

    # Required by: `external_nasm` which uses an `autoconf` build-system.
    autoconf
    automake
    libtool

    # Meta-build system used by various packages.
    meson
    # Builds generated by meson use Ninja for the actual build.
    ninja-build

    # Required by Blender build option: `WITH_GHOST_X11`.
    libXrandr-devel
    libXinerama-devel
    libXcursor-devel
    libXi-devel
    libX11-devel
    libXt-devel

    # Required by Blender build option: `WITH_GHOST_WAYLAND`.
    mesa-libEGL-devel
    # Required by: Blender & `external_opensubdiv` (probably others).
    mesa-libGL-devel
    mesa-libGLU-devel

    # Required by: `external_ispc`.
    zlib-devel
    # TODO: dependencies build without this, consider removal.
    rubygem-asciidoctor
    # TODO: dependencies build without this, consider removal.
    wget
    # Required by: `external_sqlite` as a build-time dependency (needed for the `tclsh` command).
    tcl
    # Required by: `external_aom`.
    # TODO: Blender is already building `external_nasm` which is listed as an alternative to `yasm`.
    # Why are both needed?
    yasm

    # Required by: `meson` (Python based build system).
    python36
    # Required by: `mako` (Python module used for building `external_mesa`)
    python-setuptools

    # Required by: `external_igc` & `external_osl` as a build-time dependency.
    bison
    # Required by: `external_osl` as a build-time dependency.
    flex
    # TODO: dependencies build without this, consider removal.
    ncurses-devel
)

# Additional packages needed for building Blender.
PACKAGES_FOR_BLENDER=(
    # Required by Blender build option: `WITH_GHOST_WAYLAND`.
    libxkbcommon-devel
)

yum -y install -y ${PACKAGES_FOR_LIBS[@]} ${PACKAGES_FOR_BLENDER[@]}

# Dependencies for Mesa
yum -y install expat-devel
python3 -m pip install mako

# Dependencies for pip (needed for buildbot-worker).
yum -y install python36-pip python36-devel

# Dependencies for asound.
yum -y install -y  \
    alsa-lib-devel pulseaudio-libs-devel

alternatives --install /usr/local/bin/cmake cmake /usr/bin/cmake 10  \
    --slave /usr/local/bin/ctest ctest /usr/bin/ctest  \
    --slave /usr/local/bin/cpack cpack /usr/bin/cpack  \
    --slave /usr/local/bin/ccmake ccmake /usr/bin/ccmake  \
    --family cmake

alternatives --install /usr/local/bin/cmake cmake /usr/bin/cmake3 20  \
    --slave /usr/local/bin/ctest ctest /usr/bin/ctest3  \
    --slave /usr/local/bin/cpack cpack /usr/bin/cpack3  \
    --slave /usr/local/bin/ccmake ccmake /usr/bin/ccmake3  \
    --family cmake

alternatives --install /usr/local/bin/cmake cmake /usr/bin/cmake3 20  \
    --slave /usr/local/bin/ctest ctest /usr/bin/ctest3  \
    --slave /usr/local/bin/cpack cpack /usr/bin/cpack3  \
    --slave /usr/local/bin/ccmake ccmake /usr/bin/ccmake3  \
    --family cmake
