// @HEADER
// *****************************************************************************
//            LOCA: Library of Continuation Algorithms Package
//
// Copyright 2001-2005 NTESS and the LOCA contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u
 * --- + lambda * u - alpha * u**2 + beta * u**3 = 0
 * dx2
 *
 * subject to the boundar conditions u(-1) = u(1) = beta.
 */

#ifndef _PITCHFORK_FINITEELEMENTPROBLEM_H
#define _PITCHFORK_FINITEELEMENTPROBLEM_H

#include "FiniteElementProblem.H" // base class

// Forward Declarations
class Epetra_Comm;
class Epetra_Map;
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Finite Element Problem Class
class Pitchfork_FiniteElementProblem : public FiniteElementProblem {

public:

  // Constructor
  Pitchfork_FiniteElementProblem(int NumGlobalElements, Epetra_Comm& Comm);

  // Destructor
  virtual ~Pitchfork_FiniteElementProblem();

  // Evaluates the function (F) and/or the Jacobian using the solution
  // values in solnVector.
  virtual bool evaluate(FillType f, const Epetra_Vector *solnVector,
            Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix,
            double jac_coeff = 1.0,
            double mass_coeff = 0.0);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the FiniteElementProblem class.
  virtual Epetra_Vector& getSolution();

  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the FiniteElementProblem class.
  virtual Epetra_CrsMatrix& getJacobian();

  // Set a bifurcation parameter in the application physics
  virtual bool setParameter(std::string label, double value);

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

  // Computes the source term
  double source_term(double x);

  // Computes the derivative of the source term
  double source_deriv(double x);

  // Private to prohibit copying
  Pitchfork_FiniteElementProblem(const Pitchfork_FiniteElementProblem&);

  // Private to prohibit copying
  Pitchfork_FiniteElementProblem& operator=(const Pitchfork_FiniteElementProblem&);

private:

  FillType flag;
  Epetra_Map *StandardMap;
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Vector *initialSolution;
  Epetra_Vector *rhs;
  Epetra_CrsGraph *AA;
  Epetra_CrsMatrix *A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

 public:

  double lambda;          // factor used on nonlinear term
  double alpha;           // factor used on nonlinear term
  double beta;            // factor used on nonlinear term

};
#endif




