/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.rdg.resc.ncwms.coords;

import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.geotoolkit.metadata.iso.extent.DefaultGeographicBoundingBox;
import org.opengis.metadata.extent.GeographicBoundingBox;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.ArrayFloat;
import ucar.nc2.constants.AxisType;
import ucar.nc2.dataset.CoordinateAxis;
import ucar.nc2.dataset.CoordinateAxis2D;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dt.GridCoordSystem;
import uk.ac.rdg.resc.ncwms.coords.LonLatPosition;
import uk.ac.rdg.resc.ncwms.coords.LonLatPositionImpl;
import uk.ac.rdg.resc.ncwms.coords.Longitude;

final class CurvilinearGrid
implements Iterable<Cell> {
    private static final Logger logger = LoggerFactory.getLogger(CurvilinearGrid.class);
    private final int ni;
    private final int nj;
    private final float[] longitudes;
    private final float[] latitudes;
    private final ArrayFloat.D2 cornerLons;
    private final ArrayFloat.D2 cornerLats;
    private final GeographicBoundingBox latLonBbox;
    private static final Set<NetcdfDataset.Enhance> SCALE_MISSING = EnumSet.of(NetcdfDataset.Enhance.ScaleMissing);

    public CurvilinearGrid(GridCoordSystem coordSys) {
        CoordinateAxis xAxis = coordSys.getXHorizAxis();
        CoordinateAxis yAxis = coordSys.getYHorizAxis();
        if (xAxis == null || yAxis == null || !(xAxis instanceof CoordinateAxis2D) || !(yAxis instanceof CoordinateAxis2D) || xAxis.getAxisType() != AxisType.Lon || yAxis.getAxisType() != AxisType.Lat) {
            throw new IllegalArgumentException("Coordinate system must consist of two-dimensional latitude and longitude axes");
        }
        CoordinateAxis2D lonAxis = (CoordinateAxis2D)coordSys.getXHorizAxis();
        CoordinateAxis2D latAxis = (CoordinateAxis2D)coordSys.getYHorizAxis();
        if (!Arrays.equals(lonAxis.getShape(), latAxis.getShape())) {
            throw new IllegalArgumentException(String.format("Lon and Lat axes must have the same shape. Lon: %s; Lat: %s", Arrays.toString(lonAxis.getShape()), Arrays.toString(latAxis.getShape())));
        }
        lonAxis.enhance(SCALE_MISSING);
        latAxis.enhance(SCALE_MISSING);
        this.ni = lonAxis.getShape(1);
        this.nj = lonAxis.getShape(0);
        this.longitudes = new float[this.ni * this.nj];
        this.latitudes = new float[this.ni * this.nj];
        double minLon = 180.0;
        double maxLon = -180.0;
        double minLat = 90.0;
        double maxLat = -90.0;
        int index = 0;
        for (int j = 0; j < this.nj; ++j) {
            for (int i = 0; i < this.ni; ++i) {
                boolean isNaN;
                double lon = lonAxis.getCoordValue(j, i);
                double lat = latAxis.getCoordValue(j, i);
                boolean bl = isNaN = Double.isNaN(lon) || Double.isNaN(lat);
                if (!isNaN) {
                    lon = Longitude.constrain180(lon);
                    minLon = Math.min(minLon, lon);
                    maxLon = Math.max(maxLon, lon);
                    minLat = Math.min(minLat, lat);
                    maxLat = Math.max(maxLat, lat);
                }
                this.longitudes[index] = isNaN ? Float.NaN : (float)lon;
                this.latitudes[index] = isNaN ? Float.NaN : (float)lat;
                ++index;
            }
        }
        if (maxLon < minLon || maxLat < minLat) {
            throw new IllegalStateException("Invalid bounding box");
        }
        this.latLonBbox = new DefaultGeographicBoundingBox(minLon, maxLon, minLat, maxLat);
        logger.debug("Bounding box = {},{},{},{}", new Object[]{this.latLonBbox.getWestBoundLongitude(), this.latLonBbox.getSouthBoundLatitude(), this.latLonBbox.getEastBoundLongitude(), this.latLonBbox.getNorthBoundLatitude()});
        logger.debug("Making longitude corners");
        this.cornerLons = this.makeCorners(this.longitudes, true);
        logger.debug("Making latitude corners");
        this.cornerLats = this.makeCorners(this.latitudes, false);
        logger.debug("Made curvilinear grid");
    }

    public LonLatPosition getMidpoint(int i, int j) {
        int index = this.getIndex(i, j);
        return new LonLatPositionImpl(this.longitudes[index], this.latitudes[index]);
    }

    private int getIndex(int i, int j) {
        return j * this.ni + i;
    }

    private List<LonLatPosition> getCorners(int i, int j) {
        ArrayList<LonLatPosition> corners = new ArrayList<LonLatPosition>(4);
        corners.add(this.getCorner(i, j));
        corners.add(this.getCorner(i + 1, j));
        corners.add(this.getCorner(i + 1, j + 1));
        corners.add(this.getCorner(i, j + 1));
        return corners;
    }

    private LonLatPosition getCorner(int cornerI, int cornerJ) {
        return new LonLatPositionImpl(this.cornerLons.get(cornerJ, cornerI), this.cornerLats.get(cornerJ, cornerI));
    }

    public Cell getCell(int i, int j) {
        if (i < 0 || j < 0 || i >= this.ni || j >= this.nj) {
            throw new IllegalArgumentException(i + "," + j + " is not a valid cell in this grid");
        }
        return new Cell(i, j);
    }

    private ArrayFloat.D2 makeCorners(float[] midpoints, boolean isLongitude) {
        ArrayFloat.D2 edges = new ArrayFloat.D2(this.nj + 1, this.ni + 1);
        for (int j = 0; j < this.nj - 1; ++j) {
            for (int i = 0; i < this.ni - 1; ++i) {
                double midpoint1 = midpoints[this.getIndex(i, j)];
                double midpoint2 = midpoints[this.getIndex(i + 1, j)];
                double midpoint3 = midpoints[this.getIndex(i, j + 1)];
                double midpoint4 = midpoints[this.getIndex(i + 1, j + 1)];
                if (isLongitude) {
                    midpoint2 = CurvilinearGrid.harmonizeLongitudes(midpoint1, midpoint2);
                    midpoint3 = CurvilinearGrid.harmonizeLongitudes(midpoint1, midpoint3);
                    midpoint4 = CurvilinearGrid.harmonizeLongitudes(midpoint1, midpoint4);
                }
                double xval = (midpoint1 + midpoint2 + midpoint3 + midpoint4) / 4.0;
                edges.set(j + 1, i + 1, (float)xval);
            }
            edges.set(j + 1, 0, edges.get(j + 1, 1) - (edges.get(j + 1, 2) - edges.get(j + 1, 1)));
            edges.set(j + 1, this.ni, edges.get(j + 1, this.ni - 1) + (edges.get(j + 1, this.ni - 1) - edges.get(j + 1, this.ni - 2)));
        }
        for (int x = 0; x < this.ni + 1; ++x) {
            edges.set(0, x, edges.get(1, x) - (edges.get(2, x) - edges.get(1, x)));
            edges.set(this.nj, x, edges.get(this.nj - 1, x) + (edges.get(this.nj - 1, x) - edges.get(this.nj - 2, x)));
        }
        return edges;
    }

    private static double harmonizeLongitudes(double ref, double test) {
        if (ref < -180.0 || ref > 180.0) {
            throw new IllegalArgumentException("Reference longitude must be in the range [-180,180]");
        }
        double lon1 = Longitude.constrain180(test);
        double lon2 = ref < 0.0 ? lon1 - 360.0 : lon1 + 360.0;
        double d1 = Math.abs(ref - lon1);
        double d2 = Math.abs(ref - lon2);
        return d1 < d2 ? lon1 : lon2;
    }

    public int getNi() {
        return this.ni;
    }

    public int getNj() {
        return this.nj;
    }

    public int size() {
        return this.longitudes.length;
    }

    public GeographicBoundingBox getBoundingBox() {
        return this.latLonBbox;
    }

    public int hashCode() {
        int hashCode = 17;
        hashCode = 31 * hashCode + this.ni;
        hashCode = 31 * hashCode + this.nj;
        hashCode = 31 * hashCode + Arrays.hashCode(this.longitudes);
        hashCode = 31 * hashCode + Arrays.hashCode(this.latitudes);
        return hashCode;
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof CurvilinearGrid)) {
            return false;
        }
        CurvilinearGrid other = (CurvilinearGrid)obj;
        return this.ni == other.ni && this.nj == other.nj && Arrays.equals(this.longitudes, other.longitudes) && Arrays.equals(this.latitudes, other.latitudes);
    }

    @Override
    public Iterator<Cell> iterator() {
        return new CellIterator();
    }

    private static double getArea(Point2D p1, Point2D p2, Point2D p3, Point2D p4) {
        double a2 = p1.distanceSq(p2);
        double b2 = p2.distanceSq(p3);
        double c2 = p3.distanceSq(p4);
        double d2 = p4.distanceSq(p1);
        double f2 = p1.distanceSq(p3);
        double g2 = p2.distanceSq(p4);
        double term = b2 + d2 - a2 - c2;
        return Math.sqrt(4.0 * f2 * g2 - term * term) / 4.0;
    }

    public double getMeanCellArea() {
        double sumArea = 0.0;
        int nans = 0;
        for (Cell cell : this) {
            double cellArea = cell.getArea();
            if (Double.isNaN(cellArea)) {
                ++nans;
                continue;
            }
            sumArea += cellArea;
        }
        logger.debug("{} cells out of {} had area = NaN", (Object)nans, (Object)this.size());
        return sumArea / (double)(this.size() - nans);
    }

    public class Cell {
        private final int i;
        private final int j;

        private Cell(int i, int j) {
            this.i = i;
            this.j = j;
        }

        public int getI() {
            return this.i;
        }

        public int getJ() {
            return this.j;
        }

        public LonLatPosition getCentre() {
            return CurvilinearGrid.this.getMidpoint(this.i, this.j);
        }

        public List<Point2D> getCorners() {
            List corners = CurvilinearGrid.this.getCorners(this.i, this.j);
            ArrayList<Point2D> cornerPoints = new ArrayList<Point2D>(corners.size());
            for (LonLatPosition corner : corners) {
                Point2D.Double cornerPoint = new Point2D.Double(this.harmonizeWithCentre(corner.getLongitude()), corner.getLatitude());
                cornerPoints.add(cornerPoint);
            }
            return cornerPoints;
        }

        public List<Cell> getEdgeNeighbours() {
            ArrayList<Cell> neighbours = new ArrayList<Cell>(4);
            if (this.i > 0) {
                neighbours.add(new Cell(this.i - 1, this.j));
            }
            if (this.j > 0) {
                neighbours.add(new Cell(this.i, this.j - 1));
            }
            if (this.i < CurvilinearGrid.this.ni - 1) {
                neighbours.add(new Cell(this.i + 1, this.j));
            }
            if (this.j < CurvilinearGrid.this.nj - 1) {
                neighbours.add(new Cell(this.i, this.j + 1));
            }
            return neighbours;
        }

        public List<Cell> getCornerNeighbours() {
            ArrayList<Cell> neighbours = new ArrayList<Cell>(4);
            if (this.i > 0 && this.j > 0) {
                neighbours.add(new Cell(this.i - 1, this.j - 1));
            }
            if (this.i < CurvilinearGrid.this.ni - 1 && this.j > 0) {
                neighbours.add(new Cell(this.i + 1, this.j - 1));
            }
            if (this.i < CurvilinearGrid.this.ni - 1 && this.j < CurvilinearGrid.this.nj - 1) {
                neighbours.add(new Cell(this.i + 1, this.j + 1));
            }
            if (this.i > 0 && this.j < CurvilinearGrid.this.nj - 1) {
                neighbours.add(new Cell(this.i - 1, this.j + 1));
            }
            return neighbours;
        }

        public double getArea() {
            List<Point2D> corners = this.getCorners();
            return CurvilinearGrid.getArea(corners.get(0), corners.get(1), corners.get(2), corners.get(3));
        }

        public Path2D getBoundaryPath() {
            Path2D.Double path = new Path2D.Double();
            boolean firstTime = true;
            for (Point2D point : this.getCorners()) {
                if (firstTime) {
                    ((Path2D)path).moveTo(point.getX(), point.getY());
                } else {
                    ((Path2D)path).lineTo(point.getX(), point.getY());
                }
                firstTime = false;
            }
            path.closePath();
            return path;
        }

        public boolean contains(LonLatPosition latLonPoint) {
            Path2D path = this.getBoundaryPath();
            double lon = this.harmonizeWithCentre(latLonPoint.getLongitude());
            return path.contains(lon, latLonPoint.getLatitude());
        }

        private double harmonizeWithCentre(double lon) {
            return CurvilinearGrid.harmonizeLongitudes(this.getCentre().getLongitude(), lon);
        }

        public int hashCode() {
            int hashCode = 17;
            hashCode = 31 * hashCode + this.i;
            hashCode = 31 * hashCode + this.j;
            return hashCode;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Cell)) {
                return false;
            }
            Cell other = (Cell)obj;
            return this.i == other.i && this.j == other.j;
        }

        public String toString() {
            LonLatPosition centre = this.getCentre();
            List<Point2D> corners = this.getCorners();
            return String.format("[%d,%d]: [%f,%f] %s", this.i, this.j, centre.getLongitude(), centre.getLatitude(), corners);
        }
    }

    private final class CellIterator
    implements Iterator<Cell> {
        private int index = 0;

        private CellIterator() {
        }

        @Override
        public boolean hasNext() {
            return this.index < CurvilinearGrid.this.size();
        }

        @Override
        public Cell next() {
            int i = this.index % CurvilinearGrid.this.ni;
            int j = this.index / CurvilinearGrid.this.ni;
            ++this.index;
            return new Cell(i, j);
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Not supported.");
        }
    }
}

