/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.rdg.resc.ncwms.coords;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DirectColorModel;
import java.awt.image.WritableRaster;
import org.opengis.metadata.extent.GeographicBoundingBox;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.rdg.resc.ncwms.coords.CurvilinearGrid;

final class LookUpTable {
    private static final Logger logger = LoggerFactory.getLogger(LookUpTable.class);
    private DataBuffer iIndices;
    private DataBuffer jIndices;
    private final int nLon;
    private final int nLat;
    private final AffineTransform transform = new AffineTransform();
    private static final ColorModel COLOR_MODEL = new DirectColorModel(16, 0, 65280, 255, 0);
    private static final int MISSING_VALUE = 65535;
    private static final int MAX_INDEX = 65534;

    public LookUpTable(CurvilinearGrid curvGrid, double minResolution) {
        GeographicBoundingBox bbox = curvGrid.getBoundingBox();
        double lonDiff = bbox.getEastBoundLongitude() - bbox.getWestBoundLongitude();
        double latDiff = bbox.getNorthBoundLatitude() - bbox.getSouthBoundLatitude();
        this.nLon = (int)Math.ceil(lonDiff / minResolution);
        this.nLat = (int)Math.ceil(latDiff / minResolution);
        if (this.nLon <= 0 || this.nLat <= 0) {
            String msg = String.format("nLon (=%d) and nLat (=%d) must be positive and > 0", this.nLon, this.nLat);
            throw new IllegalStateException(msg);
        }
        double lonStride = lonDiff / (double)(this.nLon - 1);
        double latStride = latDiff / (double)(this.nLat - 1);
        this.transform.scale(1.0 / lonStride, 1.0 / latStride);
        this.transform.translate(-bbox.getWestBoundLongitude(), -bbox.getSouthBoundLatitude());
        this.makeLuts(curvGrid);
    }

    private void makeLuts(CurvilinearGrid curvGrid) {
        BufferedImage iIm = this.createBufferedImage();
        BufferedImage jIm = this.createBufferedImage();
        Graphics2D ig2d = iIm.createGraphics();
        Graphics2D jg2d = jIm.createGraphics();
        ig2d.setTransform(this.transform);
        jg2d.setTransform(this.transform);
        for (CurvilinearGrid.Cell cell : curvGrid) {
            Path2D path = cell.getBoundaryPath();
            if (cell.getI() > 65534 || cell.getJ() > 65534) {
                throw new IllegalStateException("Can't store indices greater than 65534");
            }
            ig2d.setPaint(new Color(cell.getI()));
            jg2d.setPaint(new Color(cell.getJ()));
            ig2d.fill(path);
            jg2d.fill(path);
            double shiftLon = cell.getCentre().getLongitude() > 0.0 ? -360.0 : 360.0;
            path.transform(AffineTransform.getTranslateInstance(shiftLon, 0.0));
            ig2d.fill(path);
            jg2d.fill(path);
        }
        this.iIndices = iIm.getRaster().getDataBuffer();
        this.jIndices = jIm.getRaster().getDataBuffer();
    }

    private BufferedImage createBufferedImage() {
        WritableRaster raster = COLOR_MODEL.createCompatibleWritableRaster(this.nLon, this.nLat);
        BufferedImage im = new BufferedImage(COLOR_MODEL, raster, true, null);
        for (int y = 0; y < im.getHeight(); ++y) {
            for (int x = 0; x < im.getWidth(); ++x) {
                im.setRGB(x, y, 65535);
            }
        }
        logger.debug("Created BufferedImage of size {},{}, data buffer type {}", new Object[]{im.getWidth(), im.getHeight(), im.getRaster().getDataBuffer().getClass()});
        return im;
    }

    public int[] getGridCoordinates(double longitude, double latitude) {
        Point2D indexPoint = this.transform.transform(new Point2D.Double(longitude, latitude), null);
        int iLon = (int)Math.round(indexPoint.getX());
        int iLat = (int)Math.round(indexPoint.getY());
        if (iLon < 0 || iLat < 0 || iLon >= this.nLon || iLat >= this.nLat) {
            return null;
        }
        int index = iLon + iLat * this.nLon;
        int iIndex = this.iIndices.getElem(index);
        int jIndex = this.jIndices.getElem(index);
        if (iIndex < 0 || iIndex > 65534 || jIndex < 0 || jIndex > 65534) {
            return null;
        }
        return new int[]{iIndex, jIndex};
    }

    public int getNumLonPoints() {
        return this.nLon;
    }

    public int getNumLatPoints() {
        return this.nLat;
    }

    public AffineTransform getTransform() {
        return this.transform;
    }
}

