/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.rdg.resc.ncwms.coords;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.opengis.referencing.operation.TransformException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.rdg.resc.ncwms.coords.CrsHelper;
import uk.ac.rdg.resc.ncwms.coords.HorizontalCoordSys;
import uk.ac.rdg.resc.ncwms.coords.HorizontalGrid;
import uk.ac.rdg.resc.ncwms.coords.HorizontalPosition;
import uk.ac.rdg.resc.ncwms.coords.LatLonCoordSys;
import uk.ac.rdg.resc.ncwms.coords.LonLatPosition;
import uk.ac.rdg.resc.ncwms.coords.PointList;

public final class PixelMap
implements Iterable<PixelMapEntry> {
    private static final Logger logger = LoggerFactory.getLogger(PixelMap.class);
    private long[] pixelMapEntries;
    private int numEntries = 0;
    private final int chunkSize;
    private final int sourceGridISize;
    private int minIIndex = Integer.MAX_VALUE;
    private int minJIndex = Integer.MAX_VALUE;
    private int maxIIndex = -1;
    private int maxJIndex = -1;

    public PixelMap(HorizontalCoordSys horizCoordSys, PointList pointList) throws TransformException {
        long start = System.currentTimeMillis();
        this.sourceGridISize = horizCoordSys.getXAxisSize();
        this.chunkSize = pointList.size() < 1000 ? pointList.size() : pointList.size() / 10;
        this.pixelMapEntries = new long[this.chunkSize];
        if (pointList instanceof HorizontalGrid) {
            this.initFromGrid(horizCoordSys, (HorizontalGrid)pointList);
        } else {
            this.initFromPointList(horizCoordSys, pointList);
        }
        logger.debug("Built pixel map in {} ms", (Object)(System.currentTimeMillis() - start));
    }

    private void initFromPointList(HorizontalCoordSys horizCoordSys, PointList pointList) throws TransformException {
        logger.debug("Using generic method based on iterating over the PointList");
        CrsHelper crsHelper = pointList.getCrsHelper();
        int pixelIndex = 0;
        for (HorizontalPosition point : pointList.asList()) {
            LonLatPosition lonLat;
            int[] gridCoords;
            if (crsHelper.isPointValidForCrs(point) && (gridCoords = horizCoordSys.lonLatToGrid(lonLat = crsHelper.crsToLonLat(point))) != null) {
                this.put(gridCoords[0], gridCoords[1], pixelIndex);
            }
            ++pixelIndex;
        }
    }

    private void initFromGrid(HorizontalCoordSys horizCoordSys, HorizontalGrid grid) throws TransformException {
        if (grid.isLatLon() && horizCoordSys instanceof LatLonCoordSys) {
            logger.debug("Using optimized method for lat-lon coordinates with 1D axes");
            LatLonCoordSys latLonGrid = (LatLonCoordSys)horizCoordSys;
            int pixelIndex = 0;
            int[] xIndices = new int[grid.getXAxisValues().length];
            for (int i = 0; i < grid.getXAxisValues().length; ++i) {
                xIndices[i] = latLonGrid.getLonIndex(grid.getXAxisValues()[i]);
            }
            for (double lat : grid.getYAxisValues()) {
                if (lat >= -90.0 && lat <= 90.0) {
                    int yIndex = latLonGrid.getLatIndex(lat);
                    for (int xIndex : xIndices) {
                        this.put(xIndex, yIndex, pixelIndex);
                        ++pixelIndex;
                    }
                    continue;
                }
                pixelIndex += xIndices.length;
            }
        } else {
            this.initFromPointList(horizCoordSys, grid);
        }
    }

    private void put(int i, int j, int targetGridIndex) {
        if (i < 0 || j < 0) {
            return;
        }
        if (i < this.minIIndex) {
            this.minIIndex = i;
        }
        if (i > this.maxIIndex) {
            this.maxIIndex = i;
        }
        if (j < this.minJIndex) {
            this.minJIndex = j;
        }
        if (j > this.maxJIndex) {
            this.maxJIndex = j;
        }
        int sourceGridIndex = j * this.sourceGridISize + i;
        if (this.numEntries >= this.pixelMapEntries.length) {
            long[] newArray = new long[this.pixelMapEntries.length + this.chunkSize];
            System.arraycopy(this.pixelMapEntries, 0, newArray, 0, this.pixelMapEntries.length);
            this.pixelMapEntries = newArray;
        }
        this.pixelMapEntries[this.numEntries] = (long)sourceGridIndex << 32 | (long)targetGridIndex;
        ++this.numEntries;
    }

    public boolean isEmpty() {
        return this.numEntries == 0;
    }

    public int getMinIIndex() {
        return this.minIIndex;
    }

    public int getMinJIndex() {
        return this.minJIndex;
    }

    public int getMaxIIndex() {
        return this.maxIIndex;
    }

    public int getMaxJIndex() {
        return this.maxJIndex;
    }

    public int getNumUniqueIJPairs() {
        int count = 0;
        for (PixelMapEntry pme : this) {
            ++count;
        }
        return count;
    }

    public int getBoundingBoxSize() {
        return (this.maxIIndex - this.minIIndex + 1) * (this.maxJIndex - this.minJIndex + 1);
    }

    @Override
    public Iterator<PixelMapEntry> iterator() {
        return new Iterator<PixelMapEntry>(){
            private int index = 0;

            @Override
            public boolean hasNext() {
                return this.index < PixelMap.this.numEntries;
            }

            @Override
            public PixelMapEntry next() {
                long packed = PixelMap.this.pixelMapEntries[this.index];
                int[] unpacked = PixelMap.unpack(packed);
                ++this.index;
                final int sourceGridIndex = unpacked[0];
                final ArrayList<Integer> targetGridIndices = new ArrayList<Integer>();
                targetGridIndices.add(unpacked[1]);
                boolean done = false;
                while (!done && this.hasNext()) {
                    packed = PixelMap.this.pixelMapEntries[this.index];
                    unpacked = PixelMap.unpack(packed);
                    if (unpacked[0] == sourceGridIndex) {
                        targetGridIndices.add(unpacked[1]);
                        ++this.index;
                        continue;
                    }
                    done = true;
                }
                return new PixelMapEntry(){

                    @Override
                    public int getSourceGridIIndex() {
                        return sourceGridIndex % PixelMap.this.sourceGridISize;
                    }

                    @Override
                    public int getSourceGridJIndex() {
                        return sourceGridIndex / PixelMap.this.sourceGridISize;
                    }

                    @Override
                    public List<Integer> getTargetGridPoints() {
                        return targetGridIndices;
                    }
                };
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException("Not supported yet.");
            }
        };
    }

    private static int[] unpack(long packed) {
        return new int[]{(int)(packed >> 32), (int)(packed & 0xFFFFFFFFFFFFFFFFL)};
    }

    public static interface PixelMapEntry {
        public int getSourceGridIIndex();

        public int getSourceGridJIndex();

        public List<Integer> getTargetGridPoints();
    }
}

