/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.rdg.resc.ncwms.graphics;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DataBufferByte;
import java.awt.image.IndexColorModel;
import java.awt.image.Raster;
import java.awt.image.SampleModel;
import java.awt.image.WritableRaster;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.rdg.resc.ncwms.exceptions.StyleNotDefinedException;
import uk.ac.rdg.resc.ncwms.graphics.ColorPalette;
import uk.ac.rdg.resc.ncwms.util.Range;
import uk.ac.rdg.resc.ncwms.util.Ranges;
import uk.ac.rdg.resc.ncwms.util.WmsUtils;
import uk.ac.rdg.resc.ncwms.wms.Layer;
import uk.ac.rdg.resc.ncwms.wms.VectorLayer;

public final class ImageProducer {
    private static final Logger logger = LoggerFactory.getLogger(ImageProducer.class);
    private Layer layer;
    private Style style;
    private int picWidth;
    private int picHeight;
    private boolean transparent;
    private int opacity;
    private int numColourBands;
    private boolean logarithmic;
    private Color bgColor;
    private ColorPalette colorPalette;
    private Range<Float> scaleRange;
    private float arrowLength = 10.0f;
    private List<BufferedImage> renderedFrames = new ArrayList<BufferedImage>();
    private List<List<List<Float>>> frameData;
    private List<String> labels;

    private ImageProducer() {
    }

    public BufferedImage getLegend() {
        return this.colorPalette.createLegend(this.numColourBands, this.layer, this.logarithmic, this.scaleRange);
    }

    public int getPicWidth() {
        return this.picWidth;
    }

    public int getPicHeight() {
        return this.picHeight;
    }

    public boolean isTransparent() {
        return this.transparent;
    }

    public void addFrame(List<List<Float>> data, String label) {
        logger.debug("Adding frame with label {}", (Object)label);
        if (this.scaleRange.isEmpty()) {
            logger.debug("Auto-scaling, so caching frame");
            if (this.frameData == null) {
                this.frameData = new ArrayList<List<List<Float>>>();
                this.labels = new ArrayList<String>();
            }
            this.frameData.add(data);
            this.labels.add(label);
        } else {
            logger.debug("Scale is set, so rendering image");
            this.createImage(data, label);
        }
    }

    private void createImage(List<List<Float>> data, String label) {
        byte[] pixels = new byte[this.picWidth * this.picHeight];
        List<Float> magnitudes = data.size() == 1 ? data.get(0) : WmsUtils.getMagnitudes(data.get(0), data.get(1));
        for (int i = 0; i < pixels.length; ++i) {
            pixels[i] = (byte)this.getColourIndex(magnitudes.get(i));
        }
        IndexColorModel colorModel = this.colorPalette.getColorModel(this.numColourBands, this.opacity, this.bgColor, this.transparent);
        DataBufferByte buf = new DataBufferByte(pixels, pixels.length);
        SampleModel sampleModel = ((ColorModel)colorModel).createCompatibleSampleModel(this.picWidth, this.picHeight);
        WritableRaster raster = Raster.createWritableRaster(sampleModel, buf, null);
        BufferedImage image = new BufferedImage(colorModel, raster, false, null);
        if (label != null && !label.equals("")) {
            Graphics2D gfx = (Graphics2D)image.getGraphics();
            gfx.setPaint(new Color(0, 0, 143));
            gfx.fillRect(1, image.getHeight() - 19, image.getWidth() - 1, 18);
            gfx.setPaint(new Color(255, 151, 0));
            gfx.drawString(label, 10, image.getHeight() - 5);
        }
        if (this.style == Style.VECTOR) {
            Graphics2D g = image.createGraphics();
            g.setColor(Color.BLACK);
            logger.debug("Drawing vectors, length = {} pixels", (Object)Float.valueOf(this.arrowLength));
            List<Float> east = data.get(0);
            List<Float> north = data.get(1);
            int i = 0;
            while (i < this.picWidth) {
                int j = 0;
                while (j < this.picHeight) {
                    int dataIndex = j * this.picWidth + i;
                    Float eastVal = east.get(dataIndex);
                    Float northVal = north.get(dataIndex);
                    if (eastVal != null && northVal != null) {
                        double angle = Math.atan2(northVal.doubleValue(), eastVal.doubleValue());
                        double iEnd = (double)i + (double)this.arrowLength * Math.cos(angle);
                        double jEnd = (double)j - (double)this.arrowLength * Math.sin(angle);
                        g.fillOval(i - 2, j - 2, 4, 4);
                        g.setStroke(new BasicStroke(1.0f));
                        g.drawLine(i, j, (int)Math.round(iEnd), (int)Math.round(jEnd));
                    }
                    j = (int)((double)j + Math.ceil((double)this.arrowLength * 1.2));
                }
                i = (int)((double)i + Math.ceil((double)this.arrowLength * 1.2));
            }
        }
        this.renderedFrames.add(image);
    }

    private int getColourIndex(Float value) {
        double max;
        double min;
        if (value == null) {
            return this.numColourBands;
        }
        if (!this.scaleRange.contains(value)) {
            return this.numColourBands + 1;
        }
        float scaleMin = this.scaleRange.getMinimum().floatValue();
        float scaleMax = this.scaleRange.getMaximum().floatValue();
        double val = this.logarithmic ? Math.log(value.floatValue()) : (double)value.floatValue();
        double frac = (val - (min = this.logarithmic ? Math.log(scaleMin) : (double)scaleMin)) / ((max = this.logarithmic ? Math.log(scaleMax) : (double)scaleMax) - min);
        int index = (int)(frac * (double)this.numColourBands);
        if (index == this.numColourBands) {
            --index;
        }
        return index;
    }

    public List<BufferedImage> getRenderedFrames() {
        this.setScale();
        if (this.frameData != null) {
            logger.debug("Rendering image frames...");
            for (int i = 0; i < this.frameData.size(); ++i) {
                logger.debug("    ... rendering frame {}", (Object)i);
                this.createImage(this.frameData.get(i), this.labels.get(i));
            }
        }
        return this.renderedFrames;
    }

    private void setScale() {
        if (this.scaleRange.isEmpty()) {
            Float scaleMin = null;
            Float scaleMax = null;
            logger.debug("Setting the scale automatically");
            for (List<List<Float>> data : this.frameData) {
                Range range = Ranges.findMinMax((Collection)data.get(0));
                if (range.isEmpty()) continue;
                if (scaleMin == null || ((Float)range.getMinimum()).compareTo(scaleMin) < 0) {
                    scaleMin = (Float)range.getMinimum();
                }
                if (scaleMax != null && ((Float)range.getMaximum()).compareTo(scaleMax) <= 0) continue;
                scaleMax = (Float)range.getMaximum();
            }
            this.scaleRange = Ranges.newRange(scaleMin, scaleMax);
        }
    }

    public int getOpacity() {
        return this.opacity;
    }

    public static final class Builder {
        private Layer layer = null;
        private int picWidth = -1;
        private int picHeight = -1;
        private boolean transparent = false;
        private int opacity = 100;
        private int numColourBands = 254;
        private Boolean logarithmic = null;
        private Color bgColor = Color.WHITE;
        private Range<Float> scaleRange = null;
        private Style style = null;
        private ColorPalette colorPalette = null;

        public Builder layer(Layer layer) {
            if (layer == null) {
                throw new NullPointerException();
            }
            this.layer = layer;
            return this;
        }

        public Builder style(String styleSpec) throws StyleNotDefinedException {
            if (styleSpec == null) {
                return this;
            }
            String[] styleStrEls = styleSpec.split("/");
            String styleType = styleStrEls[0];
            if (styleType.equalsIgnoreCase("boxfill")) {
                this.style = Style.BOXFILL;
            } else if (styleType.equalsIgnoreCase("vector")) {
                this.style = Style.VECTOR;
            } else {
                throw new StyleNotDefinedException("The style " + styleSpec + " is not supported by this server");
            }
            String paletteName = null;
            if (styleStrEls.length > 1) {
                paletteName = styleStrEls[1];
            }
            this.colorPalette = ColorPalette.get(paletteName);
            if (this.colorPalette == null) {
                throw new StyleNotDefinedException("There is no palette with the name " + paletteName);
            }
            return this;
        }

        public Builder width(int width) {
            if (width < 0) {
                throw new IllegalArgumentException();
            }
            this.picWidth = width;
            return this;
        }

        public Builder height(int height) {
            if (height < 0) {
                throw new IllegalArgumentException();
            }
            this.picHeight = height;
            return this;
        }

        public Builder transparent(boolean transparent) {
            this.transparent = transparent;
            return this;
        }

        public Builder opacity(int opacity) {
            if (opacity < 0 || opacity > 100) {
                throw new IllegalArgumentException();
            }
            this.opacity = opacity;
            return this;
        }

        public Builder colourScaleRange(Range<Float> scaleRange) {
            this.scaleRange = scaleRange;
            return this;
        }

        public Builder numColourBands(int numColourBands) {
            if (numColourBands < 0 || numColourBands > 254) {
                throw new IllegalArgumentException();
            }
            this.numColourBands = numColourBands;
            return this;
        }

        public Builder logarithmic(Boolean logarithmic) {
            this.logarithmic = logarithmic;
            return this;
        }

        public Builder backgroundColour(Color bgColor) {
            if (bgColor != null) {
                this.bgColor = bgColor;
            }
            return this;
        }

        public ImageProducer build() throws StyleNotDefinedException {
            if (this.layer == null) {
                throw new IllegalStateException("Must set the Layer object");
            }
            if (this.picWidth < 0 || this.picHeight < 0) {
                throw new IllegalStateException("picture width and height must be >= 0");
            }
            if (this.style == Style.VECTOR && !(this.layer instanceof VectorLayer)) {
                throw new StyleNotDefinedException("The style " + (Object)((Object)this.style) + " is not supported by this layer");
            }
            ImageProducer ip = new ImageProducer();
            ip.layer = this.layer;
            ip.picWidth = this.picWidth;
            ip.picHeight = this.picHeight;
            ip.opacity = this.opacity;
            ip.transparent = this.transparent;
            ip.bgColor = this.bgColor;
            ip.numColourBands = this.numColourBands;
            ip.style = this.style == null ? (this.layer instanceof VectorLayer ? Style.VECTOR : Style.BOXFILL) : this.style;
            ip.colorPalette = this.colorPalette == null ? this.layer.getDefaultColorPalette() : this.colorPalette;
            ip.logarithmic = this.logarithmic == null ? this.layer.isLogScaling() : this.logarithmic.booleanValue();
            ip.scaleRange = this.scaleRange == null ? this.layer.getApproxValueRange() : this.scaleRange;
            return ip;
        }
    }

    private static enum Style {
        BOXFILL,
        VECTOR;

    }
}

