#ifdef Fortran2003
Module netcdf_nc_interfaces

! Fortran interfaces to netCDF C functions using FORTRAN 2003 C 
! Interoperability features. These interfaces are for the base
! netCDF C routines in the libsrc directory

! Written by: Richard Weed
!             Engineering Research Center
!             Mississippi State University
!             rweed@erc.msstate.edu

! Version 1.:  Sept. 2005 - Initial Cray X1 version
! Version 2.:  May 2006   - Updated to support g95
          
 USE ISO_C_BINDING  ! FORTRAN interface functions will inherit association

 Implicit NONE

! Define processor/compiler dependent parameters for ptrdiff_t, signed char,
! and short types. Note prtdiff_t is not defined in the FORTRAN 2003
! standard as an interoperable type in ISO_C_BINDING so we have to make our
! own.

!
! Set C_PTRDIFF_T to the ISO_C_BINDING parameter C_INTPTR_T
! (and hope they map to the same value in C)

 Integer, Parameter :: C_PTRDIFF_T = C_INTPTR_T

! Set KIND parameters for 1 and 2 byte integers if the compiler
! supports them. Set the KINDs to default integers otherwise.

#ifdef HAVE_INT1
 Integer, Parameter :: CINT1 = C_SIGNED_CHAR
 Integer(1), PRIVATE :: i1=1
 Integer, Parameter :: KINT1 = KIND(i1) 
#else
 Integer(KIND=C_INT), Parameter :: CINT1 = C_INT
 Integer, Parameter :: KINT1 = KIND(1)
#endif

#ifdef HAVE_INT2
 Integer, Parameter :: CINT2 = C_SHORT
 Integer(2), PRIVATE :: i2=1
 Integer, Parameter :: KINT2 = KIND(i2)
#else
 Integer, Parameter :: CINT2 = C_INT
 Integer, Parameter :: KINT2 = KIND(1)
#endif

!--------- Define default C interface parameters from netcdf.h ---------------

! Define data type enumerator - And yes, F2K3 supports enumerators
! Note CRAY X1 has a different (and non-standard) syntax

#ifdef CRAY_X1
 Enum, BIND(C) :: nc_type
#else
 Enum, BIND(C)
#endif
  ENUMERATOR :: NC_NAT    = 0, &
                NC_BYTE   = 1, &
                NC_CHAR   = 2, &
                NC_SHORT  = 3, &
                NC_INT    = 4, &
                NC_FLOAT  = 5, &
                NC_DOUBLE = 6
 End Enum


! Default fill values

 Integer(KIND=C_INT), Parameter :: NC_FILL_BYTE    = -127
 Integer(KIND=C_INT), Parameter :: NC_FILL_CHAR    = 0
 Integer(KIND=C_INT), Parameter :: NC_FILL_SHORT   = -32767
 Integer(KIND=C_INT), Parameter :: NC_FILL_INT     = -2147483647
 Real(KIND= C_FLOAT), Parameter :: NC_FILL_FLOAT   = 9.9692099683868690E+36
 Real(KIND=C_DOUBLE), Parameter :: NC_FILL_DOUBLE  = 9.9692099683868690E+36

! Mode flags for opening and creating datasets

 Integer(KIND=C_INT), Parameter :: NC_NOWRITE          = 0
 Integer(KIND=C_INT), Parameter :: NC_WRITE            = 1
 Integer(KIND=C_INT), Parameter :: NC_CLOBBER          = 0
 Integer(KIND=C_INT), Parameter :: NC_NOCLOBBER        = 4
 Integer(KIND=C_INT), Parameter :: NC_FILL             = 0
 Integer(KIND=C_INT), Parameter :: NC_NOFILL           = 256
 Integer(KIND=C_INT), Parameter :: NC_LOCK             = 1024
 Integer(KIND=C_INT), Parameter :: NC_SHARE            = 2048
 Integer(KIND=C_INT), Parameter :: NC_64BIT_OFFSET     = 512
 Integer(KIND=C_INT), Parameter :: NC_SIZEHINT_DEFAULT = 0
 Integer(KIND=C_INT), Parameter :: NC_ALIGN_CHUNK      = -1
 Integer(KIND=C_INT), Parameter :: NC_FORMAT_CLASSIC   = 1
 Integer(KIND=C_INT), Parameter :: NC_FORMAT_64BIT     = 2
 Integer(KIND=C_INT), Parameter :: NC_FORMAT_NETCDF4   = 3
 Integer(KIND=C_INT), Parameter :: NC_FORMAT_NETCDF4_CLASSIC = 4

! Unlimited dimension size argument and global attibute ID

 Integer(KIND=C_INT), Parameter :: NC_UNLIMITED = 0
 Integer(KIND=C_INT), Parameter :: NC_GLOBAL    = -1

! Implementation limits (WARNING!  SHOULD BE THE SAME AS C INTERFACE)

 Integer(KIND=C_INT), Parameter :: NC_MAX_DIMS     = 1024 
 Integer(KIND=C_INT), Parameter :: NC_MAX_ATTRS    = 8192 
 Integer(KIND=C_INT), Parameter :: NC_MAX_VARS     = 8192 
 Integer(KIND=C_INT), Parameter :: NC_MAX_NAME     = 256 
 Integer(KIND=C_INT), Parameter :: NC_MAX_VAR_DIMS = NC_MAX_DIMS

! Error codes

 Integer(KIND=C_INT), Parameter :: NC_NOERR        = 0
 Integer(KIND=C_INT), Parameter :: NC_EBADID       = -33
 Integer(KIND=C_INT), Parameter :: NC_EEXIST       = -35
 Integer(KIND=C_INT), Parameter :: NC_EINVAL       = -36
 Integer(KIND=C_INT), Parameter :: NC_EPERM        = -37
 Integer(KIND=C_INT), Parameter :: NC_ENOTINDEFINE = -38
 Integer(KIND=C_INT), Parameter :: NC_EINDEFINE    = -39
 Integer(KIND=C_INT), Parameter :: NC_EINVALCOORDS = -40
 Integer(KIND=C_INT), Parameter :: NC_EMAXDIMS     = -41
 Integer(KIND=C_INT), Parameter :: NC_ENAMEINUSE   = -42
 Integer(KIND=C_INT), Parameter :: NC_ENOTATT      = -43
 Integer(KIND=C_INT), Parameter :: NC_EMAXATTS     = -44
 Integer(KIND=C_INT), Parameter :: NC_EBADTYPE     = -45
 Integer(KIND=C_INT), Parameter :: NC_EBADDIM      = -46
 Integer(KIND=C_INT), Parameter :: NC_EUNLIMPOS    = -47
 Integer(KIND=C_INT), Parameter :: NC_EMAXVARS     = -48
 Integer(KIND=C_INT), Parameter :: NC_ENOTVAR      = -49
 Integer(KIND=C_INT), Parameter :: NC_EGLOBAL      = -50
 Integer(KIND=C_INT), Parameter :: NC_ENOTNC       = -51
 Integer(KIND=C_INT), Parameter :: NC_ESTS         = -52
 Integer(KIND=C_INT), Parameter :: NC_EMAXNAME     = -53
 Integer(KIND=C_INT), Parameter :: NC_EUNLIMIT     = -54
 Integer(KIND=C_INT), Parameter :: NC_ENORECVARS   = -55
 Integer(KIND=C_INT), Parameter :: NC_ECHAR        = -56
 Integer(KIND=C_INT), Parameter :: NC_EEDGE        = -57
 Integer(KIND=C_INT), Parameter :: NC_ESTRIDE      = -58
 Integer(KIND=C_INT), Parameter :: NC_EBADNAME     = -59
 Integer(KIND=C_INT), Parameter :: NC_ERANGE       = -60
 Integer(KIND=C_INT), Parameter :: NC_ENOMEM       = -61
 Integer(KIND=C_INT), Parameter :: NC_EVARSIZE     = -62
 Integer(KIND=C_INT), Parameter :: NC_EDIMSIZE     = -63
 Integer(KIND=C_INT), Parameter :: NC_ETRUNC       = -64

! Error handling codes

 Integer(KIND=C_INT), Parameter :: NC_FATAL   = 1
 Integer(KIND=C_INT), Parameter :: NC_VERBOSE = 2

! Procedure interface for f2c_xtype conversion utility

 Interface f2c_xtype
  Module Procedure f2c_xtype
 End Interface

! Begin explicit interfaces for nc_ functions. Note that some interfaces
! expect data to be passed as C_PTR type variables. These data are arrays
! that could have a NULL pointer passed instead of the array. Also, note
! the use of generic interfaces to support routines that handle text data
! to allow text to be passed as either a single string or an array of
! single characters 


!---------------------------------- nc_strerror -------------------------------
Interface
 Function nc_strerror(ncerr) BIND(C)

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncerr
 Type(C_PTR) :: nc_strerror

 End Function nc_strerror
End Interface

!---------------------------------- nc_inq_libvers ----------------------------
Interface
 Function nc_inq_libvers() BIND(C)

  IMPORT :: C_PTR

 Implicit NONE

 Type(C_PTR) :: nc_inq_libvers

 End Function nc_inq_libvers
End Interface

!---------------------------------- nc_create ---------------------------------
Interface
 Function nc_create(path, cmode, ncidp) BIND(C)

  IMPORT :: C_CHAR, C_INT

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer(KIND=C_INT), VALUE :: cmode
 Integer(KIND=C_INT), Intent(OUT) :: ncidp

 Integer(KIND=C_INT) :: nc_create

 End Function nc_create
End Interface

!---------------------------------- nc__create --------------------------------
Interface
 Function nc__create(path, cmode, initialsz, chunksizehintp, ncidp) BIND(C)

  IMPORT :: C_CHAR, C_INT, C_SIZE_T

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer(KIND=C_INT), VALUE :: cmode
 Integer(KIND=C_SIZE_T), VALUE :: initialsz
 Integer(KIND=C_SIZE_T), Intent(IN) :: chunksizehintp 
 Integer(KIND=C_INT), Intent(OUT) :: ncidp

 Integer(KIND=C_INT) :: nc__create

 End Function nc__create
End Interface

!---------------------------------- nc__create_mp -----------------------------
Interface
 Function nc__create_mp(path, cmode, initialsz, basepe, chunksizehintp, ncidp) &
                        BIND(C)

  IMPORT :: C_CHAR, C_INT, C_SIZE_T, C_PTR

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer(KIND=C_INT), VALUE :: cmode
 Integer(KIND=C_SIZE_T), VALUE :: initialsz
 Integer(KIND=C_SIZE_T), Intent(IN) :: chunksizehintp
 Type(C_PTR) :: basepe
 Integer(KIND=C_INT), Intent(OUT) :: ncidp

 Integer(KIND=C_INT) :: nc__create_mp

 End Function nc__create_mp
End Interface

!---------------------------------- nc_open -----------------------------------
Interface
 Function nc_open(path, mode, ncidp) BIND(C)

  IMPORT :: C_CHAR, C_INT

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer(KIND=C_INT), VALUE :: mode
 Integer(KIND=C_INT), Intent(OUT) :: ncidp

 Integer(KIND=C_INT) :: nc_open

 End Function nc_open
End Interface

!---------------------------------- nc__open ----------------------------------
Interface
 Function nc__open(path, mode, chunksizehintp, ncidp) BIND(C)

  IMPORT :: C_CHAR, C_INT, C_SIZE_T

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer(KIND=C_INT), VALUE :: mode
 Integer(KIND=C_SIZE_T), Intent(IN) :: chunksizehintp
 Integer(KIND=C_INT), Intent(OUT) :: ncidp

 Integer(KIND=C_INT) :: nc__open

 End Function nc__open
End Interface

!---------------------------------- nc__open_mp -------------------------------
Interface
 Function nc__open_mp(path, mode, basepe, chunksizehintp, ncidp) BIND(C)

  IMPORT :: C_CHAR, C_INT, C_SIZE_T, C_PTR

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer(KIND=C_INT), VALUE :: mode
 Integer(KIND=C_SIZE_T), Intent(IN) :: chunksizehintp
 Integer(KIND=C_INT), Intent(OUT) :: ncidp
 Type(C_PTR) :: basepe

 Integer(KIND=C_INT) :: nc__open_mp

 End Function nc__open_mp
End Interface

!---------------------------------- nc_set_fill -------------------------------
Interface
 Function nc_set_fill(ncid, fillmode, old_modep) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), VALUE :: fillmode 
 Integer(KIND=C_INT), Intent(OUT) :: old_modep

 Integer(KIND=C_INT) :: nc_set_fill

 End Function nc_set_fill
End Interface

!---------------------------------- nc_redef ----------------------------------
Interface
 Function nc_redef(ncid) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid

 Integer(KIND=C_INT) :: nc_redef

 End Function nc_redef
End Interface

!---------------------------------- nc_enddef ---------------------------------
Interface
 Function nc_enddef(ncid) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid

 Integer(KIND=C_INT) :: nc_enddef

 End Function nc_enddef
End Interface

!---------------------------------- nc__enddef --------------------------------
Interface
 Function nc__enddef(ncid, h_minfree, v_align, v_minfree, r_align) BIND(C)

  IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_SIZE_T), VALUE :: h_minfree, v_align, v_minfree, &
                                  r_align

 Integer(KIND=C_INT) :: nc__enddef

 End Function nc__enddef
End Interface

!---------------------------------- nc_sync -----------------------------------
Interface
 Function nc_sync(ncid) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid

 Integer(KIND=C_INT) :: nc_sync

 End Function nc_sync
End Interface

!---------------------------------- nc_abort ----------------------------------
Interface
 Function nc_abort(ncid) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid

 Integer(KIND=C_INT) :: nc_abort

 End Function nc_abort
End Interface

!---------------------------------- nc_close ----------------------------------
Interface
 Function nc_close(ncid) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid

 Integer(KIND=C_INT) :: nc_close

 End Function nc_close
End Interface

!---------------------------------- nc_delete --------------------------------
Interface
 Function nc_delete(path) BIND(C)
 
  IMPORT :: C_INT

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path

 Integer(KIND=C_INT) :: nc_delete

 End Function nc_delete
End Interface

!---------------------------------- nc_delete_mp -----------------------------
Interface
 Function nc_delete_mp(path, pe) BIND(C)
 
  IMPORT :: C_INT

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer(KIND=C_INT), VALUE :: pe

 Integer(KIND=C_INT) :: nc_delete_mp

 End Function nc_delete_mp
End Interface

!---------------------------------- nc_set_base_pe ----------------------------
Interface
 Function nc_set_base_pe(ncid, pe) BIND(C)

  IMPORT :: C_INT

 Integer(KIND=C_INT), VALUE :: ncid, pe

 Integer(KIND=C_INT) :: nc_set_base_pe

 End Function nc_set_base_pe
End Interface

!---------------------------------- nc_inq_base_pe ----------------------------
Interface
 Function nc_inq_base_pe(ncid, pe) BIND(C)

  IMPORT :: C_INT

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), Intent(OUT) ::  pe

 Integer(KIND=C_INT) :: nc_inq_base_pe

 End Function nc_inq_base_pe
End Interface

!---------------------------------- nc_inq ------------------------------------
Interface
 Function nc_inq(ncid, ndimsp, nvarsp, ngattsp, unlimdimidp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), Intent(OUT) :: ndimsp, nvarsp, ngattsp, unlimdimidp

 Integer(KIND=C_INT) :: nc_inq

 End Function nc_inq
End Interface

!---------------------------------- nc_inq_ndims ------------------------------
Interface
 Function nc_inq_ndims(ncid, ndimsp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), Intent(OUT) ::  ndimsp

 Integer(KIND=C_INT) :: nc_inq_ndims

 End Function nc_inq_ndims
End Interface

!---------------------------------- nc_inq_nvars ------------------------------
Interface
 Function nc_inq_nvars(ncid, nvarsp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), Intent(OUT) ::  nvarsp

 Integer(KIND=C_INT) :: nc_inq_nvars

 End Function nc_inq_nvars
End Interface

!---------------------------------- nc_inq_natts ------------------------------
Interface
 Function nc_inq_natts(ncid, ngattsp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), Intent(OUT) ::  ngattsp

 Integer(KIND=C_INT) :: nc_inq_natts

 End Function nc_inq_natts
End Interface

!---------------------------------- nc_inq_unlimdim ---------------------------
Interface
 Function nc_inq_unlimdim(ncid, unlimdimidp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), Intent(OUT) :: unlimdimidp 

 Integer(KIND=C_INT) :: nc_inq_unlimdim

 End Function nc_inq_unlimdim
End Interface
!---------------------------------- nc_inq_format -----------------------------
Interface
 Function nc_inq_format(ncid, formatp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), Intent(OUT) :: formatp

 Integer(KIND=C_INT) :: nc_inq_format

 End Function nc_inq_format
End Interface

!---------------------------------- nc_def_dim --------------------------------
Interface
 Function nc_def_dim(ncid, name, nlen, idp) BIND(C)

  IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_SIZE_T), VALUE :: nlen
 Integer(KIND=C_INT), Intent(OUT) :: idp 

 Integer(KIND=C_INT) :: nc_def_dim

 End Function nc_def_dim
End Interface

!---------------------------------- nc_inq_dimid ------------------------------
Interface
 Function nc_inq_dimid(ncid, name, idp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_INT), Intent(OUT) :: idp 

 Integer(KIND=C_INT) :: nc_inq_dimid

 End Function nc_inq_dimid
End Interface

!---------------------------------- nc_inq_dim --------------------------------
Interface
 Function nc_inq_dim(ncid, dimid, name, lenp) BIND(C)

  IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), VALUE :: dimid 
 Character(LEN=*), Intent(OUT) :: name
 Integer(KIND=C_SIZE_T), Intent(OUT) :: lenp 

 Integer(KIND=C_INT) :: nc_inq_dim

 End Function nc_inq_dim
End Interface

!---------------------------------- nc_inq_dimname ----------------------------
Interface
 Function nc_inq_dimname(ncid, dimid, name) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), VALUE :: dimid 
 Character(LEN=*), Intent(OUT) :: name

 Integer(KIND=C_INT) :: nc_inq_dimname

 End Function nc_inq_dimname
End Interface

!---------------------------------- nc_inq_dimlen -----------------------------
Interface
 Function nc_inq_dimlen(ncid, dimid, lenp) BIND(C)

  IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), VALUE :: dimid 
 Integer(KIND=C_SIZE_T), Intent(OUT) :: lenp 

 Integer(KIND=C_INT) :: nc_inq_dimlen

 End Function nc_inq_dimlen
End Interface

!---------------------------------- nc_rename_dim -----------------------------
Interface
 Function nc_rename_dim(ncid, dimid, name) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Integer(KIND=C_INT), VALUE :: dimid 
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_INT) :: nc_rename_dim

 End Function nc_rename_dim
End Interface

!---------------------------------- nc_def_var --------------------------------
Interface
 Function nc_def_var(ncid, name, xtype, ndims, dimidsp, varidp) BIND(C)

!  IMPORT :: C_INT, nc_type
  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Character(LEN=*), Intent(IN) :: name
! Type(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Integer(KIND=C_INT), VALUE :: ndims
 Integer(KIND=C_INT), Intent(IN) :: dimidsp(*)
 Integer(KIND=C_INT), Intent(OUT) :: varidp
 Integer(KIND=C_INT) :: nc_def_var

 End Function nc_def_var
End Interface

!---------------------------------- nc_inq_var --------------------------------
Interface
 Function nc_inq_var(ncid, varid, name, xtypep, ndimsp, dimidsp, nattsp) &
                     BIND(C)

!  IMPORT :: C_INT, nc_type, C_PTR
  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(OUT) :: name
! Type(nc_type), Intent(OUT) :: xtypep
 Integer, Intent(OUT) :: xtypep
 Integer(KIND=C_INT), Intent(OUT):: ndimsp
 Integer(KIND=C_INT), Intent(OUT) :: dimidsp(*)
 Integer(KIND=C_INT), Intent(OUT) :: nattsp

 Integer(KIND=C_INT) :: nc_inq_var

 End Function nc_inq_var
End Interface

!---------------------------------- nc_inq_varid ------------------------------
Interface
 Function nc_inq_varid(ncid, name, varidp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_INT), Intent(OUT) :: varidp

 Integer(KIND=C_INT) :: nc_inq_varid

 End Function nc_inq_varid
End Interface

!---------------------------------- nc_inq_varname ----------------------------
Interface
 Function nc_inq_varname(ncid, varid, name) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(OUT) :: name

 Integer(KIND=C_INT) :: nc_inq_varname

 End Function nc_inq_varname
End Interface

!---------------------------------- nc_inq_vartype ----------------------------
Interface
 Function nc_inq_vartype(ncid, varid, xtypep) BIND(C)

!  IMPORT :: C_INT, nc_type
  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
! Type(nc_type), Intent(OUT) :: xtypep
 Integer, Intent(OUT) :: xtypep

 Integer(KIND=C_INT) :: nc_inq_vartype

 End Function nc_inq_vartype
End Interface

!---------------------------------- nc_inq_varndims ---------------------------
Interface
 Function nc_inq_varndims(ncid, varid, ndimsp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_INT), Intent(OUT) :: ndimsp

 Integer(KIND=C_INT) :: nc_inq_varndims

 End Function nc_inq_varndims
End Interface

!---------------------------------- nc_inq_vardimid ---------------------------
Interface
 Function nc_inq_vardimid(ncid, varid, dimidsp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_INT), Intent(OUT) :: dimidsp(*)

 Integer(KIND=C_INT) :: nc_inq_vardimid

 End Function nc_inq_vardimid
End Interface

!---------------------------------- nc_inq_varnatts ---------------------------
Interface
 Function nc_inq_varnatts(ncid, varid, nattsp) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_INT), Intent(OUT) :: nattsp

 Integer(KIND=C_INT) :: nc_inq_varnatts

 End Function nc_inq_varnatts
End Interface

!---------------------------------- nc_rename_var -----------------------------
Interface
 Function nc_rename_var(ncid, varid, name) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name

 Integer(KIND=C_INT) :: nc_rename_var

 End Function nc_rename_var
End Interface

!---------------------------------- nc_put_var_text ---------------------------
Interface nc_put_var_text
 Function nc_put_var_text_s(ncid, varid, op) BIND(C, NAME='nc_put_var_text')

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var_text_s

 End Function nc_put_var_text_s
! array of characters
 Function nc_put_var_text_a(ncid, varid, op) BIND(C, NAME='nc_put_var_text')

  IMPORT :: C_INT, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=C_CHAR), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_text_a

 End Function nc_put_var_text_a
End Interface

!---------------------------------- nc_get_var_text ---------------------------
Interface nc_get_var_text
 Function nc_get_var_text_s(ncid, varid, ip) BIND(C, NAME='nc_get_var_text')

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var_text_s

 End Function nc_get_var_text_s
! array of characters
 Function nc_get_var_text_a(ncid, varid, ip) BIND(C, NAME='nc_get_var_text')

  IMPORT :: C_INT, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=C_CHAR), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_text_a

 End Function nc_get_var_text_a
End Interface

!---------------------------------- nc_put_var_uchar --------------------------
Interface
 Function nc_put_var_uchar(ncid, varid, op) BIND(C)

  IMPORT :: C_INT, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_uchar

 End Function nc_put_var_uchar
End Interface

!---------------------------------- nc_get_var_uchar --------------------------
Interface
 Function nc_get_var_uchar(ncid, varid, ip) BIND(C)

  IMPORT :: C_INT, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_uchar

 End Function nc_get_var_uchar
End Interface

!---------------------------------- nc_put_var_schar --------------------------
Interface
 Function nc_put_var_schar(ncid, varid, op) BIND(C)

  IMPORT :: C_INT, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_schar

 End Function nc_put_var_schar
End Interface

!---------------------------------- nc_get_var_schar --------------------------
Interface
 Function nc_get_var_schar(ncid, varid, ip) BIND(C)

  IMPORT :: C_INT, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_schar

 End Function nc_get_var_schar
End Interface

!---------------------------------- nc_put_var_short --------------------------
Interface
 Function nc_put_var_short(ncid, varid, op) BIND(C)

  IMPORT :: C_INT, CINT2

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=CINT2), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_short

 End Function nc_put_var_short
End Interface

!---------------------------------- nc_get_var_short --------------------------
Interface
 Function nc_get_var_short(ncid, varid, ip) BIND(C)

  IMPORT :: C_INT, CINT2

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=CINT2), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_short

 End Function nc_get_var_short
End Interface

!---------------------------------- nc_put_var_int ----------------------------
Interface
 Function nc_put_var_int(ncid, varid, op) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_INT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_int

 End Function nc_put_var_int
End Interface

!---------------------------------- nc_get_var_int ----------------------------
Interface
 Function nc_get_var_int(ncid, varid, ip) BIND(C)

  IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_INT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_int

 End Function nc_get_var_int
End Interface

!---------------------------------- nc_put_var_long ----------------------------
Interface
 Function nc_put_var_long(ncid, varid, op) BIND(C)

  IMPORT :: C_INT, C_LONG

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_LONG), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_long

 End Function nc_put_var_long
End Interface

!---------------------------------- nc_get_var_long ---------------------------
Interface
 Function nc_get_var_long(ncid, varid, ip) BIND(C)

  IMPORT :: C_INT, C_LONG

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_LONG), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_long

 End Function nc_get_var_long
End Interface

!---------------------------------- nc_put_var_float --------------------------
Interface
 Function nc_put_var_float(ncid, varid, op) BIND(C)

  IMPORT :: C_INT, C_FLOAT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Real(KIND=C_FLOAT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_float

 End Function nc_put_var_float
End Interface

!---------------------------------- nc_get_var_float --------------------------
Interface
 Function nc_get_var_float(ncid, varid, ip) BIND(C)

  IMPORT :: C_INT, C_FLOAT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Real(KIND=C_FLOAT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_float

 End Function nc_get_var_float
End Interface

!---------------------------------- nc_put_var_double -------------------------
Interface
 Function nc_put_var_double(ncid, varid, op) BIND(C)

  IMPORT :: C_INT, C_DOUBLE

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Real(KIND=C_DOUBLE), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_var_double

 End Function nc_put_var_double
End Interface

!---------------------------------- nc_get_var_double -------------------------
Interface
 Function nc_get_var_double(ncid, varid, ip) BIND(C)

  IMPORT :: C_INT, C_DOUBLE

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Real(KIND=C_DOUBLE), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_var_double

 End Function nc_get_var_double
End Interface

!---------------------------------- nc_put_var1_text --------------------------
Interface
 Function nc_put_var1_text(ncid, varid, indexp, op) BIND(C) 

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Character(LEN=C_CHAR), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_text

 End Function nc_put_var1_text
End Interface

!---------------------------------- nc_get_var1_text --------------------------
Interface
 Function nc_get_var1_text(ncid, varid, indexp, ip) BIND(C)

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Character(LEN=C_CHAR), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_text

 End Function nc_get_var1_text
End Interface

!---------------------------------- nc_put_var1_uchar -------------------------
Interface
 Function nc_put_var1_uchar(ncid, varid, indexp, op) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=CINT1),  Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_uchar

 End Function nc_put_var1_uchar
End Interface

!---------------------------------- nc_get_var1_uchar -------------------------
Interface
 Function nc_get_var1_uchar(ncid, varid, indexp, ip) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=CINT1),  Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_uchar

 End Function nc_get_var1_uchar
End Interface

!---------------------------------- nc_put_var1_schar -------------------------
Interface
 Function nc_put_var1_schar(ncid, varid, indexp, op) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=CINT1),  Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_schar

 End Function nc_put_var1_schar
End Interface

!---------------------------------- nc_get_var1_schar -------------------------
Interface
 Function nc_get_var1_schar(ncid, varid, indexp, ip) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=CINT1),  Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_schar

 End Function nc_get_var1_schar
End Interface

!---------------------------------- nc_put_var1_short -------------------------
Interface
 Function nc_put_var1_short(ncid, varid, indexp, op) BIND(C)

  IMPORT :: C_INT, CINT2, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=CINT2), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_short

 End Function nc_put_var1_short
End Interface

!---------------------------------- nc_get_var1_short -------------------------
Interface
 Function nc_get_var1_short(ncid, varid, indexp, ip) BIND(C) 

  IMPORT :: C_INT, CINT2, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=CINT2), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_short

 End Function nc_get_var1_short
End Interface

!---------------------------------- nc_put_var1_int ---------------------------
Interface
 Function nc_put_var1_int(ncid, varid, indexp, op) BIND(C)

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=C_INT), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_int

 End Function nc_put_var1_int
End Interface

!---------------------------------- nc_get_var1_int ---------------------------
Interface
 Function nc_get_var1_int(ncid, varid, indexp, ip) BIND(C)

!  IMPORT :: C_INT, C_SIZE_T
  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=C_INT), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_int

 End Function nc_get_var1_int
End Interface

!---------------------------------- nc_put_var1_long --------------------------
Interface
 Function nc_put_var1_long(ncid, varid, indexp, op) BIND(C)

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=C_LONG), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_long

 End Function nc_put_var1_long
End Interface

!---------------------------------- nc_get_var1_long --------------------------
Interface
 Function nc_get_var1_long(ncid, varid, indexp, ip) BIND(C)

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Integer(KIND=C_LONG), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_long

 End Function nc_get_var1_long
End Interface

!---------------------------------- nc_put_var1_float -------------------------
Interface
 Function nc_put_var1_float(ncid, varid, indexp, op) BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Real(KIND=C_FLOAT), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_float

 End Function nc_put_var1_float
End Interface

!---------------------------------- nc_get_var1_float -------------------------
Interface
 Function nc_get_var1_float(ncid, varid, indexp, ip) BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Real(KIND=C_FLOAT), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_float

 End Function nc_get_var1_float
End Interface

!---------------------------------- nc_put_var1_double ------------------------
Interface
 Function nc_put_var1_double(ncid, varid, indexp, op) BIND(C)

  IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Real(KIND=C_DOUBLE), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_var1_double

 End Function nc_put_var1_double
End Interface

!---------------------------------- nc_get_var1_double ------------------------
Interface
 Function nc_get_var1_double(ncid, varid, indexp, ip) BIND(C)

  IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: indexp
 Real(KIND=C_DOUBLE), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_var1_double

 End Function nc_get_var1_double
End Interface

!---------------------------------- nc_put_vara_text --------------------------
Interface nc_put_vara_text
 Function nc_put_vara_text(ncid, varid, startp, countp, op)  &
                          BIND(C, NAME='nc_put_vara_text')

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Character(LEN=*), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_vara_text

 End Function nc_put_vara_text
 Function nc_put_vara_text_a(ncid, varid, startp, countp, op)  & 
                             BIND(C, NAME='nc_put_vara_text')

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Character(LEN=C_CHAR), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_text_a

 End Function nc_put_vara_text_a
End Interface

!---------------------------------- nc_get_vara_text --------------------------
Interface nc_get_vara_text
 Function nc_get_vara_text(ncid, varid, startp, countp, ip)  &
                           BIND(C, NAME='nc_get_vara_text')

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Character(LEN=*), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_vara_text

 End Function nc_get_vara_text
! array of characters
 Function nc_get_vara_text_a(ncid, varid, startp, countp, ip)  & 
                             BIND(C, NAME='nc_get_vara_text')

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Character(LEN=C_CHAR), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_text_a

 End Function nc_get_vara_text_a
End Interface

!---------------------------------- nc_put_vara_uchar -------------------------
Interface
 Function nc_put_vara_uchar(ncid, varid, startp, countp, op) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_uchar

 End Function nc_put_vara_uchar
End Interface

!---------------------------------- nc_get_vara_uchar -------------------------
Interface
 Function nc_get_vara_uchar(ncid, varid, startp, countp, ip) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_uchar

 End Function nc_get_vara_uchar
End Interface

!---------------------------------- nc_put_vara_schar -------------------------
Interface
 Function nc_put_vara_schar(ncid, varid, startp, countp, op) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_schar

 End Function nc_put_vara_schar
End Interface

!---------------------------------- nc_get_vara_schar -------------------------
Interface
 Function nc_get_vara_schar(ncid, varid, startp, countp, ip) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_schar

 End Function nc_get_vara_schar
End Interface

!---------------------------------- nc_put_vara_short -------------------------
Interface
 Function nc_put_vara_short(ncid, varid, startp, countp, op) BIND(C)

  IMPORT :: C_INT, CINT2, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=CINT2), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_short

 End Function nc_put_vara_short
End Interface

!---------------------------------- nc_get_vara_short -------------------------
Interface
 Function nc_get_vara_short(ncid, varid, startp, countp, ip) BIND(C)

  IMPORT :: C_INT, CINT2, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=CINT2), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_short

 End Function nc_get_vara_short
End Interface

!--------------------------------- nc_put_vara_int ----------------------------
Interface
 Function nc_put_vara_int(ncid, varid, startp, countp, op) BIND(C)

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=C_INT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_int

 End Function nc_put_vara_int
End Interface

!--------------------------------- nc_get_vara_int ----------------------------
Interface
 Function nc_get_vara_int(ncid, varid, startp, countp, ip) BIND(C) 

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=C_INT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_int

 End Function nc_get_vara_int
End Interface

!--------------------------------- nc_put_vara_long ---------------------------
Interface
 Function nc_put_vara_long(ncid, varid, startp, countp, op) BIND(C)

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=C_LONG), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_long

 End Function nc_put_vara_long
End Interface

!--------------------------------- nc_get_vara_long ---------------------------
Interface
 Function nc_get_vara_long(ncid, varid, startp, countp, ip) BIND(C)

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Integer(KIND=C_LONG), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_long

 End Function nc_get_vara_long
End Interface

!--------------------------------- nc_put_vara_float --------------------------
Interface
 Function nc_put_vara_float(ncid, varid, startp, countp, op) BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Real(KIND=C_FLOAT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_float

 End Function nc_put_vara_float
End Interface

!--------------------------------- nc_get_vara_float --------------------------
Interface
 Function nc_get_vara_float(ncid, varid, startp, countp, ip) BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Real(KIND=C_FLOAT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_float

 End Function nc_get_vara_float
End Interface

!--------------------------------- nc_put_vara_double -------------------------
Interface
 Function nc_put_vara_double(ncid, varid, startp, countp, op) BIND(C)

  IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Real(KIND=C_DOUBLE), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vara_double

 End Function nc_put_vara_double
End Interface

!--------------------------------- nc_get_vara_double -------------------------
Interface
 Function nc_get_vara_double(ncid, varid, startp, countp, ip) BIND(C)

  IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp
 Real(KIND=C_DOUBLE), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vara_double

 End Function nc_get_vara_double
End Interface

!--------------------------------- nc_put_vars_text ---------------------------
Interface nc_put_vars_text
 Function nc_put_vars_text(ncid, varid, startp, countp, stridep, op)  &
                           BIND(C, NAME='nc_put_vars_text')

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Character(LEN=*), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_vars_text

 End Function nc_put_vars_text
! array of characters
 Function nc_put_vars_text_a(ncid, varid, startp, countp, stridep, op)  &
                             BIND(C, NAME='nc_put_vars_text')

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Character(LEN=C_CHAR), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_text_a

 End Function nc_put_vars_text_a
End Interface

!--------------------------------- nc_get_vars_text ---------------------------
Interface nc_get_vars_text
 Function nc_get_vars_text(ncid, varid, startp, countp, stridep, ip)  &
                           BIND(C, NAME='nc_get_vars_text')

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Character(LEN=*), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_vars_text

 End Function nc_get_vars_text
 Function nc_get_vars_text_a(ncid, varid, startp, countp, stridep, ip)  &
                             BIND(C, NAME='nc_get_vars_text')

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Character(LEN=C_CHAR), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_text_a

 End Function nc_get_vars_text_a
End Interface

!--------------------------------- nc_put_vars_uchar --------------------------
Interface
 Function nc_put_vars_uchar(ncid, varid, startp, countp, stridep, op) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_uchar

 End Function nc_put_vars_uchar
End Interface

!--------------------------------- nc_get_vars_uchar --------------------------
Interface
 Function nc_get_vars_uchar(ncid, varid, startp, countp, stridep, ip) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_uchar

 End Function nc_get_vars_uchar
End Interface

!--------------------------------- nc_put_vars_schar --------------------------
Interface
 Function nc_put_vars_schar(ncid, varid, startp, countp, stridep, op) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_schar

 End Function nc_put_vars_schar
End Interface

!--------------------------------- nc_get_vars_schar --------------------------
Interface
 Function nc_get_vars_schar(ncid, varid, startp, countp, stridep, ip) BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_schar

 End Function nc_get_vars_schar
End Interface

!--------------------------------- nc_put_vars_short --------------------------
Interface
 Function nc_put_vars_short(ncid, varid, startp, countp, stridep, op) BIND(C) 

  IMPORT :: C_INT, CINT2, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=CINT2), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_short

 End Function nc_put_vars_short
End Interface

!--------------------------------- nc_get_vars_short --------------------------
Interface
 Function nc_get_vars_short(ncid, varid, startp, countp, stridep, ip) BIND(C) 

  IMPORT :: C_INT, CINT2,  C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=CINT2), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_short

 End Function nc_get_vars_short
End Interface

!--------------------------------- nc_put_vars_int ----------------------------
Interface
 Function nc_put_vars_int(ncid, varid, startp, countp, stridep, op) BIND(C)

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=C_INT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_int

 End Function nc_put_vars_int
End Interface

!--------------------------------- nc_get_vars_int ----------------------------
Interface
 Function nc_get_vars_int(ncid, varid, startp, countp, stridep, ip) BIND(C)

  IMPORT :: C_INT, C_SIZE_T, C_PTRDIFF_T, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=C_INT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_int

 End Function nc_get_vars_int
End Interface

!--------------------------------- nc_put_vars_long ---------------------------
Interface
 Function nc_put_vars_long(ncid, varid, startp, countp, stridep, op) BIND(C) 

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=C_LONG), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_long

 End Function nc_put_vars_long
End Interface

!--------------------------------- nc_get_vars_long ---------------------------
Interface
 Function nc_get_vars_long(ncid, varid, startp, countp, stridep, ip) BIND(C)

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Integer(KIND=C_LONG), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_long

 End Function nc_get_vars_long
End Interface

!--------------------------------- nc_put_vars_float --------------------------
Interface
 Function nc_put_vars_float(ncid, varid, startp, countp, stridep, op) BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Real(KIND=C_FLOAT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_float

 End Function nc_put_vars_float
End Interface

!--------------------------------- nc_get_vars_float --------------------------
Interface
 Function nc_get_vars_float(ncid, varid, startp, countp, stridep, ip) BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Real(KIND=C_FLOAT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_float

 End Function nc_get_vars_float
End Interface

!--------------------------------- nc_put_vars_double -------------------------
Interface
 Function nc_put_vars_double(ncid, varid, startp, countp, stridep, op) BIND(C)

  IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE
  
 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Real(KIND=C_DOUBLE), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_vars_double

 End Function nc_put_vars_double
End Interface

!--------------------------------- nc_get_vars_double -------------------------
Interface
 Function nc_get_vars_double(ncid, varid, startp, countp, stridep, ip) BIND(C)

 IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep
 Real(KIND=C_DOUBLE), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_vars_double

 End Function nc_get_vars_double
End Interface

!--------------------------------- nc_put_varm_text ---------------------------
Interface nc_put_varm_text
 Function nc_put_varm_text(ncid, varid, startp, countp, stridep, imapp,op)  &
                           BIND(C, NAME='nc_put_varm_text')

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Character(LEN=*), Intent(IN) :: op

 Integer(KIND=C_INT) :: nc_put_varm_text

 End Function nc_put_varm_text
! array of characters
 Function nc_put_varm_text_a(ncid, varid, startp, countp, stridep, imapp,op)  &
                             BIND(C, NAME='nc_put_varm_text')

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Character(LEN=C_CHAR), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_text_a

 End Function nc_put_varm_text_a
End Interface

!--------------------------------- nc_get_varm_text ---------------------------
Interface nc_get_varm_text
 Function nc_get_varm_text(ncid, varid, startp, countp, stridep, imapp,ip)  &
                           BIND(C, NAME='nc_get_varm_text')

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Character(LEN=*), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_varm_text

 End Function nc_get_varm_text
 Function nc_get_varm_text_a(ncid, varid, startp, countp, stridep, imapp,ip)  &
                             BIND(C, NAME='nc_get_varm_text')

  IMPORT :: C_INT, C_PTR, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Character(LEN=C_CHAR), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_text_a

 End Function nc_get_varm_text_a
End Interface

!--------------------------------- nc_put_varm_uchar --------------------------
Interface
 Function nc_put_varm_uchar(ncid, varid, startp, countp, stridep, imapp, op)  &
                            BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_uchar

 End Function nc_put_varm_uchar
End Interface

!--------------------------------- nc_get_varm_uchar --------------------------
Interface
 Function nc_get_varm_uchar(ncid, varid, startp, countp, stridep, imapp,ip)  &
                            BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_uchar

 End Function nc_get_varm_uchar
End Interface

!--------------------------------- nc_put_varm_schar --------------------------
Interface
 Function nc_put_varm_schar(ncid, varid, startp, countp, stridep, imapp,op)  &
                            BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=CINT1),  Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_schar

 End Function nc_put_varm_schar
End Interface

!--------------------------------- nc_get_varm_schar --------------------------
Interface
 Function nc_get_varm_schar(ncid, varid, startp, countp, stridep, imapp,ip)  &
                            BIND(C)

  IMPORT :: C_INT, CINT1, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=CINT1),  Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_schar

 End Function nc_get_varm_schar
End Interface

!--------------------------------- nc_put_varm_short --------------------------
Interface
 Function nc_put_varm_short(ncid, varid, startp, countp, stridep, imapp,op)  &
                            BIND(C)

  IMPORT :: C_INT, CINT2, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=CINT2), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_short

 End Function nc_put_varm_short
End Interface

!--------------------------------- nc_get_varm_short --------------------------
Interface
 Function nc_get_varm_short(ncid, varid, startp, countp, stridep, imapp,ip)  &
                            BIND(C)

  IMPORT :: C_INT, CINT2,  C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=CINT2), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_short

 End Function nc_get_varm_short
End Interface

!--------------------------------- nc_put_varm_int ----------------------------
Interface
 Function nc_put_varm_int(ncid, varid, startp, countp, stridep, imapp, op)  &
                          BIND(C)

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=C_INT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_int

 End Function nc_put_varm_int
End Interface

!--------------------------------- nc_get_varm_int ----------------------------
Interface
 Function nc_get_varm_int(ncid, varid, startp, countp, stridep, imapp, ip)  &
                          BIND(C)

  IMPORT :: C_INT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=C_INT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_int

 End Function nc_get_varm_int
End Interface

!--------------------------------- nc_put_varm_long ---------------------------
Interface
 Function nc_put_varm_long(ncid, varid, startp, countp, stridep, imapp, op)  &
                           BIND(C)

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=C_LONG), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_long

 End Function nc_put_varm_long
End Interface

!--------------------------------- nc_get_varm_long ---------------------------
Interface
 Function nc_get_varm_long(ncid, varid, startp, countp, stridep, imapp, ip)  &
                           BIND(C)

  IMPORT :: C_INT, C_LONG, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Integer(KIND=C_LONG), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_long

 End Function nc_get_varm_long
End Interface

!--------------------------------- nc_put_varm_float --------------------------
Interface
 Function nc_put_varm_float(ncid, varid, startp, countp, stridep, imapp,op)  &
                            BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Real(KIND=C_FLOAT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_float

 End Function nc_put_varm_float
End Interface

!--------------------------------- nc_get_varm_float --------------------------
Interface
 Function nc_get_varm_float(ncid, varid, startp, countp, stridep, imapp,ip)  &
                            BIND(C)

  IMPORT :: C_INT, C_FLOAT, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Real(KIND=C_FLOAT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_float

 End Function nc_get_varm_float
End Interface

!--------------------------------- nc_put_varm_double -------------------------
Interface
 Function nc_put_varm_double(ncid, varid, startp, countp, stridep,imapp,op)  &
                             BIND(C)

  IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE
  
 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Real(KIND=C_DOUBLE), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_varm_double

 End Function nc_put_varm_double
End Interface

!--------------------------------- nc_get_varm_double -------------------------
Interface
 Function nc_get_varm_double(ncid, varid, startp, countp, stridep,imapp,ip)  &
                             BIND(C)

 IMPORT :: C_INT, C_DOUBLE, C_PTR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Type(C_PTR), VALUE :: startp, countp, stridep, imapp
 Real(KIND=C_DOUBLE), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_varm_double

 End Function nc_get_varm_double
End Interface

!--------------------------------- nc_inq_att --------------------------------
Interface
 Function nc_inq_att(ncid, varid, name, xtypep, lenp)  BIND(C)

! IMPORT :: C_INT, C_SIZE_T, nc_type
 IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
! Type(nc_type), Intent(OUT) :: xtypep
 Integer, Intent(OUT) :: xtypep
 Integer(KIND=C_SIZE_T), Intent(OUT) :: lenp

 Integer(KIND=C_INT) :: nc_inq_att

 End Function nc_inq_att
End Interface

!--------------------------------- nc_inq_attid ------------------------------
Interface
 Function nc_inq_attid(ncid, varid, name, attnump)   BIND(C)

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_INT), Intent(OUT) :: attnump

 Integer(KIND=C_INT) :: nc_inq_attid

 End Function nc_inq_attid
End Interface

!--------------------------------- nc_inq_atttype ----------------------------
Interface
 Function nc_inq_atttype(ncid, varid, name, xtypep)   BIND(C)

! IMPORT :: C_INT, nc_type
 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
! Type(nc_type), Intent(OUT) :: xtypep
 Integer, Intent(OUT) :: xtypep

 Integer(KIND=C_INT) :: nc_inq_atttype

 End Function nc_inq_atttype
End Interface

!--------------------------------- nc_inq_attlen -----------------------------
Interface
 Function nc_inq_attlen(ncid, varid, name, lenp)   BIND(C)

 IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_SIZE_T), Intent(OUT) :: lenp

 Integer(KIND=C_INT) :: nc_inq_attlen

 End Function nc_inq_attlen
End Interface

!--------------------------------- nc_inq_attname ----------------------------
Interface
 Function nc_inq_attname(ncid, varid, attnum, name)   BIND(C)

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid, attnum
 Character(LEN=*), Intent(OUT) :: name

 Integer(KIND=C_INT) :: nc_inq_attname

 End Function nc_inq_attname
End Interface

!--------------------------------- nc_copy_att -------------------------------
Interface
 Function nc_copy_att(ncid_in, varid_in, name, ncid_out, varid_out )  BIND(C)

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid_in, varid_in, varid_out, &
                                           ncid_out 
 Character(LEN=*), Intent(IN) :: name

 Integer(KIND=C_INT) :: nc_copy_att

 End Function nc_copy_att
End Interface

!--------------------------------- nc_rename_att -----------------------------
Interface
 Function nc_rename_att(ncid, varid, name, newname)   BIND(C)

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name, newname

 Integer(KIND=C_INT) :: nc_rename_att

 End Function nc_rename_att
End Interface

!--------------------------------- nc_del_att --------------------------------
Interface
 Function nc_del_att(ncid, varid, name)  BIND(C)

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name

 Integer(KIND=C_INT) :: nc_del_att

 End Function nc_del_att
End Interface

!--------------------------------- nc_put_att_text ---------------------------
Interface nc_put_att_text
 Function nc_put_att_text_s(ncid, varid, name, nlen, op)  &
                           BIND(C, NAME='nc_put_att_text')

 IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
 Character(LEN=*), Intent(IN) :: name, op

 Integer(KIND=C_INT) :: nc_put_att_text_s

 End Function nc_put_att_text_s
 Function nc_put_att_text_a(ncid, varid, name, nlen, op)  &
                           BIND(C, NAME='nc_put_att_text')

 IMPORT :: C_INT, C_SIZE_T, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
 Character(LEN=*), Intent(IN) :: name
 Character(LEN=C_CHAR), Intent(IN) ::  op(*)

 Integer(KIND=C_INT) :: nc_put_att_text_a

 End Function nc_put_att_text_a
End Interface

!--------------------------------- nc_get_att_text ---------------------------
Interface nc_get_att_text
 Function nc_get_att_text_s(ncid, varid, name, ip)  &
                            BIND(C, NAME='nc_get_att_text')

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Character(LEN=*), Intent(OUT) :: ip

 Integer(KIND=C_INT) :: nc_get_att_text_s

 End Function nc_get_att_text_s
 Function nc_get_att_text_a(ncid, varid, name, ip)  &
                            BIND(C, NAME='nc_get_att_text')

 IMPORT :: C_INT, C_CHAR

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Character(LEN=C_CHAR), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_att_text_a

 End Function nc_get_att_text_a
End Interface

!--------------------------------- nc_put_att_uchar --------------------------
Interface
 Function nc_put_att_uchar(ncid, varid, name, xtype, nlen, op)   BIND(C)

! IMPORT :: C_INT, C_SIZE_T, CINT1, nc_type
 IMPORT :: C_INT, C_SIZE_T, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
! Type(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=CINT1), Intent(IN) :: op(*) 

 Integer(KIND=C_INT) :: nc_put_att_uchar

 End Function nc_put_att_uchar
End Interface

!--------------------------------- nc_get_att_uchar --------------------------
Interface
 Function nc_get_att_uchar(ncid, varid, name, ip)   BIND(C)

 IMPORT :: C_INT, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=CINT1), Intent(OUT) :: ip(*) 

 Integer(KIND=C_INT) :: nc_get_att_uchar

 End Function nc_get_att_uchar
End Interface

!--------------------------------- nc_put_att_schar --------------------------
Interface
 Function nc_put_att_schar(ncid, varid, name, xtype, nlen, op)   BIND(C)

! IMPORT :: C_INT, C_SIZE_T, CINT1, nc_type
 IMPORT :: C_INT, C_SIZE_T, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
! Type(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=CINT1), Intent(IN) :: op(*) 

 Integer(KIND=C_INT) :: nc_put_att_schar

 End Function nc_put_att_schar
End Interface

!--------------------------------- nc_get_att_schar --------------------------
Interface
 Function nc_get_att_schar(ncid, varid, name, ip)   BIND(C)

 IMPORT :: C_INT, CINT1

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=CINT1), Intent(OUT) :: ip(*) 

 Integer(KIND=C_INT) :: nc_get_att_schar

 End Function nc_get_att_schar
End Interface

!--------------------------------- nc_put_att_short --------------------------
Interface
 Function nc_put_att_short(ncid, varid, name, xtype, nlen, op)   BIND(C)

! IMPORT :: C_INT, C_SIZE_T, CINT2, nc_type
 IMPORT :: C_INT, C_SIZE_T, CINT2

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
! Type(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=CINT2), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_att_short

 End Function nc_put_att_short
End Interface

!--------------------------------- nc_get_att_short --------------------------
Interface
 Function nc_get_att_short(ncid, varid, name, ip)   BIND(C)

 IMPORT :: C_INT, CINT2

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=CINT2), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_att_short

 End Function nc_get_att_short
End Interface

!--------------------------------- nc_put_att_int --------------------------
Interface
 Function nc_put_att_int(ncid, varid, name, xtype, nlen, op)   BIND(C)

! IMPORT :: C_INT, C_SIZE_T, nc_type
 IMPORT :: C_INT, C_SIZE_T

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
! TYPE(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_INT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_att_int

 End Function nc_put_att_int
End Interface

!--------------------------------- nc_get_att_int -----------------------------
Interface
 Function nc_get_att_int(ncid, varid, name, ip)   BIND(C)

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_INT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_att_int

 End Function nc_get_att_int
End Interface

!--------------------------------- nc_put_att_long --------------------------
Interface
 Function nc_put_att_long(ncid, varid, name, xtype, nlen, op)   BIND(C)

! IMPORT :: C_INT, C_SIZE_T, C_LONG, nc_type
 IMPORT :: C_INT, C_SIZE_T, C_LONG

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
! TYPE(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_LONG), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_att_long

 End Function nc_put_att_long
End Interface

!--------------------------------- nc_get_att_long --------------------------
Interface
 Function nc_get_att_long(ncid, varid, name, ip)   BIND(C)

 IMPORT :: C_INT, C_LONG

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Integer(KIND=C_LONG), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_att_long

 End Function nc_get_att_long
End Interface

!--------------------------------- nc_put_att_float --------------------------
Interface
 Function nc_put_att_float(ncid, varid, name, xtype, nlen, op)   BIND(C)

! IMPORT :: C_INT, C_SIZE_T, C_FLOAT, nc_type
 IMPORT :: C_INT, C_SIZE_T, C_FLOAT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
! TYPE(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Character(LEN=*), Intent(IN) :: name
 Real(KIND=C_FLOAT), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_att_float

 End Function nc_put_att_float
End Interface

!--------------------------------- nc_get_att_float --------------------------
Interface
 Function nc_get_att_float(ncid, varid, name, ip)   BIND(C)

 IMPORT :: C_INT, C_FLOAT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Real(KIND=C_FLOAT), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_att_float

 End Function nc_get_att_float
End Interface

!--------------------------------- nc_put_att_double -------------------------
Interface
 Function nc_put_att_double(ncid, varid, name, xtype, nlen, op)   BIND(C)

! IMPORT :: C_INT, C_SIZE_T, C_DOUBLE, nc_type
 IMPORT :: C_INT, C_SIZE_T, C_DOUBLE

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Integer(KIND=C_SIZE_T), VALUE :: nlen
! TYPE(nc_type), VALUE :: xtype
 Integer, VALUE :: xtype
 Character(LEN=*), Intent(IN) :: name
 Real(KIND=C_DOUBLE), Intent(IN) :: op(*)

 Integer(KIND=C_INT) :: nc_put_att_double

 End Function nc_put_att_double
End Interface

!------------------------------- nc_get_att_double -------------------------
Interface
 Function nc_get_att_double(ncid, varid, name, ip)   BIND(C)

 IMPORT :: C_INT, C_DOUBLE

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid, varid
 Character(LEN=*), Intent(IN) :: name
 Real(KIND=C_DOUBLE), Intent(OUT) :: ip(*)

 Integer(KIND=C_INT) :: nc_get_att_double

 End Function nc_get_att_double
End Interface

!------------------------------- nc_copy_var --------------------------------
Interface
 Function nc_copy_var(ncid_in, varid, ncid_out)  BIND(C)

 IMPORT :: C_INT

 Implicit NONE

 Integer(KIND=C_INT), VALUE :: ncid_in, varid, ncid_out
 Integer(KIND=C_INT) :: nc_copy_var

 End Function nc_copy_var
End Interface

!------------------------------- nc_set_default_format -----------------------
Interface
 Function nc_set_default_format(newform, old_format)  BIND(C)
!
 IMPORT :: C_INT

 Integer(KIND=C_INT), VALUE :: newform
 Integer(KIND=C_INT), Intent(OUT) :: old_format
 Integer(KIND=C_INT) :: nc_set_default_format

 End Function nc_set_default_format
End Interface

!---------------------------- Start of module procedure ----------------------

CONTAINS
 
!------------------------------- f2c_xtype -----------------------------------
 Function f2c_xtype(fxtype) RESULT(cxtype)

! Take in Fortran parameter value for xtype and return
! equivalent value of C enumerator cxtype. Original code
! on X1 allowed cxtype to be defined as a derived type
! Can't do this with compilers that don't support a name
! on the enum statement so we have to make cxtype an integer
!
 
 Implicit NONE

 Integer, Intent(IN) :: fxtype

! Type(nc_type) :: cxtype
 Integer(KIND=C_INT) :: cxtype

 Select Case (fxtype)

  CASE (0)
   cxtype = NC_NAT
  
  CASE (1)
   cxtype = NC_BYTE

  CASE (2)
   cxtype = NC_CHAR

  CASE (3)
   cxtype = NC_SHORT

  CASE (4)
   cxtype = NC_INT

  CASE (5)
   cxtype = NC_FLOAT

  CASE (6)
   cxtype = NC_DOUBLE

  CASE DEFAULT
   cxtype = 0

 End Select
 
End Function f2c_xtype

!----------------------------End of Module netcdf_c_interfaces ----------------
End Module netcdf_nc_interfaces
#endif
