#ifndef NO_NETCDF_2
#ifdef Fortran2003
!                  Netcdf Version 2 Fortran interfaces

! These routines replace the cfortran.h defined functions from fort-v2compat.c
! Integer arrays are used as pass-through variables for functions that
! support multiple data types as void pointers in the C routines

! Written by: Richard Weed
!             Engineering Research Center
!             Mississippi State University
!             rweed@erc.msstate.edu

! Version 1: May 2006 - Initial version 2 interfaces

! ------------------------------- ncpopt -------------------------------------- 
 Subroutine ncpopt(ncopts)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncopts
 Integer(KIND=C_INT) :: cncopts

 cncopts = ncopts
 Call c_ncpopt(cncopts)

 End Subroutine ncpopt 
! ------------------------------- ncgopt -------------------------------------- 
 Subroutine ncgopt(ncopts)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(INOUT) :: ncopts
 Integer(KIND=C_INT) :: cncopts

 cncopts = 0 
 Call c_ncgopt(cncopts)
 ncopts = cncopts

 End Subroutine ncgopt 
! ------------------------------- nccre --------------------------------------- 
 Function nccre(filename, cmode, rcode) RESULT(ncid)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: filename
 Integer, Intent(IN) :: cmode
 Integer, Intent(OUT) :: rcode
 Integer :: ncid

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: ccmode, crcode, cncid 

 ccmode = cmode
 cncid  = 0
 rcode  = 0
 crcode = 0

! check for a null character on end of filename

 ilen = LEN_TRIM(filename)
 inull = SCAN(filename, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 cncid = c_nccre(filename(1:ilen)//C_NULL_CHAR,ccmode, crcode )

 rcode = crcode 
 ncid = cncid

 End Function nccre
! ------------------------------- ncopn --------------------------------------- 
 Function ncopn(filename, rwmode, rcode) RESULT(ncid)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: filename
 Integer, Intent(IN) :: rwmode
 Integer, Intent(OUT) :: rcode
 Integer :: ncid

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: crwmode, crcode, cncid 

 crwmode = rwmode
 rcode = 0
 crcode=0
 cncid=0

! check for a null character on end of filename

 ilen = LEN_TRIM(filename)
 inull = SCAN(filename, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 cncid = c_ncopn(filename(1:ilen)//C_NULL_CHAR,crwmode, crcode )

 rcode = crcode 
 ncid = cncid

 End Function ncopn
! ------------------------------- ncddef -------------------------------------- 
 Function ncddef(ncid, dimname, dimlen, rcode) RESULT(ndimid)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: dimname 
 Integer, Intent(IN) :: ncid, dimlen
 Integer, Intent(OUT) :: rcode
 Integer :: ndimid

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cdimlen, cndimid, crcode

 cncid = ncid
 cdimlen = dimlen
 cndimid = 0
 rcode = 0
 ndimid=0

! check for a null character on end of dimname

 ilen = LEN_TRIM(dimname)
 inull = SCAN(dimname, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 cndimid = c_ncddef(cncid, dimname(1:ilen)//C_NULL_CHAR,cdimlen, crcode )
 
 rcode = crcode 
 ndimid = cndimid

 End Function ncddef
! ------------------------------- ncdid --------------------------------------- 
 Function ncdid(ncid, dimname, rcode) RESULT(ndimid)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: dimname 
 Integer, Intent(IN) :: ncid
 Integer, Intent(OUT) :: rcode
 Integer :: ndimid

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, crcode, cndimid

 cncid = ncid
 cndimid = 0
 rcode = 0
 ndimid=0

! check for a null character on end of dimname

 ilen = LEN_TRIM(dimname)
 inull = SCAN(dimname, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 cndimid = c_ncdid(cncid, dimname(1:ilen)//C_NULL_CHAR, crcode )

 rcode = crcode 
 ndimid = cndimid

 End Function ncdid
! ------------------------------- ncvdef -------------------------------------- 
 Function ncvdef(ncid, varname, vartype, nvdims, vdims, rcode) RESULT(nvarid)
!
 USE netcdf_nc_interfaces, ONLY : NC_MAX_DIMS
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: varname 
 Integer, Intent(IN) :: ncid, vartype, nvdims
 Integer, Intent(IN) :: vdims(*)
 Integer, Intent(OUT) :: rcode
 Integer :: nvarid

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, crcode, cnvdims, cvartype, cnvarid
 Integer(KIND=C_INT) :: cvdims(NC_MAX_DIMS)

 cncid = ncid
 cnvdims = nvdims 
 cvartype = vartype
 crcode  = 0
 rcode   = 0
 nvarid  = 0
 cnvarid = 0
 
! check for a null character on end of varname

 ilen = LEN_TRIM(varname)
 inull = SCAN(varname, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
 
 ! mimic f2c_dimids

 cvdims=0
 If (nvdims > 0) Then
   cvdims(1:nvdims) = vdims(nvdims:1:-1) - 1
 EndIf

 cnvarid = c_ncvdef(cncid, varname(1:ilen)//C_NULL_CHAR, cvartype, &
                    cnvdims, cvdims, crcode )

 rcode = crcode 
 nvarid = cnvarid

 End Function ncvdef
! ------------------------------- ncvid --------------------------------------- 
 Function ncvid(ncid, varname, rcode) RESULT(nvarid)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: varname 
 Integer, Intent(IN) :: ncid
 Integer, Intent(OUT) :: rcode
 Integer :: nvarid

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, crcode, cnvarid

 cncid = ncid
 crcode  = 0
 rcode   = 0
 nvarid  = 0
 cnvarid = 0
 
! check for a null character on end of varname

 ilen = LEN_TRIM(varname)
 inull = SCAN(varname, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
 
 cnvarid = c_ncvid(cncid, varname(1:ilen)//C_NULL_CHAR, crcode)

 rcode = crcode 
 nvarid = cnvarid

 End Function ncvid
! ------------------------------- nctlen -------------------------------------- 
 Function nctlen(datatype, rcode) RESULT(nvarlen)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: datatype 
 Integer, Intent(OUT) :: rcode
 Integer :: nvarlen

 Integer(KIND=C_INT) :: crcode, cnvarlen, cdtype

 cdtype = datatype
 crcode  = 0
 rcode   = 0
 nvarlen  = 0
 cnvarlen = 0
 
 cnvarlen = c_nctlen(cdtype, crcode)

 rcode = crcode 
 nvarlen = cnvarlen

 End Function nctlen
! ------------------------------- ncclos ------------------------------------- 
 Subroutine ncclos(ncid, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid 
 Integer, Intent(OUT) :: rcode

 Integer(KIND=C_INT) :: crcode, cncid

 cncid   = ncid
 crcode  = 0
 rcode   = 0
 
 Call c_ncclos(cncid, crcode)

 rcode = crcode 

 End Subroutine ncclos 
! ------------------------------- ncredf ------------------------------------- 
 Subroutine ncredf(ncid, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid 
 Integer, Intent(OUT) :: rcode

 Integer(KIND=C_INT) :: crcode, cncid

 cncid   = ncid
 crcode  = 0
 rcode   = 0
 
 Call c_ncredf(cncid, crcode)
 rcode = crcode 

 End Subroutine ncredf
! ------------------------------- ncendf -------------------------------------- 
 Subroutine ncendf(ncid, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid 
 Integer, Intent(OUT) :: rcode

 Integer(KIND=C_INT) :: crcode, cncid

 cncid   = ncid
 crcode  = 0
 rcode   = 0
 
 Call c_ncendf(cncid, crcode)

 rcode = crcode 

 End Subroutine ncendf
! ------------------------------- ncinq --------------------------------------- 
 Subroutine ncinq(ncid, ndims, nvars, natts, recdim, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid 
 Integer, Intent(OUT) :: ndims, nvars, natts, recdim, rcode

 Integer(KIND=C_INT) :: crcode, cncid, cndims, cnvars, cnatts, crecdim

 cncid   = ncid
 crcode  = 0
 rcode   = 0
 cndims  = 0
 cnvars  = 0
 cnatts  = 0
 ndims   = 0
 nvars   = 0
 natts   = 0
  
 Call c_ncinq(cncid, cndims, cnvars, cnatts, crecdim, crcode)

 ndims = cndims
 nvars = cnvars
 natts = cnatts
 If (crecdim == -1) Then ! no unlimited dimension
   recdim = -1
 Else
   recdim = crecdim + 1  ! shift by plus one for FORTRAN
 EndIf 

 rcode = crcode 

 End Subroutine ncinq
! ------------------------------- ncsnc --------------------------------------- 
 Subroutine ncsnc(ncid, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid 
 Integer, Intent(OUT) :: rcode

 Integer(KIND=C_INT) :: crcode, cncid

 cncid   = ncid
 crcode  = 0
 rcode   = 0
 
 Call c_ncsnc(cncid, crcode)

 rcode = crcode 

 End Subroutine ncsnc
! ------------------------------- ncabor -------------------------------------- 
 Subroutine ncabor(ncid, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid 
 Integer, Intent(OUT) :: rcode

 Integer(KIND=C_INT) :: crcode, cncid

 cncid   = ncid
 crcode  = 0
 rcode   = 0
 
 Call c_ncabor(cncid, crcode)

 rcode = crcode 

 End Subroutine ncabor
! ------------------------------- ncdinq -------------------------------------- 
 Subroutine ncdinq(ncid, dimid, dimname, dimlen, rcode)
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_NAME
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(OUT) :: dimname
 Integer, Intent(IN) :: ncid, dimid
 Integer, Intent(OUT) :: dimlen, rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, crcode, cdimlen, cdimid
 Character(LEN=(NC_MAX_NAME+1)) :: cdimname

 cncid = ncid
 cdimid = dimid - 1
 crcode  = 0
 rcode   = 0
 cdimlen = 0
 dimname = REPEAT(" ", LEN(dimname))
 cdimname = REPEAT(" ", LEN(cdimname))
 
 Call c_ncdinq(cncid, cdimid, cdimname, cdimlen, crcode)

! check for a null character on end of cdimname

 ilen = LEN_TRIM(cdimname)
 inull = SCAN(cdimname, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
 dimname(1:ilen) = cdimname(1:ilen)
 dimlen = cdimlen
 rcode = crcode

 End Subroutine ncdinq
! ------------------------------- ncdren -------------------------------------- 
 Subroutine ncdren(ncid, dimid, dimname, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: dimname
 Integer, Intent(IN) :: ncid, dimid
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, crcode, cdimid

 cncid = ncid
 cdimid = dimid - 1
 crcode  = 0
 rcode   = 0

! check for a null character on end of dimname

 ilen = LEN_TRIM(dimname)
 inull = SCAN(dimname, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)

 Call c_ncdren(cncid, cdimid, dimname(1:ilen)//C_NULL_CHAR, crcode)

 rcode = crcode 

 End Subroutine ncdren
! ------------------------------- ncvinq -------------------------------------- 
 Subroutine ncvinq(ncid, varid, varname, vartype, nvdims, vdims, &
                   nvatts, rcode)
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_MAX_NAME
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(INOUT) :: varname
 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(OUT) :: vartype, nvdims, nvatts, rcode
 Integer, Intent(INOUT) :: vdims(*)

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cvartype, cnvdims, &
                        cnvatts, cstatus
 Integer(KIND=C_INT) :: cvdims(NC_MAX_DIMS)
 Character(LEN=NC_MAX_NAME+1) :: cvarname

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cvdims = 0
 cvdims  = 0
 vartype = 0
 nvdims = 0
 nvatts = 0
 cnvdims = 0
 cvdims = 0
 cnvatts = 0
 cvartype = 0
 varname = REPEAT(" ", LEN(varname))
 cvarname = REPEAT(" ", LEN(cvarname))

 Call c_ncvinq(cncid, cvarid, cvarname, cvartype, cnvdims, cvdims, cnvatts, &
               crcode)      

 nvdims  = cnvdims
 vartype = cvartype
 nvatts  = cnvatts 
 rcode = crcode 

! strip C null character from cvarname
   
 ilen = LEN_TRIM(cvarname)
 inull = SCAN(cvarname, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
 varname(1:ilen) = cvarname(1:ilen)

! convert C dimids to FORTRAN order and rank
! Replaces call to c2f_dimids in C code

 If (nvdims > 0) Then
   vdims(1:nvdims) = cvdims(nvdims:1:-1) + 1
 End If

 End Subroutine ncvinq
! ------------------------------- ncvpt1 -------------------------------------- 
 Subroutine ncvpt1(ncid, varid, mindex, values, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: mindex(*), values(*)
 Integer, Intent(OUT) :: rcode

 Integer :: ndims
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cmindex(NC_MAX_DIMS)
 Type(C_PTR) :: cmindexptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cmindex = 0
 cndims = 0
 ndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cmindexptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords in C code
   If (ndims > 0) Then
     cmindex(1:ndims) = mindex(ndims:1:-1) - 1
   Endif
   cmindexptr = C_LOC(cmindex)
 Endif

 Call c_ncvpt1(cncid, cvarid, cmindexptr, values, crcode)

 rcode = crcode

 End Subroutine ncvpt1
! ------------------------------- ncvp1c -------------------------------------- 
 Subroutine ncvp1c(ncid, varid, mindex, strings, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: mindex(*)
 Character(LEN=*), Intent(IN) :: strings
 Integer, Intent(OUT) :: rcode

 Integer :: ndims
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cmindex(NC_MAX_DIMS)
 Type(C_PTR) :: cmindexptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cmindex = 0
 cndims = 0
 ndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cmindexptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords in C code
   If (ndims > 0) Then
     cmindex(1:ndims) = mindex(ndims:1:-1) - 1
   Endif
   cmindexptr = C_LOC(cmindex)
 Endif

 Call c_ncvp1c(cncid, cvarid, cmindexptr, strings, crcode)

 rcode = crcode

 End Subroutine ncvp1c
! ------------------------------- ncvpt --------------------------------------- 
 Subroutine ncvpt(ncid, varid, start, counts, values, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: start(*), counts(*), values(*)
 Integer, Intent(OUT) :: rcode

 Integer :: ndims
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Type(C_PTR) :: cstartptr, ccountsptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
   Endif
   cstartptr  = C_LOC(cstart)
   ccountsptr = C_LOC(ccounts) 
 Endif

 Call c_ncvpt(cncid, cvarid, cstartptr, ccountsptr, values, crcode)

 rcode = crcode

 End Subroutine ncvpt
! ------------------------------- ncvptc--------------------------------------- 
 Subroutine ncvptc(ncid, varid, start, counts, strings, lenstr, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid, lenstr
 Integer, Intent(IN) :: start(*), counts(*)
 Character(LEN=*) :: strings
 Integer, Intent(OUT) :: rcode

 Integer :: ndims
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims, clenstr
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Type(C_PTR) :: cstartptr, ccountsptr

 cncid = ncid
 cvarid = varid - 1
 clenstr = lenstr
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
   Endif
   cstartptr  = C_LOC(cstart)
   ccountsptr = C_LOC(ccounts)
 Endif

 Call c_ncvptc(cncid, cvarid, cstartptr, ccountsptr, strings(1:lenstr),&
               clenstr, crcode)

 rcode = crcode

 End Subroutine ncvptc
! ------------------------------- ncvptg -------------------------------------- 
 Subroutine ncvptg(ncid, varid, start, counts, strides, imap, values, &
                   rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: start(*), counts(*), values(*), strides(*), imap(*)
 Integer, Intent(OUT) :: rcode

 Integer :: ndims, inullp
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Integer(KIND=C_PTRDIFF_T), TARGET :: cstrides(NC_MAX_DIMS), cimap(NC_MAX_DIMS)
                           
 Type(C_PTR) :: cstartptr, ccountsptr, cimapptr, cstridesptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0
 inullp = 0

 Call convert_v2_imap(cncid, cvarid, imap, cimap, inullp)
 
 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 cstridesptr = C_NULL_PTR
 cimapptr =  C_LOC(cimap)
 If (inullp /= 0) cimapptr = C_NULL_PTR 
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
     cstrides(1:ndims) = strides(ndims:1:-1) - 1
   Endif
   cstartptr   = C_LOC(cstart)
   ccountsptr  = C_LOC(ccounts)
   cstridesptr = C_LOC(cstrides)
 Endif

 Call c_ncvptg(cncid, cvarid, cstartptr, ccountsptr, cstridesptr, &
               cimapptr, values, crcode)

 rcode = crcode

 End Subroutine ncvptg
! ------------------------------- ncvpgc -------------------------------------- 
 Subroutine ncvpgc(ncid, varid, start, counts, strides, imap, string, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: start(*), counts(*), strides(*), imap(*)
 Character(LEN=*), Intent(IN) :: string
 Integer, Intent(OUT) :: rcode

 Integer :: ndims, inullp
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Integer(KIND=C_PTRDIFF_T), TARGET ::cstrides(NC_MAX_DIMS), cimap(NC_MAX_DIMS)
                           
 Type(C_PTR) :: cstartptr, ccountsptr, cstridesptr, cimapptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0
 inullp = 0

 Call convert_v2_imap(cncid, cvarid, imap, cimap, inullp)
 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 cstridesptr = C_NULL_PTR
 cimapptr    = C_LOC(cimap)
 If (inullp /= 0) cimapptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
     cstrides(1:ndims) = strides(ndims:1:-1) - 1
   Endif
   cstartptr = C_LOC(cstart)
   ccountsptr = C_LOC(ccounts)
   cstridesptr = C_LOC(cstrides)
 Endif

 Call c_ncvpgc(cncid, cvarid, cstartptr, ccountsptr, cstridesptr, &
               cimapptr, string, crcode)

 rcode = crcode

 End Subroutine ncvpgc
! ------------------------------- ncvgt1 -------------------------------------- 
 Subroutine ncvgt1(ncid, varid, mindex, values, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN)    :: ncid, varid
 Integer, Intent(IN)    :: mindex(*)
 Integer, Intent(INOUT) :: values(*)
 Integer, Intent(OUT)   :: rcode

 Integer :: ndims
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cmindex(NC_MAX_DIMS)
 Type(C_PTR) :: cmindexptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cmindex = 0
 cndims = 0
 ndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cmindexptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords in C code
   If (ndims > 0) Then
     cmindex(1:ndims) = mindex(ndims:1:-1) - 1
   Endif
   cmindexptr = C_LOC(cmindex)
 Endif

 Call c_ncvgt1(cncid, cvarid, cmindexptr, values, crcode)

 rcode = crcode

 End Subroutine ncvgt1
! ------------------------------- ncvg1c -------------------------------------- 
 Subroutine ncvg1c(ncid, varid, mindex, string, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN)    :: ncid, varid
 Integer, Intent(IN)    :: mindex(*)
 Character(LEN=*), Intent(INOUT) :: string
 Integer, Intent(OUT)   :: rcode

 Integer :: ndims
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cmindex(NC_MAX_DIMS)
 Type(C_PTR) :: cmindexptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cmindex = 0
 cndims = 0
 ndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cmindexptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords in C code
   If (ndims > 0) Then
     cmindex(1:ndims) = mindex(ndims:1:-1) - 1
   Endif
   cmindexptr = C_LOC(cmindex)
 Endif

 Call c_ncvg1c(cncid, cvarid, cmindexptr, string, crcode)

 rcode = crcode

 End Subroutine ncvg1c
! ------------------------------- ncvgt --------------------------------------- 
 Subroutine ncvgt(ncid, varid, start, counts, values, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: start(*), counts(*)
 Integer, Intent(INOUT) :: values(*)
 Integer, Intent(OUT) :: rcode

 Integer :: ndims
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Type(C_PTR) :: cstartptr, ccountsptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
   Endif
   cstartptr = C_LOC(cstart)
   ccountsptr = C_LOC(ccounts)
 Endif

 Call c_ncvgt(cncid, cvarid, cstartptr, ccountsptr, values, crcode)

 rcode = crcode

 End Subroutine ncvgt
! ------------------------------- ncvgtc -------------------------------------- 
 Subroutine ncvgtc(ncid, varid, start, counts, string, lenstr, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid, lenstr
 Integer, Intent(IN) :: start(*), counts(*)
 Character(LEN=*), Intent(INOUT) :: string
 Integer, Intent(OUT) :: rcode

 Integer :: ndims, slen
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims, clenstr
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Type(C_PTR) :: cstartptr, ccountsptr
 Character(LEN=lenstr+1) :: cstring

 cncid = ncid
 cvarid = varid - 1
 clenstr = lenstr
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0
 string = REPEAT(" ", LEN(string))
 cstring = REPEAT(" ", LEN(cstring))

 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
   Endif
   cstartptr = C_LOC(cstart)
   ccountsptr = C_LOC(ccounts)
 Endif

 Call c_ncvgtc(cncid, cvarid, cstartptr, ccountsptr, cstring, clenstr, crcode)

 If (LEN(string) >= lenstr) Then
   string(1:lenstr) = cstring(1:lenstr)
 Else
   slen = LEN(string)
   string(1:slen) = cstring(1:slen)
 EndIf
 rcode = crcode

 End Subroutine ncvgtc
! ------------------------------- ncvgtg -------------------------------------- 
 Subroutine ncvgtg(ncid, varid, start, counts, strides, imap, values, &
                   rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: start(*), counts(*), strides(*), imap(*)
 Integer, Intent(INOUT) :: values(*)
 Integer, Intent(OUT) :: rcode

 Integer :: ndims, inullp
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Integer(KIND=C_PTRDIFF_T), TARGET :: cstrides(NC_MAX_DIMS), cimap(NC_MAX_DIMS)
                           
 Type(C_PTR) :: cstartptr, ccountsptr, cimapptr, cstridesptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0
 inullp = 0

 Call convert_v2_imap(cncid, cvarid, imap, cimap, inullp)
 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 cstridesptr = C_NULL_PTR
 cimapptr = C_LOC(cimap)
 If (inullp /= 0) cimapptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
     cstrides(1:ndims) = strides(ndims:1:-1) - 1
   Endif
   cstartptr = C_LOC(cstart)
   ccountsptr = C_LOC(ccounts)
   cstridesptr = C_LOC(cstrides)
 Endif

 Call c_ncvgtg(cncid, cvarid, cstartptr, ccountsptr, cstridesptr, &
               cimapptr, values, crcode)

 rcode = crcode

 End Subroutine ncvgtg
! ------------------------------- ncvggc -------------------------------------- 
 Subroutine ncvggc(ncid, varid, start, counts, strides, imap, string, rcode) 
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_DIMS, NC_NOERR, nc_inq_varndims
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(IN) :: start(*), counts(*), strides(*), imap(*)
 Character(LEN=*), Intent(INOUT) :: string
 Integer, Intent(OUT) :: rcode

 Integer :: ndims, inullp,slen
 Integer(KIND=C_INT) :: cncid, crcode, cvarid, cstatus, cndims
 Integer(KIND=C_SIZE_T), TARGET :: cstart(NC_MAX_DIMS), ccounts(NC_MAX_DIMS)
 Integer(KIND=C_PTRDIFF_T), TARGET ::cstrides(NC_MAX_DIMS), cimap(NC_MAX_DIMS)
 Character(LEN=(LEN(string)+1)) :: cstring
 Type(C_PTR) :: cstartptr, ccountsptr, cstridesptr, cimapptr

 cncid = ncid
 cvarid = varid - 1
 crcode  = 0
 rcode   = 0
 cstart = 0
 ccounts = 0
 cndims = 0
 ndims = 0
 inullp = 0
 string = REPEAT(" ", LEN(string))
 cstring = REPEAT(" ", LEN(cstring)) 

 Call convert_v2_imap(cncid, cvarid, imap, cimap, inullp)
 cstatus = nc_inq_varndims(cncid, cvarid, cndims)

 cstartptr  = C_NULL_PTR
 ccountsptr = C_NULL_PTR
 cstridesptr = C_NULL_PTR
 cimapptr = C_LOC(cimap)
 If (inullp /= 0) cimapptr = C_NULL_PTR
 ndims = cndims 
 If (cstatus == NC_NOERR) Then ! mimic f2c_coords, etc. in C code
   If (ndims > 0) Then
     cstart(1:ndims) = start(ndims:1:-1) - 1
     ccounts(1:ndims) = counts(ndims:1:-1)
     cstrides(1:ndims) = strides(ndims:1:-1) - 1
   Endif
   cstartptr = C_LOC(cstart)
   ccountsptr = C_LOC(ccounts)
   cstridesptr = C_LOC(cstrides)
 Endif

 Call c_ncvggc(cncid, cvarid, cstartptr, ccountsptr, cstridesptr, &
               cimapptr, cstring, crcode)

 slen = LEN(string)
 string(1:slen) = cstring(1:slen)
 rcode = crcode

 End Subroutine ncvggc
!-------------------------------- ncvren --------------------------------------
 Subroutine ncvren(ncid, varid, newnam, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: newnam
 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, crcode

 cncid = ncid
 cvarid = varid - 1
 rcode = 0

! check for a null character on end of newnam

 ilen = LEN_TRIM(newnam)
 inull = SCAN(newnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncvren(cncid, cvarid, newnam(1:ilen)//C_NULL_CHAR, crcode )

 rcode = crcode 

 End Subroutine ncvren
!-------------------------------- ncapt ---------------------------------------
 Subroutine ncapt(ncid, varid, attnam, attype, attlen, value, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam
 Integer, Intent(IN) :: ncid, varid, attype, attlen
 Integer, Intent(IN) :: value(*)
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, cattype, crcode
 Integer(KIND=C_SIZE_T) :: cattlen

 cncid = ncid
 cvarid = varid - 1
 cattype = attype
 cattlen = attlen
 rcode = 0

! check for a null character on end of attname

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncapt(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR, cattype, &
             cattlen, value, crcode )

 rcode = crcode 

 End Subroutine ncapt
!-------------------------------- ncaptc ------------------------------------
 Subroutine ncaptc(ncid, varid, attnam, attype, lenstr, string, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam
 Integer, Intent(IN) :: ncid, varid, attype, lenstr
 Character(LEN=*), Intent(IN) :: string 
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, cattype, crcode
 Integer(KIND=C_SIZE_T) :: clenstr

 cncid = ncid
 cvarid = varid - 1
 cattype = attype
 clenstr = lenstr
 rcode = 0

! check for a null character on end of attname

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncaptc(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR, cattype, &
               clenstr, string, crcode )

 rcode = crcode 

 End Subroutine ncaptc
!-------------------------------- ncainq --------------------------------------
 Subroutine ncainq(ncid, varid, attnam, attype, attlen, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam
 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(OUT) :: attype, attlen, rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, cattype, crcode, cattlen

 cncid = ncid
 cvarid = varid - 1
 cattype = 0
 cattlen = 0
 rcode = 0

! check for a null character on end of attnam

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncainq(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR, cattype, &
             cattlen, crcode )

 attype = cattype
 attlen = cattlen
 rcode = crcode

 End Subroutine ncainq
!-------------------------------- ncagt ---------------------------------------
 Subroutine ncagt(ncid, varid, attnam, values, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam
 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(INOUT) :: values(*)
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, crcode

 cncid = ncid
 cvarid = varid - 1
 rcode = 0

! check for a null character on end of attnam

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncagt(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR, values, crcode)

 rcode = crcode

 End Subroutine ncagt
!-------------------------------- ncagtc --------------------------------------
 Subroutine ncagtc(ncid, varid, attnam, string, lenstr, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam
 Integer, Intent(IN) :: ncid, varid, lenstr
 Character(LEN=*), Intent(INOUT) :: string 
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, crcode
 Character(LEN=(lenstr+1)) :: cstring

 cncid = ncid
 cvarid = varid - 1
 rcode = 0
 string = REPEAT(" ", LEN(string))
 cstring = REPEAT(" ", LEN(cstring))

! check for a null character on end of attnam

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncagtc(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR, cstring, lenstr, &
               crcode)

 string(1:lenstr) = cstring(1:lenstr)
 rcode = crcode

 End Subroutine ncagtc
!-------------------------------- ncacpy --------------------------------------
 Subroutine ncacpy(ncid, varid, attnam, outcdf, outvar, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam
 Integer, Intent(IN) :: ncid, varid, outcdf, outvar
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, coutcdf, coutvar, crcode

 cncid = ncid
 cvarid = varid - 1
 coutcdf = outcdf
 coutvar = outvar-1
 rcode = 0

! check for a null character on end of attnam

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncacpy(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR, coutcdf, &
               coutvar, crcode)

 rcode = crcode

 End Subroutine ncacpy
!-------------------------------- ncanam --------------------------------------
 Subroutine ncanam(ncid, varid, attnum, attnam, rcode)
!
 USE netcdf_nc_interfaces, ONLY: NC_MAX_NAME
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(INOUT) :: attnam
 Integer, Intent(IN) :: ncid, varid, attnum
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen
 Integer(KIND=C_INT) :: cncid, cvarid, cattnum, crcode
 Character(LEN=NC_MAX_NAME+1) :: cattnam
 cncid = ncid
 cvarid = varid - 1
 cattnum = attnum - 1
 rcode = 0
 attnam = REPEAT(" ", LEN(attnam))
 cattnam = REPEAT(" ", LEN(cattnam))

 Call c_ncanam(cncid, cvarid, cattnum, cattnam, crcode)

! check for a null character on end of cattnam

 ilen = LEN_TRIM(cattnam)
 inull = SCAN(cattnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
 attnam(1:ilen) = cattnam(1:ilen)
 rcode = crcode

 End Subroutine ncanam
!-------------------------------- ncaren --------------------------------------
 Subroutine ncaren(ncid, varid, attnam, newnam, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam, newnam
 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen, ilen2
 Integer(KIND=C_INT) :: cncid, cvarid, crcode

 cncid = ncid
 cvarid = varid - 1
 rcode = 0

! check for a null character on end of attnam and newnam

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 ilen2 = LEN_TRIM(newnam)
 inull = SCAN(newnam, C_NULL_CHAR)
 If (inull /= 0) ilen2 = inull-1
 ilen2 = MAX(1,ilen2)

 Call c_ncaren(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR,  &
               newnam(1:ilen2)//C_NULL_CHAR, crcode)

 rcode = crcode

 End Subroutine ncaren
!-------------------------------- ncadel --------------------------------------
 Subroutine ncadel(ncid, varid, attnam, rcode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE

 Character(LEN=*), Intent(IN) :: attnam
 Integer, Intent(IN) :: ncid, varid
 Integer, Intent(OUT) :: rcode

 Integer :: inull, ilen, ilen2
 Integer(KIND=C_INT) :: cncid, cvarid, crcode

 cncid = ncid
 cvarid = varid - 1
 rcode = 0

! check for a null character on end of attnam

 ilen = LEN_TRIM(attnam)
 inull = SCAN(attnam, C_NULL_CHAR)
 If (inull /= 0) ilen = inull-1
 ilen = MAX(1,ilen)
  
 Call c_ncadel(cncid, cvarid, attnam(1:ilen)//C_NULL_CHAR,crcode)

 rcode = crcode

 End Subroutine ncadel
!-------------------------------- ncsfil --------------------------------------
 Function ncsfil(ncid, fillmode, rcode) RESULT(currentmode)
!
 USE netcdf_fortv2_c_interfaces
!
 Implicit NONE
 Integer, Intent(IN) :: ncid, fillmode
 Integer, Intent(OUT) :: rcode
 Integer :: currentmode

 Integer(KIND=C_INT) :: cncid, cfillmode, crcode, cstatus  

 cncid = ncid
 cfillmode = fillmode

 cstatus = c_ncsfil(cncid, cfillmode, crcode)
 rcode = crcode
 currentmode = cstatus

 End Function ncsfil
#endif
#endif
