#ifdef Fortran2003
! ------------ Routines to create/open/close/redefine netcdf files ------------ 

! Replacement for fort-control.c

! Written by: Richard Weed
!             Engineering Research Center
!             Mississippi State University
!             rweed@erc.msstate.edu

! Version 1.: Sept. 2005 - Initial Cray X1 version
! Version 2.: May,  2006 - Updated to support g95
         
!-------------------------------- nf_create --------------------------------
 Function nf_create(path, cmode, ncid) RESULT (status)

! Creates a new NetCDF file given a file name and a creation mode and returns
! the file id and a status flag

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer, Intent(IN) :: cmode
 Integer, Intent(OUT) :: ncid
 
 Integer :: status

 Integer(KIND=C_INT) :: ccmode, cncid, cstatus, inull, ie

 ccmode = cmode
 cncid = 0
 
! Check for C null character on path. We will always add a null
! char so we don't need a second one
 
 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie = inull-1

! Call nc_create to create file

 cstatus = nc_create(path(1:ie)//C_NULL_CHAR, ccmode, cncid)
 
 ncid   = cncid 
 status = cstatus

 End Function nf_create
!-------------------------------- nf__create -------------------------------
 Function nf__create(path, cmode, initialsz, chunksizehintp, ncid) &
                        RESULT(status)

! Creates a new NetCDF file and returns the file id and a status flag
! This is an alternate form of nf_create that allows user to input
! two additional tuning parameters

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer, Intent(IN) :: cmode, initialsz, chunksizehintp
 Integer, Intent(OUT) :: ncid
 
 Integer :: status

 Integer(KIND=C_INT) :: ccmode, cncid, cstatus
 Integer(KIND=C_SIZE_T) :: cinit, cchunk
 Integer :: inull, ie

 ccmode = cmode
 cchunk = chunksizehintp
 cinit  = initialsz
 cncid = 0
 
! Check for C null character on path. We will always add a null
! char so we don't need a second one
 
 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie = inull-1
 ie = MAX(1,ie)

! Call nc_create to create file

 cstatus = nc__create(path(1:ie)//C_NULL_CHAR, ccmode, cinit, cchunk, cncid)
 
 ncid   = cncid 
 status = cstatus

 End Function nf__create
!-------------------------------- nf__create_mp ------------------------------
 Function nf__create_mp(path, cmode, initialsz, basepe, chunksizehintp, ncid) &
                        RESULT(status)

! Creates a new NetCDF file and returns the file id and a status flag
! This is an alternate form of nf__create for shared memory MPP systems 
! two additional tuning parameters

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer, Intent(IN) :: cmode, initialsz, chunksizehintp, basepe
 Integer, Intent(OUT) :: ncid
 
 Integer :: status

 Integer(KIND=C_INT) :: ccmode, cncid, cstatus
 Integer(KIND=C_INT), TARGET :: cbasepe
 Integer(KIND=C_SIZE_T) :: cinit, cchunk
 Type(C_PTR) :: cbasepeptr
 Integer :: inull, ie

 ccmode = cmode
 cchunk = chunksizehintp
 cinit  = initialsz
 cncid = 0
 cbasepe = basepe
 cbasepeptr = C_LOC(cbasepe)

! Check for C null character on path. We will always add a null
! char so we don't need a second one

 
 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie = inull-1
 ie = MAX(1,ie)

! Call nc_create_mp to create file for base pe

 cstatus = nc__create_mp(path(1:ie)//C_NULL_CHAR, ccmode, cinit, cbasepeptr, &
                         cchunk, cncid)
 
 ncid   = cncid 
 status = cstatus

 End Function nf__create_mp
!-------------------------------- nf_open ----------------------------------
 Function nf_open(path, mode, ncid) RESULT (status)

! Open an existing NetCDF file and return file id and a status flag

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer, Intent(IN) :: mode
 Integer, Intent(INOUT) :: ncid
 
 Integer :: status

 Integer(KIND=C_INT) :: cmode, cncid, cstatus, inull, ie

 cmode = mode
 cncid = 0
 
! Check for C null character on path. We will always add a null
! char so we don't need a second one
 
 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie = inull-1

! Call nc_create to create file

 cstatus = nc_open(path(1:ie)//C_NULL_CHAR, cmode, cncid)
 
 ncid   = cncid
 status = cstatus

 End Function nf_open
!-------------------------------- nf__open ---------------------------------
 Function nf__open(path, mode, chunksizehintp, ncid) RESULT (status)

! Open an existing NetCDF file and return file id and a status flag
! Alternate form of nf_open with extra tuning parameter

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer, Intent(IN) :: mode, chunksizehintp
 Integer, Intent(INOUT) :: ncid
 
 Integer :: status

 Integer(KIND=C_INT) :: cmode, cncid, cstatus
 Integer(KIND=C_SIZE_T) :: cchunk
 Integer :: inull, ie

 cmode = mode
 cchunk = chunksizehintp
 cncid = 0
 
! Check for C null character in path. A null character is always added
! before we pass path to C we don't need a second one
 
 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie = inull-1

! Call nc_create to create file

 cstatus = nc__open(path(1:ie)//C_NULL_CHAR, cmode, cchunk, cncid)
 
 ncid   = cncid
 status = cstatus

 End Function nf__open
!-------------------------------- nf__open_mp --------------------------------
 Function nf__open_mp(path, mode, basepe, chunksizehintp, ncid) RESULT (status)

! Open an existing NetCDF file and return file id and a status flag
! Alternate form of nf__open with parameter to designate basepe on
! shared memory MPP systems. 

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer, Intent(IN) :: mode, chunksizehintp, basepe
 Integer, Intent(INOUT) :: ncid
 
 Integer :: status

 Integer(KIND=C_INT) :: cmode, cncid, cstatus
 Integer(KIND=C_INT), TARGET :: cbasepe
 Integer(KIND=C_SIZE_T) :: cchunk
 Type(C_PTR) :: cbasepeptr
 Integer :: inull, ie

 cmode = mode
 cchunk = chunksizehintp
 cncid = 0
 cbasepe = basepe
 cbasepeptr = C_LOC(cbasepe)
 
! Check for C null character in path. A null character is always added
! before we pass path to C we don't need a second one
 
 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie = inull-1

! Call nc_create to create file

 cstatus = nc__open_mp(path(1:ie)//C_NULL_CHAR, cmode, cbasepeptr, cchunk, &
                       cncid)
 
 ncid   = cncid
 status = cstatus

 End Function nf__open_mp
!-------------------------------- nf_set_fill ------------------------------
 Function nf_set_fill(ncid, fillmode, old_mode) RESULT(status)
 
! Sets fill mode for given netcdf file returns old mode if present

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid, fillmode
 Integer, Intent(OUT) :: old_mode

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cfill, coldmode, cstatus

 cncid = ncid
 cfill = fillmode
 coldmode=0

 cstatus = nc_set_fill(cncid, cfill, coldmode)

 old_mode = coldmode
 status = cstatus

 End Function nf_set_fill
!-------------------------------- nf_set_default_format --------------------
 Function nf_set_default_format(newform, old_format) RESULT(status)
 
! Sets new default data format. Used to toggle between 64 bit offset and
! classic mode 

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: newform 
 Integer, Intent(OUT) :: old_format

 Integer :: status

 Integer(KIND=C_INT) :: cnew, cold, cstatus

 cnew = newform

 cstatus = nc_set_default_format(cnew,cold)
 old_format = cold

 status = cstatus

 End Function nf_set_default_format
!-------------------------------- nf_redef ---------------------------------
 Function nf_redef(ncid) RESULT(status)
 
! Re-Enter definition mode for NetCDF file id ncid 

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus

 cncid = ncid

 cstatus = nc_redef(cncid)

 status = cstatus

 End Function nf_redef
!-------------------------------- nf_enddef --------------------------------
 Function nf_enddef(ncid) RESULT(status)
 
! Exit definition mode for NetCDF file id ncid

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus

 cncid = ncid

 cstatus = nc_enddef(cncid)

 status = cstatus

 End Function nf_enddef
!-------------------------------- nf__enddef -------------------------------
 Function nf__enddef(ncid, h_minfree, v_align, v_minfree, r_align) &
                        RESULT(status)
 
! Exit definition mode for NetCDF file id ncid. Alternate version
! with additional tuning parameters

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid, h_minfree, v_align, v_minfree, r_align

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus
 Integer(KIND=C_SIZE_T) :: chminfree, cvalign, cvminfree, cralign

 cncid = ncid
 chminfree = h_minfree
 cvalign   = v_align
 cvminfree = v_minfree
 cralign   = r_align

 cstatus = nc__enddef(cncid, chminfree, cvalign, cvminfree, cralign)

 status = cstatus

 End Function nf__enddef
!-------------------------------- nf_sync ----------------------------------
 Function nf_sync(ncid) RESULT(status)
 
! synch up all open NetCDF files 

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus

 cncid = ncid

 cstatus = nc_sync(cncid)

 status = cstatus

 End Function nf_sync
!-------------------------------- nf_abort ---------------------------------
 Function nf_abort(ncid) RESULT(status)
 
! Abort netCDF file creation and exit 

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus

 cncid = ncid

 cstatus = nc_abort(cncid)

 status = cstatus

 End Function nf_abort
!-------------------------------- nf_close ---------------------------------
 Function nf_close(ncid) RESULT(status)
 
! Close netCDF file id ncid 

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus

 cncid = ncid

 cstatus = nc_close(cncid)

 status = cstatus

 End Function nf_close
!-------------------------------- nf_delete --------------------------------
 Function nf_delete(path) RESULT(status)
 
! Delete netCDF file id ncid 

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer :: status

 Integer(KIND=C_INT) :: cstatus
 Integer :: ie, inull

 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie=inull-1

 cstatus = nc_delete(path(1:ie)//C_NULL_CHAR)

 status = cstatus

 End Function nf_delete
!-------------------------------- nf_delete_mp -------------------------------
 Function nf_delete_mp(path, pe) RESULT(status)
 
! Delete netCDF file id ncid. Alternate form of nf_delete for shared memory
! MPP systems.

 USE netcdf_nc_interfaces

 Implicit NONE

 Character(LEN=*), Intent(IN) :: path
 Integer, Intent(IN) :: pe
 Integer :: status

 Integer(KIND=C_INT) :: cstatus, cpe
 Integer :: ie, inull

 cpe = pe

 ie = LEN_TRIM(path)
 inull = SCAN(path, C_NULL_CHAR)
 If (inull /= 0) ie=inull-1

 cstatus = nc_delete_mp(path(1:ie)//C_NULL_CHAR, cpe)

 status = cstatus

 End Function nf_delete_mp
!-------------------------------- nf_set_base_pe ------------------------------
 Function nf_set_base_pe(ncid, pe) RESULT(status)

! Sets base pe number on shared memory MPP systems

 Use netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid, pe

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cpe, cstatus

 cncid = ncid
 cpe = pe

 cstatus = nc_set_base_pe(cncid, cpe)

 status = cstatus

 End Function nf_set_base_pe
!-------------------------------- nf_inq_base_pe ------------------------------
 Function nf_inq_base_pe(ncid, pe) RESULT(status)

! Gets previously set base pe number on shared memory MPP systems

 Use netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid
 Integer, Intent(OUT) :: pe

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cpe, cstatus

 cncid = ncid

 cstatus = nc_inq_base_pe(cncid, cpe)

 pe = cpe
 status = cstatus

 End Function nf_inq_base_pe
#endif 
