#ifdef Fortran2003
!------ Routines for defining and obtaining info about dataset variables ------

! Replacement for fort-genvar.c

! Written by: Richard Weed
!             Engineering Research Center
!             Mississippi State University
!             rweed@erc.msstate.edu

! Version 1.: Sept. 2005 - Initial Cray X1 version
! Version 2.: May   2006 - Updated to support g95
          
!-------------------------------- nf_def_var -------------------------------
 Function nf_def_var(ncid, name, xtype, nvdims, vdims, varid) RESULT (status)

! Define name, datatype, number of dimensions, and dimension ids for a
! dataset variable. Returns variable id

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)  :: ncid, xtype, nvdims
 Integer, Intent(IN)  :: vdims(*)
 Integer, Intent(OUT) :: varid
 Character(LEN=*), Intent(IN)  :: name

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cnvdims, cvarid, cstatus
 Integer(KIND=C_INT) :: cvdims(NC_MAX_DIMS)
 Integer :: inull, ie
! Type(nc_type) :: cxtype

 cncid = ncid
 cnvdims = nvdims

! Check if a C NULL character was appended to name in calling routine

 ie = LEN_TRIM(name)
 inull = SCAN(name, C_NULL_CHAR)
 if (inull /= 0) ie=inull-1
 ie = MAX(1,ie)

! Flip dimids to C order and subtract -1 to yield C ids prior
! to calling nc_def_var. Replaces C utility f2c_dimids

 cvdims = 0
 If (nvdims /= 0) Then 
   cvdims(1:nvdims) = vdims(nvdims:1:-1)-1
 EndIf

 cstatus = nc_def_var(cncid, name(1:ie)//C_NULL_CHAR, f2c_xtype(xtype), &
                     cnvdims, cvdims, cvarid)

! Add one to returned C varid to yield FORTRAN id

 varid = cvarid + 1

 status = cstatus

 End Function nf_def_var
!-------------------------------- nf_inq_varndims --------------------------
 Function nf_inq_varndims(ncid, varid, vndims) RESULT (status)

! Get variable dimension for a given varid from NetCDF dataset ncid

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)           :: ncid, varid
 Integer, Intent(OUT)          :: vndims

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cvarid, cvndims, cstatus

 cncid   = ncid
 cvarid  = varid - 1

 vndims = 0

 cstatus = nc_inq_varndims(cncid, cvarid, cvndims)

 vndims = cvndims
 status = cstatus

 End Function nf_inq_varndims
!-------------------------------- nf_inq_var ----------------------------
 Function nf_inq_var(ncid, varid, name, xtype, ndims, dimids, natts) &
                        RESULT (status)

! Get variable name, data type, dimension length, dimension ids, and
! number of attributes for a given varid 

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)  :: ncid, varid
 Character(LEN=*), Intent(OUT) :: name
 Integer, Intent(OUT) :: dimids(*)
 Integer, Intent(OUT) :: ndims, xtype, natts

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus, cndims, cvarid, cnatts
 Integer(KIND=C_INT) :: cdimids(NC_MAX_DIMS)
 Integer(KIND=C_INT) :: cxtype
! Type(nc_type) :: cxtype
 Integer :: ie, inull, nlen
 Character(LEN=NC_MAX_NAME+1) :: tmpname

 cncid = ncid
 cvarid = varid - 1 ! subtract -1 to yield cvarid

 tmpname = REPEAT(" ", LEN(tmpname))
 nlen = LEN(name)
 name = REPEAT(" ", nlen)

 cndims = 0
 dimids(1) = 0
 xtype = 0
 natts = 0
 ndims = 0

 cstatus = nc_inq_var(cncid, cvarid, tmpname, cxtype, cndims, cdimids, cnatts)

! If (cstatus == NC_NOERR) Then
   xtype = cxtype
   natts = cnatts
   ndims = cndims

! Check tmpname for a C null character and strip it and trailing blanks

   ie = LEN_TRIM(tmpname)
   inull = SCAN(tmpname,C_NULL_CHAR)
   If (inull /= 0) ie = inull-1
   ie = MIN(nlen, MAX(1,ie)) ! limit ie to >=1 and <= LEN(name)

! Set name to tmpname. tmpname string is truncated if name is not big 
! enough to hold tmpname

   name(1:ie) = tmpname(1:ie)

! Reverse order of cdimids and add one to yield FORTRAN id numbers
! Replaces c2f_dimids C utility
 
   If (ndims > 0) Then
     dimids(1:ndims) = cdimids(ndims:1:-1)+1
   EndIf
    
! EndIf

 status = cstatus

 End Function nf_inq_var
!-------------------------------- nf_inq_vardimid -----------------------
 Function nf_inq_vardimid(ncid, varid, dimids) RESULT (status)

! Get variable dimension id for a dimension name

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)  :: ncid, varid
 Integer, Intent(OUT) :: dimids(*)

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cstatus, cstat2, cndims, cvarid
 Integer(KIND=C_INT) :: cvdimids(NC_MAX_DIMS)
 Integer :: ndims

 cncid = ncid
 cvarid = varid - 1 ! subtract -1 to get C id number
 ndims = 0

 dimids(1) = 0
 cvdimids = 0
 cndims = 0

 cstat2  = nc_inq_varndims(cncid, cvarid, cndims)
 cstatus = nc_inq_vardimid(cncid, cvarid, cvdimids)

 
! Reverse order of cdimids and add 1 to yield FORTRAN id numbers
! Replaces c2f_dimids C utility
 
 If (cstat2 == 0 .AND. cstatus==0) Then
   ndims = cndims
   If (ndims > 0) Then    
     dimids(1:ndims) = cvdimids(ndims:1:-1)+1
   EndIf
 Else
   ndims = 0
 EndIf
 
 status = cstatus

 End Function nf_inq_vardimid
!-------------------------------- nf_inq_varid --------------------------
 Function nf_inq_varid(ncid, name, varid) RESULT (status)

! Get variable id for a variable name from NetCDF dataset ncid

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)  :: ncid
 Integer, Intent(OUT) :: varid
 Character(LEN=*), Intent(IN)  :: name

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cvarid, cstatus
 Integer :: inull, ie

 cncid = ncid

! Check name for a C NULL character added in calling routine

 ie = LEN_TRIM(name)
 inull = SCAN(name, C_NULL_CHAR)
 If (inull /= 0) ie = inull-1
 ie = MAX(1,ie)

 varid = 0
 cvarid = 0
 
 cstatus = nc_inq_varid(cncid, name(1:ie)//C_NULL_CHAR, cvarid)

 varid = cvarid + 1  ! add one to get Fortran id number
 status = cstatus

 End Function nf_inq_varid
!-------------------------------- nf_inq_varname ------------------------
 Function nf_inq_varname (ncid, varid, name) RESULT (status)

! Get variable name for a given varid from NetCDF dataset ncid

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)           :: ncid, varid
 Character(LEN=*), Intent(OUT)  :: name

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cvarid, cstatus
 Integer :: inull, ie, nlen
 Character(LEN=NC_MAX_NAME+1) :: tmpname 

 cncid   = ncid
 cvarid  = varid - 1 ! subtract one to get C id number

 tmpname = REPEAT(" ", LEN(tmpname))
 nlen = LEN(name)
 name    = REPEAT(" ", nlen)

! Get tmpname from C interface

 cstatus = nc_inq_varname(cncid, cvarid, tmpname)

! Find first C null character in tmpname if present and set end of string

 ie = LEN_TRIM(tmpname)
 inull = SCAN(tmpname, C_NULL_CHAR)
 if (inull /= 0) ie=inull-1
 ie = MIN(nlen, MAX(1,ie)) ! Limit ie to >=1 and <= LEN(name)

! Set name to tmpname. tmpname string is truncated if name is not big 
! enough to hold tmpname

 name(1:ie) = tmpname(1:ie)

 status = cstatus

 End Function nf_inq_varname
!-------------------------------- nf_inq_vartype -------------------------
 Function nf_inq_vartype(ncid, varid, xtype) RESULT(status)

! Inquire about numeric type of variable attributes for NetCDF file id ncid

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)  :: ncid, varid
 Integer, Intent(OUT) :: xtype

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cvarid, cstatus
 Integer(KIND=C_INT) :: cxtype
! TYPE(nc_type) :: cxtype

 cncid = ncid
 cvarid = varid - 1 ! Subtract one to get C id number

 xtype = 0

 cstatus = nc_inq_vartype(cncid, cvarid, cxtype)

 xtype = cxtype
 status = cstatus

 End Function nf_inq_vartype
!-------------------------------- nf_inq_varnatts -----------------------
 Function nf_inq_varnatts(ncid, varid, nvatts) RESULT(status)

! Inquire about number of variable attributes for NetCDF file id ncid

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)  :: ncid, varid
 Integer, Intent(OUT) :: nvatts

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cvarid, cnvatts, cstatus

 cncid  = ncid
 cvarid = varid - 1 ! subtract one to get C id number

 nvatts = 0

 cstatus = nc_inq_varnatts(cncid, cvarid, cnvatts)

 nvatts = cnvatts

 status = cstatus

 End Function nf_inq_varnatts
!-------------------------------- nf_rename_var -------------------------
 Function nf_rename_var(ncid, varid, name) RESULT (status)

! Rename dimension name for a given dimension id

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN)  :: ncid, varid 
 Character(LEN=*), Intent(IN)  :: name

 Integer :: status

 Integer(KIND=C_INT) :: cncid, cvarid, cstatus
 Integer :: inull, ie

 cncid  = ncid
 cvarid = varid - 1 ! Subtract one to get C id number

! Check name for a C NULL character added in calling routine

 ie = LEN_TRIM(name)
 inull = SCAN(name, C_NULL_CHAR)
 if (inull /= 0) ie=inull-1
 ie = MAX(1,ie)
 
 cstatus = nc_rename_var(cncid, cvarid, name(1:ie)//C_NULL_CHAR)

 status = cstatus

 End Function nf_rename_var
!-------------------------------- nf_copy_var ---------------------------
 Function nf_copy_var(ncid_in, varid, ncid_out) RESULT(status)

! Copies a given variable from dataset ncid_in to dataset ncid_out

 USE netcdf_nc_interfaces

 Implicit NONE

 Integer, Intent(IN) :: ncid_in, varid, ncid_out
 Integer :: status

 Integer(KIND=C_INT) :: cncidin, cvarid, cncidout, cstatus

 cncidin  = ncid_in
 cncidout = ncid_out
 cvarid = varid - 1 ! Subtract one to get C id number

 cstatus = nc_copy_var(cncidin, cvarid, cncidout)

 status = cstatus

 End Function nf_copy_var
#endif
