/*
Limitations: 
 - we don't handle "char" attributes vectors, if they even exist.
 - the data parser appears a bit flaky, esp. with text data.
*/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <malloc.h>
#include <netcdf.h>
#include <sys/stat.h>
#include "xmlparse.h"

#define	T_UNKNOWN		0

#define T_NETCDF		1

#define	T_DIM			2
#define	T_VAR			3
#define	T_ATT			4

#define	T_TYPE			5
#define	T_NAME			6
#define	T_SIZE			7 
#define	T_VALUE			8
#define	T_DATA			9
#define	T_RECORD		10

/* maximum number of values in attribute vector */
#define MAX_VALUES	10

/* maximum number of records */
#define MAX_RECORDS 100

/*
Information on each dimension.  We need to retain this information 
throughout the run of the program.
*/
struct DimInfo {
	char *id;
	char *name;
	long size;
	int dimid;
};


/*
Information on each attribute.  We only need to keep attributes tied 
to a variable for the lifetime of that variable, but for now we retain
it for debugging purposes.
*/
struct AttInfo {
	nc_type type;
	char *name;
	char *value;
};

/*
Information on each variable.  We only need to keep variables while the
element is in scope, but for now we retain it for debugging purposes.  
*/
struct VarInfo {
	nc_type type;
	char *name;
	int ndims;
	int dims[NC_MAX_VAR_DIMS];	/* index into diminfo[] */
	int record;					/* 0 = fixed-size, 1 = record */
	int varid;
	int natts;
	struct {					/* want 'union', but that corrupts data */
		signed char b;
		char c;
		short s;
		int i;
		long l;
		double d;
	} fillvalue;
	struct AttInfo *attinfo[NC_MAX_ATTRS];
	int nrecs;
	void *data[MAX_RECORDS];
};

/*
The overall state of what we know about the netcdf file.
*/
struct UserData {
	int depth;
	int ncid;
	char *name;
	int stack[10];

	size_t len, blen;
	char *buffer;

	int ndims;
	struct DimInfo *diminfo[NC_MAX_DIMS];

	int nvars;
	struct VarInfo *varinfo[NC_MAX_VARS];

	/* global attributes */
	int natts;
	struct AttInfo *attinfo[NC_MAX_ATTRS];

	/* 'current' pointers */
	struct VarInfo *cvar;
	struct AttInfo *catt;
} userData;


/*
*/
#define	NC_CHECK(s)		\
	{	\
	int ncerr = (s); \
	if (ncerr != 0) { \
		fprintf (stderr, "%s\n", nc_strerror (ncerr)); \
		exit (0); \
	 } \
	}

/*
Identify the element
we should use something like perf/gperf, but many people don't have 
those tools
*/
static int 
lookup_element (const char *name)
{
	int t;

	if (strcmp (name, "netcdf") == 0)
		t = T_NETCDF;
	else if (strcmp (name, "dim") == 0)
		t = T_DIM;
	else if (strcmp (name, "var") == 0)
		t = T_VAR;
	else if (strcmp (name, "att") == 0)
		t = T_ATT;
	else if (strcmp (name, "type") == 0)
		t = T_TYPE;
	else if (strcmp (name, "name") == 0)
		t = T_NAME;
	else if (strcmp (name, "size") == 0)
		t = T_SIZE;
	else if (strcmp (name, "value") == 0)
		t = T_VALUE;
	else if (strcmp (name, "data") == 0)
		t = T_DATA;
	else if (strcmp (name, "record") == 0)
		t = T_RECORD;
	else
		t = T_UNKNOWN;

	return t;
}

/*
Identify the data type
we should use something like perf/gperf, but many people don't have 
those tools
*/
static nc_type
lookup_type (const char *name)
{
	nc_type t;

	if (strcasecmp (name, "byte") == 0)
		t = NC_BYTE;
	else if (strcasecmp (name, "char") == 0)
		t = NC_CHAR;
	else if (strcasecmp (name, "short") == 0)
		t = NC_SHORT;
	else if (strcasecmp (name, "int") == 0)
		t = NC_INT;
	else if (strcasecmp (name, "long") == 0)
		t = NC_LONG;
	else if (strcasecmp (name, "float") == 0)
		t = NC_FLOAT;
	else if (strcasecmp (name, "double") == 0)
		t = NC_DOUBLE;
	else
		t = 0;

	return t;
}


/*
Put a single attribute
*/
static int put_attribute (int ncid, int varid, struct AttInfo *att)
{
	signed char b[MAX_VALUES];
	short sh[MAX_VALUES];
	int i[MAX_VALUES];
	long l[MAX_VALUES];
	float f[MAX_VALUES];
	double d[MAX_VALUES];

	char *s;
	int len;

	s = att->value;
	switch (att->type) {
	case NC_BYTE:
		for (len = 0; s && *s && len < MAX_VALUES; s = strchr (s, ',')) {
			if (*s == ',') s++;
			b[len++]= atoi (s);
		}
		NC_CHECK( nc_put_att_schar (ncid, varid, att->name, att->type, len, b) );
		break;

 	case NC_CHAR:
		for ( ; *s != '\0' && *s != '"'; s++)
			s++;
		if (*s)
			s++;
		for (len = 0; s[len] != '\0' && s[len] != '"'; len++)
			;
		NC_CHECK( nc_put_att_text (ncid, varid, att->name, len, s) );
		break;

 	case NC_SHORT:
		for (len = 0; s && *s && len < MAX_VALUES; s = strchr (s, ',')) {
			if (*s == ',') s++;
			sh[len++]= atoi (s);
		}
		NC_CHECK( nc_put_att_short (ncid, varid, att->name, att->type, len, sh) );
		break;

 	case NC_INT:
		for (len = 0; s && *s && len < MAX_VALUES; s = strchr (s, ',')) {
			if (*s == ',') s++;
			i[len++] = atoi (s);
		}
		NC_CHECK( nc_put_att_int (ncid, varid, att->name, att->type, len, i) );
		break;

#if (NC_INT != NC_LONG)
 	case NC_LONG:
		for (len = 0; s && *s && len < MAX_VALUES; s = strchr (s, ',')) {
			if (*s == ',') s++;
			l[len++] = atoi (s);
		}
		NC_CHECK( nc_put_att_long (ncid, varid, att->name, att->type, len, l) );
		break;
#endif

 	case NC_FLOAT:
		for (len = 0; s && *s && len < MAX_VALUES; s = strchr (s, ',')) {
			if (*s == ',') s++;
			f[len++] = atof (s);
		}
		NC_CHECK( nc_put_att_float (ncid, varid, att->name, att->type, 
			len, f) );
		break;

 	case NC_DOUBLE:
		for (len = 0; s && *s && len < MAX_VALUES; s = strchr (s, ',')) {
			if (*s == ',') s++;
			d[len++] = atof (s);
		}
		NC_CHECK( nc_put_att_double (ncid, varid, att->name, att->type, 
			len, d) );
		break;

	default:
		fprintf (stderr, 
			"%s, line %d: unhandled data type (%d) for attribute %s\n", 
			__FILE__, __LINE__, att->type, att->name);
		break;
	}
}


/*
Allocate and initialize 
*/
static void * init_data (struct UserData *userData, struct VarInfo *var)
{
	long i, nels;
	void *data;
	char *cdata, cfill;
	short *sdata, sfill;
	int *idata, ifill;
	long *ldata, lfill;
	float *fdata;
	double *ddata, dfill;

	nels = 1;
	for (i = 0; i < var->ndims; i++)
		if (userData->diminfo[var->dims[i]]->size != NC_UNLIMITED)
			nels *= userData->diminfo[var->dims[i]]->size;

	switch (var->type) {
	case NC_BYTE:
		cdata = (char *) data = malloc (nels * sizeof (char));
		if (data == NULL) {
			perror ("malloc");
			exit (0);
		}
		cfill = var->fillvalue.b;
		for (i = 0; i < nels; i++)
			cdata[i] = cfill;
		break;
	case NC_CHAR:
		cdata = (char *) data = malloc (nels * sizeof (char));
		if (data == NULL) {
			perror ("malloc");
			exit (0);
		}
		cfill = var->fillvalue.c;
		for (i = 0; i < nels; i++)
			cdata[i] = cfill;
		break;
	case NC_SHORT:
		sdata = (short *) data = malloc (nels * sizeof (short));
		if (data == NULL) {
			perror ("malloc");
			exit (0);
		}
		sfill = var->fillvalue.s;
		for (i = 0; i < nels; i++)
			sdata[i] = sfill;
		break;
	case NC_INT:
		idata = (int *) data = malloc (nels * sizeof (int));
		if (data == NULL) {
			perror ("malloc");
			exit (0);
		}
		ifill = var->fillvalue.i;
		for (i = 0; i < nels; i++)
			idata[i] = ifill;
		break;
#if (NC_INT != NC_LONG)
	case NC_LONG:
		ldata = (long *) data = malloc (nels * sizeof (long));
		if (data == NULL) {
			perror ("malloc");
			exit (0);
		}
		lfill = var->fillvalue.l;
		for (i = 0; i < nels; i++)
			ldata[i] = lfill;
		break;
#endif
	case NC_FLOAT:
		fdata = (float *) data = malloc (nels * sizeof (float));
		if (data == NULL) {
			perror ("malloc");
			exit (0);
		}
		dfill = var->fillvalue.d;
		for (i = 0; i < nels; i++)
			fdata[i] = dfill;
		break;
	case NC_DOUBLE:
		ddata = (double *) data = malloc (nels * sizeof (double));
		if (data == NULL) {
			perror ("malloc");
			exit (0);
		}
		dfill = var->fillvalue.d;
		for (i = 0; i < nels; i++)
			ddata[i] = dfill;
		break;
	}
	return data;
}


/*
Write the data to a netCDF file.
*/
static void put_data (struct UserData *userData, int *start, int *count, 
	void *data)
{
	switch (userData->cvar->type) {
	case NC_BYTE:
		NC_CHECK( nc_put_vara_schar (
			userData->ncid, userData->cvar->varid, start, count,
			(const signed char *) data) );
		break;
	case NC_CHAR:
		NC_CHECK( nc_put_vara_text (
			userData->ncid, userData->cvar->varid, start, count,
			(const char *) data) );
		break;
	case NC_SHORT:
		NC_CHECK( nc_put_vara_short (
			userData->ncid, userData->cvar->varid, start, count,
			(const short *) data) );
		break;
	case NC_INT:
		NC_CHECK( nc_put_vara_int (
			userData->ncid, userData->cvar->varid, start, count,
			(const int *) data) );
		break;
#if (NC_INT != NC_LONG)
	case NC_LONG:
		NC_CHECK( nc_put_vara_long (
			userData->ncid, userData->cvar->varid, start, count,
			(const long *) data) );
		break;
#endif
	case NC_FLOAT:
		NC_CHECK( nc_put_vara_float (
			userData->ncid, userData->cvar->varid, start, count,
			(const float *) data) );
		break;
	case NC_DOUBLE:
		NC_CHECK( nc_put_vara_double (
			userData->ncid, userData->cvar->varid, start, count,
			(const double *) data) );
		break;
	default:
		fprintf (stderr, "%s, line %d: unhandled data type (%d) for %s\n", 
				__FILE__, __LINE__, userData->cvar->type, userData->cvar->name);
		break;
	}
}

/*
This procedure is called as we *enter* each element.  We allocate buffers,
and (occasionally) perform some other task that can't be deferred until the
end of the element.
*/
void startElement (void *ud, const char *name, const char **atts)
{
	int i;
	nc_type t;
	char *s, *p, *data;
	struct UserData *userData = (struct UserData *) ud;

	t = lookup_element (name);

	switch (t) {
	case T_DIM:
		/* read dimension ID */
		userData->diminfo[userData->ndims] = 
			(struct DimInfo *) malloc (sizeof (struct DimInfo));
		if (userData->diminfo[userData->ndims] == NULL) {
			perror ("malloc");
			exit (0);
		}
		memset (userData->diminfo[userData->ndims], 0, sizeof (struct DimInfo));
		for ( ; *atts; atts += 2) {
			if (strcmp (*atts, "id") == 0) {
				userData->diminfo[userData->ndims]->id = strdup (atts[1]);
			}

			/* legacy support */
			else if (strcmp (*atts, "name") == 0) {
				userData->diminfo[userData->ndims]->name = strdup (atts[1]);
			}
			else if (strcmp (*atts, "size") == 0) {
				if (isalpha (*atts[1]))
					userData->diminfo[userData->ndims]->size = NC_UNLIMITED;
				else
					userData->diminfo[userData->ndims]->size = atoi (atts[1]);
			}
		}
		break;

	case T_VAR:
		/* read list of  */
		userData->cvar = (struct VarInfo *) malloc (sizeof (struct VarInfo));
		if (userData->cvar == NULL) {
			perror ("malloc");
			exit (0);
		}
		memset (userData->cvar, 0, sizeof (struct VarInfo));
		userData->varinfo[userData->nvars++] = userData->cvar;

		userData->cvar->record = 0;
		userData->cvar->natts = 0;
		memset (userData->cvar->data, 0, sizeof userData->cvar->data);

		for ( ; *atts; atts += 2) {
			if (strcmp (*atts, "dims") == 0) {
				strcpy (userData->buffer, atts[1]);
				s = userData->buffer;
				userData->cvar->ndims = 0;
				while (*s) {
					p = s;
					while (*s && !isspace (*s))
						s++;
					while (*s && isspace (*s))
						*s++ = '\0';
					for (i = 0; i < userData->ndims; i++) {
						if (strcmp (p, userData->diminfo[i]->id) == 0) {
							userData->cvar->dims[userData->cvar->ndims] = i;
							userData->cvar->ndims++;
							if (userData->diminfo[i]->size == NC_UNLIMITED)
								userData->cvar->record = 1;
							break;
						}
					}
					if (i == userData->ndims) {
						fprintf (stderr, "unrecognized dimension in %s!\n",
							userData->cvar->name);
						exit (0);
					}
				}
			}
			/* legacy support */
			else if (strcmp (*atts, "name") == 0) {
				userData->varinfo[userData->ndims]->name = strdup (atts[1]);
			}
			else if (strcmp (*atts, "type") == 0) {
				userData->varinfo[userData->ndims]->type = lookup_type(atts[1]);
			}
		}
		break;

	case T_ATT:
		userData->catt = (struct AttInfo *) malloc (sizeof (struct AttInfo *));
		if (userData->catt == NULL) {
			perror ("malloc");
			exit (0);
		}
		memset (userData->catt, 0, sizeof (struct AttInfo));

		switch (userData->stack[userData->depth]) {
		case T_NETCDF:
			userData->attinfo[userData->natts++] = userData->catt;
			break;
		case T_VAR:
			userData->cvar->attinfo[userData->cvar->natts++] = userData->catt;
			break;
		default:
			assert (0);
		}

		for ( ; *atts; atts += 2) {
			/* legacy support */
			if (strcmp (*atts, "name") == 0) {
				userData->catt->name = strdup (atts[1]);
			}
			else if (strcmp (*atts, "type") == 0) {
				userData->catt->type = lookup_type (atts[1]);
			}
			if (strcmp (*atts, "value") == 0) {
				userData->catt->value = strdup (atts[1]);
			}
		}
		break;

	case T_DATA:
		userData->cvar->nrecs = 0;
		break;

	case T_VALUE:
		switch (userData->stack[userData->depth]) {
		case T_ATT:
			break;
		case T_DATA:
		case T_RECORD:
			userData->cvar->data[userData->cvar->nrecs] = 
				init_data (userData, userData->cvar);
			break;
		default:
			assert (0);
		}
		break;
	}

	userData->stack[++userData->depth] = t;
}


/*
This procedure is called as we *leave* each element.  We can often, but
not always, defer any action until this point.
*/
void endElement (void *ud, const char *name)
{
	struct UserData *userData = (struct UserData *) ud;
	int i, j, nels;
	int lastdim;
	nc_type t;
	char *s, *p;
	void *d;

	switch (userData->stack[userData->depth]) {
	case T_DIM:
		userData->ndims++;
		break;

	case T_ATT:
		switch (userData->stack[userData->depth-1]) {
		case T_VAR:
			if (strcasecmp (userData->catt->name, "_FillValue") == 0) {
				switch (userData->catt->type) {
				case NC_BYTE:
					userData->cvar->fillvalue.b = atoi (userData->catt->value);
					break;
				case NC_CHAR:
					userData->cvar->fillvalue.c = userData->catt->value[1];
					break;
				case NC_SHORT:
					userData->cvar->fillvalue.s = atoi (userData->catt->value);
					break;
				case NC_INT:
					userData->cvar->fillvalue.i = atoi (userData->catt->value);
					break;
#if (NC_INT != NC_LONG)
				case NC_LONG:
					userData->cvar->fillvalue.l = atol (userData->catt->value);
					break;
#endif
				case NC_FLOAT:
				case NC_DOUBLE:
					userData->cvar->fillvalue.d = atof (userData->catt->value);
					break;
				}
			}
			break;

		case T_NETCDF:
			/* we don't care about global _FillValue */
			break;

		default:
			assert (0); /* impossible case */
		}
		userData->catt = 0;
		break;

	case T_TYPE:
		userData->buffer[userData->len] = '\0';
		t = lookup_type (userData->buffer);

		switch (userData->stack[userData->depth-1]) {
		case T_VAR:
			userData->cvar->type = t;
			switch (userData->cvar->type) {
			case NC_BYTE:   userData->cvar->fillvalue.b = NC_FILL_BYTE;   break;
			case NC_CHAR:   userData->cvar->fillvalue.c = NC_FILL_CHAR;   break;
			case NC_SHORT:  userData->cvar->fillvalue.s = NC_FILL_SHORT;  break;
			case NC_INT:    userData->cvar->fillvalue.i = NC_FILL_INT;    break;
#if (NC_INT != NC_LONG)
			case NC_LONG:   userData->cvar->fillvalue.l = NC_FILL_LONG;   break;
#endif
			case NC_FLOAT:  userData->cvar->fillvalue.d = NC_FILL_FLOAT;  break;
			case NC_DOUBLE: userData->cvar->fillvalue.d = NC_FILL_DOUBLE; break;
			}
			break;
		case T_ATT:
			assert (userData->catt);
			if (userData->catt)
				userData->catt->type = t;
			break;
		default:
			assert (0);
		}
		userData->len = 0;
		break;

	case T_NAME:
		userData->buffer[userData->len] = '\0';
		switch (userData->stack[userData->depth-1]) {
		case T_NETCDF:
			userData->name = strdup (userData->buffer);
			break;
		case T_DIM:
			/* both legacy & new format? */
			if (userData->diminfo[userData->ndims]->name)
				free (userData->diminfo[userData->ndims]->name);
			userData->diminfo[userData->ndims]->name = strdup(userData->buffer);
			break;
		case T_VAR:
			/* both legacy & new format? */
			if (userData->cvar->name)
				free (userData->cvar->name);
			userData->cvar->name = strdup (userData->buffer);
			break;
		case T_ATT:
			if (userData->catt) {
				/* both legacy & new format? */
				if (userData->catt->name)
					free (userData->catt->name);
				userData->catt->name = strdup (userData->buffer);
			}
			break;
		default:
			assert (0);
		}
		userData->len = 0;
		break;

	case T_SIZE:
		assert (userData->stack[userData->depth-1] == T_DIM);
		userData->buffer[userData->len] = '\0';
		if (strcasecmp (userData->buffer, "unlimited") == 0)
			userData->diminfo[userData->ndims]->size = NC_UNLIMITED;
		else
			userData->diminfo[userData->ndims]->size = atoi (userData->buffer);
		userData->len = 0;
		break;

	case T_VALUE:
		switch (userData->stack[userData->depth-1]) {
		case T_ATT:
			userData->buffer[userData->len] = '\0';
			if (userData->catt) {
				/* both legacy & new format? */
				if (userData->catt->value)
					free (userData->catt->value);
				userData->catt->value = strdup (userData->buffer);
			}
			userData->len = 0;
			break;

		case T_DATA:
		case T_RECORD:
			d = userData->cvar->data[userData->cvar->nrecs];
			if (d == NULL)
				break;

			nels = 1;
			for (i = 0; i < userData->cvar->ndims; i++) {
				lastdim = userData->diminfo[userData->cvar->dims[i]]->size;
				if (lastdim != NC_UNLIMITED)
					nels *= lastdim;
			}

			userData->buffer[userData->blen] = '\0';
			s = userData->buffer;
			for (i = 0; s && *s && i < nels; i++) {
				switch (userData->cvar->type) {
				case NC_BYTE:
					if (*s != '_')
						((signed char *) d)[i] = atoi (s);
					break;

				case NC_CHAR:
					/* this logic doesn't match legacy behavior :-( */
					memset (d, '\0', lastdim);
					while (*s != '\0' && *s != '"' && *s != ',')
						s++;
					if (*s == '"') {
						s++;
						for (j = 0; j < lastdim && *s != '"'; j++)
							((char *) d)[j] = *s++;
						while (*s != '"')
							s++;
						s++;
					}
					i += lastdim;
					break;

				case NC_SHORT:
					if (*s != '_')
						((short *) d)[i] = atoi (s);
					break;

				case NC_INT:
					if (*s != '_')
						((int *) d)[i] = atoi (s);
					break;

#if (NC_INT != NC_LONG)
				case NC_LONG:
					if (*s != '_')
						((long *) d)[i] = atol (s);
					break;
#endif

				case NC_FLOAT:
					if (*s != '_')
						((float *) d)[i] = atof (s);
					break;

				case NC_DOUBLE:
					if (*s != '_')
						((double *) d)[i] = atof (s);
					break;

				default:
					fprintf (stderr, "%s, line %d: unhandled type (%d) for %s\n",
						__FILE__, __LINE__, userData->cvar->type,
						userData->cvar->name);
				}
			
				/* skip to next comma-delimited item */
				s = strchr (s, ',');
				if (s) {
					s++;
					while (*s != '\0' && isspace (*s))
						s++;
				}
			}
			userData->len = 0;
			break;

		default:
			assert (0);
		}
		break;

	case T_RECORD:
		userData->cvar->nrecs++;
		break;
	}

	userData->depth--;
}


/*
This procedure can be a little tricky to write since the SAX
parser will introduce breaks on newlines, entities, and even
"randomly" due to the read blocksize.
*/
void characterData(void *ud, const char *s, int len)
{
	struct UserData *userData = (struct UserData *) ud;
	int n;

	switch (userData->stack[userData->depth]) {
	case T_NAME:
	case T_SIZE:
	case T_TYPE:
	case T_VALUE:
		n = (userData->blen - 1) - userData->len;
		if (n > len)
			n = len;
		strncpy (&userData->buffer[userData->len], s, n);
		userData->len += n;
		break;
	}
}


/*
We could have done this incrementally as part of the parser, but by
pulling it out we separate the parsing stage and NetCDF specific stage.
*/
void generate_netcdf (struct UserData *userData)
{
	char path[256];
	int mode;
	int i, j;
	int dims[NC_MAX_DIMS];
	int start[NC_MAX_DIMS];
	int count[NC_MAX_DIMS];

	snprintf (path, sizeof path, "%s.nc", userData->name);
	mode = NC_WRITE | NC_CLOBBER | NC_LOCK;
	NC_CHECK( nc_create (path, mode, &userData->ncid) );

	/*	
	 * Specify all metadata
	 */

	/* define dimensions */
	for (i = 0; i < userData->ndims; i++) {
		NC_CHECK( nc_def_dim (userData->ncid, 
			userData->diminfo[i]->name,
			userData->diminfo[i]->size,
			&userData->diminfo[i]->dimid) );
	}

	/* define variables and captive attributes */
	for (i = 0; i < userData->nvars; i++) {
		userData->cvar = userData->varinfo[i];
		if (userData->cvar->type != 0) {
			for (j = 0; j < userData->cvar->ndims; j++)
				dims[j] = userData->diminfo[j]->dimid;

			NC_CHECK( nc_def_var (userData->ncid, 
				userData->cvar->name, userData->cvar->type,
				userData->cvar->ndims, dims, &userData->cvar->varid) );

			for (j = 0; j < userData->cvar->natts; j++) {
				put_attribute (userData->ncid, userData->cvar->varid, 
					userData->cvar->attinfo[j]);
			}
		}
	}

	/* create global attributes */
	for (i = 0; i < userData->natts; i++) {
		put_attribute (userData->ncid, NC_GLOBAL, userData->attinfo[i]);
	}

	/* create the non-record variable data */
	NC_CHECK( nc_set_fill (userData->ncid, NC_NOFILL, &mode) );
	NC_CHECK( ncendef (userData->ncid) );

	/*	
	 * Specify all fixed-size data
	 */
	for (i = 0; i < userData->nvars; i++) {
		userData->cvar = userData->varinfo[i];
		if (userData->cvar->record == 0)
			continue;

		for (j = 0; j < userData->cvar->ndims; j++) {
			start[j] = 0;
			count[j] = userData->diminfo[userData->cvar->dims[j]]->size;
		}
		put_data (userData, start, count, userData->cvar->data[0]);
	}

	/*	
	 * Specify all "record" data
	 */
	/* create the record variable data */
	for (i = 0; i < userData->nvars; i++) {
		userData->cvar = userData->varinfo[i];
		if (!userData->cvar->record)
			continue;

		for (j = 1; j < userData->cvar->ndims; j++) {
			start[j] = 0;
			count[j] = userData->diminfo[userData->cvar->dims[j]]->size;
		}
		for (j = 0; j < userData->cvar->nrecs; j++) {
			start[0] = j;
			count[0] = 1;
			put_data (userData, start, count, userData->cvar->data[j]);
		}
	}
}


int main (int argc, const char *argv[])
{
	XML_Parser parser;
	char buf[BUFSIZ];
	int done;
	int i, j;
	struct UserData userData;
	struct stat sbuf;

	if (argc != 2) {
		fprintf (stderr, "usage: %s filename\n", argv[0]);
		exit (0);
	}

	if (stat (argv[1], &sbuf) == -1) {
		perror ("stat");
		exit (0);
	}

	userData.name = 0;
	userData.depth = 0;
	userData.ncid = -1;

	/* allocate an array as large as the input file so we *know*
	 * we can hold all of the <data>/<record> information for later
	 * processing.  We should perform incremental processing in
	 * the cdata handler, but I'm deferring that until we know that
	 * everything else works.
	 */
	userData.len = 0;
	userData.blen = sbuf.st_size;
	userData.buffer = malloc (userData.blen);
	if (userData.buffer == NULL) {
		perror ("malloc");
		exit (0);
	}

	userData.ndims = 0;
	userData.nvars = 0;
	userData.natts = 0;

	memset (userData.diminfo, 0, sizeof userData.diminfo);
	memset (userData.varinfo, 0, sizeof userData.varinfo);
	memset (userData.attinfo, 0, sizeof userData.attinfo);

	userData.cvar = NULL;
	userData.catt = NULL;

	freopen (argv[1], "r", stdin);

	parser = XML_ParserCreate(NULL);

	XML_SetUserData(parser, &userData);

	XML_SetElementHandler(parser, startElement, endElement);
	XML_SetCharacterDataHandler(parser, characterData);

	do {
		size_t len = fread(buf, 1, sizeof(buf), stdin);
		done = len < sizeof(buf);
		if (!XML_Parse(parser, buf, len, done)) {
		fprintf(stderr,
			"%s at line %d\n",
			XML_ErrorString(XML_GetErrorCode(parser)),
			XML_GetCurrentLineNumber(parser));
		exit (1);
		}
	} while (!done);

	XML_ParserFree(parser);

	generate_netcdf (&userData);

	NC_CHECK( nc_close (userData.ncid) );
	userData.ncid = -1;

	free (userData.name);
	userData.name = 0;

	/* we free everything to make life easier for tools like purify */
	for (i = 0; i < userData.ndims; i++) {
		free (userData.diminfo[i]->id);
		userData.diminfo[i]->id = 0;
		free (userData.diminfo[i]->name);
		userData.diminfo[i]->name = 0;
		free (userData.diminfo[i]);
		userData.diminfo[i] = 0;
	}

	for (i = 0; i < userData.nvars; i++) {
		for (j = 0; j < userData.varinfo[i]->natts; j++) {
			free (userData.varinfo[i]->attinfo[j]->name);
			userData.varinfo[i]->attinfo[j]->name = 0;
			free (userData.varinfo[i]->attinfo[j]->value);
			userData.varinfo[i]->attinfo[j]->value = 0;
			free (userData.varinfo[i]->attinfo[j]);
			userData.varinfo[i]->attinfo[j] = 0;
		}
		for (j = 0; j < userData.varinfo[i]->nrecs; j++) {
#if 0
			if (userData.varinfo[i]->data[j]) {
				free (userData.varinfo[i]->data[j]);
				userData.varinfo[i]->data[j] = 0;
			}
#endif
		}
		free (userData.varinfo[i]->name);
		userData.varinfo[i]->name = 0;
		free (userData.varinfo[i]);
		userData.varinfo[i] = 0;
	}

	for (i = 0; i < userData.natts; i++) {
		free (userData.attinfo[i]->name);
		userData.attinfo[i]->name = 0;
		free (userData.attinfo[i]->value);
		userData.attinfo[i]->value = 0;
		free (userData.attinfo[i]);
		userData.attinfo[i] = 0;
	}

	return 0;
}
