/*

This file is part of netcdf-4, a netCDF-like interface for HDF5, or a
HDF5 backend for netCDF, depending on your point of view.

This file handles the nc4 dimension functions.

Copyright 2003-5, University Corporation for Atmospheric Research. See
the COPYRIGHT file for copying and redistribution conditions.

$Id: nc4dim.c,v 1.19 2006/09/06 10:24:55 ed Exp $
*/

#include "nc4internal.h"

/* Netcdf-4 files might have more than one unlimited dimension, but
   return the first one anyway. */
int 
nc_inq_unlimdim(int ncid, int *unlimdimidp)
{
   NC_FILE_INFO_T *nc;
   NC_GRP_INFO_T *grp, *g;
   NC_HDF5_FILE_INFO_T *h5;
   NC_DIM_INFO_T *dim;
   int found = 0;
   int retval;
 
   LOG((2, "called nc_inq_unlimdim"));

   if ((retval = find_nc_grp_h5(ncid, &nc, &grp, &h5)))
      return retval;

   if (!h5)
      return nc3_inq_unlimdim(nc->int_ncid, unlimdimidp);

   /* According to netcdf-3 manual, return -1 if there is no unlimited
      dimension. */
   *unlimdimidp = -1;
   for (g = grp; g && !found; g = g->parent)
   {
      for (dim = g->dim; dim; dim = dim->next)
      {
	 if (dim->unlimited)
	 {
	    *unlimdimidp = dim->dimid;
	    found++;
	    break;
	 }
      }
   }

   return NC_NOERR;
}

/* Dimensions are defined in attributes attached to the appropriate
   group in the data file. */
int
nc_def_dim(int ncid, const char *name, size_t len, int *idp)
{
   NC_FILE_INFO_T *nc;
   NC_GRP_INFO_T *grp;
   NC_HDF5_FILE_INFO_T *h5;
   NC_DIM_INFO_T *dim;
   int retval = NC_NOERR;

   LOG((2, "nc_def_dim: ncid 0x%x name %s len %d", ncid, name, 
	(int)len));

   /* Find our global metadata structure. */
   if ((retval = find_nc_grp_h5(ncid, &nc, &grp, &h5)))
      return retval;
   
   /* Take care of netcdf-3 files. */
   if (!h5)
      return nc3_def_dim(nc->int_ncid, name, len, idp);

   assert(h5 && nc && grp);

   /* If the file is read-only, return an error. */
   if (h5->no_write)
     return NC_EPERM;

   /* Check some stuff if strict nc3 rules are in effect. */
   if (h5->cmode & NC_CLASSIC_MODEL)
   {
      /* Only one limited dimenson for strict nc3. */
      if (len == NC_UNLIMITED)
	 for (dim = grp->dim; dim; dim = dim->next)
	    if (dim->unlimited)
	       return NC_EUNLIMIT;

      /* Must be in define mode for stict nc3. */
      if (!(h5->flags & NC_INDEF))
	 return NC_ENOTINDEFINE;
   }   

   /* If it's not in define mode, enter define mode. */
   if (!(h5->flags & NC_INDEF))
      if ((retval = nc_redef(ncid)))
	 return retval;

   /* Make sure this is a valid netcdf name. */
   if ((retval = NC_check_name(name)))
      return retval;

   /* The dimension length needs to be positive (cast needed for
      braindead systems with signed size_t). */
   if((unsigned long) len > X_INT_MAX) /* Backward compat */
      return NC_EDIMSIZE;

   /* Make sure the name is not already in use. */
   for (dim = grp->dim; dim; dim = dim->next)
      if (!strncmp(dim->name, name, NC_MAX_NAME))
	 return NC_ENAMEINUSE;

   /* Add a dimension to the list. The ID must come from the file
    * information, since dimids are visible in more than one group. */
   dim_list_add(&grp->dim);
   grp->dim->dimid = grp->file->nc4_info->next_dimid++;

   /* Initialize the metadata for this dimension. */
   strcpy(grp->dim->name, name);
   grp->dim->len = len;
   grp->dim->dirty++;
   if (len == NC_UNLIMITED)
      grp->dim->unlimited++;

   /* Pass back the dimid. */
   if (idp)
      *idp = grp->dim->dimid;

   return retval;
}

/* Given dim name, find it's id. */
int
nc_inq_dimid(int ncid, const char *name, int *idp)
{
   NC_FILE_INFO_T *nc;
   NC_GRP_INFO_T *grp;
   NC_HDF5_FILE_INFO_T *h5;
   NC_DIM_INFO_T *dim;
   int retval;

   LOG((2, "nc_inq_dimid: ncid 0x%x name %s", ncid, name));

   /* Find metadata for this file. */
   if ((retval = find_nc_grp_h5(ncid, &nc, &grp, &h5)))
      return retval;

   /* Handle netcdf-3 files. */
   if (!h5)
      return nc3_inq_dimid(nc->int_ncid, name, idp);

   assert(nc && grp);

   /* Go through each dim and check the name. */
   for (dim = grp->dim; dim; dim = dim->next)
   {
      if (!strncmp(dim->name, name, NC_MAX_NAME))
      {
	 *idp = dim->dimid;
	 return NC_NOERR;
      }
   }
   return NC_EBADDIM;
}

/* Find out name and len of a dim. For an unlimited dimension, the
   length is the largest lenght so far written. If the name of lenp
   pointers are NULL, they will be ignored. */
int
nc_inq_dim(int ncid, int dimid, char *name, size_t *lenp)
{
   NC_FILE_INFO_T *nc;
   NC_HDF5_FILE_INFO_T *h5;
   NC_GRP_INFO_T *grp;
   NC_DIM_INFO_T *dim;
   int ret = NC_NOERR;

   LOG((2, "nc_inq_dim: ncid 0x%x dimid %d", ncid, dimid));

   /* Find our global metadata structure. */
   if ((ret = find_nc_grp_h5(ncid, &nc, &grp, &h5)))
      return ret;
   
   /* Take care of netcdf-3 files. */
   if (!h5)
      return nc3_inq_dim(nc->int_ncid, dimid, name, lenp);   
   
   assert(nc && grp);

   /* Find the dimension. */
   if ((ret = find_dim(grp, dimid, &dim)))
      return ret;
   assert(dim);

   /* Return the dimension name, if the caller wants it. */
   if (name && dim->name) 
      strcpy(name, dim->name);	    
   
   /* Return the dimension length, if the caller wants it. */
   if (lenp)
   {
      if (dim->unlimited)
      {
	 /* Since this is an unlimited dimension, go to the file
	    and see how many records there are. Take the max number
	    of records from all the vars that share this
	    dimension. */
	 *lenp = 0;
	 if ((ret = find_dim_len(grp, dimid, &lenp)))
	    return ret;
      }
      else
      {
	 *lenp = dim->len;
      }
   }

   return NC_NOERR;
}

/* Given a dimid, find it's name. */
int 
nc_inq_dimname(int ncid, int dimid, char *name)
{
   size_t len;
   return nc_inq_dim(ncid, dimid, name, &len);   
}

/* Get the length of a dimension. For an unlimited dimension get the
   current nuber of records. */
int 
nc_inq_dimlen(int ncid, int dimid, size_t *lenp)
{
   return nc_inq_dim(ncid, dimid, NULL, lenp);
}

/* Rename a dimension, for those who like to prevaricate. */
int
nc_rename_dim(int ncid, int dimid, const char *name)
{
   NC_FILE_INFO_T *nc;
   NC_GRP_INFO_T *grp;
   NC_HDF5_FILE_INFO_T *h5;
   NC_DIM_INFO_T *dim;
   int retval;

   if (!name)
      return NC_EINVAL;

   LOG((2, "nc_rename_dim: ncid 0x%x dimid %d name %s", ncid, 
	dimid, name));

   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_nc_grp_h5(ncid, &nc, &grp, &h5)))      
      return retval;
   assert(nc);
   
   /* Handle netcdf-3 cases. */
   if (!h5)
      return nc3_rename_dim(nc->int_ncid, dimid, name);
   assert(h5 && grp);

   /* Trying to write to a read-only file? No way, Jose! */
   if (h5->no_write)
      return NC_EPERM;

   /* Make sure the new name is not already in use in this group. */
   for (dim = grp->dim; dim; dim = dim->next)
      if (!strncmp(dim->name, name, NC_MAX_NAME))
	 return NC_ENAMEINUSE;

   /* Find the dim. */
   for (dim = grp->dim; dim; dim = dim->next)
      if (dim->dimid == dimid)
	 break;
   if (!dim)
      return NC_EBADDIM;

   /* Make sure this is a valid netcdf name. */
   if ((retval = NC_check_name(name)))
      return retval;

   /* If not in define mode, switch to it, unless the new name is
    * shorter. (This is in accordance with the v3 interface.) */
/*    if (!(h5->flags & NC_INDEF) && strlen(name) > strlen(dim->name)) */
/*    { */
/*       if (h5->cmode & NC_CLASSIC_MODEL) */
/* 	 return NC_ENOTINDEFINE; */
/*       if ((retval = nc_redef(ncid))) */
/* 	 return retval; */
/*    } */

   /* Save the old name, we'll need it to rename this object when we
    * sync to HDF5 file. But if there already is an old_name saved,
    * just stick with what we've got, since the user might be renaming
    * the shit out of this thing, without ever syncing with the
    * file. When the sync does take place, we only need the original
    * name of the dim, not any of the intermediate ones. */
   if (!strlen(dim->old_name))
      strcpy(dim->old_name, dim->name);

   /* Give the dimension it's new name in metadata. */
   strcpy(dim->name, name);

   return NC_NOERR;
}

/* Only netcdf-4 has this function. 

   Returns an array of unlimited dimension ids.The user can get the
   number of unlimited dimensions by first calling this with NULL for
   the second pointer.  Ed Hartnett, 12/19/03
*/
int 
nc_inq_unlimdims(int ncid, int *nunlimdimsp, int *unlimdimidsp) 
{
  NC_DIM_INFO_T *dim;
  NC_GRP_INFO_T *grp;
  NC_HDF5_FILE_INFO_T *h5;
  int retval;

  LOG((2, "nc_inq_unlimdims: ncid 0x%x", ncid));

   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;
   
   /*  if (h5->cmode & NC_CLASSIC_MODEL)
       return NC_ESTRICTNC3;*/

   LOG((3, "called nc_inq_unlimdims: ncid %d", ncid));

   /* Get our dim info. */
   if (nunlimdimsp)
   {
      *nunlimdimsp = 0;
      for (dim=grp->dim; dim; dim=dim->next)
      {
	 if (dim->unlimited)
	 {
	    if (unlimdimidsp)
	       unlimdimidsp[*nunlimdimsp] = dim->dimid;
	    (*nunlimdimsp)++;
	 }
      }
   }

   return NC_NOERR;
}


