/*

This file is part of netcdf-4, a netCDF-like interface for HDF5, or a
HDF5 backend for netCDF, depending on your point of view.

This file handles the nc4 groups.

Copyright 2005, University Corporation for Atmospheric Research. See
netcdf-4/docs/COPYRIGHT file for copying and redistribution
conditions.

$Id: nc4grp.c,v 1.16 2007/03/06 20:01:10 ed Exp $
*/

#include "nc4internal.h"

/* Create a group. It's ncid is returned in the new_ncid pointer. */
int
nc_def_grp(int parent_ncid, char *name, int *new_ncid)
{
   NC_GRP_INFO_T *grp, *g;
   NC_HDF5_FILE_INFO_T *h5;
   int retval;

   /* Make sure this is a valid netcdf name. */
   if ((retval = NC_check_name(name)))
      return retval;

   LOG((2, "nc_def_grp: parent_ncid 0x%x name %s", parent_ncid, name));

   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_grp_h5(parent_ncid, &grp, &h5)))
      return retval;
   if (!h5)
      return NC_ENOTNC4;

   /* No groups in netcdf-3! */
   if (h5->cmode & NC_CLASSIC_MODEL)
      return NC_ESTRICTNC3;

   /* Check to make sure this name isn't in use. */
   for (g = grp->children; g; g = g->next)
      if (!strcmp(g->name, name))
	 return NC_ENAMEINUSE;
   
   /* Update internal lists to reflect new group. The actual HDF5
    * group creation will be done when metadata is written by a
    * sync. */
   if ((retval = nc4_grp_list_add(&(grp->children), h5->next_nc_grpid, 
				  grp, grp->file, name, &g)))
      return retval;
   if (new_ncid)
      *new_ncid = grp->file->ext_ncid | h5->next_nc_grpid;
   h5->next_nc_grpid++;
   
   return NC_NOERR;
}

/* Given an ncid and group name (NULL gets root group), return
 * the ncid of that group. */
int
nc_inq_ncid(int ncid, char *name, int *grp_ncid)
{
   NC_GRP_INFO_T *grp, *g;
   NC_HDF5_FILE_INFO_T *h5;
   int retval;

   LOG((2, "nc_inq_ncid: ncid 0x%x name %s", ncid, name));
   
   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;
   if (!h5)
      return NC_ENOTNC4;

   if (grp_ncid)
      for (g = grp->children; g; g = g->next)
	 if (!strcmp(name, g->name)) /* found it! */
	 {
	    *grp_ncid = grp->file->ext_ncid | g->nc_grpid;
	    break;
	 }
   
   return NC_NOERR;
}

/* Given a location id, return the number of groups it contains, and
 * an array of their locids. */
int
nc_inq_grps(int ncid, int *numgrps, int *ncids)
{
   NC_GRP_INFO_T *grp, *g;
   NC_HDF5_FILE_INFO_T *h5;
   int num = 0;
   int retval;

   LOG((2, "nc_inq_grps: ncid 0x%x", ncid));

   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;
   if (!h5)
      return NC_ENOTNC4;

   /* Count the number of groups in this group. */
   for (g = grp->children; g; g = g->next)
   {
      if (ncids)
      {
	 /* Combine the nc_grpid in a bitwise or with the ext_ncid,
	  * which allows the returned ncid to carry both file and
	  * group information. */
	 *ncids = g->nc_grpid | g->file->ext_ncid;
	 ncids++;
      }
      num++;
   }
   
   if (numgrps)
      *numgrps = num;

   return NC_NOERR;
}

/* Given locid, find name of group. (Root group is named "/".) */
int
nc_inq_grpname(int ncid, char *name)
{
   NC_GRP_INFO_T *grp;
   NC_HDF5_FILE_INFO_T *h5;
   int retval;

   LOG((2, "nc_inq_grpname: ncid 0x%x", ncid));

   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;
   if (!h5)
      return NC_ENOTNC4;
   
   /* I don't know what kind of weirdo would call this function with a
    * NULL pointer for name! */
   if (name)
      strcpy(name, grp->name);

   return NC_NOERR;
}

/* Given locid, find name of group. (Root group is named "/".) */
int
nc_inq_grp_parent(int ncid, int *parent_ncid)
{
   NC_GRP_INFO_T *grp;
   NC_HDF5_FILE_INFO_T *h5;
   int retval;

   LOG((2, "nc_inq_grp_parent: ncid 0x%x", ncid));

   /* I don't know what kind of tinfoil hat wearing nut job would call
    * this function with a NULL pointer for parent_ncid! */
   if (!parent_ncid)
      return NC_NOERR;

   /* Find info for this file and group. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;

   /* Groups only work with netCDF-4/HDF5 files... */
   if (!h5)
      return NC_ENOTNC4;

   /* Set the parent ncid, if there is one. */
   if (grp->parent)
      *parent_ncid = grp->file->ext_ncid | grp->parent->nc_grpid;
   else
      return NC_ENOGRP;
   
   return NC_NOERR;
}

/* Given parent ncid and name, find ncid. (Root group is named "/".) */
int
nc_inq_grp_ncid(int ncid, char *grp_name, int *parent_ncid)
{
   NC_GRP_INFO_T *grp, *g1;
   NC_HDF5_FILE_INFO_T *h5;
   int retval;

   LOG((2, "nc_inq_grp_parent: ncid 0x%x", ncid));

   /* I don't know what kind of sleezy hack would call this function
    * with a NULL pointers! */
   if (!grp_name || !parent_ncid)
      return NC_NOERR;

   if (strlen(grp_name) > NC_MAX_NAME)
      return NC_EINVAL;

   /* Find info for this file and group. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;

   /* Groups only work with netCDF-4/HDF5 files... */
   if (!h5)
      return NC_ENOTNC4;

   /* Find group with this name, if any. */
   for (g1 = grp->children; g1; g1 = g1->next)
      if (!strcmp(g1->name, grp_name))
      {
	 *parent_ncid = grp->file->ext_ncid | g1->nc_grpid; 
	 break;
      }

   /* Didn't find it? */
   if (!g1)
      return NC_ENOGRP;

   return NC_NOERR;
}

/* Get a list of ids for all the variables in a group. */
int nc_inq_varids(int ncid, int *nvars, int *varids)
{
   NC_GRP_INFO_T *grp;
   NC_HDF5_FILE_INFO_T *h5;
   NC_VAR_INFO_T *var;
   int v, num_vars = 0;
   int retval;

   LOG((2, "nc_inq_varids: ncid 0x%x", ncid));

   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;

   if (!h5)
   {
      /* If this is a netcdf-3 file, there is only one group, the root
       * group, and it's vars have ids 0 thru nvars - 1. */
      if ((retval = nc_inq(ncid, NULL, &num_vars, NULL, NULL)))
	 return retval;
      if (varids)
	 for (v = 0; v < num_vars; v++)
	    varids[v] = v;
   }
   else
   {
      /* This is a netCDF-4 group. Round up them doggies and count
       * 'em. The list is in correct (i.e. creation) order. */
      if (grp->var)
      {
	 for (var = grp->var; var; var = var->next)
	 {
	    if (varids)
	       varids[num_vars] = var->varid;
	    num_vars++;
	 }
      }
   }

   /* If the user wants to know how many vars in the group, tell
    * him. */
   if (nvars)
      *nvars = num_vars;

   return NC_NOERR;
}

/* Find all dimids for a location. This finds all dimensions in a
 * group, with or without any of it's parents, depending on last
 * parameter. */
int nc_inq_dimids(int ncid, int *ndims, int *dimids, int include_parents)
{
   NC_GRP_INFO_T *grp, *g;
   NC_HDF5_FILE_INFO_T *h5;
   NC_DIM_INFO_T *dim;
   int d, num = 0;
   int retval;

   LOG((2, "nc_inq_dimids: ncid 0x%x include_parents: %d", ncid, 
	include_parents));

   /* Find info for this file and group, and set pointer to each. */
   if ((retval = find_grp_h5(ncid, &grp, &h5)))
      return retval;

   if (!h5)
   {
      /* If this is a netcdf-3 file, then the dimids are going to be 0
       * thru ndims-1, so just provide them. */
      if ((retval = nc_inq(ncid, &num, NULL, NULL, NULL)))
	 return retval;
      if (dimids)
	 for (d = 0; d < num; d++)
	    dimids[d] = d;
   }
   else
   {
      /* For netcdf-4 files, round up them doggies and count 'em. They
       * are stored in reverse order in the list. */
      if (grp->dim)
      {
	 for (dim = grp->dim; dim->next; dim = dim->next)
	    ;
	 for (; dim; dim = dim->prev)
	 {
	    if (dimids)
	       dimids[num] = dim->dimid;
	    num++;
	 }
      }

      /* If the user wants, check all parent directories also. */
      if (include_parents)
      {
	 for (g = grp->parent; g; g = g->parent)
	    if (grp->dim)
	    {
	       for (dim = grp->dim; dim->next; dim = dim->next)
		  ;
	       for (; dim; dim = dim->prev)
	       {
		  if (dimids)
		     dimids[num] = dim->dimid;
		  num++;
	       }
	    }
      }
   }

   /* If the user wants the number of dims, give it. */
   if (ndims)
      *ndims = num;

   return NC_NOERR;
}

