/* This is part of the netCDF package.
   Copyright 2005 University Corporation for Atmospheric Research/Unidata
   See COPYRIGHT file for conditions of use.

   Test HDF5 file code. These are not intended to be exhaustive tests,
   but they use HDF5 the same way that netCDF-4 does, so if these
   tests don't work, than netCDF-4 won't work either.

   This program deals with HDF5 compound types.

   $Id: tst_h_compounds.c,v 1.7 2005/12/06 19:01:01 ed Exp $
*/
#include <nc_tests.h>

#define FILE_NAME "tst_h_compounds.h5"
#define DIM1_LEN 3
#define OSMONDS "TheOsmonds"
#define WHO "TheWho"
#define BOOZE_VAR "Alcohol_Consumed"
#define BEER_OR_WINE "Beer_or_Wine"
#define LIQUOR "The_Hard_Stuff"
#define COMPOUND_NAME "Booze_Index"
#define ARRAY_LEN 5

int
main()
{
   hid_t fileid, osmonds_grpid, who_grpid, spaceid, typeid;
   hid_t datasetid, datasetid1, typeid1;
   hsize_t dims[1];
   struct s1 {
	 int i1, i2;
   } data[DIM1_LEN];
   struct s2 {
	 int i1[ARRAY_LEN], i2;
   } data2[DIM1_LEN];
   int i, j;

   for (i=0; i<DIM1_LEN; i++)
   {
      data[i].i1 = 99;
      data[i].i2 = -99;
      data2[i].i2 = -99;
      for (j=0; j<ARRAY_LEN; j++)
	 data2[i].i1[j] = 99;
  }

   printf("\n*** Checking HDF5 compound types.\n");
   printf("*** Checking simple HDF5 compound types...");
   
   /* Open file and create group. */
   if ((fileid = H5Fcreate(FILE_NAME, H5F_ACC_TRUNC, H5P_DEFAULT, 
			   H5P_DEFAULT)) < 0) ERR;
   if ((osmonds_grpid = H5Gcreate(fileid, OSMONDS, 0)) < 0) ERR;

   /* Create a simple compound type. */
   if ((typeid = H5Tcreate(H5T_COMPOUND, sizeof(struct s1))) < 0) ERR;
   if (H5Tinsert(typeid, BEER_OR_WINE, HOFFSET(struct s1, i1), H5T_NATIVE_INT) < 0) ERR;
   if (H5Tinsert(typeid, LIQUOR, HOFFSET(struct s1, i2), H5T_NATIVE_INT) < 0) ERR;
   if (H5Tcommit(osmonds_grpid, COMPOUND_NAME, typeid) < 0) ERR;

   /* Create a space. */
   dims[0] = DIM1_LEN;
   if ((spaceid = H5Screate_simple(1, dims, dims)) < 0) ERR;

   /* Create a dataset of this compound type. */
   if ((datasetid = H5Dcreate(osmonds_grpid, BOOZE_VAR, typeid, 
			      spaceid, H5P_DEFAULT)) < 0) ERR;

   /* Write some data. */
   if (H5Dwrite(datasetid, typeid, H5S_ALL, H5S_ALL, 
		H5P_DEFAULT, data) < 0) ERR;

   /* Release all resources. */
   if (H5Dclose(datasetid) < 0 ||
       H5Tclose(typeid) < 0 ||
       H5Sclose(spaceid) < 0 ||
       H5Gclose(osmonds_grpid) < 0 ||
       H5Fclose(fileid) < 0) ERR;

   /* Now open the file and read it. */
   if ((fileid = H5Fopen(FILE_NAME, H5F_ACC_RDONLY, H5P_DEFAULT)) < 0) ERR;
   if ((osmonds_grpid = H5Gopen(fileid, OSMONDS)) < 0) ERR;
   if ((datasetid = H5Dopen(osmonds_grpid, BOOZE_VAR)) < 0) ERR;
   if ((typeid = H5Dget_type(datasetid)) < 0) ERR;
   if (H5Tget_class(typeid) != H5T_COMPOUND) ERR;
   if (H5Tget_nmembers(typeid) != 2) ERR;
   /* This doesn't work because all I have is a reference to the type! 
   if (H5Iget_name(typeid, type_name, NC_MAX_NAME) < 0) ERR;
   if (strcmp(type_name, COMPOUND_NAME)) ERR;*/

   /* Release all resources. */
   if (H5Dclose(datasetid) < 0 ||
       H5Tclose(typeid) < 0 ||
       H5Gclose(osmonds_grpid) < 0 ||
       H5Fclose(fileid) < 0) ERR;

   SUMMARIZE_ERR;

   printf("*** Checking HDF5 compound types and groups...");
   
   /* Open file and create two group. */
   if ((fileid = H5Fcreate(FILE_NAME, H5F_ACC_TRUNC, H5P_DEFAULT, 
			   H5P_DEFAULT)) < 0) ERR;
   if ((osmonds_grpid = H5Gcreate(fileid, OSMONDS, 0)) < 0) ERR;
   if ((who_grpid = H5Gcreate(fileid, WHO, 0)) < 0) ERR;

   /* Create a simple compound type. */
   if ((typeid = H5Tcreate(H5T_COMPOUND, sizeof(struct s1))) < 0) ERR;
   if (H5Tinsert(typeid, BEER_OR_WINE, HOFFSET(struct s1, i1), H5T_NATIVE_INT) < 0) ERR;
   if (H5Tinsert(typeid, LIQUOR, HOFFSET(struct s1, i2), H5T_NATIVE_INT) < 0) ERR;
   if (H5Tcommit(osmonds_grpid, COMPOUND_NAME, typeid) < 0) ERR;

   /* Create a space. */
   dims[0] = DIM1_LEN;
   if ((spaceid = H5Screate_simple(1, dims, dims)) < 0) ERR;

   /* Create a dataset of this compound type in the same group. */
   if ((datasetid = H5Dcreate(osmonds_grpid, BOOZE_VAR, typeid, 
			      spaceid, H5P_DEFAULT)) < 0) ERR;

   /* Write some data. */
   if (H5Dwrite(datasetid, typeid, H5S_ALL, H5S_ALL, 
		H5P_DEFAULT, data) < 0) ERR;

   /* Create a dataset of this compound type in a different group. */
   if ((datasetid1 = H5Dcreate(who_grpid, BOOZE_VAR, typeid, 
			       spaceid, H5P_DEFAULT)) < 0) ERR;

   /* Write some data. */
   if (H5Dwrite(datasetid1, typeid, H5S_ALL, H5S_ALL, 
		H5P_DEFAULT, data) < 0) ERR;

   /* Release all resources. */
   if (H5Dclose(datasetid) < 0 ||
       H5Dclose(datasetid1) < 0 ||
       H5Tclose(typeid) < 0 ||
       H5Sclose(spaceid) < 0 ||
       H5Gclose(osmonds_grpid) < 0 ||
       H5Gclose(who_grpid) < 0 ||
       H5Fclose(fileid) < 0) ERR;

   {
      hsize_t num_obj;
      int i, obj_type;
      char name[NC_MAX_NAME + 1];
      htri_t equal;


      /* Now open the file and read it. */
      if ((fileid = H5Fopen(FILE_NAME, H5F_ACC_RDONLY, H5P_DEFAULT)) < 0) ERR;
      if ((osmonds_grpid = H5Gopen(fileid, OSMONDS)) < 0) ERR;
      if (H5Gget_num_objs(osmonds_grpid, &num_obj) < 0) ERR;
      for (i=0; i<num_obj; i++)
      {
	 if (H5Gget_objname_by_idx(osmonds_grpid, i, name, NC_MAX_NAME+1) < 0) ERR;
	 if ((obj_type = H5Gget_objtype_by_idx(osmonds_grpid, i)) < 0) ERR;
	 switch(obj_type)
	 {
	    case H5G_DATASET:
	       if ((datasetid = H5Dopen(osmonds_grpid, name)) < 0) ERR;
	       break;
	    case H5G_TYPE:
	       if ((typeid = H5Topen(osmonds_grpid, name)) < 0) ERR;
	       if (H5Tget_class(typeid) != H5T_COMPOUND) ERR;
	       if (H5Tget_nmembers(typeid) != 2) ERR;
	       if (strcmp(name, COMPOUND_NAME)) ERR;
	       break;
	    default:
	       ERR;
	 }
      }

      /* Open the other dataset, and learn about its type. */
      if ((who_grpid = H5Gopen(fileid, WHO)) < 0) ERR;
      if ((datasetid1 = H5Dopen(who_grpid, BOOZE_VAR)) < 0) ERR;
      if ((typeid1 = H5Dget_type(datasetid1)) < 0) ERR;
      if ((equal = H5Tequal(typeid, typeid1)) < 0) ERR;
      if (!equal) ERR;
   }

   /* Release all resources. */
   if (H5Dclose(datasetid) < 0 ||
       H5Dclose(datasetid1) < 0 ||
       H5Tclose(typeid) < 0 ||
       H5Tclose(typeid1) < 0 ||
       H5Gclose(osmonds_grpid) < 0 ||
       H5Gclose(who_grpid) < 0 ||
       H5Fclose(fileid) < 0) ERR;

   SUMMARIZE_ERR;

   printf("*** Checking HDF5 compound type which contains an array...");
   
   {
      hsize_t array_dims[] = {ARRAY_LEN};
      hid_t array_typeid;

      /* Open file and create group. */
      if ((fileid = H5Fcreate(FILE_NAME, H5F_ACC_TRUNC, H5P_DEFAULT, 
			      H5P_DEFAULT)) < 0) ERR;
      if ((osmonds_grpid = H5Gcreate(fileid, OSMONDS, 0)) < 0) ERR;

      /* Create an array type. */
      if ((array_typeid = H5Tarray_create(H5T_NATIVE_INT, 1, array_dims, NULL)) < 0) ERR;

      /* Create a compound type containing an array. */
      if ((typeid = H5Tcreate(H5T_COMPOUND, sizeof(struct s2))) < 0) ERR;
      if (H5Tinsert(typeid, BEER_OR_WINE, HOFFSET(struct s2, i1), array_typeid) < 0) ERR;
      if (H5Tinsert(typeid, LIQUOR, HOFFSET(struct s2, i2), H5T_NATIVE_INT) < 0) ERR;
      if (H5Tcommit(osmonds_grpid, COMPOUND_NAME, typeid) < 0) ERR;

      /* Create a space. */
      dims[0] = DIM1_LEN;
      if ((spaceid = H5Screate_simple(1, dims, dims)) < 0) ERR;

      /* Create a dataset of this compound type. */
      if ((datasetid = H5Dcreate(osmonds_grpid, BOOZE_VAR, typeid, 
				 spaceid, H5P_DEFAULT)) < 0) ERR;

      /* Write some data. */
      if (H5Dwrite(datasetid, typeid, H5S_ALL, H5S_ALL, 
		   H5P_DEFAULT, data2) < 0) ERR;
      
      /* Release all resources. */
      if (H5Dclose(datasetid) < 0 ||
	  H5Tclose(typeid) < 0 ||
	  H5Tclose(array_typeid) < 0 ||
	  H5Sclose(spaceid) < 0 ||
	  H5Gclose(osmonds_grpid) < 0 ||
	  H5Fclose(fileid) < 0) ERR;

      /* Now open the file and read it. */
      if ((fileid = H5Fopen(FILE_NAME, H5F_ACC_RDONLY, H5P_DEFAULT)) < 0) ERR;
      if ((osmonds_grpid = H5Gopen(fileid, OSMONDS)) < 0) ERR;
      if ((datasetid = H5Dopen(osmonds_grpid, BOOZE_VAR)) < 0) ERR;
      if ((typeid = H5Dget_type(datasetid)) < 0) ERR;
      if (H5Tget_class(typeid) != H5T_COMPOUND) ERR;
      if (H5Tget_nmembers(typeid) != 2) ERR;
      /* This doesn't work because all I have is a reference to the type! 
	 if (H5Iget_name(typeid, type_name, NC_MAX_NAME) < 0) ERR;
	 if (strcmp(type_name, COMPOUND_NAME)) ERR;*/

      /* Release all resources. */
      if (H5Dclose(datasetid) < 0 ||
	  H5Tclose(typeid) < 0 ||
	  H5Gclose(osmonds_grpid) < 0 ||
	  H5Fclose(fileid) < 0) ERR;
   }

   SUMMARIZE_ERR;

   FINAL_RESULTS;
}
