/* This is part of the netCDF package.  Copyright 2005 University
   Corporation for Atmospheric Research/Unidata See COPYRIGHT file for
   conditions of use.

   This program excersizes HDF5 variable length array code.

   $Id: tst_h_enums.c,v 1.4 2005/12/21 20:55:16 ed Exp $
*/
#include <nc_tests.h>

#define FILE_NAME "tst_h_enums.h5"
#define DIM1_LEN 12
#define ATT_NAME "Browings_reverse_enumerations"
#define SIZE 9
#define GRP_NAME "Browning"
#define NUM_VALS 12

/* This seems like a good sonnet for enumation:

How do I love thee? Let me count the ways.
I love thee to the depth and breadth and height
My soul can reach, when feeling out of sight
For the ends of Being and ideal Grace.
I love thee to the level of everyday's
Most quiet need, by sun and candle-light.
I love thee freely, as men strive for Right;
I love thee purely, as they turn from Praise.
I love thee with a passion put to use
In my old griefs, and with my childhood's faith.
I love thee with a love I seemed to lose
With my lost saints, --- I love thee with the breath,
Smiles, tears, of all my life! --- and, if God choose,
I shall but love thee better after death.
*/

int
main()
{
   hid_t fileid, grpid, spaceid, typeid, attid;

   printf("\n*** Checking HDF5 enum types.\n");
   printf("*** Checking simple HDF5 enum type...");
   {
      hsize_t dims[1] = {DIM1_LEN};
      short data[DIM1_LEN];
      short data_in[DIM1_LEN];
      int i;
      short val[NUM_VALS];
      char love_how[NUM_VALS][NC_MAX_NAME + 1] = {"Depth", "Bredth", 
						  "Height", "Level", 
						  "Freely", "Purely", 
						  "Passionately", "Lost", 
						  "Breath", "Smiles", 
						  "Tears", "After Death"};
      H5T_class_t type_class;
      size_t size;
      int num_members;
      short the_value;
      char member_name[NC_MAX_NAME];
      htri_t types_equal;
      hid_t base_hdf_typeid;
	 
      
      for (i=0; i < NUM_VALS; i++)
	 val[i] = i*2;
      for (i=0; i < DIM1_LEN; i++)
	 data[i] = i*2;
      
      /* Open file. */
      if ((fileid = H5Fcreate(FILE_NAME, H5F_ACC_TRUNC, H5P_DEFAULT, 
			      H5P_DEFAULT)) < 0) ERR;
      if ((grpid = H5Gcreate(fileid, GRP_NAME, 0)) < 0) ERR;
      
      /* Create enum type. */
      /* Both methods do the same thing, but Quincey says to prefer
       * H5Tcreate_enum. */
      /*if ((typeid =  H5Tcreate(H5T_ENUM, sizeof(short))) < 0) ERR;*/
      if ((typeid =  H5Tenum_create(H5T_NATIVE_SHORT)) < 0) ERR;

      /* Insert some values. */
      for (i=0; i<NUM_VALS; i++)
	 if (H5Tenum_insert(typeid, love_how[i], &val[i]) < 0) ERR;
      
      /* Write an attribute of this type. */
      if ((spaceid = H5Screate_simple(1, dims, NULL)) < 0) ERR;
      if ((attid = H5Acreate(grpid, ATT_NAME, typeid, spaceid,
			     H5P_DEFAULT)) < 0) ERR;
      if (H5Awrite(attid, typeid, data) < 0) ERR;

      /* Close everything. */
      if (H5Aclose(attid) < 0 ||
	  H5Tclose(typeid) < 0 ||
	  H5Gclose(grpid) < 0 ||
	  H5Fclose(fileid) < 0) ERR; 

      /* Reopen the file. */
      if ((fileid = H5Fopen(FILE_NAME, H5F_ACC_RDWR, 
			    H5P_DEFAULT)) < 0) ERR;
      if ((grpid = H5Gopen(fileid, GRP_NAME)) < 0) ERR;

      /* Check the attribute's type. */
      if ((attid = H5Aopen_name(grpid, ATT_NAME)) < 0) ERR;
      if ((typeid = H5Aget_type(attid)) < 0) ERR;
      if ((type_class = H5Tget_class(typeid)) != H5T_ENUM) ERR;
      if ((num_members = H5Tget_nmembers(typeid)) != NUM_VALS) ERR;
      if ((size = H5Tget_size(typeid)) != 2) ERR;
      if ((base_hdf_typeid = H5Tget_super(typeid)) < 0) ERR;
      if ((types_equal = H5Tequal(base_hdf_typeid, H5T_NATIVE_SHORT)) < 0) ERR;
      if (!types_equal) ERR;
      
      /* Check each value and number in the enum. */
      for (i=0; i < NUM_VALS; i++)
      {
	 if (H5Tget_member_value(typeid, i, &the_value) < 0) ERR;
	 if (the_value != val[i]) ERR;
	 strcpy(member_name, H5Tget_member_name(typeid, i));
	 if (!member_name) ERR;
	 if (strcmp(member_name, love_how[i])) ERR;
      }

      /* Now read the data in the attribute and make sure it's what we
       * expect it to be. */
      if (H5Aread(attid, typeid, data_in) < 0) ERR;
      for (i=0; i < DIM1_LEN; i++)
	 if (data_in[i] != data[i]) ERR;

      /* Close it all again. */
      if (H5Aclose(attid) < 0 ||
	  H5Tclose(typeid) < 0 ||
	  H5Gclose(grpid) < 0 ||
	  H5Fclose(fileid) < 0) ERR; 
   }
   SUMMARIZE_ERR;

   FINAL_RESULTS;
}
