/* This is part of the netCDF package.
   Copyright 2005 University Corporation for Atmospheric Research/Unidata
   See COPYRIGHT file for conditions of use.

   Test netcdf-4 variables. 
   $Id: tst_vars2.c,v 1.9 2007/04/14 15:05:19 ed Exp $
*/

#include <nc_tests.h>
#include <netcdf.h>

#define FILE_NAME "tst_vars2.nc"
#define NUM_DIMS 1
#define DIM1_LEN NC_UNLIMITED
#define DIM1_NAME "Hoplites_Engaged"
#define VAR_NAME "Battle_of_Marathon"
#define LOSSES_NAME "Miltiades_Losses"   

#define MAX_CNUM 4

int
main(int argc, char **argv)
{
   int ncid, dimids[NUM_DIMS];
   int varid;
   int nvars_in, varids_in[NUM_DIMS];
   signed char fill_value = 42, fill_value_in;
   nc_type xtype_in;
   size_t len_in;
   char name_in[NC_MAX_NAME + 1];
   int attnum_in;
   int cnum;

   printf("\n*** Testing netcdf-4 variable functions, even more.\n");
   for (cnum = 0; cnum < MAX_CNUM; cnum++)
   {
      int cmode;
      
      switch(cnum)
      {
	 case 0:
	    printf("*** Testing with classic format:\n");
	    cmode = 0;
	    break;
	 case 1:
	    printf("*** Testing with 64-bit offset format:\n");
	    cmode = NC_64BIT_OFFSET;
	    break;
	 case 2:
	    printf("*** Testing with HDF5:\n");
	    cmode = NC_NETCDF4|NC_CLOBBER;
	    break;
	 case 3:
	    printf("*** Testing with HDF5, netCDF Classic Model:\n");
	    cmode = NC_CLASSIC_MODEL | NC_NETCDF4;
      }

      printf("**** testing simple fill value attribute creation...");
      {
	 /* Create a netcdf-4 file with one scalar var. Add fill
	  * value. */
	 if (nc_create(FILE_NAME, cmode, &ncid)) ERR;
	 if (nc_def_var(ncid, VAR_NAME, NC_BYTE, 0, NULL, &varid)) ERR;
	 if (nc_enddef(ncid)) ERR;
 	 if (nc_redef(ncid)) ERR;
  	 if (nc_put_att_schar(ncid, varid, _FillValue, NC_BYTE, 1, &fill_value)) ERR;              
	 if (nc_close(ncid)) ERR;

	 /* Open the file and check. */
	 if (nc_open(FILE_NAME, NC_WRITE, &ncid)) ERR;
	 if (nc_inq_varids(ncid, &nvars_in, varids_in)) ERR;
	 if (nvars_in != 1 || varids_in[0] != 0) ERR;
	 if (nc_inq_varname(ncid, 0, name_in)) ERR;
	 if (strcmp(name_in, VAR_NAME)) ERR;
	 if (nc_inq_att(ncid, varid, _FillValue, &xtype_in, &len_in)) ERR;
	 if (xtype_in != NC_BYTE && len_in != 1) ERR;
	 if (nc_get_att(ncid, varid, _FillValue, &fill_value_in)) ERR;
	 if (fill_value_in != fill_value) ERR;
	 if (nc_close(ncid)) ERR;
      }

      SUMMARIZE_ERR;
      printf("**** testing simple fill value with data read...");
      {
	 size_t start[NUM_DIMS], count[NUM_DIMS];
	 signed char data = 99, data_in;

	 /* Create a netcdf-4 file with one unlimited dim and one
	  * var. Add fill value. */
	 if (nc_create(FILE_NAME, cmode, &ncid)) ERR;
	 if (nc_def_dim(ncid, DIM1_NAME, DIM1_LEN, &dimids[0])) ERR;
	 if (nc_def_var(ncid, VAR_NAME, NC_BYTE, NUM_DIMS, dimids, &varid)) ERR;
	 if (nc_enddef(ncid)) ERR;
	 if (nc_redef(ncid)) ERR;
	 if (nc_put_att_schar(ncid, varid, _FillValue, NC_BYTE, 1, &fill_value)) ERR;            
	 if (nc_enddef(ncid)) ERR;

	 /* Write the second record. */
	 start[0] = 1;
	 count[0] = 1;
	 if (nc_put_vara_schar(ncid, varid, start, count, &data)) ERR;

	 /* Read the first record, it should be the fill value. */
	 start[0] = 0;
	 if (nc_get_vara_schar(ncid, varid, start, count, &data_in)) ERR;
	 if (data_in != fill_value) ERR;

	 /* Read the second record, it should be the value we just wrote
	  * there. */
	 start[0] = 1;
	 if (nc_get_vara_schar(ncid, varid, start, count, &data_in)) ERR;
	 if (data_in != data) ERR;
      
	 /* Close up. */
	 if (nc_close(ncid)) ERR;

	 /* Open the file and check. */
	 if (nc_open(FILE_NAME, NC_NOWRITE, &ncid)) ERR;

	 /* Check metadata. */
	 if (nc_inq_varids(ncid, &nvars_in, varids_in)) ERR;
	 if (nvars_in != 1 || varids_in[0] != 0) ERR;
	 if (nc_inq_varname(ncid, 0, name_in)) ERR;
	 if (strcmp(name_in, VAR_NAME)) ERR;

	 /* Check fill value att. */
	 if (nc_inq_att(ncid, varid, _FillValue, &xtype_in, &len_in)) ERR;
	 if (xtype_in != NC_BYTE && len_in != 1) ERR;
	 if (nc_get_att(ncid, varid, _FillValue, &fill_value_in)) ERR;
	 if (fill_value_in != fill_value) ERR;

	 /* Read the first record, it should be the fill value. */
	 start[0] = 0;
	 if (nc_get_vara_schar(ncid, varid, start, count, &data_in)) ERR;
	 if (data_in != fill_value) ERR;

	 /* Read the second record, it should be the value we just wrote
	  * there. */
	 start[0] = 1;
	 if (nc_get_vara_schar(ncid, varid, start, count, &data_in)) ERR;
	 if (data_in != data) ERR;
      
	 if (nc_close(ncid)) ERR;
      }

      SUMMARIZE_ERR;
      printf("**** testing fill value with one other attribute...");

      {
	 int losses_value = 192, losses_value_in;

	 /* Create a netcdf-4 file with one dim and one var. Add another
	  * attribute, then fill value. */
	 if (nc_create(FILE_NAME, cmode, &ncid)) ERR;
	 if (nc_def_dim(ncid, DIM1_NAME, DIM1_LEN, &dimids[0])) ERR;
	 if (nc_def_var(ncid, VAR_NAME, NC_BYTE, NUM_DIMS, dimids, &varid)) ERR;
	 if (nc_put_att_int(ncid, varid, LOSSES_NAME, NC_INT, 1, &losses_value)) ERR;
	 if (nc_put_att_schar(ncid, varid, _FillValue, NC_BYTE, 1, &fill_value)) ERR;            
	 if (nc_close(ncid)) ERR;

	 /* Open the file and check. */
	 if (nc_open(FILE_NAME, NC_NOWRITE, &ncid)) ERR;
	 if (nc_inq_att(ncid, 0, LOSSES_NAME, &xtype_in, &len_in)) ERR;
	 if (xtype_in != NC_INT && len_in != 1) ERR;
	 if (nc_get_att(ncid, 0, LOSSES_NAME, &losses_value_in)) ERR;
	 if (losses_value_in != losses_value) ERR;
	 if (nc_inq_att(ncid, 0, _FillValue, &xtype_in, &len_in)) ERR;
	 if (xtype_in != NC_BYTE && len_in != 1) ERR;
	 if (nc_get_att(ncid, 0, _FillValue, &fill_value_in)) ERR;
	 if (fill_value_in != fill_value) ERR;
	 if (nc_inq_attid(ncid, 0, LOSSES_NAME, &attnum_in)) ERR;
	 if (attnum_in != 0) ERR;
	 if (nc_inq_attid(ncid, 0, _FillValue, &attnum_in)) ERR;
	 if (attnum_in != 1) ERR;
	 if (nc_close(ncid)) ERR;
      }

      SUMMARIZE_ERR;
      printf("**** testing fill value with three other attributes...");
      {
#define NUM_LEADERS 3
	 char leader[NUM_LEADERS][NC_MAX_NAME + 1] = {"hair_length_of_strategoi", 
						      "hair_length_of_Miltiades", 
						      "hair_length_of_Darius_I"};
	 short hair_length[NUM_LEADERS] = {3, 11, 4};
	 short short_in;
	 int a;

	 /* Create a netcdf file with one dim and one var. Add 3
	  * attributes, then fill value. */
	 if (nc_create(FILE_NAME, cmode, &ncid)) ERR;
	 if (nc_def_dim(ncid, DIM1_NAME, DIM1_LEN, &dimids[0])) ERR;
	 if (nc_def_var(ncid, VAR_NAME, NC_BYTE, NUM_DIMS, dimids, &varid)) ERR;
	 for (a = 0; a < NUM_LEADERS; a++)
	    if (nc_put_att_short(ncid, varid, leader[a], NC_SHORT, 1, &hair_length[a])) ERR;
	 if (nc_put_att_schar(ncid, varid, _FillValue, NC_BYTE, 1, &fill_value)) ERR;            
	 if (nc_close(ncid)) ERR;

	 /* Open the file. */
	 if (nc_open(FILE_NAME, NC_NOWRITE, &ncid)) ERR;

	 /* Check our three hair-related attributes. */
	 for (a = 0; a < NUM_LEADERS; a++)
	 {
	    if (nc_inq_att(ncid, 0, leader[a], &xtype_in, &len_in)) ERR;
	    if (xtype_in != NC_SHORT && len_in != 1) ERR;
	    if (nc_get_att(ncid, 0, leader[a], &short_in)) ERR;
	    if (short_in != hair_length[a]) ERR;
	    if (nc_inq_attid(ncid, 0, leader[a], &attnum_in)) ERR;
	    if (attnum_in != a) ERR;
	 }

	 /* Check our fill value attribute. */
	 if (nc_inq_att(ncid, 0, _FillValue, &xtype_in, &len_in)) ERR;
	 if (xtype_in != NC_BYTE && len_in != 1) ERR;
	 if (nc_get_att(ncid, 0, _FillValue, &fill_value_in)) ERR;
	 if (fill_value_in != fill_value) ERR;

	 if (nc_close(ncid)) ERR;
      }

      SUMMARIZE_ERR;
      printf("**** testing fill value with simple example...");
      {
/* Dims stuff. */
#define NDIMS 3
#define VAR_DIMS 3
#define DIM_A "dim1"
#define DIM_A_LEN 4
#define DIM_B "dim2"
#define DIM_B_LEN 3
#define DIM_C "dim3"
#define DIM_C_LEN NC_UNLIMITED

/* Var stuff. */
#define CXX_VAR_NAME "P"

/* Att stuff. */
#define NUM_ATTS 4
#define LONG_NAME "long_name"
#define PRES_MAX_WIND "pressure at maximum wind"
#define UNITS "units"
#define HECTOPASCALS "hectopascals"

	 int dimid[NDIMS], var_dimids[VAR_DIMS] = {2, 1, 0};
	 float fill_value = -9999.0f;
	 char long_name[] = PRES_MAX_WIND;

	 if (nc_create(FILE_NAME, cmode, &ncid)) ERR;

	 /* Create dims. */
	 if (nc_def_dim(ncid, DIM_A, DIM_A_LEN, &dimid[0])) ERR;
	 if (nc_def_dim (ncid, DIM_B, DIM_B_LEN, &dimid[1])) ERR;
	 if (nc_def_dim(ncid, DIM_C, DIM_C_LEN, &dimid[2])) ERR;

	 /* Create var. */
	 if (nc_def_var(ncid, CXX_VAR_NAME, NC_FLOAT, VAR_DIMS, 
			var_dimids, &varid)) ERR;
	 if (varid) ERR;

	 if (nc_put_att(ncid, varid, LONG_NAME, NC_CHAR, strlen(long_name) + 1, 
			long_name)) ERR;
	 if (nc_put_att(ncid, varid, UNITS, NC_CHAR, strlen(UNITS) + 1, 
			UNITS)) ERR;

	 /* Check to ensure the atts have their expected attnums. */
	 if (nc_inq_attid(ncid, 0, LONG_NAME, &attnum_in)) ERR;
	 if (attnum_in != 0) ERR;
	 if (nc_inq_attid(ncid, 0, UNITS, &attnum_in)) ERR;
	 if (attnum_in != 1) ERR;

	 /* Now add a fill value. This will acutually cause HDF5 to
	  * destroy the dataset and recreate it, recreating also the
	  * three attributes that are attached to it. */
	 if (nc_put_att(ncid, varid, _FillValue, NC_FLOAT, 
			1, &fill_value)) ERR;

	 /* Check to ensure the atts have their expected attnums. */
	 if (nc_inq_attid(ncid, 0, LONG_NAME, &attnum_in)) ERR;
	 if (attnum_in != 0) ERR;
	 if (nc_inq_attid(ncid, 0, UNITS, &attnum_in)) ERR;
	 if (attnum_in != 1) ERR;

	 if (nc_close(ncid)) ERR;

	 /* Open the file and check. */
	 if (nc_open(FILE_NAME, 0, &ncid)) ERR;
	 if (nc_inq_attid(ncid, 0, LONG_NAME, &attnum_in)) ERR;
	 if (attnum_in != 0) ERR;
	 if (nc_inq_attid(ncid, 0, UNITS, &attnum_in)) ERR;
	 if (attnum_in != 1) ERR;
	 if (nc_inq_attid(ncid, 0, _FillValue, &attnum_in)) ERR;
	 if (attnum_in != 2) ERR;

	 if (nc_close(ncid)) ERR;
      }
      SUMMARIZE_ERR;

#ifndef NO_NETCDF_2
      /* The following test is an attempt to recreate a problem occuring
	 in the cxx tests. The file is created in c++ in nctsts.cpp. */
      printf("**** testing fill value with example from cxx tests in v2 api...");
      {
/* Dims stuff. */
#define NDIMS_1 4
#define VAR_DIMS 3
#define LAT "lat"
#define LAT_LEN 4
#define LON "lon"
#define LON_LEN 3
#define FRTIMED "frtimed"
#define FRTIMED_LEN NC_UNLIMITED
#define TIMELEN "timelen"
#define TIMELEN_LEN 20

/* Var stuff. */
#define CXX_VAR_NAME "P"

/* Att stuff. */
#define NUM_ATTS 4
#define LONG_NAME "long_name"
#define UNITS "units"

	 int dimid[NDIMS_1], var_dimids[VAR_DIMS] = {2, 0, 1};
	 float fill_value = -9999.0f;
	 char long_name[] = PRES_MAX_WIND;
	 int i, attid[NUM_ATTS];

	 ncid = nccreate(FILE_NAME, NC_NETCDF4);

	 /* Create dims. */
	 dimid[0] = ncdimdef(ncid, LAT, LAT_LEN);
	 dimid[1] = ncdimdef(ncid, LON, LON_LEN);
	 dimid[2] = ncdimdef(ncid, FRTIMED, FRTIMED_LEN);
	 dimid[3] = ncdimdef(ncid, TIMELEN, TIMELEN_LEN);

	 /* Just check our dimids to see that they are correct. */
	 for (i = 0; i < NDIMS_1; i++)
	    if (dimid[i] != i) ERR;

	 /* Create var. */
	 varid = ncvardef(ncid, CXX_VAR_NAME, NC_FLOAT, VAR_DIMS, var_dimids);
	 if (varid) ERR;

	 /* Add three atts to the var, long_name, units, and
	  * valid_range. */
	 if (nc_put_att(ncid, varid, LONG_NAME, NC_CHAR, strlen(long_name) + 1,
			long_name)) ERR;
	 if (nc_put_att(ncid, varid, UNITS, NC_CHAR, strlen(UNITS) + 1,
			UNITS)) ERR;

	 /* Check to ensure the atts have their expected attnums. */
	 if (nc_inq_attid(ncid, 0, LONG_NAME, &attnum_in)) ERR;
	 if (attnum_in != 0) ERR;
	 if (nc_inq_attid(ncid, 0, UNITS, &attnum_in)) ERR;
	 if (attnum_in != 1) ERR;

	 /* Now add a fill value. This will acutually cause HDF5 to
	  * destroy the dataset and recreate it, recreating also the
	  * three attributes that are attached to it. */
	 attid[3] = ncattput(ncid, varid, _FillValue, NC_FLOAT,
			     1, &fill_value);

	 /* Check to ensure the atts have their expected attnums. */
	 if (nc_inq_attid(ncid, 0, LONG_NAME, &attnum_in)) ERR;
	 if (attnum_in != 0) ERR;
	 if (nc_inq_attid(ncid, 0, UNITS, &attnum_in)) ERR;
	 if (attnum_in != 1) ERR;

	 ncclose(ncid);

	 /* Open the file and check. */
	 ncid = ncopen(FILE_NAME, 0);
	 if (nc_inq_attid(ncid, 0, LONG_NAME, &attnum_in)) ERR;
	 if (attnum_in != 0) ERR;
	 if (nc_inq_attid(ncid, 0, UNITS, &attnum_in)) ERR;
	 if (attnum_in != 1) ERR;
	 if (nc_inq_attid(ncid, 0, _FillValue, &attnum_in)) ERR;
	 if (attnum_in != 2) ERR;
	 ncclose(ncid);
      }
      SUMMARIZE_ERR;
#endif /* NO_NETCDF_2 */
   }

   printf("**** testing endianness...");

#define NDIMS4 1
#define DIM4_NAME "Joe"
#define VAR_NAME4 "Ed"
#define DIM4_LEN 10
   {
      int dimids[NDIMS4], dimids_in[NDIMS4];
      int varid;
      int ndims, nvars, natts, unlimdimid;
      nc_type xtype_in;
      char name_in[NC_MAX_NAME + 1];
      int data[DIM4_LEN];
      int endian_in;
      int i;

      for (i = 0; i < DIM4_LEN; i++)
	 data[i] = i;

      /* Create a netcdf-4 file with one dim and one var. */
      if (nc_create(FILE_NAME, NC_NETCDF4, &ncid)) ERR;
      if (nc_def_dim(ncid, DIM4_NAME, DIM4_LEN, &dimids[0])) ERR;
      if (dimids[0] != 0) ERR;
      if (nc_def_var(ncid, VAR_NAME4, NC_INT, NDIMS4, dimids, &varid)) ERR;
      if (nc_def_var_endian(ncid, varid, NC_ENDIAN_BIG)) ERR;
      if (varid != 0) ERR;

      /* Check stuff. */
      if (nc_inq(ncid, &ndims, &nvars, &natts, &unlimdimid)) ERR;
      if (ndims != NDIMS4 || nvars != 1 || natts != 0 ||
	  unlimdimid != -1) ERR;
      if (nc_inq_varids(ncid, &nvars, varids_in)) ERR;
      if (nvars != 1) ERR;
      if (varids_in[0] != 0) ERR;
      if (nc_inq_var(ncid, 0, name_in, &xtype_in, &ndims,
		     dimids_in, &natts)) ERR;
      if (strcmp(name_in, VAR_NAME4) || xtype_in != NC_INT ||
	  ndims != 1 || natts != 0 || dimids_in[0] != 0) ERR;
      if (nc_inq_var_endian(ncid, 0, &endian_in)) ERR;
      if (endian_in != NC_ENDIAN_BIG) ERR;
      if (nc_close(ncid)) ERR;

      /* Open the file and check the same stuff. */
      if (nc_open(FILE_NAME, NC_NOWRITE, &ncid)) ERR;

      if (nc_inq(ncid, &ndims, &nvars, &natts, &unlimdimid)) ERR;
      if (ndims != NDIMS4 || nvars != 1 || natts != 0 ||
	  unlimdimid != -1) ERR;
      if (nc_inq_varids(ncid, &nvars, varids_in)) ERR;
      if (nvars != 1) ERR;
      if (varids_in[0] != 0) ERR;
      if (nc_inq_var(ncid, 0, name_in, &xtype_in, &ndims,
		     dimids_in, &natts)) ERR;
      if (strcmp(name_in, VAR_NAME4) || xtype_in != NC_INT ||
	  ndims != 1 || natts != 0 || dimids_in[0] != 0) ERR;
      if (nc_inq_var_endian(ncid, 0, &endian_in)) ERR;
      if (endian_in != NC_ENDIAN_BIG) ERR;

      if (nc_close(ncid)) ERR;
   }

   SUMMARIZE_ERR;
   printf("**** testing create order varids...");

#define UNITS "units"
#define DIMNAME "x"
#define VARNAME "data"
   {
      /* This test contributed by Jeff Whitaker of NOAA - Thanks Jeff! */
      int ncid, dimid, varid, xvarid;
      char units[] = "zlotys";

      if (nc_set_default_format(NC_FORMAT_NETCDF4_CLASSIC, NULL)) ERR;
      if (nc_create(FILE_NAME, NC_CLOBBER, &ncid)) ERR;
      if (nc_def_dim(ncid, DIMNAME, 1, &dimid)) ERR;
      if (nc_enddef(ncid)) ERR;
      if (nc_redef(ncid)) ERR;
      if (nc_def_var(ncid, DIMNAME, NC_INT, 1, &dimid, &xvarid)) ERR;
      if (nc_put_att_text(ncid, xvarid, UNITS, strlen(units), units)) ERR;
      if (nc_def_var(ncid, VARNAME, NC_INT, 1, &dimid, &varid)) ERR;
      if (nc_close(ncid)) ERR;

      if (nc_open(FILE_NAME, 0, &ncid)) ERR;
      if (nc_close(ncid)) ERR;
   }
   SUMMARIZE_ERR; 

/*    printf("*** testing 2D array of NC_CHAR..."); */
/*    { */
/*       int dimid[NDIMS_1], var_dimids[VAR_DIMS] = {2, 0, 1}; */
/*       float fill_value = -9999.0f; */
/*       char long_name[] = PRES_MAX_WIND; */
/*       int i, attid[NUM_ATTS]; */
/*       char[][] = {who would fardels bear, */
/* 		  To grunt and sweat under a weary life, */
/* 		  But that the dread of something after death, */
/* 		  The undiscover'd country from whose bourn */
/*                   No traveller returns, puzzles the will */
/*                   And makes us rather bear those ills we have */
/*                   Than fly to others that we know not of? */
/*                   Thus conscience does make cowards of us all; */
/*                   And thus the native hue of resolution */
/*                   Is sicklied o'er with the pale cast of thought, */
/*                   And enterprises of great pith and moment */
/* 		  With this regard their currents turn awry, */
/* 		  And lose the name of action.} */

/*       ncid = nccreate(FILE_NAME, NC_NETCDF4); */

/*       /\* Create dims. *\/ */
/*       dimid[0] = ncdimdef(ncid, LAT, LAT_LEN); */
/*       dimid[1] = ncdimdef(ncid, LON, LON_LEN); */

/*       /\* Create var. *\/ */
/*       varid = ncvardef(ncid, CXX_VAR_NAME, NC_FLOAT, VAR_DIMS, var_dimids); */
/*       if (varid) ERR; */

/*       ncclose(ncid); */

/*       /\* Open the file and check. *\/ */
/*       ncid = ncopen(FILE_NAME, 0); */
/*       ncclose(ncid); */
/*    } */

/*    SUMMARIZE_ERR; */
   FINAL_RESULTS;
}


