/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* ## Notice
*
* The original C code, long comment, copyright, license, and constants are from [Cephes]{@link http://www.netlib.org/cephes}. The implementation follows the original, but has been modified for JavaScript.
*
* ```text
* Copyright 1984, 1995, 2000 by Stephen L. Moshier
*
* Some software in this archive may be from the book _Methods and Programs for Mathematical Functions_ (Prentice-Hall or Simon & Schuster International, 1989) or from the Cephes Mathematical Library, a commercial product. In either event, it is copyrighted by the author. What you see here may be used freely but it comes with no support or guarantee.
*
* Stephen L. Moshier
* moshier@na-net.ornl.gov
* ```
*/

'use strict';

// TODO: replace with TOMS (Openlibm) algo (updating license header and long comment)

// MODULES //

var FLOAT64_MAX_BASE2_EXPONENT = require( '@stdlib/constants/float64/max-base2-exponent' ); // eslint-disable-line id-length
var FLOAT64_MIN_BASE2_EXPONENT = require( '@stdlib/constants/float64/min-base2-exponent' ); // eslint-disable-line id-length
var round = require( './../../../../base/special/round' );
var ldexp = require( './../../../../base/special/ldexp' );
var isnan = require( './../../../../base/assert/is-nan' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var polyvalP = require( './polyval_p.js' );
var polyvalQ = require( './polyval_q.js' );


// MAIN //

/**
* Evaluates the base `2` exponential function.
*
* ## Method
*
* -   Range reduction is accomplished by separating the argument into an integer \\( k \\) and fraction \\( f \\) such that
*
*     ```tex
*     2^x = 2^k 2^f
*     ```
*
* -   A Pade' approximate
*
*     ```tex
*     1 + 2x \frac{\mathrm{P}\left(x^2\right)}{\mathrm{Q}\left(x^2\right) - x \mathrm{P}\left(x^2\right)}
*     ```
*
*     approximates \\( 2^x \\) in the basic range \\( \[-0.5, 0.5] \\).
*
*
* ## Notes
*
* -   Relative error:
*
*     | arithmetic | domain      | # trials | peak    | rms     |
*     |:----------:|:-----------:|:--------:|:-------:|:-------:|
*     | IEEE       | -1022,+1024 | 30000    | 1.8e-16 | 5.4e-17 |
*
*
* @param {number} x - input value
* @returns {number} function value
*
* @example
* var v = exp2( 3.0 );
* // returns 8.0
*
* @example
* var v = exp2( -9.0 );
* // returns ~0.002
*
* @example
* var v = exp2( 0.0 );
* // returns 1.0
*
* @example
* var v = exp2( NaN );
* // returns NaN
*/
function exp2( x ) {
	var px;
	var xx;
	var n;
	if ( isnan( x ) ) {
		return x;
	}
	if ( x > FLOAT64_MAX_BASE2_EXPONENT ) {
		return PINF;
	}
	if ( x < FLOAT64_MIN_BASE2_EXPONENT ) {
		return 0.0;
	}
	// Separate into integer and fractional parts...
	n = round( x );
	x -= n;

	xx = x * x;
	px = x * polyvalP( xx );
	x = px / ( polyvalQ( xx ) - px );
	x = 1.0 + ldexp( x, 1 );

	// Scale by power of 2:
	return ldexp( x, n );
}


// EXPORTS //

module.exports = exp2;
