/* NVTV TV mmio header -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: mmio.h,v 1.8 2004/03/01 21:08:10 dthierbach Exp $
 *
 * Contents:
 *
 * Header: Extra memory mapped I/O defines
 *
 */

#ifndef _MMIO_H
#define _MMIO_H

/* This is the interface all MMIO and port access goes through. It has
 * to provide mappings for the following situations:
 *
 * 1) Standalone usage: Directly access MMIO hardware
 * 2) Inside X: Be compatible with X definitions (maybe extra include file?)
 * 3) During testing: Call test routines
 *
 * There is a fourth situation, when the old debug code just printf's
 * the MMIO accesses.
 * 
 * The same thing applies to port access, with the additional obligation
 * to allow for different OSs (Linux, NetBSD, ...)
 *
 */

/* -------- Ports -------- */

void mmio_port_perm (unsigned long from, unsigned long num, int turn_on);

#if !defined(FAKE_MMIO) && !defined(TEST_HOOKS)

/* from xfree common/compiler.h */

#ifndef __NetBSD__
#ifdef linux 
#include <sys/io.h>

#define OUTB(p,v) outb(v,p)
#define INB(p) inb(p)

#else /* linux */

/* FIXME: we won't need that for now under Windows */

#define OUTB(p,v)
#define INB(p) 0

#endif 

#if 0 /* Linux def, maybe reusable for Cygwin ... */

static __inline void
outb (unsigned char value, unsigned short int port)
{
  __asm__ __volatile__ ("outb %b0,%w1": :"a" (value), "Nd" (port));
}

static __inline unsigned char
inb (unsigned short int port)
{
  unsigned char _v;

  __asm__ __volatile__ ("inb %w1,%0":"=a" (_v):"Nd" (port));
  return _v;
}

#endif /* Linux def */

#else /* NetBSD */
static __inline__ u_int8_t
inb(port)
	u_int16_t port;
{
	u_int8_t r;

	__asm__ __volatile__ ("inb %w1,%0" : "=a" (r) : "Nd" (port));

	return r;
}

static __inline__ void
outb(value, port)
	u_int8_t value;
	u_int16_t port;
{
	__asm__ __volatile__ ("outb %b0,%w1" : : "a" (value), "Nd" (port));
}

#define OUTB(p,v) outb(v,p)
#define INB(p) inb(p)
#endif /* NetBSD */

#else /* FAKE_MMIO/TEST_HOOKS */

void port_outb (CARD16 port, CARD8 val);
CARD8 port_inb (CARD16 port);

void port_outw (CARD16 port, CARD16 val);
CARD16 port_inw (CARD16 port);

#define OUTB(p,v) port_outb(p,v)
#define OUTW(p,v) port_outw(p,v)
#define INB(p) port_inb(p)
#define INW(p) port_inw(p)

#endif /* FAKE_MMIO/TEST_HOOKS */


/* -------- MMIO -------- */

/* MMIO defines from X: common/compiler.h */

#define MMIO_IN8(base, offset) \
	*(volatile CARD8 *)(((CARD8*)(base)) + (offset))
#define MMIO_IN16(base, offset) \
	*(volatile CARD16 *)(void *)(((CARD8*)(base)) + (offset))
#define MMIO_IN32(base, offset) \
	*(volatile CARD32 *)(void *)(((CARD8*)(base)) + (offset))
#define MMIO_OUT8(base, offset, val) \
	*(volatile CARD8 *)(((CARD8*)(base)) + (offset)) = (val)
#define MMIO_OUT16(base, offset, val) \
	*(volatile CARD16 *)(void *)(((CARD8*)(base)) + (offset)) = (val)
#define MMIO_OUT32(base, offset, val) \
	*(volatile CARD32 *)(void *)(((CARD8*)(base)) + (offset)) = (val)


/* Extra MMIO defines */

#include "debug.h"

#define HEAD 0x2000

#define CRT_INDEX(h) (0x3d4 + (h) * HEAD)
#define CRT_DATA(h)  (0x3d5 + (h) * HEAD)

#define MMIO_H_OUT8(base,h,offset,val) MMIO_OUT8(base,(offset)+(h)*HEAD,val)
#define MMIO_H_OUT32(base,h,offset,val) MMIO_OUT32(base,(offset)+(h)*HEAD,val)
#define MMIO_H_IN8(base,h,offset) MMIO_IN8(base,(offset)+(h)*HEAD)
#define MMIO_H_IN32(base,h,offset) MMIO_IN32(base,(offset)+(h)*HEAD)

#define MMIO_H_AND32(base,h,offset,val) MMIO_AND32(base,(offset)+(h)*HEAD,val)
#define MMIO_H_OR32(base,h,offset,val) MMIO_OR32(base,(offset)+(h)*HEAD,val)

#if !defined(FAKE_MMIO) && !defined(TEST_HOOKS)

#define MMIO_AND32(base, offset, val) \
	*(volatile CARD32 *)(void *)(((CARD8*)(base)) + (offset)) &= (val)
#define MMIO_OR32(base, offset, val) \
	*(volatile CARD32 *)(void *)(((CARD8*)(base)) + (offset)) |= (val)

#else /* FAKE_MMIO */

#undef MMIO_IN8
#undef MMIO_IN16
#undef MMIO_IN32
#undef MMIO_OUT8
#undef MMIO_OUT16
#undef MMIO_OUT32

#endif

#ifdef TEST_HOOKS

CARD8 mmio_in8 (void* base, unsigned offset);
CARD16 mmio_in16 (void* base, unsigned offset);
CARD32 mmio_in32 (void* base, unsigned offset);
void mmio_out8 (void* base, unsigned offset, CARD8 val);
void mmio_out16 (void* base, unsigned offset, CARD16 val);
void mmio_out32 (void* base, unsigned offset, CARD32 val);
void mmio_and32 (void* base, unsigned offset, CARD32 val);
void mmio_or32 (void* base, unsigned offset, CARD32 val);

#define MMIO_IN8(base, offset)		mmio_in8   ((void *)base, offset)
#define MMIO_IN16(base, offset)		mmio_in16  ((void *)base, offset)
#define MMIO_IN32(base, offset)		mmio_in32  ((void *)base, offset)
#define MMIO_OUT8(base, offset, val)	mmio_out8  ((void *)base, offset, val)
#define MMIO_OUT16(base, offset, val)	mmio_out16 ((void *)base, offset, val)
#define MMIO_OUT32(base, offset, val)	mmio_out32 ((void *)base, offset, val)
#define MMIO_AND32(base, offset, val)	mmio_and32 ((void *)base, offset, val)
#define MMIO_OR32(base, offset, val)	mmio_or32  ((void *)base, offset, val)

#else /* TEST_HOOKS */

#ifdef FAKE_MMIO

#ifdef CHECK_MMIO

#ifndef CHECK_MMIO_ABORT
#define MMIO_CAUGHT FPRINTF ("\nMMIO CHECK trigger in %s:%i\n", __FUNCTION__, __LINE__)
#else
#define MMIO_CAUGHT FPRINTF ("OOPS %i", 1/0)
#endif

#define MMIO_RANGE(base, offset) \
  ((((((int) base) >> 12) & 0xfff) == CHECK_MMIO_BASE && offset >= CHECK_MMIO_MIN && offset <= CHECK_MMIO_MAX) ? MMIO_CAUGHT : FPRINTF("")),

#else

#define MMIO_RANGE(base, offset)

#endif

#define MMIO_IN8(base, offset) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X?8] ", \
      (((int) base) >> 12) & 0xfff, offset), 0 )
#define MMIO_IN16(base, offset) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X?16] ", \
      (((int) base) >> 12) & 0xfff, offset), 0 )
#define MMIO_IN32(base, offset) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X?32] ", \
      (((int) base) >> 12) & 0xfff, offset), 0L )
#define MMIO_OUT8(base, offset, val) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X=%02X] ", \
    (((int) base) >> 12) & 0xfff, offset, (int) (val)) )
#define MMIO_OUT16(base, offset, val) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X=%04X] ", \
    (((int) base) >> 12) & 0xfff, offset, (int) (val)) )
#define MMIO_OUT32(base, offset, val) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X=%08X] ", \
     (((int) base) >> 12) & 0xfff, offset, (int) (val)) )
#define MMIO_AND32(base, offset, val) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X&=%08X] ", \
      (((int) base) >> 12) & 0xfff, offset, (int) (val)) )
#define MMIO_OR32(base, offset, val) \
  ( MMIO_RANGE(base, offset) FPRINTF("[%03X/%04X|=%08X] ", \
      (((int) base) >> 12) & 0xfff, offset, (int) (val)) )

#endif /* FAKE_MMIO */
#endif /* TEST_HOOKS */

#endif /* _MMIO_H */

