#!/bin/sh

# Copyright (C) 2014-2021 Daniel Baumann <daniel.baumann@open-infrastructure.net>
#
# SPDX-License-Identifier: GPL-3.0+
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Description: example for automated Debian base system container image creation
# Requires:    debootstrap plzip xz-utils
# Usage:       sudo ./container-images.sh

set -e

ARCHITECTURES="amd64 i386"
DISTRIBUTIONS="jessie stretch buster sid"
MIRROR="https://deb.debian.org/debian"
INCLUDE="dbus"

KEY="0x55CF1BF986ABB9C7"

COMPRESSIONS="gz lz xz"

DATE="$(date +%Y%m%d)"

for DISTRIBUTION in ${DISTRIBUTIONS}
do
	for ARCHITECTURE in ${ARCHITECTURES}
	do
		TITLE="Debian ${DISTRIBUTION} ${DATE}/${ARCHITECTURE}"
		SYSTEM="debian-${DISTRIBUTION}-${DATE}_${ARCHITECTURE}"

		sudo debootstrap --arch=${ARCHITECTURE} --include=${INCLUDE} ${DISTRIBUTION} ${SYSTEM} ${MIRROR}
		sudo chroot "${SYSTEM}" apt-get clean

		VERSION="$(cat ${SYSTEM}/etc/debian_version)"

		case "${VERSION}" in
			[0-9]*)
				TITLE="Debian ${VERSION} (${DISTRIBUTION}) ${DATE}/${ARCHITECTURE}"
				SYSTEM="debian-${VERSION}-${DATE}_${ARCHITECTURE}"

				sudo mv "debian-${DISTRIBUTION}-${DATE}_${ARCHITECTURE}" "${SYSTEM}"
				;;
		esac

		sudo rm -f "${SYSTEM}/etc/apt/apt.conf.d/01autoremove-kernels"
		sudo rm -f "${SYSTEM}/etc/hostname"
		sudo rm -f "${SYSTEM}/etc/machine-id"
		sudo rm -f "${SYSTEM}/etc/resolv.conf"
		sudo rm -f "${SYSTEM}/var/lib/systemd/catalog/database"

		for COMPRESSION in ${COMPRESSIONS}
		do
			case "${COMPRESSION}" in
				gz)
					TAR_OPTIONS="--gzip"
					;;

				lz)
					TAR_OPTIONS="--lzip"
					;;

				xz)
					TAR_OPTIONS="--xz"
					;;
			esac

			echo "Creating ${SYSTEM}.system.tar.${COMPRESSION}"
			sudo tar ${TAR_OPTIONS} -cf "${SYSTEM}.system.tar.${COMPRESSION}" "${SYSTEM}"

			echo "Creating ${SYSTEM}.system.tar.${COMPRESSION}.sha512"
			sha512sum "${SYSTEM}.system.tar.${COMPRESSION}" > "${SYSTEM}.system.tar.${COMPRESSION}.sha512"

			if [ -n "${KEY}" ]
			then
				echo "Creating ${SYSTEM}.system.tar.${COMPRESSION}.sign"
				gpg -a -b --default-key ${KEY} ${SYSTEM}.system.tar.${COMPRESSION}
				mv "${SYSTEM}.system.tar.${COMPRESSION}.asc" "${SYSTEM}.system.tar.${COMPRESSION}.sign"
			fi

			echo "Creating ${SYSTEM}.system.tar.${COMPRESSION} symlink"
			ln -sf "${SYSTEM}.system.tar.${COMPRESSION}" "$(echo ${SYSTEM}.system.tar.${COMPRESSION} | sed -e "s|${DATE}|current|")"

			echo "Creating ${SYSTEM}.system.tar.${COMPRESSION}.sha512 copy"
			sed -e "s|${DATE}|current|" "${SYSTEM}.system.tar.${COMPRESSION}.sha512" > "$(echo ${SYSTEM}.system.tar.${COMPRESSION}.sha512 | sed -e "s|${DATE}|current|")"

			if [ -e "${SYSTEM}.system.tar.${COMPRESSION}.sign" ]
			then
				echo "Creating ${SYSTEM}.system.tar.${COMPRESSION}.sign copy"
				cp "${SYSTEM}.system.tar.${COMPRESSION}.sign" "$(echo ${SYSTEM}.system.tar.${COMPRESSION}.sign | sed -e "s|${DATE}|current|")"
			fi
		done

		sudo rm -rf "${SYSTEM}"

cat >> container-list.txt << EOF
${SYSTEM}.system.tar | ${TITLE}
EOF

	done
done
