/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * implementation of all the various actions for the gui (play, stop, open, pause...)
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pthread.h>
#include <strings.h>

#ifdef HAVE_ALLOCA_H
#include <alloca.h>
#endif

#include <xine/xineutils.h>

#include "common.h"
#include "xine-toolkit/xitk.h"
#include "xine-toolkit/skin.h"
#include "xine-toolkit/backend.h"
#include "config_wrapper.h"
#include "actions.h"
#include "event.h"
#include "acontrol.h"
#include "control.h"
#include "event_sender.h"
#include "help.h"
#include "kbindings.h"
#include "mediamark.h"
#include "mrl_browser.h"
#include "osd.h"
#include "panel.h"
#include "playlist.h"
#include "post.h"
#include "setup.h"
#include "stream_infos.h"
#include "tvset.h"
#include "viewlog.h"
#include "videowin.h"
#include "file_browser.h"
#include "skins.h"
#include "tvout.h"
#include "stdctl.h"
#include "download.h"
#include "errors.h"
#include "lirc.h"
#include "oxine/oxine.h"

static gGui_t *_gui_get_nextprev (void *data, int *value) {
  gGui_t **p = data, *gui;
  if (!p)
    return NULL;
  gui = *p;
  if ((p < gui->nextprev) || (p >= gui->nextprev + sizeof (gui->nextprev) / sizeof (gui->nextprev[0])))
    return NULL;
  *value = p - gui->nextprev - 1;
  return gui;
}

static void gui_messages_off (gGui_t *gui, int drop) {
  pthread_mutex_lock (&gui->no_messages.mutex);
  ++gui->no_messages.level;
  gui->no_messages.until.tv_sec = sizeof (gui->no_messages.until.tv_sec) > 4 ? ((~(uint64_t)0) >> 1) : 0x7fffffff;
  if (drop)
    gui->no_messages.used = -1;
  pthread_mutex_unlock (&gui->no_messages.mutex);
}

static void gui_messages_on (gGui_t *gui) {
  int i = 0, n;

  pthread_mutex_lock (&gui->no_messages.mutex);
  if (--gui->no_messages.level > 0) {
    pthread_mutex_unlock (&gui->no_messages.mutex);
    return;
  }
  xitk_gettime_tv (&gui->no_messages.until);
  n = gui->no_messages.used;
  if (n < 0) {
    gui->no_messages.used = 0;
    pthread_mutex_unlock (&gui->no_messages.mutex);
    return;
  }
  pthread_mutex_unlock (&gui->no_messages.mutex);

  while (i < n) {
    xitk_lock (gui->xitk, 1);
    gui_msg (gui, gui->no_messages.flags[i], "%s", gui->no_messages.msg[i]);
    xitk_lock (gui->xitk, 0);
    i++;
    pthread_mutex_lock (&gui->no_messages.mutex);
    if (--gui->no_messages.level > 0) {
      pthread_mutex_unlock (&gui->no_messages.mutex);
      return;
    }
    n = gui->no_messages.used;
    if (i >= n)
      gui->no_messages.used = n = 0;
    pthread_mutex_unlock (&gui->no_messages.mutex);
  }
}

void gui_raise_window (gGui_t *gui, xitk_window_t *xwin) {
  if (gui && xwin) {
    if ((video_window_mode (gui->vwin, TOGGLE_MODE) & WINDOWED_MODE)
      || (xitk_get_wm_type (gui->xitk) & WM_TYPE_EWMH_COMP)) {
      xitk_window_flags (xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
      xitk_window_raise_window (xwin);
      gui_layer_above (gui, xwin);
    } else {
      /* Don't unmap this window, because on re-mapping, it will not be visible until
       * its ancestor, the video window, is visible. That's not what's intended. */
      xitk_window_set_wm_window_type (xwin, video_window_is_visible (gui->vwin) < 2 ? WINDOW_TYPE_NORMAL : WINDOW_TYPE_NONE);
      xitk_window_flags (xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
      xitk_window_raise_window (xwin);
    }
    xitk_window_set_input_focus (xwin);
  }
}

#define _v3cpy(_d,_s) (_d)[0] = (_s)[0], (_d)[1] = (_s)[1], (_d)[2] = (_s)[2]
int gui_xine_get_pos_length (gGui_t *gui, xine_stream_t *stream, int pos_time_length[3]) {
  int ret = 0, v[3], t;

  do {
    /* basic fail */
    if (!gui)
      break;
    if (!stream)
      break;
    /* may be NULL */
    if (!pos_time_length)
      pos_time_length = v;
    /* shortcut the seeking case */
    if (stream == gui->stream) {
      pthread_mutex_lock (&gui->seek.mutex);
      if (gui->seek.running) {
        /* xine_play (), xine_get_* () etc. all grab stream frontend lock.
         * Dont wait for seek thread, just use its values. */
        _v3cpy (pos_time_length, gui->seek.pos_time_length);
        pthread_mutex_unlock (&gui->seek.mutex);
        return 2;
      }
      pthread_mutex_unlock (&gui->seek.mutex);
    }
    /* safe defaults */
    pos_time_length[0] = pos_time_length[1] = pos_time_length[2] = 0;
    /* not playing */
    if (xine_get_status (stream) != XINE_STATUS_PLAY)
      break;
    /* ask libxine */
    t = 0;
    while (((ret = xine_get_pos_length (stream, pos_time_length, pos_time_length + 1,
      pos_time_length + 2)) == 0) && (++t < 10) && (!gui->on_quit)) {
      printf ("gui_xine_get_pos_length: wait 100 ms (%d).\n", t);
      xine_usec_sleep (100000); /* wait before trying again */
    }
    if (ret) {
      /* ok */
      if (stream == gui->stream) {
        /* save for shortcut */
        pthread_mutex_lock (&gui->seek.mutex);
        _v3cpy (gui->seek.pos_time_length, pos_time_length);
        pthread_mutex_unlock (&gui->seek.mutex);
      }
      ret = ret == 2 ? 3 : 1;
    } else if (stream == gui->stream) {
      /* fallback */
      pthread_mutex_lock (&gui->seek.mutex);
      _v3cpy (pos_time_length, gui->seek.pos_time_length);
      pthread_mutex_unlock (&gui->seek.mutex);
      ret = 4;
    }
  } while (0);

  return ret;
}

/*
 *
 */
void gui_display_logo (gGui_t *gui) {
  /* this is also part of xitk_run () startup. */
  xitk_lock (gui->xitk, 1);
  gui->logo_mode = 2;
  if (xine_get_status (gui->stream) == XINE_STATUS_PLAY) {
    gui->ignore_next = 1;
    xine_stop (gui->stream);
    gui->ignore_next = 0;
  }
  if (gui->visual_anim.running)
    visual_anim_play (gui, 0);
  xine_set_param (gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL, -1);
  xine_set_param (gui->stream, XINE_PARAM_SPU_CHANNEL, -1);
  if (gui->display_logo)
    gui_xine_open_and_play (gui, gui->logo_mrl, NULL, 0, 0, 0, 0, 1);
  gui->logo_mode = 1;

  panel_update_channel_display (gui->panel);
  panel_reset_slider (gui->panel);
  stream_infos_update_infos (gui->streaminfo);
  video_window_reset_ssaver (gui->vwin, 0);
  xitk_lock (gui->xitk, 0);
}

/* return 0 (failure), 1 (ok), 2 (stream started from logo). */
static int _gui_xine_play (gGui_t *gui, xine_stream_t *stream, int start_pos, int start_ms, int update_mmk, int lock_xitk) {
  int ret, has_video = -1, xine_status = xine_get_status (stream), already_playing;

  if (lock_xitk)
    xitk_lock (gui->xitk, 1);

  already_playing = (xine_status == XINE_STATUS_PLAY) && !gui->logo_mode;

  if (gui->visual_anim.post_changed && (xine_status == XINE_STATUS_STOP)) {
    post_rewire_visual_anim (gui);
    gui->visual_anim.post_changed = 0;
  }

  if (gui->visual_anim.enabled == 1) {
    has_video = xine_get_stream_info (stream, XINE_STREAM_INFO_HAS_VIDEO);
    if (has_video)
      has_video = !xine_get_stream_info (stream, XINE_STREAM_INFO_IGNORE_VIDEO);
    if (has_video) {
      if (gui->visual_anim.running) {
        if (post_rewire_audio_port_to_stream (gui, stream))
          gui->visual_anim.running = 0;
      }
    } else {
      if (!gui->visual_anim.running && gui->visual_anim.post_output_element.post) {
        if (post_rewire_audio_post_to_stream (gui, stream))
          gui->visual_anim.running = 1;
      }
    }
  }

  if (lock_xitk)
    xitk_lock (gui->xitk, 0);
  ret = xine_play (stream, start_pos, start_ms) ? 1 : 0;
  if (!ret) {
    if (lock_xitk)
      xitk_lock (gui->xitk, 1);
    gui_handle_xine_error (gui, stream, NULL);
    if (lock_xitk)
      xitk_lock (gui->xitk, 0);
    return ret;
  }
  if (already_playing)
    return ret;

  if (lock_xitk)
    xitk_lock (gui->xitk, 1);

  if (gui->logo_mode != 2)
    gui->logo_mode = 0;

  if (!gui->logo_mode) {
    char ident[2048];

    stream_infos_update_infos (gui->streaminfo);

    if (update_mmk && (stream_infos_get_ident_from_stream (stream, ident, sizeof (ident)))) {
      gui_playlist_set_str_val (gui, ident, MMK_VAL_IDENT, GUI_MMK_CURRENT);
      gui_current_set_index (gui, GUI_MMK_CURRENT);
    }

    if (has_video < 0) {
      has_video = xine_get_stream_info (stream, XINE_STREAM_INFO_HAS_VIDEO);
      if (has_video)
        has_video = !xine_get_stream_info (stream, XINE_STREAM_INFO_IGNORE_VIDEO);
    }
    if (has_video) {
      if ((gui->visual_anim.enabled == 2) && gui->visual_anim.running)
        visual_anim_play (gui, 0);
      if (gui->flags & XUI_FLAG_auto_vo_vis) {
        if (video_window_is_visible (gui->vwin) < 2)
          video_window_set_visibility (gui->vwin, 1);
      }
      if ((gui->flags & XUI_FLAG_auto_panel_vis) && (video_window_is_visible (gui->vwin) > 1) &&
        panel_is_visible (gui->panel) > 1)
        panel_toggle_visibility (NULL, gui->panel);
    } else {
      if (gui->flags & XUI_FLAG_auto_vo_vis) {
        if (panel_is_visible (gui->panel) < 2)
          panel_toggle_visibility (NULL, gui->panel);
        if (video_window_is_visible (gui->vwin) > 1)
          video_window_set_visibility (gui->vwin, 0);
      }
      if ((gui->flags & XUI_FLAG_auto_panel_vis) && (video_window_is_visible (gui->vwin) > 1) &&
        panel_is_visible (gui->panel) < 2)
        panel_toggle_visibility (NULL, gui->panel);
      if (video_window_is_visible (gui->vwin) > 1) {
        if (!gui->visual_anim.running)
          visual_anim_play (gui, 1);
      } else {
        gui->visual_anim.running = 2;
      }
    }

    ret = 2;
  }

  if (lock_xitk)
    xitk_lock (gui->xitk, 0);

  return ret;
}

void gui_pl_updated (gGui_t *gui, int mmk_changed_flags) {
  if (mmk_changed_flags & MMK_CHANGED_IDENT)
    video_window_set_mrl (gui->vwin, gui->mmk.ident);
  if (mmk_changed_flags & MMK_CHANGED_MRL)
    event_sender_update_menu_buttons (gui);
  if (mmk_changed_flags & (MMK_CHANGED_IDENT | MMK_CHANGED_MRL | MMK_CHANGED_LENGTH))
    panel_message (gui->panel, NULL);
  if (mmk_changed_flags & (MMK_CHANGED_IDENT | MMK_CHANGED_MRL | MMK_CHANGED_LENGTH | MMK_CHANGED_INDEX)) {
    playlist_update_playlist (gui);
    oxine_playlist_update (gui->oxine);
  }
}

static void _start_anyway_done (void *data, int state) {
  gGui_t *gui = data;
  gui->play_data.running = 0;

  if (state == 2) {
    if (_gui_xine_play (gui, gui->play_data.stream, gui->play_data.start_pos,
      gui->play_data.start_time_in_secs * 1000, gui->play_data.update_mmk, 0) == 2)
      osd_play_status (gui, 0);
  } else {
    gui_playlist_start_next (gui, 1);
  }
}

int gui_xine_play (gGui_t *gui, xine_stream_t *stream, int start_pos, int start_time_in_secs, int update_mmk) {
  int v_unhandled, a_unhandled;
  int has[] = {XINE_STREAM_INFO_HAS_VIDEO, XINE_STREAM_INFO_VIDEO_HANDLED, XINE_STREAM_INFO_IGNORE_VIDEO,
               XINE_STREAM_INFO_HAS_AUDIO, XINE_STREAM_INFO_AUDIO_HANDLED, XINE_STREAM_INFO_IGNORE_AUDIO, -1};

  if (gui->play_data.running)
    return 0;

#ifdef XINE_QUERY_STREAM_INFO
  xine_query_stream_info (stream, NULL, 0, NULL, has);
#else
  has[0] = xine_get_stream_info (stream, has[0]);
  has[1] = xine_get_stream_info (stream, has[1]);
  has[2] = xine_get_stream_info (stream, has[2]);
  has[3] = xine_get_stream_info (stream, has[3]);
  has[4] = xine_get_stream_info (stream, has[4]);
  has[5] = xine_get_stream_info (stream, has[5]);
#endif

  if (has[2])
    has[0] = 0;
  if (has[5])
    has[3] = 0;

  v_unhandled = (has[0] && !has[1]);
  a_unhandled = (has[3] && !has[4]);
  if (v_unhandled || a_unhandled) {
    char buf[4096], *p = buf, *e = buf + sizeof (buf) - 1;

    if (v_unhandled && a_unhandled) {
      p += snprintf (p, e - p, _("The stream '%s' isn't supported by xine:\n\n"),
        (stream == gui->stream) ? gui->mmk.mrl : gui->visual_anim.mrls[gui->visual_anim.current]);
    } else {
      p += snprintf (p, e - p, _("The stream '%s' uses an unsupported codec:\n\n"),
        (stream == gui->stream) ? gui->mmk.mrl : gui->visual_anim.mrls[gui->visual_anim.current]);
    }

    if (v_unhandled) {
      const char *minfo;
      uint32_t    vfcc;
      char        tmp[32];

      minfo = xine_get_meta_info (stream, XINE_META_INFO_VIDEOCODEC);
      vfcc = xine_get_stream_info (stream, XINE_STREAM_INFO_VIDEO_FOURCC);
      p += snprintf (p, e - p, _("Video Codec: %s (%s)\n"),
        (minfo && minfo[0]) ? minfo : _("Unavailable"), get_fourcc_string (tmp, sizeof(tmp), vfcc));
    }

    if (a_unhandled) {
      const char *minfo;
      uint32_t    afcc;
      char        tmp[32];

      minfo = xine_get_meta_info (stream, XINE_META_INFO_AUDIOCODEC);
      afcc = xine_get_stream_info (stream, XINE_STREAM_INFO_AUDIO_FOURCC);
      p += snprintf (p, e - p, _("Audio Codec: %s (%s)\n"),
        (minfo && minfo[0]) ? minfo : _("Unavailable"), get_fourcc_string (tmp, sizeof (tmp), afcc));
    }
    *p = 0;

    if (v_unhandled && a_unhandled) {
      gui_msg (gui, XUI_MSG_ERROR, "%s", buf);
      return 0;
    }

    if (!(gui->flags & XUI_FLAG_play_anyway)) {
      xitk_register_key_t key;

      gui->play_data.stream             = stream;
      gui->play_data.start_pos          = start_pos;
      gui->play_data.start_time_in_secs = start_time_in_secs;
      gui->play_data.update_mmk         = update_mmk;
      gui->play_data.running            = 1;

      key = xitk_window_dialog_3 (gui->xitk, NULL, gui_layer_above (gui, NULL), 400,
        _("Start Playback ?"), _start_anyway_done, gui,
        NULL, XITK_LABEL_YES, XITK_LABEL_NO, NULL, 0, ALIGN_CENTER,
        "%s%s", buf, _("\nStart playback anyway ?\n"));

      video_window_set_transient_for (gui->vwin, xitk_get_window (gui->xitk, key));

      /* Doesn't work so well yet
         use gui->play_data.running hack for a while
         xitk_window_dialog_set_modal(xw);
      */

      return 1;
    }
  }

  {
    int ret = _gui_xine_play (gui, stream, start_pos, start_time_in_secs * 1000, update_mmk, 0);
    if (ret == 2)
      osd_play_status (gui, 0);
    return ret;
  }
}

int gui_xine_open_and_play (gGui_t *gui, const char *_mrl, const char *_sub, int start_pos,
			   int start_time, int av_offset, int spu_offset, int report_error) {
  char fullfilename[4096];
  const char *mrl = _mrl;
  int ret;

  if (gui->verbosity)
    printf("%s():\n\tmrl: '%s',\n\tsub '%s',\n\tstart_pos %d, start_time %d, av_offset %d, spu_offset %d.\n",
	   __func__, _mrl, (_sub) ? _sub : "NONE", start_pos, start_time, av_offset, spu_offset);

  if(!strncasecmp(mrl, "cfg:/", 5)) {
    config_mrl (gui->xine, mrl);
    gui_playlist_start_next (gui, 0);
    return 1;
  }

  if (/*(!strncasecmp(mrl, "ftp://", 6)) ||*/ (!strncasecmp(mrl, "dload:/", 7)))  {
    const char *url = mrl;
    download_t  download;

    if(!strncasecmp(mrl, "dload:/", 7))
      url = _mrl + 7;

    download.gui    = gui;
    download.buf    = NULL;
    download.error  = NULL;
    download.size   = 0;
    download.status = 0;

    if((network_download(url, &download))) {
      char *filename;

      filename = strrchr(url, '/');
      if(filename && filename[1]) { /* we have a filename */
	FILE *fd;

	filename++;
	snprintf(fullfilename, sizeof(fullfilename), "%s/%s", xine_get_homedir(), filename);

	if((fd = fopen(fullfilename, "w+b")) != NULL) {
	  fwrite(download.buf, download.size, 1, fd);
	  fflush(fd);
	  fclose(fd);

          gui_playlist_lock (gui);
          mediamark_set_str_val (&gui->playlist.mmk[gui->playlist.cur], fullfilename, MMK_VAL_MRL);
          gui_current_set_index (gui, GUI_MMK_CURRENT);
          gui_playlist_mark_played (gui);
          gui_playlist_unlock (gui);
          mrl = fullfilename;
        }
      }
    }

    free(download.buf);
    free(download.error);

  }

  if (mrl_look_like_playlist (gui, mrl)) {
    gui_playlist_lock (gui);
    if (gui_playlist_add_item (gui, mrl, 1, GUI_ITEM_TYPE_AUTO, 0)) {
      gui_current_set_index (gui, GUI_MMK_CURRENT);
      strlcpy (fullfilename, gui->mmk.mrl, sizeof (fullfilename));
      start_pos  = 0;
      start_time = gui->mmk.start;
      av_offset  = gui->mmk.av_offset;
      spu_offset = gui->mmk.spu_offset;
      playlist_update_playlist (gui);
      oxine_playlist_update (gui->oxine);
      gui_playlist_mark_played (gui);
      mrl = fullfilename;
    }
    gui_playlist_unlock (gui);
  }

  gui_messages_off (gui, !report_error);
  ret = xine_open (gui->stream, mrl);
  gui_messages_on (gui);
  if (!ret) {
#ifdef XINE_PARAM_GAPLESS_SWITCH
    if( xine_check_version(1,1,1) )
      xine_set_param(gui->stream, XINE_PARAM_GAPLESS_SWITCH, 0);
#endif
    /* libxine error messages are more detailled than ours. */
    if (report_error)
      gui_handle_xine_error (gui, gui->stream, mrl);
    return 0;
  }

  if (_sub) {
    gui_messages_off (gui, 1);
    ret = xine_open (gui->spu_stream, _sub);
    gui_messages_on (gui);
    if (ret)
      xine_stream_master_slave (gui->stream, gui->spu_stream, XINE_MASTER_SLAVE_PLAY | XINE_MASTER_SLAVE_STOP);
  } else {
    xine_close (gui->spu_stream);
  }

  xine_set_param(gui->stream, XINE_PARAM_AV_OFFSET, av_offset);
  xine_set_param(gui->stream, XINE_PARAM_SPU_OFFSET, spu_offset);

  if (!gui_xine_play (gui, gui->stream, start_pos, start_time, 1))
    return 0;

  {
    int v[3], res = gui_xine_get_pos_length (gui, gui->stream, v);
    if (((res == 1) || (res == 2)) && (gui_playlist_set_length (gui, v[2]) > 1)) {
      playlist_update_playlist (gui);
      oxine_playlist_update (gui->oxine);
    }
  }

  gui_playlist_probe_start (gui);

  if (gui->stdctl_enable)
    stdctl_playing (gui, mrl);

  return 1;
}

static int gui_seek_wait (gGui_t *gui) {
  int n;

  pthread_mutex_lock (&gui->seek.mutex);
  for (n = 25; n > 0; n--) {
    if (gui->seek.running == 0)
      break;
    pthread_mutex_unlock (&gui->seek.mutex);
    xine_usec_sleep (50000);
    pthread_mutex_lock (&gui->seek.mutex);
  }
  pthread_mutex_unlock (&gui->seek.mutex);
  if (!n && (gui->verbosity >= 2))
    printf ("gui.seek.wait: WARNING: seek thread still running!\n");
  return n;
}

/*
 * Callback-functions for widget-button click
 */
void gui_exit (xitk_widget_t *w, void *data) {
  gGui_t *gui = data;

  /* NOTE: This runs as xitk callback, with xitk lock held.
   * Do not wait for threads that try to use xitk here -- see gui_exit_2 () below. */
  (void)w;

  if (gui->verbosity)
    printf ("xine_ui: gui_exit ().\n");

  oxine_exit (&gui->oxine);

  if(xine_get_status(gui->stream) == XINE_STATUS_PLAY) {
    gui->ignore_next = 1;

    if(gui->visual_anim.running) {
      xine_post_out_t * audio_source;

      xine_stop(gui->visual_anim.stream);

      while(xine_get_status(gui->visual_anim.stream) == XINE_STATUS_PLAY)
	xine_usec_sleep(50000);

      audio_source = xine_get_audio_source(gui->stream);
      (void) xine_post_wire_audio_port(audio_source, gui->ao_port);
    }

    xine_stop (gui->stream);
    while(xine_get_status(gui->stream) == XINE_STATUS_PLAY)
      xine_usec_sleep(50000);

    gui->ignore_next = 0;
  }

  gui->on_quit = 1;
  xitk_stop (gui->xitk);
}

void gui_exit_2 (gGui_t *gui) {

  if (gui->verbosity)
    printf ("xine_ui: gui_exit_2 ().\n");

  /* paranoia: xitk internal error exit (eg. parent console close */
  if (!gui->on_quit)
    gui_exit (NULL, gui);

  /* shut down event queue threads */
  /* do it first, events access gui elements ... */
  xine_event_dispose_queue(gui->event_queue);
  xine_event_dispose_queue(gui->visual_anim.event_queue);
  gui->event_queue = gui->visual_anim.event_queue = NULL;

  acontrol_main (XUI_W_OFF, gui);
  control_main (XUI_W_DEINIT, gui);
  event_sender_main (XUI_W_OFF, gui);
  help_main (XUI_W_OFF, gui);
  kbedit_main (XUI_W_OFF, gui);
  mrl_browser_main (XUI_W_OFF, gui);
  playlist_main (XUI_W_OFF, gui);
  pplugin_main (XUI_W_OFF, &gui->post_audio);
  pplugin_main (XUI_W_OFF, &gui->post_video);
  setup_main (XUI_W_OFF, gui);
#ifdef HAVE_TAR
  skin_download_end (gui->skdloader);
#endif
  stream_infos_main (XUI_W_OFF, gui);
  tvset_main (XUI_W_OFF, gui);
  viewlog_main (XUI_W_OFF, gui);

  panel_deinit (gui->panel);

  gui_deinit (gui);

  if (gui->load_stream) {
    filebrowser_end (gui->load_stream);
    gui->load_stream = NULL;
  }
  if (gui->load_sub) {
    filebrowser_end (gui->load_sub);
    gui->load_sub = NULL;
  }

  if (video_window_is_visible (gui->vwin) > 1)
    video_window_set_visibility (gui->vwin, 0);

  gui_seek_wait (gui);

  tvout_deinit(gui->tvout);

  osd_deinit (gui);

  config_update_num (gui->xine, "gui.amp_level", gui->mixer.level[SOFTWARE_MIXER]);
  xine_config_save (gui->xine, gui->cfg_file);

  xine_close(gui->stream);
  xine_close(gui->visual_anim.stream);
  xine_close (gui->spu_stream);

  /* we are going to dispose this stream, so make sure slider_loop
   * won't use it anymore (otherwise -> segfault on exit).
   */
  gui->running = 0;

  if(gui->visual_anim.post_output_element.post)
    xine_post_dispose (gui->xine, gui->visual_anim.post_output_element.post);
  gui->visual_anim.post_output_element.post = NULL;

  /* unwire post plugins before closing streams */
  post_deinit (gui);

  xine_dispose(gui->stream);
  xine_dispose(gui->visual_anim.stream);
  xine_dispose (gui->spu_stream);
  gui->stream = gui->visual_anim.stream = gui->spu_stream = NULL;

  if(gui->vo_port)
    xine_close_video_driver (gui->xine, gui->vo_port);
  if(gui->vo_none)
    xine_close_video_driver (gui->xine, gui->vo_none);
  gui->vo_port = gui->vo_none = NULL;

  if(gui->ao_port)
    xine_close_audio_driver (gui->xine, gui->ao_port);
  if(gui->ao_none)
    xine_close_audio_driver (gui->xine, gui->ao_none);
  gui->ao_port = gui->ao_none = NULL;

  video_window_exit (gui->vwin);
  gui->vwin = NULL;

  if (gui->flags & XUI_FLAG_start_lirc)
    lirc_stop (gui);

  if (gui->stdctl_enable)
    stdctl_stop (gui);

  xitk_skin_delete (&gui->skin_config);

  gui_playlist_free (gui);
  gui_current_free (gui);

  if (gui->icon)
    xitk_image_free_image (&gui->icon);
}

void gui_stop (xitk_widget_t *w, void *data) {
  gGui_t *gui = data;

  (void)w;
  gui->ignore_next = 1;
  xine_stop (gui->stream);
  gui->ignore_next = 0;

  gui_playlist_lock (gui);
  if (!(gui->playlist.control & PLAYLIST_CONTROL_STOP_PERSIST))
    gui->playlist.control &= ~PLAYLIST_CONTROL_STOP;
  gui_playlist_unlock (gui);

  pthread_mutex_lock (&gui->seek.mutex);
  gui->seek.pos_time_length[0] = gui->seek.pos_time_length[1] = gui->seek.pos_time_length[2] = 0;
  pthread_mutex_unlock (&gui->seek.mutex);

  mediamark_reset_played_state (gui);
  if(gui->visual_anim.running) {
    xine_stop(gui->visual_anim.stream);
    if(gui->visual_anim.enabled == 2)
      gui->visual_anim.running = 0;
  }

  osd_hide (gui);
  osd_play_status (gui, 0);
  panel_reset_slider (gui->panel);
  panel_check_pause (gui->panel);
  panel_update_runtime_display (gui->panel, 0);
  panel_playback_ctrl (gui->panel, !(gui_current_set_index (gui, GUI_MMK_CURRENT) & MMK_CHANGED_FAIL));

  gui_display_logo (gui);
}

void gui_close (xitk_widget_t *w, void *data) {
  gGui_t *gui = data;

  (void)w;
  gui->ignore_next = 1;
  xine_stop (gui->stream);
  gui->ignore_next = 0;

  gui_playlist_lock (gui);
  if (!(gui->playlist.control & PLAYLIST_CONTROL_STOP_PERSIST))
    gui->playlist.control &= ~PLAYLIST_CONTROL_STOP;
  gui_playlist_unlock (gui);

  pthread_mutex_lock (&gui->seek.mutex);
  gui->seek.pos_time_length[0] = gui->seek.pos_time_length[1] = gui->seek.pos_time_length[2] = 0;
  pthread_mutex_unlock (&gui->seek.mutex);

  mediamark_reset_played_state (gui);
  if(gui->visual_anim.running) {
    xine_stop(gui->visual_anim.stream);
    if(gui->visual_anim.enabled == 2)
      gui->visual_anim.running = 0;
  }

  osd_hide (gui);
  osd_play_status (gui, 0);
  panel_reset_slider (gui->panel);
  panel_check_pause (gui->panel);
  panel_update_runtime_display (gui->panel, 0);
  panel_playback_ctrl (gui->panel, !(gui_current_set_index (gui, GUI_MMK_CURRENT) & MMK_CHANGED_FAIL));
  gui->ignore_next = 1;
  xine_close (gui->stream);
  gui->ignore_next = 0;
}

void gui_pause (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  gGui_t *gui = data;
  int speed = xine_get_param(gui->stream, XINE_PARAM_SPEED);

  (void)w;
  (void)state;
  (void)modifier;
  if (speed != XINE_SPEED_PAUSE) {
    gui->last_playback_speed = speed;
    xine_set_param(gui->stream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
    panel_update_runtime_display (gui->panel, 0);
    video_window_reset_ssaver (gui->vwin, 0);
  } else {
    xine_set_param (gui->stream, XINE_PARAM_SPEED, gui->last_playback_speed);
    video_window_reset_ssaver (gui->vwin, 1);
  }

  panel_check_pause (gui->panel);
  /* Give xine engine some time before updating OSD, otherwise the */
  /* time disp may be empty when switching to XINE_SPEED_PAUSE.    */
  xine_usec_sleep(10000);
  osd_play_status (gui, 0);
}

void gui_eject (xitk_widget_t *w, void *data) {
  gGui_t *gui = data;

  (void)w;
  if (xine_get_status (gui->stream) == XINE_STATUS_PLAY)
    gui_stop (NULL, gui);

  gui_playlist_remove (gui, GUI_MMK_CURRENT,
    xine_eject (gui->stream) ? GUI_PLAYLIST_REMOVE_SAME_DEVICE : GUI_PLAYLIST_REMOVE_1);
  gui_current_set_index (gui, GUI_MMK_CURRENT);
}

void gui_set_fullscreen_mode (xitk_widget_t *w, void *data) {
  (void)w;
  gGui_t *gui = (gGui_t *)data;

  if ((video_window_is_visible (gui->vwin) < 2) || gui->use_root_window)
    return;
  video_window_mode (gui->vwin, TOGGLE_MODE | FULLSCR_MODE);
}

void gui_toggle_aspect (gGui_t *gui, int aspect) {
  static const char * const ratios[XINE_VO_ASPECT_NUM_RATIOS + 1] = {
    "Auto",
    "Square",
    "4:3",
    "Anamorphic",
    "DVB",
    NULL
  };

  if (!gui)
    return;
  if(aspect == -1)
    aspect = xine_get_param(gui->stream, XINE_PARAM_VO_ASPECT_RATIO) + 1;

  xine_set_param(gui->stream, XINE_PARAM_VO_ASPECT_RATIO, aspect);

  osd_message (gui, _("Aspect ratio: %s"),
    ratios [xine_get_param (gui->stream, XINE_PARAM_VO_ASPECT_RATIO)]);

  panel_raise_window(gui->panel);
}

void gui_direct_change_audio_channel (xitk_widget_t *w, void *data, int value) {
  gGui_t *gui = data;
  (void)w;
  xine_set_param(gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL, value);
  panel_update_channel_display (gui->panel);
  osd_lang (gui, 0);
}

void gui_nextprev_audio_channel (xitk_widget_t *w, void *data) {
  int channel;
  gGui_t *gui = _gui_get_nextprev (data, &channel);

  if (!gui)
    return;
  channel += xine_get_param (gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL);
  gui_direct_change_audio_channel (w, gui, channel);
}

void gui_direct_change_spu_channel(xitk_widget_t *w, void *data, int value) {
  gGui_t *gui = data;
  (void)w;
  if (!gui)
    return;
  xine_set_param(gui->stream, XINE_PARAM_SPU_CHANNEL, value);
  panel_update_channel_display (gui->panel);
  osd_lang (gui, 1);
}

void gui_nextprev_spu_channel (xitk_widget_t *w, void *data) {
  int channel;
  gGui_t *gui = _gui_get_nextprev (data, &channel);
  int maxchannel;

  (void)w;
  if (!gui)
    return;
  channel += xine_get_param(gui->stream, XINE_PARAM_SPU_CHANNEL);
  maxchannel = xine_get_stream_info(gui->stream, XINE_STREAM_INFO_MAX_SPU_CHANNEL);

  if (xine_get_status(gui->spu_stream) != XINE_STATUS_IDLE) /* if we have a slave SPU channel, take it into account */
    maxchannel += xine_get_stream_info(gui->spu_stream, XINE_STREAM_INFO_MAX_SPU_CHANNEL);

  /* XINE_STREAM_INFO_MAX_SPU_CHANNEL actually returns the number of available spu channels, i.e.
   * 0 means no SPUs, 1 means 1 SPU channel, etc. */
  --maxchannel;

  if (channel > maxchannel)
    channel = -2; /* -2 == off, -1 == auto, 0 == 1st channel */
  else if (channel < -2)
    channel = maxchannel;

  gui_direct_change_spu_channel (w, gui, channel);
}

void gui_nextprev_speed (xitk_widget_t *w, void *data) {
  int oldspeed, speed, d;
  gGui_t *gui = _gui_get_nextprev (data, &d);

  (void)w;
  if (!gui)
    return;

  oldspeed = speed = xine_get_param (gui->stream, XINE_PARAM_SPEED);
  do {
    if (d < 0) { /* slow down. */
      if (speed > XINE_SPEED_PAUSE) {
        speed /= 2;
      } else {
#ifdef XINE_PARAM_VO_SINGLE_STEP
        /* no more than 25 steps per second, please. */
        struct timespec ts = {0, 0};
        int step_time, diff;
        xitk_gettime_ts (&ts);
        step_time = (ts.tv_sec & 0x00ffffff) * 25 + (int)ts.tv_nsec / (1000000000 / 25);
        if (step_time != gui->last_step_time) {
          gui->last_step_time = step_time;
          /* performance issue: single step will
           * 1. find next frame,
           * 2. update the info returned by xine_get_pos_length (),
           * 3. blend previous osd,
           * 4. call our frame output hook, and
           * 5. display frame.
           * we can post new osd after the step, but that would
           * trigger redisplay of the same frame.
           * HACK: post osd with estimated time of next frame first,
           * then step, then post osd again only if the estimate
           * was too far off. */
          osd_play_status (gui, 1);
          xine_set_param (gui->stream, XINE_PARAM_VO_SINGLE_STEP, 1);
          osd_play_status (gui, 2);
          panel_update_runtime_display (gui->panel, 0);
          /* if the step was unusually slow, there may pile up many
           * key press repeats. skip them in that case. */
          xitk_gettime_ts (&ts);
          step_time = (ts.tv_sec & 0x00ffffff) * 25 + (int)ts.tv_nsec / (1000000000 / 25);
          diff = step_time - gui->last_step_time;
          if (diff < 0)
            diff += 0x01000000 * 25;
          if (diff > 1)
            gui->last_step_time = step_time;
        }
#endif
        break;
      }
    } else if (d > 0) { /* speed up. */
      if (speed >= XINE_SPEED_FAST_4)
        break;
      speed = (speed > XINE_SPEED_PAUSE) ? speed * 2 : XINE_SPEED_SLOW_4;
    } else { /* reset. */
      speed = XINE_SPEED_NORMAL;
    }
    xine_set_param (gui->stream, XINE_PARAM_SPEED, speed);
    osd_play_status (gui, 0);
  } while (0);

  if (oldspeed != speed)
    video_window_reset_ssaver (gui->vwin, speed == XINE_SPEED_PAUSE ? 0 : 1);

  if (speed != XINE_SPEED_PAUSE)
    gui->last_playback_speed = speed;

  panel_check_pause (gui->panel);
}

static void *gui_seek_thread (void *data) {
  gGui_t *gui = data;
  int update_mmk = 0;

  pthread_detach (pthread_self ());

  do {
    int status = xine_get_status (gui->stream);

    if (gui->logo_mode || status != XINE_STATUS_PLAY) {
      /* the implicit start by seek. dont repeat errors here. */
      /* xitk_lock (gui->xitk, 1); << not needed here. */
      if (gui_playlist_play_current (gui)) {
        /* xitk_lock (gui->xitk, 0); */
        pthread_mutex_lock (&gui->seek.mutex);
        gui->seek.pos_time_length[0] = gui->seek.pos_time_length[1] = gui->seek.pos_time_length[2] = 0;
        pthread_mutex_unlock (&gui->seek.mutex);
        xitk_lock (gui->xitk, 1);
        panel_update_runtime_display (gui->panel, _PURD_LOCK | _PURD_SLIDER);
        gui_display_logo (gui);
        xitk_lock (gui->xitk, 0);
        break;
      }
      /* xitk_lock (gui->xitk, 0); */
      gui_messages_on (gui);
      status = xine_get_status (gui->stream);
    }

    if (!xine_get_stream_info (gui->stream, XINE_STREAM_INFO_SEEKABLE) || (gui->ignore_next == 1))
      break;

    gui->ignore_next = 1;

    gui_playlist_lock (gui);
    if ( gui->playlist.num &&
        (gui->playlist.cur >= 0) &&
         gui->playlist.mmk &&
         gui->playlist.mmk[gui->playlist.cur] &&
        (gui->playlist.mmk[gui->playlist.cur]->mrl == gui->mmk.mrl))
      update_mmk = 1;
    gui_playlist_unlock (gui);

    {
      int pos = -1, v[3] = {0, 0, 0};

      if (xine_get_pos_length (gui->stream, v + 0, v + 1, v + 2)) {
        pthread_mutex_lock (&gui->seek.mutex);
        _v3cpy (gui->seek.pos_time_length, v);
        pthread_mutex_unlock (&gui->seek.mutex);
      }

      while (1) {
        pthread_mutex_lock (&gui->seek.mutex);
        if ((pos == gui->seek.newpos) && (gui->seek.timestep == 0)) {
          gui->seek.running = 0;
          gui->seek.newpos = -1;
          pthread_mutex_unlock (&gui->seek.mutex);
          break;
        }
        if (pos != gui->seek.newpos) {
          int ret;

          pos = gui->seek.newpos;
          pthread_mutex_unlock (&gui->seek.mutex);

          ret = _gui_xine_play (gui, gui->stream, pos, 0, update_mmk, 1);
          /* for performance, stream restart by xine_play (, 0, 0) returns early
           * (first frame decoded but not yet displayed). assume pos = 0 on success. */
          if (!pos && (ret >= 1)) {
            pthread_mutex_lock (&gui->seek.mutex);
            gui->seek.pos_time_length[0] = gui->seek.pos_time_length[1] = 0;
            pthread_mutex_unlock (&gui->seek.mutex);
            panel_update_runtime_display (gui->panel, _PURD_LOCK);
            osd_stream_position (gui, gui->seek.pos_time_length);
          } else if (xine_get_pos_length (gui->stream, v + 0, v + 1, v + 2)) {
            pthread_mutex_lock (&gui->seek.mutex);
            _v3cpy (gui->seek.pos_time_length, v);
            pthread_mutex_unlock (&gui->seek.mutex);
            if (ret >= 1) {
              panel_update_runtime_display (gui->panel, _PURD_LOCK);
              osd_stream_position (gui, gui->seek.pos_time_length);
            }
          }

        } else {

          int timestep = gui->seek.timestep;
          gui->seek.timestep = 0;
          pthread_mutex_unlock (&gui->seek.mutex);

          if ( xine_get_stream_info (gui->stream, XINE_STREAM_INFO_SEEKABLE) &&
              (xine_get_status (gui->stream) == XINE_STATUS_PLAY)) {
            int ret;

            gui->ignore_next = 1;
            v[1] += timestep;
            if (v[1] < 0)
              v[1] = 0;
            ret = _gui_xine_play (gui, gui->stream, 0, v[1], 1, 1);
            /* for performance, stream restart by xine_play (, 0, 0) returns early
             * (first frame decoded but not yet displayed). assume pos = 0 on success. */
            if (!v[1] && (ret >= 1)) {
              pthread_mutex_lock (&gui->seek.mutex);
              gui->seek.pos_time_length[0] = gui->seek.pos_time_length[1] = 0;
              pthread_mutex_unlock (&gui->seek.mutex);
              panel_update_runtime_display (gui->panel, _PURD_LOCK);
              osd_stream_position (gui, gui->seek.pos_time_length);
            } else if (xine_get_pos_length (gui->stream, v + 0, v + 1, v + 2)) {
              pthread_mutex_lock (&gui->seek.mutex);
              _v3cpy (gui->seek.pos_time_length, v);
              pthread_mutex_unlock (&gui->seek.mutex);
              if (ret >= 1) {
                panel_update_runtime_display (gui->panel, _PURD_LOCK);
                osd_stream_position (gui, gui->seek.pos_time_length);
              }
            }
            gui->ignore_next = 0;
          }

        }
      }
    }

    gui->ignore_next = 0;
    /* osd_hide_status (gui); */
    xitk_lock (gui->xitk, 1);
    panel_check_pause (gui->panel);
    xitk_lock (gui->xitk, 0);

    return NULL;
  } while (0);

  pthread_mutex_lock (&gui->seek.mutex);
  gui->seek.running = 0;
  gui->seek.newpos = -1;
  gui->seek.timestep = 0;
  pthread_mutex_unlock (&gui->seek.mutex);
  return NULL;
}

void gui_set_current_position (gGui_t *gui, int pos) {
  pthread_mutex_lock (&gui->seek.mutex);
  if (gui->seek.running == 0) {
    pthread_t pth;
    int err;
    gui->seek.running = 1;
    gui->seek.newpos = pos;
    pthread_mutex_unlock (&gui->seek.mutex);
    err = pthread_create (&pth, NULL, gui_seek_thread, gui);
    if (err) {
      printf (_("%s(): can't create new thread (%s)\n"), __XINE_FUNCTION__, strerror (err));
      pthread_mutex_lock (&gui->seek.mutex);
      gui->seek.running = 0;
      gui->seek.newpos = -1;
      gui->seek.timestep = 0;
      pthread_mutex_unlock (&gui->seek.mutex);
    }
  } else {
    gui->seek.newpos = pos;
    pthread_mutex_unlock (&gui->seek.mutex);
  }
}

void gui_seek_relative (gGui_t *gui, int off_sec) {
  if (off_sec == 0)
    return;

  pthread_mutex_lock (&gui->seek.mutex);
  if (gui->seek.running == 0) {
    pthread_t pth;
    int err;
    gui->seek.running = 1;
    gui->seek.timestep = off_sec * 1000;
    pthread_mutex_unlock (&gui->seek.mutex);
    err = pthread_create (&pth, NULL, gui_seek_thread, gui);
    if (err) {
      printf (_("%s(): can't create new thread (%s)\n"), __XINE_FUNCTION__, strerror (err));
      pthread_mutex_lock (&gui->seek.mutex);
      gui->seek.running = 0;
      gui->seek.newpos = -1;
      gui->seek.timestep = 0;
      pthread_mutex_unlock (&gui->seek.mutex);
    }
  } else {
    gui->seek.timestep = off_sec * 1000;
    pthread_mutex_unlock (&gui->seek.mutex);
  }
}

void gui_dndcallback (void *_gui, const char *filename) {
  gGui_t *gui = _gui;
  int n;

  n = gui_playlist_add_item (gui, filename, GUI_MAX_DIR_LEVELS, GUI_ITEM_TYPE_AUTO, 0);
  /* this includes the filname == NULL and filename == "" cases.
   * the latter often happens with KDE5 because that one \r\n terminates _all_ lines
   * of a text/uri-list. */
  if (!n)
    return;

  playlist_update_playlist (gui);
  oxine_playlist_update (gui->oxine);

  if (!(gui->playlist.control & PLAYLIST_CONTROL_IGNORE)) {
    if ((xine_get_status (gui->stream) == XINE_STATUS_STOP) || gui->logo_mode) {
      gui_current_set_index (gui, GUI_MMK_LAST - n + 1);
      if (gui->flags & XUI_FLAG_smart_mode)
        gui_play (NULL, gui);
    }
  }

  if (gui->playlist.num > 0)
    panel_playback_ctrl (gui->panel, 1);
}

void gui_nextprev_mrl (xitk_widget_t *w, void *data) {
  int by = 0;
  gGui_t *gui = _gui_get_nextprev (data, &by);

  (void)w;
  gui_playlist_start_next (gui, by);
}

int gui_layer_above (gGui_t *gui, xitk_window_t *xwin) {
  if (!gui)
    return 0;
  if ((gui->layer_above & 2) || ((gui->layer_above & 1) 
    && ((video_window_mode (gui->vwin, TOGGLE_MODE) & WINDOWED_MODE) || (video_window_is_visible (gui->vwin) <= 1)))) {
    int level = xitk_get_layer_level (gui->xitk);

    if (level <= 0)
      level = -1;
    if (xwin)
      xitk_window_set_window_layer (xwin, level);
    return level;
  }
  return 0;
}

int gui_set_amp_level (gGui_t *gui, int value) {
  if (value > GUI_AUDIO_VOL_RELATIVE / 2)
    value += gui->mixer.level[SOFTWARE_MIXER] - GUI_AUDIO_VOL_RELATIVE;
  if (value < 0)
    value = 0;
  else if (value > 200)
    value = 200;
  if (gui->mixer.level[SOFTWARE_MIXER] == value)
    return value;
  gui->mixer.level[SOFTWARE_MIXER] = value;
  xine_set_param (gui->stream, XINE_PARAM_AUDIO_AMP_LEVEL, value);
  panel_update_mixer_display (gui->panel);
  acontrol_update_mixer_display (gui->actrl);
  osd_bar (gui, _("Amplification Level"), 0, 200, value, OSD_BAR_STEPPER);
  return value;
}

int gui_set_audio_vol (gGui_t *gui, int value) {
  static const uint8_t range[LAST_MIXER] = {
    [SOUND_CARD_MIXER] = 100, [SOFTWARE_MIXER] = 200
  };
  static const int param[LAST_MIXER] = {
    [SOUND_CARD_MIXER] = XINE_PARAM_AUDIO_VOLUME, [SOFTWARE_MIXER] = XINE_PARAM_AUDIO_AMP_LEVEL
  };
  if (value > GUI_AUDIO_VOL_RELATIVE / 2)
    value += gui->mixer.level[gui->mixer.type_volume] - GUI_AUDIO_VOL_RELATIVE;
  if (value < 0)
    value = 0;
  else if (value > range[gui->mixer.type_volume])
    value = range[gui->mixer.type_volume];
  if (gui->mixer.level[gui->mixer.type_volume] == value)
    return value;
  gui->mixer.level[gui->mixer.type_volume] = value;
  xine_set_param (gui->stream, param[gui->mixer.type_volume], value);
  panel_update_mixer_display (gui->panel);
  if (gui->mixer.type_volume == SOFTWARE_MIXER)
    acontrol_update_mixer_display (gui->actrl);
  osd_bar (gui, _("Audio Volume"), 0, range[gui->mixer.type_volume], value, OSD_BAR_STEPPER);
  return value;
}

static void fileselector_cancel_callback (filebrowser_t *fb, void *userdata) {
  gGui_t *gui = userdata;

  if (fb == gui->load_stream) {
    if (filebrowser_get_string (fb, gui->curdir, sizeof (gui->curdir), 0))
      config_update_string (gui->xine, "media.files.origin_path", gui->curdir);
    gui->load_stream = NULL;
  }
  else if (fb == gui->load_sub)
    gui->load_sub = NULL;
}


static void fileselector_play (gGui_t *gui, int first) {
  if (gui->playlist.num <= 0)
    return;
  playlist_update_playlist (gui);
  oxine_playlist_update (gui->oxine);
  panel_playback_ctrl (gui->panel, 1);
  /* If an MRL is not being played, select the first file appended. If in "smart mode" start
     playing the entry.  If an MRL is currently being played, let it continue normally. */
  if ((first != gui->playlist.num) && (gui->logo_mode || (xine_get_status (gui->stream) == XINE_STATUS_STOP))) {
    gui->playlist.cur = first;
    if (gui->flags & XUI_FLAG_smart_mode) {
      gui_current_set_index (gui, GUI_MMK_CURRENT);
      gui_play (NULL, gui);
    }
  }
}

/* Callback function for file select button or double-click in file browser.
   Append selected file to the current playlist */
static void fileselector_callback (filebrowser_t *fb, void *userdata) {
  gGui_t *gui = userdata;
  char file[XITK_PATH_MAX + XITK_NAME_MAX + 2];

  /* Upate configuration with the selected directory path */
  if (filebrowser_get_string (fb, gui->curdir, sizeof (gui->curdir), 0))
    config_update_string (gui->xine, "media.files.origin_path", gui->curdir);

  /* Get the file path/name */
  if (filebrowser_get_string (fb, file, sizeof (file), 1)) {
    int first = gui->playlist.num;
    /* If the file has an extension which could be a playlist, attempt to append
       it to the current list as a list; otherwise, append it as an ordinary file. */
    gui_playlist_add_item (gui, file, 1, GUI_ITEM_TYPE_AUTO, 0);
    fileselector_play (gui, first);
  }

  gui->load_stream = NULL;
}


/* Callback function for "Select All" button in file browser. Append all files in the
   currently selected directory to the current playlist. */
static void fileselector_all_callback (filebrowser_t *fb, void *userdata) {
  gGui_t *gui = userdata;
  char path[XITK_PATH_MAX + XITK_NAME_MAX + 2], *add, *e = path + sizeof (path);

  add = path + filebrowser_get_string (fb, path, sizeof (path) - 3, 0);
  if (add > path) {
    const char * const *files;
    /* Update the configuration with the current path */
    strlcpy (gui->curdir, path, sizeof (gui->curdir));
    config_update_string (gui->xine, "media.files.origin_path", gui->curdir);
    /* Get all of the file names in the current directory as an array of pointers to strings */
    files = filebrowser_get_all_files (fb);
    if (files && files[0]) {
      int i, first = gui->playlist.num; /* current count of entries in playlist */

      if (add[-1] != '/')
        *add++ = '/';
      for (i = 0; files[i]; i++) {
        strlcpy (add, files[i], e - add);
        /* If the file has an extension which could be a playlist, attempt to append
           it to the current list as a list; otherwise, append it as an ordinary file. */
        gui_playlist_add_item (gui, path, 1, GUI_ITEM_TYPE_AUTO, 0);
      }
      fileselector_play (gui, first);
    }
  }

  gui->load_stream = NULL;
}


void gui_file_selector (gGui_t *gui) {
  gui->nongui_error_msg = NULL;
  if (gui->load_stream) {
    filebrowser_raise_window (gui->load_stream);
  } else {
    filebrowser_callback_button_t  cbb[3] = {
      {
        .label = _("Select"),
        .callback = fileselector_callback,
        .userdata = gui,
        .need_a_file = 0 /** << also take the dir itself. */
      }, {
        .callback = fileselector_cancel_callback,
        .userdata = gui
      }, {
        .label = _("Select all"),
        .callback = fileselector_all_callback,
        .userdata = gui
      }
    };
    gui->load_stream = filebrowser_create (gui, NULL,
      _("Stream(s) Loading"), gui->curdir, cbb, 3, XUI_EXTS_MEDIA);
  }
}

static void subselector_callback (filebrowser_t *fb, void *userdata) {
  gGui_t *gui = userdata;
  char file[XITK_PATH_MAX + XITK_NAME_MAX + 2];
  int ret;

  if (filebrowser_get_string (fb, file, sizeof (file), 1)) {
    mediamark_set_str_val (&gui->playlist.mmk[gui->playlist.cur], file, MMK_VAL_SUB);
    gui_current_set_index (gui, GUI_MMK_CURRENT);

    if (xine_get_status (gui->stream) == XINE_STATUS_PLAY) {
      int v[3];
      xine_close (gui->spu_stream);
      gui_messages_off (gui, 0);
      ret = xine_open (gui->spu_stream, file);
      gui_messages_on (gui);
      if (ret)
        xine_stream_master_slave (gui->stream,
          gui->spu_stream, XINE_MASTER_SLAVE_PLAY | XINE_MASTER_SLAVE_STOP);
      if (gui_xine_get_pos_length (gui, gui->stream, v)) {
        xine_stop (gui->stream);
        gui_set_current_position (gui, v[0]);
      }
    }
  }

  gui->load_sub = NULL;
}

void gui_select_sub (gGui_t *gui) {
  mediamark_t *cur;
  const char *path;
  char open_path[4096];

  if (gui->load_sub) {
    filebrowser_raise_window (gui->load_sub);
    return;
  }

  gui_playlist_lock (gui);

  cur = mediamark_get_current_mmk (gui);
  if (!cur) {
    gui_playlist_unlock (gui);
    return;
  }

  path = cur->sub ? cur->sub : cur->mrl;
  if (mrl_look_like_file (path)) {
    char *p;

    if (!strncasecmp (path, "file:", 5))
      path += 5;
    strlcpy (open_path, path, sizeof (open_path));
    gui_playlist_unlock (gui);
    p = strrchr (open_path, '/');
    if (p && p[0])
      *p = 0;
  } else {
    gui_playlist_unlock (gui);
    strlcpy (open_path, gui->curdir, sizeof (open_path));
  }
  {
    filebrowser_callback_button_t cbb[2] = {
      {
        .label = _("Select"),
        .callback = subselector_callback,
        .userdata = gui,
        .need_a_file = 1
      }, {
        .callback = fileselector_cancel_callback,
        .userdata = gui
      }
    };
    gui->load_sub = filebrowser_create (gui, NULL,
      _("Pick a subtitle file"), open_path, cbb, 2, XUI_EXTS_SUBTITLE);
  }
}

/*
 *
 */
void visual_anim_init (gGui_t *gui) {
  gui->visual_anim.num_mrls = 0;
  gui->visual_anim.mrls = malloc (sizeof (gui->visual_anim.mrls[0]));
  if (gui->visual_anim.mrls)
    gui->visual_anim.mrls[0] = NULL;
}

void visual_anim_done (gGui_t *gui) {
  if (gui->visual_anim.mrls) {
    int i;
    for (i = 0; i <= gui->visual_anim.num_mrls; i++)
      free (gui->visual_anim.mrls[i]);
    free (gui->visual_anim.mrls);
    gui->visual_anim.mrls = NULL;
  }
  gui->visual_anim.num_mrls = 0;
}

int visual_anim_add (gGui_t *gui, const char *mrl, int add) {
  if (add > 0) {
    char **n;
    if (!mrl)
      return gui->visual_anim.num_mrls;
    n = realloc (gui->visual_anim.mrls, sizeof (gui->visual_anim.mrls[0]) * (gui->visual_anim.num_mrls + 2));
    if (!n)
      return gui->visual_anim.num_mrls;
    gui->visual_anim.mrls = n;
    n[gui->visual_anim.num_mrls++] = strdup (mrl);
    n[gui->visual_anim.num_mrls]   = NULL;
  } else if (gui->visual_anim.mrls) {
    if (!mrl)
      mrl = XINE_VISDIR "/default.mpv";
    free (gui->visual_anim.mrls[gui->visual_anim.num_mrls]);
    gui->visual_anim.mrls[gui->visual_anim.num_mrls] = strdup (mrl);
  }
  return gui->visual_anim.num_mrls;
}

void visual_anim_play (gGui_t *gui, int stop_play_next) {
  int r;

  if (gui->visual_anim.enabled != 2)
    return;
  switch (stop_play_next) {
    case 0: /* stop */
      xine_stop (gui->visual_anim.stream);
      gui->visual_anim.running = 0;
      break;
    case 2: /* next */
      if (gui->visual_anim.num_mrls > 0) {
        gui->visual_anim.current++;
        if (gui->visual_anim.current >= gui->visual_anim.num_mrls)
          gui->visual_anim.current = 0;
      }
      /* fall through */
    case 1: /* play */
      gui_messages_off (gui, 0);
      r = xine_open (gui->visual_anim.stream, gui->visual_anim.mrls[gui->visual_anim.current]);
      gui_messages_on (gui);
      if (r) {
        gui->visual_anim.running = 1;
        if (xine_play (gui->visual_anim.stream, 0, 0))
          break;
      }
      gui_handle_xine_error (gui, gui->visual_anim.stream, gui->visual_anim.mrls[gui->visual_anim.current]);
      break;
    default: ;
  }
}
