/*
 * MCarray: Multidimensional Contiguous Array 
 *          numa@ees.hokudai.ac.jp
 *
 *         modified from
 *
 * Multi Dimensional Array extension module
 *   (C) Copyright by Akinori ITO
 *
 * Array in ruby is basically one-dimensional and extensible. In
 * scientific computation field, one often needs multi dimensional array
 * with fixed size. Of cource, you can simulate multi dimensional array
 * using Array-of-Array in standard ruby class. But it is inefficient,
 * and requires extra initialization. I decided to write extension module
 * for fixed multi-dimensional array to cope with this problem.
 */

#include <ruby.h>

/* struct for Fortran REAL multi-dimensional array */
struct MC_ARRAY {
    int type;        /* 0: int, 1: float */ 
    int dimension;   /* numbwer of dimension */
    int *size;       /* array size for each dimension */
    int total;       /* total size */
    union {
      int    *i;  /* body of the array; expressed as one-dimensitonal array */
      float  *f;  /* body of the array; expressed as one-dimensitonal array */
      double *d;  /* body of the array; expressed as one-dimensitonal array */
    } ptr;
};

#define MCA_Int    0
#define MCA_Float  1
#define MCA_Double 2

#define GetMCA(obj,var)  Data_Get_Struct(obj,struct MC_ARRAY, var)

static int
mca_index_pos_raw _((int argc, int *argv, struct MC_ARRAY *self));

/* global variables within this module */
static VALUE cMCArray;
static VALUE cMCIArray;
static VALUE cMCFArray;
static VALUE cMCDArray;

static void
mca_free(struct MC_ARRAY* self)
{
    free(self->size);
    switch(self->type) {
    case MCA_Int:
      free(self->ptr.f);
      break;
    case MCA_Float:
      free(self->ptr.i);
      break;
    case MCA_Double:
      free(self->ptr.d);
      break;
    }
}

/* allocation of MC_ARRAY */
static struct MC_ARRAY *
alloc_MC_ARRAY(int dim, int *sizes, int type) 
{
    int total_size = 1;
    int i;
    struct MC_ARRAY *ary;

    ary = ALLOC(struct MC_ARRAY);
    ary->size = ALLOC_N(int,dim);
    for (i = 0; i < dim; i++) {
	total_size *= sizes[i];
	ary->size[i] = sizes[i];
    }
    ary->dimension = dim;
    ary->total = total_size;
    ary->type = type;
    switch(type) {
    case MCA_Int:
      ary->ptr.i = ALLOC_N(int,total_size);
      break;
    case MCA_Float:
      ary->ptr.f = ALLOC_N(float,total_size);
      break;
    case MCA_Double:
      ary->ptr.d = ALLOC_N(double,total_size);
      break;
    }
    return ary;
}

/* search max dimension of an array */
static int
search_dim(struct RArray *ary)
{
    int dim,i,j,max;

    dim = 1;
    max = 0;
    for (i = 0; i < ary->len; i++) {
	if (TYPE(ary->ptr[i]) == T_ARRAY) {
	    j = search_dim(RARRAY(ary->ptr[i]));
	    if (j > max)
		max = j;
	} 
    }
    return dim+max;
}

/* search each dimension of an array */
static void
search_each_dim(struct RArray *ary, int dim, int thisdim, int *args)
{
    int i;

    if (ary->len > args[thisdim])
	args[thisdim] = ary->len;
    for (i = 0; i < ary->len; i++) {
	if (TYPE(ary->ptr[i]) == T_ARRAY) {
	    search_each_dim(RARRAY(ary->ptr[i]),dim,thisdim+1,args);
	}
    }
}

/* copy array to Mcarray */
static void
copy_ary_to_mca(struct RArray *ary, struct MC_ARRAY *fa,
		int dim, int thisdim, int *idx)
{
    int i;
    int pos;

    if (thisdim == dim-1) {
	i = 0;
	if (ary) {
	    for (; i < ary->len; i++) {
		idx[thisdim] = i;
		pos = mca_index_pos_raw(dim,idx,fa);
		switch(fa->type) {
		case MCA_Int:
		  if (FIXNUM_P(ary->ptr[i])) 
		    fa->ptr.i[pos] = (int)FIX2INT(ary->ptr[i]);
		  else
		    fa->ptr.i[pos] = (int)RFLOAT(ary->ptr[i])->value;
		  break;
		case MCA_Float:
		  if (FIXNUM_P(ary->ptr[i])) 
		    fa->ptr.f[pos] = (float)FIX2INT(ary->ptr[i]);
		  else
		    fa->ptr.f[pos] = (float)RFLOAT(ary->ptr[i])->value;
		  break;
		case MCA_Double:
		  if (FIXNUM_P(ary->ptr[i])) 
		    fa->ptr.d[pos] = (double)FIX2INT(ary->ptr[i]);
		  else
		    fa->ptr.d[pos] = (double)RFLOAT(ary->ptr[i])->value;
		  break;
		}
	    }
	}
    }
    else {
	i = 0;
	if (ary) {
	    for (; i < ary->len; i++) {
		idx[thisdim] = i;
		if (TYPE(ary->ptr[i]) == T_ARRAY)
		  copy_ary_to_mca(RARRAY(ary->ptr[i]),fa,dim,thisdim+1,idx);
		else
		  copy_ary_to_mca(0,fa,dim,thisdim+1,idx);
	    }
	}
	for (; i < fa->size[thisdim]; i++) {
	    idx[thisdim] = i;
	    copy_ary_to_mca(0,fa,dim,thisdim+1,idx);
	}
    }
}
	    

/* class method: to_mcarray(array) */
static VALUE
mca_typed_to_mcarray(VALUE self, VALUE ary, int type)
{
    int dim;
    int *args,*idx;
    struct MC_ARRAY *fa;
    int i;

    dim = search_dim(RARRAY(ary));
    args = ALLOCA_N(int,dim);  
    for (i = 0; i < dim; i++)
	args[i] = 0;
    search_each_dim(RARRAY(ary),dim,0,args);
    fa = alloc_MC_ARRAY(dim,args,type);
    idx = ALLOCA_N(int,dim);  
    for (i = 0; i < dim; i++)
	idx[i] = 0;
    copy_ary_to_mca(RARRAY(ary),fa,dim,0,idx);
    return Data_Wrap_Struct(cMCArray,0,mca_free,fa);
}

static VALUE
mca_i_to_mcarray(VALUE self, VALUE ary)
{
return mca_typed_to_mcarray(self, ary, MCA_Int);
}

static VALUE
mca_f_to_mcarray(VALUE self, VALUE ary)
{
return mca_typed_to_mcarray(self, ary, MCA_Float);
}

static VALUE
mca_d_to_mcarray(VALUE self, VALUE ary)
{
return mca_typed_to_mcarray(self, ary, MCA_Double);
}

/* class method: new(size1,size2,...,sizeN) */
static VALUE
mca_typed_new(int argc, VALUE *argv, VALUE class, int type)
{
    VALUE v;
    struct MC_ARRAY *fa;
    int *args;
    int dim,i;

    if (argc == 0)
	rb_raise(rb_eRuntimeError, "argument mismatch");
    if (TYPE(argv[0]) == T_ARRAY) {
	return mca_typed_to_mcarray(class,argv[0],type);
    }
    else {
	args = ALLOCA_N(int,argc);
	for (i = 0; i < argc; i++)
	    args[i] = NUM2INT(argv[i]);
	fa = alloc_MC_ARRAY(argc,args,type);
	return Data_Wrap_Struct(cMCArray,0,mca_free,fa);
    }
}

/* class method: new(size1,size2,...,sizeN) */
static VALUE
mca_i_new(int argc, VALUE *argv, VALUE class)
{
  return mca_typed_new(argc, argv, class, MCA_Int);
}

static VALUE
mca_f_new(int argc, VALUE *argv, VALUE class)
{
  return mca_typed_new(argc, argv, class, MCA_Float);
}

static VALUE
mca_d_new(int argc, VALUE *argv, VALUE class)
{
  return mca_typed_new(argc, argv, class, MCA_Double);
}

/* get index from multiple-index */
static int
mca_index_pos_raw(int argc, int *argv, struct MC_ARRAY *self)
{
    int pos = 0;
    int i;
    if (argc != self->dimension) {
	rb_raise(rb_eRuntimeError, "dimension mismatch");
    }
    for (i = argc-1; i >=0; i--) {
	int idx = argv[i];
	if (idx < 0 || self->size[i] <= idx)
	    rb_raise(rb_eRuntimeError, "Subsctipt out of range");
	pos = pos*self->size[i]+idx;
    }
    return pos;
}

/* get index from multiple-index */
static int
mca_index_pos(int argc, VALUE *argv, struct MC_ARRAY *self)
{
    int pos = 0;
    int i;
    if (argc != self->dimension) {
	rb_raise(rb_eRuntimeError, "dimension mismatch");
    }
    for (i = argc-1; i >=0; i--) {
	int idx = NUM2INT(argv[i]);
	if (idx < 0 || self->size[i] <= idx)
	    rb_raise(rb_eRuntimeError, "Subsctipt out of range");
	pos = pos*self->size[i]+idx;
    }
    return pos;
}

/* method: [](idx1,idx2,...,idxN) */
static VALUE
mca_fetch(int argc, VALUE *argv, VALUE self)
{
    struct MC_ARRAY *myself;
    int i;

    GetMCA(self,myself);
    switch(myself->type) {
    case MCA_Int:
      return INT2FIX(myself->ptr.i[mca_index_pos(argc,argv,myself)]);
      break;
    case MCA_Float:
      return rb_float_new(myself->ptr.f[mca_index_pos(argc,argv,myself)]);
      break;
    case MCA_Double:
      return rb_float_new(myself->ptr.d[mca_index_pos(argc,argv,myself)]);
      break;
    }  
}

/* method: []=(idx1,idx2,...,idxN,value) */
static VALUE
mca_store(int argc, VALUE *argv, VALUE self)
{
    struct MC_ARRAY *myself;
    int i;
    float f;
    double d;
    VALUE v = argv[argc-1];

    GetMCA(self,myself);
    switch(myself->type) {
    case MCA_Int:
      if (FIXNUM_P(v)) 
	i = FIX2INT(v);
      else
	i = (int)RFLOAT(v)->value;
      myself->ptr.i[mca_index_pos(argc-1,argv,myself)] = i;
      break;
    case MCA_Float:
      if (FIXNUM_P(v)) 
	f = (float)FIX2INT(v);
      else
	f = (float)RFLOAT(v)->value;
      myself->ptr.f[mca_index_pos(argc-1,argv,myself)] = f;
      break;
    case MCA_Double:
      if (FIXNUM_P(v)) 
	d = (double)FIX2INT(v);
      else
	d = (double)RFLOAT(v)->value;
      myself->ptr.d[mca_index_pos(argc-1,argv,myself)] = f;
      break;
    }
    return v;
}

/* method: size() -- returns an array of size of each dimension */
static VALUE
mca_size(VALUE obj)
{
    struct MC_ARRAY *myself;
    VALUE *sizes;
    int i;

    GetMCA(obj,myself);
    sizes = ALLOCA_N(VALUE,myself->dimension);
    for (i = 0; i < myself->dimension; i++)
	sizes[i] = INT2FIX(myself->size[i]);
    return rb_ary_new4(myself->dimension,sizes);
}

/* method: dimension() -- returns the dimension of the array */
static VALUE
mca_dimension(VALUE obj)
{
    struct MC_ARRAY *myself;
    GetMCA(obj,myself);
    return INT2FIX(myself->dimension);
}

/* iterator: each() */
static VALUE
mca_each(VALUE obj)
{
    struct MC_ARRAY *myself;
    int i;

    GetMCA(obj,myself);
    switch(myself->type) {
    case MCA_Int:
      for (i = 0; i < myself->total; i++)
	rb_yield(INT2FIX(myself->ptr.i[i]));
      break;
    case MCA_Float:
      for (i = 0; i < myself->total; i++)
	rb_yield(rb_float_new(myself->ptr.f[i]));
      break;
    case MCA_Double:
      for (i = 0; i < myself->total; i++)
	rb_yield(rb_float_new(myself->ptr.d[i]));
      break;
    }
    return Qnil;
}

/* iterator: each_index() */
static VALUE
mca_each_index(VALUE obj)
{
    struct MC_ARRAY *myself;
    int *size;
    struct RArray *vsize;
    int i,dim,loop;

    GetMCA(obj,myself);
    dim = myself->dimension;
    size = ALLOCA_N(int,dim);
    vsize = RARRAY(rb_ary_new2(dim));
    for (i = 0; i < dim; i++) {
	size[i] = 0;
	rb_ary_store(vsize,i,INT2FIX(0));
    }
    while (size[dim-1] < myself->size[dim-1]) {
	rb_yield(vsize);
	i = 0;
	do {
	    loop = 0;
	    size[i]++;
	    if (size[i] >= myself->size[i] && i < dim-1) {
		loop = 1;
		rb_ary_store(vsize,i,INT2FIX(0));
		size[i] = 0;
		i++;
	    }
	    else {
		rb_ary_store(vsize,i,INT2FIX(size[i]));
	    }
	} while (loop);
    }
    return Qnil;
}

#define SWAP(type,x,y) {type tmp; tmp=(x); (x)=(y); (y)=tmp; }

/* method: transpose([dim1,dim2]) */
static VALUE
mca_transpose(int argc, VALUE *argv, VALUE obj)
{
    int dim1, dim2, dim;
    struct MC_ARRAY *self;
    struct MC_ARRAY *newary;
    int *newsize;
    int *idx;
    int i,j,k,loop;

    GetMCA(obj,self);
    dim = self->dimension;
    if (argc == 0) {
	dim1 = 0;
	dim2 = dim-1;
    }
    else if (argc == 2) {
	dim1 = NUM2INT(argv[0]);
	dim2 = NUM2INT(argv[1]);
	if (dim1 < 0 || dim <= dim1 ||
	    dim2 < 0 || dim <= dim2)
	    rb_raise(rb_eRuntimeError, "Mcarray#transpose: illegal dimension spec");
	if (dim1 == dim2)
	    rb_raise(rb_eRuntimeError, "Mcarray#transpose: two dimensions are identical");
    }
    else
	rb_raise(rb_eRuntimeError, "Mcarray#transpose: argument number mismatch");
    
    newsize = ALLOCA_N(int,dim);
    idx = ALLOCA_N(int,dim);
    for (i = 0; i < dim; i++) {
	if (i == dim1) {
	    newsize[i] = self->size[dim2];
	}
	else if (i == dim2) {
	    newsize[i] = self->size[dim1];
	}
	else {
	    newsize[i] = self->size[i];
	}
	idx[i] = 0;
    }
    newary = alloc_MC_ARRAY(dim,newsize,self->type);

    while (idx[dim-1] < self->size[dim-1]) {
	i = 0;
	do {
	    loop = 0;
	    j = mca_index_pos_raw(dim,idx,self);
	    SWAP(int,idx[dim1],idx[dim2]);
	    k = mca_index_pos_raw(dim,idx,newary);
	    SWAP(int,idx[dim1],idx[dim2]);
	    switch (self->type) {
	    case MCA_Int:
	      newary->ptr.i[k] = self->ptr.i[j];
	      break;
	    case MCA_Float:	    
	      newary->ptr.f[k] = self->ptr.f[j];
	      break;
	    case MCA_Double:	    
	      newary->ptr.d[k] = self->ptr.d[j];
	      break;
	    }
	    idx[i]++;
	    if (idx[i] >= self->size[i] && i < dim-1) {
		loop = 1;
		idx[i] = 0;
		i++;
	    }
	} while (loop);
    }
    return Data_Wrap_Struct(cMCArray,0,mca_free,newary);
}


/* convert Mcarray to Array */
static VALUE
mca_to_array0(struct MC_ARRAY* fa, int *idx, int thisdim)
{
    int i;
    int pos;
    struct RArray *ary;

    ary = (struct RArray*)rb_ary_new2(fa->size[thisdim]);
    ary->len = fa->size[thisdim];
    if (thisdim == fa->dimension-1) {
	for (i = 0; i < fa->size[thisdim]; i++) {
	    idx[thisdim] = i;
	    pos = mca_index_pos_raw(fa->dimension,idx,fa);
	    switch(fa->type) {
	    case MCA_Int:
	      ary->ptr[i] = INT2FIX(fa->ptr.i[pos]);	
	      break;
	    case MCA_Float:	    
	      ary->ptr[i] = rb_float_new(fa->ptr.f[pos]);
	      break;
	    case MCA_Double:	    
	      ary->ptr[i] = rb_float_new(fa->ptr.d[pos]);
	      break;
	    }
	}
    }
    else {
	for (i = 0; i < fa->size[thisdim]; i++) {
	    idx[thisdim] = i;
	    ary->ptr[i] = mca_to_array0(fa,idx,thisdim+1);
	}
    }
    return (VALUE)ary;
}

static VALUE
mca_to_array(VALUE obj)
{
    struct MC_ARRAY *self;
    int *idx,i;

    GetMCA(obj,self);
    idx = ALLOCA_N(int,self->dimension);
    for (i = 0; i < self->dimension; i++) idx[i] = 0;
    return mca_to_array0(self,idx,0);
}

static VALUE
mca_clone(VALUE ary)
{
    VALUE ary2;
    struct MC_ARRAY *mca;
    struct MC_ARRAY *mca2;

    GetMCA(ary,mca);
    mca2 = alloc_MC_ARRAY(mca->dimension,mca->size,mca->type);
    switch(mca->type) {
    case MCA_Int:
      MEMCPY(mca2->ptr.i, mca->ptr.i, int, mca->total);
    case MCA_Float:
      MEMCPY(mca2->ptr.f, mca->ptr.f, float, mca->total);
    case MCA_Double:
      MEMCPY(mca2->ptr.d, mca->ptr.d, double, mca->total);
    }
    mca2->total = mca->total;
    ary2 = Data_Wrap_Struct(cMCArray,0,mca_free,mca2);
    CLONESETUP(ary2, ary);
    return ary2;
}

static VALUE
mca_inspect(VALUE obj)
{
    struct MC_ARRAY *myself;
    int i;

    GetMCA(obj,myself);
    switch(myself->type) {
    case MCA_Int:
      for (i = 0; i < myself->total; i++)
	printf("%d %d\n",i,myself->ptr.i[i]);
      break;
    case MCA_Float:
      for (i = 0; i < myself->total; i++)
	printf("%d %f\n",i,myself->ptr.f[i]);
      break;
    case MCA_Double:
      for (i = 0; i < myself->total; i++)
	printf("%d %f\n",i,myself->ptr.d[i]);
      break;
    }
    return Qnil;
}

/* initialization of this module */
Init_mcarray()
{
    /* define MCArray class */
    cMCArray = rb_define_class("MCArray",rb_cObject);

    /* methods */
    rb_define_method(cMCArray,"[]",mca_fetch,-1);
    rb_define_method(cMCArray,"[]=",mca_store,-1);
    rb_define_method(cMCArray,"size",mca_size,0);
    rb_define_method(cMCArray,"dimension",mca_dimension,0);
    rb_define_method(cMCArray,"each",mca_each,0);
    rb_define_method(cMCArray,"each_index",mca_each_index,0);
    rb_define_method(cMCArray,"transpose",mca_transpose,-1);
    rb_define_method(cMCArray,"to_array",mca_to_array,0);
    rb_define_method(cMCArray,"inspect",mca_inspect,0);
    rb_define_method(cMCArray,"clone",mca_clone,0);

    /* define MCIArray class */
    cMCIArray = rb_define_class("IntArray",cMCArray);

    /* class methods */
    rb_define_singleton_method(cMCIArray,"new",mca_i_new,-1);
    rb_define_singleton_method(cMCIArray,"to_intarray",mca_i_to_mcarray,1);

    /* define MCFArray class */
    cMCFArray = rb_define_class("FloatArray",cMCArray);

    /* class methods */
    rb_define_singleton_method(cMCFArray,"new",mca_f_new,-1);
    rb_define_singleton_method(cMCFArray,"to_floatarray",mca_f_to_mcarray,1);

    /* define MCDArray class */
    cMCDArray = rb_define_class("DoubleArray",cMCArray);

    /* class methods */
    rb_define_singleton_method(cMCDArray,"new",mca_f_new,-1);
    rb_define_singleton_method(cMCDArray,"to_doublearray",mca_f_to_mcarray,1);

}
