!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------
module Map_Function
!  Ͽ޺ɸϤˤ뤤餫Ѵؿ

interface ll2rt
  module procedure ll2rt_f, ll2rt_d
end interface ll2rt

interface ll2radi
  module procedure ll2radi_f, ll2radi_d
end interface ll2radi

interface dis2lon
  module procedure dis2lon_f, dis2lon_d
end interface dis2lon

interface dis2lat
  module procedure dis2lat_f, dis2lat_d
end interface dis2lat

interface rt2ll
  module procedure rt2ll_f, rt2ll_d
end interface rt2ll

interface lat2mery
  module procedure lat2mery_f, lat2mery_d
end interface lat2mery

interface ll2lamdis
  module procedure ll2lamdis_f, ll2lamdis_d
end interface ll2lamdis

interface ll2lamxy
  module procedure ll2lamxy_f, ll2lamxy_d
end interface ll2lamxy

interface ll2merdis
  module procedure ll2merdis_f, ll2merdis_d
end interface ll2merdis

interface lon2merx
  module procedure lon2merx_f, lon2merx_d
end interface lon2merx

interface vec2lam
  module procedure vec2lam_f, vec2lam_d
end interface vec2lam

interface x2merlon
  module procedure x2merlon_f, x2merlon_d
end interface x2merlon

interface xy2lamll
  module procedure xy2lamll_f, xy2lamll_d
end interface xy2lamll

interface y2merlat
  module procedure y2merlat_f, y2merlat_d
end interface y2merlat

interface Lambert_coe
  module procedure Lambert_coe_f, Lambert_coe_d
end interface Lambert_coe

contains

subroutine ll2rt_f( lon0, lat0, lon1, lat1, r, theta )
! ϵε̺ɸϤˤ, (lon0, lat0) ΰ֤Ȥ (r,\theta) ɸ
! ŸȤ, (lon1, lat1) ΰ֤ˤΥ r, Ʊ̳ theta 
! ׻. Υ r ϵ̾α߸̵ΥȤ, theta = 90, 270 deg ̤
! Ҹ̤̾褦 theta ֤.
! rt2ll εմؿ, Υ ll2radi ѤФ褤.
! ϿƤȤƤ, ̿ˡ˳.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lon0          ! ˺ɸη [rad]
  real, intent(in) :: lat0          ! ˺ɸΰ [rad]
  real, intent(in) :: lon1          ! ׻줿 [rad]
  real, intent(in) :: lat1          ! ׻줿 [rad]
  real, intent(inout) :: r          ! lon0 α߸̵Υ [m]
  real, intent(inout) :: theta      ! lon0 ǤƱ̳ [rad]
  real :: tmpcos, tmpsin, tmptan

  r=ll2radi_f( lon0, lat0, lon1, lat1 )

  if(r>0.0)then
     tmpcos=cos(lat1)*sin(lon1-lon0)
     tmpsin=sin(lat1)*cos(lat0)-cos(lat1)*sin(lat0)*cos(lon1-lon0)
     if(tmpcos==0.0.and.tmpsin==0.0)then
        theta=0.0
     else if(tmpcos==0.0.and.tmpsin>0.0)then
        theta=0.5*pi
     else if(tmpcos==0.0.and.tmpsin<0.0)then
        theta=1.5*pi
     else if(tmpcos>0.0.and.tmpsin==0.0)then
        theta=0.0
     else if(tmpcos<0.0.and.tmpsin==0.0)then
        theta=pi
     else
        if(tmpcos>0.0.and.tmpsin>0.0)then
           tmptan=tmpsin/tmpcos
           theta=atan(tmptan)
        else if(tmpcos<0.0.and.tmpsin>0.0)then
           tmptan=tmpsin/abs(tmpcos)
           theta=pi-atan(tmptan)
        else if(tmpcos>0.0.and.tmpsin<0.0)then
           tmptan=abs(tmpsin)/tmpcos
           theta=2.0*pi-atan(tmptan)
        else if(tmpcos<0.0.and.tmpsin<0.0)then
           tmptan=tmpsin/tmpcos
           theta=pi+atan(tmptan)
        end if
     end if
  else
     theta=0.0
  end if

end subroutine ll2rt_f

!-----------------------------------------------
!-----------------------------------------------

subroutine ll2rt_d( lon0, lat0, lon1, lat1, r, theta )
! ϵε̺ɸϤˤ, (lon0, lat0) ΰ֤Ȥ (r,\theta) ɸ
! ŸȤ, (lon1, lat1) ΰ֤ˤΥ r, Ʊ̳ theta 
! ׻. Υ r ϵ̾α߸̵ΥȤ, theta = 90, 270 deg ̤
! Ҹ̤̾褦 theta ֤.
! rt2ll εմؿ, Υ ll2radi ѤФ褤.
! ϿƤȤƤ, ̿ˡ˳.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lon0          ! ˺ɸη [rad]
  double precision, intent(in) :: lat0          ! ˺ɸΰ [rad]
  double precision, intent(in) :: lon1          ! ׻줿 [rad]
  double precision, intent(in) :: lat1          ! ׻줿 [rad]
  double precision, intent(inout) :: r          ! lon0 α߸̵Υ [m]
  double precision, intent(inout) :: theta      ! lon0 ǤƱ̳ [rad]
  double precision :: tmpcos, tmpsin, tmptan

  r=ll2radi_d( lon0, lat0, lon1, lat1 )

  if(r>0.0d0)then
     tmpcos=dcos(lat1)*dsin(lon1-lon0)
     tmpsin=dsin(lat1)*dcos(lat0)-dcos(lat1)*dsin(lat0)*dcos(lon1-lon0)
     if(tmpcos==0.0d0.and.tmpsin==0.0d0)then
        theta=0.0d0
     else if(tmpcos==0.0d0.and.tmpsin>0.0d0)then
        theta=0.5d0*pi_dp
     else if(tmpcos==0.0d0.and.tmpsin<0.0d0)then
        theta=1.5d0*pi_dp
     else if(tmpcos>0.0d0.and.tmpsin==0.0d0)then
        theta=0.0d0
     else if(tmpcos<0.0d0.and.tmpsin==0.0d0)then
        theta=pi_dp
     else
        if(tmpcos>0.0d0.and.tmpsin>0.0d0)then
           tmptan=tmpsin/tmpcos
           theta=datan(tmptan)
        else if(tmpcos<0.0d0.and.tmpsin>0.0d0)then
           tmptan=tmpsin/dabs(tmpcos)
           theta=pi_dp-datan(tmptan)
        else if(tmpcos>0.0d0.and.tmpsin<0.0d0)then
           tmptan=dabs(tmpsin)/tmpcos
           theta=2.0d0*pi_dp-datan(tmptan)
        else if(tmpcos<0.0d0.and.tmpsin<0.0d0)then
           tmptan=tmpsin/tmpcos
           theta=pi_dp+datan(tmptan)
        end if
     end if
  else
     theta=0.0d0
  end if

end subroutine ll2rt_d

!-----------------------------------------------
!-----------------------------------------------

subroutine rt2ll_f( r, theta, lon0, lat0, lon, lat )
! ϵε̺ɸϤˤ, (lon0, lat0) ΰ֤Ȥ (r,\theta) ɸ
! ŸȤ, Υ r, Ʊ̳ theta ΰ֤ˤ̾Ǥη٤
! ׻. Υ r ϵ̾α߸̵ΥȤ, theta = 90, 270 deg ̤
! Ҹ̤̾褦 theta ֤.
! ϿƤȤƤ, ̿ˡ˳.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: r          ! lon0 α߸̵Υ [m]
  real, intent(in) :: theta      ! lon0 ǤƱ̳ [rad]
  real, intent(in) :: lon0       ! ˺ɸη [rad]
  real, intent(in) :: lat0       ! ˺ɸΰ [rad]
  real, intent(inout) :: lon     ! ׻줿 [rad]
  real, intent(inout) :: lat     ! ׻줿 [rad]
  real :: thetad, lond, latd, tmplon, tmplat, rratio

  thetad=180.0*theta/pi
  lond=180.0*lon0/pi
  latd=180.0*lat0/pi
  rratio=r/radius

  do while(thetad>360.0)
     thetad=thetad-360.0
  end do

  if(thetad==-90.0.or.thetad==270.0)then
     lon=lon0
     lat=lat0-rratio
  else if(thetad==90.0)then
     lon=lon0
     lat=lat0+rratio
  else if((-90.0<thetad.and.90.0>thetad).or.  &
  &  (270.0<thetad.and.360.0>=thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0+asin(tmplon)
  else if((90.0<thetad.and.270.0>thetad).or.  &
  &       (-180.0<=thetad.and.-90.0>thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=-sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0-asin(tmplon)
  else
     write(*,*) "### ERROR : (rt2ll:Map_Function)"
     write(*,*) "argument 'theta' is not valid : ", theta
     write(*,*) "STOP."
     stop
  end if

end subroutine rt2ll_f

!-----------------------------------------------
!-----------------------------------------------

subroutine rt2ll_d( r, theta, lon0, lat0, lon, lat )
! ϵε̺ɸϤˤ, (lon0, lat0) ΰ֤Ȥ (r,\theta) ɸ
! ŸȤ, Υ r, Ʊ̳ theta ΰ֤ˤ̾Ǥη٤
! ׻. Υ r ϵ̾α߸̵ΥȤ, theta = 90, 270 deg ̤
! Ҹ̤̾褦 theta ֤.
! ϿƤȤƤ, ̿ˡ˳.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: r          ! lon0 α߸̵Υ [m]
  double precision, intent(in) :: theta      ! lon0 ǤƱ̳ [rad]
  double precision, intent(in) :: lon0       ! ˺ɸη [rad]
  double precision, intent(in) :: lat0       ! ˺ɸΰ [rad]
  double precision, intent(inout) :: lon     ! ׻줿 [rad]
  double precision, intent(inout) :: lat     ! ׻줿 [rad]
  double precision :: thetad, lond, latd, tmplon, tmplat, rratio

  thetad=180.0d0*theta/pi_dp
  lond=180.0d0*lon0/pi_dp
  latd=180.0d0*lat0/pi_dp
  rratio=r/radius_dp

  do while(thetad>360.0d0)
     thetad=thetad-360.0d0
  end do

  if(thetad==-90.0d0.or.thetad==270.0d0)then
     lon=lon0
     lat=lat0-rratio
  else if(thetad==90.0d0)then
     lon=lon0
     lat=lat0+rratio
  else if((-90.0d0<thetad.and.90.0d0>thetad).or.  &
  &  (270.0d0<thetad.and.360.0d0>=thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0+asin(tmplon)
  else if((90.0d0<thetad.and.270.0d0>thetad).or.  &
  &       (-180.0d0<=thetad.and.-90.0d0>thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=-sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0-asin(tmplon)
  else
     write(*,*) "### ERROR : (rt2ll:Map_Function)"
     write(*,*) "argument 'theta' is not valid : ", theta
     write(*,*) "STOP."
     stop
  end if

end subroutine rt2ll_d

!-----------------------------------------------
!-----------------------------------------------

real function ll2radi_f( lon1, lat1, lon2, lat2 )
! ̾Ǥ 2 Ǥΰ, ٤򸵤, 򽪻Ȥ߸̤εΥ
! ׻.
  use Phys_Const
  implicit none
  real, intent(in) :: lon1    !  1 Ǥη [rad]
  real, intent(in) :: lat1    !  1 Ǥΰ [rad]
  real, intent(in) :: lon2    !  2 Ǥη [rad]
  real, intent(in) :: lat2    !  2 Ǥΰ [rad]
  double precision :: lond1, lond2, latd1, latd2, tmp

  lond1=dble(lon1)
  lond2=dble(lon2)
  latd1=dble(lat1)
  latd2=dble(lat2)

  if(lond1==lond2.and.latd1==latd2)then
     ll2radi_f=0.0
  else
     tmp=sin(latd1)*sin(latd2)+cos(latd1)*cos(latd2)*cos(lond2-lond1)
     if(tmp<-1.0d0.or.tmp>1.0d0)then
        write(*,*) "*** ERROR (ll2radi) *** : Detect error", tmp, latd1, latd2, lond1, lond2
        stop
     end if
     ll2radi_f=real(acos(tmp))*radius
  end if

  return
end function ll2radi_f

!-----------------------------------------------
!-----------------------------------------------

real function ll2radi_d( lon1, lat1, lon2, lat2 )
! ̾Ǥ 2 Ǥΰ, ٤򸵤, 򽪻Ȥ߸̤εΥ
! ׻.
  use Phys_Const
  implicit none
  double precision, intent(in) :: lon1    !  1 Ǥη [rad]
  double precision, intent(in) :: lat1    !  1 Ǥΰ [rad]
  double precision, intent(in) :: lon2    !  2 Ǥη [rad]
  double precision, intent(in) :: lat2    !  2 Ǥΰ [rad]
  double precision :: lond1, lond2, latd1, latd2, tmp

  lond1=lon1
  lond2=lon2
  latd1=lat1
  latd2=lat2

  if(lond1==lond2.and.latd1==latd2)then
     ll2radi_d=0.0d0
  else
     tmp=sin(latd1)*sin(latd2)+cos(latd1)*cos(latd2)*cos(lond2-lond1)
     if(tmp<-1.0d0.or.tmp>1.0d0)then
        write(*,*) "*** ERROR (ll2radi) *** : Detect error", tmp, latd1, latd2, lond1, lond2
        stop
     end if
     ll2radi_d=real(acos(tmp))*radius_dp
  end if

  return
end function ll2radi_d

!-----------------------------------------------
!-----------------------------------------------

real function dis2lon_f( x, lon0, phi0 )
!   lon0 ֵΥ x [m] Υ줿֤ˤ [rad].
!  ,  phi0 [rad] ƱپǷ¬ΥѤ.
!  x ˷׻ݤ, ˷׻ݤͤͿз׻ǽ.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: x     !  lon0 ζֵΥ [m] ()
  real, intent(in) :: lon0  !  [rad]
  real, intent(in) :: phi0  !  [rad]

  dis2lon_f=x/(radius*cos(phi0))+lon0

  return
end function

!-----------------------------------------------

double precision function dis2lon_d( x, lon0, phi0 )
!   lon0 ֵΥ x [m] Υ줿֤ˤ [rad].
!  ,  phi0 [rad] ƱپǷ¬ΥѤ.
!  x ˷׻ݤ, ˷׻ݤͤͿз׻ǽ.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: x     !  lon0 ζֵΥ [m] ()
  double precision, intent(in) :: lon0  !  [rad]
  double precision, intent(in) :: phi0  !  [rad]

  dis2lon_d=x/(radius_dp*cos(phi0))+lon0

  return
end function

!---------------------------------

!---------------------------------

real function dis2lat_f( y, phi0 )
!   phi0 ֵΥ y [m] Υ줿֤ˤ [rad].
!  y ̤˷׻ݤ, ˷׻ݤͤͿз׻ǽ.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: y     !  phi0 ζֵΥ [m] (̸)
  real, intent(in) :: phi0  !  [rad]

  dis2lat_f=y/radius+phi0

  return
end function

!---------------------------------

!---------------------------------

double precision function dis2lat_d( y, phi0 )
!   phi0 ֵΥ y [m] Υ줿֤ˤ [rad].
!  y ̤˷׻ݤ, ˷׻ݤͤͿз׻ǽ.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: y     !  phi0 ζֵΥ [m] (̸)
  double precision, intent(in) :: phi0  !  [rad]

  dis2lat_d=y/radius_dp+phi0

  return
end function

! ʹ, ٷٺɸϿƺɸˤѴؿ
!---------------------------------
! 1. ʹߤϥ륫ȥɸ
!---------------------------------

real function x2merlon_f(x,lam0)
!   lam0 Ͽ޵Υ x [m] Υ줿֤ˤ [rad].
!  , x , Ϳз׻ǽ.
  use Phys_Const
  implicit none
  real, intent(in) :: x     !  lam0 ζֵΥ [m] ()
  real, intent(in) :: lam0  !  [rad]

  x2merlon_f=x/radius+lam0

  return
end function

!---------------------------------

double precision function x2merlon_d(x,lam0)
!   lam0 Ͽ޵Υ x [m] Υ줿֤ˤ [rad].
!  , x , Ϳз׻ǽ.
  use Phys_Const
  implicit none
  double precision, intent(in) :: x     !  lam0 ζֵΥ [m] ()
  double precision, intent(in) :: lam0  !  [rad]

  x2merlon_d=x/radius_dp+lam0

  return
end function

!---------------------------------

real function y2merlat_f(y,phi0)
!   phi0 ֵΥ y [m] Υ줿֤ˤ [rad].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: y     !  phi0 ζֵΥ [rad] (̸).
  real, intent(in) :: phi0  !  [rad]

  y2merlat_f=2.0*atan(exp(y/radius)*abs(tan(0.25*pi+0.5*phi0)))-0.5*pi

  return
end function

!---------------------------------

double precision function y2merlat_d(y,phi0)
!   phi0 ֵΥ y [m] Υ줿֤ˤ [rad].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: y     !  phi0 ζֵΥ [rad] (̸).
  double precision, intent(in) :: phi0  !  [rad]

  y2merlat_d=2.0d0*datan(dexp(y/radius_dp)*dabs(dtan(0.25d0*pi_dp+0.5d0*phi0)))-0.5d0*pi_dp

  return
end function

!---------------------------------
!---------------------------------

real function lon2merx_f(lam,lam0)
!   lam0  lam ޤǤϿ޵Υ x [m].
!  , x , Ϳз׻ǽ.
  use Phys_Const
  implicit none
  real, intent(in) :: lam   ! ׻ [rad]
  real, intent(in) :: lam0  !  [rad]

  lon2merx_f=radius*(lam-lam0)

  return
end function

!---------------------------------
!---------------------------------

double precision function lon2merx_d(lam,lam0)
!   lam0  lam ޤǤϿ޵Υ x [m].
!  , x , Ϳз׻ǽ.
  use Phys_Const
  implicit none
  double precision, intent(in) :: lam   ! ׻ [rad]
  double precision, intent(in) :: lam0  !  [rad]

  lon2merx_d=radius_dp*(lam-lam0)

  return
end function

!---------------------------------
!---------------------------------

real function lat2mery_f(phi,phi0)
!   phi0  phi ޤǤϿ޵Υ y [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: phi   ! ׻ [rad]
  real, intent(in) :: phi0  !  [rad]

  lat2mery_f=radius*log(abs(tan(0.25*pi+0.5*phi)/tan(0.25*pi+0.5*phi0)))

  return
end function

!---------------------------------
!---------------------------------

double precision function lat2mery_d(phi,phi0)
!   phi0  phi ޤǤϿ޵Υ y [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: phi   ! ׻ [rad]
  double precision, intent(in) :: phi0  !  [rad]

  lat2mery_d=radius_dp*dlog(dabs(dtan(0.25d0*pi_dp+0.5d0*phi)/dtan(0.25d0*pi_dp+0.5d0*phi0)))

  return
end function

!---------------------------------
!---------------------------------

real function ll2merdis_f(lambda1,phi1,lambda2,phi2,lambda0,phi0)
!  lambda1, phi1  lambda2, phi2 ޤǤεΥ [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda1  ! Ǥ 1 Ǥη [rad]
  real, intent(in) :: phi1     ! Ǥ 1 Ǥΰ [rad]
  real, intent(in) :: lambda2  ! Ǥ 2 Ǥη [rad]
  real, intent(in) :: phi2     ! Ǥ 2 Ǥΰ [rad]
  real, intent(in) :: lambda0  ! ̺ɸθб [rad]
  real, intent(in) :: phi0     ! ̺ɸθб [rad]

  real :: x1, x2, y1, y2

  x1=lon2merx_f( lambda1, lambda0 )
  x2=lon2merx_f( lambda2, lambda0 )
  y1=lat2mery_f( phi1, phi0 )
  y2=lat2mery_f( phi2, phi0 )

  ll2merdis_f=sqrt((x2-x1)**2+(y2-y1)**2)

  return
end function

!---------------------------------
!---------------------------------

double precision function ll2merdis_d(lambda1,phi1,lambda2,phi2,lambda0,phi0)
!  lambda1, phi1  lambda2, phi2 ޤǤεΥ [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda1  ! Ǥ 1 Ǥη [rad]
  double precision, intent(in) :: phi1     ! Ǥ 1 Ǥΰ [rad]
  double precision, intent(in) :: lambda2  ! Ǥ 2 Ǥη [rad]
  double precision, intent(in) :: phi2     ! Ǥ 2 Ǥΰ [rad]
  double precision, intent(in) :: lambda0  ! ̺ɸθб [rad]
  double precision, intent(in) :: phi0     ! ̺ɸθб [rad]

  double precision :: x1, x2, y1, y2

  x1=lon2merx_d( lambda1, lambda0 )
  x2=lon2merx_d( lambda2, lambda0 )
  y1=lat2mery_d( phi1, phi0 )
  y2=lat2mery_d( phi2, phi0 )

  ll2merdis_d=dsqrt((x2-x1)**2+(y2-y1)**2)

  return
end function

!---------------------------------
! 2. ʹߤϥ٥Ⱥɸ
!---------------------------------

subroutine xy2lamll_f( x, y, lambda, phi, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!   lambda0,  phi1, phi2 ̤ˤ,
!   lambda0,  phi0 Ͽ޺ɸθȤȤ,
!  Ͽ޺ɸ (x, y) [m] Ǥΰٷ [rad].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: x          ! Ͽ޺ɸʬ [m]
  real, intent(in) :: y          ! Ͽ޺ɸʬ [m]
  real, intent(inout) :: lambda  ! (x,y) б [rad]
  real, intent(inout) :: phi     ! (x,y) б [rad]
  real, intent(in) :: lambda0    !  [rad]
  real, intent(in) :: phi0       !  [rad]
  real, intent(in) :: phi1       ! ٥ƴ [rad]
  real, intent(in) :: phi2       ! ٥ƴ [rad]
  real, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  real, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  real, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  real :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, w0

  cos1=cos(phi1)
  cos2=cos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=tan(0.25*pi+0.5*phi0)
     tan1=tan(0.25*pi+0.5*phi1)
     tan2=tan(0.25*pi+0.5*phi2)
     ncoe=log(cos1/cos2)/log(tan2/tan1)
     w0=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tan0)))
  end if

  lambda=lambda0+atan(-x/(y-w0))/ncoe
  phi=2.0*atan(exp((1.0/ncoe)*log(radius*cos1/(ncoe*sqrt(x**2+(y-w0)**2))))*tan1)-0.5*pi

end subroutine

!---------------------------------
!---------------------------------

subroutine xy2lamll_d( x, y, lambda, phi, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!   lambda0,  phi1, phi2 ̤ˤ,
!   lambda0,  phi0 Ͽ޺ɸθȤȤ,
!  Ͽ޺ɸ (x, y) [m] Ǥΰٷ [rad].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: x          ! Ͽ޺ɸʬ [m]
  double precision, intent(in) :: y          ! Ͽ޺ɸʬ [m]
  double precision, intent(inout) :: lambda  ! (x,y) б [rad]
  double precision, intent(inout) :: phi     ! (x,y) б [rad]
  double precision, intent(in) :: lambda0    !  [rad]
  double precision, intent(in) :: phi0       !  [rad]
  double precision, intent(in) :: phi1       ! ٥ƴ [rad]
  double precision, intent(in) :: phi2       ! ٥ƴ [rad]
  double precision, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  double precision, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  double precision, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  double precision :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, w0

  cos1=cos(phi1)
  cos2=cos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=dtan(0.25d0*pi_dp+0.5d0*phi0)
     tan1=dtan(0.25d0*pi_dp+0.5d0*phi1)
     tan2=dtan(0.25d0*pi_dp+0.5d0*phi2)
     ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)
     w0=radius_dp*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tan0)))
  end if

  lambda=lambda0+datan(-x/(y-w0))/ncoe
  phi=2.0d0*datan(dexp((1.0d0/ncoe)*dlog(radius_dp*cos1/(ncoe*dsqrt(x**2+(y-w0)**2))))*tan1)-0.5d0*pi_dp

end subroutine

!---------------------------------
!---------------------------------

subroutine ll2lamxy_f( lambda, phi, x, y, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!   lambda0,  phi1, phi2 ̤ˤ,
!   lambda0,  phi0 Ͽ޺ɸθȤȤ,
!  ̾Ǥΰٷ (phi,lambda) бϿ޺ɸ (x, y) [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda   ! (x,y) б [rad]
  real, intent(in) :: phi      ! (x,y) б [rad]
  real, intent(inout) :: x     ! Ͽ޺ɸʬ [m]
  real, intent(inout) :: y     ! Ͽ޺ɸʬ [m]
  real, intent(in) :: lambda0  !  [rad]
  real, intent(in) :: phi0       !  [rad]
  real, intent(in) :: phi1       ! ٥ƴ [rad]
  real, intent(in) :: phi2       ! ٥ƴ [rad]
  real, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  real, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  real, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  real :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, tanx, w0, w

  cos1=cos(phi1)
  cos2=cos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=tan(0.25*pi+0.5*phi0)
     tan1=tan(0.25*pi+0.5*phi1)
     tan2=tan(0.25*pi+0.5*phi2)
     ncoe=log(cos1/cos2)/log(tan2/tan1)
     w0=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tan0)))
  end if

  tanx=tan(0.25*pi+0.5*phi)
  w=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tanx)))

  x=w*sin(ncoe*(lambda-lambda0))
  y=w0-w*cos(ncoe*(lambda-lambda0))

end subroutine

!---------------------------------
!---------------------------------

subroutine ll2lamxy_d( lambda, phi, x, y, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!   lambda0,  phi1, phi2 ̤ˤ,
!   lambda0,  phi0 Ͽ޺ɸθȤȤ,
!  ̾Ǥΰٷ (phi,lambda) бϿ޺ɸ (x, y) [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda   ! (x,y) б [rad]
  double precision, intent(in) :: phi      ! (x,y) б [rad]
  double precision, intent(inout) :: x     ! Ͽ޺ɸʬ [m]
  double precision, intent(inout) :: y     ! Ͽ޺ɸʬ [m]
  double precision, intent(in) :: lambda0  !  [rad]
  double precision, intent(in) :: phi0       !  [rad]
  double precision, intent(in) :: phi1       ! ٥ƴ [rad]
  double precision, intent(in) :: phi2       ! ٥ƴ [rad]
  double precision, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  double precision, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  double precision, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  double precision :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, tanx, w0, w

  cos1=dcos(phi1)
  cos2=dcos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=dtan(0.25d0*pi_dp+0.5d0*phi0)
     tan1=dtan(0.25d0*pi_dp+0.5d0*phi1)
     tan2=dtan(0.25d0*pi_dp+0.5d0*phi2)
     ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)
     w0=radius_dp*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tan0)))
  end if

  tanx=dtan(0.25d0*pi_dp+0.5d0*phi)

  w=radius_dp*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tanx)))

  x=w*dsin(ncoe*(lambda-lambda0))
  y=w0-w*dcos(ncoe*(lambda-lambda0))

end subroutine

!---------------------------------
!---------------------------------

real function ll2lamdis_f( lambda1, phi1, lambda2, phi2,  &
  &                        lambda0, phit1, phit2, phi0,  &
  &                        nopt, uopt, ropt )
!   phit1, phit2,  lambda0 Ͽ޺ɸθ
!  phi0,  lambda 0 ȤȤ, ٷٺɸ 2 
!  (lambda1,phi1), (lambda2,phi2) ֤Ͽ޾ǤεΥ [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda1 ! ᤿ 1 η [rad].
  real, intent(in) :: phi1    ! ᤿ 1 ΰ [rad].
  real, intent(in) :: lambda2 ! ᤿ 2 η [rad].
  real, intent(in) :: phi2    ! ᤿ 2 ΰ [rad].
  real, intent(in) :: lambda0 ! ٤Ͽ޺ɸ [rad]
  real, intent(in) :: phit1   !  1 [rad]
  real, intent(in) :: phit2   !  2 [rad]
  real, intent(in) :: phi0    ! Ͽ޺ɸ 0 [rad].
  real, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  real, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  real, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f
  real :: ncoe, tan1, tan2, tan0, w0, cos1, cos2, x1, y1, x2, y2

  cos1=cos(phit1)
  cos2=cos(phit2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=tan(0.25*pi+0.5*phi0)
     tan1=tan(0.25*pi+0.5*phit1)
     tan2=tan(0.25*pi+0.5*phit2)
     ncoe=log(cos1/cos2)/log(tan2/tan1)
     w0=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tan0)))
  end if

  call ll2lamxy_f( lambda1, phi1, x1, y1, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )
  call ll2lamxy_f( lambda2, phi2, x2, y2, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )

  ll2lamdis_f=sqrt((x2-x1)**2+(y2-y1)**2)

  return

end function

!---------------------------------
!---------------------------------

double precision function ll2lamdis_d( lambda1, phi1, lambda2, phi2,  &
  &                                    lambda0, phit1, phit2, phi0,  &
  &                                    nopt, uopt, ropt )
!   phit1, phit2,  lambda0 Ͽ޺ɸθ
!  phi0,  lambda 0 ȤȤ, ٷٺɸ 2 
!  (lambda1,phi1), (lambda2,phi2) ֤Ͽ޾ǤεΥ [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda1 ! ᤿ 1 η [rad].
  double precision, intent(in) :: phi1    ! ᤿ 1 ΰ [rad].
  double precision, intent(in) :: lambda2 ! ᤿ 2 η [rad].
  double precision, intent(in) :: phi2    ! ᤿ 2 ΰ [rad].
  double precision, intent(in) :: lambda0 ! ٤Ͽ޺ɸ [rad]
  double precision, intent(in) :: phit1   !  1 [rad]
  double precision, intent(in) :: phit2   !  2 [rad]
  double precision, intent(in) :: phi0    ! Ͽ޺ɸ 0 [rad].
  double precision, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  double precision, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  double precision, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f
  double precision:: ncoe, tan1, tan2, tan0, w0, cos1, cos2, x1, y1, x2, y2

  cos1=dcos(phit1)
  cos2=dcos(phit2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=dtan(0.25d0*pi_dp+0.5d0*phi0)
     tan1=dtan(0.25d0*pi_dp+0.5d0*phit1)
     tan2=dtan(0.25d0*pi_dp+0.5d0*phit2)
     ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)
     w0=radius*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tan0)))
  end if

  call ll2lamxy_d( lambda1, phi1, x1, y1, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )
  call ll2lamxy_d( lambda2, phi2, x2, y2, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )

  ll2lamdis_d=dsqrt((x2-x1)**2+(y2-y1)**2)

  return

end function

!---------------------------------
!---------------------------------

subroutine vec2lam_f( lambda0, phi1, phi2, lambda, phi,  &
  &                   ui, vi, uo, vo, conv_flag, undef )
! ʿ٥ȥʬٷٺɸϿ޺ɸѴ (٥Ⱥɸ).
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda0     ! ٥ƴ [rad]
  real, intent(in) :: phi1        ! ٥ƴ 1 [rad]
  real, intent(in) :: phi2        ! ٥ƴ 2 [rad]
  real, intent(in) :: lambda(:,:) ! ®٤Ƥ [rad]
  real, intent(in) :: phi(:,:)    ! ®٤Ƥ [rad]
  real, intent(in) :: ui(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ x ®ʬ [m/s] (Ѵ)
  real, intent(in) :: vi(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ y ®ʬ [m/s] (Ѵ)
  real, intent(inout) :: uo(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ x ®ʬ [m/s] (Ѵ)
  real, intent(inout) :: vo(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ y ®ʬ [m/s] (Ѵ)
  character(5), intent(in) :: conv_flag  ! Ѵ
                              ! 'll2xy' = ٷٺɸ٥ƺɸ
                              ! 'xy2ll' = ٥ƺɸٷٺɸ

  real, intent(in), optional :: undef   ! ̤

  integer :: ix, jy, i, j
  real :: rundef, cos1, cos2, tan1, tan2, ncoe, cosp, sinp

  ix=size(phi,1)
  jy=size(phi,2)

  cos1=cos(phi1)
  cos2=cos(phi2)
  tan1=tan(0.25*pi+0.5*phi1)
  tan2=tan(0.25*pi+0.5*phi2)
  ncoe=log(cos1/cos2)/log(tan2/tan1)

  if(present(undef))then

     uo=undef
     vo=undef

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=sin(ncoe*(lambda(i,j)-lambda0))
                 cosp=cos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
                 vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=sin(ncoe*(lambda(i,j)-lambda0))
                 cosp=cos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
                 vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     end if

  else

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              sinp=sin(ncoe*(lambda(i,j)-lambda0))
              cosp=cos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
              vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              sinp=sin(ncoe*(lambda(i,j)-lambda0))
              cosp=cos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
              vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     end if

  end if

end subroutine

!---------------------------------
!---------------------------------

subroutine vec2lam_d( lambda0, phi1, phi2, lambda, phi,  &
  &                   ui, vi, uo, vo, conv_flag, undef )
! ʿ٥ȥʬٷٺɸϿ޺ɸѴ (٥Ⱥɸ).
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda0     ! ٥ƴ [rad]
  double precision, intent(in) :: phi1        ! ٥ƴ 1 [rad]
  double precision, intent(in) :: phi2        ! ٥ƴ 2 [rad]
  double precision, intent(in) :: phi(:,:)    ! ®٤Ƥ [rad]
  double precision, intent(in) :: lambda(:,:) ! ®٤Ƥ [rad]
  double precision, intent(in) :: ui(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ x ®ʬ [m/s] (Ѵ)
  double precision, intent(in) :: vi(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ y ®ʬ [m/s] (Ѵ)
  double precision, intent(inout) :: uo(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ x ®ʬ [m/s] (Ѵ)
  double precision, intent(inout) :: vo(size(phi,1),size(phi,2))
                      !  or Ͽ޺ɸ y ®ʬ [m/s] (Ѵ)
  character(5), intent(in) :: conv_flag  ! Ѵ
                              ! 'll2xy' = ٷٺɸ٥ƺɸ
                              ! 'xy2ll' = ٥ƺɸٷٺɸ

  double precision, intent(in), optional :: undef   ! ̤

  integer :: ix, jy, i, j
  double precision :: rundef, cos1, cos2, tan1, tan2, ncoe, cosp, sinp

  ix=size(phi,1)
  jy=size(phi,2)

  cos1=dcos(phi1)
  cos2=dcos(phi2)
  tan1=dtan(0.25d0*pi_dp+0.5d0*phi1)
  tan2=dtan(0.25d0*pi_dp+0.5d0*phi2)
  ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)

  if(present(undef))then

     uo=undef
     vo=undef

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=dsin(ncoe*(lambda(i,j)-lambda0))
                 cosp=dcos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
                 vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=dsin(ncoe*(lambda(i,j)-lambda0))
                 cosp=dcos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
                 vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     end if

  else

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              sinp=dsin(ncoe*(lambda(i,j)-lambda0))
              cosp=dcos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
              vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              sinp=dsin(ncoe*(lambda(i,j)-lambda0))
              cosp=dcos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
              vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     end if

  end if

end subroutine

!---------------------------------
!---------------------------------

subroutine lambert_coe_f( lambda0, phi0, phi1, phi2, ncoe, uphi, rhophi0 )
!   phi1, phi2,  lambda0, ƺɸθ phi0
!  ξˤ, Ƽ Lambert ѿ׻ݤɬץѥ᡼
!  n, U, rho .
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda0        ! ٤Ͽ޺ɸ [rad]
  real, intent(in) :: phi1           !  1 [rad]
  real, intent(in) :: phi2           !  2 [rad]
  real, intent(in) :: phi0           ! Ͽ޺ɸ 0 [rad].
  real, intent(inout) :: ncoe        ! ̼ܰ N
  real, intent(inout) :: uphi(0:2)   ! ܷ U(phi0,phi1,phi2)
  real, intent(inout) :: rhophi0     ! Ⱦ r(phi0)

  real :: cos1, cos2

  cos1=cos(phi1)
  cos2=cos(phi2)

  uphi(0)=tan(0.25*pi+0.5*phi0)
  uphi(1)=tan(0.25*pi+0.5*phi1)
  uphi(2)=tan(0.25*pi+0.5*phi2)
  ncoe=log(cos1/cos2)/log(uphi(2)/uphi(1))
  rhophi0=radius*(cos1/ncoe)*exp(ncoe*log(uphi(1)/uphi(0)))

end subroutine

!---------------------------------
!---------------------------------

subroutine lambert_coe_d( lambda0, phi0, phi1, phi2, ncoe, uphi, rhophi0 )
!   phi1, phi2,  lambda0, ƺɸθ phi0
!  ξˤ, Ƽ Lambert ѿ׻ݤɬץѥ᡼
!  n, U, rho .
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda0        ! ٤Ͽ޺ɸ [rad]
  double precision, intent(in) :: phi1           !  1 [rad]
  double precision, intent(in) :: phi2           !  2 [rad]
  double precision, intent(in) :: phi0           ! Ͽ޺ɸ 0 [rad].
  double precision, intent(inout) :: ncoe        ! ̼ܰ N
  double precision, intent(inout) :: uphi(0:1)   ! ܷ U(phi0,phi1)
  double precision, intent(inout) :: rhophi0     ! Ⱦ r(phi0)

  double precision :: cos1, cos2, tanx

  cos1=dcos(phi1)
  cos2=dcos(phi2)
  tanx=dtan(0.25d0*pi_dp+0.5d0*phi2)

  uphi(0)=dtan(0.25d0*pi_dp+0.5d0*phi0)
  uphi(1)=dtan(0.25d0*pi_dp+0.5d0*phi1)
  ncoe=dlog(cos1/cos2)/dlog(tanx/uphi(1))
  rhophi0=radius_dp*(cos1/ncoe)*dexp(ncoe*dlog(uphi(1)/uphi(0)))

end subroutine

!---------------------------------
!---------------------------------


end module
