program initial_make
!  モデルの計算に必要な初期値ファイルを生成するプログラム
!  必要に応じてソースの書き換えが可能.
!  読み込むサウンディングで鉛直格子を決定する.
!  壁雲の数を複数設置する.
  use file_operate
  use Thermo_Const
  use Math_Const
  use Phys_Const
  use Thermo_Function
  use Thermo_Routine
  use Algebra
  use Typhoon_analy
  use gtool_history
  use derivation
  use max_min

  implicit none

!-- namelist valiables
  integer :: nr, nz
  integer :: mom_flag
  real :: coril
  real :: dr, dz
  character(4) :: bc
  character(80) :: fname
  character(80) :: sound_name  ! サウンディングファイルの名前

!-- internal valiables
  integer :: i, j
  real, allocatable, dimension(:) :: r, z, zd  ! 軸対称 2 次元の動径, 鉛直座標
  real, allocatable, dimension(:) :: rm, vm  ! 各高度での最大風速と MWR
  real, allocatable, dimension(:) :: r1, r2  ! inner, outer core boundary
  real, allocatable, dimension(:) :: pres_s, temp_s, pt_s, rho_s  ! サウンディング
  real, allocatable, dimension(:,:) :: v0, pres, qv  ! 初期の接線風, 気圧, 加熱率
  real, allocatable, dimension(:,:) :: mom  ! 運動量ソース
  real, allocatable, dimension(:,:) :: theta0, rho  ! 初期の温位, 密度
  real, allocatable, dimension(:,:) :: x  ! 座標変数
  real, allocatable, dimension(:,:) :: coriol  ! コリオリパラメータ
  real, allocatable, dimension(:,:) :: N2  ! 浮力振動数 **2
  character(20), allocatable :: val(:,:)
  real :: nibun_left, nibun_right, func_left, func_right
  real :: err, max_val
  real, allocatable, dimension(:) :: ac

!-- constant value
  integer, parameter :: sline=2  ! サウンディングファイルの読み飛ばし行
  integer, parameter :: n=1  ! eye での速度の傾き
  integer, parameter :: m_bell=2  ! 非断熱加熱のベル関数の次数
  real, parameter :: x1=300.0e3
  real, parameter :: r_qv=-2.0e3   ! qv と rm の距離
  real, parameter :: dRm=16.0/18.0  ! Rmax の傾き (dr/dz)
  real, parameter :: rmax=50.0e3  ! 地表面での Rmax
  real, parameter :: vmax=50.0  ! 地表面での Vmax
  real, parameter :: z_v0=18.0e3  ! vmax がゼロになる高度 (地表からここまで線形に減少する)
  real, parameter :: z_qv=18.0e3  ! qv がゼロになる高度
  real, parameter :: dr12=20.0e3  ! r1 と r2 の距離
  real, parameter :: err_max=1.0e-5  ! 二分法の収束条件

!-- 多重壁雲の場合
  real, allocatable, dimension(:) :: rm2, vm2  ! 各高度での最大風速と MWR
  real, parameter :: dRm2=128.0/18.0  ! Rmax の傾き (dr/dz)
  real, parameter :: rmax2=100.0e3  ! 地表面での Rmax
  real, parameter :: vmax2=50.0  ! 地表面での Vmax
  real, parameter :: dr122=40.0e3  ! r1 と r2 の距離
  real, allocatable, dimension(:) :: r12, r22  ! inner, outer core boundary
  real, allocatable, dimension(:,:) :: x2  ! 座標変数

!-- function name
  real :: func_a, Bell

!-- namelist からのパラメータの代入
  namelist /input /nr, nz, dr, dz, bc, fname, sound_name, coril, mom_flag
  read(5,nml=input)

!-- 配列の割付
  allocate(r(nr))
  allocate(z(nz))
  allocate(zd(nz))  ! 非断熱加熱の分布計算で用いる.
  allocate(rm(nz))
  allocate(vm(nz))
  allocate(r1(nz))
  allocate(r2(nz))
  allocate(pres_s(nz))
  allocate(temp_s(nz))
  allocate(pt_s(nz))
  allocate(rho_s(nz))
  allocate(val(4,nz))
  allocate(v0(nr,nz))
  allocate(pres(nr,nz))
  allocate(qv(nr,nz))
  allocate(mom(nr,nz))
  allocate(theta0(nr,nz))
  allocate(rho(nr,nz))
  allocate(x(nr,nz))
  allocate(coriol(nr,nz))
  allocate(N2(nr,nz))
  allocate(ac(nz))
  allocate(rm2(nz))
  allocate(vm2(nz))
  allocate(r12(nz))
  allocate(r22(nz))
  allocate(x2(nr,nz))

  coriol=2.0*omega*sin(coril*pi/180.0)
  mom=0.0  ! 現在のところ, 運動量ソースは initial_make では陽に作成しない.

!-- サウンディングから基本場の熱力学変数・鉛直座標を設定.
!-- サウンディングのデータ数は namelist で設定されているのと同じ値.

  call read_file_text( trim(sound_name), 4, nz, val, skip=sline )

  do i=1,nz
     read(val(1,i),*) z(i)
     read(val(2,i),*) temp_s(i)
     read(val(3,i),*) pres_s(i)
     read(val(4,i),*) pt_s(i)
     rho_s(i)=TP_2_rho( temp_s(i), pres_s(i) )
write(*,*) rho_s(i)
  end do

!-- 座標値の設定
  r=(/(dr*real(i-1),i=1,nr)/)

!-- 傾度風場の生成に必要なパラメータの設定
  rm(1)=rmax
  vm(1)=vmax
  rm2(1)=rmax2
  vm2(1)=vmax2
  do j=2,nz
     rm(j)=rm(1)+dRm*z(j)
     rm2(j)=rm2(1)+dRm2*z(j)
     if(z(j)<1000.0)then
        vm(j)=vmax*(z(j)/1000.0)
        vm2(j)=vmax2*(z(j)/1000.0)
     else
        vm(j)=vmax-vmax*(z(j)/z_v0)
        vm2(j)=vmax2-vmax2*(z(j)/z_v0)
     end if
!     vm(j)=vm(1)*func_a((z_v0-z(j))/z_v0)
  end do

!-- r1, r2 の決定 (Willoughby 2006)
  do j=1,nz
     ac(j)=(real(n)*x1)/(real(n)*x1+rm(j))
!-- r1 を求めるため, 初期値を 0, 1 とした二分法開始
     nibun_left=0.0
     nibun_right=1.0
     func_left=func_a(nibun_left)-ac(j)
     func_right=func_a(nibun_right)-ac(j)
     do while(func_left>err_max)
        err=0.0
        func_left=func_a((nibun_left+nibun_right)*0.5)-ac(j)
        func_right=func_a(nibun_right)-ac(j)
        if(func_left*func_right<0.0)then
           nibun_left=(nibun_left+nibun_right)*0.5
        else
           nibun_right=(nibun_left+nibun_right)*0.5
        end if
     end do
     r1(j)=rm(j)-dr12*0.5*(nibun_left+nibun_right)
     r2(j)=r1(j)+dr12
  end do

  do j=1,nz
     do i=1,nr
        x(i,j)=(r(i)-r1(j))/(r2(j)-r1(j))
     end do
  end do

  do j=1,nz
     ac(j)=(real(n)*x1)/(real(n)*x1+rm2(j))
!-- r1 を求めるため, 初期値を 0, 1 とした二分法開始
     nibun_left=0.0
     nibun_right=1.0
     func_left=func_a(nibun_left)-ac(j)
     func_right=func_a(nibun_right)-ac(j)
     do while(func_left>err_max)
        err=0.0
        func_left=func_a((nibun_left+nibun_right)*0.5)-ac(j)
        func_right=func_a(nibun_right)-ac(j)
        if(func_left*func_right<0.0)then
           nibun_left=(nibun_left+nibun_right)*0.5
        else
           nibun_right=(nibun_left+nibun_right)*0.5
        end if
     end do
     r12(j)=rm2(j)-dr122*0.5*(nibun_left+nibun_right)
     r22(j)=r12(j)+dr122
write(*,*) "r12", r1(j), r2(j), r12(j), r22(j)
  end do

  do j=1,nz
     do i=1,nr
        x2(i,j)=(r(i)-r12(j))/(r22(j)-r12(j))
     end do
  end do

!-- 初期値の生成
!-- 傾度風場の生成
  do j=1,nz
     do i=1,nr
        if(r(i)<=r1(j))then
           v0(i,j)=vm(j)*(r(i)/rm(j))**n
        else if(r1(j)<r(i).and.r(i)<=r2(j))then
           v0(i,j)=(vm(j)*(r(i)/rm(j))**n)*(1.0-func_a(x(i,j)))  &
  &                +(vm(j)*exp(-(r(i)-rm(j))/x1))*func_a(x(i,j))
        else if(r2(j)<r(i).and.r(i)<=r12(j))then
           v0(i,j)=vm(j)*exp(-(r(i)-rm(j))/x1)
        else if(r12(j)<r(i).and.r(i)<=r22(j))then
           v0(i,j)=(vm2(j)*(r(i)/rm2(j))**n)*(1.0-func_a(x2(i,j)))  &
  &                +(vm2(j)*exp(-(r(i)-rm2(j))/x1))*func_a(x2(i,j))
        else if(r22(j)<r(i))then
           v0(i,j)=vm2(j)*exp(-(r(i)-rm2(j))/x1)
        else
           write(*,*) "dokonimo hairanai", r(i)
        end if
!        if(z(j)>=z_v0)then
!           v0(i,j)=0.0
!        end if
     end do
  end do

!-- サウンディングと軸対称流から静力学・傾度風平衡場の生成.

  call hydro_grad_eqb( r, z, coriol, v0, pres_s, rho_s, pres, rho )

!-- ソルバに必要な pt を計算

  do j=1,nz
     do i=1,nr
        theta0(i,j)=theta_dry( rhoP_2_T( rho(i,j), pres(i,j) ), pres(i,j) )
     end do
  end do

!-- 加熱率の計算
!-- 浮力振動数の計算
  do i=1,nr
     call grad_1d( z, theta0(i,:), N2(i,:) )
     do j=1,nz
        N2(i,j)=g*N2(i,j)/theta0(i,j)
     end do
  end do

!  call max_val_2d( N2, maxnx, maxny, max_val )  ! この値は適宜変更
  max_val=1.5e-4

!-- zd の設定
  zd=(/(((z_qv-z(j))/(z_qv-5.0e3)),j=1,nz)/)

  do j=1,nz
     do i=1,nr
        qv(i,j)=max_val*func_a(zd(j))*(2.5*Bell( m_bell, x(i,j) )+2.5*Bell( m_bell, x2(i,j) ))
        qv(i,j)=qv(i,j)*Cpd*theta0(i,j)/g
     end do
  end do


!-- 初期ファイルへの書き込み

!-- 以下は GrADS 形式
!  call write_file( trim(fname), nr, nz, 1, v0 )
!  call write_file( trim(fname), nr, nz, 2, pres, mode='old' )
!  call write_file( trim(fname), nr, nz, 3, theta0, mode='old' )
!  call write_file( trim(fname), nr, nz, 4, rho, mode='old' )
!  call write_file( trim(fname), nr, nz, 5, qv, mode='old' )

!-- 以下は netcdf 形式
  call HistoryCreate( file=trim(fname), title='SEQ initial data', &
    & source='test', institution='test', dims=(/'r','z'/), dimsizes=(/nr,nz/),  &
    & longnames=(/'r-coordinate','z-coordinate'/),  &
    & units=(/'m','m'/), origin=0.0, interval=0.0 )

  call HistoryPut( 'r', r )
  call HistoryPut( 'z', z )

  call HistoryAddVariable( varname='v', dims=(/'r','z'/), &
    & longname='tangential wind', units='m/s', xtype='float')

  call HistoryAddVariable( varname='pres', dims=(/'r','z'/), &
    & longname='pressure', units='Pa', xtype='float')

  call HistoryAddVariable( varname='theta', dims=(/'r','z'/), &
    & longname='potential temperature', units='K', xtype='float')

  call HistoryAddVariable( varname='rho', dims=(/'r','z'/), &
    & longname='density', units='kg/m3', xtype='float')

  call HistoryAddVariable( varname='Q', dims=(/'r','z'/), &
    & longname='heating rate', units='J/s', xtype='float')

  call HistoryAddVariable( varname='mom', dims=(/'r','z'/), &
    & longname='momentum source rate', units='m/s2', xtype='float')

  call HistoryPut( 'v',v0 )

  call HistoryPut( 'pres',pres )

  call HistoryPut( 'theta', theta0 )

  call HistoryPut( 'rho', rho )

  call HistoryPut( 'Q', qv )

  call HistoryPut( 'mom', mom )

  call HistoryClose

!-- 終了のお知らせ

  write(*,*) "initial_make is normally conplete."

end program


real function func_a( val )
  implicit none
  real, intent(in) :: val  ! 座標変数

  if(val<=0.0)then
     func_a=0.0
  end if

  if(val>=1.0)then
     func_a=1.0
  end if

  if(val>0.0.and.val<1.0)then
     func_a=(10.0-15.0*val+6.0*val*val)*val*val*val
  end if

  return

end function


real function Bell( n, val )
!  Willoughby 2006 で提唱されたベル型関数を計算する.
  implicit none
  integer, intent(in) :: n  ! 次数
  real, intent(in) :: val  ! 引数

  if(val>=0.0.and.val<=1.0)then
     Bell=(2.0**(2*n))*((val*(1.0-val))**n)
  else
     Bell=0.0
  end if

  return

end function
