module force_solv

  use derivation
  use sub_mod

contains

subroutine ADV_term( val, u, v, ADV )
!-- calculating advection terms
! adv form = udvdx+vdvdy
! flx form = d(uv)dx+d(vv)dy
  use saveval_define
  implicit none
  double precision, dimension(size(x),size(y)), intent(in) :: val   ! advection variables
  double precision, dimension(size(x),size(y)), intent(in) :: u
                                            ! advecting speed of x direction
  double precision, dimension(size(x),size(y)), intent(in) :: v
                                            ! advecting speed of y direction
  double precision, dimension(size(x),size(y)), intent(inout) :: ADV   ! advection term

  integer :: i, j
  double precision, dimension(size(x),size(y)) :: dvdx, dvdy

  select case (adv_flag(1:1))

  case ('a')
     call grad_2d( x, y, val, dvdx, dvdy )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           ADV(i,j)=u(i,j)*dvdx(i,j)+v(i,j)*dvdy(i,j)
        end do
     end do

!$omp end do
!$omp end parallel

  case ('f')

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           dvdx(i,j)=u(i,j)*val(i,j)
           dvdy(i,j)=v(i,j)*val(i,j)
        end do
     end do

!$omp end do
!$omp end parallel

  call div( x, y, dvdx, dvdy, ADV )

  end select

end subroutine ADV_term


subroutine BETA_term( v, BETA )
!-- calculating beta effect
! -beta * v
  use saveval_define
  implicit none
  double precision, dimension(:,:), intent(in) :: v          ! y-component of wind
  double precision, dimension(size(v,1),size(v,2)), intent(inout) :: BETA   ! beta effect

  integer :: i, j

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do j=1,ny
     do i=1,nx
        BETA(i,j)=-betaf*v(i,j)
!        BETA(i,j)=-betaf(j)*v(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine BETA_term


subroutine DIFF_term( zeta, DIFF )
!-- calculating diffusion term
! nu*lap(zeta)
  use saveval_define
  implicit none
  double precision, dimension(size(x),size(y)), intent(in) :: zeta      ! zeta
  double precision, dimension(size(x),size(y)), intent(inout) :: DIFF   ! diffusion term

  integer :: i, j

  call Laplacian_2d( x, y, zeta, DIFF )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do j=1,ny
     do i=1,nx
        DIFF(i,j)=nu*DIFF(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine DIFF_term


subroutine calc_FORCE( zeta, psi, force )
!-- calculating total forcing terms
  use saveval_define
  implicit none
  double precision, dimension(size(x),size(y)), intent(in) :: zeta      ! zeta
  double precision, dimension(size(x),size(y)), intent(in) :: psi       ! psi
  double precision, dimension(size(x),size(y)), intent(inout) :: force  ! total force

  integer :: i, j
  double precision, dimension(size(x),size(y)) :: u, v, ADV, BETA, DIFF

!-- calculating advecting term 

  call psi2uv( psi, u, v )

  call ADV_term( zeta, u, v, ADV )

!-- calculating beta effect

  call BETA_term( v, BETA )

!-- calculating diffusion term

  call DIFF_term( zeta, DIFF )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do j=1,ny
     do i=1,nx
        force(i,j)=-ADV(i,j)+BETA(i,j)+DIFF(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine calc_FORCE


end module force_solv
