program make_init
!-- producing initial data of zeta
!-- Version Moon et al. (2010,JAS).

  use gtool_history
  use Math_Const

  implicit none

  integer, parameter :: nc=100

!-- namelist variables

  integer :: nx, ny
  integer :: nnum, pnum
  integer, dimension(nc) :: ndip
  real, allocatable, dimension(:) :: xr, yr
  double precision :: xmin, ymin, dx, dy
  double precision :: zcore, rcore, spd, alpha
  double precision, dimension(nc) :: xdip, ydip, zdip, adip
  double precision :: zetaa
  double precision :: zetap, ap, rpi, rpo, radi, rado
  character(1000) :: fname

  integer :: i, j, k
  double precision :: r, theta
  double precision :: tcx, tcy, xpc, ypc
  double precision, allocatable, dimension(:) :: x, y
  double precision, allocatable, dimension(:,:) :: zeta
  double precision :: makez

  namelist /initial /nx, ny, xmin, ymin, dx, dy, fname,  &
  &                  tcx, tcy
  namelist /vprof /zcore, rcore, spd, alpha
  namelist /dipole /nnum, xdip, ydip, zdip, adip, ndip
  namelist /patch /pnum, zetap, rpi, rpo, radi, rado, ap
  read(5,nml=initial)
  read(5,nml=vprof)
  read(5,nml=dipole)
  read(5,nml=patch)

  if(nnum>nc)then
     write(*,*) "*** ERROR (main) *** : namelist variable 'nnum' is <=", nnum, '.'
     write(*,*) "STOP."
     stop
  end if

  allocate(x(nx))
  allocate(y(ny))
  allocate(xr(nx))
  allocate(yr(ny))
  allocate(zeta(nx,ny))

  x=(/((xmin+dx*dble(i-1)),i=1,nx)/)
  y=(/((ymin+dy*dble(j-1)),j=1,ny)/)

  xr=(/((xmin+dx*real(i-1)),i=1,nx)/)
  yr=(/((ymin+dy*real(j-1)),j=1,ny)/)

  zeta=0.0d0

!-- producing mean zeta.

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,r,theta)

  do j=1,ny
     do i=1,nx
        r=dsqrt((x(i)-tcx)**2+(y(j)-tcy)**2)
        if(x(i)-tcx>0.0d0.and.y(j)-tcy>0.0d0)then
           theta=atan((y(j)-tcy)/(x(i)-tcx))
        else if(x(i)-tcx<0.0d0.and.y(j)-tcy>0.0d0)then
           theta=dabs(dacos(-1.0d0))-atan((y(j)-tcy)/dabs(x(i)-tcx))
        else if(x(i)-tcx<0.0d0.and.y(j)-tcy<0.0d0)then
           theta=dabs(dacos(-1.0d0))+atan(dabs(y(j)-tcy)/dabs(x(i)-tcx))
        else if(x(i)-tcx>0.0d0.and.y(j)-tcy<0.0d0)then
           theta=2.0d0*dabs(dacos(-1.0d0))-atan(dabs(y(j)-tcy)/(x(i)-tcx))
        else if(x(i)-tcx==0.0d0.and.y(j)-tcy<0.0d0)then
           theta=1.5d0*dabs(dacos(-1.0d0))
        else if(x(i)-tcx==0.0d0.and.y(j)-tcy>0.0d0)then
           theta=0.5d0*dabs(dacos(-1.0d0))
        else if(x(i)-tcx>0.0d0.and.y(j)-tcy==0.0d0)then
           theta=0.0d0
        else if(x(i)-tcx<0.0d0.and.y(j)-tcy==0.0d0)then
           theta=dabs(dacos(-1.0d0))
        end if
        zeta(i,j)=makez( r, theta, 2, (/zcore, 0.5d0*zcore/),  &
  &                      (/rcore/), (/spd/), (/alpha/) )
     end do
  end do

!$omp end do
!$omp end parallel

!-- producing perturbation zeta.

  if(nnum>0)then
     do j=1,ny
        do i=1,nx
           do k=1,nnum
              r=dsqrt((x(i)-xdip(k))**2+(y(j)-ydip(k))**2)
              zeta(i,j)=zeta(i,j)+zdip(k)*dexp(-(r/adip(k))**ndip(k))
           end do
        end do
     end do
  end if

!-- producing rainband like structure (in pnum > 0).

  if(pnum>0)then
     do j=1,ny
        do i=1,nx
           do k=1,pnum
              r=(rpo-rpi)*dble(k-1)/dble(pnum-1)+rpi
              theta=(rado-radi)*dble(k-1)/dble(pnum-1)+radi
              xpc=tcx+r*dcos(theta*pi_dp/180.0d0)
              ypc=tcy+r*dsin(theta*pi_dp/180.0d0)

              r=dsqrt((x(i)-xpc)**2+(y(j)-ypc)**2)
              zeta(i,j)=zeta(i,j)+zetap*dexp(-(r/ap)**6)
           end do
        end do
     end do
  end if

  call HistoryCreate( file=trim(adjustl(fname)), title='BAROTRO initial data', &
  & source='test', institution='test', dims=(/'x', 'y'/),  &
  & dimsizes=(/ nx, ny /),  &
  & longnames=(/'X-coordinate','Y-coordinate'/),  &
  & units=(/'m', 'm'/) )

  call HistoryPut( 'x', xr )
  call HistoryPut( 'y', yr )

  call HistoryAddVariable( varname='zeta', dims=(/'x','y'/), &
    & longname='vorticity', units='s-1', xtype='double' )

  call HistoryAddVariable( varname='xd', dims=(/'x'/), &
  &    longname='X-coord double', units='m', xtype='double' )

  call HistoryAddVariable( varname='yd', dims=(/'y'/), &
  &    longname='Y-coord double', units='m', xtype='double' )

  call HistoryPut( 'zeta', zeta )
  call HistoryPut( 'xd', x )
  call HistoryPut( 'yd', y )

  call HistoryClose()

end program

!contains



double precision function makez( radi, ang, nnum, zbar, rn, dn, an )

  use Math_Const

  implicit none

  double precision :: radi  ! radius [m]
  double precision :: ang   ! angle [rad]
  integer :: nnum
  double precision :: zbar(nnum)
  double precision :: rn(nnum-1)
  double precision :: dn(nnum-1)
  double precision :: an(nnum-1)
  integer :: i, k
  double precision :: tmpz, FS, val, coe, s, s1, s2, zptmp

  FS(val)=1.0d0-3.0d0*(val**2)+2.0d0*(val**3)

  if(radi>=0.0d0.and.radi<rn(1)-dn(1))then
     tmpz=0.0d0

     makez=zbar(1)

  else if(radi>=rn(nnum-1)+dn(nnum-1))then
     makez=zbar(nnum)*(1-an(nnum-1))*((rn(nnum-1)/radi)**(1.0d0+an(nnum-1)))

  else if(radi>=rn(1)-dn(1).and.radi<rn(nnum-1)+dn(nnum-1))then
     do k=1,nnum-1
        if(radi>=rn(k)-dn(k).and.radi<rn(k)+dn(k))then  ! Hermite
           coe=0.5d0/dn(k)
           s1=coe*(radi-rn(k)+dn(k))
           s2=coe*(rn(k)+dn(k)-radi)
           makez=zbar(k)*FS(s1)  &
  &             +zbar(k+1)*(1-an(k))*((rn(k)/radi)**(1.0d0+an(k)))*FS(s2)
           exit

        else if(radi>=rn(k)+dn(k))then
           if(radi<rn(k+1)-dn(k+1))then  ! constant
              makez=zbar(k+1)*(1-an(k))*((rn(k)/radi)**(1.0d0+an(k)))
              exit
           end if
        end if
     end do

  end if

  return

end function makez

!end program
