module time_scheme
  use force_nbm_solv
  use force_mbl_solv
  use mpi_mod

contains

subroutine time_integration( it, psiko, uko_mbl, vko_mbl,  &
  &                          psikn, ukn_mbl, vkn_mbl, pk,  &
  &                          nbm_opt, mbl_optu, mbl_optv )
!-- calculating time integration NBM and MBL, simultaneously
  use savegloval_define
  use Math_Const

  implicit none
  integer, intent(in) :: it                               ! time step
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: psiko ! psik of zo
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout) :: uko_mbl ! uk of zo
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout) :: vko_mbl ! vk of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: psikn   ! psik of zn
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(out) :: ukn_mbl   ! uk of zn
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(out) :: vkn_mbl   ! vk of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: pk     ! pk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: nbm_opt
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout), optional :: mbl_optu
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout), optional :: mbl_optv
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1

  integer :: i, j, k, ct
  complex(kind(0d0)), dimension(kxnt,kynt) :: psikt, forcea_zeta, force_zeta
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1) :: ukt_mbl, vkt_mbl
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1) :: forcea_uk, force_uk
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1) :: forcea_vk, force_vk
  double precision, dimension(jynt,jxnt) :: u_nbm_isp  ! u for ISPACK
  double precision, dimension(jynt,jxnt) :: v_nbm_isp  ! v for ISPACK

  psikt=psiko
  ukt_mbl=uko_mbl
  vkt_mbl=vko_mbl
  u_nbm_isp=0.0d0
  v_nbm_isp=0.0d0
  pk=0.0d0
  force_uk=0.0d0
  force_vk=0.0d0
  forcea_uk=0.0d0
  forcea_vk=0.0d0
  force_zeta=0.0d0
  forcea_zeta=0.0d0

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     call force_NBM( psiko, force_zeta,  &
  &                  pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )

     call force_MBL( psiko, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                  force_uk, force_vk )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,2*hynt+1
        do i=1,hxnt+1
           psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nzp
        do j=1,2*hynt+1
           do i=1,hxnt+1
              ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*force_uk(i,j,k)
              vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*force_vk(i,j,k)
           end do
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nzp
        do j=2,hynt+1
           do i=2,hxnt+1
              ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
              ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
              vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
              vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
           end do
        end do
     end do

!$omp end do
!$omp end parallel

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
        call force_MBL( psiko, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)
        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(0.5d0*dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(0.5d0*dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (2)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 psiko(i,j)=psikt(i,j)+dt*(-forcea_zeta(i,j)+2.0d0*force_zeta(i,j))
                 forcea_zeta(i,j)=forcea_zeta(i,j)+4.0d0*force_zeta(i,j)
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+dt*(-forcea_uk(i,j,k)+2.0d0*force_uk(i,j,k))
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+dt*(-forcea_vk(i,j,k)+2.0d0*force_vk(i,j,k))
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+4.0d0*force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+4.0d0*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    ukn_mbl(i,j,k)=ukt_mbl(i,j,k)+(dt/6.0d0)*forcea_uk(i,j,k)
                    vkn_mbl(i,j,k)=vkt_mbl(i,j,k)+(dt/6.0d0)*forcea_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    uko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(uko_mbl(i,j,k))
                    vko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vko_mbl(i,j,k))
                    uko_mbl(kxnt-i+2,j,k)=dconjg(uko_mbl(i,kynt-j+2,k))
                    vko_mbl(kxnt-i+2,j,k)=dconjg(vko_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
                    vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
                    ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
                    vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
        call force_MBL( psiko, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)

        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(0.5d0*dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(0.5d0*dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (2)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+2.0d0*force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+2.0d0*force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(0.5d0*dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(0.5d0*dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (3)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(dt)*force_zeta(i,j)
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+2.0d0*force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+2.0d0*force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (4)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    ukn_mbl(i,j,k)=ukt_mbl(i,j,k)+(dt/6.0d0)*forcea_uk(i,j,k)
                    vkn_mbl(i,j,k)=vkt_mbl(i,j,k)+(dt/6.0d0)*forcea_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2,3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    uko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(uko_mbl(i,j,k))
                    vko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vko_mbl(i,j,k))
                    uko_mbl(kxnt-i+2,j,k)=dconjg(uko_mbl(i,kynt-j+2,k))
                    vko_mbl(kxnt-i+2,j,k)=dconjg(vko_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        case (4)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
              end do
           end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
                    vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
                    ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
                    vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     call force_NBM( psiko, force_zeta,  &
  &                  pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
     call force_MBL( psiko, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              nbm_opt(i,j)=force_zeta(i,j)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*force_uk(i,j,k)
                 vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*force_vk(i,j,k)
                 mbl_optu(i,j,k)=force_uk(i,j,k)
                 mbl_optv(i,j,k)=force_vk(i,j,k)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*(1.5d0*force_zeta(i,j)-0.5d0*nbm_opt(i,j))
              nbm_opt(i,j)=force_zeta(i,j)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*(1.5d0*force_uk(i,j,k)-0.5d0*mbl_optu(i,j,k))
                 vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*(1.5d0*force_vk(i,j,k)-0.5d0*mbl_optv(i,j,k))
                 mbl_optu(i,j,k)=force_uk(i,j,k)
                 mbl_optv(i,j,k)=force_vk(i,j,k)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nzp
        do j=2,hynt+1
           do i=2,hxnt+1
              ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
              vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
              ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
              vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('L-F')  ! Leap Frog scheme

     call force_NBM( psiko, force_zeta,  &
  &                  pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
     call force_MBL( psiko, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              nbm_opt(i,j)=psiko(i,j)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*force_uk(i,j,k)
                 vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*force_vk(i,j,k)
                 mbl_optu(i,j,k)=uko_mbl(i,j,k)
                 mbl_optv(i,j,k)=vko_mbl(i,j,k)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=nbm_opt(i,j)+2.0d0*dt*force_zeta(i,j)
              nbm_opt(i,j)=psiko(i,j)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=mbl_optu(i,j,k)+2.0d0*dt*force_uk(i,j,k)
                 vkn_mbl(i,j,k)=mbl_optv(i,j,k)+2.0d0*dt*force_vk(i,j,k)
                 mbl_optu(i,j,k)=uko_mbl(i,j,k)
                 mbl_optv(i,j,k)=vko_mbl(i,j,k)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nzp
        do j=2,hynt+1
           do i=2,hxnt+1
              ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
              vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
              ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
              vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

  psiko=psikt
  uko_mbl=ukt_mbl
  vko_mbl=vkt_mbl

end subroutine time_integration


subroutine time_integration_NBM( it, psiko, psikn, nbm_opt, pk, u_isp, v_isp )
!-- calculating time integration
  use savegloval_define
  use Math_Const

  implicit none
  integer, intent(in) :: it                               ! time step
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: psiko ! psi of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: psikn   ! psi of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: nbm_opt
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out), optional :: pk   ! p
  double precision, dimension(jynt,jxnt), intent(out), optional :: u_isp  ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(out), optional :: v_isp  ! v for ISPACK

  integer :: i, j, ct
  complex(kind(0d0)), dimension(kxnt,kynt) :: psikt, forcea_zeta, force_zeta

  psikt=psiko
  force_zeta=0.0d0
  forcea_zeta=0.0d0

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     if(present(pk))then
        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_isp, v_isp=v_isp )
     else
        call force_NBM( psiko, force_zeta )
     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,2*hynt+1
        do i=1,hxnt+1
           psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        if(present(pk))then
           call force_NBM( psiko, force_zeta,  &
  &                        pk=pk, u_isp=u_isp, v_isp=v_isp )
        else
           call force_NBM( psiko, force_zeta )
        end if

!$omp parallel default(shared)
        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (2)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 psiko(i,j)=psikt(i,j)+dt*(-forcea_zeta(i,j)+2.0d0*force_zeta(i,j))
                 forcea_zeta(i,j)=forcea_zeta(i,j)+4.0d0*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        if(present(pk))then
           call force_NBM( psiko, force_zeta,  &
  &                        pk=pk, u_isp=u_isp, v_isp=v_isp )
        else
           call force_NBM( psiko, force_zeta )
        end if

!$omp parallel default(shared)

        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (2)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (3)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (4)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2,3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (4)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     if(present(pk))then
        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_isp, v_isp=v_isp )
     else
        call force_NBM( psiko, force_zeta )
     end if

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              nbm_opt(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*(1.5d0*force_zeta(i,j)-0.5d0*nbm_opt(i,j))
              nbm_opt(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

  case ('L-F')  ! Leap Frog scheme

     if(present(pk))then
        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_isp, v_isp=v_isp )
     else
        call force_NBM( psiko, force_zeta )
     end if

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              nbm_opt(i,j)=psiko(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=nbm_opt(i,j)+2.0d0*dt*force_zeta(i,j)
              nbm_opt(i,j)=psiko(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

  psiko=psikt

end subroutine time_integration_NBM

subroutine time_integration_MBL( it, pk, psiko_nbm, u_nbm_isp, v_nbm_isp,  &
  &                              uko_mbl, vko_mbl, ukn_mbl, vkn_mbl, mbl_optu, mbl_optv )
!-- calculating time integration
  use savegloval_define
  use Math_Const

  implicit none
  integer, intent(in) :: it                               ! time step
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: pk  ! pressure in NBM
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: psiko_nbm ! psik in NBM
  double precision, dimension(jynt,jxnt), intent(in)  :: u_nbm_isp  ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(in)  :: v_nbm_isp  ! v for ISPACK
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout) :: uko_mbl ! u of zo
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout) :: vko_mbl ! v of zo
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(out) :: ukn_mbl   ! u of zn
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(out) :: vkn_mbl   ! v of zn
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout), optional :: mbl_optu
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1), intent(inout), optional :: mbl_optv
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  integer :: i, j, k, ct
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1) :: ukt_mbl, vkt_mbl, forcea_uk, force_uk
  complex(kind(0d0)), dimension(kxnt,kynt,0:nzp+1) :: forcea_vk, force_vk

  ukt_mbl=uko_mbl
  vkt_mbl=vko_mbl
  force_uk=0.0d0
  force_vk=0.0d0
  forcea_uk=0.0d0
  forcea_vk=0.0d0

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     call force_MBL( psiko_nbm, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                  force_uk, force_vk )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nzp
        do j=1,2*hynt+1
           do i=1,hxnt+1
              ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*force_uk(i,j,k)
              vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*force_vk(i,j,k)
           end do
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nzp
        do j=2,hynt+1
           do i=2,hxnt+1
              ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
              ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
              vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
              vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
           end do
        end do
     end do

!$omp end do
!$omp end parallel

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        call force_MBL( psiko_nbm, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)

        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(0.5d0*dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(0.5d0*dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (2)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+dt*(-forcea_uk(i,j,k)+2.0d0*force_uk(i,j,k))
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+dt*(-forcea_vk(i,j,k)+2.0d0*force_vk(i,j,k))
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+4.0d0*force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+4.0d0*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    ukn_mbl(i,j,k)=ukt_mbl(i,j,k)+(dt/6.0d0)*forcea_uk(i,j,k)
                    vkn_mbl(i,j,k)=vkt_mbl(i,j,k)+(dt/6.0d0)*forcea_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    uko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(uko_mbl(i,j,k))
                    vko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vko_mbl(i,j,k))
                    uko_mbl(kxnt-i+2,j,k)=dconjg(uko_mbl(i,kynt-j+2,k))
                    vko_mbl(kxnt-i+2,j,k)=dconjg(vko_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
                    vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
                    ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
                    vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        call force_MBL( psiko_nbm, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)
        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(0.5d0*dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(0.5d0*dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (2)                         
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+2.0d0*force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+2.0d0*force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(0.5d0*dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(0.5d0*dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (3)                         
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+2.0d0*force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+2.0d0*force_vk(i,j,k)
                    uko_mbl(i,j,k)=ukt_mbl(i,j,k)+(dt)*force_uk(i,j,k)
                    vko_mbl(i,j,k)=vkt_mbl(i,j,k)+(dt)*force_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        case (4)                         
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=1,2*hynt+1
                 do i=1,hxnt+1
                    forcea_uk(i,j,k)=forcea_uk(i,j,k)+force_uk(i,j,k)
                    forcea_vk(i,j,k)=forcea_vk(i,j,k)+force_vk(i,j,k)
                    ukn_mbl(i,j,k)=ukt_mbl(i,j,k)+(dt/6.0d0)*forcea_uk(i,j,k)
                    vkn_mbl(i,j,k)=vkt_mbl(i,j,k)+(dt/6.0d0)*forcea_vk(i,j,k)
                 end do
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2,3)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    uko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(uko_mbl(i,j,k))
                    vko_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vko_mbl(i,j,k))
                    uko_mbl(kxnt-i+2,j,k)=dconjg(uko_mbl(i,kynt-j+2,k))
                    vko_mbl(kxnt-i+2,j,k)=dconjg(vko_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        case (4)
!$omp do schedule(runtime) private(i,j,k)
           do k=1,nzp
              do j=2,hynt+1
                 do i=2,hxnt+1
                    ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
                    vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
                    ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
                    vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
                 end do
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     call force_MBL( psiko_nbm, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*force_uk(i,j,k)
                 vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*force_vk(i,j,k)
                 mbl_optu(i,j,k)=force_uk(i,j,k)
                 mbl_optv(i,j,k)=force_vk(i,j,k)
              end do
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*(1.5d0*force_uk(i,j,k)-0.5d0*mbl_optu(i,j,k))
                 vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*(1.5d0*force_vk(i,j,k)-0.5d0*mbl_optv(i,j,k))
                 mbl_optu(i,j,k)=force_uk(i,j,k)
                 mbl_optv(i,j,k)=force_vk(i,j,k)
              end do
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nzp
        do j=2,hynt+1
           do i=2,hxnt+1
              ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
              vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
              ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
              vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
           end do
        end do
     end do

!$omp end do
!$omp end parallel

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

  case ('L-F')  ! Leap Frog scheme

     call force_MBL( psiko_nbm, u_nbm_isp, v_nbm_isp, pk, uko_mbl, vko_mbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=uko_mbl(i,j,k)+dt*force_uk(i,j,k)
                 vkn_mbl(i,j,k)=vko_mbl(i,j,k)+dt*force_vk(i,j,k)
                 mbl_optu(i,j,k)=uko_mbl(i,j,k)
                 mbl_optv(i,j,k)=vko_mbl(i,j,k)
              end do
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

        do k=1,nzp
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 ukn_mbl(i,j,k)=mbl_optu(i,j,k)+2.0d0*dt*force_uk(i,j,k)
                 vkn_mbl(i,j,k)=mbl_optv(i,j,k)+2.0d0*dt*force_vk(i,j,k)
                 mbl_optu(i,j,k)=uko_mbl(i,j,k)
                 mbl_optv(i,j,k)=vko_mbl(i,j,k)
              end do
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nzp
        do j=2,hynt+1
           do i=2,hxnt+1
              ukn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(ukn_mbl(i,j,k))
              vkn_mbl(kxnt-i+2,kynt-j+2,k)=dconjg(vkn_mbl(i,j,k))
              ukn_mbl(kxnt-i+2,j,k)=dconjg(ukn_mbl(i,kynt-j+2,k))
              vkn_mbl(kxnt-i+2,j,k)=dconjg(vkn_mbl(i,kynt-j+2,k))
           end do
        end do
     end do

!$omp end do
!$omp end parallel

     if(smt_mbl_opt>0)then
        call smooth_val_MBL( ukn_mbl )
        call smooth_val_MBL( vkn_mbl )
     end if

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

  uko_mbl=ukt_mbl
  vko_mbl=vkt_mbl

end subroutine time_integration_MBL

end module time_scheme
