module force_sbl_solv
!-- 全ての項は d (psi) / d t で計算されている.

  use fftsub_mod
  use diag_function

contains

subroutine ADV_term_SBL( u_isp, v_isp, w_isp,  &
  &                      ADVu, ADVv, NLu_isp, NLv_isp )
!-- calculating advection terms
  use Math_Const
  use savegloval_define
  implicit none
  double precision, dimension(jynt,jxnt), intent(in) :: u_isp  ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: v_isp  ! v for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: w_isp  ! w for ISPACK
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: ADVu  ! advection term for U
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: ADVv  ! advection term for V
  double precision, dimension(jynt,jxnt), intent(inout) :: NLu_isp ! NL term in u for ISPACK
  double precision, dimension(jynt,jxnt), intent(inout) :: NLv_isp ! NL term in v for ISPACK

  integer :: i, j
  double precision :: pi2, lxi, lyi, hi
  complex(kind(0d0)), dimension(kxnt,kynt) :: auukl, auvkl, avvkl
  double precision, dimension(jynt,jxnt) :: auu_isp, auv_isp, avv_isp

  lxi=1.0d0/Lx
  lyi=1.0d0/Ly
  hi=1.0d0/h_sbl
  pi2=2.0d0*pi_dp

  auu_isp=0.0d0
  auv_isp=0.0d0
  avv_isp=0.0d0
  auukl=0.0d0
  auvkl=0.0d0
  avvkl=0.0d0

  ADVu=0.0d0
  ADVv=0.0d0
  NLu_isp=0.0d0
  NLv_isp=0.0d0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do i=1,jxnt
     do j=1,jynt
        auu_isp(j,i)=u_isp(j,i)*u_isp(j,i)
        auv_isp(j,i)=u_isp(j,i)*v_isp(j,i)
        avv_isp(j,i)=v_isp(j,i)*v_isp(j,i)
        Nlu_isp(j,i)=-u_isp(j,i)*w_isp(j,i)*hi
        Nlv_isp(j,i)=-v_isp(j,i)*w_isp(j,i)*hi
     end do
  end do

!$omp end do
!$omp end parallel

  call phys2spec_isp( auu_isp(1:jynt,1:jxnt), auukl(1:kxnt,1:kynt) )
  call phys2spec_isp( auv_isp(1:jynt,1:jxnt), auvkl(1:kxnt,1:kynt) )
  call phys2spec_isp( avv_isp(1:jynt,1:jxnt), avvkl(1:kxnt,1:kynt) )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     ADVu(i,1)=-img_cdp*pi2*(dble(i-1)*lxi*auukl(i,1))
     ADVv(i,1)=-img_cdp*pi2*(dble(i-1)*lxi*auvkl(i,1))
     ADVu(kxnt-i+2,1)=dconjg(ADVu(i,1))
     ADVv(kxnt-i+2,1)=dconjg(ADVv(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     ADVu(1,j)=-img_cdp*pi2*(dble(j-1)*lyi*auvkl(1,j))
     ADVv(1,j)=-img_cdp*pi2*(dble(j-1)*lyi*avvkl(1,j))
     ADVu(1,kynt-j+2)=img_cdp*pi2*(dble(j-1)*lyi*auvkl(1,kynt-j+2))
     ADVv(1,kynt-j+2)=img_cdp*pi2*(dble(j-1)*lyi*avvkl(1,kynt-j+2))
!-- ここでは, ISPACK の仕様により, l 方向は -l について独立
!-- なので, jy-j+2 では l 方向の波数が奇数乗でかかる場合,
!-- (-l) でかける必要がある (偶数乗はかけなくても同じ結果). 
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        ADVu(i,j)=-img_cdp*pi2*(dble(i-1)*lxi*auukl(i,j)  &
  &                            +dble(j-1)*lyi*auvkl(i,j))
        ADVv(i,j)=-img_cdp*pi2*(dble(i-1)*lxi*auvkl(i,j)  &
  &                            +dble(j-1)*lyi*avvkl(i,j))
        ADVu(kxnt-i+2,kynt-j+2)=dconjg(ADVu(i,j))
        ADVv(kxnt-i+2,kynt-j+2)=dconjg(ADVv(i,j))
        ADVu(i,kynt-j+2)=-img_cdp*pi2*(dble(i-1)*lxi*auukl(i,kynt-j+2)  &
  &                                   -dble(j-1)*lyi*auvkl(i,kynt-j+2))
        ADVv(i,kynt-j+2)=-img_cdp*pi2*(dble(i-1)*lxi*auvkl(i,kynt-j+2)  &
  &                                   -dble(j-1)*lyi*avvkl(i,kynt-j+2))
        ADVu(kxnt-i+2,j)=dconjg(ADVu(i,kynt-j+2))
        ADVv(kxnt-i+2,j)=dconjg(ADVv(i,kynt-j+2))
!-- ここでは, ISPACK の仕様により, l 方向は -l について独立
!-- なので, jy-j+2 では l 方向の波数が奇数乗でかかる場合,
!-- (-l) でかける必要がある (偶数乗はかけなくても同じ結果). 
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine ADV_term_SBL


subroutine CORIL_term_SBL( uk, vk, CORILu, CORILv )
!-- calculating Coriolis term
!-- +fvk, -fuk
  use savegloval_define
  implicit none
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: uk        ! uk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: vk        ! vk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: CORILu ! Coriolis term for U
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: CORILv ! Coriolis term for V

  integer :: i, j

  CORILu=0.0d0
  CORILv=0.0d0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     CORILu(i,1)=f0*vk(i,1)
     CORILv(i,1)=-f0*uk(i,1)
     CORILu(kxnt-i+2,1)=dconjg(vk(i,1))
     CORILv(kxnt-i+2,1)=dconjg(uk(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     CORILu(1,j)=f0*vk(1,j)
     CORILv(1,j)=-f0*uk(1,j)
     CORILu(1,kynt-j+2)=f0*vk(1,kynt-j+2)
     CORILv(1,kynt-j+2)=-f0*uk(1,kynt-j+2)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        CORILu(i,j)=f0*vk(i,j)
        CORILv(i,j)=-f0*uk(i,j)
        CORILu(kxnt-i+2,kynt-j+2)=dconjg(CORILu(i,j))
        CORILv(kxnt-i+2,kynt-j+2)=dconjg(CORILv(i,j))
        CORILu(i,kynt-j+2)=f0*vk(i,kynt-j+2)
        CORILv(i,kynt-j+2)=-f0*uk(i,kynt-j+2)
        CORILu(kxnt-i+2,j)=dconjg(CORILu(i,kynt-j+2))
        CORILv(kxnt-i+2,j)=dconjg(CORILv(i,kynt-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine CORIL_term_SBL


subroutine DIFF_term_SBL( uk, vk, DIFFu, DIFFv )
!-- calculating diffusion term
! K*lap(val)
  use Math_Const
  use savegloval_define
  implicit none
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: uk        ! uk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: vk        ! vk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: DIFFu  ! diffusion term for U
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: DIFFv  ! diffusion term for V

  integer :: i, j
  double precision :: lxi, lyi, pi4

  DIFFu=0.0d0
  DIFFv=0.0d0

  lxi=1.0d0/Lx
  lyi=1.0d0/Ly
  pi4=4.0d0*pi_dp*pi_dp

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     DIFFu(i,1)=-K_sbl*pi4*((dble(i-1)*lxi)**2)*uk(i,1)
     DIFFv(i,1)=-K_sbl*pi4*((dble(i-1)*lxi)**2)*vk(i,1)
     DIFFu(kxnt-i+2,1)=dconjg(DIFFu(i,1))
     DIFFv(kxnt-i+2,1)=dconjg(DIFFv(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     DIFFu(1,j)=-K_sbl*pi4*((dble(j-1)*lyi)**2)*uk(1,j)
     DIFFv(1,j)=-K_sbl*pi4*((dble(j-1)*lyi)**2)*vk(1,j)
     DIFFu(1,kynt-j+2)=-K_sbl*pi4*((dble(j-1)*lyi)**2)*uk(1,kynt-j+2)
     DIFFv(1,kynt-j+2)=-K_sbl*pi4*((dble(j-1)*lyi)**2)*vk(1,kynt-j+2)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        DIFFu(i,j)=-K_sbl*pi4*((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)*uk(i,j)
        DIFFv(i,j)=-K_sbl*pi4*((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)*vk(i,j)
        DIFFu(kxnt-i+2,kynt-j+2)=dconjg(DIFFu(i,j))
        DIFFv(kxnt-i+2,kynt-j+2)=dconjg(DIFFv(i,j))
        DIFFu(i,kynt-j+2)=-K_sbl*pi4*((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)  &
  &                       *uk(i,kynt-j+2)
        DIFFv(i,kynt-j+2)=-K_sbl*pi4*((dble(i-1)*lxi)**2+(dble(j-1)*lyi)**2)  &
  &                       *vk(i,kynt-j+2)
        DIFFu(kxnt-i+2,j)=dconjg(DIFFu(i,kynt-j+2))
        DIFFv(kxnt-i+2,j)=dconjg(DIFFv(i,kynt-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine DIFF_term_SBL


subroutine STRETCH_term_phy_SBL( u_sbl_isp, v_sbl_isp, w_sbl_isp,  &
  &                              u_nbm_isp, v_nbm_isp,  &
  &                              STRETCHu_isp, STRETCHv_isp )
!-- calculating stretching terms (for ISPACK on physical space)
  use Math_Const
  use savegloval_define
  implicit none
  double precision, dimension(jynt,jxnt), intent(in) :: u_sbl_isp   ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: v_sbl_isp   ! v for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: w_sbl_isp   ! w for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: u_nbm_isp   ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: v_nbm_isp   ! v for ISPACK
  double precision, dimension(jynt,jxnt), intent(inout) :: STRETCHu_isp
                                                  ! streching term of U for ISPACK
  double precision, dimension(jynt,jxnt), intent(inout) :: STRETCHv_isp
                                                  ! streching term of V for ISPACK

  integer :: i, j
  double precision :: hi

  STRETCHu_isp=0.0d0
  STRETCHv_isp=0.0d0
  hi=1.0d0/h_sbl

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do i=1,jxnt
     do j=1,jynt
!        STRETCHu_isp(j,i)=-w_sbl_isp(j,i)  &
!  &                       *(u_nbm_isp(j,i)-u_sbl_isp(j,i))*hi
!        STRETCHv_isp(j,i)=-w_sbl_isp(j,i)  &
!  &                       *(v_nbm_isp(j,i)-v_sbl_isp(j,i))*hi
        STRETCHu_isp(j,i)=-0.5d0*(w_sbl_isp(j,i)-dabs(w_sbl_isp(j,i)))  &
  &                       *(u_nbm_isp(j,i)-u_sbl_isp(j,i))*hi
        STRETCHv_isp(j,i)=-0.5d0*(w_sbl_isp(j,i)-dabs(w_sbl_isp(j,i)))  &
  &                       *(v_nbm_isp(j,i)-v_sbl_isp(j,i))*hi
!        STRETCHu_isp(j,i)=-0.5d0*(dabs(w_sbl_isp(j,i))-w_sbl_isp(j,i))  &
!  &                       *(u_nbm_isp(j,i)-u_sbl_isp(j,i))*hi
!        STRETCHv_isp(j,i)=-0.5d0*(dabs(w_sbl_isp(j,i))-w_sbl_isp(j,i))  &
!  &                       *(v_nbm_isp(j,i)-v_sbl_isp(j,i))*hi
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine STRETCH_term_phy_SBL


subroutine FRICTION_term_phy_SBL( u_sbl_isp, v_sbl_isp,  &
  &                               FRICTIONu_isp, FRICTIONv_isp )
!-- calculating friction terms (for ISPACK on physical space)
  use Math_Const
  use savegloval_define
  implicit none
  double precision, dimension(jynt,jxnt), intent(in) :: u_sbl_isp   ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: v_sbl_isp   ! v for ISPACK
  double precision, dimension(jynt,jxnt), intent(inout) :: FRICTIONu_isp
                                                  ! friction term of U for ISPACK
  double precision, dimension(jynt,jxnt), intent(inout) :: FRICTIONv_isp
                                                  ! friction term of V for ISPACK

  integer :: i, j
  double precision :: vs, Cd_diag, coef, hi

  FRICTIONu_isp=0.0d0
  FRICTIONv_isp=0.0d0
  hi=1.0d0/h_sbl

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,vs,Cd_diag,coef)

     do i=1,jxnt
        do j=1,jynt
           Cd_diag=Cd_K16( u_sbl_isp(j,i), v_sbl_isp(j,i) )
           vs=0.78d0*dsqrt(u_sbl_isp(j,i)**2+v_sbl_isp(j,i)**2)
           coef=Cd_diag*vs*hi
           FRICTIONu_isp(j,i)=-coef*u_sbl_isp(j,i)
           FRICTIONv_isp(j,i)=-coef*v_sbl_isp(j,i)
        end do
     end do

!$omp end do
!$omp end parallel

end subroutine FRICTION_term_phy_SBL


subroutine TOTNL_term_SBL( ADV_isp, STRETCH_isp, FRIC_isp, TOTNL )
!-- calculating stretching terms (on physical space)
  use Math_Const
  use savegloval_define
  implicit none
  double precision, dimension(jynt,jxnt), intent(in) :: ADV_isp      ! NL in ADV_term_PBL for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: STRETCH_isp  ! STRETCH_term_phys_PBL for ISPACK
  double precision, dimension(jynt,jxnt), intent(in) :: FRIC_isp     ! FRIC_term_phys_PBL for ISPACK
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: TOTNL   ! total nonlinear term

  integer :: i, j
  double precision, dimension(jynt,jxnt) :: TOTNL_isp

  TOTNL=0.0d0
  TOTNL_isp=0.0d0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do i=1,jxnt
     do j=1,jynt
        TOTNL_isp(j,i)=ADV_isp(j,i)+STRETCH_isp(j,i)+FRIC_isp(j,i)
     end do
  end do

!$omp end do
!$omp end parallel

  call phys2spec_isp( TOTNL_isp(1:jynt,1:jxnt), TOTNL(1:kxnt,1:kynt) )

end subroutine TOTNL_term_SBL


subroutine PGRAD_term_SBL( pk, PGRADu, PGRADv )
!-- calculating pressure gradient terms
! -mu*zeta
  use Math_Const
  use savegloval_define
  implicit none
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: pk      ! pk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: PGRADu
                                            ! pressure gradient term for U
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: PGRADv
                                            ! pressure gradient term for V

  integer :: i, j
  double precision :: lxi, lyi, pi2, rhoi

  PGRADu=0.0d0
  PGRADv=0.0d0

  lxi=1.0d0/Lx
  lyi=1.0d0/Ly
  pi2=2.0d0*pi_dp
  rhoi=1.0d0/rho0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     PGRADu(i,1)=-img_cdp*pi2*(dble(i-1)*lxi)*pk(i,1)
     PGRADu(kxnt-i+2,1)=dconjg(PGRADu(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     PGRADv(1,j)=-img_cdp*pi2*(dble(j-1)*lyi)*pk(1,j)
     PGRADv(1,kynt-j+2)=img_cdp*pi2*(dble(j-1)*lyi)*pk(1,kynt-j+2)
!-- ここでは, ISPACK の仕様により, l 方向は -l について独立
!-- なので, jy-j+2 では l 方向の波数が奇数乗でかかる場合,
!-- (-l) でかける必要がある (偶数乗はかけなくても同じ結果). 
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        PGRADu(i,j)=-img_cdp*pi2*(dble(i-1)*lxi)*pk(i,j)
        PGRADv(i,j)=-img_cdp*pi2*(dble(j-1)*lyi)*pk(i,j)
        PGRADu(kxnt-i+2,kynt-j+2)=dconjg(PGRADu(i,j))
        PGRADv(kxnt-i+2,kynt-j+2)=dconjg(PGRADv(i,j))
        PGRADu(i,kynt-j+2)=-img_cdp*pi2*(dble(i-1)*lxi)*pk(i,kynt-j+2)
        PGRADv(i,kynt-j+2)=img_cdp*pi2*(dble(j-1)*lyi)*pk(i,kynt-j+2)
        PGRADu(kxnt-i+2,j)=dconjg(PGRADu(i,kynt-j+2))
        PGRADv(kxnt-i+2,j)=dconjg(PGRADv(i,kynt-j+2))
!-- ここでは, ISPACK の仕様により, l 方向は -l について独立
!-- なので, jy-j+2 では l 方向の波数が奇数乗でかかる場合,
!-- (-l) でかける必要がある (偶数乗はかけなくても同じ結果). 
     end do
  end do

!$omp end do
!$omp end parallel

  PGRADu=PGRADu*rhoi
  PGRADv=PGRADv*rhoi

end subroutine PGRAD_term_SBL


subroutine smooth_val_SBL( valk )
!-- calculating vertical flow based on a mass continuity
  use Math_Const
  use savegloval_define
  implicit none
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: valk

  integer :: i, j
  double precision :: coefk, coefl

  coefk=pi_dp/hxnt
  coefl=pi_dp/hynt

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=2,hxnt+1
     valk(i,1)=valk(i,1)*dsin(dble(i-1)*coefk)/(dble(i-1)*coefk)
     valk(kxnt-i+2,1)=dconjg(valk(i,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=2,hynt+1
     valk(1,j)=valk(1,j)*dsin(dble(j-1)*coefl)/(dble(j-1)*coefl)
     valk(1,kynt-j+2)=valk(1,kynt-j+2)*dsin(dble(j-1)*coefl)/(dble(j-1)*coefl)
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=2,hynt+1
     do i=2,hxnt+1
        valk(i,j)=valk(i,j)*dsin(dble(i-1)*coefk)/(dble(i-1)*coefk)  &
  &                        *dsin(dble(j-1)*coefl)/(dble(j-1)*coefl)
        valk(kxnt-i+2,kynt-j+2)=dconjg(valk(i,j))
        valk(i,kynt-j+2)=valk(i,kynt-j+2)*dsin(dble(i-1)*coefk)/(dble(i-1)*coefk)  &
  &                                      *dsin(dble(j-1)*coefl)/(dble(j-1)*coefl)
        valk(kxnt-i+2,j)=dconjg(valk(i,kynt-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine smooth_val_SBL


subroutine force_SBL( u_nbm_isp, v_nbm_isp, pk, uk_sbl, vk_sbl, forceu, forcev )
!-- calculating total forcing terms in NBM
  use savegloval_define
  implicit none
  double precision, dimension(jynt,jxnt), intent(in) :: u_nbm_isp   ! U for NBM on jxnt, jynt
  double precision, dimension(jynt,jxnt), intent(in) :: v_nbm_isp   ! V for NBM on jxnt, jynt
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: pk        ! pk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: uk_sbl    ! uk for SBL
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: vk_sbl    ! vk for SBL
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: forceu ! total force for U
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: forcev ! total force for V

  integer :: i, j
  double precision, dimension(jynt,jxnt) :: NLu_isp, NLv_isp, FRICu_isp, FRICv_isp
  double precision, dimension(jynt,jxnt) :: STRETCHu_isp, STRETCHv_isp
  double precision, dimension(jynt,jxnt) :: u_sbl_isp, v_sbl_isp, w_sbl_isp
  complex(kind(0d0)), dimension(kxnt,kynt) :: ADVu, ADVv, CORILu, CORILv
  complex(kind(0d0)), dimension(kxnt,kynt) :: DIFFu, DIFFv, PGRADu, PGRADv
  complex(kind(0d0)), dimension(kxnt,kynt) :: TOTNLu, TOTNLv, wk_sbl

  forceu=0.0d0
  forcev=0.0d0

!-- calculating wk based on mass continuity

  call W_divergence( uk_sbl, vk_sbl, wk_sbl )

!-- converting {u,v,w}k to {u,v,w}_sbl (for calculating nonlinear terms)

  call spec2phys_isp( uk_sbl(1:kxnt,1:kynt), u_sbl_isp(1:jynt,1:jxnt) )
  call spec2phys_isp( vk_sbl(1:kxnt,1:kynt), v_sbl_isp(1:jynt,1:jxnt) )
  call spec2phys_isp( wk_sbl(1:kxnt,1:kynt), w_sbl_isp(1:jynt,1:jxnt) )

!-- calculating advecting term 

  call ADV_term_SBL( u_sbl_isp, v_sbl_isp, w_sbl_isp,  &
  &                  ADVu, ADVv, NLu_isp, NLv_isp )

!-- calculating Coriolis term

  call CORIL_term_SBL( uk_sbl, vk_sbl, CORILu, CORILv )

!-- calculating diffusion term

  call DIFF_term_SBL( uk_sbl, vk_sbl, DIFFu, DIFFv )

!-- calculating pressure gradient term

  call PGRAD_term_SBL( pk, PGRADu, PGRADv )

!-- calculating stretching term on physical space

  call STRETCH_term_phy_SBL( u_sbl_isp, v_sbl_isp, w_sbl_isp,  &
  &                          u_nbm_isp, v_nbm_isp,  &
  &                          STRETCHu_isp, STRETCHv_isp )

!-- calculating surface friction term

  call FRICTION_term_phy_SBL( u_sbl_isp, v_sbl_isp,  &
  &                           FRICu_isp, FRICv_isp )

!-- converting nonlinear terms to spectral space

  call TOTNL_term_SBL( NLu_isp, STRETCHu_isp, FRICu_isp, TOTNLu )
  call TOTNL_term_SBL( NLv_isp, STRETCHv_isp, FRICv_isp, TOTNLv )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do j=1,2*hynt+1
     do i=1,hxnt+1
        forceu(i,j)=ADVu(i,j)+TOTNLu(i,j)+CORILu(i,j)  &
  &                +DIFFu(i,j)+PGRADu(i,j)
        forcev(i,j)=ADVv(i,j)+TOTNLv(i,j)+CORILv(i,j)  &
  &                +DIFFv(i,j)+PGRADv(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine force_SBL


end module force_sbl_solv
