module sub_calc

! サブ処理ルーチン
  use Thermo_Function

contains

subroutine dmp_val( r_u, r_s, z_w, z_s,  &
  &                 u_i, v_i, w_i, p_i, pb_s, t_i, qv_i, qc_i, ql_i,  &
  &                 u_dmp, v_dmp, w_dmp, p_dmp, t_dmp, qv_dmp, qc_dmp, ql_dmp )
!-- 予報変数をスカラー点に再配置.
  use Thermo_Const
  implicit none
  real, intent(in) :: r_u(:)
  real, intent(in) :: r_s(size(r_u))
  real, intent(in) :: z_w(:)
  real, intent(in) :: z_s(size(z_w))
  real, intent(in) :: u_i(size(r_u),size(z_w))
  real, intent(in) :: v_i(size(r_u),size(z_w))
  real, intent(in) :: w_i(size(r_u),size(z_w))
  real, intent(in) :: p_i(size(r_u),size(z_w))
  real, intent(in) :: pb_s(size(r_u),size(z_w))
  real, intent(in) :: t_i(size(r_u),size(z_w))
  real, intent(in) :: qv_i(size(r_u),size(z_w))
  real, intent(in) :: qc_i(size(r_u),size(z_w))
  real, intent(in) :: ql_i(size(r_u),size(z_w))
  real, intent(inout) :: u_dmp(size(r_u),size(z_w))
  real, intent(inout) :: v_dmp(size(r_u),size(z_w))
  real, intent(inout) :: w_dmp(size(r_u),size(z_w))
  real, intent(inout) :: p_dmp(size(r_u),size(z_w))
  real, intent(inout) :: t_dmp(size(r_u),size(z_w))
  real, intent(inout) :: qv_dmp(size(r_u),size(z_w))
  real, intent(inout) :: qc_dmp(size(r_u),size(z_w))
  real, intent(inout) :: ql_dmp(size(r_u),size(z_w))
  integer :: j, k, nr, nz

  nr=size(r_u)
  nz=size(z_w)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(k)
  do k=1,nz
     call auto_interpolation_1d( r_u, r_s, u_i(:,k), u_dmp(:,k), stdopt=.true. )
  end do
!$omp end do

!$omp do schedule(dynamic) private(j)
  do j=1,nr
     call auto_interpolation_1d( z_w, z_s, w_i(j,:), w_dmp(j,:), stdopt=.true. )
  end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz
     do j=1,nr
        v_dmp(j,k)=v_i(j,k)
        p_dmp(j,k)=p0*((p_i(j,k)+pb_s(j,k))**(Cpd/Rd))  ! p_dmp -> Pa
        t_dmp(j,k)=t_i(j,k)
        qv_dmp(j,k)=qv_i(j,k)
        qc_dmp(j,k)=qc_i(j,k)
        ql_dmp(j,k)=ql_i(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine dmp_val


subroutine dmp_suf( z_s, u_i, v_i, w_i, p_i, t_i, qv_i, qc_i, ql_i, Tsurf,  &
  &                 u_dmp, v_dmp, w_dmp, p_dmp, t_dmp, qv_dmp, qc_dmp, ql_dmp )
!-- 予報変数をスカラー点に再配置.
  use Thermo_Const
  implicit none
  real, intent(in) :: z_s
  real, intent(in) :: u_i(:)
  real, intent(in) :: v_i(size(u_i))
  real, intent(in) :: w_i(size(u_i))
  real, intent(in) :: p_i(size(u_i))
  real, intent(in) :: t_i(size(u_i))
  real, intent(in) :: qv_i(size(u_i))
  real, intent(in) :: qc_i(size(u_i))
  real, intent(in) :: ql_i(size(u_i))
  real, intent(in) :: Tsurf(size(u_i))
  real, intent(inout) :: u_dmp(size(u_i))
  real, intent(inout) :: v_dmp(size(u_i))
  real, intent(inout) :: w_dmp(size(u_i))
  real, intent(inout) :: p_dmp(size(u_i))
  real, intent(inout) :: t_dmp(size(u_i))
  real, intent(inout) :: qv_dmp(size(u_i))
  real, intent(inout) :: qc_dmp(size(u_i))
  real, intent(inout) :: ql_dmp(size(u_i))
  integer :: j, k, nr

  nr=size(u_i)

!!$omp parallel default(shared)
!!$omp do schedule(dynamic) private(k)
!  do k=1,nz
!     call auto_interpolation_1d( r_u, r_s, u_i(:,k), u_dmp(:,k), stdopt=.true. )
!  end do
!!$omp end do

!!$omp do schedule(dynamic) private(j)
!  do j=1,nr
!     call auto_interpolation_1d( z_w, z_s, w_i(j,:), w_dmp(j,:), stdopt=.true. )
!  end do
!!$omp end do
!!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)
  do j=1,nr
     p_dmp(j)=hypsometric_form( p_i(j), z_s, Tsurf(j), 0.0 )  ! p_i == Pa
     t_dmp(j)=theta_dry( Tsurf(j), p_dmp(j) )
     u_dmp(j)=0.0
     v_dmp(j)=0.0
     w_dmp(j)=0.0
     qv_dmp(j)=TP_2_qvs( Tsurf(j), p_dmp(j) )
     qc_dmp(j)=0.0
     ql_dmp(j)=0.0
  end do
!$omp end do
!$omp end parallel

end subroutine dmp_suf


subroutine bound_set( u_in, u_out, w_bot, w_top )
! 境界条件を設定する. ここでは, u_out は読み込むだけ. 何もしない.
  implicit none
  real, intent(inout) :: u_in(:)            ! r 方向内側水平風速 [m/s]
  real, intent(inout) :: u_out(size(u_in))  ! r 方向外側水平風速 [m/s]
  real, intent(inout) :: w_bot(:)            ! z 方向下端鉛直風速 [m/s]
  real, intent(inout) :: w_top(size(w_bot))  ! z 方向上端鉛直風速 [m/s]

  integer :: i, j, nw, nu

  nu=size(u_in)
  nw=size(w_bot)

  do i=1,nw  ! 現在, rigid lid のみ
     w_bot(i)=0.0
     w_top(i)=0.0
  end do

  do i=1,nu  ! 現在, 内側 rigid lid のみ
     u_in(i)=0.0
  end do

end subroutine bound_set


subroutine grad_back_1d( x, u, val )
! 1 次元のスカラー変数の勾配をスタッガード格子で計算する.
! ある点 xv(i) における変数 u の勾配を求めるとき,
! u は xu という座標で定義されており, 求めたい勾配はその半格子ずれた xv という
! 点で求めたいとする.
! このとき, xv(i) での勾配は,
! (u(i)-u(i-1))/(xu(i)-xu(i-1)) で計算される.
  implicit none
  real, intent(in) :: x(:)  ! u の定義点座標
  real, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  real, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  real :: scalex(size(x))

  nx=size(x)

  do i=2,nx
     val(i)=(u(i)-u(i-1))/(x(i)-x(i-1))
  end do

!-- データ数のない両端の処理 ---
  val(1)=0.0

end subroutine grad_back_1d


subroutine grad_for_1d( x, u, val )
! 1 次元のスカラー変数の勾配をスタッガード格子で計算する.
! ある点 xv(i) における変数 u の勾配を求めるとき,
! u は xu という座標で定義されており, 求めたい勾配はその半格子ずれた xv という
! 点で求めたいとする.
! このとき, xv(i) での勾配は,
! (u(i+1)-u(i))/(xu(i+1)-xu(i)) で計算される.
  implicit none
  real, intent(in) :: x(:)  ! u の定義点座標
  real, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  real, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  real :: scalex(size(x))

  nx=size(x)

  do i=1,nx-1
     val(i)=(u(i+1)-u(i))/(x(i+1)-x(i))
  end do

!-- データ数のない両端の処理 ---
  val(nx)=0.0

end subroutine grad_for_1d


real function hydro_calc( z1, t1, p1, z2, t2, p2, z0 )
! 静力学平衡を仮定したときに, 地表面の圧力 [Pa] を上 2 層から
! 外挿計算する.
! 実は, z0 を調節すれば任意の高度での値を計算可能.
! 計算方針は, (p_2+p_0-2p_1)/2dz=-grho_1 から逆算する.
  use Phys_Const
  implicit none
  real, intent(in) :: z1   ! 第 1 層スカラー点の高度座標
  real, intent(in) :: p1   ! 第 1 層スカラー点の圧力 [Pa]
  real, intent(in) :: t1   ! 第 1 層スカラー点の温度 [K]
  real, intent(in) :: z2   ! 第 2 層スカラー点の高度座標
  real, intent(in) :: p2   ! 第 2 層スカラー点の圧度 [Pa]
  real, intent(in) :: t2   ! 第 2 層スカラー点の温位 [K]
  real, intent(in) :: z0   ! 地表面の高度座標
  real :: rho

  rho=TP_2_rho( t1, p1 )

  hydro_calc=2.0*g*rho*(z2-z0)+p2

  return

end function hydro_calc


subroutine val_check( val, thr )
  real, intent(in) :: val(:,:)
  real, intent(in) :: thr
  integer :: n1, n2, i, j

  n1=size(val,1)
  n2=size(val,2)

  do j=1,n2
     do i=1,n1
        if(val(i,j)<=thr)then
           write(*,*) "detect thres", val(i,j), i, j
        end if
     end do
  end do

end subroutine val_check


real function ext_1d( x1, x2, v1, v2, point )
! x1, x2 の 2 点での値 v1, v2 の値を用いて, point 点での値を線形外挿する.
! x1, x2 からの相対位置を自動判定するので, point が x1, x2 のどちら側に
! あるかは気にしなくてよい.
! ただし, x1 < x2 でなければならない.

  implicit none

  real, intent(in) :: x1  ! 第 1 点
  real, intent(in) :: x2  ! 第 2 点
  real, intent(in) :: v1  ! 第 1 点での値
  real, intent(in) :: v2  ! 第 2 点での値
  real, intent(in) :: point  ! 外挿点
  real :: df, dx

  if(x1>=x2)then
     write(*,*) "*** ERROR (ext_1d) ***"
     write(*,*) "You must set x1 < x2. STOP."
     stop
  end if

  df=v2-v1
  dx=x2-x1

  if(point<x1)then
     ext_1d=v1+(df/dx)*(point-x1)
  else if(point>x2)then
     ext_1d=v2+(df/dx)*(point-x2)
  end if

end function ext_1d


subroutine set_zero( val )
! setting 0 in val.
  implicit none
  real, intent(inout) :: val(:,:)
  integer :: n1, n2, i, j

  n1=size(val,1)
  n2=size(val,2)

  do j=1,n2
     do i=1,n1
        if(val(i,j)<0.0)then
           val(i,j)=0.0
        end if
     end do
  end do

end subroutine set_zero

end module
