!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Derivation  ! 微分演算計算モジュール

  use Basis

  public :: grad_1d
  public :: grad_2d
  public :: grad_3d
  public :: div
  public :: curl
  public :: div_3d
  public :: curl_3d
  public :: grad4_1d
  public :: grad4_2d

  interface grad_1d
     module procedure grad_1df,  &
  &                   grad_1dd
  end interface grad_1d

  interface grad_2d
     module procedure grad_2df,  &
  &                   grad_2dd
  end interface grad_2d

  interface grad_3d
     module procedure grad_3df,  &
  &                   grad_3dd
  end interface grad_3d

  interface div
     module procedure divf,  &
  &                   divd
  end interface div

  interface curl
     module procedure curlf,  &
  &                   curld
  end interface curl

  interface div_3d
     module procedure div_3df,  &
  &                   div_3dd
  end interface div_3d

  interface curl_3d
     module procedure curl_3df,  &
  &                   curl_3dd
  end interface curl_3d

  interface grad4_1d
     module procedure grad4_1df,  &
  &                   grad4_1dd
  end interface grad4_1d

  interface grad4_2d
     module procedure grad4_2df,  &
  &                   grad4_2dd
  end interface grad4_2d

  interface laplacian_1d
     module procedure laplacian_1df,  &
  &                   laplacian_1dd
  end interface laplacian_1d

  interface laplacian_2d
     module procedure laplacian_2df,  &
  &                   laplacian_2dd
  end interface laplacian_2d

  interface laplacian_3d
     module procedure laplacian_3df,  &
  &                   laplacian_3dd
  end interface laplacian_3d

  interface local_peak_checker_1d
     module procedure local_peak_checker_1df,  &
  &                   local_peak_checker_1dd
  end interface local_peak_checker_1d

contains

subroutine divf( x, y, u, v, val, undeff, hx, hy )
! 2次元発散計算ルーチン
! 
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i,j)$ での発散は
! $\frac{u_{i+1,j}-u_{i-1,j}}{2dx} + \frac{v_{i,j+1}-v_{i,j-1}}{2dy} $
! とできる. これを用いて2次元発散を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
! 実質的には, grad_1d ルーチンを組み合わせることで計算を行う.
! du/dx = grad_1d( x, u, dudx ), dv/dy = grad_1d( y, v, dvdy ) という形で計算を行えば,
! 境界も自動的に計算可能.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: u(size(x),size(y))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(x),size(y))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(inout) :: val(size(x),size(y))  ! 2次元発散値
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))
  real :: dudx(size(x),size(y)), dvdy(size(x),size(y))
  real :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "div" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "div" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "div" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

!-- 方針は, x 方向に dvdy を計算し, y 方向に dudx を計算する.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scaley(i,j)*u(i,j)
              tmpv(i,j)=scalex(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpv(i,:), dvdy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1df( x, tmpu(:,j), dudx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudx(i,j)/=undeff.and.dvdy(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
                 val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scaley(i,j)*u(i,j)
           tmpv(i,j)=scalex(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpv(i,:), dvdy(i,:) )
     end do
     do j=1,ny
        call grad_1df( x, tmpu(:,j), dudx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
              val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0
           end if
        end do
     end do
  end if

end subroutine divf

!-----------------------------------------
!-----------------------------------------

subroutine divd( x, y, u, v, val, undeff, hx, hy )
! 2次元発散計算ルーチン
! 
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i,j)$ での発散は
! $\frac{u_{i+1,j}-u_{i-1,j}}{2dx} + \frac{v_{i,j+1}-v_{i,j-1}}{2dy} $
! とできる. これを用いて2次元発散を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
! 実質的には, grad_1d ルーチンを組み合わせることで計算を行う.
! du/dx = grad_1d( x, u, dudx ), dv/dy = grad_1d( y, v, dvdy ) という形で計算を行えば,
! 境界も自動的に計算可能.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  double precision, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  double precision, intent(in) :: u(size(x),size(y))  ! x に対応する方向の 2 次元ベクトル成分
  double precision, intent(in) :: v(size(x),size(y))  ! y に対応する方向の 2 次元ベクトル成分
  double precision, intent(inout) :: val(size(x),size(y))  ! 2次元発散値
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))
  double precision :: dudx(size(x),size(y)), dvdy(size(x),size(y))
  double precision :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "div" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "div" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "div" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

!-- 方針は, x 方向に dvdy を計算し, y 方向に dudx を計算する.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scaley(i,j)*u(i,j)
              tmpv(i,j)=scalex(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpv(i,:), dvdy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1dd( x, tmpu(:,j), dudx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudx(i,j)/=undeff.and.dvdy(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
                 val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0d0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scaley(i,j)*u(i,j)
           tmpv(i,j)=scalex(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpv(i,:), dvdy(i,:) )
     end do
     do j=1,ny
        call grad_1dd( x, tmpu(:,j), dudx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
              val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0d0
           end if
        end do
     end do
  end if

end subroutine divd

!-----------------------------------------
!-----------------------------------------

subroutine curlf( x, y, u, v, val, undeff, hx, hy, ord )
! 2 次元渦度計算ルーチン
!
! x, y は配列の次元の若い順に必ず並べること.
!
! u, v は偶置換の向きに配置すれば, 任意の2次元渦度が計算可能
! ただし, du/dz-dw/dx を計算するときのみ, (x,z,u,w) の順番で, ord オプション false.
!
! $\frac{\partial v}{\partial x} -\frac{\partial u}{\partial y} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i,j)$ での発散は
! $\frac{v_{i,j+1}-v_{i,j-1}}{2dx} -\frac{u_{i+1,j}-u_{i-1,j}}{2dy} $
! とできる. これを用いて2次元発散を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
! 実質的には grad_1d が微分計算を担当するので, 境界の計算も自動で行う.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: u(size(x),size(y))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(x),size(y))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(inout) :: val(size(x),size(y))  ! 2次元渦度
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  logical, intent(in),  optional :: ord  ! 微分の順番を入れ替えるオプション.
                 ! true なら, 入れ替えない, false なら, 入れ替える.
                 ! デフォルトは true, du/dz-dw/dx を計算するときのみ用いる.
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  logical :: order
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))
  real :: dvdx(size(x),size(y)), dudy(size(x),size(y))
  real :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "curl" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "curl" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "curl" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

!-- 方針は, x 方向に dvdy を計算し, y 方向に dudx を計算する.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scalex(i,j)*u(i,j)
              tmpv(i,j)=scaley(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpu(i,:), dudy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1df( x, tmpv(:,j), dvdx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudy(i,j)/=undeff.and.dvdx(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
                 val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scalex(i,j)*u(i,j)
           tmpv(i,j)=scaley(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpu(i,:), dudy(i,:) )
     end do
     do j=1,ny
        call grad_1df( x, tmpv(:,j), dvdx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
              val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0
           end if
        end do
     end do
  end if

!-- 回転の順番を逆にするオプション
!-- false なら, 順番を入れ替えて出力する.

  if(present(ord))then
     order=ord
  else
     order=.true.
  end if

  if(order.eqv..false.)then
     if(present(undeff))then
        do j=1,ny
           do i=1,nx
              if(val(i,j)/=undeff)then
                 val(i,j)=-val(i,j)
              end if
           end do
        end do
     else
        do j=1,ny
           do i=1,nx
              val(i,j)=-val(i,j)
           end do
        end do
     end if
  end if

end subroutine curlf

!-----------------------------------------
!-----------------------------------------

subroutine curld( x, y, u, v, val, undeff, hx, hy, ord )
! 2 次元渦度計算ルーチン
!
! x, y は配列の次元の若い順に必ず並べること.
!
! u, v は偶置換の向きに配置すれば, 任意の2次元渦度が計算可能
! ただし, du/dz-dw/dx を計算するときのみ, (x,z,u,w) の順番で, ord オプション false.
!
! $\frac{\partial v}{\partial x} -\frac{\partial u}{\partial y} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i,j)$ での発散は
! $\frac{v_{i,j+1}-v_{i,j-1}}{2dx} -\frac{u_{i+1,j}-u_{i-1,j}}{2dy} $
! とできる. これを用いて2次元発散を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
! 実質的には grad_1d が微分計算を担当するので, 境界の計算も自動で行う.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  double precision, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  double precision, intent(in) :: u(size(x),size(y))  ! x に対応する方向の 2 次元ベクトル成分
  double precision, intent(in) :: v(size(x),size(y))  ! y に対応する方向の 2 次元ベクトル成分
  double precision, intent(inout) :: val(size(x),size(y))  ! 2次元渦度
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  logical, intent(in),  optional :: ord  ! 微分の順番を入れ替えるオプション.
                 ! true なら, 入れ替えない, false なら, 入れ替える.
                 ! デフォルトは true, du/dz-dw/dx を計算するときのみ用いる.
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  logical :: order
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))
  double precision :: dvdx(size(x),size(y)), dudy(size(x),size(y))
  double precision :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "curl" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "curl" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "curl" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

!-- 方針は, x 方向に dvdy を計算し, y 方向に dudx を計算する.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scalex(i,j)*u(i,j)
              tmpv(i,j)=scaley(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpu(i,:), dudy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1dd( x, tmpv(:,j), dvdx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudy(i,j)/=undeff.and.dvdx(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
                 val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0d0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scalex(i,j)*u(i,j)
           tmpv(i,j)=scaley(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpu(i,:), dudy(i,:) )
     end do
     do j=1,ny
        call grad_1dd( x, tmpv(:,j), dvdx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
              val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0d0
           end if
        end do
     end do
  end if

!-- 回転の順番を逆にするオプション
!-- false なら, 順番を入れ替えて出力する.

  if(present(ord))then
     order=ord
  else
     order=.true.
  end if

  if(order.eqv..false.)then
     if(present(undeff))then
        do j=1,ny
           do i=1,nx
              if(val(i,j)/=undeff)then
                 val(i,j)=-val(i,j)
              end if
           end do
        end do
     else
        do j=1,ny
           do i=1,nx
              val(i,j)=-val(i,j)
           end do
        end do
     end if
  end if

end subroutine curld

!-----------------------------------------
!-----------------------------------------

subroutine div_3df( x, y, z, u, v, w, val, undeff, hx, hy, hz )
! 3次元発散計算ルーチン
!
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} +\frac{\partial w}{\partial z} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i,j,k)$ での発散は
! $\frac{u_{i+1,j,k}-u_{i-1,j,k}}{2dx} + \frac{v_{i,j+1,k}-v_{i,j-1,k}}{2dy} + \frac{w_{i,j,k+1}-w_{i,j,k-1}}{2dz} $
! とできる. これを用いて 3 次元発散を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
! 実質的には, grad_1d ルーチンが計算を行うので, 境界の処理も自動で行う.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: w(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 3 次元発散値
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  real, dimension(size(x),size(y),size(z)) :: dudx, dvdy, dwdz
  real, dimension(size(x),size(y),size(z)) :: tmpu, tmpv, tmpw
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "div_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "div_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "div_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "div_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- スケーリング変数の設定.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

!-- 方針は, x 方向に dvdy を計算し, y 方向に dudx を計算する.

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)/=undeff.and.v(i,j,k)/=undeff.and.w(i,j,k)/=undeff  &
  &              .and.scalex(i,j,k)/=undeff.and.scaley(i,j,k)/=undeff.and.  &
  &              scalez(i,j,k)/=undeff)then
                 tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
                 tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
                 tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
              else
                 tmpu(i,j,k)=undeff
                 tmpv(i,j,k)=undeff
                 tmpw(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1df( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- キャッシュから
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1df( x, tmpu(:,j,k), dudx(:,j,k), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1df( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- キャッシュから
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(dudx(i,j,k)/=undeff.and.dvdy(i,j,k)/=undeff.and.  &
  &              dwdz(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff.and.  &
  &              scaley(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0.and.scaley(i,j,k)/=0.0.and.  &
  &                 scalez(i,j,k)/=0.0)then
                    val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                            (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
                 else
                    val(i,j,k)=0.0
                 end if
              else
                 val(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
              tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
              tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1df( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1df( x, tmpu(:,j,k), dudx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1df( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0.and.scaley(i,j,k)/=0.0.and.scalez(i,j,k)/=0.0)then
                 val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                         (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
              else
                 val(i,j,k)=0.0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine div_3df

!-----------------------------------------
!-----------------------------------------

subroutine div_3dd( x, y, z, u, v, w, val, undeff, hx, hy, hz )
! 3次元発散計算ルーチン
!
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} +\frac{\partial w}{\partial z} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i,j,k)$ での発散は
! $\frac{u_{i+1,j,k}-u_{i-1,j,k}}{2dx} + \frac{v_{i,j+1,k}-v_{i,j-1,k}}{2dy} + \frac{w_{i,j,k+1}-w_{i,j,k-1}}{2dz} $
! とできる. これを用いて 3 次元発散を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
! 実質的には, grad_1d ルーチンが計算を行うので, 境界の処理も自動で行う.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  double precision, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  double precision, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  double precision, intent(in) :: v(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  double precision, intent(in) :: w(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  double precision, intent(inout) :: val(size(x),size(y),size(z))  ! 3 次元発散値
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  double precision, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  double precision, dimension(size(x),size(y),size(z)) :: dudx, dvdy, dwdz
  double precision, dimension(size(x),size(y),size(z)) :: tmpu, tmpv, tmpw
  double precision, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "div_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "div_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "div_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "div_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- スケーリング変数の設定.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

!-- 方針は, x 方向に dvdy を計算し, y 方向に dudx を計算する.

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)/=undeff.and.v(i,j,k)/=undeff.and.w(i,j,k)/=undeff  &
  &              .and.scalex(i,j,k)/=undeff.and.scaley(i,j,k)/=undeff.and.  &
  &              scalez(i,j,k)/=undeff)then
                 tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
                 tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
                 tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
              else
                 tmpu(i,j,k)=undeff
                 tmpv(i,j,k)=undeff
                 tmpw(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1dd( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- キャッシュから
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1dd( x, tmpu(:,j,k), dudx(:,j,k), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1dd( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- キャッシュから
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(dudx(i,j,k)/=undeff.and.dvdy(i,j,k)/=undeff.and.  &
  &              dwdz(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff.and.  &
  &              scaley(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0d0.and.scaley(i,j,k)/=0.0d0.and.  &
  &                 scalez(i,j,k)/=0.0d0)then
                    val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                            (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
                 else
                    val(i,j,k)=0.0d0
                 end if
              else
                 val(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
              tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
              tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1dd( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1dd( x, tmpu(:,j,k), dudx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1dd( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0d0.and.scaley(i,j,k)/=0.0d0.and.scalez(i,j,k)/=0.0d0)then
                 val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                         (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
              else
                 val(i,j,k)=0.0d0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine div_3dd

!-----------------------------------------
!-----------------------------------------

subroutine curl_3df( x, y, z, u, v, w, zeta, eta, xi, undeff, hx, hy, hz )
! 3 次元渦度を計算する.
! 引数の順番は右手系で x, y, z のデカルト座標系,
! それに対応するベクトル成分 u, v, w を代入すると,
! それに対応した渦度ベクトル成分 zeta, eta, xi が計算される.
! 実質は grad_1d が計算を担当するので, 境界の処理も自動で行う.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: w(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(inout) :: zeta(size(x),size(y),size(z))  ! 渦度ベクトル x 成分
  real, intent(inout) :: eta(size(x),size(y),size(z))  ! 渦度ベクトル y 成分
  real, intent(inout) :: xi(size(x),size(y),size(z))  ! 渦度ベクトル z 成分
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  real, allocatable, dimension(:,:,:) :: tmpyz1, tmpyz2, tmpyz3  ! キャッシュ用
  real, allocatable, dimension(:,:,:) :: tmpsyz1, tmpsyz2  ! キャッシュ用
  real, allocatable, dimension(:,:,:) :: tmpxz1, tmpxz2, tmpxz3  ! キャッシュ用
  real, allocatable, dimension(:,:,:) :: tmpsxz1, tmpsxz2  ! キャッシュ用

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, xi ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, eta ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, zeta ),  &
  &                                     "curl_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "curl_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "curl_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "curl_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpyz1(ny,nz,ompnum))
  allocate(tmpyz2(ny,nz,ompnum))
  allocate(tmpyz3(ny,nz,ompnum))
  allocate(tmpsyz1(ny,nz,ompnum))
  allocate(tmpsyz2(ny,nz,ompnum))
  allocate(tmpxz1(nx,nz,ompnum))
  allocate(tmpxz2(nx,nz,ompnum))
  allocate(tmpxz3(nx,nz,ompnum))
  allocate(tmpsxz1(nx,nz,ompnum))
  allocate(tmpsxz2(nx,nz,ompnum))

!-- スケーリング変数の設定.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x 軸に垂直な面上での x 方向の渦度ベクトルを 3 次元全域で計算.
     do i=1,nx
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curlf( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe), undeff,  &
  &                hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- キャッシュから
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x 軸に垂直な面上での x 方向の渦度ベクトルを 3 次元全域で計算.
     do i=1,nx
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curlf( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe),  &
  &                hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- キャッシュから
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! y 軸に垂直な面上での y 方向の渦度ベクトルを 3 次元全域で計算.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curlf( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe),  &
  &                undeff, ord=.false., &
  &                hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- キャッシュから
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curlf( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe), ord=.false., &
  &                hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- キャッシュから
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! z 軸に垂直な面上での z 方向の渦度ベクトルを 3 次元全域で計算.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curlf( x, y, u(:,:,k), v(:,:,k), xi(:,:,k), undeff,  &
  &                hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curlf( x, y, u(:,:,k), v(:,:,k), xi(:,:,k),  &
  &                hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine curl_3df

!-----------------------------------------
!-----------------------------------------

subroutine curl_3dd( x, y, z, u, v, w, zeta, eta, xi, undeff, hx, hy, hz )
! 3 次元渦度を計算する.
! 引数の順番は右手系で x, y, z のデカルト座標系,
! それに対応するベクトル成分 u, v, w を代入すると,
! それに対応した渦度ベクトル成分 zeta, eta, xi が計算される.
! 実質は grad_1d が計算を担当するので, 境界の処理も自動で行う.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  double precision, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  double precision, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  double precision, intent(in) :: v(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  double precision, intent(in) :: w(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  double precision, intent(inout) :: zeta(size(x),size(y),size(z))  ! 渦度ベクトル x 成分
  double precision, intent(inout) :: eta(size(x),size(y),size(z))  ! 渦度ベクトル y 成分
  double precision, intent(inout) :: xi(size(x),size(y),size(z))  ! 渦度ベクトル z 成分
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  double precision, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  double precision, allocatable, dimension(:,:,:) :: tmpyz1, tmpyz2, tmpyz3  ! キャッシュ用
  double precision, allocatable, dimension(:,:,:) :: tmpsyz1, tmpsyz2  ! キャッシュ用
  double precision, allocatable, dimension(:,:,:) :: tmpxz1, tmpxz2, tmpxz3  ! キャッシュ用
  double precision, allocatable, dimension(:,:,:) :: tmpsxz1, tmpsxz2  ! キャッシュ用

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, xi ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, eta ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, zeta ),  &
  &                                     "curl_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "curl_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "curl_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "curl_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpyz1(ny,nz,ompnum))
  allocate(tmpyz2(ny,nz,ompnum))
  allocate(tmpyz3(ny,nz,ompnum))
  allocate(tmpsyz1(ny,nz,ompnum))
  allocate(tmpsyz2(ny,nz,ompnum))
  allocate(tmpxz1(nx,nz,ompnum))
  allocate(tmpxz2(nx,nz,ompnum))
  allocate(tmpxz3(nx,nz,ompnum))
  allocate(tmpsxz1(nx,nz,ompnum))
  allocate(tmpsxz2(nx,nz,ompnum))

!-- スケーリング変数の設定.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x 軸に垂直な面上での x 方向の渦度ベクトルを 3 次元全域で計算.
     do i=1,nx
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curld( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                 tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe), undeff,  &
  &                 hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- キャッシュから
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x 軸に垂直な面上での x 方向の渦度ベクトルを 3 次元全域で計算.
     do i=1,nx
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curld( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                 tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe),  &
  &                 hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- キャッシュから
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! y 軸に垂直な面上での y 方向の渦度ベクトルを 3 次元全域で計算.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curld( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                 tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe),  &
  &                 undeff, ord=.false., &
  &                 hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- キャッシュから
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- キャッシュへ
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curld( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                 tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe), ord=.false., &
  &                 hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- キャッシュから
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! z 軸に垂直な面上での z 方向の渦度ベクトルを 3 次元全域で計算.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curld( x, y, u(:,:,k), v(:,:,k), xi(:,:,k), undeff,  &
  &                 hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curld( x, y, u(:,:,k), v(:,:,k), xi(:,:,k),  &
  &                 hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine curl_3dd

!-----------------------------------------
!-----------------------------------------

subroutine grad_1df( x, u, val, undef, hx )
! 1 次元のスカラー変数の勾配を計算する
! $\frac{\partial p}{\partial x} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}-p_{i-1}}{2dx} $
! とできる. これを用いて 1 次元勾配を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
  implicit none
  real, intent(in) :: x(:)  ! 空間座標
  real, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  real, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x))  ! x のスケール因子
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  real :: scalex(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad_1d" )
     end if
  end if

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0
     end do
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0)then
              val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
           else
              val(i)=0.0
           end if
        end if
     end do
!-- データ数のない両端の処理 ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0
        end if
     end if
  else
     do i=2,nx-1
        if(scalex(i)/=0.0)then
           val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
        else
           val(i)=0.0
        end if
     end do
!-- データ数のない両端の処理 ---
     if(scalex(1)/=0.0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0
     end if
     if(scalex(nx)/=0.0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0
     end if
  end if

end subroutine grad_1df

!-----------------------------------------
!-----------------------------------------

subroutine grad_1dd( x, u, val, undef, hx )
! 1 次元のスカラー変数の勾配を計算する
! $\frac{\partial p}{\partial x} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}-p_{i-1}}{2dx} $
! とできる. これを用いて 1 次元勾配を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
  implicit none
  double precision, intent(in) :: x(:)  ! 空間座標
  double precision, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  double precision, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x))  ! x のスケール因子
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  double precision :: scalex(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad_1d" )
     end if
  end if

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0d0
     end do
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0d0)then
              val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
           else
              val(i)=0.0d0
           end if
        end if
     end do
!-- データ数のない両端の処理 ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0d0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0d0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0d0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0d0
        end if
     end if
  else
     do i=2,nx-1
        if(scalex(i)/=0.0d0)then
           val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
        else
           val(i)=0.0d0
        end if
     end do
!-- データ数のない両端の処理 ---
     if(scalex(1)/=0.0d0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0d0
     end if
     if(scalex(nx)/=0.0d0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0d0
     end if
  end if

end subroutine grad_1dd

!-----------------------------------------
!-----------------------------------------

subroutine grad_2df( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 次元スカラー勾配のルーチンを用いて 2 次元勾配のベクトルを計算
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ を
  ! 2 次の中央差分近似で書き換えると, 点 $(i,j)$ での勾配は
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! とできる. これを用いて2次元勾配を計算.
  ! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
  ! 落ちる.
  ! 先に用いた 1 次元勾配計算ルーチンを 2 回呼び出すことにしている.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  real, intent(in) :: u(size(x),size(y))  ! 勾配をとる 2 次元スカラー成分
  real, intent(inout) :: valx(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  real, intent(inout) :: valy(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! x の配列要素数(size 関数で自動的に計算)
  integer :: ny  ! y の配列要素数(size 関数で自動的に計算)
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad_1df(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad_1df(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(valx(i,j)/=undeff.and.scalex(i,j)/=undeff)then
              if(scalex(i,j)/=0.0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0
              end if
           end if

           if(valy(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scaley(i,j)/=0.0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0
              end if
!-- ここで, else しないのは, grad_1d ルーチンですでに undeff が入っているから同じ作業に
!-- なるので, 割愛.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad_1df(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad_1df(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0
           end if
           if(scaley(i,j)/=0.0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0
           end if
        end do
     end do

  end if

end subroutine grad_2df

!-----------------------------------------
!-----------------------------------------

subroutine grad_2dd( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 次元スカラー勾配のルーチンを用いて 2 次元勾配のベクトルを計算
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ を
  ! 2 次の中央差分近似で書き換えると, 点 $(i,j)$ での勾配は
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! とできる. これを用いて2次元勾配を計算.
  ! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
  ! 落ちる.
  ! 先に用いた 1 次元勾配計算ルーチンを 2 回呼び出すことにしている.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  double precision, intent(in) :: u(size(x),size(y))  ! 勾配をとる 2 次元スカラー成分
  double precision, intent(inout) :: valx(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  double precision, intent(inout) :: valy(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! x の配列要素数(size 関数で自動的に計算)
  integer :: ny  ! y の配列要素数(size 関数で自動的に計算)
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad_1dd(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad_1dd(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(valx(i,j)/=undeff.and.scalex(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0d0
              end if
           end if

           if(valy(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scaley(i,j)/=0.0d0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0d0
              end if
!-- ここで, else しないのは, grad_1d ルーチンですでに undeff が入っているから同じ作業に
!-- なるので, 割愛.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad_1dd(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad_1dd(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0d0
           end if
           if(scaley(i,j)/=0.0d0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0d0
           end if
        end do
     end do

  end if

end subroutine grad_2dd

!-----------------------------------------
!-----------------------------------------

subroutine grad_3df( x, y, z, u, valx, valy, valz, undeff, hx, hy, hz )
  ! 1 次元スカラー勾配のルーチンを用いて 3 次元勾配のベクトルを計算.
  ! $\nabla p =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} ,\; \frac{\partial p}{\partial z} \right) $ を
  ! 2 次の中央差分近似で書き換えると, 点 $(i,j,k)$ での勾配は
  ! $\left(\frac{p_{i+1,j,k}-p_{i-1,j,k}}{2dx} ,\; \frac{p_{i,j+1,k}-p_{i,j-1,k}}{2dy} ,\; \frac{p_{i,j,k+1}-p_{i,j,k-1}}{2dz} \right) $
  ! とできる. これを用いて 3 次元勾配を計算.
  ! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
  ! 落ちる.
  ! 先に用いた 1 次元勾配計算ルーチンを 3 回呼び出すことにしている.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  real, intent(in) :: z(:)  ! z 方向の座標変数 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! 勾配をとる 2 次元スカラー成分
  real, intent(inout) :: valx(size(x),size(y),size(z))  ! 計算された y 方向の 2 次元勾配ベクトル
  real, intent(inout) :: valy(size(x),size(y),size(z))  ! 計算された y 方向の 2 次元勾配ベクトル
  real, intent(inout) :: valz(size(x),size(y),size(z))  ! 計算された z 方向の 2 次元勾配ベクトル
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valx ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valy ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valz ),  &
  &                                     "grad_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "grad_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "grad_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "grad_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1df( x, u(:,j,i), valx(:,j,i), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- キャッシュから
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- キャッシュから
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(valx(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0)then
                    valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
                 else
                    valx(i,j,k)=0.0
                 end if
              end if

              if(valy(i,j,k)/=undeff.and.scaley(i,j,k)/=undeff)then
                 if(scaley(i,j,k)/=0.0)then
                    valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
                 else
                    valy(i,j,k)=0.0
                 end if
              end if

              if(valz(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalez(i,j,k)/=0.0)then
                    valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
                 else
                    valz(i,j,k)=0.0
                 end if
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1df(x, u(:,j,i), valx(:,j,i))
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0)then
                 valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
              else
                 valx(i,j,k)=0.0
              end if
              if(scaley(i,j,k)/=0.0)then
                 valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
              else
                 valy(i,j,k)=0.0
              end if
              if(scalez(i,j,k)/=0.0)then
                 valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
              else
                 valz(i,j,k)=0.0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine grad_3df

!-----------------------------------------
!-----------------------------------------

subroutine grad_3dd( x, y, z, u, valx, valy, valz, undeff, hx, hy, hz )
  ! 1 次元スカラー勾配のルーチンを用いて 3 次元勾配のベクトルを計算.
  ! $\nabla p =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} ,\; \frac{\partial p}{\partial z} \right) $ を
  ! 2 次の中央差分近似で書き換えると, 点 $(i,j,k)$ での勾配は
  ! $\left(\frac{p_{i+1,j,k}-p_{i-1,j,k}}{2dx} ,\; \frac{p_{i,j+1,k}-p_{i,j-1,k}}{2dy} ,\; \frac{p_{i,j,k+1}-p_{i,j,k-1}}{2dz} \right) $
  ! とできる. これを用いて 3 次元勾配を計算.
  ! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
  ! 落ちる.
  ! 先に用いた 1 次元勾配計算ルーチンを 3 回呼び出すことにしている.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  double precision, intent(in) :: z(:)  ! z 方向の座標変数 [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! 勾配をとる 2 次元スカラー成分
  double precision, intent(inout) :: valx(size(x),size(y),size(z))  ! 計算された y 方向の 2 次元勾配ベクトル
  double precision, intent(inout) :: valy(size(x),size(y),size(z))  ! 計算された y 方向の 2 次元勾配ベクトル
  double precision, intent(inout) :: valz(size(x),size(y),size(z))  ! 計算された z 方向の 2 次元勾配ベクトル
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  double precision, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用
  double precision, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valx ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valy ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valz ),  &
  &                                     "grad_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "grad_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "grad_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "grad_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1dd( x, u(:,j,i), valx(:,j,i), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- キャッシュから
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- キャッシュから
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(valx(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0d0)then
                    valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
                 else
                    valx(i,j,k)=0.0d0
                 end if
              end if

              if(valy(i,j,k)/=undeff.and.scaley(i,j,k)/=undeff)then
                 if(scaley(i,j,k)/=0.0d0)then
                    valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
                 else
                    valy(i,j,k)=0.0d0
                 end if
              end if

              if(valz(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalez(i,j,k)/=0.0d0)then
                    valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
                 else
                    valz(i,j,k)=0.0d0
                 end if
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1dd(x, u(:,j,i), valx(:,j,i))
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0d0)then
                 valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
              else
                 valx(i,j,k)=0.0d0
              end if
              if(scaley(i,j,k)/=0.0d0)then
                 valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
              else
                 valy(i,j,k)=0.0d0
              end if
              if(scalez(i,j,k)/=0.0d0)then
                 valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
              else
                 valz(i,j,k)=0.0d0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine grad_3dd

!-----------------------------------------
!-----------------------------------------

subroutine grad4_1df( x, u, val, undef, hx )
! 1 次元のスカラー変数の勾配を計算する
! $\frac{\partial p}{\partial x} $ を
! 4 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $(2/3)*\frac{p_{i+1}-p_{i-1}}{dx} -(p_{i+2}-p_{i-2})/(12dx)$
! とできる. これを用いて 1 次元勾配を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
  implicit none
  real, intent(in) :: x(:)  ! 空間座標
  real, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  real, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x))  ! x のスケール因子
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  real :: scalex(size(x)), dx(size(x)), coe23, coe112

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad4_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad4_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad4_1d" )
     end if
  end if

  do i=2,nx-1
     dx(i)=0.5*(x(i+1)-x(i-1))
  end do
  dx(1)=x(2)-x(1)
  dx(nx)=x(nx)-x(nx-1)
  coe23=2.0/3.0
  coe112=1.0/12.0

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0
     end do
  end if

  if(present(undef))then
     do i=3,nx-2
        if(u(i+1)==undef.or.u(i-1)==undef.or.  &
  &        u(i+2)==undef.or.u(i-2)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0)then
              val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
           else
              val(i)=0.0
           end if
        end if
     end do

!-- データ数のない両端の処理 (両端の 1 つ内側) ---
     if(u(1)==undef.or.u(3)==undef.or.scalex(2)==undef)then
        val(2)=undef
     else
        if(scalex(2)/=0.0)then
           val(2)=0.5*(u(3)-u(1))/(scalex(2)*dx(2))
        else
           val(2)=0.0
        end if
     end if

     if(u(nx)==undef.or.u(nx-2)==undef.or.scalex(nx-1)==undef)then
        val(nx-1)=undef
     else
        if(scalex(nx-1)/=0.0)then
           val(nx-1)=0.5*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
        else
           val(nx-1)=0.0
        end if
     end if

!-- データ数のない両端の処理 ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0
        end if
     end if
  else
     do i=3,nx-2
        if(scalex(i)/=0.0)then
           val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
        else
           val(i)=0.0
        end if
     end do

!-- データ数のない両端の処理 (両端の 1 つ内側) ---
     if(scalex(2)/=0.0)then
        val(2)=0.5*(u(3)-u(1))/(scalex(2)*dx(2))
     else
        val(2)=0.0
     end if

     if(scalex(nx-1)/=0.0)then
        val(nx-1)=0.5*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
     else
        val(nx-1)=0.0
     end if

!-- データ数のない両端の処理 ---
     if(scalex(1)/=0.0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0
     end if
     if(scalex(nx)/=0.0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0
     end if
  end if

end subroutine grad4_1df

!-----------------------------------------
!-----------------------------------------

subroutine grad4_1dd( x, u, val, undef, hx )
! 1 次元のスカラー変数の勾配を計算する
! $\frac{\partial p}{\partial x} $ を
! 4 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $(2/3)*\frac{p_{i+1}-p_{i-1}}{dx} -(p_{i+2}-p_{i-2})/(12dx)$
! とできる. これを用いて 1 次元勾配を計算.
! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
! 落ちる.
  implicit none
  double precision, intent(in) :: x(:)  ! 空間座標
  double precision, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  double precision, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x))  ! x のスケール因子
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  double precision :: scalex(size(x)), dx(size(x)), coe23, coe112

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad4_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad4_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad4_1d" )
     end if
  end if

  do i=2,nx-1
     dx(i)=0.5d0*(x(i+1)-x(i-1))
  end do
  dx(1)=x(2)-x(1)
  dx(nx)=x(nx)-x(nx-1)
  coe23=2.0d0/3.0d0
  coe112=1.0d0/12.0d0

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0d0
     end do
  end if

  if(present(undef))then
     do i=3,nx-2
        if(u(i+1)==undef.or.u(i-1)==undef.or.  &
  &        u(i+2)==undef.or.u(i-2)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0d0)then
              val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
           else
              val(i)=0.0d0
           end if
        end if
     end do

!-- データ数のない両端の処理 (両端の 1 つ内側) ---
     if(u(1)==undef.or.u(3)==undef.or.scalex(2)==undef)then
        val(2)=undef
     else
        if(scalex(2)/=0.0d0)then
           val(2)=0.5d0*(u(3)-u(1))/(scalex(2)*dx(2))
        else
           val(2)=0.0d0
        end if
     end if

     if(u(nx)==undef.or.u(nx-2)==undef.or.scalex(nx-1)==undef)then
        val(nx-1)=undef
     else
        if(scalex(nx-1)/=0.0d0)then
           val(nx-1)=0.5d0*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
        else
           val(nx-1)=0.0d0
        end if
     end if

!-- データ数のない両端の処理 ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0d0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0d0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0d0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0d0
        end if
     end if
  else
     do i=3,nx-2
        if(scalex(i)/=0.0d0)then
           val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
        else
           val(i)=0.0d0
        end if
     end do

!-- データ数のない両端の処理 (両端の 1 つ内側) ---
     if(scalex(2)/=0.0d0)then
        val(2)=0.5d0*(u(3)-u(1))/(scalex(2)*dx(2))
     else
        val(2)=0.0d0
     end if

     if(scalex(nx-1)/=0.0d0)then
        val(nx-1)=0.5d0*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
     else
        val(nx-1)=0.0d0
     end if

!-- データ数のない両端の処理 ---
     if(scalex(1)/=0.0d0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0d0
     end if
     if(scalex(nx)/=0.0d0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0d0
     end if
  end if

end subroutine grad4_1dd

!-----------------------------------------
!-----------------------------------------

subroutine grad4_2df( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 次元スカラー勾配のルーチンを用いて 2 次元勾配のベクトルを計算
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ を
  ! 4 次の中央差分近似で書き換えると, 点 $(i,j)$ での勾配は
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! $(2/3)*\frac{p_{i+1,j}-p_{i-1,j}}{dx} -(p_{i+2,j}-p_{i-2,j})/(12dx),\; 
  ! (2/3)*\frac{p_{i,j+1}-p_{i,j-1}}{dy} -(p_{i,j+2}-p_{i,j-2})/(12dy)$
  ! とできる. これを用いて2次元勾配を計算.
  ! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
  ! 落ちる.
  ! 先に用いた 1 次元勾配計算ルーチンを 2 回呼び出すことにしている.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  real, intent(in) :: u(size(x),size(y))  ! 勾配をとる 2 次元スカラー成分
  real, intent(inout) :: valx(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  real, intent(inout) :: valy(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! x の配列要素数(size 関数で自動的に計算)
  integer :: ny  ! y の配列要素数(size 関数で自動的に計算)
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad4_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad4_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad4_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad4_1df(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad4_1df(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(valx(i,j)/=undeff.and.scalex(i,j)/=undeff)then
              if(scalex(i,j)/=0.0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0
              end if
           end if

           if(valy(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scaley(i,j)/=0.0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0
              end if
!-- ここで, else しないのは, grad_1d ルーチンですでに undeff が入っているから同じ作業に
!-- なるので, 割愛.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad4_1df(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad4_1df(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0
           end if
           if(scaley(i,j)/=0.0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0
           end if
        end do
     end do

  end if

end subroutine grad4_2df

!-----------------------------------------
!-----------------------------------------

subroutine grad4_2dd( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 次元スカラー勾配のルーチンを用いて 2 次元勾配のベクトルを計算
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ を
  ! 4 次の中央差分近似で書き換えると, 点 $(i,j)$ での勾配は
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! $(2/3)*\frac{p_{i+1,j}-p_{i-1,j}}{dx} -(p_{i+2,j}-p_{i-2,j})/(12dx),\; 
  ! (2/3)*\frac{p_{i,j+1}-p_{i,j-1}}{dy} -(p_{i,j+2}-p_{i,j-2})/(12dy)$
  ! とできる. これを用いて2次元勾配を計算.
  ! データ点が足りない隅の領域では, 1 次の差分近似で計算するので, 少し精度が
  ! 落ちる.
  ! 先に用いた 1 次元勾配計算ルーチンを 2 回呼び出すことにしている.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  double precision, intent(in) :: u(size(x),size(y))  ! 勾配をとる 2 次元スカラー成分
  double precision, intent(inout) :: valx(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  double precision, intent(inout) :: valy(size(x),size(y))  ! 計算された y 方向の 2 次元勾配ベクトル
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! x の配列要素数(size 関数で自動的に計算)
  integer :: ny  ! y の配列要素数(size 関数で自動的に計算)
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad4_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad4_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad4_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad4_1dd(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad4_1dd(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(valx(i,j)/=undeff.and.scalex(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0d0
              end if
           end if

           if(valy(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scaley(i,j)/=0.0d0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0d0
              end if
!-- ここで, else しないのは, grad_1d ルーチンですでに undeff が入っているから同じ作業に
!-- なるので, 割愛.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad4_1dd(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad4_1dd(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0d0
           end if
           if(scaley(i,j)/=0.0d0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0d0
           end if
        end do
     end do

  end if

end subroutine grad4_2dd

!-----------------------------------------
!-----------------------------------------

subroutine zast_2_w_2d( x, y, zeta, zf, zt, u, v, w, wh, undef )
! terrain following 座標系で評価されている各風速成分をデカルト座標系での
! 各風速成分に変換する.
! ただし, terrain following 系の水平風速はデカルト座標系でも
! 大きさが変化しないため, このルーチンでは鉛直風速のみ変換を行う.
! また, 3 次元デカルト系の格子点上に変換するのではなく, terrain following 系の
! 各点で風速成分をデカルト系方向に変換するだけ.
! 現在, 水平方向にはデカルト系にのみ対応している.
! もし, 座標点も変換する場合は, vert_coord_conv モジュールを使用のこと.
  implicit none
  real, dimension(:), intent(in) :: x  ! 東西方向の座標 [m]
  real, dimension(:), intent(in) :: y  ! 東西方向の座標 [m]
  real, dimension(size(x),size(y)), intent(in) :: zeta  ! terrain 系鉛直座標 [m]
  real, dimension(size(x),size(y)), intent(in) :: zf  ! 地表面高度座標 [m]
  real, dimension(size(x),size(y)), intent(in) :: zt  ! 座標最上端 [m]
  real, dimension(size(x),size(y)), intent(in) :: u  ! zeta における東西風 [m/s]
  real, dimension(size(x),size(y)), intent(in) :: v  ! zeta における南北風 [m/s]
  real, dimension(size(x),size(y)), intent(in) :: w  ! zeta における鉛直風 [m/s]
  real, dimension(size(x),size(y)), intent(inout) :: wh  ! デカルト系における鉛直風 [m/s]
  real, intent(in), optional :: undef  ! 欠損値
  integer :: i, j, nx, ny
  real, dimension(size(x),size(y)) :: dx, dy
  real :: j31, j32, jd, coe

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zeta ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zf ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zt ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, w ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, wh ),  &
  &                                     "zast_2_w_2d" )
  end if

  call grad_2d( x, y, zf, dx, dy )

  if(present(undef))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undef.and.v(i,j)/=undef.and.w(i,j)/=undef)then
              jd=1.0-zf(i,j)/zt(i,j)
              coe=zeta(i,j)/zt(i,j)-1.0
              j31=coe*dx(i,j)
              j32=coe*dy(i,j)
              wh(i,j)=(u(i,j)*j31+v(i,j)*j32+w(i,j))/jd
           else
              wh(i,j)=undef
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           jd=1.0-zf(i,j)/zt(i,j)
           coe=zeta(i,j)/zt(i,j)-1.0
           j31=coe*dx(i,j)
           j32=coe*dy(i,j)
           wh(i,j)=(u(i,j)*j31+v(i,j)*j32+w(i,j))/jd
        end do
     end do
  end if

end subroutine zast_2_w_2d

!-----------------------------------------
!-----------------------------------------

subroutine zast_2_w_3d( x, y, zeta, zf, zt, u, v, w, wh, undef )
! terrain following 座標系で評価されている各風速成分をデカルト座標系での
! 各風速成分に変換する.
! ただし, terrain following 系の水平風速はデカルト座標系でも
! 大きさが変化しないため, このルーチンでは鉛直風速のみ変換を行う.
! また, 3 次元デカルト系の格子点上に変換するのではなく, terrain following 系の
! 各点で風速成分をデカルト系方向に変換するだけ.
! 現在, 水平方向にはデカルト系にのみ対応している.
! もし, 座標点も変換する場合は, vert_coord_conv モジュールを使用のこと.
  implicit none
  real, dimension(:,:,:), intent(in) :: zeta  ! terrain 系の鉛直座標 [m]
  real, dimension(size(zeta,1)), intent(in) :: x  ! 東西方向の座標 [m]
  real, dimension(size(zeta,2)), intent(in) :: y  ! 東西方向の座標 [m]
  real, dimension(size(zeta,1),size(zeta,2)), intent(in) :: zf  ! 地表面高度座標 [m]
  real, dimension(size(zeta,1),size(zeta,2)), intent(in) :: zt  ! 座標最上端 [m]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(in) :: u  ! zeta における東西風 [m/s]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(in) :: v  ! zeta における南北風 [m/s]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(in) :: w  ! zeta における鉛直風 [m/s]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(inout) :: wh  ! デカルト系における鉛直風 [m/s]
  real, intent(in), optional :: undef  ! 欠損値
  integer :: k, nx, ny, nz

  nx=size(zeta,1)
  ny=size(zeta,2)
  nz=size(zeta,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, x ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_1d( ny, y ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zf ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zt ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, wh ),  &
  &                                     "zast_2_w_3d" )
  end if

  if(present(undef))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call zast_2_w_2d( x, y, zeta(:,:,k), zf, zt,  &
  &                       u(:,:,k), v(:,:,k), w(:,:,k),  &
  &                       wh(:,:,k), undef )
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call zast_2_w_2d( x, y, zeta(:,:,k), zf, zt,  &
  &                       u(:,:,k), v(:,:,k), w(:,:,k),  &
  &                       wh(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine zast_2_w_3d

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_1df( x, u, val, undef, hx )
! 1 次元のスカラー変数のラプラシアンを計算する
! $\frac{\partial ^2p}{\partial x^2} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! とできる. これを用いて 1 次元ラプラシアンを計算.
! データ点が足りない隅の領域では, undef を定義する.
! option で undef が定義されていない場合は, 0.0 を代入する.
! At each boundary, res = (val_n-val_{n-1})/(x_n-x_{n-1})^2
!                        -(val_n-val_{n-2})/{(x_n-x_{n-1})(x_n-x_{n-2})}
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  real, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  real, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x))  ! x 方向のスケール因子
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  real :: scalex(size(x)), tmpu(size(x)), tmpv(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "laplacian_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "laplacian_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "laplacian_1d" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx))then
     do i=1,nx
        if(hx(i)==0.0)then
           if(present(undef))then
              scalex(i)=undef
           else
              scalex(i)=0.0
           end if
        else
           scalex(i)=1.0/(hx(i)**2)
        end if
     end do
     if(present(undef))then
        call grad_1df( x, scalex, tmpv, undef=undef )
        call grad_1df( x, u, tmpu, undef=undef )
     else
        call grad_1df( x, scalex, tmpv )
        call grad_1df( x, u, tmpu )
     end if
  else
     do i=1,nx
        scalex(i)=1.0
     end do
     tmpv=0.0
     tmpu=0.0
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.u(i)==undef.or.  &
  &        scalex(i)==undef.or.tmpu(i)==undef.or.tmpv(i)==undef)then
           val(i)=undef
        else
           val(i)=4.0*((u(i+1)+u(i-1)-2.0*u(i))/((x(i+1)-x(i-1))**2))  &
  &              *scalex(i)+0.5*tmpu(i)*tmpv(i)
        end if
     end do
!-- データ数のない両端の処理 ---
     if(u(1)==undef.or.u(2)==undef.or.u(3)==undef.or.  &
  &     scalex(1)==undef.or.tmpu(1)==undef.or.tmpv(1)==undef)then
        val(1)=undef
     else
        val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &            -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &            +0.5*tmpu(1)*tmpv(1)
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.u(nx-2)==undef.or.  &
  &     scalex(nx)==undef.or.tmpu(nx)==undef.or.tmpv(nx)==undef)then
        val(nx)=undef
     else
        val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &            -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &            +0.5*tmpu(nx)*tmpv(nx)
     end if
  else
     do i=2,nx-1
        val(i)=4.0*((u(i+1)+u(i-1)-2.0*u(i))/((x(i+1)-x(i-1))**2))  &
  &           *scalex(i)+0.5*tmpu(i)*tmpv(i)
     end do
!-- データ数のない両端の処理 ---
     val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &         -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &         +0.5*tmpu(1)*tmpv(1)
     val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &         -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &         +0.5*tmpu(nx)*tmpv(nx)
  end if

end subroutine laplacian_1df

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_1dd( x, u, val, undef, hx )
! 1 次元のスカラー変数のラプラシアンを計算する
! $\frac{\partial ^2p}{\partial x^2} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! とできる. これを用いて 1 次元ラプラシアンを計算.
! データ点が足りない隅の領域では, undef を定義する.
! option で undef が定義されていない場合は, 0.0d0 を代入する.
! At each boundary, res = (val_n-val_{n-1})/(x_n-x_{n-1})^2
!                        -(val_n-val_{n-2})/{(x_n-x_{n-1})(x_n-x_{n-2})}
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  double precision, intent(in) :: u(size(x))  ! 上の空間配列に対応する 1 次元スカラー値
  double precision, intent(inout) :: val(size(x))  ! スカラー値の x 方向の勾配
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x))  ! x 方向のスケール因子
  integer :: i  ! イタレーション用添字
  integer :: nx  ! 配列要素数
  double precision :: scalex(size(x)), tmpu(size(x)), tmpv(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "laplacian_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "laplacian_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "laplacian_1d" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx))then
     do i=1,nx
        if(hx(i)==0.0d0)then
           if(present(undef))then
              scalex(i)=undef
           else
              scalex(i)=0.0d0
           end if
        else
           scalex(i)=1.0d0/(hx(i)**2)
        end if
     end do
     if(present(undef))then
        call grad_1dd( x, scalex, tmpv, undef=undef )
        call grad_1dd( x, u, tmpu, undef=undef )
     else
        call grad_1dd( x, scalex, tmpv )
        call grad_1dd( x, u, tmpu )
     end if
  else
     do i=1,nx
        scalex(i)=1.0d0
     end do
     tmpv=0.0d0
     tmpu=0.0d0
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.u(i)==undef.or.  &
  &        scalex(i)==undef.or.tmpu(i)==undef.or.tmpv(i)==undef)then
           val(i)=undef
        else
           val(i)=4.0d0*((u(i+1)+u(i-1)-2.0d0*u(i))/((x(i+1)-x(i-1))**2))  &
  &              *scalex(i)+0.5d0*tmpu(i)*tmpv(i)
        end if
     end do
!-- データ数のない両端の処理 ---
     if(u(1)==undef.or.u(2)==undef.or.u(3)==undef.or.  &
  &     scalex(1)==undef.or.tmpu(1)==undef.or.tmpv(1)==undef)then
        val(1)=undef
     else
        val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &            -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &            +0.5d0*tmpu(1)*tmpv(1)
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.u(nx-2)==undef.or.  &
  &     scalex(nx)==undef.or.tmpu(nx)==undef.or.tmpv(nx)==undef)then
        val(nx)=undef
     else
        val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &            -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &            +0.5d0*tmpu(nx)*tmpv(nx)
     end if
  else
     do i=2,nx-1
        val(i)=4.0d0*((u(i+1)+u(i-1)-2.0d0*u(i))/((x(i+1)-x(i-1))**2))  &
  &           *scalex(i)+0.5d0*tmpu(i)*tmpv(i)
     end do
!-- データ数のない両端の処理 ---
     val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &         -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &         +0.5d0*tmpu(1)*tmpv(1)
     val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &         -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &         +0.5d0*tmpu(nx)*tmpv(nx)
  end if

end subroutine laplacian_1dd

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_2df( x, y, u, val, undef, hx, hy )
! 2 次元のスカラー変数のラプラシアンを計算する
! $\frac{\partial ^2p}{\partial x^2} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! とできる. これを用いて 1 次元ラプラシアンを計算.
! データ点が足りない隅の領域では, undef を定義する.
! option で undef が定義されていない場合は, 0.0 を代入する.
! 一般直交座標系への対応はまだ.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  real, intent(in) :: u(size(x),size(y))  ! 上の空間配列に対応する 2 次元スカラー値
  real, intent(inout) :: val(size(x),size(y))  ! スカラー値の 2 次元方向のラプラシアン
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i, j  ! イタレーション用添字
  integer :: nx, ny  ! 配列要素数
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))
  real :: sx(size(x),size(y)), sy(size(x),size(y))
  real :: tmpu(size(x),size(y)), tmpv(size(x),size(y))
  real :: tmpsx(size(x),size(y)), tmpsy(size(x),size(y))
  real :: tmpsu(size(x),size(y)), tmpsv(size(x),size(y))
  real :: ddu(size(x),size(y)), ddv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "laplacian_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "laplacian_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "laplacian_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "laplacian_2d" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx).or.present(hy))then
     if(present(hx))then
        do j=1,ny
           do i=1,nx
              sx(i,j)=hx(i,j)
           end do
        end do
     else
        sx=1.0
     end if
     if(present(hy))then
        do j=1,ny
           do i=1,nx
              sy(i,j)=hy(i,j)
           end do
        end do
     else
        sy=1.0
     end if

     if(present(undef))then
        do j=1,ny
           do i=1,nx
              if(sx(i,j)/=undef.and.sx(i,j)/=0.0.and.sy(i,j)/=undef.and.  &
  &              sy(i,j)/=0.0)then
                 scalex(i,j)=1.0/(sx(i,j)**2)
                 scaley(i,j)=1.0/(sy(i,j)**2)
                 tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
                 tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
              else
                 scalex(i,j)=undef
                 scaley(i,j)=undef
                 tmpsx(i,j)=undef
                 tmpsy(i,j)=undef
              end if
           end do
        end do
        do j=1,ny
           call grad_1df( x, tmpsy(:,j), tmpsu(:,j), undef=undef )
        end do
        do i=1,nx
           call grad_1df( y, tmpsx(i,:), tmpsv(i,:), undef=undef )
        end do
        call grad_2df( x, y, u, ddu, ddv, undeff=undef )
     else
        do j=1,ny
           do i=1,nx
              scalex(i,j)=1.0/(sx(i,j)**2)
              scaley(i,j)=1.0/(sy(i,j)**2)
              tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
              tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
           end do
        end do
        do j=1,ny
           call grad_1df( x, tmpsy(:,j), tmpsu(:,j) )
        end do
        do i=1,nx
           call grad_1df( y, tmpsx(i,:), tmpsv(i,:) )
        end do
        call grad_2df( x, y, u, ddu, ddv )
     end if
  else
     scalex=1.0
     scaley=1.0
     tmpsu=0.0
     tmpsv=0.0
     ddu=0.0
     ddv=0.0
  end if

  if(present(undef))then

     val=undef

     do j=1,ny
        call laplacian_1df( x, u(1:nx,j), tmpu(1:nx,j), undef )
     end do
     do i=1,nx
        call laplacian_1df( y, u(i,1:ny), tmpv(i,1:ny), undef )
     end do

     do j=1,ny
        do i=1,nx
           if(tmpu(i,j)/=undef.and.tmpv(i,j)/=undef.and.  &
  &           scalex(i,j)/=undef.and.scaley(i,j)/=undef.and.  &
  &           tmpsu(i,j)/=undef.and.tmpsv(i,j)/=undef.and.  &
  &           ddu(i,j)/=undef.and.ddv(i,j)/=undef)then
              val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                   +tmpv(i,j)*scaley(i,j)  &
  &                   +0.5*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                   +0.5*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
           end if
        end do
     end do

  else

     val=0.0

     do j=1,ny
        call laplacian_1df( x, u(1:nx,j), tmpu(1:nx,j) )
     end do
     do i=1,nx
        call laplacian_1df( y, u(i,1:ny), tmpv(i,1:ny) )
     end do

     do j=1,ny
        do i=1,nx
           val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                +tmpv(i,j)*scaley(i,j)  &
  &                +0.5*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                +0.5*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
        end do
     end do

  end if

end subroutine laplacian_2df

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_2dd( x, y, u, val, undef, hx, hy )
! 2 次元のスカラー変数のラプラシアンを計算する
! $\frac{\partial ^2p}{\partial x^2} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! とできる. これを用いて 1 次元ラプラシアンを計算.
! データ点が足りない隅の領域では, undef を定義する.
! option で undef が定義されていない場合は, 0.0d0 を代入する.
! 一般直交座標系への対応はまだ.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  double precision, intent(in) :: u(size(x),size(y))  ! 上の空間配列に対応する 2 次元スカラー値
  double precision, intent(inout) :: val(size(x),size(y))  ! スカラー値の 2 次元方向のラプラシアン
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y 方向のスケール因子
  integer :: i, j  ! イタレーション用添字
  integer :: nx, ny  ! 配列要素数
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))
  double precision :: sx(size(x),size(y)), sy(size(x),size(y))
  double precision :: tmpu(size(x),size(y)), tmpv(size(x),size(y))
  double precision :: tmpsx(size(x),size(y)), tmpsy(size(x),size(y))
  double precision :: tmpsu(size(x),size(y)), tmpsv(size(x),size(y))
  double precision :: ddu(size(x),size(y)), ddv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "laplacian_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "laplacian_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "laplacian_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "laplacian_2d" )
     end if
  end if

!-- スケーリング変数の設定.

  if(present(hx).or.present(hy))then
     if(present(hx))then
        do j=1,ny
           do i=1,nx
              sx(i,j)=hx(i,j)
           end do
        end do
     else
        sx=1.0d0
     end if
     if(present(hy))then
        do j=1,ny
           do i=1,nx
              sy(i,j)=hy(i,j)
           end do
        end do
     else
        sy=1.0d0
     end if

     if(present(undef))then
        do j=1,ny
           do i=1,nx
              if(sx(i,j)/=undef.and.sx(i,j)/=0.0d0.and.sy(i,j)/=undef.and.  &
  &              sy(i,j)/=0.0d0)then
                 scalex(i,j)=1.0d0/(sx(i,j)**2)
                 scaley(i,j)=1.0d0/(sy(i,j)**2)
                 tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
                 tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
              else
                 scalex(i,j)=undef
                 scaley(i,j)=undef
                 tmpsx(i,j)=undef
                 tmpsy(i,j)=undef
              end if
           end do
        end do
        do j=1,ny
           call grad_1dd( x, tmpsy(:,j), tmpsu(:,j), undef=undef )
        end do
        do i=1,nx
           call grad_1dd( y, tmpsx(i,:), tmpsv(i,:), undef=undef )
        end do
        call grad_2dd( x, y, u, ddu, ddv, undeff=undef )
     else
        do j=1,ny
           do i=1,nx
              scalex(i,j)=1.0d0/(sx(i,j)**2)
              scaley(i,j)=1.0d0/(sy(i,j)**2)
              tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
              tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
           end do
        end do
        do j=1,ny
           call grad_1dd( x, tmpsy(:,j), tmpsu(:,j) )
        end do
        do i=1,nx
           call grad_1dd( y, tmpsx(i,:), tmpsv(i,:) )
        end do
        call grad_2dd( x, y, u, ddu, ddv )
     end if
  else
     scalex=1.0d0
     scaley=1.0d0
     tmpsu=0.0d0
     tmpsv=0.0d0
     ddu=0.0d0
     ddv=0.0d0
  end if

  if(present(undef))then

     val=undef

     do j=1,ny
        call laplacian_1dd( x, u(1:nx,j), tmpu(1:nx,j), undef )
     end do
     do i=1,nx
        call laplacian_1dd( y, u(i,1:ny), tmpv(i,1:ny), undef )
     end do

     do j=1,ny
        do i=1,nx
           if(tmpu(i,j)/=undef.and.tmpv(i,j)/=undef.and.  &
  &           scalex(i,j)/=undef.and.scaley(i,j)/=undef.and.  &
  &           tmpsu(i,j)/=undef.and.tmpsv(i,j)/=undef.and.  &
  &           ddu(i,j)/=undef.and.ddv(i,j)/=undef)then
              val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                   +tmpv(i,j)*scaley(i,j)  &
  &                   +0.5d0*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                   +0.5d0*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
           end if
        end do
     end do

  else

     val=0.0d0

     do j=1,ny
        call laplacian_1dd( x, u(1:nx,j), tmpu(1:nx,j) )
     end do
     do i=1,nx
        call laplacian_1dd( y, u(i,1:ny), tmpv(i,1:ny) )
     end do

     do j=1,ny
        do i=1,nx
           val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                +tmpv(i,j)*scaley(i,j)  &
  &                +0.5d0*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                +0.5d0*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
        end do
     end do

  end if

end subroutine laplacian_2dd

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_3df( x, y, z, u, val, undef, hx, hy, hz )
! 3 次元のスカラー変数のラプラシアンを計算する
! $\frac{\partial ^2p}{\partial x^2} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! とできる. これを用いて 1 次元ラプラシアンを計算.
! データ点が足りない隅の領域では, undef を定義する.
! option で undef が定義されていない場合は, 0.0 を代入する.
! 一般直交座標系への対応はまだ.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  real, intent(in) :: z(:)  ! z 方向の座標変数 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! 上の空間配列に対応する 3 次元スカラー値
  real, intent(inout) :: val(size(x),size(y),size(z))  ! スカラー値の 3 次元方向のラプラシアン
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i, j, k  ! イタレーション用添字
  integer :: nx, ny, nz  ! 配列要素数
  real :: scalex(size(x),size(y),size(z))
  real :: scaley(size(x),size(y),size(z))
  real :: scalez(size(x),size(y),size(z))
  real :: sx(size(x),size(y),size(z))
  real :: sy(size(x),size(y),size(z))
  real :: sz(size(x),size(y),size(z))
  real :: tmpu(size(x),size(y),size(z)), tmpv(size(x),size(y),size(z))
  real :: tmpw(size(x),size(y),size(z))
  real :: tmpsu(size(x),size(y),size(z)), tmpsv(size(x),size(y),size(z))
  real :: tmpsw(size(x),size(y),size(z))
  real :: h1(size(x),size(y),size(z)), h2(size(x),size(y),size(z))
  real :: h3(size(x),size(y),size(z))
  real :: ddu(size(x),size(y),size(z)), ddv(size(x),size(y),size(z))
  real :: ddw(size(x),size(y),size(z))
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! 配列要素数

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "laplacian_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "laplacian_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "laplacian_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- スケーリング変数の設定.

  if(present(hx).or.present(hy).or.present(hz))then

     if(present(hx))then
        sx=hx
     else
        sx=1.0
     end if
     if(present(hy))then
        sy=hy
     else
        sy=1.0
     end if
     if(present(hz))then
        sz=hz
     else
        sz=1.0
     end if

     if(present(undef))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(sx(i,j,k)/=undef.and.sy(i,j,k)/=undef.and.sz(i,j,k)/=undef)then
                    scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                    scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                    scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                    h1(i,j,k)=1.0/(sx(i,j,k)**2)
                    h2(i,j,k)=1.0/(sy(i,j,k)**2)
                    h3(i,j,k)=1.0/(sz(i,j,k)**2)
                 else
                    scalex(i,j,k)=undef
                    scaley(i,j,k)=undef
                    scalez(i,j,k)=undef
                    h1(i,j,k)=undef
                    h2(i,j,k)=undef
                    h3(i,j,k)=undef
                 end if
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                 scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                 scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                 h1(i,j,k)=1.0/(sx(i,j,k)**2)
                 h2(i,j,k)=1.0/(sy(i,j,k)**2)
                 h3(i,j,k)=1.0/(sz(i,j,k)**2)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     end if

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1df( x, scalex(:,j,k), tmpsu(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- キャッシュから
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- キャッシュから
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3df( x, y, z, u, ddu, ddv, ddw, undeff=undef )

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1df( x, scalex(:,j,k), tmpsu(:,j,k) )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- キャッシュから
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- キャッシュから
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3df( x, y, z, u, ddu, ddv, ddw )
     end if

  else

     scalex=1.0
     scaley=1.0
     scalez=1.0
     tmpsu=0.0
     tmpsv=0.0
     tmpsw=0.0
     ddu=0.0
     ddv=0.0
     ddw=0.0

  end if

  if(present(undef))then

     val=undef

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1df( x, u(:,j,k), tmpu(:,j,k), undef )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef )

           !-- キャッシュから
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef )

           !-- キャッシュから
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(tmpu(i,j,k)/=undef.and.tmpv(i,j,k)/=undef.and.  &
  &              h1(i,j,k)/=undef.and.h2(i,j,k)/=undef.and.  &
  &              h3(i,j,k)/=undef.and.tmpsu(i,j,k)/=undef.and.  &
  &              tmpsv(i,j,k)/=undef.and.tmpsw(i,j,k)/=undef.and.  &
  &              ddu(i,j,k)/=undef.and.ddv(i,j,k)/=undef.and.ddw(i,j,k)/=undef)then
                 val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                        +h2(i,j,k)*tmpv(i,j,k)  &
  &                        +h3(i,j,k)*tmpw(i,j,k)  &
  &                        +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                        +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                        +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
              end if
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  else

     val=0.0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1df( x, u(:,j,k), tmpu(:,j,k) )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                     +h2(i,j,k)*tmpv(i,j,k)  &
  &                     +h3(i,j,k)*tmpw(i,j,k)  &
  &                     +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                     +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                     +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine laplacian_3df

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_3dd( x, y, z, u, val, undef, hx, hy, hz )
! 3 次元のスカラー変数のラプラシアンを計算する
! $\frac{\partial ^2p}{\partial x^2} $ を
! 2 次の中央差分近似で書き換えると, 点 $(i)$ での勾配は
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! とできる. これを用いて 1 次元ラプラシアンを計算.
! データ点が足りない隅の領域では, undef を定義する.
! option で undef が定義されていない場合は, 0.0 を代入する.
! 一般直交座標系への対応はまだ.
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  double precision, intent(in) :: z(:)  ! z 方向の座標変数 [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! 上の空間配列に対応する 3 次元スカラー値
  double precision, intent(inout) :: val(size(x),size(y),size(z))  ! スカラー値の 3 次元方向のラプラシアン
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i, j, k  ! イタレーション用添字
  integer :: nx, ny, nz  ! 配列要素数
  double precision :: scalex(size(x),size(y),size(z))
  double precision :: scaley(size(x),size(y),size(z))
  double precision :: scalez(size(x),size(y),size(z))
  double precision :: sx(size(x),size(y),size(z))
  double precision :: sy(size(x),size(y),size(z))
  double precision :: sz(size(x),size(y),size(z))
  double precision :: tmpu(size(x),size(y),size(z)), tmpv(size(x),size(y),size(z))
  double precision :: tmpw(size(x),size(y),size(z))
  double precision :: tmpsu(size(x),size(y),size(z)), tmpsv(size(x),size(y),size(z))
  double precision :: tmpsw(size(x),size(y),size(z))
  double precision :: h1(size(x),size(y),size(z)), h2(size(x),size(y),size(z))
  double precision :: h3(size(x),size(y),size(z))
  double precision :: ddu(size(x),size(y),size(z)), ddv(size(x),size(y),size(z))
  double precision :: ddw(size(x),size(y),size(z))
  double precision, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! 配列要素数

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "laplacian_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "laplacian_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "laplacian_3d" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- スケーリング変数の設定.

  if(present(hx).or.present(hy).or.present(hz))then

     if(present(hx))then
        sx=hx
     else
        sx=1.0
     end if
     if(present(hy))then
        sy=hy
     else
        sy=1.0
     end if
     if(present(hz))then
        sz=hz
     else
        sz=1.0
     end if

     if(present(undef))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(sx(i,j,k)/=undef.and.sy(i,j,k)/=undef.and.sz(i,j,k)/=undef)then
                    scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                    scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                    scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                    h1(i,j,k)=1.0/(sx(i,j,k)**2)
                    h2(i,j,k)=1.0/(sy(i,j,k)**2)
                    h3(i,j,k)=1.0/(sz(i,j,k)**2)
                 else
                    scalex(i,j,k)=undef
                    scaley(i,j,k)=undef
                    scalez(i,j,k)=undef
                    h1(i,j,k)=undef
                    h2(i,j,k)=undef
                    h3(i,j,k)=undef
                 end if
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                 scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                 scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                 h1(i,j,k)=1.0/(sx(i,j,k)**2)
                 h2(i,j,k)=1.0/(sy(i,j,k)**2)
                 h3(i,j,k)=1.0/(sz(i,j,k)**2)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     end if

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1dd( x, scalex(:,j,k), tmpsu(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- キャッシュから
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- キャッシュから
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3dd( x, y, z, u, ddu, ddv, ddw, undeff=undef )

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1dd( x, scalex(:,j,k), tmpsu(:,j,k) )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- キャッシュから
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- キャッシュから
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3dd( x, y, z, u, ddu, ddv, ddw )
     end if

  else

     scalex=1.0
     scaley=1.0
     scalez=1.0
     tmpsu=0.0
     tmpsv=0.0
     tmpsw=0.0
     ddu=0.0
     ddv=0.0
     ddw=0.0

  end if

  if(present(undef))then

     val=undef

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1dd( x, u(:,j,k), tmpu(:,j,k), undef )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef )

           !-- キャッシュから
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef )

           !-- キャッシュから
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(tmpu(i,j,k)/=undef.and.tmpv(i,j,k)/=undef.and.  &
  &              h1(i,j,k)/=undef.and.h2(i,j,k)/=undef.and.  &
  &              h3(i,j,k)/=undef.and.tmpsu(i,j,k)/=undef.and.  &
  &              tmpsv(i,j,k)/=undef.and.tmpsw(i,j,k)/=undef.and.  &
  &              ddu(i,j,k)/=undef.and.ddv(i,j,k)/=undef.and.ddw(i,j,k)/=undef)then
                 val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                        +h2(i,j,k)*tmpv(i,j,k)  &
  &                        +h3(i,j,k)*tmpw(i,j,k)  &
  &                        +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                        +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                        +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
              end if
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  else

     val=0.0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1dd( x, u(:,j,k), tmpu(:,j,k) )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                     +h2(i,j,k)*tmpv(i,j,k)  &
  &                     +h3(i,j,k)*tmpw(i,j,k)  &
  &                     +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                     +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                     +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine laplacian_3dd

!-----------------------------------------
!-----------------------------------------

subroutine z_2_zeta( z, zf, zt, zeta )
! 幾何座標 z を terrain following 座標に変換する.
  implicit none
  real, intent(in) :: z(:,:,:)    ! デカルト幾何座標系鉛直座標 [m]
  real, intent(in) :: zf(size(z,1),size(z,2))   ! 地表面高度 [m]
  real, intent(in) :: zt(size(z,1),size(z,2))   ! 定義域最高度 [m]
  real, intent(inout) :: zeta(size(z,1),size(z,2),size(z,3))  ! terrain following 座標 [m]

  integer :: i, j, k
  integer :: nx, ny, nz

  nx=size(z,1)
  ny=size(z,2)
  nz=size(z,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zf ),  &
  &                                     "z_2_zeta" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zt ),  &
  &                                     "z_2_zeta" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, zeta ),  &
  &                                     "z_2_zeta" )
  end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

  do k=1,nz
     do j=1,ny
        do i=1,nx
           zeta(i,j,k)=zt(i,j)*(z(i,j,k)-zf(i,j))/(zt(i,j)-zf(i,j))
        end do
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine z_2_zeta

!-----------------------------------------
!-----------------------------------------

logical function local_peak_checker_1df( val, cs, ip, np, undef )
! 任意配列 val について, 極大・極小値があるかをチェックする.
! 存在すれば .true. を返す.
! 極値探索のアルゴリズムは近傍 3 点での勾配を計算し, 勾配の反転する
! 箇所をチェックする. さらに 2 階微分で上下の凹凸をチェックする. 
! 勾配がゼロで凹凸もゼロの場合, そこは極値とは判定しない.
! 勾配の符号が反転し, 上に凸の場合極大値, 下に凸の場合極小値と判定し,
! 勾配の符号が反転する 2 点を含めて近傍 4 点での最大・最小値を計算する. 
  implicit none
  real, dimension(:), intent(in) :: val  ! チェックする配列
  character(1), intent(in) :: cs  ! '+': 極大値, '-': 極小値
  integer, dimension(size(val)), optional :: ip  ! 極値の格子点番号
  integer, optional :: np  ! 極値の個数
  real, intent(in), optional :: undef  ! 未定義
  integer :: i, nx, j, nnp
  integer, dimension(size(val)) :: iip, mval
  real, dimension(size(val)) :: x, dvdx, d2vdx2, pmflag
  real :: tmpval
  logical :: lflag

  nx=size(val)
  nnp=0
  lflag=.false.

  x=(/((real(i)),i=1,nx)/)

  if(present(undef))then
     dvdx=undef
     d2vdx2=undef
     pmflag=undef

     call grad_1df( x, val, dvdx, undef=undef )
     call laplacian_1df( x, val, d2vdx2, undef=undef )

     do i=2,nx
        if(dvdx(i-1)/=undef.and.dvdx(i)/=undef)then
           pmflag(i)=dvdx(i-1)*dvdx(i)
        end if
     end do

     do i=3,nx-1
        if(pmflag(i)<=0.0.and.pmflag(i)/=undef)then
           lflag=.true.
           if(pmflag(i)==0.0)then  ! just peak
              if(d2vdx2(i)>0.0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              else if(d2vdx2(i)<0.0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              end if
           else
              if(d2vdx2(i)>0.0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval>val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              else if(d2vdx2(i)<0.0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval<val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              end if
           end if
        end if
     end do

  else

     call grad_1df( x, val, dvdx )
     call laplacian_1df( x, val, d2vdx2 )

     do i=2,nx
        pmflag(i)=dvdx(i-1)*dvdx(i)
     end do

     do i=3,nx-1
        if(pmflag(i)<=0.0)then
           lflag=.true.
           if(pmflag(i)==0.0)then  ! just peak
              if(d2vdx2(i)>0.0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              else if(d2vdx2(i)<0.0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              end if
           else
              if(d2vdx2(i)>0.0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval>val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              else if(d2vdx2(i)<0.0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval<val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              end if
           end if
        end if
     end do

  end if

  if(present(np))then
     np=nnp
  end if

  if(present(ip))then
     ip(1:nnp)=iip(1:nnp)
  end if

  local_peak_checker_1df=lflag

  return
end function local_peak_checker_1df

!-----------------------------------------
!-----------------------------------------

logical function local_peak_checker_1dd( val, cs, ip, np, undef )
! 任意配列 val について, 極大・極小値があるかをチェックする.
! 存在すれば .true. を返す.
! 極値探索のアルゴリズムは近傍 3 点での勾配を計算し, 勾配の反転する
! 箇所をチェックする. さらに 2 階微分で上下の凹凸をチェックする. 
! 勾配がゼロで凹凸もゼロの場合, そこは極値とは判定しない.
! 勾配の符号が反転し, 上に凸の場合極大値, 下に凸の場合極小値と判定し,
! 勾配の符号が反転する 2 点を含めて近傍 4 点での最大・最小値を計算する. 
  implicit none
  double precision, dimension(:), intent(in) :: val  ! チェックする配列
  character(1), intent(in) :: cs  ! '+': 極大値, '-': 極小値
  integer, dimension(size(val)), optional :: ip  ! 極値の格子点番号
  integer, optional :: np  ! 極値の個数
  double precision, intent(in), optional :: undef  ! 未定義
  integer :: i, nx, j, nnp
  integer, dimension(size(val)) :: iip, mval
  double precision, dimension(size(val)) :: x, dvdx, d2vdx2, pmflag
  double precision :: tmpval
  logical :: lflag

  nx=size(val)
  nnp=0
  lflag=.false.

  x=(/((dble(i)),i=1,nx)/)

  if(present(undef))then
     dvdx=undef
     d2vdx2=undef
     pmflag=undef

     call grad_1dd( x, val, dvdx, undef=undef )
     call laplacian_1dd( x, val, d2vdx2, undef=undef )

     do i=2,nx
        if(dvdx(i-1)/=undef.and.dvdx(i)/=undef)then
           pmflag(i)=dvdx(i-1)*dvdx(i)
        end if
     end do

     do i=3,nx-1
        if(pmflag(i)<=0.0d0.and.pmflag(i)/=undef)then
           lflag=.true.
           if(pmflag(i)==0.0d0)then  ! just peak
              if(d2vdx2(i)>0.0d0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0d0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              else if(d2vdx2(i)<0.0d0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0d0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              end if
           else
              if(d2vdx2(i)>0.0d0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval>val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              else if(d2vdx2(i)<0.0d0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval<val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              end if
           end if
        end if
     end do

  else

     call grad_1dd( x, val, dvdx )
     call laplacian_1dd( x, val, d2vdx2 )

     do i=2,nx
        pmflag(i)=dvdx(i-1)*dvdx(i)
     end do

     do i=3,nx-1
        if(pmflag(i)<=0.0d0)then
           lflag=.true.
           if(pmflag(i)==0.0d0)then  ! just peak
              if(d2vdx2(i)>0.0d0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0d0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              else if(d2vdx2(i)<0.0d0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 if(dvdx(i-1)==0.0d0)then
                    iip(nnp)=i-1
                 else
                    iip(nnp)=i
                 end if
              end if
           else
              if(d2vdx2(i)>0.0d0.and.cs(1:1)=='-')then  ! 下に凸 (極小値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval>val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              else if(d2vdx2(i)<0.0d0.and.cs(1:1)=='+')then  ! 上に凸 (極大値)
                 nnp=nnp+1
                 iip(nnp)=i-2

                 tmpval=val(i-2)
                 do j=i-1,i+1
                    if(tmpval<val(j))then
                       tmpval=val(j)
                       iip(nnp)=j
                    end if
                 end do
              end if
           end if
        end if
     end do

  end if

  if(present(np))then
     np=nnp
  end if

  if(present(ip))then
     ip(1:nnp)=iip(1:nnp)
  end if

  local_peak_checker_1dd=lflag

  return
end function local_peak_checker_1dd

!-----------------------------------------
!-----------------------------------------


subroutine turb_visc( signal, x, y, z, u, v, w, rho, nuh, nuv, val, undef, hx, hy, hz, sfctau )
! デカルト座標系における乱流粘性項を計算する.
  implicit none
  character(1) :: signal  ! デカルト座標系の何番目の乱流成分かを判定する.
                  ! [1] = デカルト座標右手系における x 座標成分 (方程式 u 成分)
                  ! [2] = デカルト座標右手系における y 座標成分 (方程式 v 成分)
                  ! [3] = デカルト座標右手系における z 座標成分 (方程式 w 成分)
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: w(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: rho(size(x),size(y),size(z))  ! 基本場の密度 [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! 水平渦粘性係数
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! 鉛直渦粘性係数
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 乱流フラックス
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  real, intent(in), optional :: sfctau(size(x),size(y))  ! 地表面からのフラックス
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real, dimension(size(x),size(y),size(z),3) :: tau  ! signal 方向に
              ! 作用する 1,2,3 面に垂直な応力
  character(1) :: signaltau(3)
  integer :: id
  real, dimension(size(x),size(y)) :: stau
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

  signaltau=(/ '1', '2', '3' /)
  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "turb_visc" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "turb_visc" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "turb_visc" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "turb_visc" )
     end if
     if(present(sfctau))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfctau ),  &
  &                                        "turb_visc" )
     end if
  end if

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0

  do id=1,3
     if(present(sfctau))then
        stau(:,:)=sfctau(:,:)
        if(present(undef))then
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfctau=stau, undef=undef )
        else
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfctau=stau )
        end if
     else
        if(present(undef))then
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, undef=undef )
        else
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez )
        end if
     end if
  end do

!-- 乱流項の計算は, 3 次元発散を行うことと等価であるため,
!-- 以下では, div_3d を用いて計算を行う.

  if(present(undef))then
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &                hx=scalex, hy=scaley, hz=scalez, undeff=undef )
  else
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &                hx=scalex, hy=scaley, hz=scalez )
  end if

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine Reynolds_stress( signal, x, y, z, u, v, w, rho, nuh, nuv, val,  &
  &                         undef, hx, hy, hz, sfctau )
! デカルト座標系におけるレイノルズ応力テンソルを計算する.
  implicit none
  character(2) :: signal  ! 計算するテンソル成分.
                  ! ['11', '22', '33'] = それぞれ対角テンソル成分
                  ! ['12', '13', '21', '23', '31', '32'] = それぞれ非対角
                  ! テンソル成分. ただし, 対称テンソルであるため, '12'='21' を
                  ! 計算していることに注意.
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: w(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: rho(size(x),size(y),size(z))  ! 基本場の密度 [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! 水平渦粘性係数
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! 鉛直渦粘性係数
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 計算されたテンソル成分
! 現在, 以下のオプションは使用していない.
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  real, intent(in), optional :: sfctau(size(x),size(y))  ! モデル最下層での応力が与えられていれば, その値を代入.
        ! この値は何もせず, 単に val の最下層に代入されるだけ.
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real :: stau(size(x),size(y))
  real :: sxx(size(x),size(y),size(z)), nu(size(x),size(y),size(z))
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  logical, dimension(size(x),size(y),size(z)) :: undeflag

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "Reynolds_stress" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "Reynolds_stress" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "Reynolds_stress" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "Reynolds_stress" )
     end if
     if(present(sfctau))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfctau ),  &
  &                                        "Reynolds_stress" )
     end if
  end if

  undeflag=.false.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0
  stau=0.0

!-- 地表面フラックスの値を代入.

  if(present(sfctau))then
     if(signal(2:2)=='3'.and.signal(1:1)/='3')then
        stau(:,:)=sfctau(:,:)
     end if
  end if

!-- [NOTE]
!-- 以下, 文字で case の or ができないため, 
!-- if 文の入れ子ではなく, if 文の並列表記で case と同じように見せかける.
!-- これはもちろん, 上から順に if をたどるが, どの場合も 2 種類以上の if に
!-- 合致しないことが既知であるために可能となる書き方であり,
!-- 並列表記した if の 2 パターン以上に合致してしまうような条件文では,
!-- case の代用には用いることができないことに注意.
!-- 本ライブラリでこのような紛らわしい表記をしている場合は必ず NOTE が入る.

!-- 応力テンソルの計算
  if(present(undef))then
     call deform_tensor( signal, x, y, z, u, v, w, sxx,  &
  &                      hx=scalex, hy=scaley, hz=scalez, undef=undef )
  else
     call deform_tensor( signal, x, y, z, u, v, w, sxx,  &
  &                      hx=scalex, hy=scaley, hz=scalez )
  end if

  if(signal(1:2)=='12'.or.signal(1:2)=='21')then

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuh(i,j,k)
           end do
        end do
     end do

  else if(signal(1:2)=='23'.or.signal(1:2)=='32')then

     if(signal(2:2)=='3')then
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuv(i,j,k)
              end do
           end do
        end do
     else
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuh(i,j,k)
              end do
           end do
        end do
     end if

  else if(signal(1:2)=='13'.or.signal(1:2)=='31')then

     if(signal(2:2)=='3')then
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuv(i,j,k)
              end do
           end do
        end do
     else
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuh(i,j,k)
              end do
           end do
        end do
     end if

  else if(signal(1:2)=='11')then

     if(present(undef))then
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez, undeff=undef )
     else
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez )
     end if

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuh(i,j,k)
           end do
        end do
     end do

  else if(signal(1:2)=='22')then

     if(present(undef))then
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez, undeff=undef )
     else
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez )
     end if

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuh(i,j,k)
           end do
        end do
     end do

  else if(signal(1:2)=='33')then

     if(present(undef))then
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez, undeff=undef )
     else
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez )
     end if

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuv(i,j,k)
           end do
        end do
     end do
  end if

  if(present(undef))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(val(i,j,k)==undef.or.nu(i,j,k)==undef.or.sxx(i,j,k)==undef)then
                 undeflag(i,j,k)=.true.
              end if
           end do
        end do
     end do
  end if

!-- 以下, 最下層は地表面フラックスを代入するかどうかのオプションのため, 別ループ

  if(present(sfctau))then
     do j=1,ny
        do i=1,nx
           if(undeflag(i,j,1).eqv..false.)then
              val(i,j,1)=stau(i,j)
           else
              val(i,j,1)=undef
           end if
        end do
     end do
  else
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)
     do j=1,ny
        do i=1,nx
           if(undeflag(i,j,1).eqv..false.)then
              val(i,j,1)=rho(i,j,1)*nu(i,j,1)*(sxx(i,j,1)-(2.0/3.0)*val(i,j,1))
           else
              val(i,j,1)=undef
           end if
        end do
     end do
!$omp end do
!$omp end parallel
  end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
  do k=2,nz
     do j=1,ny
        do i=1,nx
           if(undeflag(i,j,k).eqv..false.)then
              val(i,j,k)=rho(i,j,k)*nu(i,j,k)*(sxx(i,j,k)-(2.0/3.0)*val(i,j,k))
           else
              val(i,j,k)=undef
           end if
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine deform_tensor( signal, x, y, z, u, v, w, val, undef, hx, hy, hz )
! デカルト座標系における変形速度テンソルを計算する.
  implicit none
  character(2) :: signal  ! 計算するテンソル成分.
                  ! ['11', '22', '33'] = それぞれ対角テンソル成分
                  ! ['12', '13', '21', '23', '31', '32'] = それぞれ非対角
                  ! テンソル成分. ただし, 対称テンソルであるため, '12'='21' を
                  ! 計算していることに注意.
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: w(size(x),size(y),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 計算されたテンソル成分
! 現在, 以下のオプションは使用していない.
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用
  real, dimension(size(x),size(y),size(z)) :: tmpu, tmpv, tmpw
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  real, dimension(size(x),size(y),size(z)) :: ddx, ddy, ddz
  logical, dimension(size(x),size(y),size(z)) :: undeflag

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! 配列要素数

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "deform_tensor" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "deform_tensor" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "deform_tensor" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "deform_tensor" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "deform_tensor" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "deform_tensor" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "deform_tensor" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  undeflag=.false.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

!-- [NOTE]
!-- 以下, 文字で case の or ができないため, 
!-- if 文の入れ子ではなく, if 文の並列表記で case と同じように見せかける.
!-- これはもちろん, 上から順に if をたどるが, どの場合も 2 種類以上の if に
!-- 合致しないことが既知であるために可能となる書き方であり,
!-- 並列表記した if の 2 パターン以上に合致してしまうような条件文では,
!-- case の代用には用いることができないことに注意.
!-- 本ライブラリでこのような紛らわしい表記をしている場合は必ず NOTE が入る.

  if(present(undef))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)==undef.or.v(i,j,k)==undef.or.w(i,j,k)==undef)then
                 undeflag(i,j,k)=.true.
              end if
           end do
        end do
     end do
  end if

  if(signal(1:2)=='12'.or.signal(1:2)=='21')then

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 tmpu(i,j,k)=u(i,j,k)/scalex(i,j,k)
                 tmpv(i,j,k)=v(i,j,k)/scaley(i,j,k)
              else
                 tmpu(i,j,k)=undef
                 tmpv(i,j,k)=undef
              end if
           end do
        end do
     end do

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpv(:,j,k), ddx(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=tmpu(i,1:ny,k)

              !-- dtmpu/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- キャッシュから
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if((ddx(i,j,k)==undef).or.(ddy(i,j,k)==undef))then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpv(:,j,k), ddx(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=tmpu(i,1:ny,k)

              !-- dtmpu/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- キャッシュから
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=ddx(i,j,k)*scaley(i,j,k)/scalex(i,j,k)+  &
  &                         ddy(i,j,k)*scalex(i,j,k)/scaley(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='23'.or.signal(1:2)=='32')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 tmpv(i,j,k)=v(i,j,k)/scaley(i,j,k)
                 tmpw(i,j,k)=w(i,j,k)/scalez(i,j,k)
              else
                 tmpv(i,j,k)=undef
                 tmpw(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=tmpw(i,1:ny,k)

              !-- dtmpw/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- キャッシュから
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=tmpv(i,j,1:nz)

              !-- dtmpv/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- キャッシュから
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if((ddy(i,j,k)==undef).or.(ddz(i,j,k)==undef))then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=tmpw(i,1:ny,k)

              !-- dtmpw/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- キャッシュから
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=tmpv(i,j,1:nz)

              !-- dtmpv/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- キャッシュから
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=ddy(i,j,k)*scalez(i,j,k)/scaley(i,j,k)+  &
  &                         ddz(i,j,k)*scaley(i,j,k)/scalez(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='13'.or.signal(1:2)=='31')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 tmpu(i,j,k)=u(i,j,k)/scalex(i,j,k)
                 tmpw(i,j,k)=w(i,j,k)/scalez(i,j,k)
              else
                 tmpu(i,j,k)=undef
                 tmpw(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=tmpu(i,j,1:nz)

              !-- dtmpu/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- キャッシュから
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpw(:,j,k), ddx(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if((ddz(i,j,k)==undef).or.(ddx(i,j,k)==undef))then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=tmpu(i,j,1:nz)

              !-- dtmpu/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- キャッシュから
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpw(:,j,k), ddx(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel
     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=ddz(i,j,k)*scalex(i,j,k)/scalez(i,j,k)+  &
  &                         ddx(i,j,k)*scalez(i,j,k)/scalex(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='11')then

     !-- scale は undef 指定していないので, undef 計算はしない.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=scalex(i,1:ny,k)

           !-- dhx/dy => ddy
           call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

           !-- dhx/dz => ddz
           call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, u(:,j,k), ddx(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(ddx(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, u(:,j,k), ddx(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=2.0*(ddx(i,j,k)+ddy(i,j,k)*v(i,j,k)/scaley(i,j,k)+  &
  &                              ddz(i,j,k)*w(i,j,k)/scalez(i,j,k))/scalex(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='22')then

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=v(i,1:ny,k)

              !-- dv/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- キャッシュから
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(ddy(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=v(i,1:ny,k)

              !-- dv/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- キャッシュから
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

     !-- scale は undef 指定していないので, undef 計算はしない.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpz1(1:nz,omppe)=scaley(i,j,1:nz)

           !-- dhy/dz => ddz
           call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- キャッシュから
           ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1d( x, scaley(:,j,k), ddx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=2.0*(ddy(i,j,k)+ddz(i,j,k)*w(i,j,k)/scalez(i,j,k)+  &
  &                              ddx(i,j,k)*u(i,j,k)/scalex(i,j,k))/scaley(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='33')then

     !-- scale は undef 指定していないので, undef 計算はしない.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- キャッシュへ
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
           tmpy1(1:ny,omppe)=scalez(i,1:ny,k)

           !-- dhz/dy => ddy
           call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- キャッシュから
           ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=w(i,j,1:nz)

              !-- dw/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- キャッシュから
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(ddz(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=w(i,j,1:nz)

              !-- dw/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- キャッシュから
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1d( x, scalez(:,j,k), ddx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=2.0*(ddz(i,j,k)+ddx(i,j,k)*u(i,j,k)/scalex(i,j,k)+  &
  &                              ddy(i,j,k)*v(i,j,k)/scaley(i,j,k))/scalez(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine turb_diff( x, y, z, phi, rho, nuh, nuv, val, undef, hx, hy, hz, sfcphi )
! 乱流拡散項を計算する.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  real, intent(in) :: phi(size(x),size(y),size(z))  ! 乱流拡散を計算するスカラー量
  real, intent(in) :: rho(size(x),size(y),size(z))  ! 基本場の密度 [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! 水平渦粘性係数
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! 鉛直渦粘性係数
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 3 次元発散値
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  real, intent(in), optional :: sfcphi(size(x),size(y))      ! 地表面からのフラックス
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real, dimension(size(x),size(y),size(z),3) :: tau  ! signal 方向に
              ! 作用する 1,2,3 面に垂直な応力
  character(1) :: signaltau(3)
  integer :: id
  real, dimension(size(x),size(y)) :: stau
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

  signaltau=(/ '1', '2', '3' /)

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, phi ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "turb_diff" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "turb_diff" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "turb_diff" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "turb_diff" )
     end if
     if(present(sfcphi))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfcphi ),  &
  &                                        "turb_diff" )
     end if
  end if

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0

  do id=1,3
     if(present(sfcphi))then
        stau(:,:)=sfcphi(:,:)
        if(present(undef))then
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfcphi=stau, undef=undef )
        else
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfcphi=stau )
        end if
     else
        if(present(undef))then
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, undef=undef )
        else
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez )
        end if
     end if
  end do

!-- 乱流項は 3 次元発散と同じ形となるので, div_3d でまとめる.
  if(present(undef))then
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &               hx=scalex, hy=scaley, hz=scalez, undeff=undef )
  else
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &               hx=scalex, hy=scaley, hz=scalez )
  end if

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine Reynolds_scal( signal, x, y, z, phi, rho, nuh, nuv, val, undef, hx, hy, hz, sfcphi )
! スカラー量の乱流フラックスを計算する.
  implicit none
  character(1) :: signal  ! デカルト座標系の何番目の乱流フラックス成分かを判定する.
                          ! [1] = デカルト座標右手系における x 座標成分
                          ! [2] = デカルト座標右手系における y 座標成分
                          ! [3] = デカルト座標右手系における z 座標成分
  real, intent(in) :: x(:)  ! x 方向の空間座標 [m]
  real, intent(in) :: y(:)  ! y 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! z 方向の空間座標 [m]
  real, intent(in) :: phi(size(x),size(y),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: rho(size(x),size(y),size(z))  ! 基本場の密度 [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! 水平渦拡散係数
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! 鉛直渦拡散係数
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 計算されたテンソル成分
! 現在, 以下のオプションは使用していない.
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x 方向のスケール因子
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y 方向のスケール因子
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z 方向のスケール因子
  real, intent(in), optional :: sfcphi(size(x),size(y))  ! モデル最下層でのフラックスが与えられていれば, その値を代入.
        ! この値は何もせず, 単に val の最下層に代入されるだけ.
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! キャッシュ用
  real, dimension(size(x),size(y)) :: stau
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  logical, dimension(size(x),size(y),size(z)) :: undeflag

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! 配列要素数

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, phi ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "Reynolds_scal" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "Reynolds_scal" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "Reynolds_scal" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "Reynolds_scal" )
     end if
     if(present(sfcphi))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfcphi ),  &
  &                                        "Reynolds_scal" )
     end if
  end if

!-- キャッシュ用
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  undeflag=.false.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0

  if(present(sfcphi))then
     if(signal(1:1)=='3')then
        stau(:,:)=sfcphi(:,:)
     end if
  end if

  select case (signal(1:1))
  case ('1')
     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, phi(:,j,k), val(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(val(i,j,k)==undef.or.nuh(i,j,k)==undef.or.nuv(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, phi(:,j,k), val(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=rho(i,j,k)*nuh(i,j,k)*val(i,j,k)/scalex(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  case ('2')
     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=phi(i,1:ny,k)

              !-- dphi/dy=>val
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- キャッシュから
              val(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(val(i,j,k)==undef.or.nuh(i,j,k)==undef.or.nuv(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpy1(1:ny,omppe)=phi(i,1:ny,k)

              !-- dphi/dy=>val
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- キャッシュから
              val(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=rho(i,j,k)*nuh(i,j,k)*val(i,j,k)/scaley(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  case ('3')
     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=phi(i,j,1:nz)

              !-- dphi/dz=>val
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- キャッシュから
              val(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do j=1,ny
           do i=1,nx
              do k=1,nz
                 if(val(i,j,k)==undef.or.nuh(i,j,k)==undef.or.nuv(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- キャッシュへ
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合
              tmpz1(1:nz,omppe)=phi(i,j,1:nz)

              !-- dphi/dz=>val
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- キャッシュから
              val(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=rho(i,j,k)*nuv(i,j,k)*val(i,j,k)/scalez(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     do j=1,ny
        do i=1,nx
           if(present(sfcphi))then
              if(undeflag(i,j,1).eqv..false.)then
                 val(i,j,1)=stau(i,j)
              else
                 val(i,j,1)=undef
              end if
           end if
        end do
     end do

  end select

end subroutine

!-----------------------------------------
!-----------------------------------------

! 以下, オリジナル zast_2_vel_?d であるが, CReSS では水平風速に関して
! terrain following の前後で成分変化はないので, 今のところ以下ルーチンは
! 使う用途が見当たらない.
!subroutine zast_2_vel_2d( x, y, zast, u, v, w, uh, vh, wh, undef )
!! terrain following 座標系で評価されている風速を水平風と鉛直風に変換する.
!! ただし, 3 次元デカルト系に変換するのではなく, terrain following 座標系の
!! 各点で風速成分をデカルト系方向に変換するだけ.
!! 現在, 水平方向にはデカルト系にのみ対応している.
!! もし, 座標点も変換する場合は, vert_coord_conv モジュールを使用のこと.
!  implicit none
!  real, dimension(:), intent(in) :: x  ! 東西方向の座標 [m]
!  real, dimension(:), intent(in) :: y  ! 東西方向の座標 [m]
!  real, dimension(size(x),size(y)), intent(in) :: zast  ! terrain 系の, ある 1 層 [m]
!  real, dimension(size(x),size(y)), intent(in) :: u  ! zast における東西風 [m/s]
!  real, dimension(size(x),size(y)), intent(in) :: v  ! zast における南北風 [m/s]
!  real, dimension(size(x),size(y)), intent(in) :: w  ! zast における鉛直風 [m/s]
!  real, dimension(size(x),size(y)), intent(inout) :: uh  ! デカルト系における東西風 [m/s]
!  real, dimension(size(x),size(y)), intent(inout) :: vh  ! デカルト系における南北風 [m/s]
!  real, dimension(size(x),size(y)), intent(inout) :: wh  ! デカルト系における鉛直風 [m/s]
!  real, intent(in), optional :: undef  ! 欠損値
!  integer :: i, j, nx, ny
!  real, dimension(size(x),size(y)) :: dx, dy
!  real :: cosx, sinx, cosy, siny
!
!  nx=size(x)
!  ny=size(y)
!
!  call grad_2d( x, y, zast, dx, dy )
!
!  if(present(undef))then
!     do j=1,ny
!        do i=1,nx
!           if(u(i,j)/=undef.and.v(i,j)/=undef.and.w(i,j)/=undef)then
!              cosx=1.0/sqrt(1.0+dx(i,j)*dx(i,j))
!              cosy=1.0/sqrt(1.0+dy(i,j)*dy(i,j))
!              sinx=dx(i,j)*cosx
!              siny=dy(i,j)*cosy
!              uh(i,j)=u(i,j)*cosx
!              vh(i,j)=v(i,j)*cosy
!              wh(i,j)=w(i,j)+u(i,j)*sinx+v(i,j)*siny
!           else
!              uh(i,j)=undef
!              vh(i,j)=undef
!              wh(i,j)=undef
!           end if
!        end do
!     end do
!  else
!     do j=1,ny
!        do i=1,nx
!           cosx=1.0/sqrt(1.0+dx(i,j)*dx(i,j))
!           cosy=1.0/sqrt(1.0+dy(i,j)*dy(i,j))
!           sinx=dx(i,j)*cosx
!           siny=dy(i,j)*cosy
!           uh(i,j)=u(i,j)*cosx
!           vh(i,j)=v(i,j)*cosy
!           wh(i,j)=w(i,j)+u(i,j)*sinx+v(i,j)*siny
!        end do
!     end do
!  end if
!
!end subroutine zast_2_vel_2d
!
!!-----------------------------------------
!!-----------------------------------------
!
!subroutine zast_2_vel_3d( x, y, zast, u, v, w, uh, vh, wh, undef )
!! terrain following 座標系で評価されている風速を水平風と鉛直風に変換する.
!! ただし, 3 次元デカルト系に変換するのではなく, terrain following 座標系の
!! 各点で風速成分をデカルト系方向に変換するだけ.
!! 現在, 水平方向にはデカルト系にのみ対応している.
!! もし, 座標点も変換する場合は, vert_coord_conv モジュールを使用のこと.
!  implicit none
!  real, dimension(:,:,:), intent(in) :: zast  ! terrain 系の高度座標 [m]
!  real, dimension(size(zast,1)), intent(in) :: x  ! 東西方向の座標 [m]
!  real, dimension(size(zast,2)), intent(in) :: y  ! 東西方向の座標 [m]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(in) :: u  ! zast における東西風 [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(in) :: v  ! zast における南北風 [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(in) :: w  ! zast における鉛直風 [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(inout) :: uh  ! デカルト系における東西風 [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(inout) :: vh  ! デカルト系における南北風 [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(inout) :: wh  ! デカルト系における鉛直風 [m/s]
!  real, intent(in), optional :: undef  ! 欠損値
!  integer :: k, nx, ny, nz
!
!  nx=size(zast,1)
!  ny=size(zast,2)
!  nz=size(zast,3)
!
!  if(present(undef))then
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(k)
!     do k=1,nz
!        call zast_2_vel_2d( x, y, zast(:,:,k), u(:,:,k), v(:,:,k), w(:,:,k),  &
!  &                         uh(:,:,k), vh(:,:,k), wh(:,:,k), undef )
!     end do
!!$omp end do
!!$omp end parallel
!
!  else
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(k)
!     do k=1,nz
!        call zast_2_vel_2d( x, y, zast(:,:,k), u(:,:,k), v(:,:,k), w(:,:,k),  &
!  &                         uh(:,:,k), vh(:,:,k), wh(:,:,k) )
!     end do
!!$omp end do
!!$omp end parallel
!
!  end if
!
!end subroutine zast_2_vel_3d


end module Derivation
