!-----------------------------------------------------------------------
!     Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!-----------------------------------------------------------------------

module vert_coord_conv
! 鉛直方向の座標変換に関するモジュール

  use Statistics

  implicit none

contains

subroutine v1_2_v2_1d( v1, v2, val1, val2, undef, sig )
! 任意の鉛直座標 v1 から別の鉛直座標 v2 への val1 という変数の線形変換を行う.
! 本ルーチンは配列に格納される各物理量は幾何高度で見て低い方が要素番号の
! 小さいものに格納されていると仮定している.
  implicit none
  real, dimension(:), intent(in) :: v1  ! 基準となる鉛直座標 [v2 と同じ単位]
  real, dimension(:), intent(in) :: v2  ! 変換先の鉛直座標 [v1 と同じ単位]
  real, dimension(size(v1)), intent(in) :: val1  ! v1 で定義される任意変数
  real, dimension(size(v2)), intent(inout) :: val2  ! v2 で定義される任意変数
  real, intent(in), optional :: undef  ! 変換できない場合の未定義値
                                       ! デフォルトはゼロ
  character(1), intent(in), optional :: sig  ! 変換前後の座標が
                            ! 漸増, 漸減関数のどちらかを指定する.
                            ! 漸増の場合 -> "p", 漸減の場合 -> "m".
                            ! デフォルトは "p".
                            ! [例] 気圧の場合は "m", 高度の場合は "p".
  integer :: i, j, n1, n2
  real :: tmp1(size(v1)), tmp2(size(v2))
  real :: tmpv1(size(v1)), tmpv2(size(v2))
  real :: undeff
  character(1) :: vc1

  n1=size(v1)
  n2=size(v2)

!-- オプション引数の処理

  if(present(undef))then
     undeff=undef
  else
     undeff=0.0
  end if

  if(present(sig))then
     vc1(1:1)=sig(1:1)
  else
     vc1(1:1)='p'
  end if

!-- 漸減関数について, 漸増となるように置き換える.

  if(vc1(1:1)=='m')then
     do i=1,n1
        tmp1(i)=v1(n1-i+1)
        tmpv1(i)=val1(n1-i+1)
     end do
     do i=1,n2
        tmp2(i)=v2(n2-i+1)
     end do
  else
     do i=1,n1
        tmp1(i)=v1(i)
        tmpv1(i)=val1(i)
     end do
     do i=1,n2
        tmp2(i)=v2(i)
     end do
  end if

  call auto_interpolation_1d( tmp1, tmp2, tmpv1, tmpv2, undef=undeff,  &
  &                           undefr=undeff )

  if(vc1(1:1)=='m')then
     do i=1,n2
        val2(i)=tmpv2(n2-i+1)
     end do
  else
     do i=1,n2
        val2(i)=tmpv2(i)
     end do
  end if

end subroutine v1_2_v2_1d

!------------------------------------------
!------------------------------------------

subroutine v1_2_v2_3d( v1, v2, val1, val2, undef, sig )
! 任意の鉛直座標 v1 から別の鉛直座標 v2 への val1 という変数の線形変換を行う.
! 本ルーチンは配列に格納される各物理量は幾何高度で見て低い方が要素番号の
! 小さいものに格納されていると仮定している.
  implicit none
  real, dimension(:,:,:), intent(in) :: v1  ! 基準となる鉛直座標 [v2 と同じ単位]
  real, dimension(:,:,:), intent(in) :: v2
                             ! 変換先の鉛直座標 [v1 と同じ単位]
  real, dimension(size(v1,1),size(v1,2),size(v1,3)), intent(in) :: val1
                             ! v1 で定義される任意変数
  real, dimension(size(v1,1),size(v1,2),size(v2,3)), intent(inout) :: val2
                             ! v2 で定義される任意変数
  real, intent(in), optional :: undef  ! 変換できない場合の未定義値
                                       ! デフォルトはゼロ
  character(1), intent(in), optional :: sig  ! 変換前後の座標が
                            ! 漸増, 漸減関数のどちらかを指定する.
                            ! 漸増の場合 -> "p", 漸減の場合 -> "m".
                            ! デフォルトは "p".
                            ! [例] 気圧の場合は "m", 高度の場合は "p".
  integer :: i, j, k, nx, ny, nz1, nz2
  real :: tmp1(size(v1,1),size(v1,2),size(v1,3))
  real :: tmp2(size(v1,1),size(v1,2),size(v2,3))
  real :: undeff
  character(1) :: vc1

  nx=size(v1,1)
  ny=size(v1,2)
  nz1=size(v1,3)
  nz2=size(v2,3)

!-- オプション引数の処理

  if(present(undef))then
     undeff=undef
  else
     undeff=0.0
  end if

  if(present(sig))then
     vc1(1:1)=sig(1:1)
  else
     vc1(1:1)='p'
  end if

!-- 漸減関数について, 漸増となるように置き換える.

  do j=1,ny
     do k=1,nx
        call v1_2_v2_1d( v1(k,j,:), v2(k,j,:), val1(k,j,:), val2(k,j,:),  &
  &                      undeff, vc1(1:1) )
     end do
  end do

end subroutine

!------------------------------------------
!------------------------------------------

end module
