module gms_math_divide
  use datatype
  use mem_manager
  implicit none
  integer :: ix, iy, iz

  interface operator(/)
  module procedure divide_x_x, divide_x_y, divide_x_z, divide_x_xy, divide_x_xz, divide_x_yz, divide_x_xyz
  module procedure divide_y_x, divide_y_y, divide_y_z, divide_y_xy, divide_y_xz, divide_y_yz, divide_y_xyz
  module procedure divide_z_x, divide_z_y, divide_z_z, divide_z_xy, divide_z_xz, divide_z_yz, divide_z_xyz
  module procedure divide_xy_x, divide_xy_y, divide_xy_z, divide_xy_xy, divide_xy_xz, divide_xy_yz, divide_xy_xyz
  module procedure divide_xz_x, divide_xz_y, divide_xz_z, divide_xz_xy, divide_xz_xz, divide_xz_yz, divide_xz_xyz
  module procedure divide_yz_x, divide_yz_y, divide_yz_z, divide_yz_xy, divide_yz_xz, divide_yz_yz, divide_yz_xyz
  module procedure divide_xyz_x, divide_xyz_y, divide_xyz_z, divide_xyz_xy, divide_xyz_xz, divide_xyz_yz, divide_xyz_xyz
  module procedure divide_real_x, divide_real_y, divide_real_z, divide_real_xy, divide_real_xz, divide_real_yz, divide_real_xyz
  module procedure divide_x_real, divide_y_real, divide_z_real, divide_xy_real, divide_xz_real, divide_yz_real, divide_xyz_real
  end interface
contains
  function divide_x_x(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_x)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_x(new_id)
    output%id = new_id 

        do ix = lb_axis1, ub_axis1

        work_x(ix, 1, 1, new_id) & 
                      = work_x(ix, 1, 1, input1%id) & 
                      / work_x(ix, 1, 1, input2%id)
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = -1
  end function divide_x_x

  function divide_x_y(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_x(ix, 1, 1, input1%id) & 
                      / work_y(1, iy, 1, input2%id)
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = -1
  end function divide_x_y

  function divide_x_z(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_x(ix, 1, 1, input1%id) & 
                      / work_z(1, 1, iz, input2%id)
        end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = input2%grid(3)
  end function divide_x_z

  function divide_x_xy(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_x(ix, 1, 1, input1%id) & 
                      / work_xy(ix, iy, 1, input2%id)
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = -1
  end function divide_x_xy

  function divide_x_xz(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_x(ix, 1, 1, input1%id) & 
                      / work_xz(ix, 1, iz, input2%id)
        end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = input2%grid(3)
  end function divide_x_xz

  function divide_x_yz(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_x(ix, 1, 1, input1%id) & 
                      / work_yz(1, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_x_yz

  function divide_x_xyz(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_x(ix, 1, 1, input1%id) & 
                      / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_x_xyz

  function divide_x_real(input1, input2) result(output)
    type(var_x), intent(in) :: input1 
    real(8), intent(in) :: input2 
    type(var_x)             :: output 

    integer :: new_id

    call get_new_id_x(new_id)
    output%id = new_id 

        do ix = lb_axis1, ub_axis1

        work_x(ix, 1, 1, new_id) & 
                      = work_x(ix, 1, 1, input1%id) /input2
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = -1
  end function divide_x_real

  function divide_y_x(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_y(1, iy, 1, input1%id) & 
                      / work_x(ix, 1, 1, input2%id)
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_y_x

  function divide_y_y(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_y)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_y(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2

        work_y(1, iy, 1, new_id) & 
                      = work_y(1, iy, 1, input1%id) & 
                      / work_y(1, iy, 1, input2%id)
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_y_y

  function divide_y_z(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_y(1, iy, 1, input1%id) & 
                      / work_z(1, 1, iz, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_y_z

  function divide_y_xy(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_y(1, iy, 1, input1%id) & 
                      / work_xy(ix, iy, 1, input2%id)
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_y_xy

  function divide_y_xz(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_y(1, iy, 1, input1%id) & 
                      / work_xz(ix, 1, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_y_xz

  function divide_y_yz(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_y(1, iy, 1, input1%id) & 
                      / work_yz(1, iy, iz, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_y_yz

  function divide_y_xyz(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_y(1, iy, 1, input1%id) & 
                      / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_y_xyz

  function divide_y_real(input1, input2) result(output)
    type(var_y), intent(in) :: input1 
    real(8), intent(in) :: input2 
    type(var_y)             :: output 

    integer :: new_id

    call get_new_id_y(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2

        work_y(1, iy, 1, new_id) & 
                      = work_y(1, iy, 1, input1%id) /input2
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_y_real

  function divide_z_x(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) & 
                      / work_x(ix, 1, 1, input2%id)
        end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_z_x

  function divide_z_y(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) & 
                      / work_y(1, iy, 1, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_z_y

  function divide_z_z(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_z)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_z(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3

        work_z(1, 1, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) & 
                      / work_z(1, 1, iz, input2%id)
        end do

    output%grid(1) = -1
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_z_z

  function divide_z_xy(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) & 
                      / work_xy(ix, iy, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_z_xy

  function divide_z_xz(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) & 
                      / work_xz(ix, 1, iz, input2%id)
        end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_z_xz

  function divide_z_yz(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) & 
                      / work_yz(1, iy, iz, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_z_yz

  function divide_z_xyz(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) & 
                      / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_z_xyz

  function divide_z_real(input1, input2) result(output)
    type(var_z), intent(in) :: input1 
    real(8), intent(in) :: input2 
    type(var_z)             :: output 

    integer :: new_id

    call get_new_id_z(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3

        work_z(1, 1, iz, new_id) & 
                      = work_z(1, 1, iz, input1%id) /input2
        end do

    output%grid(1) = -1
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_z_real

  function divide_xy_x(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) & 
                      / work_x(ix, 1, 1, input2%id)
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_xy_x

  function divide_xy_y(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) & 
                      / work_y(1, iy, 1, input2%id)
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_xy_y

  function divide_xy_z(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) & 
                      / work_z(1, 1, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_xy_z

  function divide_xy_xy(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) & 
                      / work_xy(ix, iy, 1, input2%id)
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_xy_xy

  function divide_xy_xz(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) & 
                      / work_xz(ix, 1, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_xy_xz

  function divide_xy_yz(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) & 
                      / work_yz(1, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_xy_yz

  function divide_xy_xyz(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) & 
                      / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_xy_xyz

  function divide_xy_real(input1, input2) result(output)
    type(var_xy), intent(in) :: input1 
    real(8), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = work_xy(ix, iy, 1, input1%id) /input2
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = -1
  end function divide_xy_real

  function divide_xz_x(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) & 
                      / work_x(ix, 1, 1, input2%id)
        end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_xz_x

  function divide_xz_y(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) & 
                      / work_y(1, iy, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xz_y

  function divide_xz_z(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) & 
                      / work_z(1, 1, iz, input2%id)
        end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_xz_z

  function divide_xz_xy(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) & 
                      / work_xy(ix, iy, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xz_xy

  function divide_xz_xz(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) & 
                      / work_xz(ix, 1, iz, input2%id)
        end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_xz_xz

  function divide_xz_yz(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) & 
                      / work_yz(1, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xz_yz

  function divide_xz_xyz(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) & 
                      / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xz_xyz

  function divide_xz_real(input1, input2) result(output)
    type(var_xz), intent(in) :: input1 
    real(8), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = work_xz(ix, 1, iz, input1%id) /input2
        end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = -1
    output%grid(3) = input1%grid(3)
  end function divide_xz_real

  function divide_yz_x(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) & 
                      / work_x(ix, 1, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_x

  function divide_yz_y(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) & 
                      / work_y(1, iy, 1, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_y

  function divide_yz_z(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) & 
                      / work_z(1, 1, iz, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_z

  function divide_yz_xy(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) & 
                      / work_xy(ix, iy, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_xy

  function divide_yz_xz(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) & 
                      / work_xz(ix, 1, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_xz

  function divide_yz_yz(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) & 
                      / work_yz(1, iy, iz, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_yz

  function divide_yz_xyz(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) & 
                      / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_xyz

  function divide_yz_real(input1, input2) result(output)
    type(var_yz), intent(in) :: input1 
    real(8), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = work_yz(1, iy, iz, input1%id) /input2
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_yz_real

  function divide_xyz_x(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) & 
                      / work_x(ix, 1, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_x

  function divide_xyz_y(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) & 
                      / work_y(1, iy, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_y

  function divide_xyz_z(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) & 
                      / work_z(1, 1, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_z

  function divide_xyz_xy(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) & 
                      / work_xy(ix, iy, 1, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_xy

  function divide_xyz_xz(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) & 
                      / work_xz(ix, 1, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_xz

  function divide_xyz_yz(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) & 
                      / work_yz(1, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_yz

  function divide_xyz_xyz(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id
    if ( input1%grid(1) /= input2%grid(1) ) stop "(/)grid violation"
    if ( input1%grid(2) /= input2%grid(2) ) stop "(/)grid violation"
    if ( input1%grid(3) /= input2%grid(3) ) stop "(/)grid violation"

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) & 
                      / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_xyz

  function divide_xyz_real(input1, input2) result(output)
    type(var_xyz), intent(in) :: input1 
    real(8), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = work_xyz(ix, iy, iz, input1%id) /input2
        end do
      end do
    end do

    output%grid(1) = input1%grid(1)
    output%grid(2) = input1%grid(2)
    output%grid(3) = input1%grid(3)
  end function divide_xyz_real

  function divide_real_x(input1, input2) result(output)
    real(8), intent(in) :: input1 
    type(var_x), intent(in) :: input2 
    type(var_x)             :: output 

    integer :: new_id

    call get_new_id_x(new_id)
    output%id = new_id 

        do ix = lb_axis1, ub_axis1

        work_x(ix, 1, 1, new_id) & 
                      = input1 / work_x(ix, 1, 1, input2%id)
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = -1
    output%grid(3) = -1
  end function divide_real_x

  function divide_real_y(input1, input2) result(output)
    real(8), intent(in) :: input1 
    type(var_y), intent(in) :: input2 
    type(var_y)             :: output 

    integer :: new_id

    call get_new_id_y(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2

        work_y(1, iy, 1, new_id) & 
                      = input1 / work_y(1, iy, 1, input2%id)
      end do

    output%grid(1) = -1
    output%grid(2) = input2%grid(2)
    output%grid(3) = -1
  end function divide_real_y

  function divide_real_z(input1, input2) result(output)
    real(8), intent(in) :: input1 
    type(var_z), intent(in) :: input2 
    type(var_z)             :: output 

    integer :: new_id

    call get_new_id_z(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3

        work_z(1, 1, iz, new_id) & 
                      = input1 / work_z(1, 1, iz, input2%id)
        end do

    output%grid(1) = -1
    output%grid(2) = -1
    output%grid(3) = input2%grid(3)
  end function divide_real_z

  function divide_real_xy(input1, input2) result(output)
    real(8), intent(in) :: input1 
    type(var_xy), intent(in) :: input2 
    type(var_xy)             :: output 

    integer :: new_id

    call get_new_id_xy(new_id)
    output%id = new_id 

      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xy(ix, iy, 1, new_id) & 
                      = input1 / work_xy(ix, iy, 1, input2%id)
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = -1
  end function divide_real_xy

  function divide_real_xz(input1, input2) result(output)
    real(8), intent(in) :: input1 
    type(var_xz), intent(in) :: input2 
    type(var_xz)             :: output 

    integer :: new_id

    call get_new_id_xz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
        do ix = lb_axis1, ub_axis1

        work_xz(ix, 1, iz, new_id) & 
                      = input1 / work_xz(ix, 1, iz, input2%id)
        end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = -1
    output%grid(3) = input2%grid(3)
  end function divide_real_xz

  function divide_real_yz(input1, input2) result(output)
    real(8), intent(in) :: input1 
    type(var_yz), intent(in) :: input2 
    type(var_yz)             :: output 

    integer :: new_id

    call get_new_id_yz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2

        work_yz(1, iy, iz, new_id) & 
                      = input1 / work_yz(1, iy, iz, input2%id)
        end do
      end do

    output%grid(1) = -1
    output%grid(2) = input2%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_real_yz

  function divide_real_xyz(input1, input2) result(output)
    real(8), intent(in) :: input1 
    type(var_xyz), intent(in) :: input2 
    type(var_xyz)             :: output 

    integer :: new_id

    call get_new_id_xyz(new_id)
    output%id = new_id 

    do iz = lb_axis3, ub_axis3
      do iy = lb_axis2, ub_axis2
        do ix = lb_axis1, ub_axis1

        work_xyz(ix, iy, iz, new_id) & 
                      = input1 / work_xyz(ix, iy, iz, input2%id)
        end do
      end do
    end do

    output%grid(1) = input2%grid(1)
    output%grid(2) = input2%grid(2)
    output%grid(3) = input2%grid(3)
  end function divide_real_xyz

end module gms_math_divide
