#
#= The EasyVtk module provides methods to draw 3D graph easily using VTK[http://www.vtk.org/]
#
#Authors:: Seiya Nishizawa,
#Version:: 0.1 2006-05-11 seiya
#
module EasyVtk
  require 'vtk'

  # load narray if it exists
  # if loading narray was succeced,
  #  constant With_narray is set to true,
  # else With_narray is set to false.
  begin
    require 'narray'
    With_narray = true
  rescue LoadError
    With_narray = false
  end

  #
  #=== Method for initialization of EasyVtk
  #
  # In the initialization,
  # Vtk::Renderer, Vtk::RenderWindow, and Vtk::RenderWindowInteractior are made,
  # and set to module variable.
  #
  #* Arguments
  #  * none
  #* Return
  #  * nil
  #
  def init
    @ren = Vtk::Renderer.new
    @renWin = Vtk::RenderWindow.new
    @renWin.AddRenderer( @ren )
    @scale = nil
    @loolupTable = nil
    return nil
  end
  module_function :init

  #
  #=== Method for setting window size
  #
  # Sets the size of window.
  #
  #* Arguments
  #  * width (Numeric)
  #    * width of window
  #  * height (Numeric)
  #    * height of window
  #* Return
  #  * nil
  #
  def set_size( width, height )
    @renWin.SetSize( width, height )
    return nil
  end
  module_function :set_size

  #
  #=== Method for setting background color of window
  #
  # Sets the background color of window.
  #
  #* Arguments
  #  * red (Numeric)
  #    * red compornent of RGB form 0 to 1
  #  * green (Numeric)
  #    * green compornent of RGB form 0 to 1
  #  * blue (Numeric)
  #    * blue compornent of RGB form 0 to 1
  #* Return
  #  * nil
  #
  def set_background( red, green, blue )
    @ren.SetBackground( red, green, blue )
    return nil
  end
  module_function :set_background

  #
  #=== Method for setting position of camera
  #
  # Sets the position of the camera in world coordinate
  #
  #* Arguments
  #  * x (Numeric)
  #    * x coordinates of the position
  #  * y (Numeric)
  #    * y coordinates of the position
  #  * z (Numeric)
  #    * z coordinates of the position
  #* Return
  #  * nil
  #
  def set_camera_position( x, y, z )
    @ren.GetActiveCamera.SetPosition( x, y, z )
    return nil
  end
  module_function :set_camera_position

  #
  #=== Method for setting array of axes
  #
  # Vtk::RectilineraGrid is made.
  # Sets the three arrays of x, y and z axes.
  # Length of the all arrays must be the same.
  #
  #* Arguments
  #  * x (Vtk::DataArray or NArray)
  #    * array of values of x axis
  #  * y (Vtk::DataArray or NArray)
  #    * array of values of y axis
  #  * z (Vtk::DataArray or NArray)
  #    * array of values of z axis
  #* Optional arguments
  #  * scale (Array)
  #    * array of three compornents for scale to draw
  #* Return
  #  * nil
  def set_axes( x, y, z, scale=nil )
    x = get_vtkArray( x )
    y = get_vtkArray( y )
    z = get_vtkArray( z )
    if scale
      unless Array === scale
        raise "scale must be Array"
      end
      if scale.length != 3
        raise "length of scale must be 3"
      end
      @scale = scale
      for i in 0...x.GetNumberOfTuples
        x.SetValue( i, x.GetValue(i)*scale[0] )
      end
      for i in 0...y.GetNumberOfTuples
        y.SetValue( i, y.GetValue(i)*scale[1] )
      end
      for i in 0...z.GetNumberOfTuples
        z.SetValue( i, z.GetValue(i)*scale[2] )
      end
    end
    @grid = Vtk::RectilinearGrid.new
    @grid.SetDimensions( x.GetNumberOfTuples, y.GetNumberOfTuples, z.GetNumberOfTuples )
    @grid.SetXCoordinates( x )
    @grid.SetYCoordinates( y )
    @grid.SetZCoordinates( z )
    return nil
  end
  module_function :set_axes

  #
  #=== Method for setting array of grid points
  #
  # Vtk::StructuredGrid is made.
  # Sets the three arrays of x, y and z of grid points.
  # Length of the all arrays must be the same as that of data.
  #
  #* Arguments
  #  * x (Vtk::DataArray or NArray)
  #    * array of values of x coordinate of grid points
  #  * y (Vtk::DataArray or NArray)
  #    * array of values of y coordinate of grid points
  #  * z (Vtk::DataArray or NArray)
  #    * array of values of z coordinate of grid points
  #  * shape (Array)
  #    * array of three components of grid
  #* Optional arguments
  #  * scale (Array)
  #    * array of three components for scale to draw
  #* Return
  #  * nil
  def set_grid( x, y, z, shape, scale=nil )
    x = get_vtkArray( x )
    y = get_vtkArray( y )
    z = get_vtkArray( z )
    unless shape.length==3
      raise "length of shape must be three"
    end
    if scale
      unless Array === scale
        raise "scale must be Array"
      end
      if scale.length != 3
        raise "length of scale must be three"
      end
      @scale = scale
      for i in 0...x.GetNumberOfTuples
        x.SetValue( i, x.GetValue(i)*scale[0] )
      end
      for i in 0...y.GetNumberOfTuples
        y.SetValue( i, y.GetValue(i)*scale[1] )
      end
      for i in 0...z.GetNumberOfTuples
        z.SetValue( i, z.GetValue(i)*scale[2] )
      end
    end
    npoints = shape[0]*shape[1]*shape[2]
    if x.GetNumberOfTuples != npoints
      raise "length of x does not correspond to shape"
    end
    if y.GetNumberOfTuples != npoints
      raise "length of y does not correspond to shape"
    end
    if z.GetNumberOfTuples != npoints
      raise "length of z does not correspond to shape"
    end
    points = Vtk::Points.new
    points.SetNumberOfPoints( npoints )
    npoints.times{|i|
      points.SetPoint( i, x.GetValue(i), y.GetValue(i), z.GetValue(i) )
    }
    @grid = Vtk::StructuredGrid.new
    @grid.SetDimensions( *shape )
    @grid.SetPoints( points )
    return nil
  end
  module_function :set_grid

  #
  #=== Method for setting array of scalar data
  #
  # Sets the array of scalar data.
  #
  #* Arguments
  #  * data (Vtk::DataArray or NArray)
  #    * array of values of data
  #      Length of the array must be (length of x axis)*(length of y axis)*(length of z axis) set by set_axes or set_grid.
  #      If the class is NArray, its rank must be 1.
  #* Optional arguments
  #  * options (Hash)
  #    * options['type'] (String)
  #      * 'scalar' : data type is scalar
  #      * 'vector' : data type is vector
  #* Return
  #  * nil
  #
  def set_data( data, options={} )
    type = false
    options.each{ |key, val|
      case key
      when 'type'
        type = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    unless @grid
      raise "you must call #set_axes or #set_grid before call #set_data"
    end
    data = get_vtkArray( data )
    unless type
      case data.GetNumberOfComponents
      when 1
        type = 'scalar'
      when 3
        type = 'vector'
      else
        raise "components of data must be 1 (scalar) or 3 (vector)"
      end
    end
    case type
    when 'scalar'
      @grid.GetPointData.SetScalars( data )
    when 'vector'
      @grid.GetPointData.SetVectors( data )
    end
    return nil
  end
  module_function :set_data

  #
  #=== Method for drawing surface
  #
  # Draw isosurface
  #
  #* Arguments
  #  * value (Numeric)
  #    * value of isosurface
  #* Optional arguments
  #  * options (Hash)
  #    * options['opacity'] (Numeric)
  #      * opacity of the surface from 0 (no transparency) to 1 (full transparency)
  #    * options['axes'] (true|false or Hash)
  #      * If this is true or Hash, draws axes.
  #        * true
  #          * draw axes with default properties
  #        * options['axes']['format'] (String)
  #          * format of label (ex. '%4.2f')
  #        * options['axes']['factor'] (Numric)
  #          * factor of label size
  #        * options['axes']['color'] (Array[Numeric,Numeric,Numeric])
  #          * RGB of axes ([red, green, blue])
  #* Return
  #  * nil
  #
  def surface( value, options={} )
    opacity = false
    color = false
    axes = false
    options.each{ |key, val|
      case key
      when 'opacity'
	opacity = val
      when 'color'
	color = val
        if !(Array === color) || color.length != 3
          raise "color must be Array whose length is three"
        end
      when 'axes'
        axes = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    surface = Vtk::ContourFilter.new
    surface.SetInput( @grid )
    surface.SetValue( 0, value )
    normals = Vtk::PolyDataNormals.new
    normals.SetInput( surface.GetOutput )
    mapper = Vtk::PolyDataMapper.new
    mapper.SetInput( normals.GetOutput )
    mapper.SetScalarModeToUsePointFieldData
    actor = Vtk::Actor.new
    actor.SetMapper( mapper )
    actor.GetProperty.SetOpacity( opacity ) if opacity
    actor.GetProperty.SetDiffuseColor( *color ) if color
    @ren.AddActor( actor )
    draw_axes( surface, axes ) if axes
    return nil
  end
  module_function :surface

  #
  #=== Method for drawing plane
  #
  # Draws tone of a plane
  #
  #* Arguments
  #  * origin (Array)
  #    * point of origin of plane to be drawn
  #      length of origin must be 3.
  #  * normal (Array)
  #    * vector of normal of plane to be drawn
  #      length of origin must be 3.
  #* Optional arguments
  #  * options (Hash)
  #    * options['opacity'] (Numeric)
  #      * opacity of the surface from 0 (no transparency) to 1 (full transparency)
  #    * options['axes'] (true|false or Hash)
  #      * If this is true or Hash, draws axes.
  #        * true
  #          * draw axes with default properties
  #        * options['axes']['format'] (String)
  #          * format of label (ex. '%4.2f')
  #        * options['axes']['factor'] (Numric)
  #          * factor of label size
  #        * options['axes']['color'] (Array[Numeric,Numeric,Numeric])
  #          * RGB of axes ([red, green, blue])
  #* Return
  #  * nil
  #
  def plane( origin, normal, options={} )
    opacity = false
    axes = false
    options.each{ |key, val|
      case key
      when 'opacity'
	opacity = val
      when 'axes'
        axes = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    plane = Vtk::Plane.new
    plane.SetOrigin( origin )
    plane.SetNormal( normal )
    cutter = Vtk::Cutter.new
    cutter.SetInput( @grid )
    cutter.SetCutFunction( plane )
    mapper = Vtk::PolyDataMapper.new
    mapper.SetInput( cutter.GetOutput )
    if @lookupTable
      mapper.SetLookupTable( @lookupTable )
    else
      @lookupTable = mapper.GetLookupTable
    end
    actor = Vtk::Actor.new
    actor.SetMapper( mapper )
    actor.GetProperty.SetOpacity( options['opacity'] ) if opacity
    @ren.AddActor( actor )
    draw_axes( cutter, axes ) if axes
    return nil
  end
  module_function :plane

  #
  #=== Method for drawing contour surfaces
  #
  # Draws contour surfaces
  #
  #* Arguments
  #  * num (Fixnum)
  #    * number of contour to be drawn
  #  * min (Numric)
  #    * minimum value of range for contour
  #  * max (Numric)
  #    * maxmum value of range for contour
  #* Optional arguments
  #  * options (Hash)
  #    * options['opacity'] (Numeric)
  #      * opacity of the surface from 0 (no transparency) to 1 (full transparency)
  #    * options['axes'] (true|false or Hash)
  #      * If this is true or Hash, draws axes.
  #        * true
  #          * draw axes with default properties
  #        * options['axes']['format'] (String)
  #          * format of label (ex. '%4.2f')
  #        * options['axes']['factor'] (Numric)
  #          * factor of label size
  #        * options['axes']['color'] (Array[Numeric,Numeric,Numeric])
  #          * RGB of axes ([red, green, blue])
  #* Return
  #  * nil
  #
  def contour( num, min, max, options={} )
    opacity = false
    axes = false
    options.each{ |key, val|
      case key
      when 'opacity'
        opacity = val
      when 'axes'
        axes = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    contour = Vtk::ContourFilter.new
    contour.SetInput( @grid )
    contour.GenerateValues( num, min, max )
    normals = Vtk::PolyDataNormals.new
    normals.SetInput( contour.GetOutput )
    mapper = Vtk::PolyDataMapper.new
    mapper.SetInput( normals.GetOutput )
    mapper.SetScalarRange( min, max )
    if @lookupTable
      mapper.SetLookupTable( @lookupTable )
    else
      @lookupTable = mapper.GetLookupTable
    end
    actor = Vtk::Actor.new
    actor.SetMapper( mapper )
    actor.GetProperty.SetOpacity( options['opacity'] ) if opacity
    @ren.AddActor( actor )
    draw_axes( contour, axes ) if axes
    return nil
  end
  module_function :contour

  #
  #=== Method for drawing streamline
  #
  # Draws streamline
  #
  #* Arguments
  #  * point (Array)
  #    * start point
  #      length of point must be 3.
  #  * time (Numeric)
  #    * time for integration
  #  * ds (Numeric)
  #    * length of line segments to be drawn
  #* Optional arguments
  #  * options (Hash)
  #    * options['accuracy'] (Numric)
  #      * accuracy of integration (from 0 to 1)
  #    * options[method'] (String)
  #      * integration method
  #        * 'rk2' : RungeKutta2
  #        * 'rk4' : RungeKutta4
  #        * 'rk45': RungeKutta45
  #    * options['direction'] (String)
  #      * direction of integration
  #        * 'forward' : integrate forward
  #        * 'backward': integrate backward
  #        * 'both'    : integrate forward and backward
  #    * options['color'] (Array)
  #    * options['type'] (String)
  #      * type of line
  #        * 'line'
  #        * 'ribbon'
  #        * 'tube'
  #    * options['width'] (Numeric)
  #      * width of ribbon or tube
  #    * options['normal'] (Array)
  #      * vector of normal of ribbon or tube
  #        length of origin must be 3.
  #    * options['vary'] (String)
  #      * 'off'   : disable variation of ribbon or tube width
  #      * 'scalar': variation of ribbon or tube width with scalar value
  #      * 'vector': variation of tube width with vector value (only for tube)
  #    * options['vary_factor'] (Numeric)
  #      * the maximum ribbon or tube width in terms of a multiple of the minimum width
  #    * options['tube_sides'] (Integer)
  #      * number of sides of tube
  #* Return
  #  * nil
  #
  def streamline( point, time, ds, options={} )
    accuracy = false
    method = false
    direction = false
    color = false
    type = false
    width = false
    normal = false
    vary = false
    vary_factor = false
    tube_sides = false
    options.each{ |key, val|
      case key
      when 'accuracy'
        accuracy = val
      when 'method'
        method = val
      when 'direction'
        direction = val
      when 'color'
	color = val
        if !(Array === color) || color.length != 3
          raise "color must be Array whose length is three"
        end
      when 'type'
        type = val
      when 'width'
        width = val
      when 'normal'
        normal = val
      when 'vary'
        vary = val
      when 'vary_factor'
        vary_factor = val
      when 'tube_sides'
        tube_sides = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    stream = Vtk::StreamLine.new
    stream.SetInput( @grid )
    stream.SetStartPosition( *point )
    stream.SetMaximumPropagationTime( time )
    stream.SetStepLength( ds )
    stream.SetIntegrationStepLength( accuracy ) if accuracy
    stream.VorticityOn
    unless color
      stream.SpeedScalarsOn
    end
    if direction
      case direction
      when 'forward'
        stream.SetIntegrationDirectionToForward
      when 'backward'
        stream.SetIntegrationDirectionToBackward
      when 'both'
        stream.SetIntegrationDirectionToIntegrateBothDirections
      else
        raise "options['direction'] must be 'forward', 'backward', or 'both'"
      end
    end
    if method
      case method
      when 'rk2'
        stream.SetIntegrator( Vtk::RungeKutta2.new )
      when 'rk4'
        stream.SetIntegrator( Vtk::RungeKutta4.new )
      when 'rk45'
        stream.SetIntegrator( Vtk::RungeKutta45.new )
      else
        raise "options['direction'] must be 'rk2', 'rk4', or 'rk45'"
      end
    end
    if type
      case type
      when 'line'
        # do nothing
      when 'ribbon'
        filter = Vtk::RibbonFilter.new
        filter.SetInput( stream.GetOutput )
        filter.SetWidth( width ) if width
        filter.SetWidthFactor( 5 )
        if normal
          filter.SetDefaultNormal( *normal )
          filter.UseDefaultNormalOn
        end
        if vary
          case vary
          when 'off'
            filter.VaryWidthOff
          when 'scalar'
            filter.VaryWidthOn
          else
            raise "options['vary'] must be 'off' or 'scalar'"
          end
          filter.SetWidthFactor( vary_factor ) if vary_factor
        end
        stream = filter
      when 'tube'
        filter = Vtk::TubeFilter.new
        filter.SetInput( stream.GetOutput )
        filter.SetRadius( width/2 ) if width
        filter.SetNumberOfSides( tube_sides ) if tube_sides
        if normal
          filter.SetDefaultNormal( *normal )
          filter.UseDefaultNormalOn
        end
        if vary
          case vary
          when 'off'
            filter.SetVaryRadiusToVaryRadiusOff
          when 'scalar'
            filter.SetVaryRadiusToVaryRadiusByScalar
          when 'vector'
            filter.SetVaryRadiusToVaryRadiusByVector
          else
            raise "options['vary'] must be 'off', 'scalar', or 'vector'"
          end
          filter.SetRadiusFactor( vary_factor ) if vary_factor
        end
        stream = filter
      else
        raise "options['type'] must be 'line', 'ribbon', or 'tube'"
      end
    end
    mapper = Vtk::PolyDataMapper.new
    mapper.SetInput( stream.GetOutput )
    if color
      mapper.ScalarVisibilityOff
    end
    actor = Vtk::Actor.new
    actor.SetMapper( mapper )
    if color
      actor.GetProperty.SetColor( *color )
    end
    @ren.AddActor( actor )
    return nil
  end
  module_function :streamline

  #
  #=== Method for volume rendering
  #
  # Draws Volume Rendering
  #
  #* Optional arguments
  #  * options (Hash)
  #    * options['opacity'] (nil or Hash)
  #      * {value0 => opacity0, value1 => opacity1, ...}
  #        * opacity must be between 0 to 1
  #    * options['color'] (nil or Hash)
  #      * {value0 => [R0,G0,B0], value1 => [R1,G1,B1], ...}
  #    * options['technique'] (String)
  #      * "ray casting"
  #      * "2d texture mapping": default
  #    * options['axes'] (true|false or Hash)
  #      * If this is true or Hash, draws axes.
  #        * true
  #          * draw axes with default properties
  #        * options['axes']['format'] (String)
  #          * format of label (ex. '%4.2f')
  #        * options['axes']['factor'] (Numric)
  #          * factor of label size
  #        * options['axes']['color'] (Array[Numeric,Numeric,Numeric])
  #          * RGB of axes ([red, green, blue])
  #* Return
  #  * nil
  #
  def volume_rendering( options={} )
    opacity = {
      20  => 0.0,
      255 => 0.2
    }
    color = {
      0   => [0.0, 0.0, 0.0],
      64  => [1.0, 0.0, 0.0],
      128 => [0.0, 0.0, 1.0],
      192 => [0.0, 1.0, 0.0],
      255 => [0.0, 0.2, 0.0]
    }
    technique = "2d texture mapping"
    axes = false
    options.each{ |key, val|
      case key
      when 'opacity'
        if Hash === val
          opacity = val
        else
          raise "options['opacity'] must be Hash"
        end
      when 'color'
        if Hash === val
          color = val
        else
          raise "options['color'] must be Hash"
        end
      when 'axes'
        axes = val
      else
        raise "option (#{key}) is invalid"
      end
    }

    pd = @grid.GetPointData
    ary = pd.GetScalars
    unless [3,5].include?(ary.GetDataType)
      ary_new = Vtk::UnsignedShortArray.new
      len = ary.GetNumberOfTuples
      ary_new.SetNumberOfValues( len )
      max = ary_new.GetDataTypeMax
      for i in 0...len
        val = ary.GetValue(i)
        if val < 0
          val = 0
        elsif val > max
          val = max
        end
        ary_new.SetValue( i, val )
      end
      pd.SetScalars( ary_new )
    end
    image_data = Vtk::ImageData.new
    d = @grid.GetDimensions
    image_data.SetDimensions( d )
#    image_data.SetWholeExtent( @grid.GetWholeExtent )
#    image_data.SetExtent( @grid.GetExtent )
#    image_data.SetUpdateExtent( @grid.GetUpdateExtent )
    b = @grid.GetBounds
    image_data.SetSpacing( (b[1]-b[0])/d[0], (b[3]-b[2])/d[1], (b[5]-b[4])/d[2] )
    image_data.SetOrigin( b[0], b[2], b[4] )
    probe = Vtk::ProbeFilter.new
    probe.SetInput( image_data )
    probe.SetSource( @grid )

    opacityTransferFunction = Vtk::PiecewiseFunction.new
    opacity.each{|k,v|
      opacityTransferFunction.AddPoint(k, v)
    }
    colorTransferFunction = Vtk::ColorTransferFunction.new
    color.each{|k,v|
      colorTransferFunction.AddRGBPoint(k, *v)
    }
    property = Vtk::VolumeProperty.new
    property.SetColor(colorTransferFunction)
    property.SetScalarOpacity(opacityTransferFunction)
#    property.ShadeOn
#    property.SetInterpolationTypeToLinear
    case technique
    when "ray casting"
      function = Vtk::VolumeRayCastCompositeFunction.new
      mapper = Vtk::VolumeRayCastMapper.new
      mapper.SetVolumeRayCastFunction( function )
    when "2d texture mapping"
      mapper = Vtk::VolumeTextureMapper2D.new
    else
      raise "the technique is not supported: '#{technique}'"
    end
#    mapper.SetInput( image_data.GetOutput )
    mapper.SetInput( probe.GetOutput )
    volume = Vtk::Volume.new
    volume.SetMapper( mapper )
    volume.SetProperty( property )
    @ren.AddVolume( volume )
    draw_axes( probe, axes ) if axes
  end
  module_function :volume_rendering

  #
  #=== Method for drawing colorbar
  #
  # Draws colorbar
  #
  #* Arguments
  #  * none
  #* Optional arguments
  #  * options (Hash)
  #    * options['orientation'] (String)
  #      * 'horizontal': Sets orientation of colorbar to horizontal
  #      * 'vertical': Sets orientation of colorbar to vertical
  #    * options['height'] (Numeric)
  #      * height of colorbar
  #    * options['width'] (Numeric)
  #      * width of colorbar
  #    * options['position'] (Array)
  #      * position of bottom-left corner of colorbar (between 0 and 1)
  #        length must be 2.
  #* Return
  #  * nil
  #
  def colorbar( options={} )
    unless @lookupTable
      raise "color lookup table has not been created"
    end
    scalarBar = Vtk::ScalarBarActor.new
    scalarBar.SetLookupTable( @lookupTable )
    options.each{ |key, val|
      case key
      when 'orientation'
        case val
        when 'horizontal'
          scalarBar.SetOrientationToHorizontal
        when 'vertical'
          scalarBar.SetOrientationToVertical
        else
          raise "value of option['orientation'] (=#{val}) is invalid"
        end
      when 'height'
        scalarBar.SetHeight( val )
      when 'width'
        scalarBar.SetWidth( val )
      when 'position'
        if ! Array === val || val.length != 2
          raise "value of option['position'] must be Array whose length is 3"
        end
        scalarBar.GetPositionCoordinate.SetValue *val
      else
        raise "option (#{key}) is invalid"
      end
    }
    @ren.AddActor( scalarBar )
    return nil
  end
  module_function :colorbar

  #
  #=== Method for drawing text
  #
  # Draws text
  #
  #* Arguments
  #  * str (String)
  #    * text to be drawn
  #  * x (Numeric)
  #    * normalized x-position of text
  #  * y (Numeric)
  #    * normalized y-position of text
  #  * z (Numeric)
  #    * normalized z-position of text
  #  * size (Numeric)
  #    * size of text
  #* Return
  #  * nil
  #
  def text( str, x, y, z, size )
    text = Vtk::VectorText.new
    text.SetText( str )
    mapper = Vtk::PolyDataMapper.new
    mapper.SetInput( text.GetOutput )
    actor = Vtk::Follower.new
    actor.SetMapper( mapper )
    actor.SetScale( size, size, size )
    actor.AddPosition( x, y, x )
    @ren.AddActor( actor )
    @ren.ResetCameraClippingRange
    actor.SetCamera( @ren.GetActiveCamera )
    return nil
  end
  module_function :text

  #
  #=== Method for drawing fixed text
  #
  # Draws fixed text
  #
  #* Arguments
  #  * str (String)
  #    * text to be drawn
  #  * x (Numeric)
  #    * x-position of text
  #  * y (Numeric)
  #    * y-position of text
  #* Optional Arguments
  #  * options
  #    * options['vertical_justification'] (String)
  #      * 'bottom': Sets vertical justification to bottom
  #      * 'top': Sets vertical justification to top
  #      * 'centered': Sets vertical justification to centered
  #    * options['horizontal_justification'] (String)
  #      * 'bottom': Sets horizontal justification to bottom
  #      * 'top': Sets horizontal justification to top
  #      * 'centered': Sets horizontal justification to centered
  #    * options['color'] (Array)
  #      * RGB of axes
  #        length must be 3
  #* Return
  #  * nil
  #
  def fixed_text( str, x, y, options={} )
    vertical_justification = false
    horizontal_justification = false
    color = false
    options.each{ |key, val|
      case key
      when 'vertical_justification'
        vertical_justification = val
      when 'horizontal_justification'
        horizontal_justification = val
      when 'color'
        unless Array === val
          raise "value of option['color'] must be Array"
        end
        color = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    text = Vtk::TextMapper.new
    text.SetInput( str )
    prop = text.GetTextProperty
    if vertical_justification
      case vertical_justification
      when 'bottom'
        prop.SetVerticalJustificationToBottom
      when 'top'
        prop.SetVerticalJustificationToTop
      when 'center'
        prop.SetVerticalJustificationToCentered
      else
        raise "value of option['vertical_justification'] (=#{val}) is invalid"
      end
    end
    if horizontal_justification
      case horizontal_justification
      when 'left'
        prop.SetJustificationToLeft
      when 'right'
        prop.SetJustificationToRight
      when 'center'
        prop.SetJustificationToCentered
      else
        raise "value of option['horizontal_justification'] (=#{val}) is invalid"
      end
    end
    prop.SetColor( color ) if color
    actor = Vtk::Actor2D.new
    actor.SetMapper( text )
    actor.GetPositionCoordinate.SetCoordinateSystemToNormalizedDisplay
    actor.GetPositionCoordinate.SetValue( x, y )
    @ren.AddActor( actor )
    return nil
  end
  module_function :fixed_text

  #
  #=== Method for drawing sphere
  #
  # Draw sphere surface
  #
  #* Arguments
  # * radius (Numeric)
  #   * radius of sphere to be drawn
  #* Optional arguments
  # * resolution (Numeric)
  #   * resolution of sphere to be drawn
  #* Return
  # * nil
  #
  def sphere( radius, options={} )
    resolution = false
    opacity = false
    color = false
    wireframe = false
    options.each{ |key, val|
      case key
      when 'resolution'
        resolution = val
      when 'opacity'
        opacity = val
      when 'color'
	color = val
        if !(Array === color) || color.length != 3
          raise "color must be Array whose length is three"
        end
      when 'wireframe'
        wireframe = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    sphere = Vtk::SphereSource.new
    sphere.SetRadius( radius )
    if resolution
      sphere.SetPhiResolution( resolution )
      sphere.SetThetaResolution( resolution )
    end
    mapper = Vtk::PolyDataMapper.new
    mapper.SetInput( sphere.GetOutput )
    actor = Vtk::Actor.new
    actor.SetMapper( mapper )
    actor.GetProperty.SetDiffuseColor( *color ) if color
    actor.GetProperty.SetOpacity( options['opacity'] ) if opacity
    actor.GetProperty.SetRepresentationToWireframe if wireframe
    @ren.AddActor( actor )
    return nil
  end
  module_function :sphere


  #
  #=== Method for starting drawing and GUI
  #
  # Starts drawing and GUI
  #
  #* Arguments
  #  * none
  #* Block (optional)
  #  * this execute when 'return' key pressed
  #* Return
  #  * nil
  #
  def gui_start
    iren = Vtk::RenderWindowInteractor.new
    iren.SetRenderWindow( @renWin )
    iren.Initialize
    if block_given?
      proc = Proc.new{|widget, event|
        if widget.GetKeySym == 'Return'
          yield
          @renWin.Render
        end
      }
      iren.AddObserver('KeyPressEvent', proc )
    end
    iren.Start
    return nil
  end
  module_function :gui_start

  #
  #=== Method for looping of drawing
  #
  # Starts loop for drawing
  #
  #* Arguments
  #  * count (Integer)
  #* Optional arguments
  #  * interval (Numeric)
  #    * seconds for sleep after execution given block
  #  * clear (true|false)
  #    * set true if you want to clear actors
  #* Block
  #  * block arugument is current count of loop
  #* Return
  #  * nil
  #
  def loop( count, options={} )
    interval = 0.03
    clear = false
    filename = false
    options.each{ |key, val|
      case key
      when 'interval'
        interval = val
      when 'clear'
        clear = val
      when 'filename'
        filename = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    if filename
      w2i = Vtk::WindowToImageFilter.new
      w2i.SetInput( @renWin )
      w2i.Update
      writer = Vtk::MPEG2Writer.new
      writer.SetInput( w2i.GetOutput )
      writer.SetFileName( filename )
      writer.Start
    end
    count.times{|i|
      EasyVtk::clear if clear
      yield( i )
      @renWin.Render
      writer.Write if filename
      sleep( interval )
    }
    writer.End if filename
    return nil
  end
  module_function :loop


  #
  #=== Method for creating color lookup table automatically
  #
  # creats color lookup table automatically
  #
  #* Arguments
  #  * min (Numeric)
  #    * minimum value of data for color map
  #  * max (Numeric)
  #    * maximum value of data for color map
  #* Optional arguments
  #  * hue_range (Array)
  #    * range in hue (between 0 and 1)
  #      length must be 2
  #  * saturation_range (Array)
  #    * range in saturation (between 0 and 1)
  #      length must be 2
  #  * value_range (Array)
  #    * range in value (between 0 and 1)
  #      length must be 2
  #  * alpha_range (Array)
  #    * range in alpha (between 0 and 1)
  #      length must be 2
  #* Return
  #  * nil
  #
  def create_lookup_table( min, max, options={} )
    hue_range = false
    saturation_range = false
    value_range = false
    alpha_range = false
    options.each{ |key, val|
      case key
      when 'hue_range'
        if ! Array === val || val.length!=2
          raise "option['hue_range'] must be Array whose length is 2"
        end
        hue_range = val
      when 'saturation_range'
        if ! Array === val || val.length!=2
          raise "option['saturation_range'] must be Array whose length is 2"
        end
        saturation_range = val
      when 'value_range'
        if ! Array === val || val.length!=2
          raise "option['value_range'] must be Array whose length is 2"
        end
        value_range = val
      when 'alpha_range'
        if ! Array === val || val.length!=2
          raise "option['alpha_range'] must be Array whose length is 2"
        end
        alpha_range = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    @lookupTable = Vtk::LookupTable.new
    @lookupTable.SetTableRange( min, max )
    @lookupTable.SetHueRange( *hue_range ) if hue_range
    @lookupTable.SetSaturationRange( *saturation_range ) if saturation_range
    @lookupTable.SetValueRange( *value_range ) if value_range
    @lookupTable.SetAlphaRange( *alpha_range ) if alpha_range
    @lookupTable.Build
    return nil
  end
  module_function :create_lookup_table

  #
  #=== Method for clearing actors
  #
  # Clears actors
  #
  #* Arguments
  #  * none
  #* Return
  #  * nil
  #
  def clear
    actors = @ren.GetActors
    actors.GetNumberOfItems.times{|i|
      @ren.RemoveActor( actors.GetLastActor )
    }
    return nil
  end
  module_function :clear

  #
  #=== Method for clearing color lookup table
  #
  # Clears color lookup table
  #
  #* Arguments
  #  * none
  #* Return
  #  * nil
  #
  def clear_color_lookuptable
    @lookupTable = nil
    return nil
  end
  module_function :clear_color_lookuptable

  #
  #=== Method for saveing image file
  #
  # Saves window image to file
  #
  #* Arguments
  #  * filename (String)
  #    * filename to be saved
  #* Optional arguments
  #  * type (String)
  #    * image type to be saved
  #* Return
  #  * nil
  #
  def save_file( filename, options={} )
    type = 'png'
    options.each{ |key, val|
      case key
      when 'type'
        type = val
      else
        raise "option (#{key}) is invalid"
      end
    }
    w2i = Vtk::WindowToImageFilter.new
    w2i.SetInput( @renWin )
    case type
    when 'png'
      writer = Vtk::PNGWriter.new
    when 'jpeg'
      writer = Vtk::JPEGWriter.new
    when 'bmp'
      writer = Vtk::BMPWriter.new
    when 'tiff'
      writer = Vtk::TIFFWriter.new
    when 'ps'
      writer = Vtk::PostScriptWriter.new
    when 'pnm'
      writer = Vtk::PNMWriter.new
    end
    writer.SetInput( w2i.GetOutput )
    writer.SetFileName( filename )
    @renWin.Render
    w2i.Update
    writer.Write
    return nil
  end
  module_function :save_file


# followings are private methods
  private
  def get_vtkArray( ary )
    if Vtk::DataArray === ary
      return ary
    elsif With_narray && NArray === ary
      return ary.to_va
    else
      raise "invalid type"
    end
  end
  module_function :get_vtkArray

  def draw_axes( object, options )
    options = {} unless Hash === options
    format = false
    factor = false
    color = false
    options.each{ |key, val|
      case key
      when 'format'
        format = val
      when 'factor'
        factor = val
      when 'color'
        color = val
      else
        raise "key of option['axes'] (=#{key}) is invalid"
      end
    }
    axes = Vtk::CubeAxesActor2D.new
    axes.SetInput( object.GetOutput )
    if @scale
      bounds = axes.GetBounds
      bounds[0] /= @scale[0]
      bounds[1] /= @scale[0]
      bounds[2] /= @scale[1]
      bounds[3] /= @scale[1]
      bounds[4] /= @scale[2]
      bounds[5] /= @scale[2]
      axes.SetRanges( bounds )
      axes.UseRangesOn
    end
    @ren.ResetCameraClippingRange
    axes.SetCamera( @ren.GetActiveCamera )
    axes.SetLabelFormat( format ) if format
    axes.SetFontFactor( factor ) if factor
    axes.GetProperty.SetColor( color ) if color
    @ren.AddActor( axes )
  end
  module_function :draw_axes


end

if defined?(NArray)
  class NArray
    def to_va
      case typecode
      when NArray::BYTE
        vary = Vtk::UnsignedCharArray.new
      when NArray::SINT
        vary = Vtk::ShortArray.new
      when NArray::INT
        vary = Vtk::IntArray.new
      when NArray::SFLOAT
        vary = Vtk::FloatArray.new
      when NArray::FLOAT
        vary = Vtk::DoubleArray.new
      else
        raise "NArray type is invalid"
      end
      case rank
      when 1
        vary.SetNumberOfValues( length )
        for i in 0...length
          vary.SetValue( i, self[i] )
        end
      when 2
        numTuple, numComp = shape
        vary.SetNumberOfComponents( numComp )
        vary.SetNumberOfTuples( numTuple )
        for i in 0...numTuple
          for j in 0...numComp
            vary.SetComponent( i, j, self[i,j] )
          end
        end
      else
        raise "NArray rank must be 1 or 2: rank=#{rank}"
      end
      return vary
    end
  end

end
