
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libnc-dods, A C++ implementation of netCDF 3 which
// supports reading from local files and OPeNDAP servers.

// Copyright (c) 2004 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

#ifndef client_params_h
#define client_params_h

#include <string>
#include <map>

#include <Error.h>
#include <InternalErr.h>

using namespace std;
using namespace libdap;

class ClientParams {
private:
    string d_url;
    bool d_preload;
    bool d_string_as_char_array;
    bool d_scoped_grid_dims;
    int d_default_limit;
    map<string,int> d_limits;
    
    // Make these private to suppres their use outside the class
    ClientParams() {
        throw InternalErr(__FILE__, __LINE__, "Unimplemented method.");
    }

    void clone(const ClientParams &rhs);
    
    friend class NCConnect;

public:

    ClientParams(const string &url) throw(Error);
    
    ClientParams(const ClientParams &rhs) {
        clone(rhs);
    }

    ClientParams &operator=(const ClientParams &rhs) {
        if (this == &rhs)
            return *this;
            
        clone(rhs);
        return *this;
    }

    virtual ~ClientParams() {}
  
    /** @brief Get the value of the \c preload parameter. 
        The \c preload parameter is used to tell the clinet library to load
        certain variables right after opening the remote dataset so that 
        different sizes can be computed. 
        
        @note Not implemented. */
    bool get_preload() const { return d_preload; }

    /** @brief Get the value of the \c string_as_char_array parameter.
        The \c string_as_char_array parameter tells the library to treat DAP
        String variables as character arrays. 
        
        @note The default value is false. Set this to true when reading data 
	from HDF4 files or other data sources which use the DAP String data
	type for character arrays.
        
        @return The value of the parameter. */
    bool get_string_as_char_array() const { return d_string_as_char_array; }

    /** @brief Get the value of the \c scoped_grid_dims parameter.
        The \c scoped_grid_dims URL parameter tells the library to include 
        the name of the enclosing grid when adding Maps to the netcdf dataset's
        list of dimensions.
        
        @note The default value of this parameter is false. This is useful with
        hdf4 data sources.
        
        @return The value of the parameter. */
    bool get_scoped_grid_dims()  const { return d_scoped_grid_dims; }
    
    /// Get the value of \c limit.
    int get_limit(const string &var_name) const;
    
    /** Get the default value of \c limit. If the \c limit parameter was not 
        included with the URL, return 0. This indicates that DAP String variables
        should use a default size of 255 characters and Sequences should use a
        default size of one instance. */
    int get_default_limit() const {
        return d_default_limit;
    }
    
    /// Get the URL without the client-side parameters.
    string url_without_params() const;
};

#endif // client_params_h
