
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libnc-dods, A C++ implementation of netCDF 3 which
// supports reading from local files and OPeNDAP servers.

// Copyright (c) 2004 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.
 
#include <cppunit/TextTestRunner.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/extensions/HelperMacros.h>

// Tests for ClientParam. jhrg 10/26/04

#include "ClientParams.h"

#include "debug.h"

using namespace CppUnit;
using namespace std;

class ClientParamsTest: public TestFixture {
private:
    ClientParams *cp;

public: 
    ClientParamsTest() {}
    ~ClientParamsTest() {}

    void setUp() {}

    void tearDown() {}

    CPPUNIT_TEST_SUITE( ClientParamsTest );

    CPPUNIT_TEST(url_without_params_test);
    CPPUNIT_TEST(constructor_test);
    CPPUNIT_TEST(errors_test);
    
    CPPUNIT_TEST_SUITE_END();

    void errors_test() {
        try{
            ClientParams *cp = new ClientParams("[limot=17]http://localhost/");
            CPPUNIT_ASSERT(!"Misspelled parameter name.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }

        try{
            ClientParams *cp = new ClientParams("[limit=]http://localhost/");
            CPPUNIT_ASSERT(!"Malformed expression.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }
    
        try{
            ClientParams *cp = new ClientParams("[limit=true]http://localhost/");
            CPPUNIT_ASSERT(!"Bad value.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }    

        try{
            ClientParams *cp = new ClientParams("[limit=0.17]http://localhost/");
            CPPUNIT_ASSERT(!"Bad value.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }
        try{
            ClientParams *cp = new ClientParams("[limit-var0-error=true]http://localhost/");
            CPPUNIT_ASSERT(!"Bad value.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }    

        try{
            ClientParams *cp = new ClientParams("[limit-var1_error=0.17]http://localhost/");
            CPPUNIT_ASSERT(!"Bad value.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }


        try{
            ClientParams *cp = new ClientParams("[preload=1]http://localhost/");
            CPPUNIT_ASSERT(!"Bad value.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }
        try{
            ClientParams *cp = new ClientParams("[preload=]http://localhost/");
            CPPUNIT_ASSERT(!"Malformed expression.");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT("Caught error.");
        }
    }
    
    void constructor_test() {
        try { 
            ClientParams *cp = new ClientParams("http://localhost/");
            CPPUNIT_ASSERT(0 == cp->get_default_limit());
            CPPUNIT_ASSERT(false == cp->get_preload());
            CPPUNIT_ASSERT(0 == cp->get_limit(""));
            CPPUNIT_ASSERT(0 == cp->get_limit("test_var_name"));
            delete cp;
    
            cp = new ClientParams("[limit=100]http://localhost/");
            CPPUNIT_ASSERT(100 == cp->get_default_limit());
            CPPUNIT_ASSERT(false == cp->get_preload());
            CPPUNIT_ASSERT(100 == cp->get_limit(""));
            CPPUNIT_ASSERT(100 == cp->get_limit("test_var_name"));
            delete cp;
    
            cp = new ClientParams("[limit=10][preload=true]http://localhost/");
            CPPUNIT_ASSERT(10 == cp->get_default_limit());
            CPPUNIT_ASSERT(true == cp->get_preload());
            CPPUNIT_ASSERT(10 == cp->get_limit(""));
            CPPUNIT_ASSERT(10 == cp->get_limit("test_var_name"));
            delete cp;
    
            cp = new ClientParams("[limit-var1=13][preload=true]http://localhost/");
            CPPUNIT_ASSERT(0 == cp->get_default_limit());
            CPPUNIT_ASSERT(true == cp->get_preload());
            CPPUNIT_ASSERT(0 == cp->get_limit(""));
            CPPUNIT_ASSERT(13 == cp->get_limit("var1"));
            delete cp;
    
            cp = new ClientParams("[limit-var1=13][limit=11][preload=true]http://localhost/");
            CPPUNIT_ASSERT(11 == cp->get_default_limit());
            CPPUNIT_ASSERT(true == cp->get_preload());
            CPPUNIT_ASSERT(11 == cp->get_limit("var11"));
            CPPUNIT_ASSERT(13 == cp->get_limit("var1"));
            delete cp;
            
            cp = new ClientParams("tooth_fairy.nc");
            CPPUNIT_ASSERT(cp->url_without_params() == "tooth_fairy.nc");
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT(!"Got an Error");
        }
    }
    
    void url_without_params_test() {
        try {
            ClientParams *cp = new ClientParams("http://localhost/");
            CPPUNIT_ASSERT("http://localhost/" == cp->url_without_params());
            delete cp;

            cp = new ClientParams("[limit=100]http://localhost/");
            CPPUNIT_ASSERT("http://localhost/" == cp->url_without_params());
            delete cp;

            cp = new ClientParams("[limit=10][preload=true]http://localhost/");
            CPPUNIT_ASSERT("http://localhost/" == cp->url_without_params());
            delete cp;
        }
        catch (Error &e) {
            cerr << e.get_error_message() << endl;
            CPPUNIT_ASSERT(!"Got an Error");
        }
    }
};

CPPUNIT_TEST_SUITE_REGISTRATION(ClientParamsTest);

int 
main( int argc, char* argv[] )
{
    CppUnit::TextTestRunner runner;
    runner.addTest( CppUnit::TestFactoryRegistry::getRegistry().makeTest() );

    bool wasSuccessful = runner.run( "", false ) ;

    return wasSuccessful ? 0 : 1;
}
