=begin
= GrADS_Gridded  -- a class for GrADS gridded datasets

by T Horinouchi

==Overview

a GrADS_Gridded object corresponds to a GrADS control file,
through which the users can also access its binary data file(s).

==Current Limitations

* Multiple files are not supported (i.e., the template option is ignored)
* The "units" field in variable definitions are only partially interpreted.
  Interpreted are 99 or 0 (default), and -1,30.
* Endian conversion is not supported
* options yrev and zrev are not interpreted
* option 365_day_calendar is not interpreted
* Partial support of possible units of the time increment:
  mn (minutes), hr (hours) and dy (days) are supported, but
  mo (month) and yr (year) is not. 
    (Developer's memo): Time is stored as Julian day and is handled with
    the Date standard library. Thus, it's not difficult to support the
    unsupported units. However, it would be tricky to support the
    365_day_calendar option.

==Class Methods

---GrADS_Gridded.new(ctlfilename, mode="r")
   same as GrADS_Gridded.open

---GrADS_Gridded.open(ctlfilename, mode="r")
   make a new GrADS_Gridded object. 

   ARGUMENTS
   * ctlfilename (String): name of the control file to open
   * mode (String): IO mode. "r" or "w+" (precisely speaking, you can
     use "w", but it is interpreted as "w+". Thus, all files opened 
     will be readable). CURRENTLY, ONLY "r" IS HANDLED.

   REMARK
   * You can modify the object through instance methods even if mode=="r".
     In that case, the modification will not be written in the original
     control file.

==Methods
---to_ctl
   returns the contents of the corresponding control file as a String.

   REMARK
   * The contents is reconstructed from the internal data of the object.
     Therefore, even when the object is based on a read-only control file,
     it is not necessarily the same as the original one. It is especially
     true when the object was modified after it is opened.

---get(name, level, time)
   reads the binary data and returns as a NArray.

   ARGUMENTS
   * name (String): name of the variable to read
   * level (Integer, currently): vertical level to read (0,1,2,...; starting 
     from 0). Currently only one vertical levels must be chosen, but in the
     future, it is planned to support multiple levels.
   * time (Integer, currently): time to read (0,1,2,...;  starting 
     from 0). Currently only one time must be chosen, but in the
     future, it is planned to support multiple times.

---varnames
   Returns names of the variable in the GrADS file as an Array in the order
   placed.

---dimensions
   Returns info on the four dimensions.

   RETURN VALUE
   * an Array of 4 elements: dimension[0] for x, dimension[1] for y,
     dimension[2] for z, and dimension[3] for t. Each of them is a
     Hash like the following:
       {:name=>"x", 
       :len=>132, 
       :flag=>"LINEAR",
       :spec=>"-0.7500         1.5000",
       :start=>-0.75, :increment=>1.5,
       :description=>"longitude", 
       :units=>"degrees_east"}
     Here, :len, :flag, and :spec are directly from the control file, while
     others are derived properties for internal use. 

   WARNING
   * Each elements of the return value is not a clone but is a direct 
     association to an internal object of the object. Therefore, to 
     modify it is to modify the object. That is, dimensions[0][:len]=10
     would actually change the internal variable, while dimensions[0]=nil
     has no effect (the former is a substitution IN a Hash, while the latter
     is a substitution OF the Hash).

---ctlfilename
   path of the control file

---title
---title=
   get/set the title

---undef
---undef=
   get/set the undef value

---dset
---dset=
   get/set the dset string
=end

require "date"
require "narray"

class GrADS_Gridded

  class << self
    alias open new
  end

  def initialize(ctlfilename, mode="r")

    case(mode)
    when /^r/
      @mode = 'r'
    when /^w/
      @mode = 'w+'
    else
      raise ArgumentError, "Unsupported IO mode: #{mode}"
    end

    @ctlfile = File.open(ctlfilename, mode)
    @options = {    # initialization
      "yrev"=>nil, 
      "zrev"=>nil,
      "sequential"=>nil,
      "byteswapped"=>nil,
      "template"=>nil,
      "big_endian"=>nil,
      "little_endian"=>nil,
      "cray_32bit_ieee"=>nil,
    }

    if (File.exists?(ctlfilename))
      parse_ctl
    else
      if @mode != 'w'
	raise "File #{ctlfilename} does not exist."
      end
      initialize_new
    end
  end

  def to_ctl
    return <<EOS
DSET    #{@dset}
TITLE   #{@title}
UNDEF   #{@undef}
OPTIONS #{op=""; @options.each{|key,val| op += key+" " if(val)}; op}
XDEF    #{@dimensions[0][:len]}\t#{@dimensions[0][:flag]}\t#{@dimensions[0][:spec]}
YDEF    #{@dimensions[1][:len]}\t#{@dimensions[1][:flag]}\t#{@dimensions[1][:spec]}
ZDEF    #{@dimensions[2][:len]}\t#{@dimensions[2][:flag]}\t#{@dimensions[2][:spec]}
TDEF    #{@dimensions[3][:len]}\t#{@dimensions[3][:flag]}\t#{@dimensions[3][:spec]}
VARS
#{@variables.collect{|i| "  "+i[:name]+"\t"+i[:nlev].to_s+"\t"+i[:option].to_s+
       "\t"+i[:description]}.join("\n")}
ENDVARS
EOS
  end

  def inspect
    return <<EOS
#{self.class}
file: #{@ctlfile.path}
DSET #{@dset}
OPTIONS #{@options.inspect}
XDIM #{@dimensions[0].inspect}
YDIM #{@dimensions[1].inspect}
ZDIM #{@dimensions[2].inspect}
TDIM #{@dimensions[3].inspect}
VARS
#{@variables.collect{|i| "  "+i[:name]+"\t"+i[:nlev].to_s+"\t"+i[:option].to_s+
       "\t"+i[:description]}.join("\n")}
ENDVARS
EOS
  end

  def varnames
    @variables.collect{|i| i[:name]}
  end

  def get(name, level, time)
    start_byte = start_byte(name, level, time)
    @datafile.pos=start_byte
    if(!@map[name][:xytranspose])
      ary = NArray.to_na(@datafile.read(@xybytes), "sfloat",
			 @dimensions[0][:len],@dimensions[1][:len])
    else
      ary = NArray.to_na(@datafile.read(@xybytes), "sfloat",
			 @dimensions[1][:len],@dimensions[0][:len]).trasnpose
    end
    ary
  end

  attr_accessor (:title, :undef, :dset)
  def dimensions
    @dimensions.clone
  end
  def variables
    @variables.clone
  end
  def ctlfilename
    @ctlfile.path
  end

  ########## private methods #############
  private
  
  def parse_ctl
    @ctlfile.rewind
    @fileheader_len = 0     # defalut value
    @title = ""        
    @variables = []         # initalization
    @dimensions = []        # initalization
    while ( line = @ctlfile.gets )
      case(line)
      when /^\s*\*/,/^\s*$/
	# do nothing
      when /^\s*DSET\s*(\S*)/i
	if ($1)
	  @dset = $1
	  @datafile = File.open(@dset,@mode)
	else
	  raise "Invalid line: "+line
	end
      when /^\s*TITLE\s*(\S+.*)$/i
	if ($1)
	  @title = $1
	else
	  raise "Invalid line: "+line
	end
      when /^\s*UNDEF\s*(\S*)/i
	if ($1)
	  @undef = $1.to_f
	else
	  raise "Invalid line: "+line
	end
      when /^\s*FILEHEADER\s*(\S*)/i
	if ($1)
	  @fileheader_len = $1.to_i
	else
	  raise "Invalid line: "+line
	end
      when /^\s*OPTIONS\s*(\S+.*)$/i
	if ($1)
	  $1.split.each{ |opt|
	    if (@options.has_key?(opt))
	      @options[opt] = true
	    else
	      raise "Invalid/unsupported option: "+opt
	    end
	  }
	else
	  raise "Invalid line: "+line
	end
      when /^\s*[XYZT]DEF/i
	/^\s*([XYZT])DEF\s+(\d+)\s+(\S+)\s+(.*)$/i =~ line
	if ( (len=$2) && (flag=$3) && (spec=$4))
	  dim = {:len=>len.to_i, :flag=>flag, :spec=>spec}
	  case $1
	  when /X/i
	    idim=0
	    dim[:name] = 'x'
	    dim[:description] = 'longitude'
	    dim[:units] = 'degrees_east'
	  when /Y/i
	    idim=1
	    dim[:name] = 'y'
	    dim[:description] = 'latitude'
	    dim[:units] = 'degrees_north'
	  when /Z/i
	    idim=2
	    dim[:name] = 'z'
	    dim[:description] = 'pressure level'
	    dim[:units] = 'hPa'
	  when /T/i
	    dim[:name] = 't'
	    dim[:description] = 'time'
	    idim=3
	  end
	  if (idim!=3)
	    if (dim[:flag] =~ /LINEAR/i)
	      begin
		dim[:start],dim[:increment] = spec.split.collect!{|i| i.to_f}
	      rescue NameError,StandardError
		raise $!.to_s+"\nCannot read start and increment from: "+spec
	      end
	    elsif (dim[:flag] =~ /LEVELS/i)
	      dim[:levels] = []
	      pos = @ctlfile.pos    # back up for a one-line rewind
	      dim[:spec] = "\n"
	      while (dim[:spec] += levs = @ctlfile.gets)
		if( /^\s*[\d\-\.]/ =~ levs )
		  dim[:levels] += levs.split.collect!{|i| i.to_f}
		  #p '###  levels',dim[:levels].length,"  ",levs.split
		  pos = @ctlfile.pos    # back up for a one-line rewind
		else
		  @ctlfile.pos = pos  # one-line rewind (note: IO#lineno= doesn't work for this purpose)
		  break
		end
	      end
	    else
	      raise "invalid or not-yet-supported dimension flag: "+dim[:flag]
	    end
	  else
	    # idim = 3 --- time
	    if (dim[:flag] =~ /LINEAR/i)
	      start,increment= spec.split
	      julian_day = parse_starttime(start)
	      dim[:start] = julian_day
	      dim[:start_units] = 'days since '+Date.new1(0.0).to_s
	      dim[:increment],dim[:increment_units] = parse_timeincrement(increment)
	    else
	      raise "invalid dimension flag(only LINEAR is available for time)"
	    end
	  end
	  @dimensions[idim]=dim
	else
	  raise "Invalid line: "+line
	end
      when /^\s*VARS/i
	total_levs=0
	while ( vline = @ctlfile.gets )
	  case(vline)
	  when /^\s*\*/,/^\s*$/
	    # do nothing
	  when /^\s*ENDVARS/i
	    break
	  else
	    vline =~ /^\s*(\S+)\s+(\S+)\s+(\S+)\s+(\S+.*?)\s*$/
	    if( !($1 && $2 && $3 && $4) )
	      raise "Something is wrong with this line: "+vline
	    end
	    nlev = max($2.to_i,1)
	    total_levs += nlev
	    @variables.push({:name=>$1,:nlev=>nlev,:option=>$3,
			      :description=>$4})
	  end
	end
	@map=Hash.new
	cum_lev = 0
	@variables.each{|i|
	  varname = i[:name]
	  case (i[:option])
	  when /^0/,/^99/,/^-1,30/
	    @map[varname]={:offset=>@fileheader_len,:nlev=>i[:nlev],
	      :start=>cum_lev,:zstep=>1,:tstep=>total_levs}
	    if i[:option] =~ /^-1,30/
	      @map[varname][:xytranspose]=true
	    end
	    #p "## test - #{varname}",@map[varname]
	  else
	    raise "invalid or unsupported variable placement option: "+i[:option]
	  end
	  cum_lev += i[:nlev]
	}
      end

    end

    #<check whether all the mandatory specifications are done>
    for i in 0..3
      raise "#{i}-th dimension is not found " if( ! @dimensions[i] )
    end
    raise "UNDEF field is not found" if(!@undef)
    raise "DSET field is not found" if(!@dset)
    raise "VARS field is not found" if(!@variables)

    #<post processing>
    @xybytes = 4 * @dimensions[0][:len] * @dimensions[1][:len]

  end

  def initialize_new

    raise "Sorry, this method is far from completed"

    @dims = []
    @variables = []
    
    #<attributes>
    @dset = nil
    @title = nil
    @undef = nil
    @fileheader_len = 0
    @options = []
    
    #<internal control parameters>
    @define_mode = true
    @ctl_dumped = false
  end
  
  def start_byte(name, level, time)
    # offset to read an xy section of the variable with NAME
    # at LEVEL(counted from 0) at TIME(conted from 0)
    if (map = @map[name] )
      if (level<0 || level>=map[:nlev])
	raise "Level #{level} is out of the range of the variable #{name}"
      end
      if (time<0 || time>=@dimensions[3][:len])
	raise "Time #{time} is not in the data period"
      end
      iblock = map[:start]+level*map[:zstep]+time*map[:tstep]
      str_byte = map[:offset] + @xybytes*iblock
      if( @options["sequential"] )
	str_byte += iblock*8 + 4
      end
    else
      raise "Variable does not exist: "+name
    end
    str_byte
  end

  def parse_starttime(string)
    ## interpret the hh:mmZddmmmyyyy format for grads
    
    if (/(.*)Z(.*)/ =~ string)
      stime = $1
      sdate = $2
    else
      # must be date, not time, since month and year are mandatory
      sdate = string  
      stime = ''
    end
    
    if ( /(\d\d):(\d\d)/ =~ stime )
      begin
	shour = $1
	smin = $2
	hour = shour.to_i
	min = smin.to_i
      rescue StandardError,NameError
	raise "Cannot convert hour or time into interger: "+stime
      end
    else
      #shour = '00'
      #smin = '00'
      hour = 0
      min = 0
    end
    timeofday = min/1440.0 + hour/24.0

    if ( /(\d*)(\w\w\w)(\d\d\d\d)/ =~ sdate )
      sday = $1
      smon = $2
      syear = $3
      begin
	day = sday.to_i
	year = syear.to_i
	mon=['jan','feb','mar','apr','may','jun','jul','aug','sep',
	  'oct','nov','dec'].index(smon.downcase) + 1
	date = Date.new(year,mon,day)
      rescue StandardError, NameError
	raise $!+"\nCould not parse the date string: "+sdate
      end
    else
      raise "The date part must be [dd]mmmyyyy, but what was given is: "+sdate
    end

    #print '## full start time: ',shour+':'+smin+'Z'+sday+smon+syear,"\n"
    #print '## ',date.to_s,' ',timeofday,"\n"

    julian_day = date.jd + timeofday

    return julian_day
  end

  def parse_timeincrement(string)
    if ( /(\d+)(\w\w)/ =~ string )
      sincrement = $1.to_i
      sunits = $2
      case sunits
      when /mn/i
	fact = 1.0/1440.0   # factor to convert into days
	units = 'day'
	increment = sincrement * fact
      when /hr/i
	fact = 1.0/24.0
	units = 'day'
	increment = sincrement * fact
      when /dy/i
	fact = 1.0
	units = 'day'
	increment = sincrement * fact
      when /mo/i
	fact = 1.0
	units = 'month'
	increment = sincrement * fact
      when /yr/i
	fact = 1.0
	units = 'year'
	increment = sincrement * fact
      else
	raise "invalid units: "+sunits
      end
    else
      raise "invalid time-increment string: "+sunits
    end
    #print "### ",increment, ' ',units,"\n"
    return [increment, units]
  end

  def max(a,b)
    a>b ? a : b
  end

end

if __FILE__ == $0 then
  Dir.chdir('/murtmp/horinout/arps-data/much1-saigen-20020213_cont2/grads')
  ctlfilename = 'fi1-much-132x132x15.gradscntl'
  gr = GrADS_Gridded.open(ctlfilename)
  #p gr
  print gr.to_ctl
  p gr.varnames
  #print "start_byte  qs ",gr.start_byte('qs',0,0),"\n"
  #print "start_byte  zp ",gr.start_byte('zp',0,0),"\n"
  #print "start_byte  zp ",gr.start_byte('zp',0,1),"\n"
  p gr.get('prcrt1',0,0)
end
