!= リスタートデータ, 初期値データ入出力
!
!= Restart data, initial data input/output
!
! Authors::   Yasuhiro Morikawa, Yoshiyuki O. Takahashi
! Version::   $Id: restart_file_io.f90,v 1.26 2013/09/30 03:03:42 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2010. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module restart_file_io
  !
  != リスタートデータ, 初期値データ入出力
  !
  != Restart data, initial data input/output
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! リスタートデータもしくは初期値データの入出力を行います. 
  ! 入力ファイル, 出力ファイル, データの出力の間隔は
  ! NAMELIST#restart_file_io_nml で設定します. 
  !
  ! リスタートデータの入力ファイルが指定されない場合, 
  ! initial_data モジュールで生成された初期値データを取得します. 
  !
  ! Restart data or initial data is input/output. 
  ! Settings of input file, output file, and interval of data output 
  ! is configured by "NAMELIST#restart_file_io_nml". 
  !
  ! If input file of restart data is not set, 
  ! initial data is generated in "initial_data" module, and 
  ! obtained data from the module. 
  !
  !== Procedures List
  !
  ! RestartFileOpen   :: リスタート/初期値ファイルのオープン
  ! RestartFileOutput :: リスタート/初期値ファイルへのデータ出力
  ! InitialFileOutput :: 初期値ファイルへのデータ出力
  ! RestartFileClose  :: リスタート/初期値ファイルのクローズ
  ! RestartFileGet    :: リスタート/初期値ファイルの入力
  ! ------------      :: ------------
  ! RestartFileOpen   :: Open restart/initial file
  ! RestartFileOutput :: Data output to restart/initial file
  ! InitialFileOutput :: Data output to initial file
  ! RestartFileClose  :: Close restart/initial file
  ! RestartFileGet    :: Input restart/initial file
  !
  !== NAMELIST
  !
  ! NAMELIST#restart_file_io_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 組成に関わる配列の設定
  ! Settings of array for atmospheric composition
  !
  use composition, only: &
    &                    ncmax, &
                              ! 成分の数
                              ! Number of composition
    &                    a_QMixName, &
                              ! 成分の変数名
                              ! Name of variables for composition
    &                    a_QMixLongName, &
                              ! 成分の長い変数名
                              ! Long name of variables for composition
    &                    IndexH2OVap
                              ! 水蒸気のインデックス
                              ! Index for water vapor


  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING, &  ! 文字列.       Strings. 
    &                 TOKEN      ! キーワード.   Keywords. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! gtool4 データ出力
  ! Gtool4 data output
  !
  use gtool_history, only: GT_HISTORY


  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: RestartFileInit
  public :: RestartFileOpen, RestartFileClose, RestartFileOutPut, InitialFileOutput
  public :: RestartFileGet

  ! 公開変数
  ! Public variables
  !
  logical, save, public:: restart_file_io_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  logical, save, public:: restart_file_opened = .false.
                              ! リスタートファイルのオープンに関するフラグ. 
                              ! Flag of restart file open

  character(STRING), save, public:: InputFile
                              ! 入力するリスタートデータのファイル名
                              ! Filename of input restart data

  character(STRING), save, public:: OutputFile
                              ! 出力するリスタートデータのファイル名
                              ! Filename of output restart data
  real(DP), save, public:: IntValue
                              ! リスタートデータの出力間隔 (数値). 
                              ! Interval of restart data output (numerical value)
  character(TOKEN), save, public:: IntUnit
                              ! リスタートデータの出力間隔 (単位). 
                              ! Interval of restart data output (unit)
  real(DP), save, public:: IntTime
                              ! リスタートデータの出力間隔 [秒]. 
                              ! Interval of restart data output [sec]

  ! 非公開変数
  ! Private variables
  !
  logical, save:: flag_init_data_save
                              ! 初期値データを作成する場合は, 
                              ! この引数に .true. を与えます. 
                              ! 
                              ! If initial data is created, 
                              ! give ".true." to this argument. 

  type(GT_HISTORY), save:: gthst_rst
                              ! リスタートデータ用 gtool_history#GT_HISTORY 変数
                              ! "gtool_history#GT_HISTORY" variable for restart data

  real(DP), save:: PrevOutputTime
                              ! 前回の出力時間. 
                              ! Previous output time
  logical, save:: flag_output_end
                              ! 計算最終時刻の出力完了のフラグ. 
                              ! Flag for completion of output at the end time of calculation

  character(*), parameter:: module_name = 'restart_file_io'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: restart_file_io.f90,v 1.26 2013/09/30 03:03:42 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine RestartFileOpen( flag_init_data )
    !
    ! リスタート/初期値ファイルをオープンします. 
    !
    ! A restart/initial data file is opened. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 出力ファイルの基本情報
    ! Basic information for output files
    ! 
    use fileset, only: &
      & FileTitle, &
                              ! 出力データファイルの表題. 
                              ! Title of output data files
      & FileSource, &
                              ! データファイル作成の手段. 
                              ! Source of data file
      & FileInstitution
                              ! データファイルを最終的に変更した組織/個人. 
                              ! Institution or person that changes data files for the last time

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! 円周率. Circular constant

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon, &
                              ! $ \lambda $ [rad.] . 経度. Longitude
      & x_Lon_Weight, &
                              ! $ \Delta \lambda $ [rad.] . 
                              ! 経度座標重み. 
                              ! Weight of longitude
      & y_Lat, &
                              ! $ \varphi $ [rad.] . 緯度. Latitude
      & y_Lat_Weight, &
                              ! $ \Delta \varphi $ [rad.] . 
                              ! 緯度座標重み. 
                              ! Weight of latitude
      & z_Sigma, &
                              ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level
      & r_Sigma, &
                              ! $ \sigma $ レベル (半整数). 
                              ! Half $ \sigma $ level
      & z_DelSigma
                              ! $ \Delta \sigma $ (整数). 
                              ! $ \Delta \sigma $ (Full)

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & RestartTime, &        ! リスタート開始時刻. 
                              ! Restart time of calculation
      & InitialDate           ! 計算開始日時. 
                              ! Start date of calculation

    ! gtool4 データ出力
    ! Gtool4 data output
    !
    use gtool_history, only: HistoryCreate, HistoryAddVariable, &
      & HistoryPut, HistoryAddAttr

    ! 暦と日時の取り扱い
    ! Calendar and Date handler
    !
    use dc_calendar, only: DCCalInquire, DCCalDateInquire

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: CPrintf, LChar

    ! 組み込み関数 PRESENT の拡張版関数
    ! Extended functions of intrinsic function "PRESENT"
    !
    use dc_present, only: present_and_true

    ! 宣言文 ; Declaration statements
    !
    implicit none
    logical, intent(in), optional:: flag_init_data
                              ! 初期値データを作成する場合は, 
                              ! この引数に .true. を与えます. 
                              ! 
                              ! If initial data is created, 
                              ! give ".true." to this argument. 

    ! 作業変数
    ! Work variables
    !
    character(STRING):: title_msg
                              ! 表題に付加するメッセージ. 
                              ! Message added to title
    real(DP):: origin_time
                              ! 計算開始時刻. 
                              ! Start time of calculation

    integer:: InitialYear, InitialMonth, InitialDay, InitialHour, InitialMin
                              ! 計算開始の年月日時分. 
    real(DP):: InitialSec
                              ! 計算開始の秒. 

    character(TOKEN):: cal_type
                              ! 暦のタイプ. 
    integer:: month_in_year, hour_in_day, min_in_hour
    integer, pointer:: day_in_month_ptr(:) => null()
    real(DP):: sec_in_min
                              ! 暦の詳細情報

    logical:: flag_mpi_init

    integer :: n


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    flag_init_data_save = present_and_true( flag_init_data )

    if ( .not. restart_file_io_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    if ( restart_file_opened ) return

    ! 表題に付加するメッセージの設定
    ! Configure message added to title
    !
    if ( .not. flag_init_data_save ) then
      title_msg = ' restart data'
    else
      title_msg = ' initial data'
    end if

    ! 時刻情報の取得
    ! Get time information
    !
    if ( .not. flag_init_data_save ) then
      origin_time = RestartTime + IntTime
    else
      origin_time = RestartTime
    end if

    flag_mpi_init = .false.

    ! リスタートファイルのオープン
    ! Open a restart file
    !
    call HistoryCreate( &
      &      file = OutputFile, &                                   ! (in)
      &     title = trim(FileTitle) // trim(title_msg), &           ! (in)
      &    source = FileSource, institution = FileInstitution, &    ! (in)
      &      dims = (/ 'lon    ', 'lat    ', 'sig    ', 'sigm   ', &
      &                'timestr', 'time   ' /), &                   ! (in)
      &  dimsizes = (/ imax, jmax, kmax, kmax + 1, TOKEN, 0 /), &   ! (in)
      & longnames = (/ 'longitude                             ', &
      &                'latitude                              ', &
      &                'sigma at layer midpoints              ', &
      &                'sigma at layer end-points (half level)', &
      &                'number of characters for datetime     ', &    ! (in)
      &                'time                                  ' /), & ! (in)
      &     units = (/ 'degrees_east ', 'degrees_north', &
      &                '1            ', '1            ', &
      &                '1            ',                  &
      &                'sec          ' /), &                           ! (in)
      & xtypes = (/'double', 'double', 'double', &
      &            'double', 'int   ', 'double'/), &                  ! (in)
      &        origind = origin_time, &       ! (in) optional
      &      intervald = IntTime, &           ! (in) optional
!!$      & flag_mpi_split = flag_mpi_init, &     ! (in) optional
      & flag_mpi_split = .true., &            ! (in) optional
      &        history = gthst_rst )          ! (out) optional


    ! ※ 元々再計算データかどうかのチェックのためのフラグとして
    !    作成したものであるため、下記 flag_rst 設定完了後 
    !    削除予定 (morikawa  2010/06/13)
    !
    ! $ \Delta t $ に関する情報を追加. 
    ! Add information about $ \Delta t $. 
    !
!!$    if ( .not. flag_init_data_save ) then
!!$      call HistoryAddVariable( &
!!$        & varname = 'deltime', &             ! (in)
!!$        & dims = (/''/), &                   ! (in)
!!$        & longname = 'delta time', &         ! (in)
!!$        & units = 'sec', xtype = 'double', & ! (in)
!!$        & history = gthst_rst )              ! (inout)
!!$      call HistoryPut( &
!!$        & varname = 'deltime', &            ! (in)
!!$        & array = (/ DelTime /), &          ! (in)
!!$        & history = gthst_rst )             ! (inout)
!!$    end if

    ! 開始日時情報を "time" の属性として設定
    ! Set start date information as attributes of "time"
    !
    call DCCalDateInquire( &
      & InitialYear, InitialMonth, InitialDay, & ! (out)
      & InitialHour, InitialMin,   InitialSec, & ! (out)
      & date = InitialDate )               ! (in)

    call HistoryAddAttr( &
      & 'time', attrname = 'origin', &     ! (in)
      & value = 'origin_year origin_month origin_day origin_hour origin_min origin_sec', &
                                           ! (in)
      & history = gthst_rst )              ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'origin_year', & ! (in)
      & value = InitialYear, &                ! (in)
      & history = gthst_rst )               ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'origin_month', & ! (in)
      & value = InitialMonth, &                ! (in)
      & history = gthst_rst )                ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'origin_day', &  ! (in)
      & value = InitialDay, &                 ! (in)
      & history = gthst_rst )               ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'origin_hour', &  ! (in)
      & value = InitialHour, &                 ! (in)
      & history = gthst_rst )                ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'origin_min', &  ! (in)
      & value = InitialMin, &                 ! (in)
      & history = gthst_rst )               ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'origin_sec', &  ! (in)
      & value = InitialSec, &                 ! (in)
      & history = gthst_rst )               ! (inout)

    ! 暦情報を "time" の属性として設定
    ! Set calendar information as attributes of "time"
    !
    call DCCalInquire( &
      & cal_type         = cal_type, &          ! (out)
      & month_in_year    = month_in_year, &     ! (out)
      & day_in_month_ptr = day_in_month_ptr , & ! (out)
      & hour_in_day      = hour_in_day  , &     ! (out)
      & min_in_hour      = min_in_hour  , &     ! (out)
      & sec_in_min       = sec_in_min )         ! (out)

    call HistoryAddAttr( &
      & 'time', attrname = 'calendar', & ! (in)
      & value = cal_type, &              ! (in)
      & history = gthst_rst )            ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'month_in_year', & ! (in)
      & value = month_in_year, &              ! (in)
      & history = gthst_rst )                 ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'day_in_month', & ! (in)
      & value = day_in_month_ptr, &          ! (in)
      & history = gthst_rst )                ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'hour_in_day', & ! (in)
      & value = hour_in_day, &              ! (in)
      & history = gthst_rst )               ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'min_in_hour', & ! (in)
      & value = min_in_hour, &              ! (in)
      & history = gthst_rst )               ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'sec_in_min', & ! (in)
      & value = sec_in_min, &              ! (in)
      & history = gthst_rst )              ! (inout)

    deallocate( day_in_month_ptr )

    ! 再計算データ用のフラグ設定
    ! Set flag for a restart data file. 
    !
    call HistoryAddVariable( &
      & varname = 'flag_rst', &               ! (in)
      & dims = (/''/), &                      ! (in)
      & longname = 'flag for restart data', & ! (in)
      & units = '1', xtype = 'int', &         ! (in)
      & history = gthst_rst )                 ! (inout)
    call HistoryAddAttr( &
      & varname = 'flag_rst', &               ! (in)
      & attrname = 'comment', &               ! (in)
      & value = 'If this value is nonzero, this file provides restart data', &
                                                ! (in)
      & history = gthst_rst )                 ! (inout)

    if ( flag_init_data_save ) then
      call HistoryPut( &
        & varname = 'flag_rst', &               ! (in)
        & array = (/ 0 /), &                    ! (in)
        & history = gthst_rst )                 ! (inout)
    else
      call HistoryPut( &
        & varname = 'flag_rst', &               ! (in)
        & array = (/ 1 /), &                    ! (in)
        & history = gthst_rst )                 ! (inout)
    end if

    ! 年月日時分秒形式の日時情報変数の設定
    ! Set a date information variable with year-month-day hour:minute:second format
    !
    call HistoryAddVariable( &
      & varname = 'datetime', &                     ! (in)
      & dims =  (/'timestr', 'time   ' /), &        ! (in)
      & longname = 'time represented as strings', & ! (in)
      & units = '1', xtype = 'char', &              ! (in)
      & history = gthst_rst )                       ! (inout) optional

    ! 座標データの設定
    ! Axes data settings
    !
    call HistoryAddAttr( &
      & 'lon', attrname = 'standard_name', &     ! (in)
      & value = 'longitude', &                   ! (in)
      & history = gthst_rst )                    ! (inout)
    call HistoryAddAttr( &
      & 'lat', attrname = 'standard_name', &     ! (in)
      & value = 'latitude', &                    ! (in)
      & history = gthst_rst )                    ! (inout)
    call HistoryAddAttr( &
      & 'sig', attrname = 'standard_name', &     ! (in)
      & value = 'atmosphere_sigma_coordinate', & ! (in)
      & history = gthst_rst )                    ! (inout)
    call HistoryAddAttr( &
      & 'sigm', attrname = 'standard_name', &    ! (in)
      & value = 'atmosphere_sigma_coordinate', & ! (in)
      & history = gthst_rst )                    ! (inout)
    call HistoryAddAttr( &
      & 'time', attrname = 'standard_name', &    ! (in)
      & value = 'time', &                        ! (in)
      & history = gthst_rst )                    ! (inout)
    call HistoryAddAttr( &
      & 'sig', attrname = 'positive', &          ! (in)
      & value = 'down', &                        ! (in)
      & history = gthst_rst )                    ! (inout)
    call HistoryAddAttr( &
      & 'sigm', attrname = 'positive', &         ! (in)
      & value = 'down', &                        ! (in)
      & history = gthst_rst )                    ! (inout)

    call HistoryPut( &
      & 'lon', x_Lon / PI * 180.0_DP, & ! (in)
      & history = gthst_rst )           ! (inout)
    call HistoryPut( &
      & 'lat', y_Lat / PI * 180.0_DP, & ! (in)
      & history = gthst_rst )           ! (inout)
    call HistoryPut( &
      & 'sig', z_Sigma, &               ! (in)
      & history = gthst_rst )           ! (inout)
    call HistoryPut( & 
      & 'sigm', r_Sigma, &              ! (in)
      & history = gthst_rst )           ! (inout)

    ! 座標重みの設定
    ! Axes weights settings
    !
    call HistoryAddVariable( 'lon_weight', &               ! (in)
      & (/'lon'/), &                                       ! (in)
      & 'weight for integration in longitude', 'radian', & ! (in)
      & xtype = 'double', &                                ! (in)
      & history = gthst_rst )                              ! (inout)
    call HistoryAddAttr( &
      & 'lon', attrname = 'gt_calc_weight', &     ! (in)
      & value = 'lon_weight', &                   ! (in)
      & history = gthst_rst )                     ! (inout)
    call HistoryPut( &
      & 'lon_weight', x_Lon_Weight, &             ! (in)
      & history = gthst_rst )                     ! (inout)

    call HistoryAddVariable( 'lat_weight', &                      ! (in)
      & (/'lat'/), &                                              ! (in)
      & 'weight for integration in latitude', units = 'radian', & ! (in)
      & xtype = 'double', &                                       ! (in)
      & history = gthst_rst )                                     ! (inout)
    call HistoryAddAttr( &
      & 'lat', attrname = 'gt_calc_weight', &     ! (in)
      & value = 'lat_weight', &                   ! (in)
      & history = gthst_rst )                     ! (inout)
    call HistoryPut( &
      & 'lat_weight', y_Lat_Weight, &             ! (in)
      & history = gthst_rst )                     ! (inout)

    call HistoryAddVariable( 'sig_weight', &      ! (in)
      & (/'sig'/), &                              ! (in)
      & 'weight for integration in sigma', '1', & ! (in)
      & xtype = 'double', &                       ! (in)
      & history = gthst_rst )                     ! (inout)
    call HistoryAddAttr( &
      & 'sig', attrname = 'gt_calc_weight', &     ! (in)
      & value = 'sig_weight', &                   ! (in)
      & history = gthst_rst )                     ! (inout)
    call HistoryPut( &
      & 'sig_weight', z_DelSigma, &               ! (in)
      & history = gthst_rst )                     ! (inout)


    ! 予報変数の設定
    ! Predictional variables settings
    !
    if ( flag_init_data_save ) then

      ! 初期データファイル用
      ! For initial data file
      !
      call HistoryAddVariable( 'U', &                         ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'eastward wind', 'm s-1', &                         ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      call HistoryAddVariable( 'V', &                         ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'northward wind', 'm s-1', &                        ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      call HistoryAddVariable( 'Temp', &                      ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'temperature', 'K', &                               ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      do n = 1, ncmax
        call HistoryAddVariable( a_QMixName(n),                & ! (in)
          & (/ 'lon ', 'lat ', 'sig ', 'time' /),              & ! (in)
          & a_QMixLongName(n), 'kg kg-1',                      & ! (in)
          & xtype = 'double',                                  & ! (in)
          & history = gthst_rst                                & ! (inout)
          & )
      end do
      call HistoryAddVariable( 'Ps', &                        ! (in)
        & (/ 'lon ', 'lat ', 'time' /), &                     ! (in)
        & 'surface pressure', 'Pa', &                         ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)

    else

      ! リスタートデータファイル用
      ! For restart data file
      !
      call HistoryAddVariable( 'UB', &                        ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'eastward wind (at t-\Delta t)', 'm s-1', &         ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      call HistoryAddVariable( 'VB', &                        ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'northward wind (at t-\Delta t)', 'm s-1', &        ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      call HistoryAddVariable( 'TempB', &                     ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'temperature (at t-\Delta t)', 'K', &               ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      do n = 1, ncmax
        call HistoryAddVariable( trim(a_QMixName(n))//'B',    & ! (in)
          & (/ 'lon ', 'lat ', 'sig ', 'time' /),             & ! (in)
          & a_QMixLongName(n), 'kg kg-1',                     & ! (in)
          & xtype = 'double',                                 & ! (in)
          & history = gthst_rst                               & ! (inout)
          & )
      end do
      call HistoryAddVariable( 'PsB', &                       ! (in)
        & (/ 'lon ', 'lat ', 'time' /), &                     ! (in)
        & 'surface pressure (at t-\Delta t)', 'Pa', &         ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)

      call HistoryAddVariable( 'UN', &                        ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'eastward wind (at t)', 'm s-1', &                  ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      call HistoryAddVariable( 'VN', &                        ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'northward wind (at t)', 'm s-1', &                 ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      call HistoryAddVariable( 'TempN', &                     ! (in)
        & (/ 'lon ', 'lat ', 'sig ', 'time' /), &             ! (in)
        & 'temperature (at t)', 'K', &                        ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
      do n = 1, ncmax
        call HistoryAddVariable( trim(a_QMixName(n))//'N',    & ! (in)
          & (/ 'lon ', 'lat ', 'sig ', 'time' /),             & ! (in)
          & a_QMixLongName(n), 'kg kg-1',                     & ! (in)
          & xtype = 'double',                                 & ! (in)
          & history = gthst_rst                               & ! (inout)
          & )
      end do
      call HistoryAddVariable( 'PsN', &                       ! (in)
        & (/ 'lon ', 'lat ', 'time' /), &                     ! (in)
        & 'surface pressure (at t)', 'Pa', &                  ! (in)
        & xtype = 'double', &                                 ! (in)
        & history = gthst_rst )                               ! (inout)
    end if

    restart_file_opened = .true.
  end subroutine RestartFileOpen

  !--------------------------------------------------------------------------------------

  subroutine RestartFileOutput(                      &
    & xyz_UB, xyz_VB, xyz_TempB, xyzf_QMixB, xy_PsB, &   ! (in)
    & xyz_UN, xyz_VN, xyz_TempN, xyzf_QMixN, xy_PsN  &   ! (in)
    & )
    !
    ! リスタートデータの出力を行います. 
    !
    ! Output restart data

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ステップ $ t $ の時刻. 
                              ! Time of step $ t $. 
      & EndTime, &            ! 計算終了時刻. 
                              ! End time of calculation
      & InitialDate           ! 計算開始日時. 
                              ! Start date of calculation

    ! gtool4 データ出力
    ! Gtool4 data output
    !
    use gtool_history, only: HistoryPut, HistorySetTime

    ! 暦と日時の取り扱い
    ! Calendar and Date handler
    !
    use dc_calendar, only: DCCalDateInquire

    ! 組み込み関数 PRESENT の拡張版関数
    ! Extended functions of intrinsic function "PRESENT"
    !
    use dc_present, only: present_and_true

    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(in):: xyz_UB     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u (t-\Delta t) $ .   東西風速. Eastward wind
    real(DP), intent(in):: xyz_VB     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v (t-\Delta t) $ .   南北風速. Northward wind
    real(DP), intent(in):: xyz_TempB  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T (t-\Delta t) $ .   温度. Temperature
    real(DP), intent(in):: xyzf_QMixB (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q (t-\Delta t) $ .   比湿. Specific humidity
    real(DP), intent(in):: xy_PsB     (0:imax-1, 1:jmax)
                              ! $ p_s (t-\Delta t) $ . 地表面気圧. Surface pressure
    real(DP), intent(in):: xyz_UN     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u (t) $ .     東西風速. Eastward wind
    real(DP), intent(in):: xyz_VN     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v (t) $ .     南北風速. Northward wind
    real(DP), intent(in):: xyz_TempN  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T (t) $ .     温度. Temperature
    real(DP), intent(in):: xyzf_QMixN (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q (t) $ .     比湿. Specific humidity
    real(DP), intent(in):: xy_PsN     (0:imax-1, 1:jmax)
                              ! $ p_s (t) $ .   地表面気圧. Surface pressure

    ! 作業変数
    ! Work variables
    !
    logical:: flag_output
                              ! 出力のフラグ. 
                              ! Flag for output
    character(TOKEN):: date_str         ! 日時の年月日時分秒形式. 
                                        ! Date with year-month-day hour:minute:second format
    integer :: n


    ! 実行文 ; Executable statement
    !
    if ( .not. restart_file_opened ) call RestartFileOpen

    ! リスタートデータ出力であることのチェック
    ! Check restart data output
    !
    if ( flag_init_data_save ) then
      call MessageNotify( 'E', module_name, &
        & 'Now, initial data output mode.' )
    end if

    ! 出力タイミングのチェック
    ! Check output timing
    !
    flag_output = TimeN - PrevOutputTime >= IntTime
    if ( TimeN >= EndTime .and. .not. flag_output_end ) then
      flag_output = .true.
      flag_output_end = .true.
    end if
    flag_output = ( .not. TimeN == PrevOutputTime ) .and. flag_output
    flag_output = flag_init_data_save .or. flag_output

    if ( .not. flag_output ) return

    ! 次回用に, 今回の出力 (希望) 時刻 を保存
    ! Save output time (expected) in this time, for next time
    !
    PrevOutputTime = PrevOutputTime + IntTime

    ! 時刻の設定
    ! Set time
    !
    call HistorySetTime( timed = TimeN, history = gthst_rst )

    ! "TimeN" を年月日時分秒表記にして "datetime" 変数へ出力
    ! Put "TimeN" on "datetime" variable with year-month-day hour:minute:second format
    !
    call DCCalDateInquire( &
      & date_str   = date_str, &   ! (out)
      & elapse_sec = TimeN, &      ! (in) optional
      & date       = InitialDate ) ! (in) optional

    call HistoryPut( 'datetime', date_str, &  ! (in)
      &              history = gthst_rst )    ! (inout) optional

    ! データ出力
    ! Data output
    !
    call HistoryPut( &
      & 'UB', xyz_UB, history = gthst_rst ) ! (in)
    call HistoryPut( &
      & 'VB', xyz_VB, history = gthst_rst ) ! (in)
    call HistoryPut( &
      & 'TempB', xyz_TempB, history = gthst_rst ) ! (in)
    do n = 1, ncmax
      call HistoryPut( &
        & trim(a_QMixName(n))//'B', xyzf_QMixB(:,:,:,n), history = gthst_rst ) ! (in)
    end do
    call HistoryPut( &
      & 'PsB', xy_PsB, history = gthst_rst ) ! (in)

    call HistoryPut( &
      & 'UN', xyz_UN, history = gthst_rst ) ! (in)
    call HistoryPut( &
      & 'VN', xyz_VN, history = gthst_rst ) ! (in)
    call HistoryPut( &
      & 'TempN', xyz_TempN, history = gthst_rst ) ! (in)
    do n = 1, ncmax
      call HistoryPut( &
        & trim(a_QMixName(n))//'N', xyzf_QMixN(:,:,:,n), history = gthst_rst ) ! (in)
    end do
    call HistoryPut( &
      & 'PsN', xy_PsN, history = gthst_rst ) ! (in)

  end subroutine RestartFileOutput

  !--------------------------------------------------------------------------------------

  subroutine InitialFileOutput(                 &
    & xyz_U, xyz_V, xyz_Temp, xyzf_QMix, xy_Ps  &   ! (in)
    & )
    !
    ! 初期値データの出力を行います. 
    !
    ! Output initial data

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ステップ $ t $ の時刻. 
                              ! Time of step $ t $. 
      & InitialDate           ! 計算開始日時. 
                              ! Start date of calculation

    ! gtool4 データ出力
    ! Gtool4 data output
    !
    use gtool_history, only: HistoryPut, HistorySetTime

    ! 暦と日時の取り扱い
    ! Calendar and Date handler
    !
    use dc_calendar, only: DCCalDateInquire

    ! 組み込み関数 PRESENT の拡張版関数
    ! Extended functions of intrinsic function "PRESENT"
    !
    use dc_present, only: present_and_true

    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(in):: xyz_U    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ .     東西風速. Eastward wind
    real(DP), intent(in):: xyz_V    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ .     南北風速. Northward wind
    real(DP), intent(in):: xyz_Temp (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(in):: xyzf_QMix(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     比湿. Specific humidity
    real(DP), intent(in):: xy_Ps    (0:imax-1, 1:jmax)
                              ! $ p_s $ .   地表面気圧. Surface pressure

    ! 作業変数
    ! Work variables
    !
    character(STRING):: date_str        ! 日時の年月日時分秒形式. 
                                        ! Date with year-month-day hour:minute:second format
    integer          :: n


    ! 実行文 ; Executable statement
    !
    if ( .not. restart_file_opened ) call RestartFileOpen( flag_init_data = .true. )

    ! 初期値出力であることのチェック
    ! Check initial data output
    !
    if ( .not. flag_init_data_save ) then
      call MessageNotify( 'E', module_name, &
        & 'Now, restart data output mode.' )
    end if

    ! 時刻の設定
    ! Set time
    !
    call HistorySetTime( timed = TimeN, history = gthst_rst )

    ! "TimeN" を年月日時分秒表記にして "datetime" 変数へ出力
    ! Put "TimeN" on "datetime" variable with year-month-day hour:minute:second format
    !
    call DCCalDateInquire( &
      & date_str   = date_str, &   ! (out)
      & elapse_sec = TimeN, &      ! (in) optional
      & date       = InitialDate ) ! (in) optional

    call HistoryPut( 'datetime', date_str, &  ! (in)
      &              history = gthst_rst )    ! (inout) optional

    ! データ出力
    ! Data output
    !
    call HistoryPut( &
      & 'U', xyz_U, history = gthst_rst )       ! (in)
    call HistoryPut( &
      & 'V', xyz_V, history = gthst_rst )       ! (in)
    call HistoryPut( &
      & 'Temp', xyz_Temp, history = gthst_rst ) ! (in)
    do n = 1, ncmax
      call HistoryPut( &
        & a_QMixName(n), xyzf_QMix(:,:,:,n), history = gthst_rst ) ! (in)
    end do
    call HistoryPut( &
      & 'Ps', xy_Ps, history = gthst_rst )      ! (in)


  end subroutine InitialFileOutput

  !--------------------------------------------------------------------------------------

  subroutine RestartFileClose
    !
    ! リスタートデータファイル出力の終了処理を行います. 
    !
    ! Terminate restart data files output. 

    ! モジュール引用 ; USE statements
    !

    ! gtool4 データ出力
    ! Gtool4 data output
    !
    use gtool_history, only: HistoryClose

    ! 宣言文 ; Declaration statements
    !
    implicit none

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !
    if ( .not. restart_file_opened ) return

    call HistoryClose( history = gthst_rst ) ! (inout)

    restart_file_opened = .false.
  end subroutine RestartFileClose

  !--------------------------------------------------------------------------------------

  subroutine RestartFileGet( &
    & xyz_UB, xyz_VB, xyz_TempB, xyzf_QMixB, xy_PsB, & ! (out)
    & xyz_UN, xyz_VN, xyz_TempN, xyzf_QMixN, xy_PsN, & ! (out)
    & flag_initial                                   & ! (out) optional
    & )
    !
    ! リスタート/初期値データの入力を行います. 
    ! リスタート/初期値データファイルが存在しない場合には, 
    ! initial_data モジュールを用いて, 初期値生成を行います. 
    !
    ! Input restart/initial data. 
    ! If a restart/initial data file is not exist, 
    ! initial data is created by "initial_data". 


    ! モジュール引用 ; USE statements
    !

    ! 初期値データ (リスタートデータ) 提供
    ! Prepare initial data (restart data)
    !
    use initial_data, only: SetInitData

    ! 時刻管理
    ! Time control
    !
    use timeset, only: TimesetSetInitialDate, TimesetSetTimeN, TimesetSetCalendar, &
      & RestartTime           ! リスタート開始時刻. 
                              ! Restart time of calculation

    ! gtool4 データ入力
    ! Gtool4 data input
    !
    use gtool_history, only: HistoryGet, HistoryGetAttr

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: toChar

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen


    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(out):: xyz_UB      (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u (t-\Delta t) $ .   東西風速. Eastward wind
    real(DP), intent(out):: xyz_VB      (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v (t-\Delta t) $ .   南北風速. Northward wind
    real(DP), intent(out):: xyz_TempB   (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T (t-\Delta t) $ .   温度. Temperature
    real(DP), intent(out):: xyzf_QMixB  (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q (t-\Delta t) $ .   比湿. Specific humidity
    real(DP), intent(out):: xy_PsB      (0:imax-1, 1:jmax)
                              ! $ p_s (t-\Delta t) $ . 地表面気圧. Surface pressure
    real(DP), intent(out):: xyz_UN      (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u (t) $ .     東西風速. Eastward wind
    real(DP), intent(out):: xyz_VN      (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v (t) $ .     南北風速. Northward wind
    real(DP), intent(out):: xyz_TempN   (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T (t) $ .     温度. Temperature
    real(DP), intent(out):: xyzf_QMixN  (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q (t) $ .     比湿. Specific humidity
    real(DP), intent(out):: xy_PsN      (0:imax-1, 1:jmax)
                              ! $ p_s (t) $ .   地表面気圧. Surface pressure
    logical , intent(out), optional:: flag_initial
                              ! リスタートデータを読み込む場合には, 
                              ! .false. が, 初期値データを読み込む場合には
                              ! .true. が返ります. 
                              ! 
                              ! If restart data is loaded, .false. is returned.
                              ! On the other hand, if initial data is loaded, 
                              ! .true. is returned. 

    ! 作業変数
    ! Work variables
    !
    integer:: flag_rst        ! 初期値データから入力時は 0, 
                              ! リスタートデータ入力時は 1. 
                              ! 
                              ! If initial data is input, this value is 0. 
                              ! If restart data is input, this value is 1.

    character(STRING):: time_range
                              ! 時刻の指定. 
                              ! Specification of time
    character(TOKEN):: dummy_str
                              ! 入力チェック用のダミー変数
                              ! Dummy variable for check of input
    logical:: get_err
                              ! 入力時のエラーフラグ. 
                              ! Error flag for input
    real(DP):: returned_time
                              ! 入力データの時刻. 
                              ! Time of input data.
    logical:: flag_time_exist
                              ! 入力データの時刻用のフラグ. 
                              ! Flag for time of input data.

    integer:: origin_year, origin_month, origin_day, origin_hour, origin_min
                              ! 計算開始の年月日時分. 
    real(DP):: origin_sec
                              ! 計算開始の秒. 
    character(TOKEN):: cal_type
                              ! 暦のタイプ. 
    integer:: month_in_year, hour_in_day, min_in_hour
    integer, pointer:: day_in_month_ptr(:) => null()
    real(DP):: sec_in_min
                              ! 暦の詳細情報
    real(DP):: rst_time
                              ! リスタートデータファイルから得られる時刻変数
                              ! Time variable from a restart data file

    logical:: flag_mpi_init

    integer :: n


    ! 実行文 ; Executable statement
    !

    if ( .not. restart_file_io_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! データを initial_data モジュールから取得
    ! Data is input from "initial_data" module
    ! 
    if ( trim(InputFile) == '' ) then
      call SetInitData( &
        & xyz_UB, xyz_VB, xyz_TempB, xyzf_QMixB(:,:,:,IndexH2OVap), xy_PsB )   ! (out)

      ! The variables below are initialized here, temporarily. 
      !
      xyzf_QMixB(:,:,:,1:IndexH2OVap-1)     = 0.0_DP
      xyzf_QMixB(:,:,:,IndexH2OVap+1:ncmax) = 0.0_DP

      call MessageNotify( 'M', module_name, &
        & 'Initial data (not restart data) is input ' // &
        & 'from a module "initial_data". ' // &
        & '*B (t-dt) and *N (t) are same.' )

      xyz_UN     = xyz_UB
      xyz_VN     = xyz_VB
      xyz_TempN  = xyz_TempB
      xyzf_QMixN = xyzf_QMixB
      xy_PsN     = xy_PsB

      if ( present(flag_initial) ) flag_initial = .true.

    ! データを InputFile から取得
    ! Data is input from InputFile
    ! 
    else

!!$      flag_mpi_init = .false.
      flag_mpi_init = .true.

      ! ファイルの有無を確認
      ! Conform an existence of an input file
      ! 
      call HistoryGetAttr( InputFile, 'lon', 'units', & ! (in)
        & dummy_str, &                                  ! (out)
        & flag_mpi_split = flag_mpi_init, &             ! (in) optional
        & err = get_err )                               ! (out)

      if ( get_err ) then
        call MessageNotify( 'E', module_name, &
          & 'restart/initial data file "%c" is not found.', &
          & c1 = trim(InputFile) )
      end if

      ! 時刻情報の取得
      ! Get time information
      !
      time_range = 'time=' // toChar( RestartTime )

      ! 初期値データかリスタートデータかのチェック
      ! Check whether input data is initial data or restart data
      ! 
      call HistoryGet( &
        & InputFile, 'flag_rst', &                ! (in)
        & flag_rst, &                             ! (out)
        & err = get_err, &                        ! (out) optional
        & quiet = .true., &                       ! (in) optional
        & flag_mpi_split = flag_mpi_init )        ! (in) optional

      if ( get_err ) then
        ! flag_rst 変数が読み込めない場合、とりあえず初期値データと見なす
        !
        flag_rst = 0
        if ( present(flag_initial) ) flag_initial = .true.

      else if ( flag_rst == 0 ) then
        if ( present(flag_initial) ) flag_initial = .true.

      else
        if ( present(flag_initial) ) flag_initial = .false.
      end if

      ! 物理量データの有無のチェック ("U", "UB" のみ)
      ! Check existence or nonexistence of physical data (only "U", "UB")
      ! 
      if ( flag_rst == 0 ) then
        call HistoryGet( &
          & InputFile, 'U', &                       ! (in)
          & xyz_UB, &                               ! (out)
          & err = get_err, &                        ! (out) optional
          & quiet = .true., &                       ! (in) optional
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        if ( get_err ) then
          call MessageNotify( 'E', module_name, &
            & 'One of necessary variables "U" for start of calculation ' // &
            & 'cannot be read from "%c"', &
            & c1 = trim(InputFile) )
        end if

      else

        call HistoryGet( &
          & InputFile, 'UB', &                      ! (in)
          & xyz_UB, &                               ! (out)
          & range = time_range, &                   ! (in) optional
          & quiet = .true., &                       ! (in) optional
          & err = get_err, &                        ! (out) optional
          & returned_time = returned_time, &        ! (out) optional
          & flag_time_exist = flag_time_exist, &    ! (out) optional
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        if ( get_err ) then
          call MessageNotify( 'E', module_name, &
            & 'One of necessary variables "UB" for restart ' // &
            & 'cannot be read from "%c"', &
            & c1 = trim(InputFile) )
        end if
      end if

      ! 初期値データを入力する場合、経過時間を 0 に設定
      ! Set 0 to elapsed time, when initial data is input
      !
      if ( flag_rst == 0 ) then
        call TimesetSetTimeN( 0.0_DP )
      end if

      ! 暦、開始日時、経過時刻の読み込みと上書き設定
      ! Input and overwrite calendar, start date, and elapsed time
      !
      if ( flag_rst /= 0 ) then

        ! 暦の読み込み
        ! Input calendar
        !
        call HistoryGetAttr( &
          & InputFile, 'time', 'calendar', &          ! (in)
          & cal_type, &                               ! (out)
          & flag_mpi_split = flag_mpi_init )          ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'month_in_year', &     ! (in)
          & month_in_year, &                          ! (out)
          & flag_mpi_split = flag_mpi_init )          ! (in) optional

        allocate( day_in_month_ptr(month_in_year) )
        call HistoryGetAttr( &
          & InputFile, 'time', 'day_in_month', &      ! (in)
          & day_in_month_ptr, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )          ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'hour_in_day', &       ! (in)
          & hour_in_day, &                            ! (out)
          & flag_mpi_split = flag_mpi_init )          ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'min_in_hour', &       ! (in)
          & min_in_hour, &                            ! (out)
          & flag_mpi_split = flag_mpi_init )          ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'sec_in_min', &        ! (in)
          & sec_in_min, &                             ! (out)
          & flag_mpi_split = flag_mpi_init )          ! (in) optional

        ! 暦の上書き設定
        ! Overwrite calendar
        !
        call TimesetSetCalendar( cal_type, &       ! (in)
          & month_in_year, day_in_month_ptr, &     ! (in)
          & hour_in_day, min_in_hour, sec_in_min ) ! (in)

        deallocate( day_in_month_ptr )

        ! 開始日時の読み込み
        ! Input start date
        !
        call HistoryGetAttr( &
          & InputFile, 'time', 'origin_year', &     ! (in)
          & origin_year, &                          ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'origin_month', &    ! (in)
          & origin_month, &                         ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'origin_day', &      ! (in)
          & origin_day, &                           ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'origin_hour', &     ! (in)
          & origin_hour, &                          ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'origin_min', &      ! (in)
          & origin_min, &                           ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        call HistoryGetAttr( &
          & InputFile, 'time', 'origin_sec', &      ! (in)
          & origin_sec, &                           ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional

        ! 開始日時の上書き設定
        ! Overwrite start date
        !
        call TimesetSetInitialDate( &
          & origin_year, origin_month, origin_day, & ! (in)
          & origin_hour, origin_min,   origin_sec )  ! (in)

        ! 経過時間の読み込み
        ! Input elapsed time
        !
        call HistoryGet( &
          & InputFile, 'time', range = time_range, &  ! (in)
          & array = rst_time, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )          ! (in) optional

        call MessageNotify( 'M', module_name, &
          & '  Specified RestartTime from a NAMELIST file            = <%f>, ', &
          & d = (/ RestartTime /) )
        call MessageNotify( 'M', module_name, &
          & '  Actual initial value of TimeN  from a restart file  = <%f>', &
          & d = (/ rst_time /) )

        ! 経過時間の上書き設定
        ! Overwrite elapsed time
        !
        call TimesetSetTimeN( rst_time )

      end if

      ! データ入力
      ! Data input
      ! 
      if ( flag_rst == 0 ) then
        call HistoryGet( &
          & InputFile, 'U', &                 ! (in)
          & array = xyz_UB, &                 ! (out)
          & flag_mpi_split = flag_mpi_init )  ! (in) optional
        call HistoryGet( &
          & InputFile, 'V', &                 ! (in)
          & array = xyz_VB, &                 ! (out)
          & flag_mpi_split = flag_mpi_init )  ! (in) optional
        call HistoryGet( &
          & InputFile, 'Temp', &              ! (in)
          & array = xyz_TempB, &              ! (out)
          & flag_mpi_split = flag_mpi_init )  ! (in) optional
        do n = 1, ncmax
          call HistoryGet(                           &
            & InputFile, trim(a_QMixName(n)),        &              ! (in)
            & array = xyzf_QMixB(:,:,:,n),           &              ! (out)
            & flag_mpi_split = flag_mpi_init         &              ! (in) optional
            & )
        end do
        call HistoryGet( &
          & InputFile, 'Ps', &                ! (in)
          & array = xy_PsB, &                 ! (out)
          & flag_mpi_split = flag_mpi_init )  ! (in) optional

        call MessageNotify( 'M', module_name, &
          & 'Initial data (not restart data) is input ' // &
          & 'from a data file "%c". ' // &
          & '*B (t-dt) and *N (t) are same.', &
          & c1 = trim(InputFile) )

        xyz_UN     = xyz_UB
        xyz_VN     = xyz_VB
        xyz_TempN  = xyz_TempB
        xyzf_QMixN = xyzf_QMixB
        xy_PsN     = xy_PsB

      else
        call HistoryGet( &
          & InputFile, 'UB', range = time_range, &  ! (in)
          & array = xyz_UB, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional
        call HistoryGet( &
          & InputFile, 'VB', range = time_range, &  ! (in)
          & array = xyz_VB, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional
        call HistoryGet( &
          & InputFile, 'TempB', range = time_range, &  ! (in)
          & array = xyz_TempB, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )           ! (in) optional
        do n = 1, ncmax
          call HistoryGet(                                             &
            & InputFile, trim(a_QMixName(n))//'B', range = time_range, & ! (in)
            & array = xyzf_QMixB(:,:,:,n),                             & ! (out)
            & flag_mpi_split = flag_mpi_init                           & ! (in) optional
            & )
        end do
        call HistoryGet( &
          & InputFile, 'PsB', range = time_range, &  ! (in)
          & array = xy_PsB, &                        ! (out)
          & flag_mpi_split = flag_mpi_init )         ! (in) optional

        call HistoryGet( &
          & InputFile, 'UN', range = time_range, &  ! (in)
          & array = xyz_UN, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional
        call HistoryGet( &
          & InputFile, 'VN', range = time_range, &  ! (in)
          & array = xyz_VN, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )        ! (in) optional
        call HistoryGet( &
          & InputFile, 'TempN', range = time_range, &  ! (in)
          & array = xyz_TempN, &                       ! (out)
          & flag_mpi_split = flag_mpi_init )           ! (in) optional
        do n = 1, ncmax
          call HistoryGet(                                             &
            & InputFile, trim(a_QMixName(n))//'N', range = time_range, & ! (in)
            & array = xyzf_QMixN(:,:,:,n),                             & ! (out)
            & flag_mpi_split = flag_mpi_init                           & ! (in) optional
            & )
        end do
        call HistoryGet( &
          & InputFile, 'PsN', range = time_range, &  ! (in)
          & array = xy_PsN, &                        ! (out)
          & flag_mpi_split = flag_mpi_init )         ! (in) optional

        call MessageNotify( 'M', module_name, &
          & 'Restart data (not initial data) is input ' // &
          & 'from a data file "%c". ', &
          & c1 = trim(InputFile) )
      end if

    end if

  end subroutine RestartFileGet

  !--------------------------------------------------------------------------------------

  subroutine RestartFileInit
    !
    ! restart_file_io モジュールの初期化を行います. 
    ! NAMELIST#restart_file_io_nml の読み込みはこの手続きで行われます. 
    !
    ! "restart_file_io" module is initialized. 
    ! "NAMELIST#restart_file_io_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & RestartTime           ! リスタート開始時刻. 
                              ! Restart time of calculation

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! 暦と日時の取り扱い
    ! Calendar and Date handler
    !
    use dc_calendar, only: DCCalConvertByUnit

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 初期値データ (リスタートデータ) 提供
    ! Prepare initial data (restart data)
    !
    use initial_data, only: InitDataInit


    ! 宣言文 ; Declaration statements
    !
    implicit none

    ! 作業変数
    ! Work variables
    !
    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /restart_file_io_nml/ &
      & InputFile, &
      & OutputFile, &
      & IntValue, IntUnit
          !
          ! デフォルト値については初期化手続 "restart_file_io#RestartFileInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "restart_file_io#RestartFileInit" for the default values. 
          !


    ! 実行文 ; Executable statement
    !

    if ( restart_file_io_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    InputFile  = ''
    if ( .not. flag_init_data_save ) then
      OutputFile = 'rst.nc'
    else
      OutputFile = 'init.nc'
    end if
    IntValue   = 365.0d0
    IntUnit    = 'day'

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &         ! (in)
        & nml = restart_file_io_nml, &  ! (out)
        & iostat = iostat_nml ) ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
      if ( iostat_nml == 0 ) write( STDOUT, nml = restart_file_io_nml )
    end if

    ! 出力時間間隔の設定
    ! Configure time interval of output
    !
    IntTime = DCCalConvertByUnit( IntValue, IntUnit, 'sec' ) ! (in)
    PrevOutputTime = RestartTime

    ! フラグの初期化
    ! Initialize flag
    !
    flag_output_end = .false.


    ! Initialization of modules used in this module

    ! 初期値データ (リスタートデータ) 提供
    ! Prepare initial data (restart data)
    !
    call InitDataInit


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'Input:: ' )
    if ( trim(InputFile) /= '' ) then
      call MessageNotify( 'M', module_name, '  InputFile  = %c', c1 = trim(InputFile) )
    else
      call MessageNotify( 'M', module_name, '  InputFile  = <Non>' )
      call MessageNotify( 'M', module_name, '  ** Initial data is generated in "initial_data" module' )
    end if
    call MessageNotify( 'M', module_name, 'Output:: ' )
    call MessageNotify( 'M', module_name, '  OutputFile = %c', c1 = trim(OutputFile) )
    call MessageNotify( 'M', module_name, '  IntTime    = %f [%c]', d = (/ IntValue /), c1 = trim(IntUnit) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    restart_file_io_inited = .true.

  end subroutine RestartFileInit

  !--------------------------------------------------------------------------------------

end module restart_file_io
