!= 金星放射モデル Ver. 1 (汎惑星大気放射モデル, 金星 Ver. 1)
!
!= Venus radiation model Ver. 1 (general-purpose radiation model for planetary atmosphere, Venus Ver. 1)
!
module rad_Venus_V1
  !
  != 金星放射モデル Ver. 1 (汎惑星大気放射モデル, 金星 Ver. 1)
  !
  != Venus radiation model Ver. 1 (general-purpose radiation model for planetary atmosphere, Venus Ver. 1)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! This is a general-purpose radiation model for planetary atmospheres
  !
  !== References
  !
  !
  !== Procedures List
  !
  ! RadEarthV2Flux :: 放射フラックスの計算
  ! ------------          :: ------------
  ! RadEarthV2Flux :: Calculate radiation flux
  !
  !== NAMELIST
  !
  ! NAMELIST#rad_Earth_V2_nml
  !

  ! USE statements
  !

  !
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  !
  ! Physical constants settings
  !
  use constants, only: Grav     ! $ g $ [m s-2].
                                !
                                ! Gravitational acceleration

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数.
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数.
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数.
                             ! Number of vertical level

  implicit none

  private


  integer                       , save :: NMolNum
  integer                       , save :: NPtcl
  integer          , allocatable, save :: m_MolNum(:)
!!$  character(STRING), allocatable, save :: a_PtclName(:)
  character(64)    , allocatable, save :: a_PtclName(:)
  integer                       , save :: PrescNPress
  integer                       , save :: PrescNPressM
  real(DP)         , allocatable, save :: r_PrescPress      (:)
  real(DP)         , allocatable, save :: rm_PrescVMR       (:,:)
  real(DP)         , allocatable, save :: z_PrescPress      (:)
  real(DP)         , allocatable, save :: a_PrescPtclDens   (:)
  real(DP)         , allocatable, save :: za_PrescPtclEffRad(:,:)
  real(DP)         , allocatable, save :: za_PrescPtclMMR   (:,:)

  integer                       , save :: iLoop
  integer                       , save :: NCalcInterval

  logical                       , save :: FlagDiurnalMean
  integer                       , save :: NSZA
  real(DP)         , allocatable, save :: a_DiurnalMeanHA(:)
  real(DP)         , allocatable, save :: a_SZAGW        (:)
  real(DP)         , allocatable, save :: xya_SZADeg     (:,:,:)


  ! 公開変数
  ! Public variables
  !
  logical, save :: rad_Venus_V1_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  public :: RadVenusV1Flux
  public :: RadVenusV1Init
  public :: RadVenusV1Finalize

  character(*), parameter:: module_name = 'rad_Venus_V1'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: rad_Earth_V2.f90,v 1.9 2015/01/29 12:06:43 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine RadVenusV1Flux(                       &
    & xy_SurfAlbedo,                               & ! (in)
    & xyz_Press, xyr_Press, xyz_Temp,              & ! (in)
    & xy_SurfTemp,                                 & ! (in)
    & xyr_RadSUwFlux, xyr_RadSDwFlux,              & ! (inout)
    & xyr_RadLUwFlux, xyr_RadLDwFlux,              & ! (inout)
    & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux       & ! (inout)
    & )


    ! USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimesetClockStart, TimesetClockStop

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & y_Lat

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only : &
      & MolWtDry

    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    use set_solarconst, only : SetSolarConst

    use rad_flux_wrapper, only : RadFluxWrapper

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncome


    real(DP), intent(in   ) :: xy_SurfAlbedo   (0:imax-1, 1:jmax)
    real(DP), intent(in   ) :: xyz_Press       (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyr_Press       (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(in   ) :: xyz_Temp        (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xy_SurfTemp     (0:imax-1, 1:jmax)
    real(DP), intent(inout) :: xyr_RadSUwFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadSDwFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadLUwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadLDwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyra_DelRadLUwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
    real(DP), intent(inout) :: xyra_DelRadLDwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)


    ! Work variables
    !
    real(DP) :: xyz_DelAtmMass   (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyr_Temp      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyrm_VMR      (0:imax-1, 1:jmax, 0:kmax, NMolNum)

    real(DP) :: StrFluxTOA0
    real(DP) :: StrFluxTOA
    real(DP) :: xy_CosSZA     (0:imax-1, 1:jmax)
    real(DP) :: FluxFactor

    real(DP) :: xy_AlbedoSW   (0:imax-1, 1:jmax)
    real(DP) :: xy_AlbedoLW   (0:imax-1, 1:jmax)
    real(DP) :: xy_EmisSW     (0:imax-1, 1:jmax)
    real(DP) :: xy_EmisLW     (0:imax-1, 1:jmax)
    real(DP) :: xy_SurfEmis   (0:imax-1, 1:jmax)

    real(DP) :: xyr_RadUwFlux      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFlux      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyra_DelRadUwFlux  (0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFlux  (0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyr_RadUwFluxSW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFluxSW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadUwFluxLW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFluxLW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyra_DelRadUwFluxSW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFluxSW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadUwFluxLW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFluxLW(0:imax-1, 1:jmax, 0:kmax, -1:1)

    real(DP)          :: xyza_PtclDens   (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_PtclEffRad (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_PtclMMR    (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_DelPtclMass(0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyr_MeanMolWt   (0:imax-1, 1:jmax, 0:kmax)

    real(DP) :: DistFromStarScld
    real(DP) :: SinDel

    integer  :: j
    integer  :: k
    integer  :: l
    integer  :: m
    integer  :: n


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_Venus_V1_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    if ( ( iLoop > 0 ) .and. ( mod( iLoop, NCalcInterval ) /= 0 ) ) then
      iLoop = iLoop + 1
      return
    end if
    iLoop = iLoop + 1


    k = 0
    xyr_Temp(:,:,k) =                                    &
      &      ( xyz_Temp (:,:,k+2) - xyz_Temp (:,:,k+1) ) &
      & / log( xyz_Press(:,:,k+2) / xyz_Press(:,:,k+1) ) &
      & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k+1) ) &
      & + xyz_Temp(:,:,k+1)
    do k = 1, kmax-1
      xyr_Temp(:,:,k) =                                &
        &      ( xyz_Temp (:,:,k+1) - xyz_Temp (:,:,k) ) &
        & / log( xyz_Press(:,:,k+1) / xyz_Press(:,:,k) ) &
        & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k) ) &
        & + xyz_Temp(:,:,k)
    end do
    k = kmax
    xyr_Temp(:,:,k) = xyz_Temp(:,:,k)



!!$    xy_AlbedoSW = xy_SurfAlbedo
!!$    xy_AlbedoLW = 0.0_DP

    xy_SurfEmis = 1.0_DP
    !
    xy_AlbedoSW = xy_SurfAlbedo
    xy_AlbedoLW = 1.0_DP - xy_SurfEmis
    xy_EmisSW   = 1.0_DP - xy_SurfAlbedo
    xy_EmisLW   = xy_SurfEmis


    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    call SetSolarConst( &
      & StrFluxTOA0 & ! (out)
      & )

    ! 短波入射の計算
    ! Calculate short wave (insolation) incoming radiation
    !
    call RadShortIncome(                    &
      & xy_CosSZA, FluxFactor,                 & ! (out) optional
      & DistFromStarScld  = DistFromStarScld,  & ! (out) optional
      & SinDel            = SinDel             & ! (out) optional
      & )
    !
    if ( FlagDiurnalMean ) then
      do n = 1, NSZA
        do j = 1, jmax
          xy_CosSZA(:,j) = &
            &  + sin( y_Lat(j) ) * SinDel &
            &  + cos( y_Lat(j) ) * sqrt( 1.0_DP - SinDel**2 ) * cos( a_DiurnalMeanHA(n) )
        end do
        xya_SZADeg(:,:,n) = acos( xy_CosSZA ) * 180.0_DP / PI
      end do
    else
      xya_SZADeg(:,:,1) = acos( xy_CosSZA ) * 180.0_DP / PI
    end if


    StrFluxTOA = StrFluxTOA0 / ( DistFromStarScld * DistFromStarScld ) * FluxFactor

    xyr_MeanMolWt = MolWtDry

    call RadVenusV1Interpolate( &
      & NMolNum, PrescNPress, r_PrescPress, rm_PrescVMR, &
      & imax, jmax, kmax+1, xyr_Press, &
      & xyrm_VMR &
      & )
    do m = 1, NPtcl
      xyza_PtclDens(:,:,:,m) = a_PrescPtclDens(m)
    end do
    call RadVenusV1Interpolate( &
      & NPtcl, PrescNPressM, z_PrescPress, za_PrescPtclEffRad, &
      & imax, jmax, kmax, xyz_Press, &
      & xyza_PtclEffRad &
      & )
    call RadVenusV1Interpolate( &
      & NPtcl, PrescNPressM, z_PrescPress, za_PrescPtclMMR, &
      & imax, jmax, kmax, xyz_Press, &
      & xyza_PtclMMR &
      & )
    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do
    do m = 1, NPtcl
      xyza_DelPtclMass(:,:,:,m) = &
        &   xyz_DelAtmMass * xyza_PtclMMR(:,:,:,m)
    end do

    call RadFluxWrapper(                      &
      & imax, jmax, kmax,                           & ! (in)
      & NMolNum, m_MolNum,                          & ! (in)
      & xy_AlbedoSW, xy_AlbedoLW,                   & ! (in)
      & xy_EmisSW, xy_EmisLW,                       & ! (in)
      & StrFluxTOA,                                 & ! (in)
      & NSZA, xya_SZADeg, a_SZAGW,                  & ! (in)
      & xyr_Press, xyr_Temp, xy_SurfTemp,           & ! (in)
      & xyrm_VMR, xyr_MeanMolWt,                    & ! (in)
      & NPtcl, a_PtclName,                          & ! (in)
      & xyza_PtclDens,                              & ! (in)
      & xyza_PtclEffRad, xyza_DelPtclMass,          & ! (in)
      & xyr_RadUwFlux, xyr_RadDwFlux,               & ! (out)
      & xyra_DelRadUwFlux, xyra_DelRadDwFlux,       & ! (out)
      & xyr_RadUwFluxLW, xyr_RadDwFluxLW,           & ! (out)
      & xyr_RadUwFluxSW, xyr_RadDwFluxSW,           & ! (out)
      & xyra_DelRadUwFluxLW, xyra_DelRadDwFluxLW,   & ! (out)
      & xyra_DelRadUwFluxSW, xyra_DelRadDwFluxSW    & ! (out)
      & )

    xyr_RadLUwFlux     = xyr_RadUwFluxLW
    xyr_RadLDwFlux     = xyr_RadDwFluxLW
    xyr_RadSUwFlux     = xyr_RadUwFluxSW
    xyr_RadSDwFlux     = xyr_RadDwFluxSW

    xyra_DelRadLUwFlux(:,:,:,0) = xyra_DelRadUwFluxLW(:,:,:,-1)
    xyra_DelRadLDwFlux(:,:,:,0) = xyra_DelRadDwFluxLW(:,:,:,-1)
    !
!!$    xyra_DelRadLUwFlux(:,:,:,1) = 0.0_DP
!!$    xyra_DelRadLDwFlux(:,:,:,1) = 0.0_DP
    do l = 1, 1
      do k = 0, kmax
        xyra_DelRadLUwFlux(:,:,k,l) = &
          &   ( xyra_DelRadUwFluxLW(:,:,k,l) - xyra_DelRadUwFluxLW(:,:,k,l-1) ) &
          & / log( xyr_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & * log( xyz_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & + xyra_DelRadUwFluxLW(:,:,k,l-1)
        xyra_DelRadLDwFlux(:,:,k,l) = &
          &   ( xyra_DelRadDwFluxLW(:,:,k,l) - xyra_DelRadDwFluxLW(:,:,k,l-1) ) &
          & / log( xyr_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & * log( xyz_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & + xyra_DelRadDwFluxLW(:,:,k,l-1)
      end do
    end do


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine RadVenusV1Flux

  !--------------------------------------------------------------------------------------

  subroutine RadVenusV1Interpolate( &
    & NParam, InNLev, z_InPress, za_InValue, &
    & imax, jmax, OutNLev, xyz_OutPress, &
    & xyza_OutValue &
    & )

    integer , intent(in ) :: NParam
    integer , intent(in ) :: InNLev
    real(DP), intent(in ) :: z_InPress(InNLev)
    real(DP), intent(in ) :: za_InValue(InNLev,NParam)
    integer , intent(in ) :: imax
    integer , intent(in ) :: jmax
    integer , intent(in ) :: OutNLev
    real(DP), intent(in ) :: xyz_OutPress(imax,jmax,OutNLev)
    real(DP), intent(out) :: xyza_OutValue(imax,jmax,OutNLev,NParam)


    ! Local variables
    !
    real(DP) :: z_LogInPress(InNLev)
    real(DP) :: za_LogInValue(InNLev,NParam)
    real(DP) :: xyz_LogOutPress(imax,jmax,OutNLev)

    integer :: i
    integer :: j
    integer :: k
    integer :: kk
    integer :: m


    do k = 1, InNLev
      if ( z_InPress (k) > 0.0_DP ) then
        z_LogInPress (k) = log( z_InPress (k) )
      else
        z_LogInPress (k) = -100.0_DP
      end if
    end do
    do m = 1, NParam
      do k = 1, InNLev
        if ( za_InValue(k,m) > 0.0_DP ) then
          za_LogInValue(k,m) = log( za_InValue(k,m) )
        else
          za_LogInValue(k,m) = -100.0_DP
        end if
      end do
    end do
    do k = 1, OutNLev
      do j = 1, jmax
        do i = 1, imax
          if ( xyz_OutPress(i,j,k) > 0.0_DP ) then
            xyz_LogOutPress(i,j,k) = log( xyz_OutPress(i,j,k) )
          else
            xyz_LogOutPress(i,j,k) = -100.0_DP
          end if
        end do
      end do
    end do


    do j = 1, jmax
      do i = 1, imax

        do m = 1, NParam
          do k = 1, OutNLev

            if ( z_InPress(1) <= xyz_OutPress(i,j,k) ) then
              xyza_OutValue(i,j,k,m) = za_LogInValue(1,m)
            else if ( xyz_OutPress(i,j,k) <= z_InPress(InNLev) ) then
              xyza_OutValue(i,j,k,m) = za_LogInValue(InNLev,m)
            else
              do kk = 1+1, InNLev
                if ( z_InPress(kk) < xyz_OutPress(i,j,k) ) then
                  exit
                end if
              end do
              xyza_OutValue(i,j,k,m) = &
                &   ( za_LogInValue(kk,m)    - za_LogInValue(kk-1,m) ) &
                & / ( z_LogInPress(kk)       - z_LogInPress(kk-1)    ) &
                & * ( xyz_LogOutPress(i,j,k) - z_LogInPress(kk-1)    ) &
                & + za_LogInValue(kk-1,m)
            end if

          end do
        end do

      end do
    end do

    xyza_OutValue = exp( xyza_OutValue )


  end subroutine RadVenusV1Interpolate

  !--------------------------------------------------------------------------------------

  subroutine RadVenusV1Init

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI

    !
    ! Physical constants settings
    !
    use constants, only: Grav     ! $ g $ [m s-2].
                                  !
                                  ! Gravitational acceleration

    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    use set_solarconst, only : SetSolarConstInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncomeInit

    use rad_flux_wrapper, only : RadFluxWrapperInit

    ! ガウス重み, 分点の計算
    ! Calculate Gauss node and Gaussian weight
    !
    use gauss_quad, only : GauLeg

    ! gtool データ入力
    ! Gtool data input
    !
    use gtool_history, only: HistoryGet

    use io_profile, only : &
      & IOProfileInqDimLen, &
      & IOProfileRead, &
      & IOProfileReadPtcl


    ! 宣言文 ; Declaration statements
    !


    ! Local variables
    !
    real(DP)          :: WaveNumAlbedoSwitch
    character(STRING) :: KDOptPropNcFN
    character(STRING) :: ProfileNcFN

    real(DP), allocatable :: r_Temp(:)
    real(DP)              :: SurfTemp

    integer , allocatable :: a_PtclNum (:)

    real(DP), allocatable :: a_TmpArray(:)

    integer :: NGQDiurnalMean

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read

    integer :: k
    integer :: m


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /rad_Venus_V1_nml/ &
      & WaveNumAlbedoSwitch, &
      & KDOptPropNcFN, &
      & ProfileNcFN, &
      & NCalcInterval, &
      & FlagDiurnalMean, &
      & NGQDiurnalMean
!!$      & CloudIceREffMethod,     &
!!$      & CloudWatREff,           &
!!$      & CloudIceREff

!!$      & SWVer, LWVer
          !
          ! デフォルト値については初期化手続 "rad_Earth_V2#RadEarthV2Init"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "rad_Earth_V2#RadEarthV2Init" for the default values.
          !

    if ( rad_Venus_V1_inited ) return



    ! デフォルト値の設定
    ! Default values settings
    !
    KDOptPropNcFN = ""
    ProfileNcFN   = ""

    NCalcInterval = 1

    FlagDiurnalMean = .false.
    NGQDiurnalMean  = 0

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = rad_Venus_V1_nml,         & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    iLoop = 0


!!$    ! Identification of calculation method of cloud particle effective radius
!!$    !
!!$    call MessageNotify( 'M', module_name, &
!!$      & 'CloudIceREffMethod=<%c>.', &
!!$      & c1 = trim(CloudIceREffMethod) )
!!$    !
!!$    select case ( CloudIceREffMethod )
!!$    case ( 'Const' )
!!$      IDCloudIceREffMethod = IDCloudIceREffMethodConst
!!$    case ( 'Lin' )
!!$      IDCloudIceREffMethod = IDCloudIceREffMethodLin
!!$    case default
!!$      call MessageNotify( 'E', module_name, &
!!$        & 'CloudIceREffMethod=<%c> is not supported.', &
!!$        & c1 = trim(CloudIceREffMethod) )
!!$    end select



    call IOProfileInqDimLen(      &
      & ProfileNcFN, 'Press', &
      & PrescNPress &
      & )
    call IOProfileInqDimLen(      &
      & ProfileNcFN, 'PressM', &
      & PrescNPressM &
      & )
    call IOProfileInqDimLen(      &
      & ProfileNcFN, 'MolNum', &
      & NMolNum &
      & )
    call IOProfileInqDimLen(      &
      & ProfileNcFN, 'PtclNum', &
      & NPtcl &
      & )

    allocate( m_MolNum          (NMolNum)                  )
    allocate( a_PtclName        (NPtcl)                    )
    allocate( r_PrescPress      (1:PrescNPress )           )
    allocate( rm_PrescVMR       (1:PrescNPress ,1:NMolNum) )
    allocate( z_PrescPress      (1:PrescNPressM)           )
    allocate( a_PrescPtclDens   (NPtcl) )
    allocate( za_PrescPtclEffRad(1:PrescNPressM,1:NPtcl)   )
    allocate( za_PrescPtclMMR   (1:PrescNPressM,1:NPtcl)   )

    allocate( r_Temp(PrescNPress) )
    allocate( a_PtclNum (NPtcl) )

    call IOProfileRead(      &
      & ProfileNcFN, 'Press', 'MolNum', 'Temp', 'Ts', 'VMR', &
      & PrescNPress, NMolNum, &
      & r_PrescPress, m_MolNum, r_Temp, SurfTemp, rm_PrescVMR     &
      & )
    call IOProfileReadPtcl(      &
      & ProfileNcFN, &
      & PrescNPressM, NPtcl, &
      & z_PrescPress, a_PtclNum, a_PtclName, a_PrescPtclDens, &
      & za_PrescPtclEffRad, za_PrescPtclMMR  &
      & )

    deallocate( r_Temp     )
    deallocate( a_PtclNum  )


    if ( r_PrescPress(1) < r_PrescPress(2) ) then
      allocate( a_TmpArray( PrescNPress ) )
      !
      a_TmpArray = r_PrescPress
      do k = 1, PrescNPress
        r_PrescPress(k) = a_TmpArray(PrescNPress-k+1)
      end do
      !
      do m = 1, NMolNum
        a_TmpArray = rm_PrescVMR(:,m)
        do k = 1, PrescNPress
          rm_PrescVMR(k,m) = a_TmpArray(PrescNPress-k+1)
        end do
      end do
      !
      deallocate( a_TmpArray )

      allocate( a_TmpArray( PrescNPressM ) )
      !
      do m = 1, NPtcl
        a_TmpArray = za_PrescPtclEffRad(:,m)
        do k = 1, PrescNPressM
          za_PrescPtclEffRad(k,m) = a_TmpArray(PrescNPressM-k+1)
        end do
        !
        a_TmpArray = za_PrescPtclMMR(:,m)
        do k = 1, PrescNPressM
          za_PrescPtclMMR(k,m) = a_TmpArray(PrescNPressM-k+1)
        end do
      end do
      !
      deallocate( a_TmpArray )
    end if


    if ( FlagDiurnalMean ) then
      if ( NGQDiurnalMean < 1 ) then
        write( 6, * ) 'NGQDiurnalMean has to be greater than 0, when FlagDiurnalMean is .true.: ', &
          & FlagDiurnalMean, NGQDiurnalMean
        stop
      end if
      NSZA = NGQDiurnalMean
      allocate( a_DiurnalMeanHA( NSZA ) )
      allocate( a_SZAGW        ( NSZA ) )
      call GauLeg( 0.0_DP, PI, NSZA, a_DiurnalMeanHA, a_SZAGW )
      a_SZAGW = a_SZAGW / PI
    else
      NSZA = 1
      allocate( a_SZAGW( NSZA ) )
      a_SZAGW = 1.0_DP
    end if
    allocate( xya_SZADeg(0:imax-1, 1:jmax, NSZA) )


    ! Initialization of modules used in this module
    !
    call RadFluxWrapperInit( &
      & imax, jmax,                              &
      & Grav, WaveNumAlbedoSwitch, KDOptPropNcFN &
      & )


    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    call SetSolarConstInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    call RadShortIncomeInit

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
!!$    call SetGCMRInit

    ! O3 分布の設定
    ! Set O3 distribution
    !
!!$    call SetO3Init



    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
!!$    call HistoryAutoAddVariable( 'CloudCoverforRad', &
!!$      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
!!$      & 'cloud cover', '1' )
!!$    call HistoryAutoAddVariable( 'CloudCoverRand', &
!!$      & (/ 'lon ', 'lat ', 'time' /), &
!!$      & 'cloud cover', '1' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  WaveNumAlbedoSwitch = %f', &
      & d = (/ WaveNumAlbedoSwitch /) )
    call MessageNotify( 'M', module_name, '  KDOptPropNcFN       = %c', &
      & c1 = trim(KDOptPropNcFN) )
    call MessageNotify( 'M', module_name, '  ProfileNcFN         = %c', &
      & c1 = trim(ProfileNcFN) )
    call MessageNotify( 'M', module_name, '  NCalcInterval       = %d', &
      & i = (/ NCalcInterval /) )
    call MessageNotify( 'M', module_name, '  FlagDiurnalMean     = %b', &
      & l = (/ FlagDiurnalMean /) )
    call MessageNotify( 'M', module_name, '  NGQDiurnalMean      = %d', &
      & i = (/ NGQDiurnalMean /) )
!!$    call MessageNotify( 'M', module_name, '  CloudWatREff      = %f', &
!!$      & d = (/ CloudWatREff /) )
!!$    call MessageNotify( 'M', module_name, '  CloudIceREff      = %f', &
!!$      & d = (/ CloudIceREff /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    rad_Venus_V1_inited = .true.

  end subroutine RadVenusV1Init

  !--------------------------------------------------------------------------------------

  subroutine RadVenusV1Finalize


    use rad_flux_wrapper, only : RadFluxWrapperFinalize



    ! 宣言文 ; Declaration statements
    !


    ! Local variables
    !


    if ( rad_Venus_V1_inited ) return


    call RadFluxWrapperFinalize


    deallocate( m_MolNum           )
    deallocate( a_PtclName         )
    deallocate( r_PrescPress       )
    deallocate( rm_PrescVMR        )
    deallocate( a_PrescPtclDens    )
    deallocate( za_PrescPtclEffRad )
    deallocate( za_PrescPtclMMR    )

    if ( FlagDiurnalMean ) then
      deallocate( a_DiurnalMeanHA )
      deallocate( a_SZAGW         )
    end if
    deallocate( xya_SZADeg )

    rad_Venus_V1_inited = .false.

  end subroutine RadVenusV1Finalize

  !--------------------------------------------------------------------------------------

end module rad_Venus_V1
