!= Dennou AGCM の式を用いた飽和比湿の算出
!
!= Evaluate saturation specific humidity with method used by Dennou AGCM
!
! Authors::   Yasuhiro MORIKAWA, Yoshiyuki O. Takahashi
! Version::   $Id: saturate_DennouAGCM.f90,v 1.6 2015/01/29 12:07:16 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module saturate_DennouAGCM
  !
  != Dennou AGCM の式を用いた飽和比湿の算出
  !
  != Evaluate saturation specific humidity with method used by Dennou AGCM
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! Dennou AGCM で用いられていた式を用いて飽和蒸気圧を計算します. 
  ! この式は, 元々 沼口 (1992) に依っています.
  !
  ! Saturation specific humidity and temperature derivative of it 
  ! are calculated with a formula used in Dennou AGCM. 
  ! This formula is originaly obtained from Numaguti (1992).
  !
  !== References
  !
  ! * Numaguti, 1992: 
  !   <title in Japanese>, 
  !   Doctor thesis, University of Tokyo.
  !
  !== Procedures List
  !
  ! CalcQVapSat            :: 飽和比湿の計算
  ! CalcDQVapSatDTemp      :: 飽和比湿の温度微分の計算
  ! ------------  :: ------------
  ! CalcQVapSat            :: Calculate saturation specific humidity
  ! CalcDQVapSatDTemp      :: Calculate temperature derivative of saturation specific humidity
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#saturate_t1930_nml
  !++

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private


  ! 公開手続き
  ! Public procedure
  !
  public:: xyz_CalcQVapSatOnLiq
  public:: xyz_CalcDQVapSatDTempOnLiq
  public:: xyz_CalcQVapSatOnSol
  public:: xyz_CalcDQVapSatDTempOnSol
  public:: SaturateInit

  ! 公開変数
  ! Public variables
  !
  logical, save, public:: saturate_DennouAGCM_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  ! 非公開変数
  ! Private variables
  !
  real(DP), parameter:: Es0 = 611.0_DP
                              ! 0 ℃の飽和蒸気圧 [Pa]. 
                              ! Saturation water vapor pressure at 0 deg C [Pa]

  character(*), parameter:: module_name = 'saturate_DennouAGCM'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: saturate_DennouAGCM.f90,v 1.6 2015/01/29 12:07:16 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  function xyz_CalcQVapSatOnLiq( xyz_Temp, xyz_Press ) result( xyz_QVapSat )
    !
    ! 温度 *Temp* と気圧 *Press* を用い, 
    ! 飽和比湿 *QVapSat* を求めます. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRWet, &
                              ! $ R_v $ [J kg-1 K-1]. 
                              ! 凝結成分の気体定数. 
                              ! Gas constant of condensible elements
      & LatentHeat, &
                              ! $ L $ [J kg-1] . 
                              ! 凝結の潜熱. 
                              ! Latent heat of condensation
      & LatentHeatFusion, &
                              ! $ L $ [J kg-1] .
                              ! 融解の潜熱.
                              ! Latent heat of fusion
      & EpsV                  ! $ \epsilon_v $ . 
                              ! 水蒸気分子量比. 
                              ! Molecular weight of water vapor

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_Press(:,:,:)
                              ! $ p $ . 気圧. Air pressure

    real(DP):: xyz_QVapSat(1:size(xyz_Temp,1), 1:size(xyz_Temp,2), 1:size(xyz_Temp,3))
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity

    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatLocal
                              !
                              ! Latent heat used in this routine


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !

    if ( .not. saturate_DennouAGCM_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! Set a value for latent heat
    LatentHeatLocal = LatentHeat

    xyz_QVapSat = &
      & EpsV * Es0 * exp( LatentHeatLocal / GasRWet * ( 1.0_DP/273.0_DP - 1.0_DP/xyz_Temp ) ) &
      & / xyz_Press


  end function xyz_CalcQVapSatOnLiq

  !--------------------------------------------------------------------------------------

  function xyz_CalcDQVapSatDTempOnLiq( xyz_Temp, xyz_QVapSat ) result( xyz_DQVapSatDTemp )
    !
    ! 温度 *Temp* と飽和比湿 *QVapSat* を用い, 
    ! 飽和比湿の温度微分 *DQVapSatDTemp* を求めます. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRWet, &
                              ! $ R_v $ [J kg-1 K-1]. 
                              ! 凝結成分の気体定数. 
                              ! Gas constant of condensible elements
      & LatentHeat, &
                              ! $ L $ [J kg-1] . 
                              ! 凝結の潜熱. 
                              ! Latent heat of condensation
      & LatentHeatFusion
                              ! $ L $ [J kg-1] .
                              ! 融解の潜熱.
                              ! Latent heat of fusion

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp   (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_QVapSat(:,:,:)
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              ! $ \DP{q^{*}}{T} $ . 飽和比湿の温度微分. 
                              ! Temperature derivative of saturation specific humidity. 

    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatLocal
                              !
                              ! Latent heat used in this routine


    ! 実行文 ; Executable statement
    !

    if ( .not. saturate_DennouAGCM_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! Set a value for latent heat
    LatentHeatLocal = LatentHeat

    xyz_DQVapSatDTemp = LatentHeatLocal * xyz_QVapSat / ( GasRWet * xyz_Temp**2 )


  end function xyz_CalcDQVapSatDTempOnLiq

  !--------------------------------------------------------------------------------------

  function xyz_CalcQVapSatOnSol( xyz_Temp, xyz_Press ) result( xyz_QVapSat )
    !
    ! 温度 *Temp* と気圧 *Press* を用い, 
    ! 飽和比湿 *QVapSat* を求めます. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRWet, &
                              ! $ R_v $ [J kg-1 K-1]. 
                              ! 凝結成分の気体定数. 
                              ! Gas constant of condensible elements
      & LatentHeat, &
                              ! $ L $ [J kg-1] . 
                              ! 凝結の潜熱. 
                              ! Latent heat of condensation
      & LatentHeatFusion, &
                              ! $ L $ [J kg-1] .
                              ! 融解の潜熱.
                              ! Latent heat of fusion
      & EpsV                  ! $ \epsilon_v $ . 
                              ! 水蒸気分子量比. 
                              ! Molecular weight of water vapor

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_Press(:,:,:)
                              ! $ p $ . 気圧. Air pressure

    real(DP):: xyz_QVapSat(1:size(xyz_Temp,1), 1:size(xyz_Temp,2), 1:size(xyz_Temp,3))
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity

    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatLocal
                              !
                              ! Latent heat used in this routine


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !

    if ( .not. saturate_DennouAGCM_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! Set a value for latent heat
    LatentHeatLocal = LatentHeat + LatentHeatFusion

    xyz_QVapSat = &
      & EpsV * Es0 * exp( LatentHeatLocal / GasRWet * ( 1./273. - 1./xyz_Temp ) ) &
      & / xyz_Press


  end function xyz_CalcQVapSatOnSol

  !--------------------------------------------------------------------------------------

  function xyz_CalcDQVapSatDTempOnSol( xyz_Temp, xyz_QVapSat ) result( xyz_DQVapSatDTemp )
    !
    ! 温度 *Temp* と飽和比湿 *QVapSat* を用い, 
    ! 飽和比湿の温度微分 *DQVapSatDTemp* を求めます. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRWet, &
                              ! $ R_v $ [J kg-1 K-1]. 
                              ! 凝結成分の気体定数. 
                              ! Gas constant of condensible elements
      & LatentHeat, &
                              ! $ L $ [J kg-1] . 
                              ! 凝結の潜熱. 
                              ! Latent heat of condensation
      & LatentHeatFusion
                              ! $ L $ [J kg-1] .
                              ! 融解の潜熱.
                              ! Latent heat of fusion

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp   (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_QVapSat(:,:,:)
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              ! $ \DP{q^{*}}{T} $ . 飽和比湿の温度微分. 
                              ! Temperature derivative of saturation specific humidity. 

    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatLocal
                              !
                              ! Latent heat used in this routine


    ! 実行文 ; Executable statement
    !

    if ( .not. saturate_DennouAGCM_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! Set a value for latent heat
    LatentHeatLocal = LatentHeat + LatentHeatFusion

    xyz_DQVapSatDTemp = LatentHeatLocal * xyz_QVapSat / ( GasRWet * xyz_Temp**2 )


  end function xyz_CalcDQVapSatDTempOnSol

  !--------------------------------------------------------------------------------------

  subroutine SaturateInit
    !
    ! saturate_DennouAGCM モジュールの初期化を行います. 
    !--
    ! NAMELIST#saturate_DennouAGCM_nml の読み込みはこの手続きで行われます. 
    !++
    !
    ! "saturate_DennouAGCM" module is initialized. 
    !--
    ! "NAMELIST#saturate_DennouAGCM_nml" is loaded in this procedure. 
    !++

    ! モジュール引用 ; USE statements
    !

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
!!$    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 宣言文 ; Declaration statements
    !
    implicit none

!!$    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /saturate_DennouAGCM_nml/ &
!!$      & FlagSublimation
          !
          ! デフォルト値については初期化手続 "saturate_DennouAGCM#SaturateInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "saturate_DennouAGCM#SaturateInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( saturate_DennouAGCM_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml, &           ! (in)
!!$        & nml = saturate_DennouAGCM_nml, &  ! (out)
!!$        & iostat = iostat_nml )   ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$    end if

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
!!$    call MessageNotify( 'M', module_name, '  FlagSublimation = %b', l = (/ FlagSublimation /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    saturate_DennouAGCM_inited = .true.

  end subroutine SaturateInit

  !--------------------------------------------------------------------------------------

end module saturate_DennouAGCM
