!= 主成分相変化
!
!= Phase change of atmospheric major component
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: saturate_major_comp.f90,v 1.1 2013/09/30 02:55:27 yot Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module saturate_major_comp
  !
  != 主成分相変化
  !
  != Phase change of atmospheric major component
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  !== References
  !
  ! * Pollack, J. B., C. B. Leovy, P. W. Greiman, and Y. Mintz, 1981:
  !   A Martian general circulation experiment with large topography, 
  !   <i>J. Atmos. Sci.</i>, <b>38</b>, 3--29.
  !
  !== Procedures List
  ! 
!!$  ! DryConvAdjust :: 乾燥対流調節
!!$  ! ------------  :: ------------
!!$  ! DryConvAdjust :: Dry convective adjustment
  !
  !== NAMELIST
  !
  ! NAMELIST#saturate_major_comp_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! NAMELIST ファイル入力に関するユーティリティ
  ! Utilities for NAMELIST file input
  !
  use namelist_util, only: MaxNmlArySize
                              ! NAMELIST から読み込む配列の最大サイズ. 
                              ! Maximum size of arrays loaded from NAMELIST

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private


  integer, save      :: IndexSpc
  integer, parameter :: IndexSpcCO2     = 1


!!$  real(DP), parameter :: LatentHeatCO2Subl = 5.9d5
!!$                              ! Latent heat of CO2 (J kg^{-1})
!!$                              ! This value is obtained from Pollack et al. 
!!$                              ! (1981) and Forget et al. (1998).
  real(DP), parameter :: LatentHeatCO2Subl = 5.98d5
                              ! Latent heat of CO2 sublimation (J kg^{-1})
                              ! This value is obtained from Stull (1947).

  ! 公開手続き
  ! Public procedure
  !
  public :: SaturateMajorCompCondTemp
  public :: SaturateMajorCompPressSat
  public :: SaturateMajorCompDPressSatDT
  public :: SaturateMajorCompInqLatentHeat
  public :: SaturateMajorCompInit

  ! 公開変数
  ! Public variables
  !
  logical, save, public:: saturate_major_comp_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  ! 非公開変数
  ! Private variables
  !
  character(*), parameter:: module_name = 'saturate_major_comp'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: saturate_major_comp.f90,v 1.1 2013/09/30 02:55:27 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version


  interface SaturateMajorCompCondTemp
    module procedure &
      SaturateMajorCompCondTemp3D, &
      SaturateMajorCompCondTemp2D, &
      SaturateMajorCompCondTemp1D, &
      SaturateMajorCompCondTemp0D
  end interface

  interface SaturateMajorCompPressSat
    module procedure &
      SaturateMajorCompPressSat3D, &
      SaturateMajorCompPressSat2D
  end interface

  interface SaturateMajorCompDPressSatDT
    module procedure &
      SaturateMajorCompDPressSatDT3D, &
      SaturateMajorCompDPressSatDT2D
  end interface

contains

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompCondTemp3D(  &
    & xyz_Press,                           &  ! (in)
    & xyz_TempCond                         &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !

    != 
    != CO2 condensation temperature (Pollack et al., 1981)
    !
    use saturate_co2_p81, only : xyz_SaturateCO2P81TempCond

    != 
    != CO2 condensation temperature (Kasting, 1991)
    !
    use saturate_co2_k91, only : xyz_SaturateCO2K91TempCond


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Press   (:,:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(out):: xyz_TempCond(:,:,:)
                              !
                              ! Condensation temperature

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    select case ( IndexSpc )
    case ( IndexSpcCO2 )
      ! CO2 condensation temperature

#ifdef LIB_SATURATE_CO2_P81
      xyz_TempCond = xyz_SaturateCO2P81TempCond( xyz_Press )
#else
      xyz_TempCond = xyz_SaturateCO2K91TempCond( xyz_Press )
#endif

    end select


  end subroutine SaturateMajorCompCondTemp3D

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompCondTemp2D(  &
    & xy_Press,                            &  ! (in)
    & xy_TempCond                          &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xy_Press   (:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(out):: xy_TempCond(:,:)
                              !
                              ! Condensation temperature

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_Press   (size(xy_Press   ,1),size(xy_Press   ,2),1)
    real(DP) :: xyz_TempCond(size(xy_TempCond,1),size(xy_TempCond,2),1)


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    xyz_Press(:,:,1) = xy_Press

    call SaturateMajorCompCondTemp3D(  &
      & xyz_Press,                     &  ! (in)
      & xyz_TempCond                   &  ! (out)
      & )

    xy_TempCond = xyz_TempCond(:,:,1)


  end subroutine SaturateMajorCompCondTemp2D

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompCondTemp1D(  &
    & x_Press,                             &  ! (in)
    & x_TempCond                           &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: x_Press   (:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(out):: x_TempCond(:)
                              !
                              ! Condensation temperature

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_Press   (size(x_Press   ,1),1,1)
    real(DP) :: xyz_TempCond(size(x_TempCond,1),1,1)


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    xyz_Press(:,1,1) = x_Press

    call SaturateMajorCompCondTemp3D(  &
      & xyz_Press,                     &  ! (in)
      & xyz_TempCond                   &  ! (out)
      & )

    x_TempCond = xyz_TempCond(:,1,1)


  end subroutine SaturateMajorCompCondTemp1D

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompCondTemp0D(  &
    & Press,                               &  ! (in)
    & TempCond                             &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: Press
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(out):: TempCond
                              !
                              ! Condensation temperature

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_Press   (1,1,1)
    real(DP) :: xyz_TempCond(1,1,1)


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    xyz_Press(1,1,1) = Press

    call SaturateMajorCompCondTemp3D(  &
      & xyz_Press,                     &  ! (in)
      & xyz_TempCond                   &  ! (out)
      & )

    TempCond = xyz_TempCond(1,1,1)


  end subroutine SaturateMajorCompCondTemp0D

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompPressSat3D(  &
    & xyz_Temp,                            &  ! (in)
    & xyz_Press                            &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !

    != 
    != CO2 condensation temperature (Pollack et al., 1981)
    !
    use saturate_co2_p81, only : xyz_SaturateCO2P81PressSat

    != 
    != CO2 condensation temperature (Kasting, 1991)
    !
    use saturate_co2_k91, only : xyz_SaturateCO2K91PressSat

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Temp (:,:,:)
                              !
                              ! Condensation temperature
    real(DP), intent(out):: xyz_Press(:,:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    select case ( IndexSpc )
    case ( IndexSpcCO2 )
      ! CO2 condensation temperature

#ifdef LIB_SATURATE_CO2_P81
      xyz_Press = xyz_SaturateCO2P81PressSat( xyz_Temp )
#else
      xyz_Press = xyz_SaturateCO2K91PressSat( xyz_Temp )
#endif

    end select


  end subroutine SaturateMajorCompPressSat3D

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompPressSat2D( &
    & xy_Temp,                            &  ! (in)
    & xy_Press                            &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xy_Temp (:,:)
                              !
                              ! Condensation temperature
    real(DP), intent(out):: xy_Press(:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_Temp (size(xy_Temp ,1),size(xy_Temp ,2),1)
    real(DP) :: xyz_Press(size(xy_Press,1),size(xy_Press,2),1)


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    xyz_Temp(:,:,1) = xy_Temp

    call SaturateMajorCompPressSat3D( &
      & xyz_Temp,                     &  ! (in)
      & xyz_Press                     &  ! (out)
      & )

    xy_Press = xyz_Press(:,:,1)


  end subroutine SaturateMajorCompPressSat2D

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompDPressSatDT3D(  &
    & xyz_Temp,                                   &  ! (in)
    & xyz_DPressSatDT                             &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !

    != 
    != CO2 condensation temperature (Pollack et al., 1981)
    !
    use saturate_co2_p81, only : xyz_SaturateCO2P81DPressSatDT

    != 
    != CO2 condensation temperature (Kasting, 1991)
    !
    use saturate_co2_k91, only : xyz_SaturateCO2K91DPressSatDT

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyz_Temp (:,:,:)
                              !
                              ! Condensation temperature
    real(DP), intent(out):: xyz_DPressSatDT(:,:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    select case ( IndexSpc )
    case ( IndexSpcCO2 )
      ! CO2 condensation temperature

#ifdef LIB_SATURATE_CO2_P81
      xyz_DPressSatDT = xyz_SaturateCO2P81DPressSatDT( xyz_Temp )
#else
      xyz_DPressSatDT = xyz_SaturateCO2K91DPressSatDT( xyz_Temp )
#endif

    end select


  end subroutine SaturateMajorCompDPressSatDT3D

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompDPressSatDT2D( &
    & xy_Temp,                               &  ! (in)
    & xy_DPressSatDT                         &  ! (out)
    & )
    !
    ! 主成分相変化
    !
    ! Major component phase change
    !

    ! モジュール引用 ; USE statements
    !


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xy_Temp (:,:)
                              !
                              ! Condensation temperature
    real(DP), intent(out):: xy_DPressSatDT(:,:)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_Temp       (size(xy_Temp       ,1),size(xy_Temp       ,2),1)
    real(DP) :: xyz_DPressSatDT(size(xy_DPressSatDT,1),size(xy_DPressSatDT,2),1)


    ! 実行文 ; Executable statement
    !


    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    xyz_Temp(:,:,1) = xy_Temp

    call SaturateMajorCompDPressSatDT3D( &
      & xyz_Temp,                        &  ! (in)
      & xyz_DPressSatDT                  &  ! (out)
      & )

    xy_DPressSatDT = xyz_DPressSatDT(:,:,1)


  end subroutine SaturateMajorCompDPressSatDT2D

  !--------------------------------------------------------------------------------------

  function SaturateMajorCompInqLatentHeat( ) result( LatentHeat )
    !
    ! 主成分の潜熱を返しします. 
    !
    ! Inquiry of latent heat of major component
    !

    ! モジュール引用 ; USE statements
    !

    ! メッセージ出力
    ! Message output
    !
    use dc_message, only: MessageNotify

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP) :: LatentHeat


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. saturate_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! Set latent heat
    select case ( IndexSpc )
    case ( IndexSpcCO2  )
      LatentHeat = LatentHeatCO2Subl
    case default
      call MessageNotify( 'E', module_name, 'This specice is not supported.' )
    end select


  end function SaturateMajorCompInqLatentHeat

  !--------------------------------------------------------------------------------------

  subroutine SaturateMajorCompInit(  &
    & CondMajCompName                & ! (in)
    & )
    !
    ! saturate_major_comp モジュールの初期化を行います. 
    ! NAMELIST#saturate_major_comp_nml の読み込みはこの手続きで行われます. 
    !
    ! "saturate_major_comp" module is initialized. 
    ! "NAMELIST#saturate_major_comp_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

!!$    ! NAMELIST ファイル入力に関するユーティリティ
!!$    ! Utilities for NAMELIST file input
!!$    !
!!$    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid
!!$
!!$    ! ファイル入出力補助
!!$    ! File I/O support
!!$    !
!!$    use dc_iounit, only: FileOpen
!!$
!!$    ! 種別型パラメタ
!!$    ! Kind type parameter
!!$    !
!!$    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA

    != 
    != CO2 condensation temperature (Pollack et al., 1981)
    !
    use saturate_co2_p81, only : SaturateCO2P81Init

    != 
    != CO2 condensation temperature (Kasting, 1991)
    !
    use saturate_co2_k91, only : SaturateCO2K91Init

    ! 宣言文 ; Declaration statements
    !
    implicit none

    character(*), intent(in) :: CondMajCompName
                                        ! Condensable major component


!!$    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /saturate_major_comp_nml/ &
!!$      & FlagUse

          ! デフォルト値については初期化手続 "saturate_major_comp#MajorCompSaturateInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "saturate_major_comp#MajorCompSaturateInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( saturate_major_comp_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml,                    &  ! (in)
!!$        & nml = saturate_major_comp_nml, &  ! (out)
!!$        & iostat = iostat_nml )             ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$!      if ( iostat_nml == 0 ) write( STDOUT, nml = cumulus_adjust_nml )
!!$    end if


    ! Set index
    select case ( CondMajCompName )
    case ( 'CO2' )
      IndexSpc = IndexSpcCO2
    case default
      call MessageNotify( 'E', module_name, 'This specice is not supported.' )
    end select


    ! Initialization of modules called in this module
    !

#ifdef LIB_SATURATE_CO2_P81
    != 
    != CO2 condensation temperature (Pollack et al., 1981)
    !
    call SaturateCO2P81Init
#else
    call SaturateCO2K91Init
#endif


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    saturate_major_comp_inited = .true.

  end subroutine SaturateMajorCompInit

  !-------------------------------------------------------------------

end module saturate_major_comp
