program rad1d_main

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types

  ! 格子点数設定
  !
  use gridset, only : kmax

  != 地球大気向け放射モデル Ver. 2 ラッパー
  !
  use rad_Earth_V2_wrapper, only : RadEarthV2Wrapper1DFlux

  implicit none

  real(DP), allocatable :: r_Press  (:)  ! 鉛直圧力座標 (Pa) [r 点]
  real(DP), allocatable :: z_Press  (:)  ! 鉛直圧力座標 (Pa) [z 点]
  real(DP)              :: SurfTemp      ! 惑星表面温度 (K)
  real(DP), allocatable :: z_Temp   (:)  ! 温度 (K)
  real(DP), allocatable :: z_QH2OVap(:)  ! 比湿 (1)
  real(DP), allocatable :: z_QCO2   (:)  ! 二酸化炭素(質量)混合比   (1)
  real(DP), allocatable :: z_QO3    (:)  ! オゾン(質量)混合比       (1)
  real(DP), allocatable :: z_QN2O   (:)  ! 一酸化二窒素(質量)混合比 (1)
  real(DP), allocatable :: z_QCH4   (:)  ! メタン(質量)混合比       (1)

  real(DP)              :: SolarConst
  real(DP)              :: CosSZA
  real(DP)              :: SurfAlbedo    ! 惑星表面アルベド

  real(DP), allocatable :: r_RadSUwFlux(:) ! 上向き短波フラックス (Wm-2) [r 点]
  real(DP), allocatable :: r_RadSDwFlux(:) ! 下向き短波フラックス (Wm-2) [r 点]
  real(DP), allocatable :: r_RadLUwFlux(:) ! 上向き長波フラックス (Wm-2) [r 点]
  real(DP), allocatable :: r_RadLDwFlux(:) ! 下向き長波フラックス (Wm-2) [r 点]

  integer :: k   ! 鉛直方向繰り返し用変数


  ! 初期化
  call Rad1DMainInit

  != 地球大気向け放射モデル Ver. 2 ラッパー
  !
  call RadEarthV2Wrapper1DFlux( &
    & SolarConst, CosSZA,                                     & ! (in)
    & SurfAlbedo,                                             & ! (in)
    & z_Press, r_Press, z_Temp, SurfTemp,                     & ! (in)
    & z_QH2OVap, z_QCO2, z_QO3, z_QN2O, z_QCH4,               & ! (in)
    & r_RadSUwFlux, r_RadSDwFlux, r_RadLUwFlux, r_RadLDwFlux  & ! (out)
    & )

  write( 6, * ) "# Result"
  write( 6, * ) "# k   Press  RadSUwFlux  RadSDwFlux  RadLUwFlux  RadLDwFlux"
  do k = 0, kmax
    write( 6, * ) k, r_Press(k), &
      & r_RadSUwFlux(k), r_RadSDwFlux(k), r_RadLUwFlux(k), r_RadLDwFlux(k)
  end do

  write( 6, * ) "# Comparison with Case 5 of Ellingson et al. (1991) (E91)"
  write( 6, * ) "# Case 5 : Isothermal atmosphere 300 K, CO2 only 300 ppmv"
  write( 6, * ) "# Results of E91 is shown in Table10 of E91."
  write( 6, * ) "# Unit : W m-2"
  write( 6, * ) "# UpLW@Sfc = ", r_RadLUwFlux(0), " (E91 Ave=458.85, StdDev=1.05)"
  write( 6, * ) "# DnLW@Sfc = ", r_RadLDwFlux(0), " (E91 Ave= 90.16, StdDev=7.66)"
  write( 6, * ) "# UpLW@TOA = ", r_RadLUwFlux(kmax), " (E91 Ave=458.90, StdDev=2.08)"

  !----------------------------------------------------------------------------

contains

  !----------------------------------------------------------------------------

  ! 初期化
  !   鉛直層の設定
  !   初期条件の設定
  !   放射モジュールの初期化
  !
  subroutine Rad1DMainInit

    != NAMELIST ファイル入力に関するユーティリティ
    !
    use namelist_util, only : NmlutilInit

    != 物理定数設定
    !
    use constants0, only : PI

    != 物理定数設定
    !
    use constants, only : ConstantsInit

    != 格子点数設定
    !
    use gridset, only : GridSetInit

    != 地球大気向け放射モデル Ver. 2
    !
    use rad_Earth_V2_wrapper, only : RadEarthV2WrapperInit



    ! サブルーチン内変数

    integer  :: kmax_set      ! 鉛直層数

    real(DP) :: Ps            ! 惑星表面圧力 (Pa)

    character(128), parameter :: namelist_filename_in = "rad1d.cntl"

    integer  :: k   ! 鉛直方向繰り返し用変数


    ! 使用モジュールの初期化
    !

    != NAMELIST ファイル入力に関するユーティリティ
    !
    call NmlutilInit(  &
      & namelist_filename_in & ! (in)
      & )

    != 物理定数設定
    !
    call ConstantsInit

    != 格子点数設定
    !
    ! 鉛直層数 [z 点の数; r 点の数は +1]
    kmax_set = 20
    call GridSetInit( kmax_set )

    != 地球大気向け放射モデル Ver. 2
    !
    call RadEarthV2WrapperInit



    ! 配列割り付け
    !   注意 : kmax は gridset.f90 内で指定している
    !
    allocate( r_Press  (0:kmax) )
    allocate( z_Press  (1:kmax) )
    allocate( z_Temp   (1:kmax) )
    allocate( z_QH2OVap(1:kmax) )
    allocate( z_QCO2   (1:kmax) )
    allocate( z_QO3    (1:kmax) )
    allocate( z_QN2O   (1:kmax) )
    allocate( z_QCH4   (1:kmax) )

    allocate( r_RadSUwFlux(0:kmax) )
    allocate( r_RadSDwFlux(0:kmax) )
    allocate( r_RadLUwFlux(0:kmax) )
    allocate( r_RadLDwFlux(0:kmax) )


    ! 惑星表面圧力 (Pa)
    !
    Ps = 1.0d5

    ! 鉛直層境界面
    !
    r_Press(0) = Ps
    do k = 1, kmax
      r_Press(k) = r_Press(k-1) * exp( - 0.2d0 )
    end do


    ! 鉛直層中心面
    !
    do k = 1, kmax-1
      z_Press(k) = sqrt( r_Press(k-1) * r_Press(k) )
    end do
    k = kmax
    if ( r_Press(k) == 0.0d0 ) then
      z_Press(k) = ( r_Press(k-1) + r_Press(k) ) / 2.0d0
    else
      z_Press(k) = sqrt( r_Press(k-1) * r_Press(k) )
    end if


    ! 太陽放射フラックス, アルベドの設定
    !
    SolarConst = 1360.0d0
    CosSZA     = cos( 30.0d0*PI/180.0d0 )
    SurfAlbedo = 0.2d0

    ! 大気構造
    !   Case 5 of Ellingson et al. (1991) (E91)
    !     Ellingson et al. (1991), JGR, 96, 8929.
    !   Case 5 : Isothermal atmosphere 300 K, CO2 only 300 ppmv
    !
    SurfTemp   = 300.0d0
    z_Temp     = 300.0d0
    z_QH2OVap  = 0.0d0
    z_QO3      = 0.0d0   * 48.0d-3/28.0d-3

    ! Values at 2000 in CMIP5 recommendation, PRE2005_MIDYR_CONC.DAT
    ! CO2 :  369.0d-6
    ! N2O :  316.0d-9
    ! CH4 : 1751.0d-9
    z_QCO2     = 300.0d-6 * 44.0d-3/28.0d-3
    z_QN2O     =   0.0d0  * 44.0d-3/28.0d-3
    z_QCH4     =   0.0d0  * 16.0d-3/28.0d-3


  end subroutine Rad1DMainInit

  !----------------------------------------------------------------------------

end program rad1d_main
