!= opacity 計算プログラム
!
!= opacity calculation module
!
! Authors::   Masanori Onishi
! Version::   $Id: main_dcrtm.f90,v 1.00 onishi$
! Tag Name::  $Name: dcrtm-20161020 $
! Copyright:: Copyright (C) GFD Dennou Club, 2016. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]


!2012/01/25(linebyline.f90から引き継ぎ)
!2014/05/21
!2014/06/03 ! add continuum
!2014/07/18
!2015/01/29: modify ReadParSum
!2015/01/30: SecRadConst comment out
!2015/02/04: add subroutine RayleighScatAir
!2015/02/12: modify ReadParSum 0 - 70 K: liner interporate
!          : modify RayleighScatAir: expand high wavenumber (from 50000cm-1)

!2015/03/30
!2016/06/02
!2016/08/05
!2016/10/18: cut off values are modifiable
!2016/10/20: isotope abundance is modifiable
!2016/10/26: readHITRAN 
!2016/10/27: ReadParSumNC, SetQTm
!2016/10/27: linebyline, mol_absorpk are modified


!line by line で吸収スペクトルを計算

! 吸収係数、光学的厚さの計算の定式化は Rothman et al., 1998
! 但し、line shift は温度依存させる(2013/05/01)
! 鉛直分布の重みづけについて、温度、混合比の依存性を追加(2013/06/07)
! physical constants: CODATA 2010
! voigt profile test: Kuntz 1997 and Ruyten 2004, and Humlicek 1982 (2013/07/01)
! add line shape function (2013/07/08)

module mod_LineByLine
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !

  ! モジュール引用 ; USE statements
  !

  ! netcdf データ入出力
  ! netcdf data input or output
  !
  use gtool_history

  ! 物理定数設定1
  ! physical constants setteing 1
  !
  use constants0, only: PI

  ! 物理定数設定2
  ! physical constants setteing 2
  !
  use planetconst, only: LightSpeed,     &
    &                    PlanckConst,    &
    &                    BoltzmannConst, &
    &                    Grav,           &
    &                    AvogadroNum,    &
    &                    SecRadConst


  implicit none

  private

  ! 物理定数, 計算条件
  ! Pysical constants, condition of calculation
  !
  real(8) :: Press_Ref = 101325.0_8
                              ! $ p_0 $ [Pa].
                              ! HITRAN/HITEMP 参照圧力
                              ! reference pressure for HITRAN/HITEMP
  real(8) :: Temp_Ref = 296.0_8
                              ! $ T $ [K].
                              ! HITRAN/HITEMP 参照温度
                              ! reference temperature for HITRAN/HITEMP

  public :: FileCheck
  public :: LineCount
  public :: SetIsoMolWt
  public :: SetIsoQ0Val
  public :: SetIsoMRatioE
  public :: SetQTm
  public :: ReadParSumNC
  public :: ReadHITRAN
  public :: ray_scatk
  public :: ray_scatcs
  public :: RayleighScattCSAir
!  public :: MolecularWeight
  !public :: mol_absorpkOld
  public :: mol_absorpk
  !public :: linebylineOld
  public :: linebyline

contains 

  subroutine FileCheck( &
    & DataBase, k_mol    ) ! (in )
    !
    ! HITRAN data existence check
    ! 
    ! 
    character(10), intent(in) :: DataBase
    integer,       intent(in) :: k_mol!mmax
    character(128) :: fn
    logical :: od
    logical :: ex
    character( 2) :: MN
    integer :: k_m

    !do k_m = 1, mmax
    k_m = k_mol

      if (k_m <10) then
        write(MN, '(A1,I1)') '0', k_m
      else
        write(MN, '(I2)') k_m
      end if

      if( DataBase == 'HITRAN1996') then 
        fn ='./../data/'// MN //'_HIT96.PAR'
      else if( DataBase == 'HITRAN2008') then
        fn ='./../data/'// MN //'_hit08.par'
      else if( DataBase == 'HITEMP2010') then
        fn ='./../data/'// MN //'_HITEMP2010.par'
      else if( DataBase == 'HITRAN2012') then
        fn ='./../data/'// MN //'_hit12.par'
      else if( DataBase == 'TestData') then
        fn = './../data/test.par'
      end if

      inquire( file = fn, opened = od, exist = ex )
      if( .not. ex ) then
        write( 6, * ) 'ERROR: File, ', trim( fn ), ', does not exist.'
        stop
      end if
      if( od ) then
        write( 6, * ) 'ERROR: File, ', trim( fn ), ', is open.'
        stop
      end if
      write( 6, * ) 'File exists', fn
    !end do

  end subroutine filecheck

  subroutine FileCheckOld( &
    & DataBase, mmax    ) ! (in )
    !
    ! HITRAN data existence check
    ! 
    ! 
    character(10), intent(in) :: DataBase
    integer,       intent(in) :: mmax
    character(128) :: fn
    logical :: od
    logical :: ex
    character( 2) :: MN
    integer :: k_m

    do k_m = 1, mmax

      if (k_m <10) then
        write(MN, '(A1,I1)') '0', k_m
      else
        write(MN, '(I2)') k_m
      end if

      if( DataBase == 'HITRAN1996') then 
        fn ='./../data/'// MN //'_HIT96.PAR'
      else if( DataBase == 'HITRAN2008') then
        fn ='./../data/'// MN //'_hit08.par'
      else if( DataBase == 'HITEMP2010') then
        fn ='./../data/'// MN //'_HITEMP2010.par'
      else if( DataBase == 'HITRAN2012') then
        fn ='./../data/'// MN //'_hit12.par'
      else if( DataBase == 'TestData') then
        fn = './../data/test.par'
      end if

      inquire( file = fn, opened = od, exist = ex )
      if( .not. ex ) then
        write( 6, * ) 'ERROR: File, ', trim( fn ), ', does not exist.'
        stop
      end if
      if( od ) then
        write( 6, * ) 'ERROR: File, ', trim( fn ), ', is open.'
        stop
      end if
      write( 6, * ) 'File exists', fn
    end do

  end subroutine FileCheckOld

  subroutine LineCount( &
    &  DataBase,        & ! (in )
    &  k_mol,           & ! (in )
    &  LineNum          ) ! (out)
    !
    !
    ! calculate HITRAN data file line number
    ! 
    character(10), intent(in ) :: DataBase
    integer      , intent(in ) :: k_mol
    integer      , intent(out) :: LineNum
    character(128) :: fn
    character( 2) :: MN
    integer :: ios
    integer :: l

    if (k_mol <10) then
      write(MN, '(A1,I1)') '0', k_mol
    else
      write(MN, '(I2)') k_mol
    end if

    if( DataBase == 'HITRAN1996') then 
      fn ='./../data/'// MN //'_HIT96.PAR'
    else if( DataBase == 'HITRAN2008') then
      fn ='./../data/'// MN //'_hit08.par'
    else if( DataBase == 'HITEMP2010') then
      fn ='./../data/'// MN //'_HITEMP2010.par'
    else if( DataBase == 'HITRAN2012') then
      fn ='./../data/'// MN //'_hit12.par'
    else if( DataBase == 'TestData') then
      fn = './../data/test.par'
    end if

    open( 50, file=fn, status='unknown' )
!       open( 50, file='datafile', status='unknown' )
    l = 0
    count_line : do
      read( 50, *, iostat = ios )
 !         print *, ios
      if ( ios == -1 ) exit count_line
        l = l + 1
    end do count_line
    close( 50 )

    !write( 6, * ) l
    LineNum = l

  end subroutine LineCount

  subroutine ReadParSumNC(              & ! 2016/10/27
    & PATHQ, k_mol,                     & ! (in )
    & imt_Qval                          ) ! (out)
    !
    ! read parsum.dat.nc file
    !
    character(99), intent(in ) :: PATHQ
    integer, intent(in ) :: k_mol
    real(8), intent(out) :: imt_Qval(1:10,1:k_mol,1:2931)
    real(8) :: iat_Qval(1:10, 1:50, 1:2931)
    integer :: k_i, k_m

    call HistoryGet(''//PATHQ//'', 'imt_Qval', iat_Qval)
 
    do k_m = 1, k_mol
      do k_i = 1, 10
        imt_Qval(k_i, k_m, :) = iat_Qval(k_i, k_m, :)
      end do
    end do

  end subroutine ReadParSumNC

  subroutine SetQTm(                    & ! 2016/10/27
    & Temp, it_Qval,                    & ! (in )
    & i_Qval                            ) ! (out)
    !
    ! calculate Q value for a Temperature and a Molecule
    !
    real(8), intent(in ) :: Temp
    real(8), intent(in ) :: it_Qval(1:10, 1:2931)
    real(8), intent(out) :: i_Qval(1:10)
    real(8) :: xT1, xT2
    real(8) :: yQ1(1:10), yQ2(1:10)
    integer :: k_i

    if    ( Temp > 3000.0_8 ) then
      print *, 'Temperature is out of range(<=3000[K]), T:', Temp
      print *, 'stop (SetQTm)'
      stop
    elseif( Temp == 3000.0_8 ) then
      i_Qval(:) = it_Qval(:,2931)
    elseif( Temp <= 0.0_8 ) then
      print *, 'Temperature is not invalid value, T:', Temp 
      print *, 'stop (SetQTm)'
      stop
    else

      if( Temp < 70.0_8 ) then
        xT1 = 0.0_8
        xT2 = 70.0_8
        yQ1(1:10) = 0.0_8
        yQ2(1:10) = it_Qval(1:10,1)
      else
        xT1 = int(Temp)*1.0_8
        xT2 = xT1 + 1.0_8
        yQ1(1:10) = it_Qval(1:10, int(Temp)-69)
        yQ2(1:10) = it_Qval(1:10, int(Temp)-68)
      end if

      do k_i = 1, 10

        i_Qval(k_i) = yQ1(k_i) + (yQ2(k_i)-yQ1(k_i))*(Temp - xT1)/(xT2 - xT1)

      end do

    end if

  end subroutine SetQTm

  subroutine ray_scatk (                & ! 2016/06/21
    & k_mol, MolWt, wmax, MinWN, ResWN, & ! (in ) 
    & w_Scatk                           ) ! (out)
    !
    ! Rayleigh scattering cross section calculation
    !
    integer, intent(in ) :: k_mol
    real(8), intent(in ) :: MolWt
    integer, intent(in ) :: wmax
    real(8), intent(in ) :: MinWN
    real(8), intent(in ) :: ResWN
    real(8), intent(out) :: w_Scatk(0:wmax)

    call ray_scatcs(k_mol, wmax, MinWN, ResWN, w_Scatk)

    w_Scatk = w_Scatk * AvogadroNum/MolWt

  end subroutine ray_scatk

  subroutine ray_scatcs(         & ! 2016/06/21
    & k_mol, wmax, MinWN, ResWN, & ! (in ) 
    & w_ScatCS                   ) ! (out)
    !
    ! Rayleigh scattering cross section calculation
    !
    integer, intent(in ) :: k_mol
    integer, intent(in ) :: wmax
    real(8), intent(in ) :: MinWN
    real(8), intent(in ) :: ResWN
    real(8), intent(out) :: w_ScatCS(0:wmax)
    integer :: k_w
    real(8) :: alpha

    
    if(k_mol == 0) then
      do k_w = 0, wmax
        call RayleighScattCSAir(MinWN + ResWN*k_w, w_ScatCS(k_w))
      end do
    else

      call AlphaTable(k_mol, alpha)

      if(alpha == 0.0_8) then

        print *, 'No such Molecules Rayleigh Scattering Cross Section: k_mol=', k_mol
        print *, 'stop'
        stop

      else    
        do k_w = 0, wmax
          call ray_scatcs_alpha(MinWN + ResWN*k_w, alpha, w_ScatCS(k_w))
        end do
      end if

    end if

  end subroutine ray_scatcs

  subroutine ray_scatcs_alpha(   &
    & WaveNum, alpha,            & ! (in )
    & RSCS                       ) ! (out)
    !
    ! Rayleigh scattering cross section
    ! 
    real(8), intent(in ) :: WaveNum
    real(8), intent(in ) :: alpha
    real(8), intent(out) :: RSCS
    
    RSCS = 128.0_8 * PI*PI*PI*PI*PI * alpha * alpha * &
      & WaveNum * WaveNum * WaveNum * WaveNum/ &
      & 3.0_8

  end subroutine ray_scatcs_alpha

  subroutine AlphaTable(         &
    & k_mol,                     & ! (in )
    & alpha                      ) ! (out)
    !
    ! CRC HANDBOOK of CHEMISTRY and PHYSICS 84th EDITION
    ! Average Electric Diple plarizabilities for Ground State
    !
    integer, intent(in ) :: k_mol
    real(8), intent(out) :: alpha
    real(8) :: m_alpha(1:50)

    !   H2O (1)
    m_alpha( 1) = 1.45d-30 ![m3]
    !   CO2 (2)
    m_alpha( 2) = 2.911d-30
    !    O3 (3)
    m_alpha( 3) = 3.21d-30
    !   N2O (4)
    m_alpha( 4) = 3.03d-30
    !    CO (5)
    m_alpha( 5) = 1.95d-30
    !   CH4 (6)
    m_alpha( 6) = 2.593d-30
    !    O2 (7)
    m_alpha( 7) = 1.5812d-30
    !    NO (8)
    m_alpha( 8) = 1.70d-30
    !   SO2 (9)
    m_alpha( 9) = 3.72d-30! (4.28)
    !   NO2 (10)
    m_alpha(10) = 3.02d-30
    !   NH3 (11)
    m_alpha(11) = 2.81d-30! (2.10, 2.26)
    !  HNO3 (12)
    m_alpha(12) = 0.0_8!NO DATA
    !    OH (13)
    m_alpha(13) = 0.0_8!NO DATA
    !    HF (14)
    m_alpha(14) = 0.80d-30
    !   HCl (15)
    m_alpha(15) = 2.63d-30! (2.77)
    !   HBr (16)
    m_alpha(16) = 3.61d-30
    !    HI (17)
    m_alpha(17) = 5.44d-30! (5.35)
    !   ClO (18)
    m_alpha(18) = 0.0_8!NO DATA
    !   OCS (19)
    m_alpha(19) = 5.71d-30! (5.2)
    !  H2CO (20)
    m_alpha(20) = 0.0_8!NO DATA
    !  HOCl (21)
    m_alpha(21) = 0.0_8 !NO DATA
    !    N2 (22)
    m_alpha(22) = 1.7403d-30
    !   HCN (23)
    m_alpha(23) = 2.59d-30! (2.46)
    ! CH3Cl (24)
    m_alpha(24) = 5.35d-30! (4.72)
    !  H2O2 (25)
    m_alpha(25) = 0.0_8!NO DATA
    !  C2H2 (26)
    m_alpha(26) = 3.33d-30! (3.93)
    !  C2H6 (27)
    m_alpha(27) = 4.47d-30! (4.43)
    !   PH3 (28)
    m_alpha(28) = 4.84d-30
    !  COF2 (29)
    m_alpha(29) = 0.0_8!NO DATA
    !   SF6 (30)
    m_alpha(30) = 6.54d-30
    !   H2S (31)
    m_alpha(31) = 3.78d-30! (3.95)
    ! HCOOH (32)
    m_alpha(32) = 0.0_8!NO DATA
    !   HO2 (33)
    m_alpha(33) = 0.0_8!NO DATA
    !     O (34)
    m_alpha(34) = 0.802d-30
    !ClONO2 (35)
    m_alpha(35) = 0.0_8!NO DATA
    !   NO+ (36)
    m_alpha(36) = 0.0_8!NO DATA
    !  HOBr (37)
    m_alpha(37) = 0.0_8!NO DATA
    !  C2H4 (38)
    m_alpha(38) = 4.252d-30
    ! CH3OH (39)
    m_alpha(39) = 0.0_8!NO DATA
    ! CH3Br (40)
    m_alpha(40) = 5.87d-30! (6.03, 5.55)
    ! CH3CN (41)
    m_alpha(41) = 0.0_8!NO DATA
    !   CF4 (42)
    m_alpha(42) = 3.838d-30
    !   C4H2 (43)
    m_alpha(43) = 0.0_8!NO DATA
    !  HC3N  (44)
    m_alpha(44) = 0.0_8!NO DATA
    !  H2 (45)
    m_alpha(45) = 0.8023d-30! (0.8045, 0.8042, 0.8059)
    !    CS (46)
    m_alpha(46) = 0.0_8!NO DATA
    !  SO3 (47)
    m_alpha(47) = 4.84d-30

    m_alpha(48:50) = 0.0_8

    alpha = m_alpha(k_mol)

  end subroutine AlphaTable

  subroutine RayleighScattCSAir( &
    &  WaveNum,                  & ! (in )
    &  RSCSAir                   ) ! (out)
    !
    ! Rayleigh scattering cross section for Air
    ! Bucholtz, 1995: eq.8
    !
    real(8), intent(in ) :: WaveNum
    real(8), intent(out) :: RSCSAir
    real(8) :: WaveLen
    real(8) :: A1, B1, C1, D1
    real(8) :: A2, B2, C2, D2

    A1 = 3.01577d-28
    B1 = 3.55212_8
    C1 = 1.35579_8
    D1 = 0.11563_8

    A2 = 4.01061d-28
    B2 = 3.99668_8
    C2 = 1.10298d-3
    D2 = 2.71393d-2

    if( WaveNum .eq. 0.0_8) then ! 2016/06/03

      RSCSAir = 0.0_8

    else

      WaveLen = 1d+6/WaveNum ![micron]

      !if( WaveLen >= 0.5_8 ) then 
      if( WaveNum <= 2d+6 ) then 
        RSCSAir = A2*WaveLen**(-(B2 + C2*WaveLen + D2/WaveLen))
      !elseif( WaveLen >= 0.2 ) then
      elseif( WaveNum <= 5d+6 ) then
        RSCSAir = A1*WaveLen**(-(B1 + C1*WaveLen + D1/WaveLen))
      else                                                                 !2015/02/12
        RSCSAir = A1*0.2_8**(-(B1 + C1*0.2_8 + D1/0.2_8)) * &              !2015/02/12
          & (0.2_8/WaveLen)*(0.2_8/WaveLen)*(0.2_8/WaveLen)*(0.2_8/WaveLen)!2015/02/12

      !print *, WaveLen
      !print *, 'wavenumber is out of range(<=50000cm-1), WN=', WaveNum
      !print *, '==subroutine RayleighScattSCAir: stop'
      !stop
      end if

      RSCSAir = RSCSAir * 1d-4 ![m2]

      if( RSCSAir/RSCSAir .ne. 1.0_8 ) then ! infinity check ! 2016/06/03
        RSCSAir = 0.0_8
      end if

    end if

  end subroutine RayleighScattCSAir

!  subroutine MolecularWeight( &
!    & Press, Temp, MixRatioH, & ! (in )
!    & MolAve                  ) ! (out)

!    ! 平均分子量の計算
!    ! calculate molecular weight
!    !
!    real(8), intent(in ) :: Press(1:2)
!    real(8), intent(in ) :: Temp(1:2)
!    real(8), intent(in ) :: MixRatioH(1:2, 1:50)
!    real(8), intent(out) :: MolAve
!    integer :: k_mol, k_iso
!    real(8) :: MolWeight

!    MolVal = 0.0_8
!    call SetMolVal(2)

!    MolAve = 0.0_8 ! 2013/07/18
    
!    do k_mol = 1, 50
!      if( (MixRatioH(1, k_mol) .ne. 0.0_8) .and. &
!        & (MixRatioH(2, k_mol) .ne. 0.0_8) ) then
!        MolWeight = 0.0_8
!        do k_iso = 1, 10
!          MolWeight = MolWeight + MolVal(k_mol, k_iso, 1) * MolVal(k_mol, k_iso, 3)        
!        end do
!        MolAve = MolAve + MolWeight * &
!          & (MixRatioH(1, k_mol) * Press(1)/Temp(1) + MixRatioH(2, k_mol) * Press(2)/Temp(2))/&
!          & (Press(1)/Temp(1) + Press(2)/Temp(2))
!      end if
!    end do  

!  end subroutine MolecularWeight

  subroutine mol_absorpk(                      & !2016/10/27
    & Flag_LINESHAPE, CutOffVal, i_Qval,       & !(in )
    & i_IsoARatio, i_IsoQ0, i_IsoMolWt,        & !(in )
    & af_IntHITRAN, af_RealHITRAN, FileEnd,    & !(in )
    & k_mol, MolWt, Press, Temp, MixRatioH,    & !(in )
    & wmax, MinWaveNum, ResWaveNum,            & !(in )
    & w_Absorpk                                ) !(out)

    integer, intent(in ) :: Flag_LINESHAPE

    real(8), intent(in ) :: CutOffVal

    real(8), intent(in ) :: i_Qval(1:10)

    real(8), intent(in ) :: i_IsoARatio(1:10)

    real(8), intent(in ) :: i_IsoQ0(1:10)

    real(8), intent(in ) :: i_IsoMolWt(1:10)

    integer, intent(in ) :: af_IntHITRAN(1:2, 1:FIleEnd)

    real(8), intent(in ) :: af_RealHITRAN(1:8, 1:FileEnd)

    integer, intent(in ) :: FileEnd

    integer, intent(in ) :: k_mol

    real(8), intent(in ) :: MolWt

    real(8), intent(in ) :: Press
                              ! $ p_0 $ [Pa].
                              ! 圧力
                              ! pressure
    real(8), intent(in ) :: Temp
                              ! $ T $ [K].
                              ! 温度
                              ! temperature
    real(8), intent(in ) :: MixRatioH
                              ! volume mixing ratio of molecules

    integer, intent(in ) :: wmax

    real(8), intent(in ) :: MinWaveNum

    real(8), intent(in ) :: ResWaveNum

    real(8), intent(out) :: w_Absorpk(0:wmax)


    call linebyline(                          &
      & Flag_LINESHAPE, CutOffVal, i_Qval,    &
      & i_IsoARatio, i_IsoQ0, i_IsoMolWt,     &
      & af_IntHITRAN, af_RealHITRAN, FileEnd, &
      & wmax, k_mol, Press, Temp,             &
      & MixRatioH, MinWaveNum, ResWaveNum,    &
      & w_Absorpk(0:wmax))

    w_Absorpk = w_Absorpk * AvogadroNum/MolWt

  end subroutine mol_absorpk

  subroutine mol_absorpkOld(                   & !2016/06/21
    & Flag_LINESHAPE, CutOffVal, PATHQ,        & !(in )
    & i_IsoARatio, i_IsoQ0, i_IsoMolWt,        & !(in )
    & DataBase, k_mol, MolWt, Press, Temp,     & !(in )
    & MixRatioH, wmax, MinWaveNum, ResWaveNum, & !(in )
    & w_Absorpk                                ) !(out)

    integer, intent(in ) :: Flag_LINESHAPE

    real(8), intent(in ) :: CutOffVal

    character(99), intent(in ) :: PATHQ

    real(8), intent(in ) :: i_IsoARatio(1:10)

    real(8), intent(in ) :: i_IsoQ0(1:10)

    real(8), intent(in ) :: i_IsoMolWt(1:10)

    character(10), intent(in ) :: DataBase

    integer, intent(in ) :: k_mol

    real(8), intent(in ) :: MolWt

    real(8), intent(in ) :: Press
                              ! $ p_0 $ [Pa].
                              ! 圧力
                              ! pressure
    real(8), intent(in ) :: Temp
                              ! $ T $ [K].
                              ! 温度
                              ! temperature
    real(8), intent(in ) :: MixRatioH
                              ! volume mixing ratio of molecules

    integer, intent(in ) :: wmax

    real(8), intent(in ) :: MinWaveNum

    real(8), intent(in ) :: ResWaveNum

    real(8), intent(out) :: w_Absorpk(0:wmax)


    call linebylineOld(Flag_LINESHAPE, CutOffVal, PATHQ,  &
      & i_IsoARatio, i_IsoQ0, i_IsoMolWt,   &
      & DataBase, wmax, k_mol, Press, Temp, &
      & MixRatioH, MinWaveNum, ResWaveNum,  &
      & w_Absorpk(0:wmax))

    w_Absorpk = w_Absorpk * AvogadroNum/MolWt

  end subroutine mol_absorpkOld

  subroutine ReadHITRAN(                       &
    & DataBase, k_mol, FileEnd,                & !(in )
    & af_IntHITRAN, af_RealHITRAN              ) !(out)
 
    character(10), intent(in ) :: DataBase
    integer, intent(in ) :: k_mol
    integer, intent(in ) :: FileEnd
    integer, intent(out) :: af_IntHITRAN(1:2,1:FileEnd)
    real(8), intent(out) :: af_RealHITRAN(1:8,1:FileEnd)
    character( 2) :: NumHi
    integer :: fHi = 31
    integer :: k_f

    if( k_mol < 10 ) then   ! 2014/04/15
      write(NumHi, '(A1,I1)') '0', k_mol
      !DnumHi = '0' // SnumHi
    else
      write(NumHi, '(I2)') k_mol
    end if

    if( DataBase == 'HITRAN1996') then                     ! 2014/05/21
      open(fHi, file='./../data/'// NumHi //'_HIT96.PAR')
    else if( DataBase == 'HITRAN2008') then
      open(fHi, file='./../data/'// NumHi //'_hit08.par')
    else if( DataBase == 'HITEMP2010') then
      open(fHi, file='./../data/'// NumHi //'_HITEMP2010.par')
    else if( DataBase == 'HITRAN2012') then
      open(fHi, file='./../data/'// NumHi //'_hit12.par')
    else if( DataBase == 'TestData') then                  ! 2016/10/18
      open(fHi, file='./../data/test.par')
    end if

    do k_f = 1, FileEnd
      read (fHi, '(I2, I1, F12.6, E10.3, E10.3, F5.4, F5.4, F10.4, F4.2, F8.6)') &
        & af_IntHITRAN(1,k_f), &    ! MolNum(k_Hi)
        & af_IntHITRAN(2,k_f), &    ! IsoNum(k_Hi)
        & af_RealHITRAN(1,k_f), &   ! WaveNum0(k_Hi)
        & af_RealHITRAN(2,k_f), &   ! Int_Ref(k_Hi)
        & af_RealHITRAN(3,k_f), &   ! EinsA(k_Hi)
        & af_RealHITRAN(4,k_f), &   ! AirBroadHarf(k_Hi)
        & af_RealHITRAN(5,k_f), &   ! SelfBroadHarf(k_Hi)
        & af_RealHITRAN(6,k_f), &   ! LowStatE(k_Hi)
        & af_RealHITRAN(7,k_f), &   ! N_air(k_Hi)
        & af_RealHITRAN(8,k_f)      ! delta_air(k_Hi)

      !  & MolNum(k_Hi), IsoNum(k_Hi), WaveNum0(k_Hi), Int_Ref(k_Hi), EinsA(k_Hi), &
      !  & AirBroadHarf(k_Hi), SelfBroadHarf(k_Hi), LowStatE(k_Hi), N_air(k_Hi), delta_air(k_Hi)
      ! 単位をSIに変換
      !WaveNum0(k_Hi) = WaveNum0(k_Hi) * 100.0_8
      af_RealHITRAN(1,k_f) = af_RealHITRAN(1,k_f) * 100.0_8
      !Int_Ref(k_Hi) = Int_Ref(k_Hi) * 0.010_8
      af_RealHITRAN(2,k_f) = af_RealHITRAN(2,k_f) * 0.010_8
      !AirBroadHarf(k_Hi) = AirBroadHarf(k_Hi) * 9.869233e-4_8 ! 1 [cm-1 atm-1] = 10^2/101325 [m-1 Pa-1]
      af_RealHITRAN(4,k_f) = af_RealHITRAN(4,k_f) * 9.869233e-4_8 ! 1 [cm-1 atm-1] = 10^2/101325 [m-1 Pa-1]
      !SelfBroadHarf(k_Hi) = SelfBroadHarf(k_Hi) * 9.869233e-4_8
      af_RealHITRAN(5,k_f) = af_RealHITRAN(5,k_f) * 9.869233e-4_8
      !LowStatE(k_Hi) = LowStatE(k_Hi) * 100.0_8
      af_RealHITRAN(6,k_f) = af_RealHITRAN(6,k_f) * 100.0_8
      !delta_air(k_Hi) = delta_air(k_Hi) * 9.869233e-4_8
      af_RealHITRAN(8,k_f) = af_RealHITRAN(8,k_f) * 9.869233e-4_8
    end do

    close(fHi) !=================================Read HITRAN end


  end subroutine ReadHITRAN

  subroutine linebyline(                       & ! 2016/10/27
    & Flag_LINESHAPE, CutOffVal, i_Qval,       & !(in )
    & i_IsoARatio, i_IsoQ0, i_IsoMolWt,        & !(in )
    & af_IntHITRAN, af_RealHITRAN, FileEnd,    & !(in ) !2016/10/26
    & wmax, k_mol, Press, Temp,                & !(in ) 
    & MixRatioH, MinWaveNum, ResWaveNum,       & !(in )
    & AbsorpG                                  ) !(out)

    integer, intent(in ) :: Flag_LINESHAPE !2016/06/30

    real(8), intent(in ) :: CutOffVal      !2016/10/18

    real(8), intent(in ) :: i_Qval(1:10)   !2016/10/27

    real(8), intent(in ) :: i_IsoARatio(1:10)

    real(8), intent(in ) :: i_IsoQ0(1:10)

    real(8), intent(in ) :: i_IsoMolWt(1:10)

    integer, intent(in ) :: af_IntHITRAN(1:2, 1:FIleEnd)

    real(8), intent(in ) :: af_RealHITRAN(1:8, 1:FileEnd)

    integer, intent(in ) :: FileEnd

    integer, intent(in ) :: wmax

    real(8), intent(in ) :: Press
                              ! $ p_0 $ [Pa].
                              ! 圧力
                              ! pressure
    real(8), intent(in ) :: Temp
                              ! $ T $ [K].
                              ! 温度
                              ! temperature
    real(8), intent(in ) :: MixRatioH
                          ! Mix Ratio of molecules in Hitran database
    real(8), intent(in ) :: MinWaveNum

    real(8), intent(in ) :: ResWaveNum

    real(8), intent(out) :: AbsorpG(0:wmax)

    integer, intent(in ) :: k_mol

!    integer, allocatable :: MolNum(:)
                              ! 
                              ! Molecule number (input HITRAN)
    integer, allocatable :: IsoNum(:)
                              !
                              ! Isotopologue number (input HITRAN)
!    real(8), allocatable :: WaveNum0(:)
                              ! $ \v_0 $ [m-1]
                              ! 波数. Vacuume wavenumber (input HITRAN)
!    real(8), allocatable :: Int_Ref(:)
                              ! $ \S_0 $ [m molecule-1]
                              ! 吸収線強度. Intensity at 296[K] (input HITRAN)
!    real(8), allocatable :: EinsA(:)
                              ! $ \A $ [s-1]
                              ! Einstein A-coefficient (input HITRAN)
!    real(8), allocatable :: AirBroadHarf(:)
                              ! $ \gamma_{air} $ [m-1 Pa-1]
                              ! Air-broadened half-width at 296[K] (input HITRAN)
!    real(8), allocatable :: SelfBroadHarf(:)
                              ! $ \gamma_{self} $ [m-1 Pa-1]
                              ! Self-broadened hafl-width at 296[K] (input HITRAN)
!    real(8), allocatable :: LowStatE(:)
                              ! $ \E^" $ [m-1]
                              ! Lower-State energy (input HITRAN)
!    real(8), allocatable :: N_air(:)
                              ! $ n_{air} $ [1]
                              ! Temperature dependence exponent for gamma_air (input HITRAN)
!    real(8), allocatable :: delta_air(:)
                              ! $ \delta_{air} $ [m-1 Pa-1]
                              ! Air pressure-induced line shift (input HITRAN)
    integer :: k_iso

    integer :: k_ln

    ! Hitran data 読み込み用
    integer :: k_Hi


    ! Hitran dataの読み込み ======================Read HITRAN Start
    !
    !allocate( MolNum       (1:FileEnd) )
    allocate( IsoNum       (1:FileEnd) )
    !allocate( WaveNum0     (1:FileEnd) )
    !allocate( Int_Ref      (1:FileEnd) )
    !allocate( EinsA        (1:FileEnd) )
    !allocate( AirBroadHarf (1:FileEnd) )
    !allocate( SelfBroadHarf(1:FileEnd) )
    !allocate( LowStatE     (1:FileEnd) )
    !allocate( N_air        (1:FileEnd) )
    !allocate( delta_air    (1:FileEnd) )

    !open (fHi, file='./data/01_0-50_HITEMP2010.par') ! 25cm-1 check 20131225
    !print *, 'start read HITRAN'
    do k_Hi = 1, FileEnd
      !read (fHi, '(I2, I1, F12.6, E10.3, E10.3, F5.4, F5.4, F10.4, F4.2, F8.6)') &
      !  & MolNum(k_Hi), IsoNum(k_Hi), WaveNum0(k_Hi), Int_Ref(k_Hi), EinsA(k_Hi), &
      !  & AirBroadHarf(k_Hi), SelfBroadHarf(k_Hi), LowStatE(k_Hi), N_air(k_Hi), delta_air(k_Hi)
    !  MolNum(k_Hi)         = af_IntHITRAN(1,k_Hi)
      IsoNum(k_Hi)         = af_IntHITRAN(2,k_Hi)
    !  WaveNum0(k_Hi)       = af_RealHITRAN(1,k_Hi)
    !  Int_Ref(k_Hi)        = af_RealHITRAN(2,k_Hi)
    !  EinsA(k_Hi)          = af_RealHITRAN(3,k_Hi)
    !  AirBroadHarf(k_Hi)   = af_RealHITRAN(4,k_Hi)
    !  SelfBroadHarf(k_Hi)  = af_RealHITRAN(5,k_Hi)
    !  LowStatE(k_Hi)       = af_RealHITRAN(6,k_Hi)
    !  N_air(k_Hi)          = af_RealHITRAN(7,k_Hi)
    !  delta_air(k_Hi)      = af_RealHITRAN(8,k_Hi)

    end do

    !! NOTE !!
    ! in HITRAN2012 CO2 isotope number '0' exists !!
    ! but this program cannot calculate isotope number '0'

    ! print *, "mixratioH=/0 kmol=", k_mol
    ! 波数範囲にあるデータを読み込む
    !
    AbsorpG(:) = 0.0_8 ! 2014/05/15
    do k_ln = 1, FileEnd

      if ( IsoNum(k_ln) .ne. 0 ) then ! 2016/10/21
      !if ( af_IntHITRAN(2,k_ln) .ne. 0 ) then ! 2016/10/21
        !if ( ( MolVal(k_mol, IsoNum(k_ln), 1) .ne. 0.0_8 ) .and. &
        if ( ( i_IsoARatio(IsoNum(k_ln)) .ne. 0.0_8 ) .and. &
          &  ( i_Qval(IsoNum(k_ln)) .ne. 0.0_8 ) ) then   ! parsumdatに無いものも除く
          !print *, 'call calc_absorption'
          !if(.true.) then
          call Calc_absorption( Flag_LINESHAPE, CutOffVal, &
            & wmax, MinWaveNum, ResWaveNum, k_mol, Press, Temp, MixRatioH, &
            & af_RealHITRAN(1,k_ln), af_RealHITRAN(2,k_ln), af_RealHITRAN(4,k_ln), &
            & af_RealHITRAN(5,k_ln), af_RealHITRAN(6,k_ln), &
            & af_RealHITRAN(7,k_ln), af_RealHITRAN(8,k_ln), i_IsoARatio(IsoNum(k_ln)), &
            & i_IsoQ0(IsoNum(k_ln)), i_IsoMolWt(IsoNum(k_ln)), i_Qval(IsoNum(k_ln)), &
            & AbsorpG(0:wmax))
          !else

          !call Calc_absorption( Flag_LINESHAPE, CutOffVal, &
          !  & wmax, MinWaveNum, ResWaveNum, k_mol, Press, Temp, MixRatioH, &
          !  & WaveNum0(k_ln), Int_Ref(k_ln), AirBroadHarf(k_ln), SelfBroadHarf(k_ln), LowStatE(k_ln), &
          !  & N_air(k_ln), delta_air(k_ln), i_IsoARatio(IsoNum(k_ln)), &
          !  & i_IsoQ0(IsoNum(k_ln)), i_IsoMolWt(IsoNum(k_ln)), i_Qval(IsoNum(k_ln)), &
          !  & AbsorpG(0:wmax))
          !end if

        end if
      end if

    end do ! k_ln

    !deallocate( MolNum        )
    deallocate( IsoNum        )
    !deallocate( WaveNum0      )
    !deallocate( Int_Ref       )
    !deallocate( EinsA         )
    !deallocate( AirBroadHarf  )
    !deallocate( SelfBroadHarf )
    !deallocate( LowStatE      )
    !deallocate( N_air         )
    !deallocate( delta_air     )

  end subroutine linebyline

  subroutine linebylineOld(                       &
    & Flag_LINESHAPE, CutOffVal, PATHQ,        & !(in )
    & i_IsoARatio, i_IsoQ0, i_IsoMolWt,        & !(in )
    & DataBase, wmax, k_mol, Press,            & !(in )
    & Temp, MixRatioH, MinWaveNum, ResWaveNum, & !(in )
    & AbsorpG                                  ) !(out)

    integer, intent(in ) :: Flag_LINESHAPE !2016/06/30

    real(8), intent(in ) :: CutOffVal      !2016/10/18

    character(99), intent(in ) :: PATHQ

    character(10), intent(in ) :: DataBase

    real(8), intent(in ) :: i_IsoARatio(1:10)

    real(8), intent(in ) :: i_IsoQ0(1:10)

    real(8), intent(in ) :: i_IsoMolWt(1:10)

    integer, intent(in ) :: wmax

    real(8), intent(in ) :: Press
                              ! $ p_0 $ [Pa].
                              ! 圧力
                              ! pressure
    real(8), intent(in ) :: Temp
                              ! $ T $ [K].
                              ! 温度
                              ! temperature
    real(8), intent(in ) :: MixRatioH
                          ! Mix Ratio of molecules in Hitran database
    real(8), intent(in ) :: MinWaveNum

    real(8), intent(in ) :: ResWaveNum

    real(8), intent(out) :: AbsorpG(0:wmax)

    integer, intent(in ) :: k_mol

    integer, allocatable :: MolNum(:)
                              ! 
                              ! Molecule number (input HITRAN)
    integer, allocatable :: IsoNum(:)
                              !
                              ! Isotopologue number (input HITRAN)
    real(8), allocatable :: WaveNum0(:)
                              ! $ \v_0 $ [m-1]
                              ! 波数. Vacuume wavenumber (input HITRAN)
    real(8), allocatable :: Int_Ref(:)
                              ! $ \S_0 $ [m molecule-1]
                              ! 吸収線強度. Intensity at 296[K] (input HITRAN)
    real(8), allocatable :: EinsA(:)
                              ! $ \A $ [s-1]
                              ! Einstein A-coefficient (input HITRAN)
    real(8), allocatable :: AirBroadHarf(:)
                              ! $ \gamma_{air} $ [m-1 Pa-1]
                              ! Air-broadened half-width at 296[K] (input HITRAN)
    real(8), allocatable :: SelfBroadHarf(:)
                              ! $ \gamma_{self} $ [m-1 Pa-1]
                              ! Self-broadened hafl-width at 296[K] (input HITRAN)
    real(8), allocatable :: LowStatE(:)
                              ! $ \E^" $ [m-1]
                              ! Lower-State energy (input HITRAN)
    real(8), allocatable :: N_air(:)
                              ! $ n_{air} $ [1]
                              ! Temperature dependence exponent for gamma_air (input HITRAN)
    real(8), allocatable :: delta_air(:)
                              ! $ \delta_{air} $ [m-1 Pa-1]
                              ! Air pressure-induced line shift (input HITRAN)
    real(8) :: Intensity
                              ! $ \S $ [m molecule-1]
                              ! Intensity
    real(8) :: Alpha_D
                              ! $ \alpha_D $ [m-1]
                              ! Harf-width of Doppler broadening
    real(8) :: Alpha_L
                              ! $ \alpha_L $ [m-1]
                              ! Harf-width of Pressure broadening
    real(8) :: WaveNumS
                              ! $ \v $ [m-1]
                              ! 波数. pressure shift wavenumber
    integer :: FileEnd(1:50)

    integer :: ParSumMol(1:50, 0:10) ! second index 1:10 -> 0:10 !2016/06/03

    real(8) :: Qval(0:108, 1:2)

    real(8) :: Q_TT(1:50, 1:10) !Q_Tからこちらに変更する

    integer :: cutoff = 1 !1: 2500[m-1], 2: HarfWidth dependent

    integer :: k_iso

    integer :: k_min, k_max

    integer :: k

    integer :: k_ln !, k_g

    real(8) :: BroadHarf

    real(8) :: CalcWidth

!    real(8) :: OffSet = 2500.0_8

    ! Hitran data 読み込み用
    integer :: k_Hi
    integer :: fHi = 13
    character(31) :: filenameHi
    character( 1) :: SnumHi
    character( 2) :: DnumHi


    ! HITRAN dataファイルの行数: FileEndの設定
    if(.false.) then
      if( DataBase == 'HITRAN1996') then   ! 2014/05/21
         FileEnd(1:50) = (/ 49444,  60802, 275133,  26174,    4477, & !  1: 5
          &                 48032,   6292,  15331,  38853,  100680, & !  6:10
          &                 11152, 165426,   8676,    107,     533, & ! 11:15
          &                   576,    237,   7230,    858,    2702, & ! 16:20
          &                 15565,    120,    772,   9355,    5444, & ! 21:25
          &                  1668,   4749,   2886,  54866,   11520, & ! 26:30
          &                  7151,   3388,  26963,      2,   32199, & ! 31:35
          &                     0,      0,      0,      0,       0, & ! 36:40
          &                     0,      0,      0,      0,       0, & ! 41:45
          &                     0,      0,      0,      0,       0 /) ! 46:50

      else if( DataBase == 'HITRAN2008') then
        FileEnd(1:50) = (/  69201, 314919, 409686,  47843,    4477, & !  1: 5
          &                290091,   6428, 105079,  58250,  104223, & !  6:10
          &                 29084, 487254,  31976,    107,     613, & ! 11:15
          &                  1293,    806,  11501,  29242,   37050, & ! 16:20
          &                 16276,    120,   4253, 196171,  126983, & ! 21:25
          &                 11340,  22402,  20099,  70601, 2889065, & ! 26:30
          &                 20788,  62684,  38804,      2,   32199, & ! 31:35
          &                  1206,   4358,  18378,  19897,   36911, & ! 36:40
          &                  3572,  60033,      0,      0,       0, & ! 41:45
          &                     0,      0,      0,      0,       0 /) ! 46:50
 
      else if( DataBase == 'HITEMP2010' ) then
        FileEnd( 1) = 114241164
        FileEnd( 2) =  11193608
        FileEnd( 5) =    113631
        FileEnd( 8) =    115610
        FileEnd(13) =     41557

      else if( DataBase == 'HITRAN2012' ) then
        FileEnd(1:50) = (/  224515, 471847, 422116,  47843,    4606, & !  1: 5
          &                 468013,  13975, 105079,  95121,  104223, & !  6:10
          &                  46392, 961962,  31979,  34376,   83691, & ! 11:15
          &                   8980,   4751,  11501,  29361,   44601, & ! 16:20
          &                  16276,   1268,   4253, 212496,  126983, & ! 21:25
          &                  20410,  49629,  22189, 184104, 2889065, & ! 26:30
          &                  54235,  62684,  38804,      2,   32199, & ! 31:35
          &                   1206,   4358,  18378,  19897,   36911, & ! 36:40
          &                   3572,  60033, 124126, 180332,    9146, & ! 41:45
          &                   2078,  10881,      0,      0,       0 /) ! 46:50
      else
        print *, 'invalid database, DataBase:', DataBase
        print *, 'stop (linebyline)'
        stop
      end if
    else

      call LineCount( DataBase, k_mol, FileEnd(k_mol) )
      !print *, FileEnd(k_mol)
    end if

    ! parsum.datのコラムと分子種の対応マトリックス
    ParSumMol(1:50, 1:10) = 0.0_8

    ParSumMol( 1, 1:6 ) = (/ 1, 2, 3, 4, 5, 6 /) !H2O
    ParSumMol( 2, 1:8 ) = (/ 7, 8, 9, 10, 11, 12, 13, 14 /) !CO2
    ParSumMol( 3, 1:10) = (/ 15, 16, 17, 18, 19, 20, 21, 22, 23, 24 /) !O3 !32まである
    ParSumMol( 4, 1:5 ) = (/ 33, 34, 35, 36, 37 /) !N2O
    ParSumMol( 5, 1:6 ) = (/ 38, 39, 40, 41, 42, 43 /) !CO2
    ParSumMol( 6, 1:3 ) = (/ 44, 45, 46 /) !CH4
    ParSumMol( 7, 1:3 ) = (/ 47, 48, 49 /) !O3
    ParSumMol( 8, 1:3 ) = (/ 50, 51, 52 /) !NO
    ParSumMol( 9, 1:2 ) = (/ 53, 54 /) !SO2
    ParSumMol(10, 1:1 ) = (/ 55 /) !NO2
    ParSumMol(11, 1:2 ) = (/ 56, 57 /) !NH3
    ParSumMol(12, 1:1 ) = (/ 58 /) !HNO3
    ParSumMol(13, 1:3 ) = (/ 59, 60, 61 /) !OH
    ParSumMol(14, 1:1 ) = (/ 62 /) !HF
    ParSumMol(15, 1:2 ) = (/ 63, 64 /) !HCl
    ParSumMol(16, 1:2 ) = (/ 65, 66 /) !HBr
    ParSumMol(17, 1:1 ) = (/ 67 /) !HI
    ParSumMol(18, 1:2 ) = (/ 68, 69 /) !ClO
    ParSumMol(19, 1:5 ) = (/ 70, 71, 72, 73, 74 /) !OCS
    ParSumMol(20, 1:3 ) = (/ 75, 76, 77 /) !H2CO
    ParSumMol(21, 1:2 ) = (/ 78, 79 /) !HOCl
    ParSumMol(22, 1:1 ) = (/ 80 /) !N2
    ParSumMol(23, 1:3 ) = (/ 81, 82, 83 /) !HCN
    ParSumMol(24, 1:2 ) = (/ 84, 85 /) !CH3Cl
    ParSumMol(25, 1:1 ) = (/ 86 /) !H2O2
    ParSumMol(26, 1:2 ) = (/ 87, 88 /) !C2H2
    ParSumMol(27, 1:1 ) = (/ 89 /) !C2H6
    ParSumMol(28, 1:1 ) = (/ 90 /) !PH3
    ParSumMol(29, 1:1 ) = (/ 91 /) !COF2
    ParSumMol(30, 1:1 ) = (/ 92 /) !SF6
    ParSumMol(31, 1:3 ) = (/ 93, 94, 95 /) !H2S
    ParSumMol(32, 1:1 ) = (/ 96 /) !HCOOH
    ParSumMol(33, 1:1 ) = (/ 97 /) !HO2
    ParSumMol(34, 1:1 ) = (/ 98 /) !O
    ParSumMol(35, 1:2 ) = (/ 99, 100 /) !ClONO2
    ParSumMol(36, 1:1 ) = (/ 101 /) !NO+
    ParSumMol(37, 1:2 ) = (/ 102, 103 /) !HOBr
    ParSumMol(38, 1:2 ) = (/ 104, 105 /) !C2H4
    ParSumMol(39, 1:1 ) = (/ 106 /) !CH3OH
    !ParSumMol(40, 1:2 ) = (/ /) !CH3Br
    ParSumMol(41, 1:1 ) = (/ 107 /) !CH3CN
    ParSumMol(42, 1:1 ) = (/ 108 /) !CF4

    ! MolValの設定
    ! 1: Earth, HITRANは地球の同位体比に規格化されている (同位体比 MolVal(:,:,1)=1)
    ! 2: Earth の同位体比
    ! (??: 同位体比を適当に設定)
    !
    !!call SetMolVal(1)

    ! Hitran dataの読み込み ======================Read HITRAN Start
    !
    allocate( MolNum       (1:FileEnd(k_mol)) )
    allocate( IsoNum       (1:FileEnd(k_mol)) )
    allocate( WaveNum0     (1:FileEnd(k_mol)) )
    allocate( Int_Ref      (1:FileEnd(k_mol)) )
    allocate( EinsA        (1:FileEnd(k_mol)) )
    allocate( AirBroadHarf (1:FileEnd(k_mol)) )
    allocate( SelfBroadHarf(1:FileEnd(k_mol)) )
    allocate( LowStatE     (1:FileEnd(k_mol)) )
    allocate( N_air        (1:FileEnd(k_mol)) )
    allocate( delta_air    (1:FileEnd(k_mol)) )

    if( k_mol < 10 ) then   ! 2014/04/15
      write(SnumHi, '(I1)') k_mol
      DnumHi = '0' // SnumHi
    else
      write(DnumHi, '(I2)') k_mol
    end if

    if( DataBase == 'HITRAN1996') then                     ! 2014/05/21
      open(fHi, file='./../data/'// DnumHi //'_HIT96.PAR')
    else if( DataBase == 'HITRAN2008') then
      open(fHi, file='./../data/'// DnumHi //'_hit08.par')
    else if( DataBase == 'HITEMP2010') then
      open(fHi, file='./../data/'// DnumHi //'_HITEMP2010.par')
    else if( DataBase == 'HITRAN2012') then
      open(fHi, file='./../data/'// DnumHi //'_hit12.par')
    else if( DataBase == 'TestData') then                  ! 2016/10/18
      open(fHi, file='./../data/test.par')
    end if

    !open (fHi, file='./data/01_0-50_HITEMP2010.par') ! 25cm-1 check 20131225
    !print *, 'start read HITRAN'
    do k_Hi = 1, FileEnd(k_mol)
      read (fHi, '(I2, I1, F12.6, E10.3, E10.3, F5.4, F5.4, F10.4, F4.2, F8.6)') &
        & MolNum(k_Hi), IsoNum(k_Hi), WaveNum0(k_Hi), Int_Ref(k_Hi), EinsA(k_Hi), &
        & AirBroadHarf(k_Hi), SelfBroadHarf(k_Hi), LowStatE(k_Hi), N_air(k_Hi), delta_air(k_Hi)
      !print *, '====k_Hi:', k_Hi, '===='
      !print *, 'Mol', MolNum(k_Hi)
      !print *, k_Hi, 'Iso', IsoNum(k_Hi)
      !if( (IsoNum(k_Hi) > 0) .or. (IsoNum(k_Hi) <11)) then
      !else
      !  print *, '===WRONG==',k_Hi, IsoNum(k_Hi)
      !  stop
      !end if 
      !print *, 'WN', WaveNum0(k_Hi)
      !print *, 'Int', Int_Ref(k_Hi)
      !print *, 'Eins', EinsA(k_Hi)
      !print *, 'AirB', AirBroadHarf(k_Hi)
      !print *, 'SelfB', SelfBroadHarf(k_Hi)
      !print *, 'LowE', LowStatE(k_Hi)
      !print *, 'Nair', N_air(k_Hi)
      !print *, 'delta', delta_air(k_Hi)
      ! 単位をSIに変換
      WaveNum0(k_Hi) = WaveNum0(k_Hi) * 100.0_8
      Int_Ref(k_Hi) = Int_Ref(k_Hi) * 0.010_8
      AirBroadHarf(k_Hi) = AirBroadHarf(k_Hi) * 9.869233e-4_8 ! 1 [cm-1 atm-1] = 10^2/101325 [m-1 Pa-1]
      SelfBroadHarf(k_Hi) = SelfBroadHarf(k_Hi) * 9.869233e-4_8
      LowStatE(k_Hi) = LowStatE(k_Hi) * 100.0_8
      delta_air(k_Hi) = delta_air(k_Hi) * 9.869233e-4_8

      !print *, WaveNum0(k_Hi)
      !print *, Int_Ref(k_Hi)
      !print *, EinsA(k_Hi)
      !print *, AirBroadHarf(k_Hi)
      !print *, SelfBroadHarf(k_Hi)
      !print *, LowStatE(k_Hi)
      !print *, N_air(k_Hi)
      !print *, delta_air(k_Hi)
    end do
    close(fHi) !=================================Read HITRAN end

    !!
    ! Q_Tの計算をこちらに移動
    call ReadParSum( PATHQ, Temp, Qval )

    ! Q_Tの計算
    !
    do k_iso = 1, 10
      !if(  ( MolVal(k_mol, k_iso, 1) .ne. 0.0_8 ) .and. &
      if(  ( i_IsoARatio(k_iso) .ne. 0.0_8 ) .and. &
        &  ( ParSumMol(k_mol, k_iso) .ne. 0.0_8 ) ) then   ! parsumdatに無いものも除く
        Q_TT(k_mol, k_iso) = func_LinearInterPol( &
          & Qval(0, 1), Qval(0, 2), Qval(ParSumMol(k_mol, k_iso), 1), &
          & Qval(ParSumMol(k_mol, k_iso), 2), Temp )
      end if
      !print *, Q_TT(k_mol, k_iso)
    end do ! k_iso

    !do k_ln = 334320, FileEnd(k_mol)
    !  print *, 'k_ln', k_ln
    !  print *, 'IsoNum', IsoNum(k_ln)
    !  if( (IsoNum(k_ln) > 0) .or. (IsoNum(k_ln) <11)) then
    !  else
    !    print *, '===WRONG==',k_ln, IsoNum(k_ln)
    !    stop
    !  end if 
    !  print *, 'i_IsoARatio',i_IsoARatio(IsoNum(k_ln))
    !end do
    !! NOTE !!
    ! in HITRAN2012 CO2 isotope number '0' exists !!
    ! but this program cannot calculate isotope number '0'

    ! print *, "mixratioH=/0 kmol=", k_mol
    ! 波数範囲にあるデータを読み込む
    !
    AbsorpG(:) = 0.0_8 ! 2014/05/15
    do k_ln = 1, FileEnd(k_mol)

      !print *, '==check'
      !print *, 'k_ln', k_ln
      !print *, 'IsoNum', IsoNum(k_ln)
      !print *, 'i_IsoARatio',i_IsoARatio(IsoNum(k_ln))

      if ( IsoNum(k_ln) .ne. 0 ) then ! 2016/10/21
        !if ( ( MolVal(k_mol, IsoNum(k_ln), 1) .ne. 0.0_8 ) .and. &
        if ( ( i_IsoARatio(IsoNum(k_ln)) .ne. 0.0_8 ) .and. &
          &  ( ParSumMol(k_mol, IsoNum(k_ln)) .ne. 0.0_8 ) ) then   ! parsumdatに無いものも除く
          !print *, 'call calc_absorption'
          call Calc_absorption( Flag_LINESHAPE, CutOffVal, &
            & wmax, MinWaveNum, ResWaveNum, k_mol, Press, Temp, MixRatioH, &
            & WaveNum0(k_ln), Int_Ref(k_ln), AirBroadHarf(k_ln), SelfBroadHarf(k_ln), LowStatE(k_ln), &
            & N_air(k_ln), delta_air(k_ln), i_IsoARatio(IsoNum(k_ln)), &
            & i_IsoQ0(IsoNum(k_ln)), i_IsoMolWt(IsoNum(k_ln)), Q_TT(k_mol, IsoNum(k_ln)), &
            & AbsorpG(0:wmax))
        end if
      end if

    end do ! k_ln

    deallocate( MolNum        )
    deallocate( IsoNum        )
    deallocate( WaveNum0      )
    deallocate( Int_Ref       )
    deallocate( EinsA         )
    deallocate( AirBroadHarf  )
    deallocate( SelfBroadHarf )
    deallocate( LowStatE      )
    deallocate( N_air         )
    deallocate( delta_air     )

  end subroutine linebylineOld

  subroutine Calc_absorption(          &
    & Flag_LINESHAPE, CutOffVal,       & !(in )
    & wmax, MinWaveNum, ResWaveNum,    & !(in )
    & k_mol, Press, Temp, MixRatioH,   & !(in )
    & WaveNum0, Int_Ref, AirBroadHarf, & !(in )
    & SelfBroadHarf, LowStatE,         & !(in )
    & N_air, delta_air, IsoARatio,     & !(in )
    & IsoQ0, IsoMolWt, Q_TT,           & !(in )
    & AbsorpG                          ) !(out)

    integer, intent(in ) :: Flag_LINESHAPE !2016/06/30
    real(8), intent(in ) :: CutOffVal      !2016/10/18
    integer, intent(in ) :: wmax
    real(8), intent(in ) :: MinWaveNum
    real(8), intent(in ) :: ResWaveNum
    integer, intent(in ) :: k_mol
    real(8), intent(in ) :: Press
    real(8), intent(in ) :: Temp
    real(8), intent(in ) :: MixRatioH
    real(8), intent(in ) :: WaveNum0
    real(8), intent(in ) :: Int_Ref
    real(8), intent(in ) :: AirBroadHarf
    real(8), intent(in ) :: SelfBroadHarf
    real(8), intent(in ) :: LowStatE
    real(8), intent(in ) :: N_air
    real(8), intent(in ) :: delta_air
    real(8), intent(in ) :: IsoARatio
    real(8), intent(in ) :: IsoQ0
    real(8), intent(in ) :: IsoMolWt
    real(8), intent(in ) :: Q_TT
    real(8), intent(inout) :: AbsorpG(0:wmax)
    real(8) :: WaveNumS
    real(8) :: Alpha_D, Alpha_L
    real(8) :: BroadHarf
    real(8) :: Intensity
    real(8) :: CalcWidth
    integer :: cutoff = 1
    integer :: k_min, k_max
    integer :: k_w

    !integer :: VOIGT = 2 ! SET voigt profile 2013/07/01
                          ! Humlicek を使う 2014/04/17

    ! pressure shift of wacuum wavenumber
    !
    WaveNumS = WaveNum0 + delta_air * Press * (Temp_Ref/ Temp) ! 温度依存性の導入 2013/05/01
    !print *, 'WNS', WaveNumS

    if(WaveNumS < 0.0_8) then           ! 2013/12/25
      AbsorpG(0:wmax) = AbsorpG(0:wmax) 
    else

      ! 半値幅の計算
      !
      Alpha_D = ( WaveNumS / LightSpeed ) * &
        & ( 2.0_8 * BoltzmannConst * Temp * AvogadroNum/ IsoMolWt )**0.50_8
        !print *, "AlphaD", Alpha_D

      !BroadHarf = SelfBroadHarf * MixRatioH * MolVal_1 + &
      !  & AirBroadHarf * (1.0_8 - MixRatioH * MolVal_1)
      BroadHarf = SelfBroadHarf * MixRatioH + &               ! 2016/10/19
        & AirBroadHarf * (1.0_8 - MixRatioH   )               ! 2016/10/19
      Alpha_L = BroadHarf * Press * ( Temp_Ref / Temp )**N_air
      !print *, "AlphaL", Alpha_L

      ! 計算する波数範囲をcalcWidthとする
      if( cutoff == 1 ) then
        !CalcWidth = 2500.0 ! 2013/08/26
        CalcWidth = CutOffVal ! 2016/10/18

      elseif( cutoff == 2 ) then
        if( Alpha_D .gt. Alpha_L ) then
          CalcWidth = Alpha_D * 5.0_8
        else
          CalcWidth = Alpha_L * 100.0_8
        end if
      end if
 
      ! Intensityの計算
      !
      Intensity = IsoARatio * func_IntensityTempVar( &
        & Temp, IsoQ0, Q_TT, &
        & Int_Ref, WaveNum0, LowStatE ) !WaveNum0
      !print *, "Temp", Temp, "LowStatE", LowStatE, "Intensity", Intensity, "N", N_air, "Q", Q_T, &
      !  & Intensity * func_Voigt( 0.0_8 , ( Alpha_L / Alpha_D )  ) / ( Alpha_D * PI**0.50_8 )
      ! 2013/06/19
   !   print *, "Intensity: ", Intensity
  
      ! スペクトルの計算 ! バグ修正 2013/05/01
      !
      !if( nint((WaveNumS - CalcWidth - MinWaveNum) /ResWaveNum ) .le. 0 ) then
      if( (int((WaveNumS - CalcWidth - MinWaveNum) /ResWaveNum - 1d-16)+1) .le. 0 ) then ! 2016/10/18
        k_min = 0
      else
      !  k_min = nint((WaveNumS - CalcWidth - MinWaveNum) /ResWaveNum )
        k_min = int((WaveNumS - CalcWidth - MinWaveNum) /ResWaveNum - 1d-16) + 1 ! 2016/10/18
      end if
      ! print *, "WaveNumS - CalcWidth - MinWaveNum", WaveNumS - CalcWidth - MinWaveNum, "k_min", k_min
      !if( nint((WaveNumS + CalcWidth - MinWaveNum) /ResWaveNum ) .ge. wmax ) then
      if( int((WaveNumS + CalcWidth - MinWaveNum) /ResWaveNum ) .ge. wmax ) then ! 2016/10/18
        k_max = wmax
      !elseif( nint((WaveNumS + CalcWidth - MinWaveNum) /ResWaveNum ) .le. 0 ) then
      elseif( int((WaveNumS + CalcWidth - MinWaveNum) /ResWaveNum ) .le. 0 ) then ! 2016/10/18
        k_max = -1
      else
      !  k_max = nint((WaveNumS + CalcWidth - MinWaveNum) /ResWaveNum )
        k_max = int((WaveNumS + CalcWidth - MinWaveNum) /ResWaveNum ) ! 2016/10/18
      end if
      ! print *, "WaveNumS + CalcWidth - MinWaveNum", WaveNumS + CalcWidth - MinWaveNum, "k_max", k_max

      ! AbsorpG(:) = 0.0_8 ! 2013/07/19 : これは間違い
      do k_w = k_min, k_max ! in kk, k_mol

        AbsorpG(k_w) = AbsorpG(k_w) +                                 &
          & Intensity *                                               &
          & func_LineShape( Flag_LINESHAPE,                           &
          &                 k_mol,                                    &
          &                 MinWaveNum + ResWavenum * k_w - WaveNumS, &
          &                 Alpha_L,                                  &
          &                 Alpha_D                                   )
        
!        if( VOIGT == 1 ) then ! add func_LineShape 2013/07/08
!          AbsorpG(k_w) = AbsorpG(k_w) + &
!            & Intensity * func_LineShape( k_mol, MinWaveNum + ResWavenum * k_w - WaveNumS ) * &
!            & func_Voigt( ( MinWaveNum + ResWavenum * k_w - WaveNumS )/ Alpha_D, &
!            & ( Alpha_L / Alpha_D )  ) / ( Alpha_D * PI**0.50_8 )
!        elseif( VOIGT == 2 ) then
!          AbsorpG(k_w) = AbsorpG(k_w) + &
!            & Intensity * func_LineShape( k_mol, MinWaveNum + ResWavenum * k_w - WaveNumS ) * &
!            & real(Humlicek( ( MinWaveNum + ResWavenum * k_w - WaveNumS )/ Alpha_D, &
!            & ( Alpha_L / Alpha_D )  ) / ( Alpha_D * PI**0.50_8 ))
!        end if

      end do ! k_w in kk, k_mol

    end if

  end subroutine Calc_absorption

  function func_LineShape( Flag_LINESHAPE, k_mol, & !2016/06/30
    & deltaWN, Alpha_L, Alpha_D) result(a)
    !
    ! Flag_LINESHAPE == 1: voigt profile (all)
    ! Flag_LINESHAPE == 2: voigt profile, CO2 sub-Lorentzian
    ! Flag_LINESHAPE == 3: Lorentz profile (all)
    ! Flag_LINESHAPE == 4: Lorentz profile, CO2 sub-Lorentzian
    !
    integer :: Flag_LINESHAPE
    integer :: k_mol
    real(8) :: deltaWN
    real(8) :: Alpha_L
    real(8) :: Alpha_D
    real(8) :: a

    if( Flag_LINESHAPE == 1) then

      !a = func_LineShapeX(1, deltaWN) * &
      a = real(Humlicek( ( deltaWN )/ Alpha_D, &
        & ( Alpha_L / Alpha_D )  ) / ( Alpha_D * PI**0.50_8 ))

    elseif(Flag_LINESHAPE == 2) then

      a = real(Humlicek( ( deltaWN )/ Alpha_D, &
        & ( Alpha_L / Alpha_D )  ) / ( Alpha_D * PI**0.50_8 ))

      if( k_mol == 2) then
        a = func_LineShapeX(2, k_mol, deltaWN) * a
      end if

    elseif( Flag_LINESHAPE == 3) then

      !a = func_LineShapeX(1, deltaWN) * &
      a = Lorentz(deltaWN, Alpha_L)

    elseif(Flag_LINESHAPE == 4) then

      a = Lorentz(deltaWN, Alpha_L)

      if( k_mol == 2) then
        a = func_LineShapeX(2, k_mol, deltaWN) * a
      end if

    end if

  end function func_LineShape

  function func_LineShapeX( FLAG, k_mol, deltaWN ) result(a) ! 2013/07/08 !2016/06/30

    integer :: FLAG
    integer :: k_mol
    real(8) :: deltaWN
    real(8) :: a
    real(8) :: WaveNumCutOff = 2500.0_8
    real(8) :: Coeff

    if(FLAG == 1) then

      a = 1.0_8

    elseif(FLAG == 2) then
      !
      ! CO2 sub-Lorentzian
      ! Pollack et al. (1993)
      !
      if( abs(deltaWN) < 300.0_8 ) then ! |deltaWN| < 3 cm-1
        a = 1.0_8
      elseif( abs(deltaWN) < 1000.0_8 ) then ! 3 cm-1 <= |deltaWN| < 10 cm-1
        a = 1.35_8 * exp(-abs(deltaWN)/1000.0_8)
      elseif( abs(deltaWN) < 12000.0_8 ) then ! 10 cm-1 <= |deltaWN| < 120 cm-1
        a = 0.614_8 * exp(-abs(deltaWN)/4700.0_8)
      else
        a = 0.0_8
      end if

    elseif(FLAG == 3) then
      !
      ! LBLRTM line function (??)
      !
      Coeff = deltaWN/WaveNumCutOff

      if(k_mol == 2) then
        a = 1.0_8 - Coeff * Coeff * ( Coeff * Coeff - 2.0_8 )
      else
        a = 1.0_8 - Coeff * Coeff
      end if

    else

      print *, 'func_LineShapeX, FLAG is non valid value, FLAG:', FLAG
      print *, 'stop'
      stop

    end if

  end function func_LineShapeX

  function func_IntensityTempVar( Temp, Q_Ref, Q_T, Int_Ref, WaveNum0, LowStatE ) result(a)

    real(8) :: Temp, Q_Ref, Q_T, Int_Ref, WaveNum0, LowStatE, a

    a = Int_Ref * exp( ( SecRadConst * LowStatE ) * ( 1.0_8 / Temp_Ref - 1.0_8 / Temp ) ) &
      & * ((1.0_8 -exp(-SecRadConst * WaveNum0 / Temp     )) / &
      &    (1.0_8 -exp(-SecRadConst * WaveNum0 / Temp_Ref )) ) &
      & * ( Q_Ref / Q_T )

  end function func_IntensityTempVar

  function func_LinearInterPol( x1, x2, y1, y2, x0 ) result(a)

    real(8) :: x1, x2, y1, y2, x0, a

    if ( (x2-x1) .ne. 0.0_8 ) then
      a = ( y2 - y1 ) * ( x0 - x1 ) / ( x2 - x1 ) + y1
    else
      print *, "値が不適切です"
    end if

  end function func_LinearInterPol

  function Lorentz( deltaWN, alpha_L) result(a) ! 2016/06/30
    ! 
    ! Lorentz line shape function calculation 
    ! Rothman et al. (1998) (A14)
    !
    real(8) :: deltaWN
    real(8) :: alpha_L
    real(8) :: a

    a = alpha_L/ ( PI * ( alpha_L * alpha_L + deltaWN * deltaWN ) )

  end function Lorentz

  function Humlicek(X, Y) result(a)  ! 2013/07/01
    !
    ! voigt function calulation by Humlicek 1982
    !
    real(8) :: X, Y
    real(8) :: S
    complex :: T, U
    complex :: a

    T = cmplx(Y, -X)
    S = ABS(X) + Y
    if(S .LT. 15. ) GOTO 1

      a = T*.5641896/(.5 + T*T)
      return
1   if(S .LT. 5.5) GOTO 2
      U = T*T
      a = T*(1.410474 + U*.5641896)/(.75+U*(3.+U))
      return
2   if(Y .LT. .195*ABS(X) -.176) GOTO 3
      a = (16.4955 + T*(20.20933 + T*(11.96482 + T*(3.778987 + T*.5642236))))/&
      & (16.4955 + T*(38.82363 + T*(39.27121 + T*(21.69274 + T*(6.699398+T)))))
      return
3   U = T*T
      a = CEXP(U) - T*(36183.31 - U*(3321.9905-U*(1540.787-U*(219.0313-U*(35.76683-U*(1.320522-U*.56419))))))/&
      & (32066.6-U*(24322.84-U*(9022.228-U*(2186.181-U*(364.2191-U*(61.57037-U*(1.841439-U)))))))
      return
  end function Humlicek  ! 2013/07/01

  function func_Voigt( x, y ) result(a)
    !
    ! voigt function calculation by Kuntz 1997/Ruyten 2004
    ! 
    real(8) :: x, y, a
    real(8) :: a1, b1, a2, b2
    real(8) :: a3, b3, c3, d3, a4, b4, c4, d4
    real(8) :: a5, b5, c5, d5, e5, a6, b6, c6, d6, e6
    real(8) :: a7, b7, c7, d7, e7, f7, g7, h7, o7, p7, q7, r7, s7, t7
    real(8) :: a8, b8, c8, d8, e8, f8, g8, h8, o8, p8, q8, r8, s8, t8 

    if ( ( abs(x) + y ) .ge. 15.0_8 ) then         !>=確認
      a1 = 0.2820948_8 * y + 0.5641896 * y**3
      b1 = 0.5641896_8 * y
      a2 = 0.5_8 + y**2 + y**4
      b2 = -1.0_8 + 2.0_8 * y**2

      a = ( a1 + b1 * x**2) / ( a2 + b2 * x**2 + x**4 )
      
    elseif ( ( ( abs(x) + y ) .ge. 5.5_8 ) .and. &      !>=確認
      &  ( ( abs(x) + y ) .lt. 15.0_8)       ) then
      a3 =   1.05786_8 * y + 4.65456_8 * y**3 + 3.10304_8 * y**5 + 0.56419_8 * y**7
      b3 =     2.962_8 * y + 0.56419_8 * y**3 + 1.69257_8 * y**5
      c3 = - 2.53885_8 * y + 1.69257_8 * y**3
      d3 =   0.56419_8 * y
      a4 = 0.5625_8 + 4.5_8 * y**2 + 10.5_8 * y**4 + 6.0_8 * y**6 + y**8
      b4 =   -4.5_8 + 9.0_8 * y**2 +  6.0_8 * y**4 + 4.0_8 * y**6
      c4 =   10.5_8 - 6.0_8 * y**2 +  6.0_8 * y**4
      d4 =   -6.0_8 + 4.0_8 * y**2

      a = ( a3 + b3 * x**2 + c3 * x**4 + d3 * x**6 ) / &
        & ( a4 + b4 * x**2 + c4 * x**4 + d4 * x**6 + x**8 )

    elseif ( ( ( abs(x) + y ) .lt. 5.5_8 ) .and. &
      &  ( y .gt. ( 0.195_8 * abs(x) - 0.176_8 ) ) ) then
      a5 = 272.102_8 + 973.778_8 * y + 1629.76_8 * y**2 + 1678.33_8 * y**3 + 1174.8_8 * y**4 + &
        & 581.746 * y**5 + 204.501_8 * y**6 + 49.5213_8 * y**7 + 7.55895_8 * y**8 + 0.564224_8 * y**9
      b5 = -60.5644_8 - 2.34403_8 * y + 220.843 * y**2 + 336.364_8 * y**3 + 247.198_8 * y**4 + &
        & 100.705_8 * y**5 + 22.6778_8 * y**6 + 2.25689_8 * y**7
      c5 = 4.58029_8 + 18.546_8 * y + 42.5683_8 * y**2 + 52.8454_8 * y**3 + 22.6798_8 * y**4 + &
        & 3.38534_8 * y**5
      d5 = -0.128922_8 + 1.66203_8 * y + 7.56186_8 * y**2 + 2.25689_8 * y**3
      e5 = 0.000971457_8 + 0.564224_8 * y
      a6 = 272.102_8 + 1280.83_8 * y + 2802.87_8 * y**2 + 3764.97 * y**3 + 3447.63_8 * y**4 + &
        & 2256.98_8 * y**5 + 1074.41 * y**6 + 369.199_8 * y**7 + 88.2674_8 * y**8 + 13.3988_8 * y**9 + y**10
      b6 = 211.678_8 + 902.306_8 * y + 1758.34_8 * y**2 + 2037.31_8 * y**3 + 1549.68_8 * y**4 + &
        & 793.427_8 * y**5 + 266.299_8 * y**6 + 53.5952_8 * y**7 + 5.0_8 * y**8
      c6 = 78.866_8 + 308.186_8 * y + 497.302_8 * y**2 + 479.258_8 * y**3 + 269.292_8 * y**4 + &
        & 80.3928_8 * y**5 + 10.0_8 * y**6
      d6 = 22.0353_8 + 55.0293_8 * y + 92.7568_8 * y**2 + 53.5952_8 * y**3 + 10.0_8 * y**4
      e6 = 1.49645_8 + 13.3988_8 * y + 5.0_8 * y**2

      a = ( a5 + b5 * x**2 + c5 * x**4 + d5 * x**6 + e5 * x**8 ) / &
        & ( a6 + b6 * x**2 + c6 * x**4 + d6 * x**6 + e6 * x**8 + x**10 )

    elseif ( ( ( abs(x) + y ) .lt. 5.5_8 ) .and. &
      &  ( y .lt. ( 0.195_8 * abs(x) - 0.176_8 ) ) ) then
      a7 = 1.16028e+9_8 * y - 9.86604e+8_8 * y**3 + 4.56662e+8_8 * y**5 - 1.53575e+8_8 * y**7 + &
        & 4.08168e+7_8 * y**9 - 9.69463e+6_8 * y**11 + 1.6841e+6_8 * y**13 - 320772.0_8 * y**15 + &
        & 40649.2_8 * y**17 - 5860.68_8 * y**19 + 571.687_8 * y**21 - 72.9359_8 * y**23 + &
        & 2.35944_8 * y**25 - 0.56419_8 * y**27
      b7 = -5.60505e+8_8 * y - 9.85386e+8_8 * y**3 + 8.06985e+8_8 * y**5 - 2.91876e+8_8 * y**7 + &
        & 8.64829e+7_8 * y**9 - 7.72359e+6_8 * y**11 + 3.59915e+6_8 * y**13 - 234417.0_8 * y**15 + &
        & 45251.3_8 * y**17 - 2269.19_8 * y**19 - 234.143_8 * y**21 + 23.0312_8 * y**23 &
        & -7.33447_8 * y**25
      c7 = -6.51523e+8_8 * y + 2.47157e+8_8 + y**3 + 2.94262e+8_8 * y**5 - 2.04467e+8_8 * y**7 + &
        & 2.29302e+7_8 * y**9 - 2.3818e+7_8 * y**11 + 576054.0_8 * y**13 + 98079.1_8 * y**15 &
        & -25338.3_8 * y**17 + 1097.77_8 * y**19 + &
        & 97.6203_8 * y**21 - 44.0068_8 * y**23
      d7 = -2.63894e+8_8 * y + 2.70167e+8_8 * y**3 - 9.96224e+7_8 * y**5 - 4.15013e+7_8 * y**7 + &
        & 3.83112e+7_8 * y**9 + 2.2404e+6_8 * y**11 - 303569.0_8 * y**13 - 66431.2_8 * y**15 + &
        & 8381.97_8 * y**17 + 228.563_8 * y**19 - 161.358_8 * y**21
      e7 = -6.31771e+7_8 * y + 1.40677e+8_8 * y**3 + 5.56965e+6_8 * y**5 + 2.46201e+7_8 * y**7 + &
        & 468142.0_8 * y**9 - 1.003e+6_8 * y**11 - 66212.1_8 * y**13 + 23507.6_8 * y**15 + &
        & 296.38_8 * y**17 - 403.396_8 * y**19
      f7 = -1.69846e+7 * y + 4.07382e+6_8 * y**3 - 3.32896e+7_8 * y**5 - 1.93114e+6_8 * y**7 &
        & -934717.0_8 * y**9 + 8820.94_8 * y**11 + 37544.8_8 * y**13 + 125.591_8 * y**15 &
        & -726.113_8 * y**17
      g7 = -1.23165e+6_8 * y + 7.52883e+6_8 * y**3 - 900010.0_8 * y**5 - 186682.0_8 * y**7 + 79902.5_8 * y**9 + &
        & 37371.9_8 * y**11 - 260.198_8 * y**13 - 968.15_8 * y**15
      h7 = -610622.0_8 * y + 86407.6_8 * y**3 + 153468.0_8 * y**5 + 72520.9_8 * y**7 + 23137.1_8 * y**9 &
        & -571.645_8 * y**11 - 968.15_8 * y**13
      o7 = -23586.5 * y + 49883.8_8 * y**3 + 26538.5_8 * y**5 + 8073.15_8 * y**7 - 575.164_8 * y**9 &
        & -726.113 * y**11
      p7 = -8009.1_8 * y + 2198.86_8 * y**3 + 953.655_8 * y**5 - 352.467_8 * y**7 - 403.396_8 * y**9
      q7 = -622.056_8 * y - 271.202_8 * y**3 - 134.792_8 * y**5 - 161.358_8 * y**7
      r7 = -77.0535_8 * y - 29.7896_8 * y**3 -44.0068_8 * y**5
      s7 = -2.92264_8 * y - 7.33447_8 * y**3
      t7 = -0.56419_8 * y

      a8 = 1.02827e+9_8 - 1.5599e+9_8 * y**2 + 1.17022e+9_8 * y**4 -5.79099e+8_8 * y**6 + 2.11107e+8_8 * y**8 &
        & -6.11148e+7_8 * y**10 + 1.44647e+7_8 * y**12 - 2.85721e+6_8 * y**14 + 483737.0_8 * y**16 - 70946.1_8 * y**18 + &
        & 9504.65_8 * y**20 - 955.194_8 * y**22 + 126.532_8 * y**24 - 3.68288_8 * y**26 + 1.0_8 * y**28
      b8 = 1.5599e+9_8 - 2.28855e+9_8 * y**2 + 1.66421e+9_8 * y**4 - 7.53828e+8_8 * y**6 + 2.89676e+8_8 * y**8 &
        & -7.01358e+7_8 * y**10 + 1.39465e+7_8 * y**12 - 2.84954e+6_8 * y**14 + 498334.0_8 * y**16 - 55600.0_8 * y**18 + &
        & 3058.26_8 * y**20 + 533.254_8 * y**22 - 40.5117_8 * y**24 + 14.0_8 * y**26
      c8 = 1.17022e+9_8 - 1.66421e+9_8 * y**2 + 1.06002e+9_8 * y**4 - 6.60078e+8_8 * y**6 + 6.33496e+7_8 * y**8 &
        & - 4.60396e+7_8 * y**10 + 1.4841e+7_8 * y**12 - 1.06352e+6_8 * y**14 - 217801.0_8 * y**16 + 48153.3_8 * y**18 &
        & - 1500.17_8 * y**20 - 198.876_8 * y**22 + 91.0_8 * y**24
      d8 = 5.79099e+8_8 - 7.53828e+8_8 * y**2 + 6.60078e+8_8 * y**4 + 5.40367e+7_8 * y**6 + 1.99846e+8_8 * y**8 &
        & - 6.87656e+6_8 * y**10 - 6.89002e+6_8 * y**12 + 280428.0_8 * y**14 + 161461.0_8 * y**16 - 16493.7_8 * y**18 &
        & - 567.164_8 * y**20 + 364.0_8 * y**22
      e8 = 2.11107e+8_8 - 2.89676e+8_8 * y**2 + 6.33496e+7_8 * y**4 - 1.99846e+8_8 * y**6 - 5.01017e+7_8 * y**8 &
        & - 5.25722e+6_8 * y**10 + 1.9547e+6_8 * y**12 + 240373.0_8 * y**14 - 55582.0_8 * y**16 - 1012.79_8 * y**18 &
        & + 1001.0_8 * y**20
      f8 = 6.11148e+7_8 - 7.01358e+7_8 * y**2 + 4.60396e+7_8 * y**4 - 6.87656e+6_8 * y**6 + 5.25722e+6_8 * y**8 &
        & + 3.04316e+6_8 * y**10 + 123052.0_8 * y**12 - 106663.0_8 * y**14 - 1093.82_8 * y**16 + 2002.0_8 * y**18
      g8 = 1.44647e+7_8 - 1.39465e+7_8 * y**2 + 1.4841e+7_8 * y**4 + 6.89002e+6_8 * y**6 + 1.9547e+6 * y**8 &
        & - 123052.0_8 * y**10 - 131337.0_8 * y**12 - 486.14_8 * y**14 + 3003.0_8 * y**16
      h8 = 2.85721e+6_8 - 2.84954e+6_8 * y**2 + 1.06352e+6_8 * y**4 + 280428.0_8 * y**6 - 240373.0_8 * y**8 &
        & - 106663.0_8 * y**10 + 486.14_8 * y**12 + 3432.0_8 * y**14
      o8 = 483737.0_8 - 498334.0_8 * y**2 - 217801.0_8 * y**4 - 161461.0_8 * y**6 - 55582.0_8 * y**8 &
        & + 1093.82_8 * y**10 + 3003.0_8 * y**12
      p8 = 70946.1_8 - 55600.0_8 * y**2 - 48153.3_8 * y**4 - 16493.7_8 * y**6 + 1012.79_8 * y**8 &
        & + 2002.0_8 * y**10
      q8 = 9504.65_8 - 3058.26_8 * y**2 - 1500.17_8 * y**4 + 567.164_8 * y**6 + 1001.0_8 * y**8
      r8 = 955.194_8 + 533.254_8 * y**2 + 198.876_8 * y**4 + 364.0_8 * y**6
      s8 = 126.532_8 + 40.5117_8 * y**2 + 91.0_8 * y**4
      t8 = 3.68288_8 + 14.0_8 * y**2

      a = exp( y**2 - x**2 ) * cos( 2.0_8 * x * y ) - &
        & (a7 + b7 * x**2 + c7 * x**4 + d7 * x**6 + e7 * x**8 + f7 * x**10 + g7 * x**12 + h7 * x**14 + &
        &  o7 * x**16 + p7 * x**18 + q7 * x**20 + r7 * x**22 + s7 * x**24 + t7 * x**26 ) / &
        & (a8 + b8 * x**2 + c8 * x**4 + d8 * x**6 + e8 * x**8 + f8 * x**10 + g8 * x**12 + h8 * x**14 + &
        &  o8 * x**16 + p8 * x**18 + q8 * x**20 + r8 * x**22 + s8 * x**24 + t8 * x**26 + x**28 )

    end if

  end function func_Voigt

  subroutine ReadParSum( PATHQ, Temp, Qval )

    character(99), intent(in ) :: PATHQ
    real(8), intent(in ) :: Temp
    real(8), intent(out) :: Qval(0:108, 1:2)
    integer :: count, ksr
    integer :: fi = 14

    !print *, 'Temp', Temp

    !if( (Temp < 70.0_8) .or. (Temp > 3000.0_8) ) then ! 2014/04/15
    !  print *, 'Temperature is out of range(70-3000[K]), T:', Temp
    !  print *, 'stop (ReadParSum)'
    !  stop
    !end if
    if( (Temp > 3000.0_8) ) then ! 2015/01/29
      print *, 'Temperature is out of range(<=3000[K]), T:', Temp
      print *, 'stop (ReadParSum)'
      stop
    end if

    if( Temp < 70.0_8 ) then   !2015/01/29
      count = 0                !2015/01/29
    else                       !2015/01/29

      count = int(Temp - 70.0_8)

    end if

    Qval = 0.0_8

    open(fi, file = PATHQ) ! 2016/07/04
!    open(fi, file = './src/optdep/parsum.dat') ! 2015/03/31

    do ksr = 1, count + 1
      read(fi, *)
      !read(fi, '()')  !2014/05/20 No Good !
    end do

    if( Temp == 3000.0_8 ) then ! 2014/04/15
      read(fi, *) Qval(:, 1)
      Qval(:, 2) = 0.0_8 
    elseif( Temp < 70.0 ) then ! 2015/02/12
      Qval(:, 1) = 0.0_8       ! 2015/02/12
      read(fi, *) Qval(:, 2)   ! 2015/02/12
    else
      read(fi, *) Qval
    end if

    close(fi)

  end subroutine ReadParSum

  subroutine SetIsoMolWt( im_IsoMolWt )
    ! set molecular weight of isotope [kg mol-1]
    !
    real(8), intent(out) :: im_IsoMolWt(1:10, 1:50)

    im_IsoMolWt = 0.0_8

    ! H2O (1)
    im_IsoMolWt(1,1) = 18.010565E-3_8
    im_IsoMolWt(2,1) = 20.014811E-3_8
    im_IsoMolWt(3,1) = 19.014780E-3_8
    im_IsoMolWt(4,1) = 19.016740E-3_8
    im_IsoMolWt(5,1) = 21.020985E-3_8
    im_IsoMolWt(6,1) = 20.020956E-3_8

    ! CO2 (2)
    im_IsoMolWt(1,2) = 43.989830E-3_8
    im_IsoMolWt(2,2) = 44.993185E-3_8
    im_IsoMolWt(3,2) = 45.994076E-3_8
    im_IsoMolWt(4,2) = 44.994045E-3_8
    im_IsoMolWt(5,2) = 46.997431E-3_8
    im_IsoMolWt(6,2) = 45.997400E-3_8
    im_IsoMolWt(7,2) = 47.998322E-3_8
    im_IsoMolWt(8,2) = 46.998291E-3_8
    im_IsoMolWt(9,2) = 49.001675E-3_8
    im_IsoMolWt(10,2) = 48.001646E-3_8

    ! O3 (3)
    im_IsoMolWt(1,3) = 47.984745E-3_8
    im_IsoMolWt(2,3) = 49.988991E-3_8
    im_IsoMolWt(3,3) = 49.988991E-3_8
    im_IsoMolWt(4,3) = 48.988960E-3_8
    im_IsoMolWt(5,3) = 48.988960E-3_8

    ! N2O (4)
    im_IsoMolWt(1,4) = 44.001062E-3_8
    im_IsoMolWt(2,4) = 44.998096E-3_8
    im_IsoMolWt(3,4) = 44.998096E-3_8
    im_IsoMolWt(4,4) = 46.005308E-3_8
    im_IsoMolWt(5,4) = 45.005278E-3_8

    ! CO (5)
    im_IsoMolWt(1,5) = 27.994915E-3_8
    im_IsoMolWt(2,5) = 28.998270E-3_8
    im_IsoMolWt(3,5) = 29.999161E-3_8
    im_IsoMolWt(4,5) = 28.999130E-3_8
    im_IsoMolWt(5,5) = 31.002516E-3_8
    im_IsoMolWt(6,5) = 30.002485E-3_8

    ! CH4 (6)
    im_IsoMolWt(1,6) = 16.031300E-3_8
    im_IsoMolWt(2,6) = 17.034655E-3_8
    im_IsoMolWt(3,6) = 17.037475E-3_8
    im_IsoMolWt(4,6) = 18.040830E-3_8

    ! O2 (7)
    im_IsoMolWt(1,7) = 31.989830E-3_8
    im_IsoMolWt(2,7) = 33.994076E-3_8
    im_IsoMolWt(3,7) = 32.994045E-3_8

    ! NO (8)
    im_IsoMolWt(1,8) = 29.997989E-3_8
    im_IsoMolWt(2,8) = 30.995023E-3_8
    im_IsoMolWt(3,8) = 32.002234E-3_8

    ! SO2 (9)
    im_IsoMolWt(1,9) = 63.961901E-3_8
    im_IsoMolWt(2,9) = 65.957695E-3_8

    ! NO2 (10)
    im_IsoMolWt(1,10) = 45.992904E-3_8

    ! NH3 (11)
    im_IsoMolWt(1,11) = 17.026549E-3_8
    im_IsoMolWt(2,11) = 18.023583E-3_8

    ! HNO3 (12)
    im_IsoMolWt(1,12) = 62.995644E-3_8

    ! OH (13)
    im_IsoMolWt(1,13) = 17.002740E-3_8
    im_IsoMolWt(2,13) = 19.006986E-3_8
    im_IsoMolWt(3,13) = 18.008915E-3_8

    ! HF (14)
    im_IsoMolWt(1,14) = 20.006229E-3_8

    ! HCl (15)
    im_IsoMolWt(1,15) = 35.976678E-3_8
    im_IsoMolWt(2,15) = 37.973729E-3_8

    ! HBr (16)
    im_IsoMolWt(1,16) = 79.926160E-3_8
    im_IsoMolWt(2,16) = 81.924115E-3_8

    ! HI (17)
    im_IsoMolWt(1,17) = 127.912297E-3_8

    ! ClO (18)
    im_IsoMolWt(1,18) = 50.963768E-3_8
    im_IsoMolWt(2,18) = 52.960819E-3_8

    ! OCS (19)
    im_IsoMolWt(1,19) = 59.966986E-3_8
    im_IsoMolWt(2,19) = 61.962780E-3_8
    im_IsoMolWt(3,19) = 60.970341E-3_8
    im_IsoMolWt(4,19) = 60.966371E-3_8
    im_IsoMolWt(5,19) = 61.971231E-3_8

    ! H2CO (20)
    im_IsoMolWt(1,20) = 30.010565E-3_8
    im_IsoMolWt(2,20) = 31.013920E-3_8
    im_IsoMolWt(3,20) = 32.014811E-3_8

    ! HOCl (21)
    im_IsoMolWt(1,21) = 51.971593E-3_8
    im_IsoMolWt(2,21) = 53.968644E-3_8

    ! N2 (22)
    im_IsoMolWt(1,22) = 28.006147E-3_8

    ! HCN (23)
    im_IsoMolWt(1,23) = 27.010899E-3_8
    im_IsoMolWt(2,23) = 28.014254E-3_8
    im_IsoMolWt(3,23) = 28.007933E-3_8

    ! CH3Cl (24)
    im_IsoMolWt(1,24) = 49.992328E-3_8
    im_IsoMolWt(2,24) = 51.989379E-3_8

    ! H2O2 (25)
    im_IsoMolWt(1,25) = 34.005480E-3_8

    ! C2H2 (26)
    im_IsoMolWt(1,26) = 26.015650E-3_8
    im_IsoMolWt(2,26) = 27.019005E-3_8
    im_IsoMolWt(3,26) = 27.021825E-3_8

    ! C2H6 (27)
    im_IsoMolWt(1,27) = 30.046950E-3_8
    im_IsoMolWt(2,27) = 31.050305E-3_8

    ! PH3 (28)
    im_IsoMolWt(1,28) = 33.997238E-3_8

    ! COF2 (29)
    im_IsoMolWt(1,29) = 65.991722E-3_8

    ! SF6 (30)
    im_IsoMolWt(1,30) = 145.962492E-3_8

    ! H2S (31)
    im_IsoMolWt(1,31) = 33.987721E-3_8
    im_IsoMolWt(2,31) = 35.983515E-3_8
    im_IsoMolWt(3,31) = 34.987105E-3_8

    ! HCOOH (32)
    im_IsoMolWt(1,32) = 46.005480E-3_8

    ! HO2 (33)
    im_IsoMolWt(1,33) = 32.997655E-3_8

    ! O (34)
    im_IsoMolWt(1,34) = 15.994915E-3_8
 
    ! ClONO2 (35)
    im_IsoMolWt(1,35) = 96.956672E-3_8
    im_IsoMolWt(2,35) = 98.953723E-3_8

    ! NO+ (36)
    im_IsoMolWt(1,36) = 29.997989E-3_8

    ! HOBr (37)
    im_IsoMolWt(1,37) = 95.921076E-3_8
    im_IsoMolWt(2,37) = 97.919027E-3_8

    ! C2H4 (38)
    im_IsoMolWt(1,38) = 28.031300E-3_8
    im_IsoMolWt(2,38) = 29.034655E-3_8

    ! CH3OH (39)
    im_IsoMolWt(1,39) = 32.026215E-3_8

    ! CH3Br (40)
    im_IsoMolWt(1,40) = 93.941811E-3_8
    im_IsoMolWt(2,40) = 95.939764E-3_8

    ! CH3CN (41)
    im_IsoMolWt(1,41) = 41.026549E-3_8

    ! CF4 (42)
    im_IsoMolWt(1,42) = 87.993616E-3_8

    ! CS (46)
    im_IsoMolWt(1,46) = 43.971036E-3_8
    im_IsoMolWt(2,46) = 45.966787E-3_8
    im_IsoMolWt(3,46) = 44.974368E-3_8
    im_IsoMolWt(4,46) = 44.970399E-3_8

  end subroutine SetIsoMolWt

  subroutine SetIsoQ0Val( im_IsoQ0Val )
    ! set Q0 value (269K)
    !
    real(8), intent(out) :: im_IsoQ0Val(1:10, 1:50)

    im_IsoQ0Val = 0.0_8

    ! H2O (1)
    im_IsoQ0Val(1,1) = 1.7464E+02_8
    im_IsoQ0Val(2,1) = 1.7511E+02_8
    im_IsoQ0Val(3,1) = 1.0479E+03_8
    im_IsoQ0Val(4,1) = 8.5901E+02_8
    im_IsoQ0Val(5,1) = 8.7519E+02_8
    im_IsoQ0Val(6,1) = 5.2204E+03_8

    ! CO2 (2)
    im_IsoQ0Val(1,2) = 2.8694E+02_8
    im_IsoQ0Val(2,2) = 5.7841E+02_8
    im_IsoQ0Val(3,2) = 6.0948E+02_8
    im_IsoQ0Val(4,2) = 3.5527E+03_8
    im_IsoQ0Val(5,2) = 1.2291E+03_8
    im_IsoQ0Val(6,2) = 7.1629E+03_8
    im_IsoQ0Val(7,2) = 3.2421E+02_8
    im_IsoQ0Val(8,2) = 3.7764E+03_8
    im_IsoQ0Val(9,2) = 6.5350E+02_8
    im_IsoQ0Val(10,2) = 7.4229E+03_8

    ! O3 (3)
    im_IsoQ0Val(1,3) = 3.4838E+03_8
    im_IsoQ0Val(2,3) = 7.4657E+03_8
    im_IsoQ0Val(3,3) = 3.6471E+03_8
    im_IsoQ0Val(4,3) = 4.3331E+04_8
    im_IsoQ0Val(5,3) = 2.1405E+04_8

    ! N2O (4)
    im_IsoQ0Val(1,4) = 5.0018E+03_8
    im_IsoQ0Val(2,4) = 3.3619E+03_8
    im_IsoQ0Val(3,4) = 3.4586E+03_8
    im_IsoQ0Val(4,4) = 5.3147E+03_8
    im_IsoQ0Val(5,4) = 3.0971E+04_8

    ! CO (5)
    im_IsoQ0Val(1,5) = 1.0712E+02_8
    im_IsoQ0Val(2,5) = 2.2408E+02_8
    im_IsoQ0Val(3,5) = 1.1247E+02_8
    im_IsoQ0Val(4,5) = 6.5934E+02_8
    im_IsoQ0Val(5,5) = 2.3582E+02_8
    im_IsoQ0Val(6,5) = 1.3809E+03_8

    ! CH4 (6)
    im_IsoQ0Val(1,6) = 5.9045E+02_8
    im_IsoQ0Val(2,6) = 1.1808E+03_8
    im_IsoQ0Val(3,6) = 4.7750E+03_8
    im_IsoQ0Val(4,6) = 1.5980E+03_8

    ! O2 (7)
    im_IsoQ0Val(1,7) = 2.1577E+02_8
    im_IsoQ0Val(2,7) = 4.5230E+02_8
    im_IsoQ0Val(3,7) = 2.6406E+03_8

    ! NO (8)
    im_IsoQ0Val(1,8) = 1.1421E+03_8
    im_IsoQ0Val(2,8) = 7.8926E+02_8
    im_IsoQ0Val(3,8) = 1.2045E+03_8

    ! SO2 (9)
    im_IsoQ0Val(1,9) = 6.3403E+03_8
    im_IsoQ0Val(2,9) = 6.3689E+03_8

    ! NO2 (10)
    im_IsoQ0Val(1,10) = 1.3578E+04_8

    ! NH3 (11)
    im_IsoQ0Val(1,11) = 1.7252E+03_8
    im_IsoQ0Val(2,11) = 1.1527E+03_8

    ! HNO3 (12)
    im_IsoQ0Val(1,12) = 2.1412E+05_8

    ! OH (13)
    im_IsoQ0Val(1,13) = 8.0362E+01_8
    im_IsoQ0Val(2,13) = 8.0882E+01_8
    im_IsoQ0Val(3,13) = 2.0931E+02_8

    ! HF (14)
    im_IsoQ0Val(1,14) = 4.1466E+01_8

    ! HCl (15)
    im_IsoQ0Val(1,15) = 1.6066E+02_8
    im_IsoQ0Val(2,15) = 1.6089E+02_8

    ! HBr (16)
    im_IsoQ0Val(1,16) = 2.0018E+02_8
    im_IsoQ0Val(2,16) = 2.0024E+02_8

    ! HI (17)
    im_IsoQ0Val(1,17) = 3.8900E+02_8

    ! ClO (18)
    im_IsoQ0Val(1,18) = 3.2746E+03_8
    im_IsoQ0Val(2,18) = 3.3323E+03_8

    ! OCS (19)
    im_IsoQ0Val(1,19) = 1.2210E+03_8
    im_IsoQ0Val(2,19) = 1.2535E+03_8
    im_IsoQ0Val(3,19) = 2.4842E+03_8
    im_IsoQ0Val(4,19) = 4.9501E+03_8
    im_IsoQ0Val(5,19) = 1.3137E+03_8

    ! H2CO (20)
    im_IsoQ0Val(1,20) = 2.8467E+03_8
    im_IsoQ0Val(2,20) = 5.8376E+03_8
    im_IsoQ0Val(3,20) = 2.9864E+03_8

    ! HOCl (21)
    im_IsoQ0Val(1,21) = 1.9274E+04_8
    im_IsoQ0Val(2,21) = 1.9616E+04_8

    ! N2 (22)
    im_IsoQ0Val(1,22) = 4.6598E+02_8

    ! HCN (23)
    im_IsoQ0Val(1,23) = 8.9529E+02_8
    im_IsoQ0Val(2,23) = 1.8403E+03_8
    im_IsoQ0Val(3,23) = 6.2141E+02_8

    ! CH3Cl (24)
    im_IsoQ0Val(1,24) = 1.1583E+05_8
    im_IsoQ0Val(2,24) = 1.1767E+05_8

    ! H2O2 (25)
    im_IsoQ0Val(1,25) = 9.8198E+03_8

    ! C2H2 (26)
    im_IsoQ0Val(1,26) = 4.1403E+02_8
    im_IsoQ0Val(2,26) = 1.6562E+03_8
    im_IsoQ0Val(3,26) = 1.5818E+03_8

    ! C2H6 (27)
    im_IsoQ0Val(1,27) = 7.0881E+04_8
    im_IsoQ0Val(2,27) = 8.0001E+04_8

    ! PH3 (28)
    im_IsoQ0Val(1,28) = 3.2486E+03_8

    ! COF2 (29)
    im_IsoQ0Val(1,29) = 7.0044E+04_8

    ! SF6 (30)
    im_IsoQ0Val(1,30) = 1.6233E+06_8

    ! H2S (31)
    im_IsoQ0Val(1,31) = 5.0307E+02_8
    im_IsoQ0Val(2,31) = 5.0435E+02_8
    im_IsoQ0Val(3,31) = 2.0149E+03_8

    ! HCOOH (32)
    im_IsoQ0Val(1,32) = 3.9133E+04_8

    ! HO2 (33)
    im_IsoQ0Val(1,33) = 4.3004E+03_8

    ! O (34)
    im_IsoQ0Val(1,34) = 6.7212E+00_8

    ! ClONO2 (35)
    im_IsoQ0Val(1,35) = 4.7884E+06_8
    im_IsoQ0Val(2,35) = 4.9102E+06_8

    ! NO+ (36)
    im_IsoQ0Val(1,36) = 3.1168E+02_8

    ! HOBr (37)
    im_IsoQ0Val(1,37) = 2.8339E+04_8
    im_IsoQ0Val(2,37) = 2.8238E+04_8

    ! C2H4 (38)
    im_IsoQ0Val(1,38) = 1.1041E+04_8
    im_IsoQ0Val(2,38) = 4.5197E+04_8

    ! CH3OH (39)
    im_IsoQ0Val(1,39) = 3.5314E+04_8

    ! CH3Br (40)
    im_IsoQ0Val(1,40) = 2.0762E+04_8
    im_IsoQ0Val(2,40) = 2.0849E+04_8

    ! CH3CN (41)
    im_IsoQ0Val(1,41) = 8.8670E+04_8

    ! CF4 (42)
    im_IsoQ0Val(1,42) = 1.2127E+05_8

    ! CS (46)
    im_IsoQ0Val(1,46) = 2.5653E+02_8
    im_IsoQ0Val(2,46) = 2.6069E+02_8
    im_IsoQ0Val(3,46) = 5.4345E+02_8
    im_IsoQ0Val(4,46) = 1.0346E+03_8

  end subroutine SetIsoQ0Val

  subroutine SetIsoMRatioE( im_IsoMRatio )
    ! set Q0 value (269K)
    !
    real(8), intent(out) :: im_IsoMRatio(1:10, 1:50)

    im_IsoMRatio = 0.0_8

    ! H2O (1)
    im_IsoMRatio(1,1) = 0.997317E+00_8
    im_IsoMRatio(2,1) = 1.99983E-03_8
    im_IsoMRatio(3,1) = 3.71884E-04_8
    im_IsoMRatio(4,1) = 3.10693E-04_8
    im_IsoMRatio(5,1) = 6.23003E-07_8
    im_IsoMRatio(6,1) = 1.15853E-07_8

    ! CO2 (2)
    im_IsoMRatio(1,2) = .984204E+00_8
    im_IsoMRatio(2,2) = 1.10574E-02_8
    im_IsoMRatio(3,2) = 3.94707E-03_8
    im_IsoMRatio(4,2) = 7.33989E-04_8
    im_IsoMRatio(5,2) = 4.43446E-05_8
    im_IsoMRatio(6,2) = 8.24623E-06_8
    im_IsoMRatio(7,2) = 3.95734E-06_8
    im_IsoMRatio(8,2) = 1.47180E-06_8
    im_IsoMRatio(9,2) = 4.44600E-08_8
    im_IsoMRatio(10,2) = 1.65354E-08_8

    ! O3 (3)
    im_IsoMRatio(1,3) = .992901E+00_8
    im_IsoMRatio(2,3) = 3.98194E-03_8
    im_IsoMRatio(3,3) = 1.99097E-03_8
    im_IsoMRatio(4,3) = 7.40475E-04_8
    im_IsoMRatio(5,3) = 3.70237E-04_8

    ! N2O (4)
    im_IsoMRatio(1,4) = .990333E+00_8
    im_IsoMRatio(2,4) = 3.64093E-03_8
    im_IsoMRatio(3,4) = 3.64093E-03_8
    im_IsoMRatio(4,4) = 1.98582E-03_8
    im_IsoMRatio(5,4) = 3.69280E-04_8

    ! CO (5)
    im_IsoMRatio(1,5) = .986544E+00_8
    im_IsoMRatio(2,5) = 1.10836E-02_8
    im_IsoMRatio(3,5) = 1.97822E-03_8
    im_IsoMRatio(4,5) = 3.67867E-04_8
    im_IsoMRatio(5,5) = 2.22250E-05_8
    im_IsoMRatio(6,5) = 4.13292E-06_8

    ! CH4 (6)
    im_IsoMRatio(1,6) = .988274E+00_8
    im_IsoMRatio(2,6) = 1.11031E-02_8
    im_IsoMRatio(3,6) = 6.15751E-04_8
    im_IsoMRatio(4,6) = 6.91785E-06_8

    ! O2 (7)
    im_IsoMRatio(1,7) = .995262E+00_8
    im_IsoMRatio(2,7) = 3.99141E-03_8
    im_IsoMRatio(3,7) = 7.42235E-04_8

    ! NO (8)
    im_IsoMRatio(1,8) = .993974E+00_8
    im_IsoMRatio(2,8) = 3.65431E-03_8
    im_IsoMRatio(3,8) = 1.99312E-03_8

    ! SO2 (9)
    im_IsoMRatio(1,9) = .945678E+00_8
    im_IsoMRatio(2,9) = 4.19503E-02_8

    ! NO2 (10)
    im_IsoMRatio(1,10) = .991616E+00_8

    ! NH3 (11)
    im_IsoMRatio(1,11) = .995872E+00_8
    im_IsoMRatio(2,11) = 3.66129E-03_8

    ! HNO3 (12)
    im_IsoMRatio(1,12) = .989110E+00_8

    ! OH (13)
    im_IsoMRatio(1,13) = .997473E+00_8
    im_IsoMRatio(2,13) = 2.00014E-03_8
    im_IsoMRatio(3,13) = 1.55371E-04_8

    ! HF (14)
    im_IsoMRatio(1,14) = .999844E+00_8

    ! HCl (15)
    im_IsoMRatio(1,15) = .757587E+00_8
    im_IsoMRatio(2,15) = .242257E+00_8

    ! HBr (16)
    im_IsoMRatio(1,16) = .506781E+00_8
    im_IsoMRatio(2,16) = .493063E+00_8

    ! HI (17)
    im_IsoMRatio(1,17) = .999844E+00_8

    ! ClO (18)
    im_IsoMRatio(1,18) = .755908E+00_8
    im_IsoMRatio(2,18) = .241720E+00_8

    ! OCS (19)
    im_IsoMRatio(1,19) = .937395E+00_8
    im_IsoMRatio(2,19) = 4.15828E-02_8
    im_IsoMRatio(3,19) = 1.05315E-02_8
    im_IsoMRatio(4,19) = 7.39908E-03_8
    im_IsoMRatio(5,19) = 1.87967E-03_8

    ! H2CO (20)
    im_IsoMRatio(1,20) = .986237E+00_8
    im_IsoMRatio(2,20) = 1.10802E-02_8
    im_IsoMRatio(3,20) = 1.97761E-03_8

    ! HOCl (21)
    im_IsoMRatio(1,21) = .755790E+00_8
    im_IsoMRatio(2,21) = .241683E+00_8

    ! N2 (22)
    im_IsoMRatio(1,22) = .992687E+00_8

    ! HCN (23)
    im_IsoMRatio(1,23) = .985114E+00_8
    im_IsoMRatio(2,23) = 1.10676E-02_8
    im_IsoMRatio(3,23) = 3.62174E-03_8

    ! CH3Cl (24)
    im_IsoMRatio(1,24) = .748937E+00_8
    im_IsoMRatio(2,24) = .239491E+00_8

    ! H2O2 (25)
    im_IsoMRatio(1,25) = .994952E+00_8

    ! C2H2 (26)
    im_IsoMRatio(1,26) = .977599E+00_8
    im_IsoMRatio(2,26) = 2.19663E-02_8
    im_IsoMRatio(3,26) = 3.04550E-04_8

    ! C2H6 (27)
    im_IsoMRatio(1,27) = .976990E+00_8
    im_IsoMRatio(2,27) = 2.19526E-02_8

    ! PH3 (28)
    im_IsoMRatio(1,28) = .999533E+00_8

    ! COF2 (29)
    im_IsoMRatio(1,29) = .986544E+00_8

    ! SF6 (30)
    im_IsoMRatio(1,30) = .950180E+00_8

    ! H2S (31)
    im_IsoMRatio(1,31) = .949884E+00_8
    im_IsoMRatio(2,31) = 4.21369E-02_8
    im_IsoMRatio(3,31) = 7.49766E-03_8

    ! HCOOH (32)
    im_IsoMRatio(1,32) = .983898E+00_8

    ! HO2 (33)
    im_IsoMRatio(1,33) = .995107E+00_8

    ! O (34)
    im_IsoMRatio(1,34) = .997628E+00_8

    ! ClONO2 (35)
    im_IsoMRatio(1,35) = .749570E+00_8
    im_IsoMRatio(2,35) = .239694E+00_8

    ! NO+ (36)
    im_IsoMRatio(1,36) = .993974E+00_8

    ! HOBr (37)
    im_IsoMRatio(1,37) = .505579E+00_8
    im_IsoMRatio(2,37) = .491894E+00_8

    ! C2H4 (38)
    im_IsoMRatio(1,38) = .977294E+00_8
    im_IsoMRatio(2,38) = 2.19595E-02_8

    ! CH3OH (39)
    im_IsoMRatio(1,39) = .985930E+00_8

    ! CH3Br (40)
    im_IsoMRatio(1,40) = .500995E+00_8
    im_IsoMRatio(2,40) = .487433E+00_8

    ! CH3CN (41)
    im_IsoMRatio(1,41) = .973866E+00_8

    ! CF4 (42)
    im_IsoMRatio(1,42) = .988890E+00_8

    ! CS (46)
    im_IsoMRatio(1,46) = .939624E+00_8
    im_IsoMRatio(2,46) = .416817E-01_8
    im_IsoMRatio(3,46) = .105565E-01_8
    im_IsoMRatio(4,46) = .741668E-02_8

  end subroutine SetIsoMRatioE


end module mod_LineByLine
