!= 雲なしモデル
!
!= No cloud model
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: cloud_none.f90,v 1.1 2015/01/29 12:19:41 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module cloud_none
  !
  != 雲なしモデル
  !
  != No cloud model
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! This is a no cloud model, i.e., condensation rate is converted to 
  ! precipitation rate. 
  !
  !== Procedures List
  !
!!$  ! RadiationFluxDennouAGCM :: 放射フラックスの計算
!!$  ! ------------            :: ------------
!!$  ! RadiationFluxDennouAGCM :: Calculate radiation flux
  !
  !== NAMELIST
  !
  ! NAMELIST#cloud_none_nml
  !

  ! モジュール引用 ; USE statements

  !
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数.
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数.
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数.
                             ! Number of vertical level

  implicit none

  private


  ! 公開手続き
  ! Public procedure
  !
  public :: CloudNoneWithIce
  public :: CloudNone
  public :: CloudNoneInit


  ! 公開変数
  ! Public variables
  !


  ! 非公開変数
  ! Private variables
  !
  logical , save        :: FlagSnow
                           ! A flag for snow

  logical , save :: FlagPRCPPC
  logical , save :: FlagPRCPEvap

  real(DP), save :: PRCPArea
    !                           a_p
  real(DP), save :: PRCPEvapArea
    !                           A = max( a_p - a, 0 )


  logical, save :: cloud_none_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  character(*), parameter:: module_name = 'cloud_none'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: cloud_none.f90,v 1.1 2015/01/29 12:19:41 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine CloudNoneWithIce(               &
    & xyr_Press, xyz_Press,                  & ! (in   )
    & xyz_DQRainDt, xyz_DQSnowDt,            & ! (in   )
    & xyz_Temp, xyz_QH2OVap,                 & ! (inout)
    & xy_SurfRainFlux, xy_SurfSnowFlux       & ! (out  )
    & )

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime            ! $ \Delta t $ [s]

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only:  &
      & CpDry,            &
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure
      & Grav
                              ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only: &
      & CloudUtilsPRCPStepPC1Grid, &
      & CloudUtilsPRCPEvap1Grid

    real(DP), intent(in   ) :: xyr_Press       ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_Press       ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQRainDt    ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQSnowDt    ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_Temp        ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OVap     ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(out  ) :: xy_SurfRainFlux ( 0:imax-1, 1:jmax )
    real(DP), intent(out  ) :: xy_SurfSnowFlux ( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_TempB   ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP) :: xyz_QH2OVapB( 0:imax-1, 1:jmax, 1:kmax )
    real(DP) :: xyz_QH2OLiqB( 0:imax-1, 1:jmax, 1:kmax )
    real(DP) :: xyz_QH2OSolB( 0:imax-1, 1:jmax, 1:kmax )
    real(DP) :: xyz_DelMass ( 0:imax-1, 1:jmax, 1:kmax )

    real(DP) :: xyz_QH2OLiqTMP( 0:imax-1, 1:jmax, 1:kmax )
    real(DP) :: xyz_QH2OSolTMP( 0:imax-1, 1:jmax, 1:kmax )

    integer  :: i
    integer  :: j
    integer  :: k


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. cloud_none_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! Save variables
    !
    xyz_TempB    = xyz_Temp
    xyz_QH2OVapB = xyz_QH2OVap
    xyz_QH2OLiqB = xyz_DQRainDt * ( 2.0_DP * DelTime )
    xyz_QH2OSolB = xyz_DQSnowDt * ( 2.0_DP * DelTime )


    do k = 1, kmax
      xyz_DelMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    ! Freezing and melting switching at temperature of TempCondWater
    xy_SurfRainFlux = 0.0_DP
    xy_SurfSnowFlux = 0.0_DP
    do j = 1, jmax
      do i = 0, imax-1
        do k = kmax, 1, -1

          if ( FlagPRCPPC ) then
            call CloudUtilsPRCPStepPC1Grid(  &
              & xyr_Press(i,j,k-1), xyr_Press(i,j,k),       & ! (in   )
              & xyz_Temp(i,j,k),                            & ! (inout)
              & xy_SurfRainFlux(i,j), xy_SurfSnowFlux(i,j)  & ! (out  )
              & )
          end if

          if ( FlagPRCPEvap ) then
            call CloudUtilsPRCPEvap1Grid(                               &
              & xyz_Press(i,j,k), xyr_Press(i,j,k-1), xyr_Press(i,j,k), & ! (in)
              & PRCPArea, PRCPEvapArea,                                 & ! (in)
              & xyz_Temp(i,j,k), xyz_QH2OVap(i,j,k),                    & ! (inout)
              & xy_SurfRainFlux(i,j), xy_SurfSnowFlux(i,j)              & ! (inout)
              & )
          end if

          xy_SurfRainFlux(i,j) = xy_SurfRainFlux(i,j) &
            & + xyz_DQRainDt(i,j,k) * xyz_DelMass(i,j,k)
          xy_SurfSnowFlux(i,j) = xy_SurfSnowFlux(i,j) &
            & + xyz_DQSnowDt(i,j,k) * xyz_DelMass(i,j,k)

        end do
      end do
    end do


    xyz_QH2OLiqTMP = 0.0_DP
    xyz_QH2OSolTMP = 0.0_DP
    call CloudNoneConsChk(                             &
!!$      & FlagIncludeIcePhaseChange,                           &
      & xyr_Press,                                               &
      & xyz_TempB, xyz_QH2OVapB, xyz_QH2OLiqB  , xyz_QH2OSolB  , &
      & xyz_Temp , xyz_QH2OVap , xyz_QH2OLiqTMP, xyz_QH2OSolTMP, &
      & xy_SurfRainFlux, xy_SurfSnowFlux                         &
      & )


  end subroutine CloudNoneWithIce

  !--------------------------------------------------------------------------------------

  subroutine CloudNone(                      &
    & xyr_Press, xyz_Press,                  & ! (in   )
    & xyz_DQRainDt,                          & ! (in   )
    & xyz_Temp, xyz_QH2OVap,                 & ! (inout)
    & xy_SurfRainFlux                        & ! (out  )
    & )

    real(DP), intent(in   ) :: xyr_Press       ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_Press       ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQRainDt    ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_Temp        ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OVap     ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(out  ) :: xy_SurfRainFlux ( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_DQSnowDt    ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP) :: xy_SurfSnowFlux ( 0:imax-1, 1:jmax )


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. cloud_none_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( FlagPRCPPC ) then
      call MessageNotify( 'E', module_name, 'FlagPRCPPC has to be .false., if CloudNone subroutine is used.' )
    end if


    xyz_DQSnowDt = 0.0_DP

    call CloudNoneWithIce(               &
      & xyr_Press, xyz_Press,                  & ! (in   )
      & xyz_DQRainDt, xyz_DQSnowDt,            & ! (in   )
      & xyz_Temp, xyz_QH2OVap,                 & ! (inout)
      & xy_SurfRainFlux, xy_SurfSnowFlux       & ! (out  )
      & )


  end subroutine CloudNone

  !--------------------------------------------------------------------------------------

  subroutine CloudNoneConsChk(                             &
!!$    & FlagIncludeIcePhaseChange,                           &
    & xyr_Press,                                           &
    & xyz_TempB, xyz_QH2OVapB, xyz_QH2OLiqB, xyz_QH2OSolB, &
    & xyz_Temp , xyz_QH2OVap , xyz_QH2OLiq , xyz_QH2OSol , &
    & xy_Rain, xy_Snow                                     &
    & )


    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime            ! $ \Delta t $ [s]

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, & 
                              ! $ g $ [m s-2]. 
                              ! 重力加速度. 
                              ! Gravitational acceleration
      & CpDry, &
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure
      & LatentHeat, &
                              ! $ L $ [J kg-1] . 
                              ! 凝結の潜熱. 
                              ! Latent heat of condensation
      & LatentHeatFusion
                              ! $ L $ [J kg-1] .
                              ! 融解の潜熱.
                              ! Latent heat of fusion

!!$    logical , intent(in) :: FlagIncludeIcePhaseChange
    real(DP), intent(in) :: xyr_Press   (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(in) :: xyz_TempB   (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyz_QH2OVapB(0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyz_QH2OLiqB(0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyz_QH2OSolB(0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyz_Temp    (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyz_QH2OVap (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyz_QH2OLiq (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyz_QH2OSol (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xy_Rain     (0:imax-1, 1:jmax)
    real(DP), intent(in) :: xy_Snow     (0:imax-1, 1:jmax)

    ! Local variables
    !
    real(DP) :: xyz_DelMass(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xy_Val(0:imax-1, 1:jmax)
    real(DP) :: xy_SumB(0:imax-1, 1:jmax)
    real(DP) :: xy_Sum(0:imax-1, 1:jmax)
    real(DP) :: xy_Ratio(0:imax-1, 1:jmax)
    integer  :: i
    integer  :: j
    integer  :: k


    do k = 1, kmax
      xyz_DelMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    xy_Sum = 0.0_DP
    do k = kmax, 1, -1
      xy_Val =   CpDry * xyz_TempB(:,:,k)               &
        &      + LatentHeat * xyz_QH2OVapB(:,:,k)       &
        &      - LatentHeatFusion * xyz_QH2OSolB(:,:,k)
      xy_Sum = xy_Sum + xy_Val * xyz_DelMass(:,:,k)
    end do

    xy_SumB = xy_Sum

    xy_Sum = 0.0_DP
    do k = kmax, 1, -1
      xy_Val =   CpDry * xyz_Temp (:,:,k)               &
        &      + LatentHeat * xyz_QH2OVap (:,:,k)       &
        &      - LatentHeatFusion * xyz_QH2OSol (:,:,k)
      xy_Sum = xy_Sum + xy_Val * xyz_DelMass(:,:,k)
    end do
!!$    if ( FlagIncludeIcePhaseChange ) then
      xy_Sum = xy_Sum - LatentHeatFusion * xy_Snow * 2.0_DP * DelTime
!!$    end if

    xy_Ratio = ( xy_Sum - xy_SumB ) / ( xy_Sum + 1.0d-100 )
    do j = 1, jmax
      do i = 0, imax-1
        if ( abs( xy_Ratio(i,j) ) > 1.0d-10 ) then
          call MessageNotify( 'M', module_name, 'Modified condensate static energy is not conserved, %f.', d = (/ xy_Ratio(i,j) /) )
        end if
      end do
    end do



    xy_Sum = 0.0_DP
    do k = kmax, 1, -1
      xy_Val = xyz_QH2OVapB(:,:,k) + xyz_QH2OLiqB(:,:,k) + xyz_QH2OSolB(:,:,k)
      xy_Sum = xy_Sum + xy_Val * xyz_DelMass(:,:,k)
    end do

    xy_SumB = xy_Sum

    xy_Sum = 0.0_DP
    do k = kmax, 1, -1
      xy_Val = xyz_QH2OVap (:,:,k) + xyz_QH2OLiq (:,:,k) + xyz_QH2OSol (:,:,k)
      xy_Sum = xy_Sum + xy_Val * xyz_DelMass(:,:,k)
    end do
    xy_Sum = xy_Sum + ( xy_Rain + xy_Snow ) * 2.0_DP * DelTime

    xy_Ratio = ( xy_Sum - xy_SumB ) / ( xy_Sum + 1.0d-100 )
    do j = 1, jmax
      do i = 0, imax-1
        if ( abs( xy_Ratio(i,j) ) > 1.0d-10 ) then
          call MessageNotify( 'M', module_name, 'H2O mass is not conserved, %f.', d = (/ xy_Ratio(i,j) /) )
        end if
      end do
    end do


  end subroutine CloudNoneConsChk

  !--------------------------------------------------------------------------------------

  subroutine CloudNoneInit( &
    & ArgFlagSnow             &
    & )

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtilsInit


    ! 宣言文 ; Declaration statements
    !

    logical, intent(in) :: ArgFlagSnow


    character(STRING) :: CloudCoverMethod
    character(STRING) :: SnowMethod

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /cloud_none_nml/ &
      & FlagPRCPPC,          &
      & FlagPRCPEvap,        &
      & PRCPArea,            &
      & PRCPEvapArea
          !
          ! デフォルト値については初期化手続 "cloud_none#CloudNoneInit"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "cloud_none#CloudNoneInit" for the default values.
          !

    ! 実行文 ; Executable statement
    !

    if ( cloud_none_inited ) return


    FlagSnow = ArgFlagSnow


    ! デフォルト値の設定
    ! Default values settings
    !
    FlagPRCPPC          = .false.
    FlagPRCPEvap        = .false.
!!$    PRCPEvapArea        = 0.5_DP
    PRCPArea            = 1.0_DP
!!$    PRCPArea            = 0.5_DP
    PRCPEvapArea        = 1.0_DP
!!$    PRCPEvapArea        = 0.5_DP


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = cloud_none_nml,           & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if



    ! Initialization of modules used in this module
    !

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    call CloudUtilsInit( &
      & FlagSnow         &
      & )


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
!!$    call HistoryAutoAddVariable( 'EffCloudCover', &
!!$      & (/ 'lon ', 'lat ', 'time' /), &
!!$      & 'effective cloud cover', '1' )



    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'FlagPRCPPC          = %b', l = (/ FlagPRCPPC /) )
    call MessageNotify( 'M', module_name, 'FlagPRCPEvap        = %b', l = (/ FlagPRCPEvap /) )
    call MessageNotify( 'M', module_name, 'PRCPArea            = %f', d = (/ PRCPArea /) )
    call MessageNotify( 'M', module_name, 'PRCPEvapArea        = %f', d = (/ PRCPEvapArea /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    cloud_none_inited = .true.

  end subroutine CloudNoneInit

  !--------------------------------------------------------------------------------------

end module cloud_none
