module pbl_shcu

contains
  subroutine pbl_shcu_run(                                               &
    fb_surf, ustar, pstar,                             &
    z_tq, u, v, t, th, exner, p_theta_levels, &
    q, qcl, qcf, q1, frac_gauss,          &
    zhpar,                                             &
    frac, wb_ng)

    use pp_vardef
    use pp_phys_const, only: epsilon, one_minus_epsilon, hlatnt, hls, cp, gdvcp, &
      & e0cw, tetn1w, tetn2w, tetn3w, c_virtual, grav, rdvcp, tkelvn, &
      & rd, pi, one_third
    use pbl_grid, only: nz, shcu_levels
    use pbl_parm, only: wb_ng_max
    use pbl_const, only: timestep
    use pp_monit, only: pp_monit_store

    implicit none

    real(r_size), intent(in) :: fb_surf
    ! buoyancy flux at the surface
    real(r_size), intent(in) :: ustar
    ! surface friction velocity
    real(r_size), intent(in) :: pstar
    ! surface pressure (Pa)

    real(r_size), intent(in) :: z_tq(nz)

    real(r_size), intent(in) :: u(nz)
    ! U at pressure points
    real(r_size), intent(in) :: v(nz)
    ! V at pressure points
    real(r_size), intent(in) :: t(nz)
    ! temperature at full levels
    real(r_size), intent(in) :: th(nz)
    real(r_size), intent(in) :: exner(nz)
    real(r_size), intent(in) :: p_theta_levels(nz)
    ! pressure at full levels (Pa)
    real(r_size), intent(in) :: q(nz)
    ! specific humidity at full levels
    real(r_size), intent(in) :: qcl(nz)
    ! liquid water content at full levels
    real(r_size), intent(in) :: qcf(nz)
    ! frozen water content at full levels
    real(r_size), intent(in) :: q1(nz)
    ! normalized excess of water
    real(r_size), intent(in) :: frac_gauss(nz)
    ! cloud fraction derived with Gaussian distribution
    ! function

    real(r_size), intent(inout) :: zhpar

    real(r_size), intent(out) :: frac(nz)
    ! cloud fraction including that by convection
    real(r_size), intent(out) :: wb_ng(nz)
    ! Non-gradint buoyancy flux due to the skewness

    integer(4) :: k

    integer(4) :: k_par
    ! level for start of parcel ascent
    integer(4) :: ktpar
    ! highest full level below inversion (at ZHPAR)
    integer(4) :: k_neut
    ! level of neutral parcel buoyancy
    integer(4) :: ktinv
    ! top level of inversion
    integer(4) :: k_lcl
    ! level of lifting condensation
    integer(4) :: interp_inv
    ! flag to interpolated inversion heights (1=yes)
    integer(4) :: topbl
    ! 1 => top of bl reached
    ! 2 => max allowable height reached

    real(r_size) :: lcrcp
    ! Evaporation-to-dT conversion factor.
    real(r_size) :: lsrcp
    ! Sublimation-to-dT conversion factor.
    real(r_size) :: grcp

    real(r_size) :: virt_factor
    ! Vfac = 1+0.61qv - qcl - qcf
    real(r_size) :: z_surf
    ! height of surface layer
    real(r_size) :: w_s
    ! velocity scale
    real(r_size) :: thv_sd
    ! standard deviation of thv in surface layer
    real(r_size) :: dqsatdt
    real(r_size) :: qsatfac
    ! saturation coefficients in buoyancy parameters
    real(r_size) :: qc_env
    ! environment liquid water
    real(r_size) :: vap_press
    ! Vapour pressure.
    real(r_size) :: t_lcl
    ! temperature of LCL
    real(r_size) :: th_par
    ! theta of parcel
    real(r_size) :: t_par
    ! temperature of parcel
    real(r_size) :: dpar_bydz
    ! parcel thv gradient
    real(r_size) :: denv_bydz
    ! environement thv gradient
    real(r_size) :: gamma_fa
    ! free atmospheric lapse rate
    real(r_size) :: gamma_cld
    ! cloud layer lapse rate
    real(r_size) :: wb_scale
    ! buoyancy flux scaling (m2/s3)
    real(r_size) :: w_cld
    ! cloud layer velocity scale (m/s)
    real(r_size) :: z_cld
    ! cloud layer depth (m)
    real(r_size) :: dz_inv_cu_rec
    ! reconstructed inversion thickness above Cu
    real(r_size) :: vscalsq_incld
    ! incloud squared velocity scale
    real(r_size) :: m_base
    ! cloud base mass flux (m/s)
    real(r_size) :: z_pr
    real(r_size) :: ze_pr
    ! scaled height
    real(r_size) :: zpr_top
    ! inversion top in scaled coordinate
    real(r_size) :: f_ng
    ! non-gradient shape functions
    real(r_size) :: fnn
    ! entrainment factor gN
    real(r_size) :: z0
    real(r_size) :: z1
    real(r_size) :: z2
    real(r_size) :: z3
    ! heights for polynomial interpolation
    real(r_size) :: d0
    real(r_size) :: d1
    real(r_size) :: d2
    real(r_size) :: d3
    ! values for polynomial interpolation
    real(r_size) :: a2
    real(r_size) :: a3
    real(r_size) :: xi
    ! work variables for polynomial interpolation
    real(r_size) :: a_poly
    real(r_size) :: b_poly
    real(r_size) :: c_poly
    ! coefficients in polynomial interpolation
    real(r_size) :: ri
    ! Richardson number
    real(r_size) :: grid_int
    ! THV integral over inversion
    real(r_size) :: zhdisc
    ! height of subgrid interpolated inversion
    real(r_size) :: lrcp_c
    ! Latent heat over heat capacity
    real(r_size) :: l_heat
    ! Latent heat
    real(r_size) :: frcu
    ! cloud fraction due to convection
    real(r_size) :: thl_par
    ! parcel thl
    real(r_size) :: qw_par
    ! parcel qw
    real(r_size) :: sl_par
    ! parcel static energy
    real(r_size) :: th_ref
    ! reference theta for parcel ascent
    real(r_size) :: th_par_kp1
    ! parcel theta at level below
    real(r_size) :: p_lcl
    ! pressure of LCL
    real(r_size) :: thv_pert
    ! threshold for parcel thv
    real(r_size) :: z_lcl
    ! height of LCL
    real(r_size) :: zh
    ! boundary layer depth (from RI)
    real(r_size) :: zhpar_old
    ! height of cloud-top on previous timestep
    real(r_size) :: zhpar_max
    ! Maximum allowed height for cloud-top
    ! (to limit growth rate of boundary layer)
    real(r_size) :: w_star
    ! sub-cloud layer velocity scale (m/s)
    real(r_size) :: cape
    ! convective available potential energy (m2/s2)
    real(r_size) :: dbdz_inv
    ! buoyancy gradient across inversion
    real(r_size) :: dz_inv_cu
    ! inversion thickness above Cu
    real(r_size) :: qsat
    ! saturated water mixing ratio
    real(r_size) :: t_ref
    ! reference temperature

    real(r_size) :: weight1
    real(r_size) :: weight2
    real(r_size) :: weight3


    real(r_size) :: dthvdz(shcu_levels)
    ! gradient of THV at half levels

    real(r_size) :: dthvdzm(shcu_levels)

    real(r_size) :: thl(shcu_levels)
    ! liquid water potential tempeature
    real(r_size) :: tl(shcu_levels)
    ! liquid water tempeature
    real(r_size) :: qw(shcu_levels)
    ! total water spec humidity
    real(r_size) :: thvl(shcu_levels)
    ! virtual thl
    real(r_size) :: thv(shcu_levels)
    ! virtual th
    real(r_size) :: thv_par(shcu_levels)
    ! virtual th for parcel
    real(r_size) :: qc_par(shcu_levels)
    ! parcel liquid water

    real(r_size) :: u_p(shcu_levels)     ! u at rho level 
    real(r_size) :: v_p(shcu_levels)     ! v at rho level
    real(r_size) :: p_rho_levels(shcu_levels)
    real(r_size) :: z_uv(shcu_levels)


    logical :: topinv
    ! indicates top of inversion being reached
    logical :: topprof
    ! indicates top of ascent being reached
    logical :: above_lcl
    ! indicates being above the LCL

    real(r_size), parameter :: a_parcel = 0.2_r_size
    real(r_size), parameter :: b_parcel = 3.26_r_size
    real(r_size), parameter :: max_t_grad = 1.0e-3_r_size
    real(r_size), parameter :: ric = 0.25_r_size


    p_rho_levels(1) = 0.5_r_size * (pstar + p_theta_levels(1))
    u_p(1) = 0.5_r_size * u(1)
    v_p(1) = 0.5_r_size * v(1)
    z_uv(1) = 0.5_r_size * z_tq(1)

    do k = 2, shcu_levels
      p_rho_levels(k) = 0.5_r_size & 
        &         * (p_theta_levels(k) + p_theta_levels(k-1))

      u_p(k) = 0.5_r_size * (u(k) + u(k-1))
      v_p(k) = 0.5_r_size * (v(k) + v(k-1))
      z_uv(k) = 0.5_r_size * (z_tq(k) + z_tq(k-1))
    end do


    lcrcp = hlatnt / cp
    lsrcp = hls / cp
    grcp = gdvcp

    zhpar_old = zhpar
    ! Limit boundary layer growth rate to 0.14 m/s
    ! (approx 500m/hour)
    zhpar_max = min( z_tq(shcu_levels-1),                  &
      zhpar_old+timestep*0.14_r_size )
    zh = 0.0_r_size

    do k = 1, shcu_levels
      ! initialise cumulus cloud fraction to zero
      !exner(k) = (p_theta_levels(k) / pref) ** kappa
      !th(k) = t(k) / exner(k)
      thl(k) = th(k)                                      &
        - (lcrcp*qcl(k) + lsrcp*qcf(k)) / exner(k)
      qw(k) = q(k) + qcl(k) + qcf(k)
      thvl(k)= thl(k) * ( 1. + c_virtual*qw(k) )
      virt_factor = 1. + c_virtual*q(k) - qcl(k) - qcf(k)
      THv(k) = th(k) * virt_factor
      thv_par(k) = THv(k)   ! default for stable bls
      wb_ng(k) = 0.0_r_size
      frac(k) = frac_gauss(k)
      tl(k) = t(k) - lcrcp*qcl(k) - lsrcp*qcf(k)
    end do

    do k = 2, shcu_levels
      dthvdz(k) = THv(k) - THv(k-1)
    end do

    do k = 3, shcu_levels
      weight1 = z_uv(k) - z_uv(k-1)
      weight2 = z_tq(k-1)- z_uv(k-1)
      weight3 = z_uv(k) - z_tq(k-1)
      dthvdzm(k) = (weight2 * dthvdz(k)                       &
        + weight3 * dthvdz(k-1)) / weight1
    end do

    k = 2
    dthvdzm(k) = dthvdz(k)

    do k = 2, shcu_levels
      ri = (u_p(k)-u_p(k-1))**2                               &
        &               +(v_p(k)-v_p(k-1))**2
      ri = (grav*(z_uv(k)-z_uv(k-1))                             &
        &             *dthvdzm(k)/THv(k)) / max( 1.e-14_r_size, ri )
      if ( ri > ric .and. zh == 0.0_r_size ) then
        zh =z_uv(k)
      end if
      qc_par(k) = 0.0_r_size
    end do
    !-----------------------------------------------------------------------
    ! 1. Set up parcel
    !-----------------------------------------------------------------------
    ! Start parcel ascent from grid-level above top of surface layer, taken
    ! to be at a height, z_surf, given by 0.1*ZH
    !-----------------------------------------------------------------------
    k_par = 1
    zhpar = zh  ! initialise to bl depth (from RI)
    k_lcl = 1
    if (fb_surf >= 0.0) then
      z_surf = 0.1_r_size * zh
      do while ( z_uv(k_par) < z_surf .and.                  &
                                ! not reached Z_SURF
        thvl(k_par+1) <= thvl(k_par) )
        ! not reached inversion
        k_par = k_par + 1
      end do
      w_s = ( fb_surf * zh + ustar** 3 ) ** one_third
      thv_sd = 1.93_r_size * fb_surf * THv(k_par)              &
        / ( grav * w_s )
      thl_par = thl(k_par)
      qw_par  = qw(k_par)
      sl_par = tl(k_par) + grcp * z_tq(k_par)
      !-----------------------------------------------------------------------
      ! Calculate temperature and pressure of lifting condensation level
      ! using approximations from Bolton (1980)
      !-----------------------------------------------------------------------
      vap_press = q(k_par) *                                 &
        p_theta_levels(k_par) / ( 100.0_r_size * EPSILON )
      if (vap_press >= 0.0_r_size) then
        t_lcl = 55.0_r_size + 2840.0_r_size &
          & / ( 3.5_r_size * log(t(k_par))         &
          - log(vap_press) - 4.805_r_size )
        p_lcl =  p_theta_levels(k_par) *                &
          ( t_lcl / t(k_par) ) ** (1.0_r_size / rdvcp)
      else
        p_lcl = pstar
      end if
      ! K_LCL is model level BELOW the lifting condensation level
      k_lcl = 1
      do k = 2, shcu_levels
        if (p_rho_levels(k) > p_lcl) then
          k_lcl = k - 1
        end if
      end do
      z_lcl = z_uv(k_lcl+1)                             &
        + ( z_uv(k_lcl)-z_uv(k_lcl+1) )         &
        * ( p_rho_levels(k_lcl+1) - p_lcl)          &
        / ( p_rho_levels(k_lcl+1)                        &
        - p_rho_levels(k_lcl) )
      z_lcl = max( z_uv(1), z_lcl )
      !-----------------------------------------------------------------------
      ! Threshold on parcel buoyancy for ascent, THV_PERT, is related to
      ! standard deviation of thv in surface layer
      !-----------------------------------------------------------------------
      thv_pert = max( a_parcel,                                   &
        min( max_t_grad * zh, b_parcel * thv_sd ) )

      th_ref = thl_par
      th_par_kp1 = thl_par
    else
      ! dummy
      th_ref = thl(1)
      z_lcl = z_uv(1)
    end if   ! test on unstable
    !-----------------------------------------------------------------------
    !! 2  Parcel ascent:
    !-----------------------------------------------------------------------
    ! Lift parcel conserving its THL and QW.
    ! Calculate parcel QC by linearising q_sat about the parcel's
    ! temperature extrapolated up to the next grid-level

    do k = 1, shcu_levels
      t_ref = th_ref*exner(k)

      qsat = e0cw * exp(tetn1w *                              &
        (t_ref  - tetn2w) / (t_ref - tetn3w) )
      qsat = epsilon * qsat / (p_theta_levels(k) - one_minus_epsilon * qsat)

      if (fb_surf > 0.0) then
        if (t_ref > tkelvn) then
          lrcp_c = lcrcp
          l_heat = hlatnt
        else
          lrcp_c = lsrcp
          l_heat = hls
        end if

        dqsatdt = EPSILON * l_heat * qsat/(rd * t_ref**2)
        qsatfac = 1.0_r_size/(1.0_r_size+(lrcp_c)*dqsatdt)
        qc_par(k)  = max( 0.0_r_size,                                    &
          qsatfac*( qw_par - qsat                      &
          - (thl_par-th_ref)                            &
          *exner(k)*dqsatdt ) )
        qc_env  = max( 0.0_r_size, qsatfac*( qw(k) - qsat           &
          &                   - (tl(k)-t_ref) *dqsatdt ) )
        qc_par(k)  = qc_par(k) + qcl(k) + qcf(k) - qc_env
        t_par = sl_par - grcp * z_tq(k) + lrcp_c * qc_par(k)
        ! recalculate if signs of T_REF and T_PAR are different
        if (t_ref <= tkelvn .and. t_par > tkelvn) then
          lrcp_c = lcrcp
          qsatfac = 1.0_r_size /(1.0_r_size+(lrcp_c)*dqsatdt)
          qc_par(k)  = max( 0.0_r_size,                                  &
            qsatfac*( qw_par - qsat                    &
            - (sl_par-grcp*z_tq(k)-t_ref) *dqsatdt ) )
          qc_par(k)  = qc_par(k) + qcl(k) + qcf(k) - qc_env
          t_par = sl_par - grcp * z_tq(k)                    &
            + lrcp_c * qc_par(k)
        end if
        th_par = t_par / exner(k)
        thv_par(k) = th_par *                                     &
          (1.0_r_size + c_virtual * qw_par          &
          -(1.0_r_size + c_virtual) * qc_par(k))
        if (k > 1 .and. k < shcu_levels - 1) then
          ! extrapolate reference TH gradient up to next grid-level
          z_pr      = (z_tq(k+1)-z_tq(k))                     &
            /(z_tq(k)-z_tq(k-1))
          th_ref = th_par * (1.0_r_size + z_pr) - th_par_kp1 * z_pr
          th_par_kp1 = th_par
        end if
      end if   ! test on unstable
    end do
    !-----------------------------------------------------------------------
    !! 3 Identify layer boundaries
    !-----------------------------------------------------------------------
    topbl = 0
    topprof = .false.
    topinv= .false.
    ktpar = 1
    k_neut = 1
    ktinv = 1
    dbdz_inv = 0.003_r_size
    ! start with a weak minimum inversion lapse rate
    ! (~1.e-4 s^-2, converted from K/m to s^-2 later)

    do k = 2, shcu_levels

      if (fb_surf > 0.0_r_size) then
        !------------------------------------------------------------
        ! Set flag to true when level BELOW is above the lcl
        ! and above LCL transition zone
        !------------------------------------------------------------
        above_lcl = k-1 > k_lcl + 1 .and. z_tq(k-1) > 1.1_r_size * z_lcl
        !-------------------------------------------------------------
        ! Calculate vertical gradients in parcel and environment THV
        !-------------------------------------------------------------
        dpar_bydz = (thv_par(k) - thv_par(k-1)) / (z_tq(k) - z_tq(k-1))
        denv_bydz = (THv(k) - THv(k-1)) / (z_tq(k) - z_tq(k-1))
        !-------------------------------------------------------------
        ! Find top of inversion - where parcel has minimum buoyancy
        !-------------------------------------------------------------
        if ( topbl > 0 .and. .not. topinv ) then
          dbdz_inv = max( dbdz_inv, denv_bydz )
          if ( k-1 > ktpar + 2 .and. (                             &
                                ! Inversion at least two grid-levels thick
            denv_bydz <= dpar_bydz .or.                  &
                                ! => at a parcel buoyancy minimum
            z_uv(k) > zhpar+min(1000.0_r_size, 0.5_r_size*zhpar)   &
            )) then
            ! restrict inversion thickness < 1/2 bl depth and 1km
            topinv = .true.
            ktinv = k-1
          end if
        end if
        !-------------------------------------------------------------
        ! Find base of inversion - where parcel has maximum buoyancy
        !                          or is negatively buoyant
        !-------------------------------------------------------------
        if ( .not. topprof .and. k > k_par .and.            &
          ((thv_par(k)-THv(k)                              &
          <= - thv_pert) .or.                      &
          k > shcu_levels - 1 )) then
          topprof = .true.
          k_neut = k-1
        end if

        if ( topbl == 0 .and. k > k_par .and.               &
          (  ( thv_par(k)-THv(k)                           &
          <= - thv_pert) .or.                    &
          !                      plume non buoyant
          
          ( above_lcl .and. (denv_bydz > 1.25_r_size*dpar_bydz) )        &
          
          !                      or environmental virtual temperature gradient
          !                      significantly larger than parcel gradient
          !                      above lifting condensation level
          
          )) then

          topbl = 1
          ktpar = k-1   ! marks most buoyant theta-level
          ! (just below inversion)
          zhpar    = z_uv(k)
          dbdz_inv = max( dbdz_inv, denv_bydz )
        end if

        if ( topbl == 0 .and.                                    &
          (z_tq(k-1) >= zhpar_max .or. k == shcu_levels)) then
          !                      gone above maximum allowed height
          topbl = 2
          ktpar = k-2
          dbdz_inv = max( dbdz_inv, denv_bydz )
        end if
      end if   ! test on unstable
    end do


    !-----------------------------------------------------------------------
    !! 3.1 Interpolate inversion base and top between grid-levels
    !-----------------------------------------------------------------------
    if ( ktpar > 1 ) then
      !-----------------------------------------------------
      ! parcel rose successfully
      !-----------------------------------------------------
      zhpar    = z_uv(ktpar+1)

      ! to determine if interpolation of the inversion is performed
      if (topbl == 2) then
        ! Stopped at max allowable height
        interp_inv= 0
        zhpar  = zhpar_max
        k = ktpar
      else
        interp_inv=1
        !-------------------------------------------------------
        ! First interpolate inversion base (max buoyancy excess)
        !-------------------------------------------------------
        !-----------------------------------------------------------
        ! interpolate height by fitting a cubic to 3 parcel excesses,
        ! at Z1 (top of cloud layer) and the two grid-levels above,
        ! and matching cloud layer gradient (D1-D0) at Z1.
        !-----------------------------------------------------------
        k = ktpar+2
        z3=z_tq(k)  -z_tq(k-2)
        d3=thv_par(k)-THv(k)
        z2=z_tq(k-1)-z_tq(k-2)
        d2=thv_par(k-1)-THv(k-1)
        z1=z_tq(k-2)-z_tq(k-2)
        d1=thv_par(k-2)-THv(k-2)
        z0=z_tq(k-3)-z_tq(k-2)
        d0=thv_par(k-3)-THv(k-3)
        c_poly = (d1-d0)/(z1-z0)
        a2= d2 - d1 - c_poly*z2
        a3= d3 - d1 - c_poly*z3
        b_poly = (a3-a2*z3**3/z2**3)/(z3*z3*(1.0-z3/z2))
        a_poly = (a2-b_poly*z2*z2)/z2**3

        xi=b_poly*b_poly-3.*a_poly*c_poly
        if (a_poly /= 0.0_r_size .and. xi > 0.0_r_size) then
          ! ZHPAR is then the height where the above
          ! polynomial has zero gradient
          zhpar = z_tq(k-2)-(b_poly+sqrt(xi))                &
            /(3.0_r_size*a_poly)
          zhpar = max( min( zhpar, z_tq(k) ), z_tq(k-2) )
          if ( zhpar > z_tq(ktpar+1) ) then
            ktpar=ktpar+1
          end if
        end if
        k = ktpar
        denv_bydz = (THv(k+1) - THv(k)) / (z_tq(k+1) - z_tq(k))
      end if
      if ( interp_inv == 1 ) then
        !-----------------------------------------------------
        ! Now interpolate inversion top
        !-----------------------------------------------------
        if ( ktinv > ktpar+1 ) then
          k = ktinv+1
          dpar_bydz = (thv_par(k) - thv_par(k-1)) /           &
            (z_tq(k) - z_tq(k-1))
          denv_bydz = (THv(k) - THv(k-1)) /                   &
            (z_tq(k) - z_tq(k-1))
          if (denv_bydz < dpar_bydz) then
            !-----------------------------------------------------------
            ! interpolate height by fitting a parabola to parcel
            ! excesses and finding the height of its minimum
            !-----------------------------------------------------------
            z1=z_tq(k)
            d1=thv_par(k)-THv(k)
            z2=z_tq(k-1)
            d2=thv_par(k-1)-THv(k-1)
            z3=z_tq(k-2)
            d3=thv_par(k-2)-THv(k-2)
            xi=z2**2-z3**2
            b_poly=( d1-d3 - (d2-d3)*(z1**2-z3**2)/xi ) /             &
              ( z1-z3 - (z2-z3)*(z1**2-z3**2)/xi )
            a_poly=(d2 - d3 - b_poly*(z2-z3) )/xi
          end if
        end if   ! inversion top grid-level 2 levels above parcel top
      end if   ! interp_inv flag
    end if   ! parcel rose
    !-----------------------------------------------------------------------
    !! 4. Integrate parcel excess buoyancy
    !-----------------------------------------------------------------------
    cape = 0.0_r_size

    do k = k_lcl + 1, k_neut-1
      cape = cape + (thv_par(k) - THv(k))         &
        * (z_uv(k+1)-z_uv(k)) / THv(k)
    end do
    !-----------------------------------------------------------------------
    !! 6. Calculate non-gradient fluxes and velocity scales
    !-----------------------------------------------------------------------
    dz_inv_cu = 0.0_r_size
    frcu = 0.0_r_size
    if (fb_surf > 0.0_r_size) then
      w_star = ( fb_surf * zhpar ) ** one_third
      ! dry bl scale
      dbdz_inv = grav * dbdz_inv / THv(ktpar)
      ! convert to buoyancy units
      dz_inv_cu = 0.2_r_size * zhpar
      ! default for no CAPE
    end if

    if (cape > 0.0_r_size .and. zhpar - z_lcl > 0.0_r_size) then
      k = k_lcl
      ! calculate velocity scales
      w_star = ( fb_surf * z_lcl ) ** one_third
      m_base    = 0.04_r_size * w_star
      cape = grav * cape
      w_cld     = ( m_base * cape ) ** one_third
      z_cld     = zhpar - z_lcl
      ! calculate fluxes at LCL
      wb_scale  = ( w_cld**3 / z_cld ) * sqrt( m_base/w_cld )

      !----------------------------------------------------------
      ! Estimate inversion thickness.
      !----------------------------------------------------------
      vscalsq_incld = 2.0_r_size * cape
      dz_inv_cu  = sqrt( vscalsq_incld / dbdz_inv )
      !
      ! If inversion is unresolved (less than 3 grid-levels thick)
      ! then use profile reconstruction
      !
      if ( ktpar <= shcu_levels - 4 ) then
        if ( dz_inv_cu  < z_tq(ktpar+3) - z_tq(ktpar) ) then
          !
          ! First interpolate to find height of discontinuous inversion
          !
          k = ktpar
          gamma_cld = (THv(k)-THv(k-1)) / (z_tq(k)-z_tq(k-1))
          if (k-2 > k_lcl) then
            gamma_cld = min( gamma_cld,                               &
              ( THv(k-1)-THv(k-2) )                &
              /(   z_tq(k-1)-  z_tq(k-2) ) )
          end if
          gamma_cld = max(0.0_r_size, gamma_cld)
          gamma_fa = (THv(k+4)-THv(k+3))                      &
            /(z_tq(k+4)-z_tq(k+3))
          gamma_fa = max(0.0_r_size, gamma_fa)
          ! Integrate thv over the inversion grid-levels
          grid_int =  (THv(k+1)-THv(k))                       &
            *(z_uv(k+2)-z_uv(k+1))         &
            + (THv(k+2)-THv(k))                          &
            *(z_uv(k+3)-z_uv(k+2))         &
            + (THv(k+3)-THv(k))                          &
            *( z_tq(k+3)-z_uv(k+3))

          c_poly = (THv(k+3)-THv(k))                          &
            *(z_tq(k+3)-z_tq(k))                  &
            - 0.5_r_size*gamma_fa*(z_tq(k+3)-z_tq(k))**2    &
            - grid_int
          b_poly = -(THv(k+3)-THv(k)                          &
            -gamma_fa*(z_tq(k+3)-z_tq(k)))
          a_poly = 0.5_r_size * (gamma_cld-gamma_fa)
          xi     = b_poly*b_poly-4.0_r_size*a_poly*c_poly

          if (xi >= 0.0_r_size .and.                              &
            ( a_poly  /= 0.0_r_size .or. b_poly  /= 0.0_r_size )) then
            if (a_poly == 0.0_r_size) then
              dz_inv_cu_rec = -c_poly/b_poly
            else
              dz_inv_cu_rec = (-b_poly-sqrt(xi))/(2.0_r_size*a_poly)
            end if
            zhdisc  = z_tq(k)+dz_inv_cu_rec
            !
            ! Now calculate inversion stability given Dz=V^2/DB
            !
            c_poly = -vscalsq_incld*THv(k+1)/grav
            b_poly = THv(k+3)-gamma_fa *(z_tq(k+3)-zhdisc)    &
              -THv(k)  -gamma_cld*(zhdisc  -z_tq(k))
            a_poly = 0.5_r_size*(gamma_cld+gamma_fa)
            xi=b_poly*b_poly-4.0_r_size*a_poly*c_poly

            if (xi >= 0.0_r_size .and.                               &
              ( a_poly /= 0.0_r_size .or. b_poly /= 0.0_r_size )) then
              if (a_poly == 0.0_r_size) then
                dz_inv_cu_rec = -c_poly/b_poly
              else
                dz_inv_cu_rec = (-b_poly+sqrt(xi))/(2.0_r_size*a_poly)
              end if
              dz_inv_cu_rec = min( dz_inv_cu_rec,                     &
                2.0_r_size*(zhdisc-z_tq(ktpar)) )
              if (dz_inv_cu_rec <= dz_inv_cu) then
                dz_inv_cu = dz_inv_cu_rec
              end if
            end if  ! interpolation for DZ_INV_CU successful
          end if    ! interpolation for ZHDISC successful

        end if  ! inversion not resolved
      end if  ! if ktpar(i,j) <= shcu_levels - 4

      zpr_top     = 1.0_r_size + min(1.0_r_size, dz_inv_cu/z_cld )
      do k = 1, shcu_levels-1
        ! Z_PR=0 at cloud-base, 1 at cloud-top
        z_pr = ( z_uv(k+1) - z_lcl )/ z_cld
        if (z_pr > 0.0_r_size) then

          !   Non-gradient function for WB

          f_ng = 0.0_r_size
          if ( z_pr <= 0.9_r_size ) then
            ! function with gradient=0 at z=0.9
            !                    f=0,1 at z=0,0.9
            ze_pr = z_pr/0.9_r_size
            f_ng  = 0.5_r_size * sqrt(ze_pr) * (3.0_r_size - ze_pr)
          else if (z_pr <= zpr_top) then
            ze_pr = (z_pr-0.9_r_size)/(zpr_top-0.9_r_size)  ! from 0 to 1
            f_ng  = 0.5_r_size * (1.+cos(pi*ze_pr))
          end if
          fnn = 0.5_r_size &
            &   * (1.0_r_size + tanh(0.8_r_size * (q1(k) + 0.5_r_size)))
          wb_ng(k) = min((1.0_r_size-fnn)*3.7_r_size*f_ng*wb_scale, wb_ng_max)
        end if   ! if Z_PR > 0

        !   Cloud fraction enhancement and sigma_s calculation (for ql)
        !   (on Z rather than ZE levels)

        z_pr = ( z_tq(k) - z_lcl )/ z_cld
        ! Z_PR=0 at cloud-base, 1 at cloud-top

        if (z_pr > 0.0_r_size) then
          f_ng = 0.0_r_size
          if ( z_pr <= 0.9_r_size ) then
            f_ng = 1.0_r_size+3.0_r_size*exp(-5.0_r_size*z_pr)     ! =4 at cloud-base
          else if ( z_pr < zpr_top ) then
            ze_pr = (z_pr-0.9_r_size)/(zpr_top-0.9_r_size)  ! from 0 to 1
            f_ng = 0.5_r_size*(1.0_r_size+cos(pi*ze_pr))
          end if
          frcu = 0.5_r_size*f_ng*min(0.5_r_size,m_base/w_cld)
        end if   ! Z_PR > 0
        frac(k)    = max( frac_gauss(k), frcu)
      end do   ! loop over K
    end if ! Test on CAPE

    do k = shcu_levels + 1, nz
      frac(k) = 0.0_r_size
      wb_ng(k) = 0.0_r_size
    end do

    call pp_monit_store(cape, 'cape')
    call pp_monit_store(zhpar, 'zhpar')
    call pp_monit_store(z_lcl, 'zlcl')
    call pp_monit_store(shcu_levels, 1, shcu_levels, wb_ng, 'wb_ng')
    call pp_monit_store(shcu_levels, 1, shcu_levels, thv_par, 'thvpar')
    call pp_monit_store(shcu_levels, 1, shcu_levels, thv, 'thvenv')
    call pp_monit_store(shcu_levels, 1, shcu_levels, thvl, 'thvl')
    call pp_monit_store(shcu_levels, 1, shcu_levels, qw, 'qw')
    call pp_monit_store(shcu_levels, 1, shcu_levels, qc_par, 'qcpar')
    call pp_monit_store(dz_inv_cu, 'dzicu')
    call pp_monit_store(zhdisc, 'zhdisc')
    call pp_monit_store(qw_par, 'qwpar')
!    call pp_monit_store(real(k_neut), 'k_neut')


  end subroutine pbl_shcu_run
end module pbl_shcu
