!= 地球大気向け放射モデル Ver. 2
!
!= radiation model for the Earth's atmosphere Ver. 2
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: rad_Earth_V2.f90,v 1.9 2015/01/29 12:06:43 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!
module rad_Earth_V2
  !
  != 地球大気向け放射モデル Ver. 2
  !
  != radiation model for the Earth's atmosphere Ver. 1
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 地球大気向け放射モデル.
  !
  ! This is a radiation model for the Earth's atmospehre.
  !
  ! Radiation in the wavenumber range from    0 to  3000 cm-1 
  ! is calculated in the routine of long wave radiation. 
  ! Radiation in the wavenumber range from 1000 to 57143 cm-1 (0.175 to 10 micron) 
  ! is calculated in the routine of shortwave radiation. 
  !
  !== References
  !
  !
  !== Procedures List
  !
  ! RadEarthV2Flux :: 放射フラックスの計算
  ! ------------          :: ------------
  ! RadEarthV2Flux :: Calculate radiation flux
  !
  !== NAMELIST
  !
  ! NAMELIST#rad_Earth_V2_nml
  !

  ! USE statements
  !

  !
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  !
  ! Physical constants settings
  !
  use constants, only: Grav     ! $ g $ [m s-2].
                                !
                                ! Gravitational acceleration

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数.
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数.
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数.
                             ! Number of vertical level

  implicit none

  private

  real(DP), save :: MeanMolWeight
  real(DP), save :: H2OMolWeight
  real(DP), save :: CO2MolWeight
  real(DP), save :: N2OMolWeight
  real(DP), save :: CH4MolWeight

  integer, save     :: IDCloudIceREffMethod
  !
  integer, parameter:: IDCloudIceREffMethodConst = 0
  integer, parameter:: IDCloudIceREffMethodLin   = 1

  real(DP), save :: CloudWatREff
  real(DP), save :: CloudIceREff

  real(DP), allocatable, save :: xyz_QO3(:,:,:)
                              ! O3 分布 (1)
                              ! O3 distribution (1)

!!$  integer , save              :: SWVer
!!$  integer , save              :: LWVer

  ! 公開変数
  ! Public variables
  !
  logical, save :: rad_Earth_V2_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  public :: RadEarthV2Flux
!!$  public :: RadEarthV2FluxforNHM
  public :: RadEarthV2Init

  character(*), parameter:: module_name = 'rad_Earth_V2'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: rad_Earth_V2.f90,v 1.9 2015/01/29 12:06:43 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine RadEarthV2Flux(                                                 &
    & xy_SurfAlbedo,                                                         & ! (in)
    & xyz_Press, xyr_Press, xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol, & ! (in)
    & xyz_CloudCover,                                                        & ! (in)
    & xy_SurfTemp,                                                           & ! (in)
    & xyr_RadSUwFlux, xyr_RadSDwFlux,                                        & ! (out)
    & xyr_RadLUwFlux, xyr_RadLDwFlux,                                        & ! (out)
    & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux                                 & ! (out)
    & )


    ! USE statements
    !

    real(DP), intent(in ) :: xy_SurfAlbedo   (0:imax-1, 1:jmax)
    real(DP), intent(in ) :: xyz_Press       (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyr_Press       (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(in ) :: xyz_Temp        (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_QH2OVap     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ q $ .   混合比. Mass mixing ratio of constituents (1)
    real(DP), intent(in ) :: xyz_QH2OLiq     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_QH2OSol     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_CloudCover  (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xy_SurfTemp     (0:imax-1, 1:jmax)
    real(DP), intent(out) :: xyr_RadSUwFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadSDwFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadLUwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadLDwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyra_DelRadLUwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
    real(DP), intent(out) :: xyra_DelRadLDwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)

!!$    real(DP):: xyr_RadLUwFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP):: xyr_RadLDwFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP):: xyra_DelRadLUwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
!!$    real(DP):: xyra_DelRadLDwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)


    ! Work variables
    !
    real(DP) :: xyz_DelAtmMass   (0:imax-1, 1:jmax, 1:kmax)

    integer  :: k


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_Earth_V2_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k ) ) / Grav
    end do

    call RadEarthV2FluxCore(                                                        &
      & xy_SurfAlbedo,                                                              &
      & xyz_DelAtmMass, xyz_Press, xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol, &
      & xyz_CloudCover,                                                             &
      & xy_SurfTemp,                                                                &
      & xyr_RadSUwFlux, xyr_RadSDwFlux,                                             &
      & xyr_RadLUwFlux, xyr_RadLDwFlux,                          & ! (out)
      & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux                   & ! (out)
      & )


  end subroutine RadEarthV2Flux

  !--------------------------------------------------------------------------------------
!!$
!!$  subroutine RadEarthV2FluxforNHM(                                          &
!!$    & r_Height, xy_SurfAlbedo,                                              &
!!$    & xyz_Dens, xyz_Press, xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol, &
!!$    & xy_SurfTemp,                                                          &
!!$    & xyr_RadSFlux, xyr_RadLFlux, xyra_DelRadLFlux                          &
!!$    & )
!!$
!!$
!!$    ! USE statements
!!$    !
!!$
!!$    real(DP), intent(in ) :: r_Height                          (0:kmax)
!!$    real(DP), intent(in ) :: xy_SurfAlbedo   (0:imax-1, 1:jmax)
!!$    real(DP), intent(in ) :: xyz_Dens        (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP), intent(in ) :: xyz_Press       (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP), intent(in ) :: xyz_Temp        (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP), intent(in ) :: xyz_QH2OVap     (0:imax-1, 1:jmax, 1:kmax)
!!$                              ! $ q $ .   混合比. Mass mixing ratio of constituents (1)
!!$    real(DP), intent(in ) :: xyz_QH2OLiq     (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP), intent(in ) :: xyz_QH2OSol     (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP), intent(in ) :: xy_SurfTemp     (0:imax-1, 1:jmax)
!!$    real(DP), intent(out) :: xyr_RadSFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP), intent(out) :: xyr_RadLFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP), intent(out) :: xyra_DelRadLFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
!!$
!!$
!!$    ! Work variables
!!$    !
!!$    real(DP) :: xyz_DelAtmMass   (0:imax-1, 1:jmax, 1:kmax)
!!$
!!$    integer  :: k
!!$
!!$
!!$    ! 初期化確認
!!$    ! Initialization check
!!$    !
!!$    if ( .not. rad_Earth_V2_inited ) then
!!$      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
!!$    end if
!!$
!!$
!!$    do k = 1, kmax
!!$      xyz_DelAtmMass(:,:,k) = xyz_Dens(:,:,k) * ( r_Height(k) - r_Height(k-1) )
!!$    end do
!!$
!!$    call RadEarthV2FluxCore(                                                        &
!!$      & xy_SurfAlbedo,                                                              &
!!$      & xyz_DelAtmMass, xyz_Press, xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol, &
!!$      & xy_SurfTemp,                                                                &
!!$      & xyr_RadSFlux, xyr_RadLFlux, xyra_DelRadLFlux                                &
!!$      & )
!!$
!!$
!!$  end subroutine RadEarthV2FluxforNHM
!!$
  !--------------------------------------------------------------------------------------
!!$
!!$  subroutine RadDcpamEV2FluxforNHM2DWrapper(                    &
!!$    & r_Height, x_SurfAlbedo,                                   &
!!$    & xz_Dens, xz_Press, xz_Temp, xz_QH2OVap, xz_QH2OLiq,       &
!!$    & x_SurfTemp,                                               &
!!$    & xr_RadSFlux, xr_RadLFlux, xra_DelRadLFlux                 &
!!$    & )
!!$
!!$
!!$    ! USE statements
!!$    !
!!$
!!$
!!$    real(DP), intent(in ) :: r_Height                 (0:kmax)
!!$    real(DP), intent(in ) :: x_SurfAlbedo   (0:imax-1)
!!$    real(DP), intent(in ) :: xz_Dens        (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: xz_Press       (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: xz_Temp        (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: xz_QH2OVap     (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: xz_QH2OLiq     (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: x_SurfTemp     (0:imax-1)
!!$    real(DP), intent(out) :: xr_RadSFlux    (0:imax-1, 0:kmax)
!!$    real(DP), intent(out) :: xr_RadLFlux    (0:imax-1, 0:kmax)
!!$    real(DP), intent(out) :: xra_DelRadLFlux(0:imax-1, 0:kmax, 0:1)
!!$
!!$
!!$    ! Work variables
!!$    !
!!$    real(DP) :: xy_SurfAlbedo   (0:imax-1, 1:jmax)
!!$    real(DP) :: xyz_Dens        (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xyz_Press       (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xyz_Temp        (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xyz_QH2OVap     (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xyz_QH2OLiq     (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xy_SurfTemp     (0:imax-1, 1:jmax)
!!$    real(DP) :: xyr_RadSFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP) :: xyr_RadLFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP) :: xyra_DelRadLFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
!!$
!!$    integer  :: k
!!$
!!$
!!$    ! 初期化確認
!!$    ! Initialization check
!!$    !
!!$    if ( .not. rad_dcpam_E_V2_inited ) then
!!$      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
!!$    end if
!!$
!!$
!!$    if ( jmax /= 1 ) then
!!$      call MessageNotify( 'E', module_name, 'jmax must be 1.' )
!!$    end if
!!$
!!$
!!$    xy_SurfAlbedo   (:, 1)       = x_SurfAlbedo   (:)
!!$    do k = 1, kmax
!!$      xyz_Dens        (:, 1, k)    = xz_Dens        (:, k)
!!$      xyz_Press       (:, 1, k)    = xz_Press       (:, k)
!!$      xyz_Temp        (:, 1, k)    = xz_Temp        (:, k)
!!$      xyz_QH2OVap     (:, 1, k)    = xz_QH2OVap     (:, k)
!!$      xyz_QH2OLiq     (:, 1, k)    = xz_QH2OLiq     (:, k)
!!$    end do
!!$    xy_SurfTemp     (:, 1)       = x_SurfTemp     (:)
!!$
!!$    call RadDcpamEV2FluxforNHM(                                       &
!!$      & r_Height, xy_SurfAlbedo,                                      &
!!$      & xyz_Dens, xyz_Press, xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq,      &
!!$      & xy_SurfTemp,                                                  &
!!$      & xyr_RadSFlux, xyr_RadLFlux, xyra_DelRadLFlux                  &
!!$      & )
!!$
!!$    xr_RadSFlux    (:, :)    = xyr_RadSFlux    (:, 1, :)
!!$    xr_RadLFlux    (:, :)    = xyr_RadLFlux    (:, 1, :)
!!$    xra_DelRadLFlux(:, :, :) = xyra_DelRadLFlux(:, 1, :, :)
!!$
!!$
!!$  end subroutine RadDcpamEV2FluxforNHM2DWrapper
!!$
  !--------------------------------------------------------------------------------------
!!$
!!$  subroutine RadDcpamEV2Flux2DWrapper(                         &
!!$    & x_SurfAlbedo,                                            &
!!$    & xz_Press, xr_Press, xz_Temp, xz_QH2OVap, xz_QH2OLiq,     &
!!$    & x_SurfTemp,                                              &
!!$    & xr_RadSFlux, xr_RadLFlux, xra_DelRadLFlux                &
!!$    & )
!!$
!!$
!!$    ! USE statements
!!$    !
!!$
!!$
!!$
!!$    real(DP), intent(in ) :: x_SurfAlbedo   (0:imax-1)
!!$    real(DP), intent(in ) :: xz_Press       (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: xr_Press       (0:imax-1, 0:kmax)
!!$    real(DP), intent(in ) :: xz_Temp        (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: xz_QH2OVap     (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: xz_QH2OLiq     (0:imax-1, 1:kmax)
!!$    real(DP), intent(in ) :: x_SurfTemp     (0:imax-1)
!!$    real(DP), intent(out) :: xr_RadSFlux    (0:imax-1, 0:kmax)
!!$    real(DP), intent(out) :: xr_RadLFlux    (0:imax-1, 0:kmax)
!!$    real(DP), intent(out) :: xra_DelRadLFlux(0:imax-1, 0:kmax, 0:1)
!!$
!!$
!!$    ! Work variables
!!$    !
!!$    real(DP) :: xy_SurfAlbedo   (0:imax-1, 1:jmax)
!!$    real(DP) :: xyz_Press       (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xyr_Press       (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP) :: xyz_Temp        (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xyz_QH2OVap     (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xyz_QH2OLiq     (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP) :: xy_SurfTemp     (0:imax-1, 1:jmax)
!!$    real(DP) :: xyr_RadSFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP) :: xyr_RadLFlux    (0:imax-1, 1:jmax, 0:kmax)
!!$    real(DP) :: xyra_DelRadLFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
!!$
!!$
!!$    ! 初期化確認
!!$    ! Initialization check
!!$    !
!!$    if ( .not. rad_dcpam_E_V2_inited ) then
!!$      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
!!$    end if
!!$
!!$
!!$    if ( jmax /= 1 ) then
!!$      call MessageNotify( 'E', module_name, 'jmax must be 1.' )
!!$    end if
!!$
!!$
!!$    xy_SurfAlbedo   (:, 1)       = x_SurfAlbedo   (:)
!!$    xyz_Press       (:, 1, :)    = xz_Press       (:, :)
!!$    xyr_Press       (:, 1, :)    = xr_Press       (:, :)
!!$    xyz_Temp        (:, 1, :)    = xz_Temp        (:, :)
!!$    xyz_QH2OVap     (:, 1, :)    = xz_QH2OVap     (:, :)
!!$    xyz_QH2OLiq     (:, 1, :)    = xz_QH2OLiq     (:, :)
!!$    xy_SurfTemp     (:, 1)       = x_SurfTemp     (:)
!!$
!!$    call RadDcpamEV2Flux(                                             &
!!$      & xy_SurfAlbedo,                                                &
!!$      & xyz_Press, xyr_Press, xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq,     &
!!$      & xy_SurfTemp,                                                  &
!!$      & xyr_RadSFlux, xyr_RadLFlux, xyra_DelRadLFlux                  &
!!$      & )
!!$
!!$    xr_RadSFlux    (:, :)    = xyr_RadSFlux    (:, 1, :)
!!$    xr_RadLFlux    (:, :)    = xyr_RadLFlux    (:, 1, :)
!!$    xra_DelRadLFlux(:, :, :) = xyra_DelRadLFlux(:, 1, :, :)
!!$
!!$
!!$  end subroutine RadDcpamEV2Flux2DWrapper
!!$
  !--------------------------------------------------------------------------------------

  subroutine RadEarthV2FluxCore(                                                  &
    & xy_SurfAlbedo,                                                              &
    & xyz_DelAtmMass, xyz_Press, xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol, &
    & xyz_CloudCover,                                                             &
    & xy_SurfTemp,                                                                &
    & xyr_RadSUwFlux, xyr_RadSDwFlux,                                             &
    & xyr_RadLUwFlux, xyr_RadLDwFlux,                          & ! (out)
    & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux                   & ! (out)
    & )


    ! USE statements
    !

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN                 ! ステップ $ t $ の時刻.
                              ! Time of step $ t $.

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    use set_gcmr, only : SetGCMR

    ! O3 分布の設定
    ! Set O3 distribution
    !
    use set_o3, only: SetO3

!!$    ! dcpam 地球大気向け短波放射モデル Ver. 2
!!$    ! dcpam short wave radiation model for the Earth's atmosphere Ver. 2
!!$    !
!!$    use radiation_dcpam_E_SW_V2, only: RadiationDcpamESWV2Flux

!!$    ! 地球大気向け短波放射モデル Ver. 2.1
!!$    ! short wave radiation model for the Earth's atmosphere Ver. 2.1
!!$    !
!!$    use rad_Earth_SW_V2_1, only: RadEarthSWV21Flux
!!$
!!$    ! 地球大気向け短波放射モデル Ver. 2.2
!!$    ! short wave radiation model for the Earth's atmosphere Ver. 2.2
!!$    !
!!$    use rad_Earth_SW_V2_2, only: RadEarthSWV22Flux

    ! 地球大気向け短波放射モデル Ver. 2.6
    ! short wave radiation model for the Earth's atmosphere Ver. 2.6
    !
    use rad_Earth_SW_V2_6, only: RadEarthSWV26Flux

!!$    ! dcpam 地球大気向け長波放射モデル Ver. 2
!!$    ! dcpam long wave radiation model for the Earth's atmosphere Ver. 2
!!$    !
!!$    use radiation_dcpam_E_LW_V2, only : RadiationDcpamELWV2Flux
!!$
!!$    ! dcpam 地球大気向け長波放射モデル Ver. 2.1
!!$    ! dcpam long wave radiation model for the Earth's atmosphere Ver. 2.1
!!$    !
!!$    use radiation_dcpam_E_LW_V2_1, only : RadiationDcpamELWV21Flux
!!$
!!$    ! dcpam 地球大気向け長波放射モデル Ver. 2.2
!!$    ! dcpam long wave radiation model for the Earth's atmosphere Ver. 2.2
!!$    !
!!$    use radiation_dcpam_E_LW_V2_2, only : RadiationDcpamELWV22Flux
!!$
!!$    ! 地球大気向け長波放射モデル Ver. 2.3
!!$    ! long wave radiation model for the Earth's atmosphere Ver. 2.3
!!$    !
!!$    use rad_Earth_LW_V2_3, only : RadEarthLWV23Flux
!!$
    ! 地球大気向け長波放射モデル Ver. 2.4
    ! long wave radiation model for the Earth's atmosphere Ver. 2.4
    !
    use rad_Earth_LW_V2_4, only : RadEarthLWV24Flux


    real(DP), intent(in ) :: xy_SurfAlbedo   (0:imax-1, 1:jmax)
    real(DP), intent(in ) :: xyz_DelAtmMass  (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_Press       (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_Temp        (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_QH2OVap     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ q $ .   混合比. Mass mixing ratio of constituents (1)
    real(DP), intent(in ) :: xyz_QH2OLiq     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_QH2OSol     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_CloudCover  (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xy_SurfTemp     (0:imax-1, 1:jmax)
    real(DP), intent(out) :: xyr_RadSUwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadSDwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadLUwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadLDwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyra_DelRadLUwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
    real(DP), intent(out) :: xyra_DelRadLDwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)


    ! Work variables
    !
    real(DP) :: VMRCO2
    real(DP) :: xyz_QCO2         (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: VMRN2O
    real(DP) :: xyz_QN2O         (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: VMRCH4
    real(DP) :: xyz_QCH4         (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_DelCO2Mass   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelH2OVapMass(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelH2OLiqMass(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelH2OSolMass(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelO3Mass    (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelN2OMass   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelCH4Mass   (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_CloudWatREff (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_CloudIceREff (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xy_CloudCoverRand(0:imax-1, 1:jmax)

    integer :: k

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_Earth_V2_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    call SetGCMR( &
      & 'CO2',    & ! (in)
      & VMRCO2    & ! (out)
      & )
    xyz_QCO2 = VMRCO2 * CO2MolWeight / MeanMolWeight

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    call SetGCMR( &
      & 'N2O',    & ! (in)
      & VMRN2O    & ! (out)
      & )
    xyz_QN2O = VMRN2O * N2OMolWeight / MeanMolWeight

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    call SetGCMR( &
      & 'CH4',    & ! (in)
      & VMRCH4    & ! (out)
      & )
    xyz_QCH4 = VMRCH4 * CH4MolWeight / MeanMolWeight

    ! O3 分布の設定
    ! Setting of O3 distribution
    !
    call SetO3( &
      & xyz_Press, & ! (in )
      & xyz_QO3    & ! (out)
      & )

    !
    ! Set cloud effective radius
    !
    xyz_CloudWatREff = CloudWatREff
    !
    select case ( IDCloudIceREffMethod )
    case ( IDCloudIceREffMethodConst )
      xyz_CloudIceREff = CloudIceREff
    case ( IDCloudIceREffMethodLin )
      ! This is a simple model mimicing a result of Ou and Liou (1995).
      ! This was used in MRI model probably.
      xyz_CloudIceREff = &
        &   ( 20.0e-6_DP              - 150.0e-6_DP             ) &
        & / ( ( 273.15_DP - 60.0_DP ) - ( 273.15_DP - 20.0_DP ) ) &
        & * ( xyz_Temp                - ( 273.15_DP - 20.0_DP ) ) &
        & + 150.0e-6_DP
      xyz_CloudIceREff = min( max( xyz_CloudIceREff, 20.0e-6_DP ), 150.0e-6_DP )
    end select


    xyz_DelCO2Mass    = xyz_DelAtmMass * xyz_QCO2
    xyz_DelH2OVapMass = xyz_DelAtmMass * xyz_QH2OVap
    xyz_DelH2OLiqMass = xyz_DelAtmMass * xyz_QH2OLiq
    xyz_DelH2OSolMass = xyz_DelAtmMass * xyz_QH2OSol
    xyz_DelO3Mass     = xyz_DelAtmMass * xyz_QO3
    xyz_DelN2OMass    = xyz_DelAtmMass * xyz_QN2O
    xyz_DelCH4Mass    = xyz_DelAtmMass * xyz_QCH4


!!$    select case ( SWVer )
!!$    case ( 0 )
!!$      ! dcpam 地球大気向け短波放射モデル Ver. 2
!!$      ! dcpam short wave radiation model for the Earth's atmosphere Ver. 2
!!$      !
!!$      call RadiationDcpamESWV2Flux(                                 &
!!$        & xy_SurfAlbedo,                                                  &
!!$        & xyz_Press, xyr_Press, xyz_Temp, xyz_QH2OVap, xyz_QO3, xyz_Height,  &
!!$        & xyr_RadSFlux                                                    &
!!$        & )
!!$    case ( 1 )
!!$      ! 地球大気向け短波放射モデル Ver. 2.1
!!$      ! short wave radiation model for the Earth's atmosphere Ver. 2.1
!!$      !
!!$      call RadEarthSWV21Flux(                                      &
!!$        & xy_SurfAlbedo,                                           &
!!$        & xyz_DelAtmMass,                                          &
!!$        & xyz_DelH2OVapMass, xyz_DelH2OLiqMass, xyz_DelH2OSolMass, &
!!$        & xyz_DelO3Mass,                                           &
!!$        & xyz_Press, xyz_Temp,                                     &
!!$        & xyz_CloudCover,                                          &
!!$        & xyr_RadSUwFlux, xyr_RadSDwFlux                           &
!!$        & )
!!$    case ( 2 )
!!$      ! 地球大気向け短波放射モデル Ver. 2.2
!!$      ! short wave radiation model for the Earth's atmosphere Ver. 2.2
!!$      !
!!$      call RadEarthSWV22Flux(                                      &
!!$        & xy_SurfAlbedo,                                           &
!!$        & xyz_DelAtmMass,                                          &
!!$        & xyz_DelH2OVapMass, xyz_DelH2OLiqMass, xyz_DelH2OSolMass, &
!!$        & xyz_DelO3Mass,                                           &
!!$        & xyz_Press, xyz_Temp,                                     &
!!$        & xyz_CloudCover,                                          &
!!$        & xyr_RadSUwFlux, xyr_RadSDwFlux                           &
!!$        & )
!!$    case ( 6 )
      ! 地球大気向け短波放射モデル Ver. 2.6
      ! short wave radiation model for the Earth's atmosphere Ver. 2.6
      !
      call RadEarthSWV26Flux(                                      &
        & xy_SurfAlbedo,                                           &
        & xyz_DelAtmMass,                                          &
        & xyz_DelH2OVapMass, xyz_DelH2OLiqMass, xyz_DelH2OSolMass, &
        & xyz_DelO3Mass,                                           &
        & xyz_Press, xyz_Temp,                                     &
        & xyz_CloudCover,                                          &
        & xyz_CloudWatREff, xyz_CloudIceREff,                      &
        & xyr_RadSUwFlux, xyr_RadSDwFlux                           &
        & )
!!$    case default
!!$      call MessageNotify( 'E', module_name, 'SW model version %d is not supported.', i = (/ LWVer /) )
!!$    end select

!!$    select case ( LWVer )
!!$    case ( 0 )
!!$      call RadiationDcpamELWV2Flux( &
!!$        & xyz_Press, xyr_Press, xyz_Temp, xy_SurfTemp,                    & ! (in )
!!$        & xyz_QH2OVap, xyz_QO3,                                           & ! (in )
!!$        & xyr_RadLFlux, xyra_DelRadLFlux                                  & ! (out)
!!$        & )
!!$    case ( 1 )
!!$      call RadiationDcpamELWV21Flux( &
!!$        & xyz_Press, xyr_Press, xyz_Temp, xy_SurfTemp,                    & ! (in )
!!$        & xyz_QH2OVap, xyz_QO3,                                           & ! (in )
!!$        & xyr_RadLFlux, xyra_DelRadLFlux                                  & ! (out)
!!$        & )
!!$    case ( 2 )
!!$      call RadiationDcpamELWV22Flux( &
!!$        & xyz_Press, xyr_Press, xyz_Temp, xy_SurfTemp,                    & ! (in )
!!$        & xyz_QH2OVap, xyz_QO3,                                           & ! (in )
!!$        & xyr_RadLFlux, xyra_DelRadLFlux                                  & ! (out)
!!$        & )
!!$    case ( 3 )
!!$      call RadEarthLWV23Flux(                                      &
!!$        & xyz_DelCO2Mass,                                          &
!!$        & xyz_DelH2OVapMass, xyz_DelH2OLiqMass, xyz_DelH2OSolMass, &
!!$        & xyz_DelO3Mass,                                           &
!!$        & xyz_Press, xyz_Temp, xy_SurfTemp,                        & ! (in )
!!$        & xyz_QCO2, xyz_QH2OVap,                                   & ! (in )
!!$        & xyz_CloudCover,                                          & ! (in )
!!$        & xyr_RadLUwFlux, xyr_RadLDwFlux,                          & ! (out)
!!$        & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux                   & ! (out)
!!$        & )
!!$    case ( 4 )
      call RadEarthLWV24Flux(                                      &
        & xyz_DelCO2Mass,                                          &
        & xyz_DelH2OVapMass, xyz_DelH2OLiqMass, xyz_DelH2OSolMass, &
        & xyz_DelO3Mass,                                           &
        & xyz_DelN2OMass, xyz_DelCH4Mass,                          &
        & xyz_Press, xyz_Temp, xy_SurfTemp,                        & ! (in )
        & xyz_QCO2, xyz_QH2OVap,                                   & ! (in )
        & xyz_QN2O, xyz_QCH4,                                      & ! (in )
        & xyz_CloudCover,                                          & ! (in )
        & xyz_CloudWatREff, xyz_CloudIceREff,                      & ! (in )
        & xyr_RadLUwFlux, xyr_RadLDwFlux,                          & ! (out)
        & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux                   & ! (out)
        & )
!!$    case default
!!$      call MessageNotify( 'E', module_name, 'LW model version %d is not supported.', i = (/ LWVer /) )
!!$    end select


    ! Output variables
    !
    call HistoryAutoPut( TimeN, 'CloudCoverforRad', xyz_CloudCover )
    !
    xy_CloudCoverRand = 1.0_DP
    do k = 1, kmax
      xy_CloudCoverRand = xy_CloudCoverRand * ( 1.0_DP - xyz_CloudCover(:,:,k) )
    end do
    xy_CloudCoverRand = 1.0_DP - xy_CloudCoverRand
    call HistoryAutoPut( TimeN, 'CloudCoverRand', xy_CloudCoverRand )


  end subroutine RadEarthV2FluxCore

  !--------------------------------------------------------------------------------------

  subroutine RadEarthV2Init

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    use set_gcmr, only : SetGCMRInit

    ! O3 分布の設定
    ! Set O3 distribution
    !
    use set_o3, only : SetO3Init

!!$    ! 地球大気向け短波放射モデル Ver. 2.1
!!$    ! short wave radiation model for the Earth's atmosphere Ver. 2.1
!!$    !
!!$    use rad_Earth_SW_V2_1, only: RadEarthSWV21Init
!!$
!!$    ! 地球大気向け短波放射モデル Ver. 2.2
!!$    ! short wave radiation model for the Earth's atmosphere Ver. 2.2
!!$    !
!!$    use rad_Earth_SW_V2_2, only: RadEarthSWV22Init

    ! 地球大気向け短波放射モデル Ver. 2.6
    ! short wave radiation model for the Earth's atmosphere Ver. 2.6
    !
    use rad_Earth_SW_V2_6, only: RadEarthSWV26Init

!!$    ! 地球大気向け長波放射モデル Ver. 2.3
!!$    ! long wave radiation model for the Earth's atmosphere Ver. 2.3
!!$    !
!!$    use rad_Earth_LW_V2_3, only : RadEarthLWV23Init

    ! 地球大気向け長波放射モデル Ver. 2.4
    ! long wave radiation model for the Earth's atmosphere Ver. 2.4
    !
    use rad_Earth_LW_V2_4, only : RadEarthLWV24Init


    ! 宣言文 ; Declaration statements
    !

    character(STRING)   :: CloudIceREffMethod

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /rad_Earth_V2_nml/ &
      & CloudIceREffMethod,     &
      & CloudWatREff,           &
      & CloudIceREff

!!$      & SWVer, LWVer
          !
          ! デフォルト値については初期化手続 "rad_Earth_V2#RadEarthV2Init"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "rad_Earth_V2#RadEarthV2Init" for the default values.
          !

    if ( rad_Earth_V2_inited ) return


    ! Molecular weights of atmosphere, CO2, H2O, N2O, CH4
    !
    MeanMolWeight = 28.0d-3
    CO2MolWeight  = 44.0d-3
    H2OMolWeight  = 18.0d-3
    N2OMolWeight  = 44.0d-3
    CH4MolWeight  = 16.0d-3


    ! デフォルト値の設定
    ! Default values settings
    !
    CloudIceREffMethod = 'Const'
!!$    CloudIceREffMethod = 'Lin'

    CloudWatREff    = 10.0d-6
    CloudIceREff    = 50.0d-6


!!$    SWVer = 1
!!$    LWVer = 3


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = rad_Earth_V2_nml,         & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    ! Identification of calculation method of cloud particle effective radius
    !
    call MessageNotify( 'M', module_name, &
      & 'CloudIceREffMethod=<%c>.', &
      & c1 = trim(CloudIceREffMethod) )
    !
    select case ( CloudIceREffMethod )
    case ( 'Const' )
      IDCloudIceREffMethod = IDCloudIceREffMethodConst
    case ( 'Lin' )
      IDCloudIceREffMethod = IDCloudIceREffMethodLin
    case default
      call MessageNotify( 'E', module_name, &
        & 'CloudIceREffMethod=<%c> is not supported.', &
        & c1 = trim(CloudIceREffMethod) )
    end select


    ! Allocate a local variable for O3 distribution
    !
    allocate( xyz_QO3(0:imax-1, 1:jmax, 1:kmax) )


    ! Initialization of modules used in this module
    !

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
    call SetGCMRInit

    ! O3 分布の設定
    ! Set O3 distribution
    !
    call SetO3Init

!!$    ! 地球大気向け短波放射モデル Ver. 2.1
!!$    ! short wave radiation model for the Earth's atmosphere Ver. 2.1
!!$    !
!!$    call RadEarthSWV21Init
!!$
!!$    ! 地球大気向け短波放射モデル Ver. 2.2
!!$    ! short wave radiation model for the Earth's atmosphere Ver. 2.2
!!$    !
!!$    call RadEarthSWV22Init

    ! 地球大気向け短波放射モデル Ver. 2.6
    ! short wave radiation model for the Earth's atmosphere Ver. 2.6
    !
    call RadEarthSWV26Init

!!$    ! 地球大気向け長波放射モデル Ver. 2.3
!!$    ! long wave radiation model for the Earth's atmosphere Ver. 2.3
!!$    !
!!$    call RadEarthLWV23Init( &
!!$      & FlagSnow            &
!!$      & )

    ! 地球大気向け長波放射モデル Ver. 2.4
    ! long wave radiation model for the Earth's atmosphere Ver. 2.4
    !
    call RadEarthLWV24Init


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'CloudCoverforRad', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'cloud cover', '1' )
    call HistoryAutoAddVariable( 'CloudCoverRand', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'cloud cover', '1' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  CloudWatREff      = %f', &
      & d = (/ CloudWatREff /) )
    call MessageNotify( 'M', module_name, '  CloudIceREff      = %f', &
      & d = (/ CloudIceREff /) )
!!$    call MessageNotify( 'M', module_name, 'SWVer = %d', i = (/ SWVer /) )
!!$    call MessageNotify( 'M', module_name, 'LWVer = %d', i = (/ LWVer /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    rad_Earth_V2_inited = .true.

  end subroutine RadEarthV2Init

  !--------------------------------------------------------------------------------------

end module rad_Earth_V2
