!= 火星放射モデル Ver. 2 (汎惑星大気放射モデル 火星 Ver. 1)
!
!= Mars radiation model Ver. 2 (general-purpose radiation model for planetary atmosphere, Mars Ver. 1)
!
module rad_Mars_V2
  !
  != 火星放射モデル Ver. 2 (汎惑星大気放射モデル 火星 Ver. 1)
  !
  != Mars radiation model Ver. 2 (general-purpose radiation model for planetary atmosphere, Mars Ver. 1)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! This is a general-purpose radiation model for planetary atmospheres
  !
  !== References
  !
  !
  !== Procedures List
  !
  ! RadEarthV2Flux :: 放射フラックスの計算
  ! ------------          :: ------------
  ! RadEarthV2Flux :: Calculate radiation flux
  !
  !== NAMELIST
  !
  ! NAMELIST#rad_Earth_V2_nml
  !

  ! USE statements
  !

  !
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  !
  ! Physical constants settings
  !
  use constants, only: Grav     ! $ g $ [m s-2].
                                !
                                ! Gravitational acceleration

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数.
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数.
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数.
                             ! Number of vertical level

  implicit none

  private

  real(DP)         , parameter         :: MolWtH2O = 18.0e-3_DP

  integer                       , save :: NMolNum
  integer                       , save :: NPtcl
  integer          , allocatable, save :: m_MolNum(:)
!!$  character(STRING), allocatable, save :: a_PtclName(:)
  character(64)    , allocatable, save :: a_PtclName(:)

  integer                       , save :: iLoop
  integer                       , save :: NCalcInterval

  logical                       , save :: FlagRadEffectH2OVap
  logical                       , save :: FlagRadEffectH2OLiq
  logical                       , save :: FlagRadEffectH2OSol
  logical                       , save :: FlagRadEffectCO2Sol

  real(DP)                      , save :: CloudLiqRad
  real(DP)                      , save :: CloudSolRad
  real(DP)                      , save :: CloudMajCompIceRad
  real(DP)                      , save :: CloudNumRatio

  logical                       , save :: FlagDiurnalMean
  integer                       , save :: NSZA
  real(DP)         , allocatable, save :: a_DiurnalMeanHA(:)
  real(DP)         , allocatable, save :: a_SZAGW        (:)
  real(DP)         , allocatable, save :: xya_SZADeg     (:,:,:)


  ! 公開変数
  ! Public variables
  !
  logical, save :: rad_Mars_V2_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  public :: RadMarsV2Flux
  public :: RadMarsV2Init
  public :: RadMarsV2Finalize

  character(*), parameter:: module_name = 'rad_Mars_V2'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: rad_Earth_V2.f90,v 1.9 2015/01/29 12:06:43 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine RadMarsV2Flux(                       &
    & xy_SurfType, xy_SurfMajCompIce,              & ! (in)
    & xy_SurfAlbedo,                               & ! (in)
    & xyz_Press, xyr_Press, xyz_Temp,              & ! (in)
    & xy_SurfTemp,                                 & ! (in)
    & xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol,       & ! (in)
    & xyz_QDust,                                   & ! (in)
    & xyz_QMajCompIce,                             & ! (in)
    & xyr_RadSUwFlux, xyr_RadSDwFlux,              & ! (inout)
    & xyr_RadLUwFlux, xyr_RadLDwFlux,              & ! (inout)
    & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux       & ! (inout)
    & )


    ! USE statements
    !

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ステップ $ t $ の時刻.
                              ! Time of step $ t $.
      & TimesetClockStart, TimesetClockStop

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & y_Lat

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only : &
      & MolWtDry

    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    use set_solarconst, only : SetSolarConst

    use rad_flux_wrapper, only : RadFluxWrapper

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncome

    use set_Mars_dust, only : &
      & SetMarsDustSetDOD067, &
      & SetMarsDustCalcDOD067

    ! 雪と海氷の定数の設定
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only: &
      & CO2IceThreshold, &
      & CO2IceEmisS,     &
      & CO2IceEmisN

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtils1BinCalcPartProp


    integer , intent(in   ) :: xy_SurfType       (0:imax-1, 1:jmax)
    real(DP), intent(in   ) :: xy_SurfMajCompIce (0:imax-1, 1:jmax)
    real(DP), intent(in   ) :: xy_SurfAlbedo     (0:imax-1, 1:jmax)
    real(DP), intent(in   ) :: xyz_Press         (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyr_Press         (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(in   ) :: xyz_Temp          (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xy_SurfTemp       (0:imax-1, 1:jmax)
    real(DP), intent(in   ) :: xyz_QH2OVap       (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyz_QH2OLiq       (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyz_QH2OSol       (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyz_QDust         (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in   ) :: xyz_QMajCompIce   (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout) :: xyr_RadSUwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadSDwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadLUwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyr_RadLDwFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout) :: xyra_DelRadLUwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
    real(DP), intent(inout) :: xyra_DelRadLDwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)


    ! Work variables
    !
    real(DP) :: xyz_DelAtmMass   (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyr_Temp      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_QH2OVap   (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyrm_VMR      (0:imax-1, 1:jmax, 0:kmax, NMolNum)

    real(DP) :: PlanetLonFromVE
    real(DP) :: xyr_DOD067  (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_DOD     (0:imax-1, 1:jmax, 0:kmax)

    real(DP) :: StrFluxTOA0
    real(DP) :: StrFluxTOA
    real(DP) :: xy_CosSZA     (0:imax-1, 1:jmax)
    real(DP) :: FluxFactor

    real(DP) :: xy_AlbedoSW   (0:imax-1, 1:jmax)
    real(DP) :: xy_AlbedoLW   (0:imax-1, 1:jmax)
    real(DP) :: xy_EmisSW     (0:imax-1, 1:jmax)
    real(DP) :: xy_EmisLW     (0:imax-1, 1:jmax)

    real(DP) :: MajCompIceThreshold
    real(DP) :: MajCompIceEmis
    real(DP) :: xy_SurfEmis   (0:imax-1, 1:jmax)

    real(DP) :: xyr_RadUwFlux      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFlux      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyra_DelRadUwFlux  (0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFlux  (0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyr_RadUwFluxSW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFluxSW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadUwFluxLW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadDwFluxLW    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyra_DelRadUwFluxSW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFluxSW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadUwFluxLW(0:imax-1, 1:jmax, 0:kmax, -1:1)
    real(DP) :: xyra_DelRadDwFluxLW(0:imax-1, 1:jmax, 0:kmax, -1:1)

    real(DP)          :: xyza_PtclDens   (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_PtclEffRad (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_PtclMMR    (0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyza_DelPtclMass(0:imax-1, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)          :: xyr_MeanMolWt   (0:imax-1, 1:jmax, 0:kmax)

    integer  :: IndexH2OLiq
    integer  :: IndexH2OSol
    integer  :: IndexCO2Sol
    integer  :: IndexDust

    real(DP) :: CondMaterialDen
    real(DP) :: CloudRad
    real(DP) :: CloudNuclRad
    real(DP) :: CloudNuclDen

    real(DP) :: xyz_DelNuclNum  (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_LiqPartRad   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_LiqPartDen   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_SolPartRad   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_SolPartDen   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_CO2IcePartRad(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_CO2IcePartDen(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DustPartRad  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DustPartDen  (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_QDustPresc(0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: DistFromStarScld
    real(DP) :: SinDel

    real(DP) :: DustExtEff = 3.04_DP   ! Ockert-Bell et al. (1997)

    real(DP) :: xyz_DustDensScledOptDep(0:imax-1, 1:jmax, 1:kmax)

    integer  :: i
    integer  :: j
    integer  :: k
    integer  :: l
    integer  :: m
    integer  :: n


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_Mars_V2_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    if ( ( iLoop > 0 ) .and. ( mod( iLoop, NCalcInterval ) /= 0 ) ) then
      iLoop = iLoop + 1
      return
    end if
    iLoop = iLoop + 1


    k = 0
    xyr_Temp(:,:,k) =                                    &
      &      ( xyz_Temp (:,:,k+2) - xyz_Temp (:,:,k+1) ) &
      & / log( xyz_Press(:,:,k+2) / xyz_Press(:,:,k+1) ) &
      & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k+1) ) &
      & + xyz_Temp(:,:,k+1)
    do k = 1, kmax-1
      xyr_Temp(:,:,k) =                                &
        &      ( xyz_Temp (:,:,k+1) - xyz_Temp (:,:,k) ) &
        & / log( xyz_Press(:,:,k+1) / xyz_Press(:,:,k) ) &
        & * log( xyr_Press(:,:,k  ) / xyz_Press(:,:,k) ) &
        & + xyz_Temp(:,:,k)
    end do
    k = kmax
    xyr_Temp(:,:,k) = xyz_Temp(:,:,k)



!!$    xy_AlbedoSW = xy_SurfAlbedo
!!$    xy_AlbedoLW = 0.0_DP

    !  Surface emissivity
    !
    xy_SurfEmis = 1.0_DP
    !
    MajCompIceThreshold = CO2IceThreshold
    do j = 1, jmax
      if ( y_Lat(j) < 0.0_DP ) then
        MajCompIceEmis = CO2IceEmisS
      else
        MajCompIceEmis = CO2IceEmisN
      end if
      do i = 0, imax-1
        if ( xy_SurfType(i,j) > 0 ) then
          if ( xy_SurfMajCompIce(i,j) > MajCompIceThreshold ) then
            xy_SurfEmis(i,j) = MajCompIceEmis
          else if ( xy_SurfMajCompIce(i,j) < 0.0_DP ) then
            xy_SurfEmis(i,j) = xy_SurfEmis(i,j)
          else
            xy_SurfEmis(i,j) =                                    &
              &   ( MajCompIceEmis         - xy_SurfEmis(i,j) )   &
              & / ( MajCompIceThreshold    - 0.0_DP           )   &
              & * ( xy_SurfMajCompIce(i,j) - 0.0_DP           )   &
              & + xy_SurfEmis(i,j)
          end if
        end if
      end do
    end do
    !
    xy_AlbedoSW = xy_SurfAlbedo
    xy_AlbedoLW = 1.0_DP - xy_SurfEmis
    xy_EmisSW   = 1.0_DP - xy_SurfAlbedo
    xy_EmisLW   = xy_SurfEmis


    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    call SetSolarConst( &
      & StrFluxTOA0 & ! (out)
      & )

    ! 短波入射の計算
    ! Calculate short wave (insolation) incoming radiation
    !
    call RadShortIncome(                       &
      & xy_CosSZA, FluxFactor,                 & ! (out)
      & DistFromStarScld  = DistFromStarScld,  & ! (out) optional
      & SinDel            = SinDel,            & ! (out) optional
      & PlanetLonFromVE   = PlanetLonFromVE    & ! (out) optional
      & )
    PlanetLonFromVE = PlanetLonFromVE * 180.0_DP / PI
    !
    if ( FlagDiurnalMean ) then
      do n = 1, NSZA
        do j = 1, jmax
          xy_CosSZA(:,j) = &
            &  + sin( y_Lat(j) ) * SinDel &
            &  + cos( y_Lat(j) ) * sqrt( 1.0_DP - SinDel**2 ) * cos( a_DiurnalMeanHA(n) )
        end do
        xya_SZADeg(:,:,n) = acos( xy_CosSZA ) * 180.0_DP / PI
      end do
    else
      xya_SZADeg(:,:,1) = acos( xy_CosSZA ) * 180.0_DP / PI
    end if


    StrFluxTOA = StrFluxTOA0 / ( DistFromStarScld * DistFromStarScld ) * FluxFactor


    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    xyr_MeanMolWt = MolWtDry

    do m = 1, NMolNum
      select case ( m_MolNum(m) )
      case ( 1 )
        if ( FlagRadEffectH2OVap ) then
          k = 0
          xyr_QH2OVap(:,:,k) = xyz_QH2OVap(:,:,k+1)
          do k = 0+1, kmax-1
            xyr_QH2OVap(:,:,k) = &
              & ( xyz_QH2OVap(:,:,k) + xyz_QH2OVap(:,:,k+1) ) / 2.0_DP
          end do
          k = kmax
          xyr_QH2OVap(:,:,k) = xyz_QH2OVap(:,:,k)
          !
          xyrm_VMR(:,:,:,m) = xyr_QH2OVap * xyr_MeanMolWt / MolWtH2O
        else
          xyrm_VMR(:,:,:,m) = 0.0d0
        end if
      case ( 2 )
        xyrm_VMR(:,:,:,m) = 0.95d0
      case default
        xyrm_VMR(:,:,:,m) = 0.0d0
      end select
    end do


    IndexH2OLiq = -1
    IndexH2OSol = -1
    IndexCO2Sol = -1
    IndexDust   = -1
    do m = 1, NPtcl
      select case ( a_PtclName(m) )
      case ( 'water' )
        IndexH2OLiq = m
      case ( 'ice' )
        IndexH2OSol = m
      case ( 'co2_ice' )
        IndexCO2Sol = m
      case ( 'mars_dust' )
        IndexDust   = m
      end select
    end do
    if ( FlagRadEffectH2OLiq ) then
      if ( IndexH2OLiq <= 0 ) then
        call MessageNotify( 'E', module_name, 'Unable to find water.' )
      end if
    end if
    if ( FlagRadEffectH2OSol ) then
      if ( IndexH2OSol <= 0 ) then
        call MessageNotify( 'E', module_name, 'Unable to find ice.' )
      end if
    end if
    if ( FlagRadEffectCO2Sol ) then
      if ( IndexCO2Sol <= 0 ) then
        call MessageNotify( 'E', module_name, 'Unable to find co2_ice.' )
      end if
    end if
    if ( IndexDust <= 0 ) then
      call MessageNotify( 'E', module_name, 'Unable to find mars_dust.' )
    end if


    xyz_DelNuclNum = xyz_DelAtmMass * CloudNumRatio
    CloudNuclRad   = 0.0d0    ! dummy
    CloudNuclDen   = 3.0e3_DP ! dummy


    ! Effective radius
!!$    xyz_LiqPartRad = 1.0d-6
!!$    xyz_LiqPartDen = 1.0d3
!!$    xyz_SolPartRad = 1.0d-6
!!$    xyz_SolPartDen = 1.0d3

    ! H2O liquid
    CondMaterialDen = 1.0d3
    CloudRad        = CloudLiqRad
    call CloudUtils1BinCalcPartProp( &
      & CondMaterialDen, CloudRad,                     & ! (in )
      & CloudNuclRad, CloudNuclDen,                    & ! (in )
      & xyr_Press,                                     & ! (in )
      & xyz_DelNuclNum,                                & ! (in )
      & xyz_QH2OLiq,                                   & ! (out)
      & xyz_LiqPartRad, xyz_LiqPartDen                 & ! (out)
      & )
    ! H2O ice
    CondMaterialDen = 1.0d3
    CloudRad        = CloudSolRad
    call CloudUtils1BinCalcPartProp( &
      & CondMaterialDen, CloudRad,                     & ! (in )
      & CloudNuclRad, CloudNuclDen,                    & ! (in )
      & xyr_Press,                                     & ! (in )
      & xyz_DelNuclNum,                                & ! (in )
      & xyz_QH2OSol,                                   & ! (out)
      & xyz_SolPartRad, xyz_SolPartDen                 & ! (out)
      & )


    ! CO2 ice
    CondMaterialDen = 1.62d3    ! density at 150 K (Mangan et al., 2017)
        ! Mangan et al. (2017)
        ! https://www.sciencedirect.com/science/article/pii/S0019103516306832
    CloudRad = CloudMajCompIceRad
    call CloudUtils1BinCalcPartProp( &
      & CondMaterialDen, CloudRad,                 & ! (in )
      & CloudNuclRad, CloudNuclDen,                & ! (in )
      & xyr_Press,                                 & ! (in )
      & xyz_DelNuclNum,                            & ! (in )
      & xyz_QMajCompIce,                           & ! (in )
      & xyz_CO2IcePartRad, xyz_CO2IcePartDen       & ! (out)
      & )


    ! dust
    xyz_DustPartRad   = 1.0d-6
    xyz_DustPartDen   = 3.0d3


    ! Prescribed dust mass mixing ratio
    call SetMarsDustSetDOD067(                  &
      & PlanetLonFromVE, xyr_Press, xyz_Press,  & ! (in)
      & xyr_DOD067                              & ! (out)
      & )
    do k = 1, kmax
      xyz_QDustPresc(:,:,k) = &
        & ( xyr_DOD067(:,:,k-1) - xyr_DOD067(:,:,k) ) &
        & / ( DustExtEff * PI * xyz_DustPartRad(:,:,k)**2 ) &
        & * 4.0_DP / 3.0_DP * PI * xyz_DustPartRad(:,:,k)**3 * xyz_DustPartDen(:,:,k) &
        & / xyz_DelAtmMass(:,:,k)
    end do


    do m = 1, NPtcl
      if ( m == IndexH2OLiq ) then
        xyza_PtclDens  (:,:,:,m) = xyz_LiqPartDen
        xyza_PtclEffRad(:,:,:,m) = xyz_LiqPartRad
        if ( FlagRadEffectH2OLiq ) then
          xyza_PtclMMR   (:,:,:,m) = xyz_QH2OLiq
        else
          xyza_PtclMMR   (:,:,:,m) = 0.0d0
        end if
      else if ( m == IndexH2OSol ) then
        xyza_PtclDens  (:,:,:,m) = xyz_SolPartDen
        xyza_PtclEffRad(:,:,:,m) = xyz_SolPartRad
        if ( FlagRadEffectH2OSol ) then
          xyza_PtclMMR   (:,:,:,m) = xyz_QH2OSol
        else
          xyza_PtclMMR   (:,:,:,m) = 0.0d0
        end if
      else if ( m == IndexCO2Sol ) then
        xyza_PtclDens  (:,:,:,m) = xyz_CO2IcePartDen
        xyza_PtclEffRad(:,:,:,m) = xyz_CO2IcePartRad
        if ( FlagRadEffectCO2Sol ) then
          xyza_PtclMMR   (:,:,:,m) = xyz_QMajCompIce
        else
          xyza_PtclMMR   (:,:,:,m) = 0.0d0
        end if
      else if ( m == IndexDust ) then
        xyza_PtclDens  (:,:,:,m) = xyz_DustPartDen
        xyza_PtclEffRad(:,:,:,m) = xyz_DustPartRad
        xyza_PtclMMR   (:,:,:,m) = xyz_QDustPresc
      else
!!$        call MessageNotify( 'E', module_name, 'Unable to set appropriate density.' )
        xyza_PtclDens  (:,:,:,m) = 1.0d3  ! dummy
        xyza_PtclEffRad(:,:,:,m) = 1.0d-6 ! dummy
        xyza_PtclMMR   (:,:,:,m) = 0.0d0
      end if
    end do


    do m = 1, NPtcl
      xyza_DelPtclMass(:,:,:,m) = &
        &   xyz_DelAtmMass * xyza_PtclMMR(:,:,:,m)
    end do

    call RadFluxWrapper(                      &
      & imax, jmax, kmax,                           & ! (in)
      & NMolNum, m_MolNum,                          & ! (in)
      & xy_AlbedoSW, xy_AlbedoLW,                   & ! (in)
      & xy_EmisSW, xy_EmisLW,                       & ! (in)
      & StrFluxTOA,                                 & ! (in)
      & NSZA, xya_SZADeg, a_SZAGW,                  & ! (in)
      & xyr_Press, xyr_Temp, xy_SurfTemp,           & ! (in)
      & xyrm_VMR, xyr_MeanMolWt,                    & ! (in)
      & NPtcl, a_PtclName,                          & ! (in)
      & xyza_PtclDens,                              & ! (in)
      & xyza_PtclEffRad, xyza_DelPtclMass,          & ! (in)
      & xyr_RadUwFlux, xyr_RadDwFlux,               & ! (out)
      & xyra_DelRadUwFlux, xyra_DelRadDwFlux,       & ! (out)
      & xyr_RadUwFluxLW, xyr_RadDwFluxLW,           & ! (out)
      & xyr_RadUwFluxSW, xyr_RadDwFluxSW,           & ! (out)
      & xyra_DelRadUwFluxLW, xyra_DelRadDwFluxLW,   & ! (out)
      & xyra_DelRadUwFluxSW, xyra_DelRadDwFluxSW    & ! (out)
      & )

    xyr_RadLUwFlux     = xyr_RadUwFluxLW
    xyr_RadLDwFlux     = xyr_RadDwFluxLW
    xyr_RadSUwFlux     = xyr_RadUwFluxSW
    xyr_RadSDwFlux     = xyr_RadDwFluxSW

    xyra_DelRadLUwFlux(:,:,:,0) = xyra_DelRadUwFluxLW(:,:,:,-1)
    xyra_DelRadLDwFlux(:,:,:,0) = xyra_DelRadDwFluxLW(:,:,:,-1)
    !
!!$    xyra_DelRadLUwFlux(:,:,:,1) = 0.0_DP
!!$    xyra_DelRadLDwFlux(:,:,:,1) = 0.0_DP
    do l = 1, 1
      do k = 0, kmax
        xyra_DelRadLUwFlux(:,:,k,l) = &
          &   ( xyra_DelRadUwFluxLW(:,:,k,l) - xyra_DelRadUwFluxLW(:,:,k,l-1) ) &
          & / log( xyr_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & * log( xyz_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & + xyra_DelRadUwFluxLW(:,:,k,l-1)
        xyra_DelRadLDwFlux(:,:,k,l) = &
          &   ( xyra_DelRadDwFluxLW(:,:,k,l) - xyra_DelRadDwFluxLW(:,:,k,l-1) ) &
          & / log( xyr_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & * log( xyz_Press(:,:,l) / xyr_Press(:,:,l-1) ) &
          & + xyra_DelRadDwFluxLW(:,:,k,l-1)
      end do
    end do


    ! Output variables
    !
    call HistoryAutoPut( TimeN, 'DOD067', xyr_DOD067 )
    do k = 1, kmax
      xyz_DustDensScledOptDep(:,:,k) =                  &
        &   ( xyr_DOD067(:,:,k-1) - xyr_DOD067(:,:,k) ) &
        & / ( xyr_Press (:,:,k-1) - xyr_Press (:,:,k) ) &
        & * Grav
    end do
    call HistoryAutoPut( TimeN, 'DustDensScledOptDep', xyz_DustDensScledOptDep )

    call HistoryAutoPut( TimeN, 'LiqCloudRadiusForRad'    , xyza_PtclEffRad(:,:,:,IndexH2OLiq) )
    call HistoryAutoPut( TimeN, 'SolCloudRadiusForRad'    , xyza_PtclEffRad(:,:,:,IndexH2OSol) )
    call HistoryAutoPut( TimeN, 'MajCompCloudRadiusForRad', xyza_PtclEffRad(:,:,:,IndexCO2Sol) )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine RadMarsV2Flux

  !--------------------------------------------------------------------------------------

  subroutine RadMarsV2Init

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI

    !
    ! Physical constants settings
    !
    use constants, only: &
      &                  ConstantsInit, &
      &                  Grav     ! $ g $ [m s-2].
                                  !
                                  ! Gravitational acceleration

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & AxnameX, &
      & AxnameY, &
      & AxnameZ, &
      & AxnameR, &
      & AxnameT

    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    use set_solarconst, only : SetSolarConstInit

    use set_Mars_dust, only : SetMarsDustInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncomeInit

    use rad_flux_wrapper, only : RadFluxWrapperInit

    ! ガウス重み, 分点の計算
    ! Calculate Gauss node and Gaussian weight
    !
    use gauss_quad, only : GauLeg

    ! gtool データ入力
    ! Gtool data input
    !
    use gtool_history, only: HistoryGet

    use ni3_module

    ! 簡単雲モデル - 1 ビン
    ! simple cloud model with 1 bin
    !
!!$    use cloud_simple_1bin, only: CloudSimple1BinInit
    ! 雲関系ルーチン
    ! Cloud-related routines
    !
!!$    use cloud_utils, only : CloudUtilsInit


    ! 宣言文 ; Declaration statements
    !


    ! Local variables
    !
    real(DP)          :: WaveNumAlbedoSwitch
    character(STRING) :: KDOptPropNcFN

    integer :: NGQDiurnalMean

    integer           :: NcID

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /rad_Mars_V2_nml/ &
      & WaveNumAlbedoSwitch, &
      & KDOptPropNcFN, &
      & NCalcInterval, &
      & FlagRadEffectH2OVap, &
      & FlagRadEffectH2OLiq, &
      & FlagRadEffectH2OSol, &
      & FlagRadEffectCO2Sol, &
      & CloudLiqRad, &
      & CloudSolRad, &
      & CloudMajCompIceRad, &
      & CloudNumRatio, &
      & FlagDiurnalMean, &
      & NGQDiurnalMean
!!$      & CloudIceREffMethod,     &
!!$      & CloudWatREff,           &
!!$      & CloudIceREff

!!$      & SWVer, LWVer
          !
          ! デフォルト値については初期化手続 "rad_Earth_V2#RadEarthV2Init"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "rad_Earth_V2#RadEarthV2Init" for the default values.
          !

    if ( rad_Mars_V2_inited ) return



    ! デフォルト値の設定
    ! Default values settings
    !
    KDOptPropNcFN = ""

    NCalcInterval = 1

    FlagRadEffectH2OVap = .true.
    FlagRadEffectH2OLiq = .true.
    FlagRadEffectH2OSol = .true.
    FlagRadEffectCO2Sol = .true.
    CloudLiqRad         =  2.0d-6
    CloudSolRad         =  2.0d-6
    CloudMajCompIceRad  =  2.0d-6
    CloudNumRatio       = -1.0d0

    FlagDiurnalMean = .false.
    NGQDiurnalMean  = 0

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = rad_Mars_V2_nml,          & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    iLoop = 0


!!$    ! Identification of calculation method of cloud particle effective radius
!!$    !
!!$    call MessageNotify( 'M', module_name, &
!!$      & 'CloudIceREffMethod=<%c>.', &
!!$      & c1 = trim(CloudIceREffMethod) )
!!$    !
!!$    select case ( CloudIceREffMethod )
!!$    case ( 'Const' )
!!$      IDCloudIceREffMethod = IDCloudIceREffMethodConst
!!$    case ( 'Lin' )
!!$      IDCloudIceREffMethod = IDCloudIceREffMethodLin
!!$    case default
!!$      call MessageNotify( 'E', module_name, &
!!$        & 'CloudIceREffMethod=<%c> is not supported.', &
!!$        & c1 = trim(CloudIceREffMethod) )
!!$    end select


    call ni3_open( KDOptPropNcFN, 'read', NcID )
    call ni3_inq_dimlen( NcID, 'MolNum' , NMolNum )
    call ni3_inq_dimlen( NcID, 'PtclNum', NPtcl   )
    !
    allocate( m_MolNum          (NMolNum)                  )
    allocate( a_PtclName        (NPtcl)                    )
    !
    call ni3_get_var( NcID, 'MolNum'  , m_MolNum   )
    call ni3_get_var( NcID, 'PtclName', a_PtclName )
    call ni3_close( NcID )


    if ( FlagDiurnalMean ) then
      if ( NGQDiurnalMean < 1 ) then
        write( 6, * ) 'NGQDiurnalMean has to be greater than 0, when FlagDiurnalMean is .true.: ', &
          & FlagDiurnalMean, NGQDiurnalMean
        stop
      end if
      NSZA = NGQDiurnalMean
      allocate( a_DiurnalMeanHA( NSZA ) )
      allocate( a_SZAGW        ( NSZA ) )
      call GauLeg( 0.0_DP, PI, NSZA, a_DiurnalMeanHA, a_SZAGW )
      a_SZAGW = a_SZAGW / PI
    else
      NSZA = 1
      allocate( a_SZAGW( NSZA ) )
      a_SZAGW = 1.0_DP
    end if
    allocate( xya_SZADeg(0:imax-1, 1:jmax, NSZA) )


    ! Initialization of modules used in this module
    !
    call RadFluxWrapperInit( &
      & imax, jmax,                              &
      & Grav, WaveNumAlbedoSwitch, KDOptPropNcFN &
      & )


    ! 物理定数設定
    ! Physical constants settings
    !
    call ConstantsInit

    ! 太陽放射フラックスの設定
    ! Set solar constant
    !
    call SetSolarConstInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    call RadShortIncomeInit

    call SetMarsDustInit

    ! 全球一定体積混合比の設定
    ! Set globally constant volume mixing ratio
    !
!!$    call SetGCMRInit

    ! O3 分布の設定
    ! Set O3 distribution
    !
!!$    call SetO3Init

    ! 簡単雲モデル - 1 ビン
    ! simple cloud model with 1 bin
    !
!!$    call CloudSimple1BinInit
    ! 雲関系ルーチン
    ! Cloud-related routines
    !
!!$    call CloudUtilsInit


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'DOD067',             &
      & (/ AxnameX, AxnameY, AxnameR, AxnameT /),      &
      & 'dust optical depth at 0.67 micron meter at the surface', '1' )
    call HistoryAutoAddVariable( 'DustDensScledOptDep',    &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /),         &
      & 'dust density-scaled optical depth at 0.67 micron meter', 'm2 kg-1' )
    call HistoryAutoAddVariable( 'LiqCloudRadiusForRad', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'water cloud radius for radiation', &
      & 'm' )
    call HistoryAutoAddVariable( 'SolCloudRadiusForRad', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'ice cloud radius for radiation', &
      & 'm' )
    call HistoryAutoAddVariable( 'MajCompCloudRadiusForRad', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'Major component cloud radius for radiation', &
      & 'm' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  WaveNumAlbedoSwitch = %f', &
      & d = (/ WaveNumAlbedoSwitch /) )
    call MessageNotify( 'M', module_name, '  KDOptPropNcFN       = %c', &
      & c1 = trim(KDOptPropNcFN) )
    call MessageNotify( 'M', module_name, '  NCalcInterval       = %d', &
      & i = (/ NCalcInterval /) )
    call MessageNotify( 'M', module_name, '  FlagRadEffectH2OVap = %b', &
      & l = (/ FlagRadEffectH2OVap /) )
    call MessageNotify( 'M', module_name, '  FlagRadEffectH2OLiq = %b', &
      & l = (/ FlagRadEffectH2OLiq /) )
    call MessageNotify( 'M', module_name, '  FlagRadEffectH2OSol = %b', &
      & l = (/ FlagRadEffectH2OSol /) )
    call MessageNotify( 'M', module_name, '  FlagRadEffectCO2Sol = %b', &
      & l = (/ FlagRadEffectCO2Sol /) )
    call MessageNotify( 'M', module_name, '  CloudLiqRad         = %f', &
      & d = (/ CloudLiqRad /) )
    call MessageNotify( 'M', module_name, '  CloudSolRad         = %f', &
      & d = (/ CloudSolRad /) )
    call MessageNotify( 'M', module_name, '  CloudMajCompIceRad  = %f', &
      & d = (/ CloudMajCompIceRad /) )
    call MessageNotify( 'M', module_name, '  CloudNumRatio       = %f', &
      & d = (/ CloudNumRatio /) )
    call MessageNotify( 'M', module_name, '  FlagDiurnalMean     = %b', &
      & l = (/ FlagDiurnalMean /) )
    call MessageNotify( 'M', module_name, '  NGQDiurnalMean      = %d', &
      & i = (/ NGQDiurnalMean /) )
!!$    call MessageNotify( 'M', module_name, '  CloudWatREff      = %f', &
!!$      & d = (/ CloudWatREff /) )
!!$    call MessageNotify( 'M', module_name, '  CloudIceREff      = %f', &
!!$      & d = (/ CloudIceREff /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    rad_Mars_V2_inited = .true.

  end subroutine RadMarsV2Init

  !--------------------------------------------------------------------------------------

  subroutine RadMarsV2Finalize


    use rad_flux_wrapper, only : RadFluxWrapperFinalize



    ! 宣言文 ; Declaration statements
    !


    ! Local variables
    !


    if ( rad_Mars_V2_inited ) return


    call RadFluxWrapperFinalize


    deallocate( m_MolNum           )
    deallocate( a_PtclName         )

    if ( FlagDiurnalMean ) then
      deallocate( a_DiurnalMeanHA )
      deallocate( a_SZAGW         )
    end if
    deallocate( xya_SZADeg )

    rad_Mars_V2_inited = .false.

  end subroutine RadMarsV2Finalize

  !--------------------------------------------------------------------------------------

end module rad_Mars_V2
