/*
 * Copyright (C) 2020 Red Hat, Inc.
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#include "config.h"

/* This gets logged as part of the (more verbose) protocol logging */
#ifdef G_LOG_DOMAIN
#undef G_LOG_DOMAIN
#endif
#define G_LOG_DOMAIN "cockpit-protocol"

#include "cockpitmemfdread.h"

#include "cockpitjson.h"

#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <unistd.h>

gchar *
cockpit_memfd_read (int      fd,
                    GError **error)
{
  int seals = fcntl (fd, F_GET_SEALS);
  if (seals == -1)
    {
      g_set_error (error, G_FILE_ERROR, g_file_error_from_errno (errno),
                   "could not query seals on fd %d: not memfd?: %m", fd);
      return NULL;
    }

  const guint expected_seals = F_SEAL_WRITE | F_SEAL_GROW | F_SEAL_SHRINK;
  if ((seals & expected_seals) != expected_seals)
    {
      g_set_error (error, G_FILE_ERROR, G_FILE_ERROR_INVAL,
                   "memfd fd %d has incorrect seals set: %u (instead of %u)\n",
                   fd, seals & expected_seals, expected_seals);
      return NULL;
    }

  struct stat buf;
  if (fstat (fd, &buf) != 0)
    {
      g_set_error (error, G_FILE_ERROR, g_file_error_from_errno (errno),
                   "Failed to stat memfd %d: %m", fd);
      return NULL;
    }

  if (buf.st_size < 1)
    {
      g_set_error (error, G_FILE_ERROR, G_FILE_ERROR_INVAL,
                   "memfd %d must not be empty", fd);
      return NULL;
    }

  /* This number is completely arbitrary: it's much larger than anything
   * we're ever going to receive, but it's much smaller than any value
   * that would ever cause us problems (with integer overflow, or malloc
   * failing).
   */
  if (buf.st_size > 10000)
    {
      g_set_error (error, G_FILE_ERROR, G_FILE_ERROR_INVAL,
                   "memfd %d is unreasonably large (%"PRId64" bytes)", fd, (gint64) buf.st_size);
      return NULL;
    }

  g_autofree gchar *content = g_malloc (buf.st_size + 1);
  gssize s = pread (fd, content, buf.st_size + 1, 0);
  if (s == -1)
    {
      g_set_error (error, G_FILE_ERROR, g_file_error_from_errno (errno),
                   "failed to read memfd %d: %m", fd);
      return NULL;
    }
  else if (s != buf.st_size)
    {
      g_set_error (error, G_FILE_ERROR, G_FILE_ERROR_INVAL,
                   "memfd %d changed size from %zu to %zu bytes", fd, (gssize) buf.st_size, s);
      return NULL;
    }

  for (gint i = 0; i < buf.st_size; i++)
    if (((signed char) content[i]) <= 0)
      {
        g_set_error (error, G_FILE_ERROR, G_FILE_ERROR_INVAL,
                     "memfd %d contains %s character", fd, content[i] ? "non-ASCII" : "nul");
        return NULL;
      }

  content[buf.st_size] = '\0';

  return g_steal_pointer (&content);
}

JsonObject *
cockpit_memfd_read_json (gint fd,
                         GError **error)
{
  g_autofree gchar *content = cockpit_memfd_read (fd, error);

  if (content == NULL)
    return NULL;

  return cockpit_json_parse_object (content, -1, error);
}

JsonObject *
cockpit_memfd_read_json_from_control_messages (CockpitControlMessages  *ccm,
                                               GError                 **error)
{
  if (ccm->n_messages == 0)
    return NULL;

  gint peeked_fd = cockpit_control_messages_peek_single_fd (ccm, error);

  if (peeked_fd == -1)
    return NULL;

  return cockpit_memfd_read_json (peeked_fd, error);
}
