/*
** Copyright (c) 2018-2022 Valve Corporation
** Copyright (c) 2018-2022 LunarG, Inc.
**
** Permission is hereby granted, free of charge, to any person obtaining a
** copy of this software and associated documentation files (the "Software"),
** to deal in the Software without restriction, including without limitation
** the rights to use, copy, modify, merge, publish, distribute, sublicense,
** and/or sell copies of the Software, and to permit persons to whom the
** Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in
** all copies or substantial portions of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
** AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
** LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
** FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
*/

#ifndef GFXRECON_DECODE_VULKAN_ASCII_CONSUMER_BASE_H
#define GFXRECON_DECODE_VULKAN_ASCII_CONSUMER_BASE_H

#include "format/platform_types.h"
#include "annotation_handler.h"
#include "generated/generated_vulkan_consumer.h"
#include "util/defines.h"
#include "util/to_string.h"

#include "vulkan/vulkan.h"

#include <cstdio>
#include <string>

GFXRECON_BEGIN_NAMESPACE(gfxrecon)
GFXRECON_BEGIN_NAMESPACE(decode)

/// @brief Factor out manually-written code from the derived class
/// VulkanAsciiConsumer which is generated.
class VulkanAsciiConsumerBase : public VulkanConsumer, public AnnotationHandler
{
  public:
    VulkanAsciiConsumerBase();

    virtual ~VulkanAsciiConsumerBase() override;

    /// @brief  Initialize the consumer for writing to the file passed in.
    /// @param file A file to output to. The caller retains ownership. Do not close this.
    /// @param gfxrVersion The version of the GFXReconstruct project the convert tool was built from,
    /// including the branch and git commit in the case of a development build.
    /// @param vulkanVersion The version of Vulkan Headers that is being built against.
    /// @param inputFilepath The path to the source file as passed to the application.
    void Initialize(FILE* file, const char gfxrVersion[], const char vulkanVersion[], const char inputFilepath[]);

    void Destroy();

    bool IsValid() const { return (file_ != nullptr); }

    /// @brief Convert annotations, which are simple {type:enum, key:string, value:string} objects.
    virtual void ProcessAnnotation(uint64_t               block_index,
                                   format::AnnotationType type,
                                   const std::string&     label,
                                   const std::string&     data) override;

    virtual void
    Process_vkAllocateCommandBuffers(const ApiCallInfo&                                         call_info,
                                     VkResult                                                   returnValue,
                                     format::HandleId                                           device,
                                     StructPointerDecoder<Decoded_VkCommandBufferAllocateInfo>* pAllocateInfo,
                                     HandlePointerDecoder<VkCommandBuffer>* pCommandBuffers) override;

    virtual void
    Process_vkAllocateDescriptorSets(const ApiCallInfo&                                         call_info,
                                     VkResult                                                   returnValue,
                                     format::HandleId                                           device,
                                     StructPointerDecoder<Decoded_VkDescriptorSetAllocateInfo>* pAllocateInfo,
                                     HandlePointerDecoder<VkDescriptorSet>* pDescriptorSets) override;

    virtual void Process_vkCmdBuildAccelerationStructuresIndirectKHR(
        const ApiCallInfo&                                                         call_info,
        format::HandleId                                                           commandBuffer,
        uint32_t                                                                   infoCount,
        StructPointerDecoder<Decoded_VkAccelerationStructureBuildGeometryInfoKHR>* pInfos,
        PointerDecoder<VkDeviceAddress>*                                           pIndirectDeviceAddresses,
        PointerDecoder<uint32_t>*                                                  pIndirectStrides,
        PointerDecoder<uint32_t*>*                                                 ppMaxPrimitiveCounts) override;

    virtual void Process_vkCmdBuildAccelerationStructuresKHR(
        const ApiCallInfo&                                                         call_info,
        format::HandleId                                                           commandBuffer,
        uint32_t                                                                   infoCount,
        StructPointerDecoder<Decoded_VkAccelerationStructureBuildGeometryInfoKHR>* pInfos,
        StructPointerDecoder<Decoded_VkAccelerationStructureBuildRangeInfoKHR*>*   ppBuildRangeInfos) override;

    virtual void Process_vkCmdPushDescriptorSetWithTemplateKHR(const ApiCallInfo& call_info,
                                                               format::HandleId   commandBuffer,
                                                               format::HandleId   descriptorUpdateTemplate,
                                                               format::HandleId   layout,
                                                               uint32_t           set,
                                                               DescriptorUpdateTemplateDecoder* pData) override;

    virtual void Process_vkGetAccelerationStructureBuildSizesKHR(
        const ApiCallInfo&                                                         call_info,
        format::HandleId                                                           device,
        VkAccelerationStructureBuildTypeKHR                                        buildType,
        StructPointerDecoder<Decoded_VkAccelerationStructureBuildGeometryInfoKHR>* pBuildInfo,
        PointerDecoder<uint32_t>*                                                  pMaxPrimitiveCounts,
        StructPointerDecoder<Decoded_VkAccelerationStructureBuildSizesInfoKHR>*    pSizeInfo) override;

    virtual void Process_vkUpdateDescriptorSetWithTemplate(const ApiCallInfo&               call_info,
                                                           format::HandleId                 device,
                                                           format::HandleId                 descriptorSet,
                                                           format::HandleId                 descriptorUpdateTemplate,
                                                           DescriptorUpdateTemplateDecoder* pData) override;

    virtual void Process_vkUpdateDescriptorSetWithTemplateKHR(const ApiCallInfo&               call_info,
                                                              format::HandleId                 device,
                                                              format::HandleId                 descriptorSet,
                                                              format::HandleId                 descriptorUpdateTemplate,
                                                              DescriptorUpdateTemplateDecoder* pData) override;

  protected:
    /// @deprecated Prefer the shorter function signature without toStringFlags,
    /// tabCount, tabSize in new code.
    template <typename ToStringFunctionType>
    inline void WriteApiCallToFile(const ApiCallInfo& call_info,
                                   const std::string& functionName,
                                   /// @todo Remove toStringFlags, tabCount, tabSize
                                   util::ToStringFlags        toStringFlags,
                                   const uint32_t&            tabCount,
                                   const uint32_t             tabSize,
                                   const ToStringFunctionType toStringFunction,
                                   /// The formatted return string value excluding trailing comma
                                   const std::string& return_val = std::string(),
                                   // Set to false for bools and ints but leave true for enums, pointers, strings.
                                   const bool quoteReturn = true)
    {
        // Output the start of a function call line, up to the arguments:
        if (return_val.empty())
        {
            fprintf(file_,
                    "{\"index\":%llu,\"vkFunc\":{\"name\":\"%s\",\"args\":{",
                    (long long unsigned int)call_info.index,
                    functionName.c_str());
        }
        else
        {
            if (quoteReturn == true)
            {
                fprintf(file_,
                        "{\"index\":%llu,\"vkFunc\":{\"name\":\"%s\",\"return\":\"%s\",\"args\":{",
                        (long long unsigned int)call_info.index,
                        functionName.c_str(),
                        return_val.c_str());
            }
            // Ugly to duplicate this but more efficient than adding conditionals above or
            // having the caller quote with std::string manipulations.
            else
            {
                fprintf(file_,
                        "{\"index\":%llu,\"vkFunc\":{\"name\":\"%s\",\"return\":%s,\"args\":{",
                        (long long unsigned int)call_info.index,
                        functionName.c_str(),
                        return_val.c_str());
            }
        }

        // Reset the per-call reusable stringstream and use it to capture the full tree of
        // the function arguments including pNexts and struct pointers:
        strStrm_.str(std::string{});
        toStringFunction(strStrm_);

        // Dump the captured argument string into the file and close the argument,
        // nested, and top-level braces:
        fputs(strStrm_.str().c_str(), file_);
        fputs("}}}\n", file_);

        // Push out the whole line to the next tool in any pipeline we may be chained
        // into so it can start processing it or to file:
        fflush(file_);
    }

    /// @brief  Allow callers to be incrementally refactored away from the explicit
    /// formatting using toStringFlags, tabCount, tabSize.
    template <typename ToStringFunctionType>
    inline void WriteApiCallToFile(const ApiCallInfo&         call_info,
                                   const std::string&         functionName,
                                   const ToStringFunctionType toStringFunction,
                                   /// The formatted return string value excluding trailing comma
                                   const std::string& return_val = std::string(),
                                   // Set to false for bools and ints but leave true for enums, pointers, strings.
                                   const bool quoteReturn = true)
    {
        WriteApiCallToFile(
            call_info, functionName, util::kToString_Unformatted, 0, 0, toStringFunction, return_val, quoteReturn);
    }

  private:
    /// File to write textual representation of capture out to.
    /// @note This is passed to us in Initialize() and owned elsewhere.
    FILE* file_{ nullptr };
    /// Reusable string stream for formatting top-level objects like vkFuncs into.
    std::stringstream strStrm_;
};

GFXRECON_END_NAMESPACE(decode)
GFXRECON_END_NAMESPACE(gfxrecon)

#endif // GFXRECON_DECODE_VULKAN_ASCII_CONSUMER_BASE_H
