// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//	http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package slhdsa_test

import (
	"encoding/hex"
	"testing"

	"github.com/tink-crypto/tink-go/v2/signature/slhdsa"
)

const (
	// Copied from Tink C++ SLH-DSA signature verification test.
	privKeySHA2128sHex = "d44f6f06a73a07451096ad4bfbd240cb54b779330a65ed34ec0cd372c96fe48bf2b907c6" +
		"b73d52125c3930a195ef650baf7f68a07f4f3435408ac5ecaafaf4f3"
	pubKeySHA2128sHex = "f2b907c6b73d52125c3930a195ef650baf7f68a07f4f3435408ac5ecaafaf4f3"
	msgSHA2128sHex    = "6d657373616765"
	sigSHA2128sHex    = "677020005c0922919fb6b837c42783b93a71a10db1794c86683f5f22dd1f0b984a66f749" +
		"3cac79fc4a344ba07d1c2b7921e7b2ce50d7e5de20a47c9883059dd8e800a433d962a332" +
		"c3a884f0ae8cc14977baa4d0c9067ea9ac54c52ff9bd715ea8dc5210ff5dee5bdf00541b" +
		"b3d5a05305ba6da0920d35a596040a53f7704a430fa4eee2a21bc6d9dd942fa201cd9316" +
		"0d8e15347460bbef6ca99b0963c6050626855738d98201e8927e8543595422897954c62e" +
		"8263e09a7f97d16a4a268afb7dbe022cc468bb3ff8fdf4cc865675045a010226a67663a1" +
		"299d05913fed902eb3738cf326c45a712ac54ec462eee8bffd0624e7f40d34f24ffec7dd" +
		"00b3c73aa92e3558c003a4103c404790eef989f2dd6c98bf225e5a465a5120b40133fc97" +
		"5fbe82c42c8a15435d55a72b97b13835e4bc6d71cb336659bb6b1ed4d4ceba4e98acd61e" +
		"1180da27efc2ac6879a389b214bc8c4bf450410c786b82c0533a65ea6ce971281ba93de1" +
		"5f3983b57accb464dc9134490e88da36604f0b8bfeee10566c9a7c12cc18a42be2449cf5" +
		"237c4d4c42ec384d77a02bbfeec73b0929dd08dddde0aefc26509c2db2de6c337af131e6" +
		"1487005ab54a3af835942a79a6abaffbd51056cc83231d894d0ca89013af28e93758dd6e" +
		"913326ba459c0261df519b200b849cbd9994b55bfb8b89add46796cea6832fc9cfcda7c4" +
		"181abc2fc119fbe7a264da627a4093e922032fb09f1d87765f3fe03fdd4695b6ffac1e1e" +
		"a638141157960e539630e1b3f8e74b4ec62ed68a349a2020f08c59689e13eabaa7371fd1" +
		"98df194c985d2df93647ebd910f9ccbfc3465fed3fa015d6a890f42e1dd1359b7cd4c59a" +
		"4b0d2743f1acf733efe9f9fc349745048c0fef0e1852fd0e9f5189eae8fb04f98cc3bee0" +
		"a1574f46c4c54a700a0a669175867395a6c7bba1c0c4881db0b8688666beef55cf98c9aa" +
		"e3357f7c5c659daf36e72c2ae7dc2e757957bc45086c207aa2151eca1399f596732d25ba" +
		"17f9de45188eec6782f919daa18e2f7a37cb51a75421bf5d1b9397cad979bbef09a87c7c" +
		"57a871c3cbe0ebf36633281531b2bc9fbde43da9164101fb98204a56fea1d015a0518fd5" +
		"930020cfaf26486c8e20ae54590a2a026ed5f240f38675fbd25b27caa2af4920645bd6df" +
		"2dc49fd393adc95593cc3596c8f9b0c6183389b6bd4a1d1a5b682a41cfaac0dd4727df26" +
		"b4443bc886ed5bda5101a7b2582f59ee41aade795e21e6cb9abc4fcdc3b150c9cea6eb49" +
		"207b806b25020a369563a5ccbff289d808f005460f133dfbf4a621e9c93fc7d74edc0716" +
		"993aa727c4038b8c4855216f2d0301a22541ec1e93846f6c89f1f16dfe5b916bf16e82a5" +
		"e35a09f8865ad28d55833e9a566f7c12cb338e48c1d22c5233d2e17fee5b09eed81dfc65" +
		"5fb003f4045082b5269ce7263234bcb68225e403626249bd462f259ffaa065cf38318403" +
		"12d7993219f79e96e42e1e46801c7e809131510007b3aac2ff7c8f897d75109a26fb0d4e" +
		"e8c76a316a666f335a453ef8507a409809dfef3c4b1cf1f89ccf677a649084519dda826e" +
		"ad3d348a290bd1d39eff5f4a21487c210c37aac96e66e4bc9604ef988242a88ae0c24c7b" +
		"034cc1063f3daff93a6c3edcaed6d27fecc521155787389d0398b268e233b1803a35ffc5" +
		"9b94847acdb4685efecafa383971a9d1ff64f093db635221fbca2ddbf90b25cb8e45ce3a" +
		"474d16a9dd3018bd8b9a60287325000c0580004c9486fe4a65f0a9a6370523e1063d08f5" +
		"862ff966e8277ccf27860505944b431e372cc2f85036828e2a3b5c50456bc41deb27aec6" +
		"11b5650b202d13634eba83ed4715f9220b1c6eff1e378a6241b9f400dffd06ac588439a2" +
		"c25ee4b2796ade04d1d4bb2c616a0c6d37c95c0db4bab167f95c2e62682f17e585610562" +
		"f5db06127a1478fdf166ef1e309f1618595b2eddf045f41899697f69bc383b1ca6f25943" +
		"75f1541cc60c9be199d5ebc585daed75d8856e156c356b65d7bb253534102c790d6a14d0" +
		"556c3949f38e3102bf2388ff8afeb0ad0338fdc1e0351293bb84b2164c5922fab682b966" +
		"d7777109d9b180749604c5cadebcaa9b668ce966a93ea0945b8ea7b763c56f23b1b266c7" +
		"d7e155008d50c18d47c8d2ea11fb78e2f97018229c2d78c9815aa50b81ab39eda8e393b3" +
		"f74ca4ad466cafdccdc0cb3eecd992a6941464cdc1d8f14fcc7ed6314efe601c4340eaf9" +
		"edf0d185d5430c26d593100d3dfc0462cd5be9537d973ac4e035b19081a994a6b7041898" +
		"eda75393007013954764a90c4bcab67bccbc19e938c421fcc06f1a30e05fc138930f1746" +
		"74b9210d86ab26e8b5bf0e6adda4bfa9bce04addcf15b0005d508e59fd40b4d0b6ff7ace" +
		"35b37a88fcb7234d6857d0eed91090c9d866b601a8715fdc4aa44a2d19c98a29fe3d68a1" +
		"530b58f862f164530dcd30152d3d048ec3f8b24f17bb1616bab31075a05b120d864cdda3" +
		"071e89f5dfdf3f6050c4456b35d6f4d1a2106fb25c8bf4fc813bcd8b788b1719d0f7b836" +
		"fd65a4c9a0b0c4e64a8fcefd4e9b36093a54019e07d2f2204f4ba1077d388e9bb87ff3a2" +
		"85f7fe70f1f43054a05478485dede6c3e88b890d01709baf64cac191fc4b87f4879b51b9" +
		"75e9c21c0cd2a9288dce44fa998c5fe998748530a5e9c4bacc8d57073c6e0d3068d127f6" +
		"0e55eaae6c0cc205fd7ff17654b4150c5e728907440e96eb49ac1e3a06dc3ddc9615c66d" +
		"4d29f2267211c41d3b2e4281e6c022d5a1fd8ae20475a638ebe5ee520a4f9cb404c0df3f" +
		"ade560e9fae29d22098377e392b2f39fd273a553d881658df7c6650edaf62a4e15d2573e" +
		"18e1c2fe194173ca84b23426463fe14d235a758ab6ad0ca4f17b12801cfa9e3461640a7c" +
		"ed3ec1fd2e9a72786315e8d2da081cf93e14f67ebf7d6482d9bc0f9e4abc1e8e8529e648" +
		"0a3dabb86d927cafb3d08bb9275788230d81c2071540f1aaea0751ebbffb673949f69e6a" +
		"5fdb6f5c9d0f28b9a827bc7325517ab4b30e742f3e98cad59c48fa12b0ef9e885a03bb83" +
		"2c1efe721070cb8c177206d00f286a93f5bea0377ba86f7129dea527caae1e9f4ae9b0cd" +
		"4be6fb174019e04e9d2b0273ce44393d0867a81938625fa7d114b1f36a727614c8cee27b" +
		"25dd4b9411a4c4fc6a9b3007d682abb5f00f8ee12ec52979fe93069bab61349248b9a782" +
		"9731c29b56e87095df83d2d02ec8513886bb7573fe755df88545fd142f682ab9792b9771" +
		"e9cf4df891a28a218ee9f60f266a292834b4fecebcdb6baf1857b622a3a0a6486773b987" +
		"2dc1e4b393ed149669f4a6c30acb931b578c2e0bca6e6d85171af86529500e1d05da6c83" +
		"0ff0bf1eb90c3ccfb0c578903565a7d733466cbf85041bce8c138a8f0f3b6e55b201570d" +
		"252b57d85eb92abcf2ab59a309fb94a78f9207bfd5bf1b828e6c1781f4ef4decf2feedba" +
		"845be48415bf09bf525942be085f426fcd2ad6a2f1f4ec92b189050a1546e7fec1d061c7" +
		"38c29d00e2462f2ce72c483f5b4613ea0c82def0bc289df9e5ba9eebdb0b870a9ab71f08" +
		"c720e4883f72dfa49a05f88d002afc4e43e00e38480fb576ec07b4c014c2973eb994fb67" +
		"771b1b46d632764f7371976219da13d1ea6586fc08e5691a41ebd19da2cdcd730b8d118c" +
		"e1d8322a60262ebd2e0e03b64ef6ea93b2d76c9b7626d760b2efa8d72f00a4667741b749" +
		"c6aa706895c299d57e6822c7a7616efed153a9c3fd907119e9c7309ae428f8154f4ddec1" +
		"953f9731cc39f98982518cdb0e476d46b3873073d0297d4eb0c33df597814fe378cb11bc" +
		"b74c1737625b0d8444e13e9c29e7eab66c06c3dfd63c71fc267199e9df570306cc340d5c" +
		"31e28a6cbe4a78bb81f8f720c5f712ec19931f9df1efb445e1247f8db0e219d03d19b515" +
		"1e50fa162d346dcc39bff43e0b2b4ecb6c5e4977b485a965d4c717ed060378beda79b321" +
		"26052c537cf829a204b81a5b1d76a121d55a00e810beb36b67bb4960480cfb85465db65c" +
		"ac24368885190d59e6424a40c95edc82e9b1088c03339043a88ddb8fc071bf8a74e8c7d2" +
		"6b3576be535a66ce7bede9ce6442402472c43bbbd3306b39275118397a92ce70af9fbeb0" +
		"7ff1b7492316c1f9186ea5117ad26ec621d318786597b3d12d8426a7b98f1308eb8d911e" +
		"501f500f6ba8dd0420e3c6cd06ca9207f2841403404a1c7e1d72982558805fab2104d4b3" +
		"5506b137fa56960099a06f5d824ad0d3a545cfc3215e9b0ce7af8be1ff774357614bd95e" +
		"61a8d5c8919ef2204bf03be20415139eb8e06cffb797445441d3d77c6c98da65d694c285" +
		"cac42657000636ea9bf4c689552a6baedddcd4bc79b3f48aeb68668ae1eee796bc13fbf4" +
		"d23352d1dcf1c501105f0e9785e44ccecf271d08c4c0d5f6578995ecfdd6144398184a98" +
		"55d37e7ea4e3364a876e80da9e025d0e8b5e8e4f61470b0f37d8b8a6739eba236d606595" +
		"3f8d125555e1798e1e8ea9a7042bca0c24aba2c6d3156f99a7ee9644f6c608b07817f670" +
		"6e15f0e77b75ab72fc09789a9e86bdf83685ad0c276d85f119382d2d461cb8bb8a39a857" +
		"4bf7dc49d9b09f1a05c243f9c9b5fbc3f0f09c3169ef44df7fc1b8f7d211139ed0941d2b" +
		"9d91c6513ad67ea19abe63d9959eba1fe354b234a51f9db4c4954e25ca9e14aa4a478381" +
		"28ccefe8fc7c34034c2e3438eb95e5232af4a47191a4bee9e3c00230543382053760a782" +
		"e0ab533d07327776adad619ad4a2cff7fefdcf9b290d07807c5b23fa7a696b914e40f905" +
		"30581ecf2270e28eec1a0c485311e0f57a4807edeabe2eb7a89c4d8411d889c6164a6ff0" +
		"fc9604e410606ad9a3f6837809316d3551491278b141d22e20a2c3eaba47657e08b5e849" +
		"593332b3d1db5f6efd4ace89bd37a3b5d4049b1c06a58c0a9901bd1a9525cfd020b8dc16" +
		"dd8154473789960b47903f66f50e89d23ce637f360076b055c21ab597f45560cc2b7d48f" +
		"463a7d3120e03579371778928c41b36ccc276f52b1e4ffe7603cce185f8acb70f902b749" +
		"e500685bd3973f33ac7232e9b6e57cdbe584db93d852ad1a2654e9bea68a6425a3b21059" +
		"f4e44688de3f301bd042cc897795c83215438157831cb00dd94bb4d9cb42c31c15b6b33b" +
		"982db25c9abd89b008a44846120489b71c24bfd1794cd110799e8d3fe4296a12d35d506d" +
		"c31cfdae7a905e191cff535e12315f35100ddfa87f0004dcb0d549ac3d27e3541a1e90cb" +
		"51eb2e7bc46b98d4eaad72d11fdfebd10ee7f0c55075ab95839c5e7f7f57f142485e15a9" +
		"66e590ee4c697ab4260046731aec31e7a5011026bdaef234dd56c55ced28b18b7aa91940" +
		"ebb509e3a5ea4ba168c553f583c665a82076fd278cf3da36afa1ad3c43bf394aab561706" +
		"39ad3a885532d563e2c285eb8a716e93f04ee1329c8b14e77f8c3c49bb3857bdae9cf91a" +
		"437cc492d24ac74ceab9319459b382035e1026519a17b2c5d6b4923815889694750ed177" +
		"37f478838bd842057f3642515c41fabd0ecc07729c5bbffa788800d916d70d2662c096d4" +
		"34cf47794dbb6db5d17ba9b8f47664759cfc7394aa69219a4c4c58d90cc520dd0c7411e3" +
		"2d7ac9f614de165cbe188a19dcea6569ae8468977aecb5f75abc80f30ece39a9b0e79d70" +
		"47b189d1e00cb69a9fb2e57bca08c50e9693eab405dec5792c92a174beed639f08a85b70" +
		"11ca43a1e7854426996857fb5df6d1951e45846a9f233050d0542d968dd2bf265c3ffa47" +
		"560861fa4ac19438898fdbe6f417e7fea8ef09d6685ac2bcca20020b77baa7b56ed3515c" +
		"e0da3db93d1b101be93e12eca5265501c061d23deb9b5faab13d5f69b0897374c6596bc8" +
		"3c20ab4cf05219a5288beb48614e0bd1e47203ce2dfb56503d076ed30096e47106d755e1" +
		"b2a6ea9a5e7927cecc2373f52c17d8f99b329d4db48b1ad5122ad6604c0e8a4628f08549" +
		"d5317a0ed383aa82f86275d167f04dc5e0a74907b5975557eeababb6e32b26938a90cfbb" +
		"854e1fa0d6aafac8b00c513bed22816919251e231c9476ac4e3b4e8ea317758d18d015af" +
		"56d29c772b00c168a7c7c23e3710a8151325b8795a3ead2144445ce0b7c3e4b2f73dce68" +
		"1352f04becbdcc3800d07a822150bc96b81a90962de171c837fdd263d323f25547aff507" +
		"01f22469310ee209b41b15c0f192325ad475b0ed269ba84244f405f66c08fdaf3e230110" +
		"8914699af8e2b63c6d29093358a30039b06a0b830f263c972210fa429de1e5669ce2a652" +
		"9effd80260121d481d7fc8f111edd46af020209b2d8c668c37fb1a82645413ccf4d6d0c2" +
		"5c4542cbfb55e65c02d5c7ee631615973f2a0a32ed413a7ff07c0c6f804f3c03db1ef539" +
		"a1b151540fc32849b45696797fddf3b970f840b2233f08377783e9cecfd77c4dd2da0c7a" +
		"afb765c6ccbecbffaad01f8460c1c2a649720f8fb61693d430623603b96d98a31401c666" +
		"787df53720f81aec8eeea773dc20279c0892028f3ab7639d98c41ef082f140463dbb7ef5" +
		"33544c7fa14ad9e0501223f2f9ba544bd5f80c5cfae08876c77333cbb8746324160082fe" +
		"9ae9ba505ad16d5ebb921c6dad8cbc75b33cb3ab792266fd49a1b109a2121df8dde069a2" +
		"a51f63eb5c90fc5e7f8df11f123f1b7a7d057b8895d24f5c4471e749aa6e00ff57fb081a" +
		"6602405f1c89a48bfd8479dcd2f6910e1f7305964e0e35c081b5aafd04c8ad7fdbc8c6b3" +
		"d44099cbfcb451907726481a04109e25c8fd70a76cd5ca80cc018384a9ae6a557d645b0f" +
		"54d3e9f76b2978a326211c67a26ced1171703b576b17d8332b38c014853e0e9a3f5aa6cc" +
		"5e30d888f2b32c54f915dfdd6dfcae626c747e85f9a0f8658c11e738bd9c8c724160c79a" +
		"cd3d9ceb4e80495d6e7b48fb63f1e55f4a9e310e92fa5ba5d93580718ae678c9ded688a0" +
		"c8e75197a334d4bc8501b34cda264cb7e0a3232e2c788f218f8e3979f0f7b280ca8f85f5" +
		"9c75b1a2740911c381f93b016d8c79453cf87dedebc2f127c65426555d9e37d971d5ede6" +
		"ba1f6b97d0b8d1558be4e72545285972961e28b70efe2d7a41076ce394d6f17a0cead75d" +
		"3f5fb62b852ecb18a85698e82dfe332f80dd8492012d5f988e66a426532c36ba2365f5fe" +
		"55b92d1b8652e83e40da5201b2f75ab8cacbe61d9800df3a7282d4c4130f3cca6aa45c6f" +
		"e73732799e612d1ef694757fc163fa3dd161c894227673a7f3193be0b652c01226ef9519" +
		"f415b0d7bb823fb240077dd1217f69445262122e74c015c2c3506f8924dba9897712168c" +
		"54f604a4ef0697265f3a3e54aaf3c93ca933bad4d707cd2b4728054b12dee2ac8ec172e9" +
		"f58924cd452aa01dbee114201d719edd8d052e1284ec9522492f4415725e76994456af3b" +
		"930778d67b891c167621fe68d9257c64fd0e14ae6d7b63bf95a1d2dab75c90f50c8c0451" +
		"bde828fdee84cf966f032bf9976aa1ea96d18aeaf0aa327071e87c19cece6d537a2fac6b" +
		"b69f0ef8b733bc7fe3fdc9a26e0da75b6422416ea6bfe69d1ee921a0937b3699069d71f9" +
		"f42e9e66a1ffff14b7d86d0209a8e7bf215cd645e52b8736c433c3191af6432164a25ed6" +
		"b4c2050d2d5709b52337d061637b230efa98f4cfe81dc46c4ffa8e4b92b17689af9a2691" +
		"5540604f5614a68cf0e0f43f98e3c7d4765a7aee54770b279c21d4c4439af654cc486ce1" +
		"92b36748f4b06a64ddb6a2d84feb7b318e18d65a5fbc307c8399a62d6954bec2485d9d92" +
		"4810adf675247fa3940e1702da6ae864af97996a20679e6e52c8e18fc8ef18a32fdabc4d" +
		"826f919622928e1900a8054ea8015eae3b6538b5833011e72488616f3033a55e1eb998e4" +
		"8ac443f3efba7d8e7349c43b0ab4df18f106d8f235bed643ec051f5c524f0e0493239cb1" +
		"77e9c2d3d6ebeec99a222ed365cc54563efe18984b13bd3a6480adba03c255ae10c4c8a2" +
		"59fca9e7ea26d487279933ca1b81718f830f8d5059416e22b7f53a2e675e4591810de044" +
		"83881f6ad08c69f027723220168d870d0e41520091fad771fe2937daa9196833f49a6982" +
		"2a7c7d71dd557c5206d2d00ccd067707aa407e29739de1898bc45978599099c5e0068414" +
		"b5a559539717c4fa8eadc9638fe9065aaa0112cd08989b35a318ba5c09354f2e46f437ac" +
		"321c89a0d6ef3201ed13f681c6962d78d5f0d0774639e42febea7dc85512c1f0f278c7aa" +
		"ab29c50323a23c852c01bcd9a0f6c86c8af9d68ffc32f40ef267849f895fc8735c08f1ef" +
		"9b09a26f1724e83b5f039950a6fb8029f13ed7cfd91068f05d4abda87f698aad0beb5b14" +
		"63edca5b6320a71ed493c78f54bf4111928969e627d791fadbecf38c5e96f221eff2abaa" +
		"66494acf6bdad1c45db239541e43159c1dc4c240b1f5117823843c1cb01c92813354094f" +
		"11e7200d963226fb265d61c496ab296a27894bce9cf1b25a9ea9725ffe2a1b7b34652281" +
		"e683f5bddd34ad920736cc179476cef3eae7f5971e4954885c9586025d65bd37f3011ae4" +
		"66dc152ad67d3fd6535637eadafb5de121abbb773fa54584074dbd83f7039a7fb10b87f4" +
		"24a24bf65fc1a7aefd998ba53ceac8ca3f272f4364a2f6e0ade0369eb57d97fbed155a55" +
		"33cf03b61ff7f5933c66b57fe8bc364d9e6c96851b9464bc50f469598646f7157efe5516" +
		"95d283409b397bdf2f2402cd0a7629986d011e871c84af690738ac8fe727736bbf00aec9" +
		"736a4c846a7f76ebfbeebd7c2c1dcb5784cb569cb86c287d4ac04905061fe5754323335c" +
		"4a0a806fa8a253c1fa3c933512650be1c0c65cfe98851accfc80fd76226fe33785003c17" +
		"6790cdccf459800f5631da92d1636f17e7e0a6eb2121e45dd1091be5ed2de90eae77c6dc" +
		"e6e9bd3d1871d2cae994e7c84fed03aa0ddb99104ff4bcf725ef9554e5ea0d7f0bb91d79" +
		"38c052df93c875e6dec129bdea6e4c7192ed11f646dd3de7ca0671841cf9254003968799" +
		"56719be890eb54fa62bbc061b8416d18f47f9275b935c055e6a01dfa8b80d6b87b022371" +
		"9d52f6c02b9e18e52faf141099f1dfa84ad237743486702715a52e43ed5ce9edc7e9f3a4" +
		"801ad0391df7af438a329bf09e0c05a0af22871f14fe31d2160a84707002383d2f5c6943" +
		"a8cd430d9f6d852b79ae14a155fd8135e8fd96cb2fc0bdbc2db636eca660dd7f95b58c35" +
		"b06371858e3c424de3bb66898f107cf2c91a3ca1e5339ed0ed70801b66755161c01f6bac" +
		"92f465fe51f829bd23d2d60e5efbc976388f67e721103888eccfe06b98d61bb06725947e" +
		"6d1b40005d6841cee60a4a44b1aea78d0e29b8d9559835bd6f30b5a6709da9dad96061f3" +
		"1c459968e8738ba35285ad5cbc43f8f8a756c4575624492516e2b8d8d0e6006ed969aa7d" +
		"20ec67ccdabaec8eda981946984d77f763e3fdb67e64a9952791882b0678f2d346642654" +
		"2c79821b1b5aafbb8fde9d0a7d05655bb6c08f4426a819859844115ed8d5fea9bb2a97da" +
		"52b7f5f79a21e199ee167675a660a7c340ca598a968e3107dadfc5b96030dae6dd8c4380" +
		"c3e97f4321ce3e477099bf0f8e675b4dfa10e9aa655c8582828fb20c85fface950f79df9" +
		"0cda4820d2ea21939cad07d8aacdfe47c3e2176663815114cdf351a5184a51a29da690b8" +
		"f75010e3196fda09957e74c3dcd9b94a677bbf40392b65eee6c1fcc254fb1fbe846b665d" +
		"42b597f58528e804250bcf81735b2c5593de39da267c4309f5d1288c4674eda10ee6ba43" +
		"520368ac3ed832d3dd57b5f9c7e449b6fd631fbef0c19996527273f695afd5b42f6eeb2d" +
		"b00a3e16210a1d9322122cbff34497065f921a61ee4bff495a7dc932481e008c05f24d08" +
		"0cba5c993ffa8f842e5de3e15c0d536b2111b886d3ec307be777a5e91523278bca5ea528" +
		"64ae0861546cecaeb99db0dd3146e582bc192bd85c6dc9b8ffbffef3b490275cbbf26f4b" +
		"b8066f874d093e6f29f7cb6a9aad327fd2390f62d87758de01f60139b25fae885e26575f" +
		"25c6996332d2ab0ee19b1b11e6bd920ef10a204ac60160dbf7908eb31fc4e9ee183b9405" +
		"5ff93c5d61c65633b2613587815fdd7d7140c0169ac6c395013d1dd940ea6eaf1e03a407" +
		"af2ef43826bf9b5ba8fb5606eba499a55c25dee71fea4d90dd0487ff878da4ae12ce3bd9" +
		"ca213fa2d02d218f7fd0288ce856303dc1c25094843eec22342b8b583549a7c34fdb82b8" +
		"b304b34da42d7730e4fbbc4632178934acfe2b1914a50c1db64615463db5b5c62c898a65" +
		"cd794d6b8a21b9c0295e406a9136d7e663f6070512dd9912a17f1f02d8f1f02909503c5d" +
		"54fefee23886d23f17c030dbc16c6e5b89b5270c4de69b77fa8ed2b976c381879854590f" +
		"2a2636688d382e27d11f29ae73db88abe4275db37622a7650ca9eccb7cebc59693170796" +
		"f4297e7846d2c9605b2362c52ec6c9601bff0a13176e491fb6f35e45f03d6773a2e6581c" +
		"c436286c40501cd05e4dd25ee9ed23cf692a227b90179884a4e340156289fa4332254265" +
		"40da3c40426197b70acdf0f096cd049264c6f2273c308aae61648b7358c3ef1108114647" +
		"79e0aec392eed07d3c19a0487022965f5a588aa27bcb08af860610174210a9b6dcad1fd5" +
		"664eff817cf90e745233e303eea0233640080920febe47372eaa67b32215982d8d268389" +
		"b0fa2746c91049dd689726821d823d3afc4f9ef2e547086aa15f6ddfd8e287e517181ba9" +
		"c6b4330720201667665ac24c6c7b73779eef718035065710b4236d132e94202f3f2cfa9d" +
		"bf52c3d448f163f0c4427cf8d933bedd364f370aeec777e79743d8184d052343720867a1" +
		"4911d35c6fd86dc51a655fd23b34ab33307f50a5de8852f607fb11a7a9fcdb7997498a56" +
		"fa7fc212b26209edcb7ac03d3be52d261e2f7f8120eec51391f386badd591a9fee82e49b" +
		"375095b31048af565178c21d89e4f1017f32457e62e41013e49a68d3eec723d4dd16c7d6" +
		"e11c1d784c5706b9740f3ad6acb5e3bd2ec10eb4f1226d947605fdc85f0d703eec9cb739" +
		"d6be581b1a2acc37bd4d90aff0b2687e3b3464a48011ee0b9a1c571c02fc4efb94091dbc" +
		"e2357266713e160718d8b5ab1bacb6cb5330ff2e9dea49d11a18eb924d1c3edb7b4fff9a" +
		"6c60f23dfd1cc4c9"

	// Generated using the reference implementation (https://github.com/sphincs/sphincsplus).
	privKeySHAKE256fHex = "5b099ffc42a7b162c9cfbc188d7bffb19e0250ab36220b7de908e6af0f494456300339e3e832ba67eba181a30933aeb62f57d117fb2d28cc8271d359e0295f53c543272795c138347744d243ee143a0bd8e978e9638fff2ae5af533614316cb6ab4de2bd00bd9d54d21dafa97fd46155f5717a8bb4ef0b854c3cef55e5c553da"
	pubKeySHAKE256fHex  = "c543272795c138347744d243ee143a0bd8e978e9638fff2ae5af533614316cb6ab4de2bd00bd9d54d21dafa97fd46155f5717a8bb4ef0b854c3cef55e5c553da"
	msgSHAKE256fHex     = "6d657373616765"
	sigSHAKE256fHex     = "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"
)

type testKeyPair struct {
	pubKey  []byte
	privKey []byte
}

func generateTestKeyPair(t *testing.T, hashType slhdsa.HashType, keySize int, sigType slhdsa.SignatureType) testKeyPair {
	t.Helper()
	if hashType == slhdsa.SHA2 && keySize == 64 && sigType == slhdsa.SmallSignature {
		pubKeyBytes, err := hex.DecodeString(pubKeySHA2128sHex)
		if err != nil {
			t.Fatalf("hex.DecodeString(pubKeyHex) err = %v, want nil", err)
		}
		privKeyBytes, err := hex.DecodeString(privKeySHA2128sHex)
		if err != nil {
			t.Fatalf("hex.DecodeString(privKeyHex) err = %v, want nil", err)
		}
		return testKeyPair{pubKeyBytes, privKeyBytes}
	}
	if hashType == slhdsa.SHAKE && keySize == 128 && sigType == slhdsa.FastSigning {
		pubKeyBytes, err := hex.DecodeString(pubKeySHAKE256fHex)
		if err != nil {
			t.Fatalf("hex.DecodeString(pubKeyHex) err = %v, want nil", err)
		}
		privKeyBytes, err := hex.DecodeString(privKeySHAKE256fHex)
		if err != nil {
			t.Fatalf("hex.DecodeString(privKeyHex) err = %v, want nil", err)
		}
		return testKeyPair{pubKeyBytes, privKeyBytes}
	}
	t.Fatalf("unsupported hashType: %v, keySize: %v, sigType: %v", hashType, keySize, sigType)
	return testKeyPair{nil, nil}
}
