if (typeof T === 'undefined') require('../setup');

T('plus', function () {

  var t = function (addendA, addendB, expected) {
    T.assertEqual(expected, new Decimal(addendA).plus(addendB).valueOf());
  }

  Decimal.config({
    precision: 200,
    rounding: 4,
    toExpNeg: -9e9,
    toExpPos: 9e9,
    minE: -9e15,
    maxE: 9e15
  });

  t(1, 0, '1');
  t(1, -0, '1');
  t(-1, 0, '-1');
  t(-1, -0, '-1');
  t(1, 'NaN', 'NaN');
  t(-1, 'NaN', 'NaN');
  t(1, 'Infinity', 'Infinity');
  t(1, '-Infinity', '-Infinity');
  t(-1, 'Infinity', 'Infinity');
  t(-1, '-Infinity', '-Infinity');
  t(0, 1, '1');
  t(0, -1, '-1');
  t(-0, 1, '1');
  t(-0, -1, '-1');

  /*
    IEEE 754 - 2008 section 6.3
    When the sum of two operands with opposite signs is exactly zero, the sign of that sum shall
    be +0 in all rounding-direction attributes except roundTowardNegative; under that attribute,
    the sign of an exact zero sum shall be −0.
    However, x + x = x −(−x) retains the same sign as x even when x is zero.
  */
  Decimal.rounding = 3;

  t('0', '-0', '-0');   //   0 + -0 = -0
  t('-0', '0', '-0');   //  -0 +  0 = -0
  t('0', '0', '0');     //   0 +  0 =  0
  t('-0', '-0', '-0');  //  -0 + -0 = -0
  t('1', '-1', '-0');   //   1 + -1 = -0
  t('-1', '1', '-0');   //  -1 +  1 = -0

  Decimal.rounding = 4;

  t('0', '-0', '0');    //   0 + -0 =  0
  t('-0', '0', '0');    //  -0 +  0 =  0
  t('0', '0', '0');     //   0 +  0 =  0
  t('-0', '-0', '-0');  //  -0 + -0 = -0
  t('1', '-1', '0');    //   1 + -1 =  0
  t('-1', '1', '0');    //  -1 +  1 =  0

  t(0, NaN, 'NaN');
  t(-0, NaN, 'NaN');
  t(0, Infinity, 'Infinity');
  t(0, -Infinity, '-Infinity');
  t(-0, Infinity, 'Infinity');
  t(-0, -Infinity, '-Infinity');
  t('NaN', 1, 'NaN');
  t('NaN', -1, 'NaN');
  t('NaN', 0, 'NaN');
  t('NaN', -0, 'NaN');
  t('NaN', 'NaN', 'NaN');
  t('NaN', Infinity, 'NaN');
  t('NaN', -Infinity, 'NaN');
  t(Infinity, 1, 'Infinity');
  t(Infinity, -1, 'Infinity');
  t(-Infinity, 1, '-Infinity');
  t(-Infinity, -1, '-Infinity');
  t(Infinity, 0, 'Infinity');
  t(Infinity, -0, 'Infinity');
  t(-Infinity, 0, '-Infinity');
  t(-Infinity, -0, '-Infinity');
  t(Infinity, 'NaN', 'NaN');
  t(-Infinity, 'NaN', 'NaN');
  t(Infinity, Infinity, 'Infinity');
  t(Infinity, -Infinity, 'NaN');
  t(-Infinity, Infinity, 'NaN');
  t(-Infinity, -Infinity, '-Infinity');

  t(1, '0', '1');
  t(1, '1', '2');
  t(1, '-45', '-44');
  t(1, '22', '23');
  t(1, 0144, '101');
  t(1, '0144', '145');
  t(1, '6.1915', '7.1915');
  t(1, '-1.02', '-0.02');
  t(1, '0.09', '1.09');
  t(1, '-0.0001', '0.9999');
  t(1, '8e5', '800001');
  t(1, '9E12', '9000000000001');
  t(1, '1e-14', '1.00000000000001');
  t(1, '3.345E-9', '1.000000003345');
  t(1, '-345.43e+4', '-3454299');
  t(1, '-94.12E+0', '-93.12');
  t('0', 0, '0');
  t('0', '0', '0');
  t(3, -0, '3');
  t(9.654, 0, '9.654');
  t(0, '0.001', '0.001');
  t(0, '111.1111111110000', '111.111111111');
  t('NaN', '0', 'NaN');
  t(-1, 1, '0');
  t(-0.01, 0.01, '0');
  t(54, -54, '0');
  t(9.99, '-9.99', '0');
  t('0.00000', '-0.000001', '-0.000001');
  t('0.0000023432495704937', '-0.0000023432495704937', '0');
  t(NaN, NaN, 'NaN');
  t(NaN, 'NaN', 'NaN');
  t('NaN', NaN, 'NaN');
  t('NaN', 4, 'NaN');
  t('NaN', '4534534.45435435', 'NaN');
  t('NaN', 99999.999, 'NaN');
  t(Infinity, '354.345341', 'Infinity');
  t(3, -Infinity, '-Infinity');
  t(-Infinity, -Infinity, '-Infinity');
  t(-Infinity, -Infinity, '-Infinity');
  t(Infinity, '-999e999', 'Infinity');
  t('1.21123e43', -Infinity, '-Infinity');
  t('-999.0', Infinity, 'Infinity');
  t('657.342e-45', -Infinity, '-Infinity');
  t(Infinity, 123, 'Infinity');
  t(100, 100, '200');
  t(-999.99, '0.01', '-999.98');
  t('10', 4, '14');
  t('03.333', -4, '-0.667');
  t(-1, -0.1, '-1.1');
  t(43534.5435, '0.054645', '43534.598145');
  t('99999', '1', '100000');

  t = function (addendA, addendB, expected, sd, rm) {
    Decimal.precision = sd;
    Decimal.rounding = rm;
    T.assertEqual(expected, new Decimal(addendA).plus(addendB).valueOf());
  }

  t('593563412042.65071284077577238642546669796903116', '7399734315.5', '600963146358.15071284077577238642546669796903116', 78, 2);
  t('91834314419826105636913315.2250760351217688154427716325', '3453589484.0922087', '91834314419826109090502799.317284735121768815442772', 50, 2);
  t('1720025.190318830518434426408143', '66109246773146613017341021115.139667467395061272917480324885924044', '66109246773147000000000000000', 14, 5);
  t('84956243086401045.957714583212857676046952761360420789675', '236266946455107694023566236.1067', '236266946540063937109967282.064414583212857676046952761360420789675', 76, 4);
  t('27679002965950558720382323007580555707.1129725764', '3902452514422.1829293695210524511304537682971792', '27679002965950558720382326910033070000', 34, 6);
  t('9182914763445141913243843569172715469835.4885696227829197845538926', '7633444625534.6271670438', '9182914763445141913243843576806160095370.115', 43, 1);
  t('29495676.658886008145597040990327830', '6931766589765486564126938677173.788', '6931766589765486564126968172850.44688600814559704099032783', 97, 4);
  t('379110485.1879790330578106922187736430522', '44778134175412745601990207594724261558.24483', '50000000000000000000000000000000000000', 1, 0);
  t('294375205.06', '0.4210755138711679324480015392679141', '294375205.4810755138711679324480015392679141', 80, 3);
  t('97.8245', '678086929547037858952090.4586', '678086929547037858952188.2831', 47, 2);
  t('88488789676361292764625272698.8', '9635357185402290148967003271409315372896.608367362907029792956117646', '9635357185490778938643364564173941000000', 34, 4);
  t('6.6986623837182087397483631708665129676441', '4979636344219290763240.0069', '4979636344219290763246.705562383718208739748363170866513', 55, 6);
  t('70719500740711222218051791089542849911.838248343393159286401111269584528', '72611217359692246954.6315', '70719500740711222290663008449235096866.469748343393159286401111269584528', 100, 2);
  t('58732556421350640725226.234521198024002553600', '58121246202.94542710436381', '58732556421408700000000', 15, 3);
  t('3974581800977029548324180832.13384409524742094306099637', '3614381973858572.7418758814906985', '3974581800000000000000000000', 9, 6);
  t('697637766264066731981534075222456637269.8150', '84841485703909.9296471188463060', '697637766264066731981534160063942341179.744', 42, 3);
  t('47061.38', '1757281333528971368573.2805225778643923494834101231347769030', '1757281333528971415634.660522577864392349483410123134776903', 76, 2);
  t('1476403577722345877.206796162385539569704', '9942200550749890696264919346845234.2', '9942200550749892172700000000000000', 20, 4);
  t('12589380812542.2800574047701400763640862141749065', '6621312631.8566626344982182410585379', '12596002125174.136720039268', 26, 6);
  t('7882195982742109440258991594228896.97647548424454', '549984552171.2', '7882195982742109440259541578781068.17647548424454', 62, 3);
  t('79344567.9583146261700423456539738617142116267', '76770085769491324895202178331658.82467912070', '76770085769491324895202257676226.7829937468700423456539738617142116267', 75, 1);
  t('9803952036104667191144020833.60300205', '9495298573633466486861.2007267111985', '9803961531403240824610507694.8037287611985', 91, 6);
  t('42717742030595859.2062217089376928894751002693', '4392.7633096899809016584837135701339842639427', '42717742030600251.9695313989185945479588138394339842639427', 84, 0);
  t('175862554420496917507.72', '11748733600781956.625570241067964572154062958', '175874300000000000000', 7, 4);
  t('388528785785093377176597943386.8074346502385319459876897943393772766', '780579558740664211027356329768421.54760979278144', '780968087526449304404532927711808.355044443019971', 48, 3);
  t('64154311030289309886070986793918832.391977325870878', '85034840868876941709096163143445.05236', '64239345871158186827780082957000000', 29, 3);
  t('558109875620.787158', '4318677092534293169485.05154202819301085696282200030', '4318677090000000000000', 9, 6);
  t('8.1572164344501210656681193', '141383742681203268175705548539.5903424774279108405668653', '141383742681203268175705548547.7475589118780319062349846', 87, 1);
  t('475020553895752975125109317246970868141.19586602169742108309356725817022171', '50527762765152355076523195636814.169841064002562503683522141227541614', '475020604423515740277460000000000000000', 23, 5);
  t('623051342929926112359884960895415731939.5562490236742100882954960168', '503245645050146765988526273.758561914511343318476865211', '623051342930429358004935107661404258213.3148109381855534067723612278', 74, 6);
  t('531029963832660460416202620349662236.9', '1698435814148807364109817.622700275250303509772194494494138982090', '531030000000000000000000000000000000', 5, 4);
  t('6222.818987', '2745.84213504008632396977994287249342', '8968.66112204008632396977994287249342', 45, 5);
  t('83277468481757396400448039342892.13637251061', '949622766461448278187285221580222572.50707178970245998726148766810515863', '949706043929930035000000000000000000', 18, 3);
  t('50515084033799949576079332969.310908542889109621643886016029143', '9995275496554.2256723021298', '50515084033799959571354000000', 23, 3);
  t('58075371184786989216823527114006.8193336921861462723001823062864617438', '831577.37480626', '58075371184786989216823527945584.1941399521861463', 48, 4);
  t('354334.661931489528109', '49318851855669319.7668000614644666331306269353669354', '49000000000000000', 2, 5);
  t('171792414269.20500878', '7228500834975.6421662524366409', '7400293249244.8471750324366409', 86, 5);
  t('3686865567196977754914570753132702.86443144944286374816106555429', '252977292743732124354509767399171006.56792658736276850447471854354498', '256664158310929102109424338152303709.43235803680563225263578409783498', 76, 1);
  t('7772413527479.0292045567366574215506978466', '6951039673098790484081676762223797578.48972170861559580041987995', '6951039673098790484081684534637325057.5189262', 44, 1);
  t('897519.488463768126475213360003650464', '243345915368302813437999412228446907.03319139912318828693096', '243345915368302813437999412229344426.521655167249663500290963650464', 69, 1);
  t('9944701104687189448523634989441.1145387693210320276079647', '76461573432179845156280348475771563.06533912678377627209249', '76471518133284532345728872110761004.179877896', 44, 1);
  t('49868358282915444463601511745776323.6756', '426.127124853', '49868358282915444463601511745776749.802724853', 62, 1);
  t('462611390833337670596267664076995136589.8078310066750996553090298147792146374', '86220896429616059447274947806526227.378336884633', '462697611729767286655714939024801662817.18616789130809965530902981477921', 71, 5);
  t('3082844058.9630699111683783220405718767649310463', '9821678552282552.46', '9821681635126611.4230699111683783220405718767649310463', 85, 6);
  t('968956231687314239915639.8479448896945159', '2013053560776294882507697425832898126329.553542945365269900429', '2013053560776295851463929113147138041969.401487835059786', 55, 0);
  t('18829358460454562606796373222566693.8406444934478081754024757172', '245696827314924767344.62999383040940', '18829358460454808303623688147334038.4706383238572081754024757172', 65, 1);
  t('87272640829480092614399030.17403303652178770296488752252', '80645627676002739119663787.1389295', '167918268505482831700000000', 19, 4);
  t('110333142216083345526934754242807378414.8112810978', '8009375616663973267889.632407900829561884426774123507672', '110333142217000000000000000000000000000', 12, 2);
  t('7306882124433855860885172398759699317308.33181885682904', '7340.5772269222099866925020556183', '7306882124433855860885172398759699324648.9090457790390266925020556183', 73, 0);
  t('451443519.5', '40545287685504171383686533617478.0379433921591774655032923922180', '40545287685504171383686985060997.537943392159177465503292392218', 85, 1);
  t('9103707265251.90817165947074590284', '7132.1726911109784489', '9103707272384.08086277044919480284', 57, 1);
  t('1327536619868001635732671456652796081.82503453353162015625727828', '777720909904.3138244842545463', '1327536619868001635732672234373705986.13885901778616645625727828', 94, 6);
  t('94785498339319783.701937', '5.5284513', '94785498339319789.2303883', 45, 3);
  t('413864849197532888901858875400764791296.16643089', '1079561.227318262349', '413864849197532888901858875400000000000', 29, 3);
  t('45087586598522825025991129716000.35589454049730836', '85381176658152430597133806358376750.386015723730182079339361073643', '85426264244750953422159797488092750.741910264227490439339361073643', 95, 3);
  t('1.577', '5867361133548635578508631112769.512', '5867361133548635578508631112771.089', 37, 6);
  t('611867270001089057.5416980', '928462251839038030221601836154.34932586048493472075', '928462251839649897491602925211.89102386048493472075', 83, 0);
  t('2420842674752.5', '95330664601887717505762272.986252619687512204447', '95330664601890138348437025.486252619687512204447', 89, 4);
  t('81132109324361724111675518092.0430089148033907345892402992266', '56280915849510057765284537881333571.97531', '56280996981619382127008649556851664.018318914803390734', 53, 1);
  t('904671681146500577093476684800344905.4695', '94797652292709883543386718515254224988.65582754951556889876984', '95702323973856384120480195200054569894.12532754951556889876984', 82, 3);
  t('7088106609362197643328448383484980.038959814107063899', '670949858005717984030520198273540550.3057', '678037964615080181673848646657025530.3446598', 43, 5);
  t('99.21410642', '9083.34303307666325286270', '9182.5571394966632528627', 54, 4);
  t('3283.8936708900357134342', '496435946.937228974879461', '496439230.830899864915174435', 27, 2);
  t('676830762646205126.3112659907032497497725497603419138', '119932542098343475889901925878.775805518069408', '119932542099020400000000000000', 16, 2);
  t('340784617186492559520867952988798537.2875804', '619036430511269842219006698087.3401406612', '340000000000000000000000000000000000', 2, 5);
  t('173.3190299633996648505', '1824690614717784789521361964710.12123119395', '1824690614717784789521361964883.4402611573496648505', 96, 0);
  t('482415314400600371304757933326735596247.502', '64121617419177033485421455407.5858664539051557486457828966476314714', '482415314464721988723934966812157051655.0878664539051557486457828966476314714', 79, 5);
  t('375737470927645435406243965552430202253.631955167073623', '6338230241298196815836217859672.793', '375730000000000000000000000000000000000', 5, 3);
  t('29573885064956243070331189.3797092614486', '93819837825028613877.417105', '29573978884794068098945066.7968142614486', 96, 6);
  t('138120.700190473452', '968621311075606384.99548987853755995993', '968621311075744505.69568035198955995993', 45, 0);
  t('973.11990', '42154692435544969993439766.300521', '42154692435544969993440739.420421', 58, 5);
  t('554388861256.869722940300672388196938972302318081540', '1155262184508008.03323100857', '1155816573369264.90295394887067238819693897230231808154', 79, 2);
  t('3416418236.26400558205105933', '5.6479733', '3416418241.91197888205105933', 38, 5);
  t('2523992943408057896347889018.4465701438462144584', '506142712311227094740144.824697116', '2524499086120369130000000000', 18, 2);
  t('308403067969970653523590678072543444693.024992850150499393849129', '732241040738289252979020347.308655416021872045866556456848136589', '308403067970702894564328967325522465040.333648266172371439715685456848136589', 93, 1);
  t('3239064136731151.02891710688580557275640729053', '9083206824661083.5207932479503224412', '12322270961392234.54971035483612801395640729053', 80, 5);
  t('475331426870705.089157016478807', '9009215435143577267843833230423.14449365133718648192722331279072569', '9009215435143577743175260101128.233650667', 40, 3);
  t('105248227507845582838.20680935912346655579395894726346139163', '65223604191.5805291176792', '105248227573069190000', 17, 6);
  t('99.215190820118104610069782046', '16578.8209964235031908525606222947444696', '16678.0361872436212954626304043407444696', 63, 2);
  t('510539424488693733.41537434134721968131486', '327009.1496755290899995046724660116828516732779', '510539424489020742.56504987043721918598', 38, 1);
  t('315204602.81489858995', '3732180545.09', '4047385147.90489858995', 86, 4);
  t('5314592166616.1', '973281963604549959893809697900212092.07971879384048258785317865', '973281963604549959893815012492378708.17971879384', 48, 4);
  t('8.7166935890371592992002394888141553196926', '958659296273258091217856839735061.3647474524043063555717437013816', '958659296273258091217856839735070.08144104144146565477198319019575532', 69, 4);
  t('8540337154693663138.922681853498773232162632659387428466959', '714421549038288078923295717348706283749.09747291', '714421549038288078931836054503399946888.020154763498773232162632659387428466959', 96, 3);
  t('64218.4910654347361732', '1904584344376749176156995834410373768727.880243614504617078941694238', '1904584344376749176156995834410373832946.371309049240790279', 58, 2);
  t('79608939157757406640105379.1099825541008835456051880', '93.887979844359072766498', '79608900000000000000000000', 6, 4);
  t('997619509.8131634548180432', '408723341842195084378669.60436714294200657842422937736174', '408723341842196081998179.41753059776004978', 41, 0);
  t('581375185201895127979063.3148256', '14356334345.7574898', '581375185201909484313409.0723154', 33, 1);
  t('3527973624305181687800039808864.5573554755188', '12630611136837256867366017776380.0', '16158584761142438555166000000000', 23, 3);
  t('367630489717480484892012992734563672974.0', '45054048927965278885126985834716.66', '367630534771529412857291900000000000000', 25, 4);
  t('787513869174.71308620942912467', '23093894176823.452215801790110947022416582614', '23881408045998.165302011219235617022416582614', 69, 0);
  t('57753452879494271824903590.413829378140475084396185', '50545413776472197781.9967870405', '57753503424908048297101372.410616418640475084396185', 58, 2);
  t('17236192031420.2163137961486', '692838850166456533709637044970916752.36688025752892121241', '692838850166456533709654281162948172.58319405367752121241', 68, 0);
  t('4171050007467938807033605772358739230.9311059803557030664429028', '0.1923799694080', '4171050007467938807033605772358739231.123485949763703066', 55, 5);
  t('28478160393.9806', '277120391846395984.25', '277130000000000000', 5, 0);
  t('3061847804826051245630.65596944768453555988908615722', '40580833038231825168334767272165347.842548699388', '40580833038234887016139593323410978.4985181470725356', 51, 5);
  t('4307943411739045.1991060215780', '2498934952289.676303798694', '4310442346691334.875409820272', 100, 2);
  t('447264.0', '296280025996717104124065506095605789.16501275', '296280025996717104124065506096053053.16501275', 76, 3);
  t('7774084135404146930548457363682921350.47767504567574932067192218196557', '6356426099126393712.50823145833574225514684676449194435', '7774084135404146936904883462809315062.9859065040114915758', 56, 3);
  t('772351722055561484188074604225369559844.02', '4826191208867.1814570305', '772351722055561484190000000000000000000', 20, 4);
  t('252673622.185680951859328', '14604.688310802671654124022197881135913861968', '252688226.87399175453098212402219789', 35, 2);
  t('9844584158.071870640', '7376636821819775622965030201208199.7842455984620137382448558190', '7376636821819775622965040045792357.856116238462013738244855819', 67, 0);
  t('7402851913927.841295065397846204951008', '3095919939427099869935455000158274400904.445835412', '3095919939427099869935455007561126314832.287130477397846204951008', 76, 1);
  t('39454068592.5426357674', '539500064406478650256867488.44171988564589', '539500064406478689710936080.98435565304589', 78, 0);
  t('711378057536130546.05030965850391817985', '3519.9006655104107', '711378057536134065.95097516891461817985', 76, 0);
  t('4961172832120828815580484602760756.2012712908', '672041039664151754962281595761407065569.95071013621060637157419754', '672046000836983875791097176246009826326.15', 41, 1);
  t('7101.2850399893410837385161594686012015', '7447039100693496598.27485', '7447039100693503699.559889989341083738', 37, 3);
  t('4017400855744231199234363.40', '8680854252488567594955443.1697318101148721333039110519819430', '12698255108232798794189806.569731810114872133303911051981943', 91, 4);
  t('2059590390363552.40', '586325932054905969708260929.92327440514351633822636210224317619490', '586325932056965560098624482.3232744051435163382263621022', 55, 5);
  t('582983617.569929797014256654694471', '9981786309799768994686892934445.9295301169296065462066657484', '9981786309799768994687475918063.4994599139438632009011367484', 68, 6);
  t('2.6165698405910796049261368090493573329246', '1760827779548.1353871194941345580463266976978', '1760827779550.751956960085214162972463506747157333', 49, 4);
  t('42513713793037649412.20337267935799701850122725370100', '4884164426571880743457.9060113', '4926678140364918392870.109383979357997018501227253701', 88, 4);
  t('615013136392771057432168892493351121.844', '7501.90719259576291230', '615013136392771057432168892493358623.7511925957629123', 56, 1);
  t('439780502800870192246198170895028338.75503651737499', '54139159349728010041929589316148734373.238719628804657454831809670308', '54578939852528880234175787487043762700', 36, 1);
  t('2500282.56691918225466823870', '4468153172374579202195051.1192110124701356844021658', '4468153172374579204695333.6861301947248039231021658', 70, 3);
  t('3947906778978419575322273.5922', '527753211579218151805229658066345458285.5885089788593758900780037', '527753211579222099712008636485920780559.1807089789', 49, 4);
  t('224244602386820798357422068121543275.9722790644', '432961.0031220586705516514763153277784', '224244602386820798357422068121976236.97540112307055165148', 56, 4);
  t('41074785256868066207935.9658329758100047703778552011774401118298', '996313038437469337782350067491224840574.3439865837306077524137342817475940790588', '996313038437469378857135324359291048510.3098195595406125227915894829250341908886', 87, 3);
  t('78270192113308576879.230', '422705378961250342057111974970896025870.232502331134017349978026', '422705400000000000000000000000000000000', 7, 0);
  t('23154961663482879829897.443500', '49041587603060876.96', '23155010705070482890774.4035', 84, 3);
  t('612963.737191014', '2731304753434157847102688238900689.5878194', '2731304753434157800000000000000000', 17, 3);
  t('79.410900071205868462305254', '498855037827626504175503191449.252294529388770858879482', '498855037827626504175503191528.663194600594639321184736', 72, 2);
  t('6993101104301088.970619872035600302449779353214753901', '509465802045548078458021.19126386538627494681427019648867', '509465809038649182759110.161883737421875249264049549703423901', 90, 5);
  t('3173346.465280699493106096344514997257165695887', '25094425785458254607988571321.6118249859981254347581599952737191037300', '25094425785458000000000000000', 14, 3);
  t('4825986092816553.7635954902708301975634371809900', '6932262398609128466540251295633274.5129783804352371207490', '6932262398609128471366237388449828.27657387070606731831243718099', 87, 1);
  t('926199681466567517287973.10295', '228084522784513379301529587994.866786459974', '228085448984194845869046875967.969736459974', 49, 6);
  t('451757469368611461226744.675918562388474905198839780224990', '152843853624419159800008131.88722', '153295611093787771261234876.56313856238847490519883978022499', 60, 2);
  t('6779555902177154177229771584.099785903065', '7705328232185839682426202564296087172997.525840974954568455125376057221984043', '7705328232192619238328379718473316944581.625626878019568455125376057221984043', 86, 1);
  t('9536.995381842', '5743683556272978375327341031353872992805.228035142323187697966003805355354634434', '5743683556272978375327341031353873002342.223416984323187697', 58, 3);
  t('794180405697345273302591220141629717.0335032652959', '79253739255343474198020799617825451.0', '873434144952688747500612019759455000', 33, 4);
  t('7150644601128617400192415586930354936122.6334767464', '9109635205088641816430727590858041637.2704471583622041768929760514400617734', '7159754236333706042008846314521212977759.9039239047622041768929760514400618', 74, 5);
  t('73562280122283.3504585511130', '9831433225592360462.8352635', '9831506787872482746.185722051113', 98, 4);
  t('8956158610374625.509531016', '326445046650.0', '8956485055421275.509532', 22, 2);
  t('74356614405271046021497883758184078570.45526096329466906744072230232', '250086499545948148713695351897361487.2424378528824906540', '74606700904816994170211579110081440057.6976988161772', 51, 0);
  t('54438066500841374310337535135292.9180799710380482218974913', '93583236200855.457936825513836204', '54438066500841374403920800000000', 24, 6);
  t('992.10612', '1853924775.3578645099684579616948779829033812', '1853925767.4639845099684579616949', 32, 2);
  t('186864386191178904666377352418476.9026', '324887104993.09478140653570710242809082', '186864386191178904666700000000000', 22, 5);
  t('73975703304587932639.2181626587592846793346557', '385064.7711364633520856985649145914899', '73975703304588317703.9892991221113703778995702914899', 65, 1);
  t('444492371187821986647273.357556318861835197', '32419538466212897.7682790461375', '444492403607360452860171.125835364999335197', 90, 5);
  t('425.36', '4.8163620735655971354915724851187759884583', '430.1763620735655971354915724851187759884583', 58, 3);
  t('47140705104264.633543653650963362791573755527599', '5038770685966504.041420875593555678620', '5085911391070768.67496452924452', 30, 5);
  t('7680406.950844967098906022738005995808', '7170039143571111215189.2720128', '8000000000000000000000', 1, 2);
  t('56766742609.4324995621156186543537021757622234796', '5600154020118.606591501', '5656920762728.039091063115618654353702176', 40, 4);
  t('570657301559247483807209644.19', '347169349198778409985.34839429745774918204191942456937431', '570657648728596682585619629.53839429745774918204191942456937431', 74, 5);
  t('5361772037663933234405322789036276.33156433900413166799767144681154137323', '286704590022711173.40544742389525', '5361772037663933521109912811747449.73701176289938166799767', 57, 1);
  t('93732046861555957196550772609968.3472471313', '1859083952987593674898181.79957615855993107', '93732048720639910184144447508150.14682328985993107', 91, 3);
  t('92567721714303533.992771142280786530426395726718479', '4173646454390713911135550874061129.722450828911285695255462583174414', '4173646454390714003703272588364663.715221971192072225681858309892893', 98, 5);
  t('1375128263616714701.94878254650062', '46092727285436763451586.053119100', '46094102413700380166288.0019016465', 33, 4);
  t('73619022388103118.782952924468', '812143.274917215669264759', '73619022388915262.1', 18, 0);
  t('5863128081.032078098051593954', '3401644100547973363487331808740.559267563791452260218198', '3401644100547973363493194936821.591345661843046214218198', 76, 2);
  t('2035229845149050434432862.3913626', '324888035708707633659073773920117.1589359337264063', '324888037743937478808124208352979.5502985337264063', 82, 3);
  t('3809229847420761545133.0', '4820445969156504840260574583039569681560.574245660256', '4820445969156500000000000000000000000000', 15, 4);
  t('265038662972437534963025477565907689803.0955954524873', '8689727198065.214479051254116375289593', '265038662972437534960000000000000000000', 20, 5);
  t('467620089714294461431888766085.933012457041824326', '8454761303295998080511054.0533223', '467628500000000000000000000000', 7, 4);
  t('87894879970166522727794755.93237106514121963', '9.21735981924330694503993149149554469907', '87894879970166522727794765.1497308843845265750399314914955447', 61, 5);
  t('415465265615530648920289751139098887220.642143071162972307015579919362', '784797421468789423953053292044037319012.57789662938605515623570039453017199', '1200262687084320072873343043183136206233.22003970054', 51, 3);
  t('919416082026906893949479156604880336.255538841432616135466943633799', '786230185237.5400725225964740467962175280141', '919416082026906893949479942835065573.7956113', 43, 3);
  t('652545610682383466438682204.42146681125712', '90000817174099471.37775277', '652545610772384283612781675.79921958125712', 83, 1);
  t('834893074961901.316219159', '3494462.496814', '834893078456363.813033159', 31, 3);
  t('30955320958921614617178138289424354927.94679217117720836736847', '9343585893221.056', '30955320958921614617178147633010248149.00279217117720836736847', 93, 1);
  t('90487985719811418246283472474823.257227209130594638697376', '91288470132046108165996.51806996', '90487985811099888378329580640819.775297169130594638697376', 97, 3);
  t('8399816940955101.576691696184560166134', '1425122.40715', '8399816942380223.983841696184560166134', 73, 3);
  t('90835019536.019916982404274694597500578', '716647297150731356312375253472992368.5862189689245116804145741973840791708889', '716647297150731356312375344308011904.6061359513287863750120748', 61, 2);
  t('1.76835213775438870258717627564722245975', '7006467747024962494048481730509926588025.86559434478599359798030950728698427731', '7006467747024962494048481730509926588027.63394648254038230056748578293420673706', 81, 0);
  t('928715435770769947371.846861119120186848130047779221215181888', '54176632694146863536.023546974', '982892068464916810907.87', 23, 6);
  t('2592460567145.9081671605659594580786953609908858698', '7864631589857045160739640205084628.0', '7864631589857045160742232665651773.90816716056595945807869536099088587', 69, 5);
  t('550167718959711561547.0354625', '452302.08587611743779', '550167718959712013849.12133861743779', 53, 5);
  t('2012913.385532423507759516640479441', '103518982004003013118420061.943654190758892648', '103518982004003013120000000', 20, 5);
  t('49943050945228733544.6734986147', '368220952556842481.13025278166305776823174166647', '50311271897785576025.80375139636305776823174166647', 71, 6);
  t('6819435979706516062524.6356313748', '61269304801758183.91645059614218918999', '6819497249011317820708.55208197094218918999', 95, 5);
  t('7233786224735497889984695259.844810632657377449889809221767092', '6807990850759856236367040269664178400670.841947660632059789845', '6807990850767090022591775767554163095930.686758293289437239734809221767092', 92, 0);
  t('93353199365969854.9000648459972051740723', '9422.3888765011593114515314770219001723580808', '93353199365979277.2889413471565166256037770219001723580808', 78, 2);
  t('30980262.05713034537675', '778.835454408', '30981040.8925847533768', 21, 4);
  t('32615753444771455868335024685114814.9238176568928810345574', '7768512679320024.55715331942029263648769217', '32615753444771455876103537364434839.48097097631317367105', 55, 2);
  t('13632097595691451297658166154832.143013272134363070648360851205244642', '338823008282322586814270.072279678874150483663858', '13632097934514459579980752969102.2152929510085135543122189', 57, 4);
  t('6422084834544364179504578.997091732441389646011113', '749563746146212443758110.4060254814560251104', '7171648580690576623262689.4031172138974147564111', 47, 4);
  t('524628228909.16050485473888934', '157911548505.727158203964341656616832412311819792', '682539777414.887663058703230996616832412311819792', 95, 0);
  t('119792544.57876596369349895973', '27424886476597422405011090514661205.0', '27424886476597422405011090634453749.57876596369349895973', 95, 0);
  t('2069402156492526629947970885.71888875649933', '9983225233803399851557109.1989727', '2079385381726330029799527994.91786145649933', 81, 3);
  t('469964156809.608491', '595074692464717339784472687248240220.861973', '595074692464717339784473157212397030.470464', 82, 2);
  t('493360767167099894326435423534395819518.19695281187005', '7766465503.425829049644163982004', '493360767167099894326435423542162285021.622781861514213982004', 99, 2);
  t('99936880718101922830812712583.65032262815572509426570474', '33036974375228768881250337405734899.3544983736263577509194755440416', '33037074312109486983173168218447483.00482100178208284518518', 58, 1);
  t('7938089.9035748249886034322', '59938235780340155825294510015684098915.6306313894846960', '59938235780340155825294510015692037005.5342062', 45, 5);
  t('12020912075170693462106301094099.55717548283295340940081', '7925012204520701892409967356193913171038.423047370', '7925012216541613967580660818300214265137.98022285283295340940081', 69, 4);
  t('2403534785699117112878638516815488696057.917378367470941297385927', '6382502380424160003.479677660153263338013088885', '2403534785699117112885021019195912856061.397056027624204635399015885', 85, 1);
  t('698965328139372391512502.0393628796146', '349.3774499346277293528646834', '698965328139372391512851.4168128142', 34, 3);
  t('4557454883.68536507001647907979028423', '461212.6999283496597694751582137548', '4557916096.3852934196762485549484979848', 77, 5);
  t('4863433553099693015144122064473263537.90580375285849442237159037230012244', '729921864795343627150.6944153828593595441', '4863433553099693745065986859816890688.60021913571785396647159037230012244', 88, 1);
  t('8341821006482531885092.98411549285365', '3649787.9056678809068225979753383', '8341821006482535534880.8897833737604725979753383', 57, 4);
  t('50900271502855123535389576514508.6', '3529291535644.0', '50900271502855123538918868050152.6', 68, 3);
  t('65048735011187538001618360.6190283754497263475739381', '544369607320210256967259524797.5031', '544434656055221444505261143158.12212838', 38, 0);
  t('9980542928.586', '2.109934', '9980542930.695934', 55, 6);
  t('4691011429542960320268638161612830.142849169', '757715397475477414366.06830046', '4691011429543718035666113639027196.211149629', 83, 4);
  t('9874.54', '3856064270543742684218503.9123367482159408', '3856064270543742684228378.4523367482159408', 87, 6);
  t('68154879448120209877595453.386110402115092955259472912357', '27424826983.14981', '68154879448120237302422436.535920402115092955259472912357', 65, 6);
  t('37249053070475232499173177378.031244401', '907464305952.1105379598699518977559941948581426', '37249053070475233406637483330.1417823608699518977559941948581426', 65, 6);
  t('41.155993138147253723002006646182', '780464.5000578752798', '780505.656051013427053723002006646182', 89, 6);
  t('96920877597978100716777218.316958009791452354794880717409', '3188642196779775872022.94066144525722263195417766', '96924066240174880492649241.257619455048674986749058377409', 71, 6);
  t('816671190796023.957', '756549967023185081275882729414.639883423689748974601796203858786', '756550000000000000000000000000', 6, 6);
  t('3076171322682840.90303474815851099552968479985931936', '2787757192.94', '3076174110440033.8430347481585109955296847998593', 47, 6);
  t('2848944214201036899739.56442582032264708210474810029831860', '26670925074193.56548167869110309938557', '2848944240871961973933.1299074990137501814903181002983186', 90, 5);
  t('948894436900441147561922778096850998.025997593986', '443727475506008.5697805641108323511255902936', '948894436900441147000000000000000000', 18, 3);
  t('399834116109759859225128.86903386714268063275622585130318202065', '83243301579.66435004343194655785121635608927271', '399834116109843102526708.53338391057462719060744220739245473065', 63, 1);
  t('858875986652204266383404217992.291603983048765141', '6.0926465541742680273373988448676561', '858875986652204266383404217998.3842505372230331683373988448676561', 77, 1);
  t('776263916931670286731399.503267531195', '210175752246089961220452910504105327677.1086634983558182634326666828', '210175752246090737484369842174392059076.6119310295508182634326666828', 82, 5);
  t('8718333630210.37077854', '37817829567644957230084174.76767663', '37817829567653675600000000', 18, 5);
  t('175798484157966149637348754749.631917825059', '191069.39198', '175798484157966149637348945819.023897825059', 68, 0);
  t('71094655118453771005757530841.0396931519906710695577045349011371640774', '940850879022443685735297736247770183.1062859472086874437627996457041', '940850950117098804189068742005301024.1459790991994', 49, 4);
  t('55771.4302072966318037483204', '5074951905603164297.9', '5074951905603220069.3302072966318037483204', 43, 1);
  t('97573612523925803.471300446034721965973356', '884547563142872953757144.226968283178396', '884547660716485477682947.699', 27, 0);
  t('534430489256092354910779169321445983.67571072478274', '8134878658.8874328', '534430489256092354910779177456324642.563143524782', 48, 3);
  t('21752977621.53545695744', '9.27129494782115597264112710202', '21752977630.80675190526115597264112710202', 87, 2);
  t('9932777897276065862443227653669365455342.97666674233819577', '9658865807142952156279.44105819698310', '9932777897276065872102093460812317611000', 37, 1);
  t('123376130230056627372753581526479.897520335045821', '210087514417423605269213204478329905.5000222444600122718', '210210890547653661896585958059856385.3975425795058332718', 65, 2);
  t('69.90142090849675528097876785026473472', '79834.1060979095860549730563533311594891', '79904.00751881808281025403512118142422382', 100, 3);
  t('55295146117469331485301.39276631674751989526', '7152150280497482.0881738719503196277', '55295153269619611982783.48094018869783952296', 100, 0);
  t('2936.4729089310337543432927171', '592041503613529215374762022103.93452080904168592', '592041503613529215374762025040.407', 33, 5);
  t('3543847956755001.889', '35208784086977202.003447129500893889382686251144250', '38752632043732203.9', 18, 0);
  t('210.12201651568251832101964897378268382', '34.753746', '244.87576251568251832101964897378', 32, 4);
  t('88716032.541028216', '7257777720664327957948893098939678.91443795578792936', '7257777720664327950000000000000000', 18, 3);
  t('44129675251569160650555860751984.44296857695163101', '567695.63279516908886789079257067', '44129675251569160650555861319700', 30, 6);
  t('83283897197333851642528226761.317486565324', '1549.2573908490966060607', '83283897197333851642528228310.5748774144206060607', 73, 2);
  t('6996221992337457337.85327025', '667897527431759884729458692764632.058245604034043675214', '667897527431766880951451030221969.911515854034043675214', 100, 3);
  t('922619961162.2208', '47542640152456.2642306344645126128579941', '48465260113618.4850306344645126128579941', 98, 5);
  t('611397868296.756734797071956883451847', '196585892481617652694421837845193747022.4082193499020', '196585892481617652694421800000000000000', 25, 4);
  t('877868949660161908540.37474593654374', '9263.734978454268710907848665', '877868949660161917804.109724390812450907848665', 79, 4);
  t('668028.0', '80293619463333150.662604016585067202571976571183668062', '80293619464001178.662604016', 26, 3);
  t('742255892921.08552278588490', '92947236971652934320.381863823954476980190144705805612087685', '92947237713908827241.46738660983937698019014470580561', 52, 1);
  t('588640312892042.5229623908713504345901589967711144', '238642085600739223534691983689116795270.9392476910711070478806933690607710023', '238642085600739223534692572329429687313.4622100819424574824708523658318854023', 84, 5);
  t('693483.30191418214', '142589882557.20048662638259170691', '142590576040.50240080852259170691', 57, 1);
  t('9117137093.74844262790935196293617898088598527612', '73795973679442547725112.33063226581350299943451760384', '73795973679500000000000', 12, 6);
  t('4.526207787', '494677.2546106854088', '494681.7808184724088', 48, 1);
  t('30748018.2039098027620849591479954965780808709', '6852918.752696887318295520093985973', '37600936.9566066900803804792419814695780808709', 45, 1);
  t('83757651224762459585278205.84804608223565851299630130742301927', '7.64047420843', '83757651224762459585278213.488520290665658', 41, 3);
  t('975947153367486.28236437261704896554620492', '43249595796844818325786438957.99323092408', '43249595796845794272939806444.27559529669704896554620492', 68, 5);
  t('11.441690234434150722494554227811313', '875817973566587174821389748.1222354707915361953423889', '875817973566590000000000000', 14, 0);
  t('16958.2140542', '788926587.3973035876122169863973821', '788943545.6113577876122169863973821', 86, 0);
  t('417606670852399308849133225.281250676092118', '381.17737', '417606700000000000000000000', 7, 5);
  t('5801004654143445689815580111944080.0', '5042.588293662243775410837328482', '5801004654143445689815580111949122.588293662243775410837328482', 88, 4);
  t('9458108973104955501.19731713109763438574297', '25483717936.348437947448383180705503437809197004644', '9458108998588000000', 13, 1);
  t('599950356499300529789528173.8770713', '268880.548220513', '599950356499300529789797054.425291813', 43, 6);
  t('606420737207328260.923737', '2793960799108258019951495192.38660756661', '2793960800000000000000000000', 9, 6);
  t('537996460782034925311563841713892430891.857623392671766153564', '71846410869540749.3122050323622539301930763629', '537996460782034925311635688124761971641.1698284250340200837570763629', 95, 3);
  t('80855500804550283711094768191389.4531358771', '224613017765431377571817090000565541368.774524616422152973', '224613098620932182122100801095333732758.227660493522152973', 58, 4);
  t('27241652.017420652090726110', '46876731265932806262337193330223860.2863772593893330', '46876731265932806262337193357465512.30379791148006', 49, 5);
  t('703120.391502363183514079067', '79671.011', '782791.402502363183514079067', 53, 2);
  t('2946391497.026572366841464125208813086920', '189111065852431753063091488548.6924212587703', '189111065852431753066037880045.718993625612', 42, 0);
  t('70423417262.00132385773544766159557', '73526366752549062277219085279.1389002553495680928843377706747975638548', '73526366752549062347642502541.140224114', 38, 2);
  t('44343536119449484.887983227307430084581470194803169', '687286972712382711.817290490656982837', '731630500000000000', 7, 1);
  t('70666678103509980014.16613492442285597381', '7718081682856158711304.0858624585029', '7788748360959668691300', 20, 3);
  t('190892658814671027948020798562.66401', '3077914596519843221.11397426172', '190892658817748942544540641783.77798426172', 87, 1);
  t('857699.4648', '215172698578303.0', '215172699436002.4648', 30, 5);
  t('71367336317503469996089975313.78956860568717151126807150128475', '6004790033798702211031253178324995.49124913354575889902983817', '6004861401135019714501249268300309', 34, 6);
  t('5738.75872446155', '29858473503717618435783009538061.2367254587951510324891489061', '29858473503717618435783009500000', 27, 1);
  t('65523975946403470734536.20026566994471', '38814.767794272562047617', '65523975946403470773350.968059943', 32, 0);
  t('1009099102.7851258876510730342999502734', '81254320103901710.149545407089813758', '81254321200000000', 9, 0);
  t('0.80914', '45600517407671269885.9496812081291114319857829', '45600517407671269886.7588212081291114319857829', 46, 0);
  t('210699773499290371305723185122.758275542646367986145982574141331', '55837310555101717797702096605946323.1231534464473', '55837521254875217088073402329131445.881428989093667986145982574141331', 72, 5);
  t('46692439158758833041970355457275102904.33760089680805727178230', '5100.209160709238112205230327', '46692439158758833041970355457275108004.5467616060461694', 54, 1);
  t('6579188486857662760325.07263745375372', '736137.85702779', '6579188486857663496462.929665', 28, 4);
  t('6886266974462041003392255811.09', '157963384397039800487731671393.108839212437491153109722732638001', '164849651371501841491123920000', 26, 1);
  t('2637149.605497368130690', '337990492.8259301750025354342970385746411677689', '340627642.4314275431332254342970385746411677689', 59, 6);
  t('570544774319240136627673486124120453.421318090125', '627713154220186466312689860553.99818847011260119751', '570545402032394356814139798813981007.4195065602', 46, 6);
  t('1318.48981623281675', '53804888180681289555782776.08394916399261146860796354721', '53804888180681289555784094.57376539680936146860796354721', 65, 4);
  t('261560536522656708116178389636.96902957482448266', '43.894270959640597', '261560536522656708116178389680.86330053446507966', 58, 1);
  t('7389329930453.70597654613990586394144740106960', '7516027589853093719738857026838910413.94175665824167286', '7516027589853093000000000000000000000', 16, 1);
  t('567517025412398867248433926.8250784803633167567', '696941460339562352367494644161068447.0679405', '696941460907079377779893511409510000', 32, 0);
  t('82945697650.550467974', '1284942280086114734292766548209853298.612173192917', '1284942280086114734292766631155550949.162641166917', 65, 3);
  t('93979849901023407697509.7504500665642', '11239154317612957016099.3416669443153809735814058303182922492', '105219005000000000000000', 9, 2);
  t('644053055639524.76460493835227869', '24146983115497166.1003169554407', '24791036171136690.86492189379297869', 51, 2);
  t('654967.7798430818094341734', '977795808332853542.993', '977795808333508500', 16, 4);
  t('61824635978480492.6150785106270847', '667798149318896631450324413241.62326275146964221151041435124939752', '667798149318958456080000000000', 20, 3);
  t('14259971637546.243488016880492813', '47.75579735608706408', '14259971637593.999285372967556893', 38, 1);
  t('588735735873509776102380058677.1286952894401', '97.779232460469346660464003276', '588700000000000000000000000000', 4, 1);
  t('4957944374566137476540.9952771548077616024854644521329663489719', '266991814901251690329155686572698080.319489183525872908575867896', '266991814901256648273530252710174621.3', 37, 4);
  t('905145247499300612520097.253707113587775050584777734446', '2.67860616', '905145247499300612520099.93231327358777505', 41, 3);
  t('187589668011557947955872515425196212.068934581576274674451435281', '3714253520965421.0362', '187589668011557948000000000000000000', 19, 2);
  t('632409948559671833918283328879700466.0707', '0.0', '632410000000000000000000000000000000', 6, 2);
  t('984658744739418870.0476', '994.96727362001617', '984658744739419865.01487362001617', 49, 6);
  t('29683677807887236717503907093343.06771889170251887946265749329856798', '45075776650325326298409643816.981009335987971232267', '29728753584537562043802316737160.0487282276904901117296575', 57, 5);
  t('7558021712013738570217625988429759952877.233977937693484040846382622449065814187', '200795399044471614083706145185994.9737509075917941414185784409908246339265', '7558021912809137614689240072135905139000', 37, 6);
  t('92727445890741842949222790332510627.7224808091', '7609679843743118851669318656.448072904050281496386026279208447', '92727453500421686690000000000000000', 19, 1);
  t('92.908806076826839332', '92015624645614561808276089841.2399', '92015624645614561808280000000', 22, 4);
  t('377398015133086006.7276532802990993135', '28.905738023202267326815256101032838442', '377398015133086035.633391303501366640315256101032838442', 78, 4);
  t('621370624941874188939244.5200921561690076892575914021', '14510649911140399825678588475466309.79501016304482572013696', '14510649911761770450620462664405554.3151023192138334093945514021', 81, 5);
  t('20081300.1067371967724120952932454323', '9675429293924225905505754922506072343.970353197935010801782552732426097906028', '9675429293924225905505754922526153644.0770904', 44, 2);
  t('54659276818717923618921400660359661.2290315794669934292061764', '8321263373.178046641719253164256231', '54659276818717923618921408981623034.4070782211862465934624074', 65, 0);
  t('7495308563429194517.4950842956952536756132', '26.98098879', '7495308563400000000', 11, 1);
  t('88323438972.9', '3947737028583992592909690304161920258.9068619264090315473722446642669787780455', '3947737028583992592909690392485359231.806861926409031547372', 58, 1);
  t('91225.90076905019990035843212878517817', '2130936640071464704116655553.147183018975', '2130936640071464704116746779.04795206917490035843212878517817', 75, 5);
  t('20407744.328013599621804041238', '2232167919397693742359658567.87265821764005695380797468', '2232167919397693742380066312.20067181726186099504597468', 69, 6);
  t('957850276076148.677550288084933', '888359678658559270946031642.373', '888359678659600000000000000', 13, 2);
  t('74703809599234519251556274.9985711463724', '709853892003739943883074863956445427.0505267451606981', '709853892100000000000000000000000000', 10, 5);
  t('709178511125702540754989933967.422236522725392985745972858235163652', '428213017390007592115871937872.386068060216', '1137391528515710132870861871839.80830458294139299', 48, 5);
  t('530337022465.5659118274', '301051142868.206537562799420298004577157182501384107', '831388165333.77245', 17, 4);
  t('39403557595010122186833725730.5058035394497957071790', '18724770932943354346673517548.401096149086217', '58128328527953476533507240000', 25, 3);
  t('82428066.0483267045298279670012562565', '4331992005957183667051602911617712754498.899', '4300000000000000000000000000000000000000', 2, 6);
  t('80476913465044941102915.632723815887522711331529', '5887544380532992755.93511005801607595687540', '80000000000000000000000', 1, 3);
  t('76964133350127894939591167080295347.29740816733871', '772928008263955477554638996.3016794002524482346051388435140772', '76964134123000000000000000000000000', 12, 1);
  t('93613415080964701839073763939463.7495190652950339', '2046269205652007290620382.1448517814887759303162856189242851406961', '93613417127233907491081054559800', 30, 3);
  t('93178915455.978234064862278834', '70156785202107872073374655857728.50328804248218', '70156785202107872073467834773184.481522107344458834', 100, 1);
  t('5930.48411006091572345293786062592928', '89716089387944665472.79179673561851851', '89716089387944671403.27590679653424196293786062592928', 75, 2);
  t('9.09', '81260831717416.97800363', '81260831717426.06800363', 89, 1);
  t('5.86180704663201447', '502623012117081747364976254577298342892.5693576219320095015755173160', '502623012117081747400000000000000000000', 19, 5);
  t('472623181838332890363488612910764474071.586394798193158796943819', '8947794.638137404', '472623181838332890363488612910773421866.2245322021931', 52, 3);
  t('2918131158533557596.17923195829208', '736594397048603541.13479231839973012229918592', '3654725555582161138', 19, 2);
  t('82175442362893730960724401268.902099552468164410386448', '539613524689316.8064962190352269180519578275596107467', '82175442362894270574249090585.7085957715033913284384058275596107467', 67, 6);
  t('78844330615540128067328106904.417352885735996754512588977537835', '69839051335233721003254224037166.023137852571150860272482848011573', '69917895665849261131321552144070.440490738307147614785071825549408', 93, 5);
  t('83665060234963974331243072338458013.41317384589891075', '36334.350471648383518695', '83665060234963974331243072338494347.763645494282429445', 57, 5);
  t('16281170009138261525916508238.0992', '5448415536509973562150706380848655442293.0', '5448415536526254732159844642374571950531.0992', 67, 0);
  t('1855445938689865341002697566158764326.79725393455165540', '565.09791800109811921652723361425597442', '1855445938689865341002697566158764891.89517193564977461652723361425597442', 97, 1);
  t('59739822482533882530.588103741051908781', '1074461088804628208978814.625', '1074520828627110742861340', 24, 3);
  t('343073566004818040807197646892403723450.116757581689904546354687420', '81463061354543518130225835483953609.7480531965302510897362653301336597', '343155029066172584330000000000000000000', 20, 0);
  t('7029.86663560321963172780', '998093479032.085', '998093486061.9516356032196317278', 97, 3);
  t('6575500176279641875577.1641', '7042204323.867599405615408232', '6575500176286680000000', 15, 3);
  t('563573389370505100921783994.63820711261296', '6183290606357615817.2201', '563573395553795707279399811.85830711261296', 93, 1);
  t('0.11775276416997354994358051', '9390124793042023844.11441973650635879154582117867', '9390124793042023844.23217250067633234148940168867', 98, 5);
  t('34682163909803912305.58279868924', '3882592798.71', '34682163913686505104.29279868924', 32, 1);
  t('45556245389463100131676399335223730722.64187559429639385721653', '3139098333707395868305892593215872544698.3980421276727128', '3184654579096858968437568992551096275421.039917', 46, 1);
  t('42868.0', '22853181.7', '22896049.7', 15, 5);
  t('13324349843364165078434315416.4219558920586598551', '634910046173465606044.94350342', '13324350478274212000000000000', 17, 2);
  t('214681567801880599741149018665.4268904627994045437530044796095081166697', '30.24438352', '214681567801880599741149018695.671273982799404543753004', 54, 1);
  t('7523592953864446224655147634517806935677.8', '321960438573180275859530550631223297.2220693619034050159922710483945976', '7523914914303019404931007165068438158975.02206936190340501599227', 63, 4);
  t('8113872603185679969728486212332590783.602055287175599032912451', '6883368621751387.904574695331503', '8113872603185679969735369580954342000', 34, 1);
  t('4473325482.521859322286620356175998427324', '5517383386767642457749850088998162.9878404839737358471115523370', '5517383386767642457749000000000000', 22, 1);
  t('548897584678465172116.263299940', '72849524332345281737.582385686658613419466441920115889', '621747109010810453853.84568562665861341946644192012', 50, 4);
  t('117624754.6743723197932701087121335', '432230085746050.6326013435', '432230203370805.3069736633', 25, 2);
  t('211869120158.6755510559122396323632849492172', '3350176216485535906730909111199110098559.434382853205751669259307', '3350176216485535906730909111410979218718.1099339091179913016225919492172', 84, 1);
  t('9503417727903696377.142688585685289539282', '9877995505774365097359676968.96332983381', '9877995515277782825263373346.106018419495289539282', 73, 3);
  t('57724427547745158119515495.3817737615142', '4694471183681522510434.9730842784869109298930451804195715', '57729122018928839642025930.354858040001110929893045180419571', 59, 3);
  t('626926.688418991670094873504536127927874926961', '44847638554.28038', '44848265480.968798991670094874', 29, 5);
  t('714421546608288.75231442', '90204.1877149702887861501577366', '714421546698492.9400294', 22, 2);
  t('316760561115100328925527829.868748', '947550435614164286486134205461396980094.329237531119278646527505237738228861', '947550435614481047047249305790322507924.1979855311', 49, 4);
  t('4787781.830769633650', '28106931688076101944325980357112766666.4147856541674471029390', '28100000000000000000000000000000000000', 4, 1);
  t('57582322933702115191953750823.452324932759532448958683849760832537', '6208160069451660574.046992350532154118230361572466326413231', '57582322939910275261405411397.499317283291686567189045422227158950231', 88, 6);
  t('1355.2928651389134946509903601457219696', '19706467852518454.51194438', '19706467852500000', 12, 3);
  t('6976348867939077662220126242.325581450', '37632000780254976.39244499406494501679005669774384375', '6976348867976709663000381218.71802644406494501679005669774384375', 78, 5);
  t('6570085371399.73547644347999335127', '756784655571328405212254444.6082917039022814', '756784655571334975297625844.34376814738227475127', 95, 3);
  t('929638604406893166255480319.1', '2.417176', '929638604406893166255480321.517176', 38, 4);
  t('15333436830100284767892.8132821094757068516782384757893828', '6449148471872656116642573.4936072838460652424614060030402960899', '6464481910000000000000000', 9, 6);
  t('8533454661767.0', '9108117482.7818582174602901110', '8542562779249.781858217460290111', 98, 4);
  t('39994199552944997944674603137102.34481054478642979579556733', '48416362753816384.1186859947', '39994199552945046361037356953486.46349653948642979579556733', 58, 0);
  t('3434813658037057456771225.691575393610', '4750727681378762.621422066', '3434813662787785138149988.31299745961', 50, 2);
  t('5421870551709971643477646229788.9916151146275027436845830667747596930491', '1294187723.63552000599434', '5421870551709971643478940417512.6271351206218427436845830667747596930491', 84, 1);
  t('788800325137015309691.9869658106091643522880904508333', '21900937194317004352606581867113.74136675', '21900937195105804677743597000000', 26, 4);
  t('9617449543131441414744.740488254808748568321093692000048461', '235857647890636847089343613221.34934036368820', '235857657508086390220785027966.089828618496948568321093692000048461', 89, 5);
  t('66516175033671157522539424787063035.289018415345287705892', '457025403297589.48109451223206526106813370983486534540', '66516175030000000000000000000000000', 10, 1);
  t('1258923932952359915757977904.322911996152689501640222239052', '516664166.6315559709401451437542901201056045029', '1258923932952359916274642070.9544679670928346453945123591576045029', 95, 5);
  t('54150784958055693201041677324555.2911379033655054172110464032863412', '33037219866829.6261839599256770108242884962194958', '54150784958055693234078897191384.917321863291182428035334899505837', 81, 6);
  t('8913941949356907370902846091451.1977723298', '18118.359607978038552071087439', '8913941949356907380000000000000', 18, 0);
  t('420410527375765723.59880359173181300286368456', '51144376163959512362961732882333687.127621078003128120894', '51144376163959512783372260258099410.726', 38, 1);
  t('2349860389256052429839524907.98078795118546317910978141', '79456576116664247.9125601988131884602', '2349860389335509005956189155.89334814999865163930978141', 83, 6);
  t('17267416260184245345149414069.320310187240', '468091424192506118640825196.9226420172656985659417202247668848', '17735507684376751463790239266.2429522045056985659417202247668848', 88, 5);
  t('43036191260946830561733461288991373.899222463453441', '10395502788.8976', '43036191260946830561733471684494162.796822463453441', 57, 0);
  t('4909087864.1506006674710514486016876552', '99.292952416278708', '4909087963.4435530837497594486016876552', 41, 2);
  t('3.21003346071', '312572971640834499.5651635013104403556411852', '312572971640834502.7751969620204403556411852', 65, 1);
  t('35598083270443235311505086492314.01162518855759640920', '802559462829546799507.20438317038165345760210289811993', '35598083300000000000000000000000', 9, 4);
  t('4920535946382178542803785670.9541584521', '170.20', '4000000000000000000000000000', 1, 3);
  t('900521977018339396017913407962.091347787805459639979561619514892821', '86767128498.266492255048676', '900521977018339396104690000000', 23, 0);
  t('75902636101686648000.6', '359043382984760880361621433096498558.8522232641204', '359043300000000000000000000000000000', 7, 1);
  t('7241120.33794391441049422', '1335.1641429496763007692077381282', '7242455.5020868640867949892077382', 32, 0);
  t('31660394971452812.41546802595892146417789641819', '9553266936159140830011359638454773.3381424723086951716056', '9553266936159140861671754609907585.75361049826761663578349641819', 69, 0);
  t('234366.419036191593359028721642040021', '60990608004.6502084059964088971099341864756873167255', '60990842371.0692445975897679258315762264966873167255', 63, 3);
  t('595275472.57284296', '585213018909659275240537.83842', '585213018909659870516010.41126296', 77, 2);
  t('686195920001876075.42907969250854530243258040339799332818', '690209177.0029006268774080543176338133518164481718', '686195920692085252.431980319385953', 33, 6);
  t('7277582821973178066168.46693414550857472667581295031419181226', '977228.354318', '7277582821973179043396.8212521455', 33, 3);
  t('16292530.1051086167480590244141', '842.22481391817784261791644440600423', '16293372.32992253492590164233054440600423', 70, 0);
  t('446843063108997245517407765134496278.709613990646498002821658685419236096', '88.30183490297', '446843063108997245517407765134496367.01144889', 44, 3);
  t('971454488708956128182425493108.4703496410268788074591551814', '1500927701243584.018518512711227442104023548', '971454488708957629110126736692.4889', 34, 2);
  t('198396844046545593344953371792608987.3415868552607561', '918181535.23275182900', '198396844046545600000000000000000000', 17, 0);
  t('2972391991248.1400667620189362208168307', '5479621704745657230709696892535281281754.7235', '5479621704745657230709696895507673273002.863566762', 49, 5);
  t('167427692328032674.25414709703925071305926860225042563182', '89766918952455057053287822756397622107.3928497608193855537950881184312381', '89766918952455057053455250448725654781.64699685786', 49, 0);
  t('43769831251746560.795026217654974915196027543695', '75537438951336483269081.02941273170981073861080518572', '75537482720000000000000', 10, 5);
  t('3.790', '9378194.8767000998322295569523865', '9378198.66670009983222956', 24, 0);
  t('24851382261264597346888356037041.885', '1207.8573780576386389443173321131', '24851382261264597346888356038249.7423780576386389443173321131', 81, 0);
  t('857.7560861034887549930443754', '18504376084922251496220282291018.19697204676', '18504376000000000000000000000000', 8, 5);
  t('5603522276434.666278645458189168052', '9478916073657380583651187460768352706264.0922039031654960785', '9478916073657380583651187466371874982698.7584825487', 50, 0);
  t('272.20163189736699207276', '8009976948562717694664282.361198237345827', '8009976948562717694664554.56283013471281907276', 93, 1);
  t('75286597915548865172794905512308251.1787842766', '5484959.61054', '75286597915548865172794905600000000', 27, 2);
  t('93525713224070.09674242841815995541364018052052', '75768676.532480641692742503493', '93525788992746.6', 15, 1);
  t('78523021357694108868692752.91646650044759904173480959883557402997', '495.59', '78523021357700000000000000', 13, 0);
  t('3238721780678562900915193565802225587392.90', '748785061.66820334924209162463741', '3238721780678562910000000000000000000000', 18, 0);
  t('98846.81464384706741923274334684308930111', '9113.005045834', '107959.81968968106741923274335', 29, 0);
  t('9.8406692310667035312191105495866801000', '9089567678.5508097764778', '9089567688.39147900754450353121911054959', 39, 0);
  t('99065135312720.84941192157958140619229438117', '53476921863610672798515553218338324.959329170020735932026093766186541396', '53476921863610672798614618353600000', 30, 3);
  t('3176881371318006310361.3543736148140570707252111', '292524070460132222826177704057904019376.40698273353941530130447408152708766116', '292524070460132226003059075375910329737.761357', 45, 2);
  t('6.90498764441988249128502282530851578', '43.50645242771270044511', '50.41144007213258293639502282530851578', 45, 6);
  t('338450.823402603747752671009120897851', '5207.0099214953', '343657.833324099047752671009120897851', 59, 6);
  t('2919769949581629422304417.9039495411631604778178068863', '58958088180274381782196182656778.3384598201347480560152781708323013', '58958091100044331363825604961196.2424093612979085338330850571323013', 81, 2);
  t('858925411393538961692691.6390687813', '429652895153162504998871559279.1146093468874157436552', '429653754078573898537833251970.7536781281874157436552', 52, 5);
  t('1744056.272665115358269071', '6706.43717222508571881815348574554893', '1750762.70983734044398788915348574554893', 39, 1);

  Decimal.toExpNeg = Decimal.toExpPos = 0;

  t('4.6011800481717419E-806104423910', '-8.6437098622551564549067098493197906561E-280114', '-8.64370986225515645490670984931979065609999999999999e-280114', 51, 1);
  t('3.4681913622863819250677595389410956725648165265014983E-53212647902', '-8.18410385421852186881398297236729212185E+188', '-8.18410385421852186881398297236729213e+188', 36, 3);
  t('6.66144513902114901386407229147E+6153706610103500', '9.0511259314724948248692894888182029430402308963553381379066E+953113360042358', '6.66144513902114901386407229147e+6153706610103500', 74, 4);
  t('3.372650568051595258210907487596E+409215479667897', '-4.183206656034048517819669530602649203970090968725509970E+26570793338', '3.372650568051595258210907487596e+409215479667897', 74, 4);
  t('-5.7656240966847308176727533610432E+62642145507', '-1.122213E+7374576131771581', '-1.12221300001e+7374576131771581', 12, 3);
  t('8.774639235678526352460104724518622822913117021566676E-1991326', '8E+45629', '8e+45629', 34, 4);
  t('4.9590402774499089404125161028829404E+1528628', '-8.110292813306920410771458213952017595580929120805599E+676055705', '-8.110292813306920410771458213952e+676055705', 32, 5);
  t('-9.94827679900718E+47261881238', '6.191921784108606448822612520770987741963619338418111021E+2397', '-9.94827679900718e+47261881238', 79, 3);
  t('3.000279398050E+68452755140307', '-2.314E+6336521643193', '3.00027939805e+68452755140307', 25, 2);
  t('-6.39233174154524919846265317272190742174E+27843778404', '-4.9206748933094636318481451339835260640508733245821012047E+581960334', '-6.3923317415452491984626531727219e+27843778404', 32, 2);
  t('9.766364616787074202022021617942580306437E+853339', '4.75E-94400', '9.76636e+853339', 6, 6);
  t('-2.477879366633922436942620379465344075853263140147E-179', '5.001E+607036568', '5.001e+607036568', 10, 5);
  t('-9.91773038789242415841400646241871432E+509270', '8.90240027196285459406834697800823E+384', '-9.9177304e+509270', 8, 3);
  t('-6.4216117911855785555196586953491283327671659034E-4267768410', '-2.71549651651035474742851384005686907E+932623751145375', '-2.71549651651035474742851384006e+932623751145375', 30, 6);
  t('7.505003437177338901537585712517292440418147498390807673406E-41013596', '-1.6377957063913576274E-2362545', '-1.637795706391357627399999999999999999999999999999999999999999999999999999e-2362545', 73, 2);
  t('6.52497142975743790E+59803', '7.70070919859302927342891E+79853350467950', '7.70070919859303e+79853350467950', 15, 5);
  t('-1.986586534614591954662873350021191E-8494702330585', '934883196.36', '9.3488319636e+8', 31, 4);
  t('-9.34393830187381530019933058983221630473183E+1322081', '-5.8113420569E+76250834', '-5.8113420569000000000000000000000000000000000001e+76250834', 47, 3);
  t('2.8472949996959453188308291822038006419E+58488338835694', '-8.40525201109E+450', '2.8472949996959453188308291823e+58488338835694', 29, 2);
  t('9.648367041406758034023991036E-59498', '-4661212.7762884155284840804331492224870144396865168', '-4.6612127763e+6', 11, 0);
  t('6.590E+563652649062720', '-2.470866644580659989139348934871662E+5393', '6.59e+563652649062720', 52, 4);
  t('7.50298455E+409885709899897', '7.8195903135920240406104586171019944177061863195512E+38137135', '7.50298455e+409885709899897', 21, 3);
  t('-2.085709164207E+556', '-8.65687769072345279667007546405481189141239951957731228E-445', '-2.085709164207e+556', 24, 6);
  t('-7.59901046473456174157176600977643736E+928', '3.1554648E+9567428', '3.1554647999999999999999999999999999999999999999999999999999999999999e+9567428', 68, 1);
  t('-1.764893485808065014931071720288894996414317946E+4373', '8.207697685678414861699991146707E-54431221200', '-1.764893485808065014931071720288894996414317946e+4373', 69, 0);
  t('1E+3438', '6.10225432462566325701089103E+413299377797', '6.1022543246256632570108910300000000000000000000000000000000000000000000000000001e+413299377797', 80, 0);
  t('-2.52752725708781022951571784E-699814829', '8.17875673731E+550628791', '8.17875673731e+550628791', 41, 5);
  t('3.4142120654239282339061777091E-137153793', '-1.920473198839259830870283143608138835313005551161E-5700', '-1.920473198839e-5700', 13, 2);
  t('4.916E+67267431864055', '-3.11232626041972E-22119634', '4.916e+67267431864055', 43, 0);
  t('2.7805554685841989593647311623180945E+631402138248535', '-3.5045033749573135402009952508161E+422962095', '2.7805554685841989594e+631402138248535', 20, 4);
  t('-6.7851124456156495E-283663', '-1.9748544E+991285', '-1.974854400000000000000000000000000000000000000000000000000001e+991285', 61, 3);
  t('3.3999495398879477127431E+26', '6.94787088935990247137662674286E-30255690293', '3.3999495398879477127431e+26', 39, 6);
  t('7.38629782790796531025130356311386902379257377504731E-65498', '-9.996570772195629416377330436034192E+586560433', '-9.99657077219562941637733043603419199e+586560433', 36, 1);
  t('-65427.2626599177561639832012637131774577343', '-2.2602461224542757053386510475184E+45', '-2.2602461224542757053387e+45', 23, 6);
  t('-7603337863100331037227641358.6358488010354381615972252', '0.079095503', '-7.60333786310033103722764135855675329804e+27', 39, 6);
  t('-8.566805532967771E+1133407170134', '-2.62014507E+59410816402904', '-2.6e+59410816402904', 2, 2);
  t('-6.378824969667553954007858399705411E+6125807', '1.5498189328683385253997376566304691018861347216E+978678273514', '1.5498189328683385254e+978678273514', 21, 4);
  t('-2.7745277583174075108879767960293E+57', '-1.04898713859989870135460729303852467980207715713082811008E-52522188', '-2.7745277583174075108879767960293e+57', 43, 4);
  t('5.6416413863730610446514344935740516542544173047838529477345E+3643189785224903', '4.325844E+10237387', '5.6416413863730610446514344935740516543e+3643189785224903', 38, 0);
  t('9.5700358273E+3281436620', '-3.49529922278383002447248797548714279747957316073562965362E+231765765674926', '-3.4952992227838300244724879754872e+231765765674926', 32, 0);
  t('5.746256598050270E+270179101110', '-8.69E+9', '5.74625659805027e+270179101110', 52, 0);
  t('-3.43703754975841976988985601863218657090565779695942448E-584', '-7.04870954464451748226487792925E+6452', '-7.04870954464451748226487792925e+6452', 65, 1);
  t('3.669191803775848342191491816028514958908276103E+6104885', '6349200469.49622348667724526141571168', '3.669191803775848342191491816028514958908276103e+6104885', 57, 4);
  t('-6.6413415085265200380682995061672743168E+6742608268982', '-5.7854761668979413E+40390639739697', '-5.7854761668979413e+40390639739697', 74, 5);
  t('8.131061529004687961E+4703014322894957', '6.6631960163605E+336774893', '8.1310615290046879610000000000000000000000000000000001e+4703014322894957', 53, 0);
  t('2.61586047680708591504734E-73618452078', '-2.82E+649040159313928', '-2.81999999999999999999999999999999999999999999999999999999999999e+649040159313928', 63, 2);
  t('3.806193950290037447482606404806098753E-310306977712459', '6.73574380489237E-6148', '6.73574380489237e-6148', 61, 4);
  t('-2.999244594782511304449990054060750394689E+665698507', '-7.86539E+624', '-2.99924459478251e+665698507', 15, 5);
  t('-1.73138291075097215041056825045207084790023E+9919178', '6.901091644374740436286975650710270008065430429870778E+112688678', '6.901091644374740436286975650710270008065430429871e+112688678', 49, 4);
  t('-4.4158131733719201035971650235401451699E+8478781', '3.43659130529850845160044047E-84414725879', '-4.415813173371920103597165023540145169899999999999e+8478781', 49, 2);
  t('8.2490530603E+563080552827', '-7.0892036675186933319756716833232912692817124518421644588185E+503', '8.249053e+563080552827', 7, 4);
  t('4.4812397022812291341308065939147761041239042598E+173199103', '-2.55358479553032556192092676679E-406', '4.4812397e+173199103', 9, 3);
  t('3.63707497063891323415475164E+19564220', '6.86E+394624254369', '6.860000000000000000001e+394624254369', 22, 2);
  t('-1.76521690385342100091110955772116032382348E+22342', '2.066371211836190804699093814076260613148846118E+930', '-1.76521690385342100091110955772116032382348e+22342', 46, 3);
  t('-9.16540861788939936221696041264140509892899684806066624E+545382', '-6.795036513456510225425505285065601005815059E+53651315619590', '-6.795036513456510225425505285065601005815059e+53651315619590', 52, 6);
  t('3.8824853845483747950121911961308E-232889426110', '-9.39085569262226048802461108556961372903363E+20482563', '-9.3908556926222604880246110856e+20482563', 29, 3);
  t('5.669348078931699485812E+91135', '-1.09634534582515905312199364521354019E+2398', '5.669348078931699485812e+91135', 43, 6);
  t('-7.108420703596696174864577599199E-17865', '-3.5821234495870040888988341424E-38476', '-7.108420703596696174864577599199e-17865', 68, 2);
  t('3.0587937455435385568686E+858411', '7.5633990990140136053230800443528E+6979747049', '7.563399099015e+6979747049', 13, 2);
  t('-9.4201452050384019635554408654254853829585833173937E-67881175400', '8.7594589610901982328939776149140582313134280', '8.759458961090198232893977614914058231313428e+0', 51, 5);
  t('-4.872E+9', '7.049434162615329092286249077949314195166656228776654449E+7179602312410905', '7.049434162615329092286249077949314195166656228776654449e+7179602312410905', 57, 2);
  t('2.498050094918515195272495E+25602819644', '1.690189280483229506407151914477E-6281659114721093', '2.498050094918515195272495e+25602819644', 37, 4);
  t('-2.872594834762158998189624857E-3872766009', '7.84730498600204522802E+1918690570', '7.8e+1918690570', 2, 3);
  t('-8.5989284208919716747311350415768937298660825849E+8362030996', '7.246737029123850373402361691951291E-8667053', '-8.598928420891971674731e+8362030996', 22, 2);
  t('8.9119677110510714267573580010087886701724745920E+21469', '2.11712768007778858425E-2709430696', '8.911967711051071426757358001008788670172474592e+21469', 61, 3);
  t('-4.6389773729060896356268643578053980E+271990', '5.4794710777462795179129364629207044179495471E+68796013', '5.479471077746279517912936462920704417949547099999999999e+68796013', 55, 1);
  t('-8.692108349146276787672E+502575719452551', '2.101910086621574891507701870425441356167851002049453E-36058', '-8.692108349146276787672e+502575719452551', 71, 4);
  t('-4.329643128068417315145546886117425E+47', '-5.180272153613851594615192162296169326241993204242351102905E+61', '-5.18027215361389489104647284646932078171085437849235111e+61', 54, 3);
  t('-4.366295227912472100879989655235852323995260684E-3443005916835', '6.6495301021110825661911619989342504206315E+243165734316192', '6.6495301021110825661911619989342504206314999999999999999999999e+243165734316192', 62, 3);
  t('-50.3340777108096062382934887949536591632954', '-2.327861974724190621584E-95290518158703', '-5.0334077711e+1', 11, 5);
  t('2.891402E+308984', '-4.4030600568577068902242588950310064E+3309250', '-4.4030600568577068902242588950310064e+3309250', 38, 5);
  t('2.9063245181838078406E-30082705149', '-5.21164603510E+700', '-5.2116460351e+700', 47, 3);
  t('9.172623893884291050640178462622576979394001E+74551900', '-0.0006992318506880861375813973557823740630779243140040', '9.17262389388429105064017846262257697939400099999999999999999999999999999e+74551900', 72, 1);
  t('-8.45743812419841308359736070937E-7', '-3.7150162603440907497300108323452688532070E-47707102611', '-8.45743812419841308359736070937000000000000000000000000000000001e-7', 63, 3);
  t('-5.64048998369545130320593838176021852293273725718E-66', '6.4546418096890813418E+8932686', '6.4546418096890813418e+8932686', 47, 5);
  t('0.000206', '-2.46E-16691451673', '2.06e-4', 72, 4);
  t('9523206424.817811834542598837948480678179531962407', '-9.246E+122', '-9.246e+122', 61, 5);
  t('-7.10637843522872405148177939345694170244998E+393578612', '-5.1686557840917978536969896875975293786383104244203680E+2623', '-7.10637843522872405148177939345694170245e+393578612', 39, 0);
  t('-9.701743299218118469518328922466346421531319820575184691E+987052143', '-2.162146263864726687779E+279733', '-9.701743299218118469518328922466346421531319820575184691e+987052143', 76, 6);
  t('2.425681641011836402167519E-745580', '-6.975349406316865333556880973581197441792824E+77261', '-6.9753494063e+77261', 11, 4);
  t('-9.9615667808682717784639047503416E+431098', '-5.411587171905E+752176928289', '-5.4115e+752176928289', 5, 2);
  t('-7.93800341064442267449278738313266140E+6661', '-4.77520007398717199039734239605332385947778750068587220E+772955635746', '-4.7752000739871719903973423960533238594777875006858722e+772955635746', 67, 5);
  t('-4.3950852214287588188735202190E+667593915681', '6.903412426009159788320109541849891301209580355174E+192215971290', '-4.395085221428758818873520219e+667593915681', 44, 3);
  t('-8.52768540649733978324430E+689', '6.5091092511E+6976032213402', '6.5091092511e+6976032213402', 47, 2);
  t('-551.540567153246944', '8.86155528029344977E+1979683', '8.86155528029344977e+1979683', 40, 6);
  t('-2.717789E+3942366982137037', '-2.4634911223721540585053403E+461', '-2.717789e+3942366982137037', 53, 5);
  t('6.166E-6244253', '-9.3214367939176849033412886654725530336291E+4194', '-9.3214367939e+4194', 11, 1);
  t('5.20725596945602124349E-6628650332', '8.3926653673662259021000369E+619', '8.39266536736622590210003690000000000000000000001e+619', 48, 2);
  t('-5.47534147329743050378910264292307917745886652072040802E+4736849', '-6.36656218431E+903', '-5.47534147329743e+4736849', 16, 2);
  t('-9.2797481508655335153340193408747852559E-762', '2.521199983786748708733455394509E-4192408404897193', '-9.2797481508655335153340193409e-762', 29, 0);
  t('3.861229821495293537193961672144521609397610358852924709084E+69897997373384', '-4.187275007792772117094721014960E-4794184599539', '3.861229821495293537193961672144521609397610358852924709084e+69897997373384', 58, 4);
  t('2.411723E+4353339572', '-7.68531916850E+98133448828193', '-7.6853191685e+98133448828193', 63, 0);
  t('0.188933017020775851110962818646204457535104092933546166', '-3.19557555791310599673359150099731090768E+6558', '-3.1955755579131059967335915009974e+6558', 32, 3);
  t('-3.77704574461102506821116718590213463452971444226E+24181884', '7.89843889931508016397563E+54501333520', '7.89e+54501333520', 3, 1);
  t('-1.694348343970E-19062624043964', '4.74085362358688899617185491288841032345E+1818992', '4.74085362358689e+1818992', 15, 6);
  t('7.827548798871167738828135403355260658546930138086182E+88209119442', '-6.873098406387768E+8107575038668', '-6.873098406387768e+8107575038668', 45, 6);
  t('-7.793847268035948651213204898194065634671277463640679113E-71021', '-9.25192017503693470678244673283726041560455402088E+468797185420', '-9.251920175037e+468797185420', 13, 0);
  t('1.5324356680755304707388397601754351260260E+51800671', '7.750256E+2067810458386606', '7.750256e+2067810458386606', 68, 3);
  t('-4284207863.48162898771759870845159620427053872002953436', '2.678461563685438181787E+2175', '2.678461563685438181787e+2175', 41, 2);
  t('3.509925697559788603605165685667402961597749778E+579972621', '9.49354694944266950726785182650136718343516254583305492E+615166919', '9.49354694944266950726785182651e+615166919', 30, 2);
  t('4.997254278769027827125462337E+116287155053', '-8.3420550960652692014E+73857', '4.99725427877e+116287155053', 12, 0);
  t('-1.5471238E+12818502839', '-2.51624E+6670', '-1.54712380000000000000000000000000000000000000001e+12818502839', 48, 3);
  t('4.03845806575613111013609373029E+6641', '2.50582882960752577743961966535833E+34733022', '2.5058289e+34733022', 8, 0);
  t('-6.27404897554E+512651292', '3.850202960342140538175124315512092E-2468457050130455', '-6.27404897554e+512651292', 37, 4);
  t('3.73543241298342843814894899458386501E+2691695896154', '-9.724815059386122233255407618425E+4203582', '3.7354324129835e+2691695896154', 14, 0);
  t('-8.8231459456021072857818669866309764613E+930262643401808', '-5.249691790665394148103E-74240164', '-8.8231459456021072857818669866309764613e+930262643401808', 73, 4);
  t('-2.15335223663222506124798820406557843215348E+4202729672452141', '8.45980008962682894307E+774202820651', '-2.1534e+4202729672452141', 5, 0);
  t('-1.7799886819301366984596688945034565698432742036E-947941197049654', '2.213372627032946638192686843830E+240141770080', '2.21337262703294663819268684383e+240141770080', 40, 2);
  t('-9.4144E+2477', '5.13E-1594851437729515', '-9.4144e+2477', 28, 6);
  t('-6.84E+94', '2.4673099077000465752320309838269659987987170400370E-6504', '-6.84e+94', 52, 3);
  t('-5.071806225212E+5417270823657', '-9.624503E+2818', '-5.07180622521200000000000000000000000000001e+5417270823657', 42, 0);
  t('-1334623402060722808866044690398799614351953.37498706276', '-9.2877551960513631E-16062', '-1.334623402061e+42', 13, 0);
  t('-2.6442090E-332', '5.661168834684021053299709525328255302698212731427678781E+171776', '5.6611688346840210532997e+171776', 23, 1);
  t('-1.78E-383610148117221', '-8.4613992413390360803256099173E+37044824', '-8.4613992413390360803256099173e+37044824', 61, 1);
  t('-0.0000062017052170970729498384391456766778884529974040624', '-5.518773320738164882E+85135165', '-5.518773320738164882e+85135165', 65, 6);
  t('1.93714405569220407313824065271476230676E+467591435511', '-5.7783767686955108423425328373372848769672002449421056718363E-7519814813', '1.937144055692204073138240652714762306759999999e+467591435511', 46, 3);
  t('4.4360034512789350E+155348', '2.25210627639241929951317896718E+9412244', '2.25210627639241929951317896718e+9412244', 76, 4);
  t('-6.481935214532330996490E-46', '-3.025123871773769737508727467472456142864922733668871E+2854517544177134', '-3.025123871773769737508727467472456142864922733668871000000000001e+2854517544177134', 64, 3);
  t('1.494273480155723127276787860353929741454585651339923384424E+7030', '3.599955E+6368385830772', '3.599955e+6368385830772', 63, 1);
  t('-3.28608292746150680004217754E+688438', '7.3551633873882835599850217061608240606832358782E-74572376626450', '-3.28608292746150680004217754e+688438', 50, 6);
  t('-1.03266796446E-8278', '-4.0460202659250511215483363070320924549117537263881023827559E+8774208', '-4.04602026593e+8774208', 12, 6);
  t('2.48389743898982686504586E+5990118820', '7.105424023427069964143E+50201321230', '7.10542402342706996e+50201321230', 18, 1);
  t('-3.6762713182092660572370854E-293263731', '2E+81311851770632', '2e+81311851770632', 45, 5);
  t('-7.922246796162E+560934167437503', '-795.111719224266738776273297740051263152270177611848', '-7.922246796162e+560934167437503', 20, 4);
  t('-7.84986371289748940384675258290319452577143E+784559037253', '4.8553700563277E+499797945', '-7.84986371289748940384675258290319452577143e+784559037253', 62, 5);
  t('-5.01451226729986E+634128641536684', '5.1772935756137731440883468553701654868056893184113342E+12620', '-5.01451226729986e+634128641536684', 73, 5);
  t('4.8148097678679009821027564058818895196777521129850224405E+68964', '9.681157078623298477505420042500552450626592789128905E+207', '4.81480976786790098210275640588188951967775212e+68964', 45, 0);
  t('2.7266500623596292085843253101131091745735239E+7581005', '-5.1142051E+379807216817167', '-5.1142051e+379807216817167', 14, 6);
  t('9.345E+4050716137', '8.25864200040756587945263712166329463926916E+989168197213499', '8.2587e+989168197213499', 5, 0);
  t('-4.4958852106290149081075323202084592539E-86', '-5.01557417551668E+7208357370', '-5.01557e+7208357370', 6, 1);
  t('-1.942702E+87956369', '-6.0893584767832E-464800', '-1.942702e+87956369', 41, 2);
  t('-1.20679614544803855898645801212207601717304402844E+3013791221389', '-8.2076647453271044803940573308E+2853907916091', '-1.20679614544803855898646e+3013791221389', 24, 5);
  t('-2.851894798868666358828959513926E+98768599339296', '-3.990652968467072965998006858161830578E+683573721', '-2.8518947988686663588289595139261e+98768599339296', 32, 3);
  t('6.25935603116583092362404788361107180421574164239073421E-9473', '-9.577742473E+686', '-9.57774247299999999999999999999999999999999999999999999999999999999999999e+686', 72, 2);
  t('-0.0058890488', '-2.8E-962', '-5.89e-3', 4, 3);
  t('5.983877037E+772055650933', '5.97790820447058150727307175945652943256022347739E+7594931381', '5.983877037e+772055650933', 52, 1);
  t('-3.49906948286962159619846775300283489060141E-4413058701961', '-5.78971523E+368830178139', '-5.78971523e+368830178139', 34, 5);
  t('7.99749151865001217378490121516E+9002510549', '2.916E+632970249941742', '2.916e+632970249941742', 32, 6);
  t('6.330E+94716', '-2.547152E+802', '6.33e+94716', 11, 5);
  t('7.6823442662814070799812306776069858647829286E-3799184', '-7.769E+52', '-7.769e+52', 17, 5);
  t('-9.12907988042607711581374160076730389488382781685292E+741508832', '2.268473599477838877813920073444162787769215604E-560719592104759', '-9.12907988042607711581374160076730389488382781685292e+741508832', 67, 3);
  t('7.7870558162217243698180022032041422347135479211630E+866', '-7.7020027224428659E-145491', '7.78706e+866', 6, 5);
  t('6.80478260690757549309246441979E+1869908145', '2.177206989317966240E+7522960', '6.80478260690757549309246442e+1869908145', 27, 2);
  t('5.07211949960267751398063785947343590299568602130E+720174', '-2.08432572E+83019', '5.07211949960267751398063786e+720174', 27, 0);
  t('4.89816458567686737756451385267113E+12101101', '-9.90080E+857552754863', '-9.900799999999999999999999e+857552754863', 25, 1);
  t('-2.48566946979385137E-538', '-7.217064208353626291334100467692468361768858565308282E+415524765', '-7.2170642083536262913341004677e+415524765', 30, 0);
  t('5.048E-551635094', '-1.439E+8813632164337913', '-1.439e+8813632164337913', 59, 5);
  t('-4.3E+752212513104', '53992722691780303486849512407.39246980952144', '-4.3e+752212513104', 30, 0);
  t('-7.142585472595838E+92892003796883', '4.74357300426553962404753275927780859812305367191190071314120E+4633239443606751', '4.7435730042655396240475327592778085981230536719119007131412e+4633239443606751', 67, 0);
  t('9.56250E+282', '-1.976191E+524639328111177', '-1.976191e+524639328111177', 80, 4);
  t('58097.0062256871554607945977818233891850896884623286405535', '9.82736E+3437548955564', '9.82736e+3437548955564', 20, 3);
  t('7.9045762440256318425790659793143E-4965904173099', '-3.6938684611042279394235224593675E+38389', '-3.69386846110422793942352245e+38389', 27, 1);
  t('-7E-462234653159', '-6.6847478792421727085344807185389002E+637004169', '-6.68474787924217270853e+637004169', 21, 6);
  t('-561.2720091692360113067783104954803622754208325102871818', '-7.4130142024929865261318486934E+32164358643', '-7.41301420249299e+32164358643', 15, 5);
  t('2.9903723964619271448953261629215250845808357901962E-51544547', '-7.2845375732905385235824832397797E+2060566536801', '-7.2845375732905385235824832397797e+2060566536801', 71, 4);
  t('-8.9582345656083E+809069666', '2.046451423561257913699884388407E+1935885444277270', '2.046451423561257913699884388407e+1935885444277270', 67, 5);
  t('-2.856771363027963E+1198409198036', '4.605132333E+709', '-2.9e+1198409198036', 2, 5);
  t('9091734240572059777337372169615492503869182301.1247975', '-8.3901616123211619672445427440292E+133319243', '-8.39016161232116196724e+133319243', 21, 1);
  t('5.233579714171457111588E-49936', '-1.2617273074210E+6667', '-1.261727307421e+6667', 14, 4);
  t('-8.654883657758180169090304049968356295218972434990113E+14841560166', '8.33657387806916277E+597026', '-8.6548836577581801690903040499683e+14841560166', 32, 2);
  t('-8059331253.58719027386485290667153', '7.9565819816680714877389072141477744303559651093340424969392E+715299', '7.95659e+715299', 6, 0);
  t('-8.2287694685E+511', '-6.606E+375989867821', '-6.607e+375989867821', 4, 0);
  t('-9.32984690617E+31581741', '-9.42018175E+74364348717760', '-9.42018175e+74364348717760', 26, 4);
  t('-6.20085923820376E+8160673', '-8.795906485970620E+579650017409', '-8.7959064859706200000000000000000000000000000000000000000000000000000000000001e+579650017409', 77, 3);
  t('6111077.73449423349820841419947108867573056064817', '4.8591E-2967852', '6.11107773449423349821e+6', 21, 6);
  t('-5.6173317869768505246856457356986780E-645911516', '-6.48234137251910937E+13672413115716', '-6.48234137251910937e+13672413115716', 75, 2);
  t('1.9262000404646981284372692010779E+487924', '-1.535749951E+3638', '1.9262000405e+487924', 11, 0);
  t('-4.10762806215627722794306907745530381615754854740321381196382E-791154205', '3.028775154170974E+25608327', '3.028775154170974e+25608327', 39, 6);
  t('-5E+1218', '2.866422112642480598913127853764805280E+99948556185675', '2.866422113e+99948556185675', 10, 6);
  t('1.3687497629329880657607582012353633048331464879213827E+17963259014145', '-5.490245106915E+23852306753', '1.3687497629329880657607582012353633048331464879213827e+17963259014145', 69, 5);
  t('3.85212724377624295090535229379E+9478758', '4.0451450081831497E+87972', '3.85212724377624295090535229379e+9478758', 48, 4);
  t('-9.08908544982E+6617337143', '-6.5142538665739447016963430365560E+7631260', '-9.08908544982e+6617337143', 44, 5);
  t('5.959457309231332E+41', '-5.070718567389E+561214349', '-5.0707185673889999999999999999999999999999999999e+561214349', 47, 2);
  t('-3.728622750545506512080484234354229671633274E+158233', '7.179319446281008245342486526801997438459E+109038903341', '7.179319446281008245342486526801997438459e+109038903341', 45, 5);
  t('-5.7614024596800067218526669656238695848489E+224985466854', '3.2427446481735120819953731236322538554448401584592050E+8378', '-5.7614024596800067218526669656238695848488999999999e+224985466854', 50, 1);
  t('-3.4835323431888473771840263159327068602635020872E+7998239783115', '6.7428100586109632952356510043358059719760985E+2038539186746204', '6.742810058610963295235651004335805971976098499999999999999999999999999999999999e+2038539186746204', 79, 1);
  t('-6229133496.4893924446161519601394345107', '-2.276916342205932216117805016795613418000924136833E+16242080', '-2.276916342205932216117805016795613418000924136833e+16242080', 70, 2);
  t('5.35984699E-72107915904', '-7.76276928415635E+61389759002525', '-7.7627692841563499999999999999999999999999999999999999999999999999999999999e+61389759002525', 74, 1);
  t('7.1119339064979752108567832E+929808812', '-1.67553193934111426722745645998029754581442648081654760882657E-2916957357', '7.1119339064979752108567832e+929808812', 56, 6);
  t('-243072788544178675499517239569256568712198846202767712.62', '-4.33480855071507017886773199908E-35324630', '-2.43072788544178675499517239569256568712198846203e+53', 48, 5);
  t('-7.86878210709758E+9527', '-1.7336937634386E-9343103', '-7.86878210709758e+9527', 65, 2);
  t('4.079210074712408981696154435299450106E+93', '2.23415225492269115022951067888975527E+47', '4.07921007471240898169615443529945010600000000023e+93', 48, 0);
  t('-3.01237791565426512047386800693447E+678081', '-2.4751113813529797739111066953E+2882403566', '-2.475111381352979773911106e+2882403566', 25, 2);
  t('-4.64E-187769790111', '-3.186089369230379696332522E+2132959976908', '-3.186089369230379696332522e+2132959976908', 78, 5);
  t('1.3727020422764194180E+4584765', '-6.3616E+78692', '1.38e+4584765', 3, 2);
  t('-4.8117877099593937804832068934927E-3538', '9.13916952072141473559928404471708806E+93168534173363', '9.13916952072141473559928404471708806e+93168534173363', 39, 2);
  t('3.9617E+17367957171734', '-2.9450450488701070872960195710709108035833204E+2040162', '3.9617e+17367957171734', 39, 6);
  t('-5.371658726088474609E+287219966201', '-6.20518769279687306882E+49565185', '-5e+287219966201', 1, 4);
  t('-7419029829396747927663790849547151266796', '-5.9454693938125138515717817737024996E+918166986842', '-5.9454693938125138515717817737024996e+918166986842', 73, 4);
  t('3.07E+95277', '7.57199363734149184622458305448877770753853237840990735194E+5053804512331', '7.57199363734149184622458305448877770753853237841e+5053804512331', 48, 5);
  t('372', '4.0265633445977342886914858196025527970624174904E+64814780518', '4.0265633445977342886914858196025527970624174904e+64814780518', 49, 1);
  t('8.2434115226873751346578963854690734024845578463E+7994295397592428', '3.278854063807E+9359', '8.243411522687375134657896385469073402484557846e+7994295397592428', 46, 6);
  t('-7.45390E-812', '7.709625209977879913214128557108751E-7981196011445', '-7.45e-812', 3, 5);
  t('-3.3067553560851282138678987501752281E+7837691181', '2.95450700E-324345', '-3.30675535608512821e+7837691181', 18, 2);
  t('-4.132725853511692382203983950527438919940659554630E-64400138', '4.360967917876345295518228613189623E-7924', '4.3609679179e-7924', 11, 5);
  t('-7.623450291473352063748519929E+11275984419428', '-4.025668803346594949576561567E-93708553284332', '-7.6234503e+11275984419428', 8, 3);
  t('3.3495505285238580755935342866555E+716714246983274', '-4.76240159579095E+7974238', '3.34955052852385807559353428665549999999999999999999999999999999999999999999999e+716714246983274', 78, 1);
  t('-1.136660052521958775081132655E+780346585615836', '-1.4135244992E-646125418487', '-1.136660052521958775081132655e+780346585615836', 57, 1);
  t('7.7725969446340451788873889127408996E-54272576921', '-9.9100709E+53123563', '-9.9100709e+53123563', 22, 4);
  t('-2.07E+838401', '-2.172095490971831419956111597314147500878965497E+926', '-2.07e+838401', 56, 2);
  t('6.80671234274E+24353107821', '1E+4750938956', '6.80671234274e+24353107821', 35, 1);
  t('4.2085E+131473', '-5.1978121837876172298218903357871360922005E+2099146', '-5.1978121837876172298e+2099146', 20, 1);
  t('7.069049473296496848033268695052361630426E+9568897081', '4.386868537351368673177869E+390306297083752', '4.38686853735136867e+390306297083752', 18, 6);
  t('-6.16478547257945184155358047295366261455E-38', '-2.05173968079268247011459019923423420235943277500358858E+5308523078501', '-2.05173968079268247011459019923423420235943277500358858e+5308523078501', 60, 5);
  t('1.383306490919538688E+81879426409289', '-2.265643768191163573874470768963025734277031037574241340460E-2379', '1.383306490919538688e+81879426409289', 55, 2);
  t('6.65075972731238922E-930707069', '9.45703E+4509158241767', '9.4570300000001e+4509158241767', 14, 2);
  t('-9.67250937130732794417157057421320331204604270225E-574', '7.737619718560572622199221865663491231727577E+782142770968324', '7.737619718560572622199221865663491231727577e+782142770968324', 61, 2);
  t('-89022.41482369848442890908256566643503482318530572528', '-5.695027009E+39', '-5.69502700900000000000000000000000008902241482369848442890908256566644e+39', 69, 4);
  t('-1.179762010268198245134282222435204048846859133388E+68346761', '8.362818497E+8896956578994', '8.3e+8896956578994', 2, 3);
  t('-3.236195523012E-83254192', '3.378251334898933005797757298070303713450750E-2142943475475', '-3.236195523012e-83254192', 47, 5);
  t('2.175532119419909257429360635876470707716347E+1182196405810284', '9.464118566590516974257535899157E+154099855', '2.175532119419909257429360635876470707716347e+1182196405810284', 58, 4);
  t('-1.85132291032742043751664841732E+6749817', '-5.51291831643359159273855611591861000641174141E-3605', '-1.85132291032742043751664841732e+6749817', 37, 1);
  t('-2.2E+623', '-9.68323984843808465603512E+183103449888', '-9.68323984843808465603512e+183103449888', 44, 1);
  t('-4.1828527968981705591657649860294016662956868724355853E+19307151738', '5E+544965526789', '5e+544965526789', 8, 5);
  t('-0.00002106140160087411819018460948258476', '5E+95325', '5e+95325', 18, 4);
  t('-9.03580108114826811062868525189870794017579791E+85492885978', '4.475647546864518189E+4051302309', '-9.0358010811482681106286852518987079401758e+85492885978', 42, 3);
  t('6.2061097171903904854635354E+72', '6.9015321E+72', '1.31076418171903904854635354e+73', 58, 0);
  t('1.1271190477663817879311605112662756265493718175791E-99055996513', '4.0284287354371351719008499649E+611668504239704', '4.0284287354371351719008499649e+611668504239704', 53, 5);
  t('-9.070491212596496609867481793574894990535496090177868969854E+658840970078', '-8.762526950581E+1026278', '-9.070491212596496609867481793574894990535496e+658840970078', 43, 6);
  t('8.369641142361236162304528128393153546691876416984E+868902939086', '88804.9739647026947806803702930945515597341930611269861526294', '8.369641142361236162304528128393153546691876416984e+868902939086', 63, 4);
  t('6.25303187109832956485812114E-332', '-4.16057982689671800552E+47', '-4.16057982689671800552e+47', 71, 3);
  t('-6.9097762286078E+103218989497771', '7.0148859392951759145156525969964358929113614070644E+965971647', '-6.90977622860779999999999999999999999999999999999999999999999999999999e+103218989497771', 69, 2);
  t('-8462160.18230790532620920473885801869', '-6.823491622045784142358976141661068180768388482486E-88', '-8.46216018230790532620920473885801869e+6', 63, 1);
  t('-1.38690581025619355662279638E+350', '-8.955517678E+1853909510957206', '-8.955517678e+1853909510957206', 43, 6);
  t('-5.662020023720939855003554858E+986417261', '-9.2325541872875317742243079569723516648762943536933E+8451619714730', '-9.232554187287531774224307956972351665e+8451619714730', 37, 6);
  t('6.3E+53457735417', '1.0697531572559526955214421502791847069093610417E+337274762167831', '1.069753157255952695521442150279184706909e+337274762167831', 40, 6);
  t('3.9639194078060371658112341512E+8634764363825', '5.6698008620633286745528866293971251897089840345693E+92903695103312', '5.6698008620633286745528866293971251897089840345693e+92903695103312', 58, 4);
  t('9.584900900395488487116295413021E+117531807633', '6.9324408293249E-1363303', '9.584900900395488487116295413021e+117531807633', 36, 1);
  t('-4.604250E-48595989500099', '9.4091100983915005728854438780350386722124E-37700994', '9.4091100984e-37700994', 11, 6);
  t('-1.2205998125948126093868793881263340265049E-998794', '3.649848500316357500547879920E+99', '3.649848500316357500547879919e+99', 28, 1);
  t('5.918177310439220307623448011380E-257', '-2E+6749795432', '-2e+6749795432', 66, 0);
  t('-6.5872656998444641442648407054E-767', '3.51329355E+43', '3.51329355e+43', 30, 0);
  t('-5.562719334179628563480E+318', '-39.08012730982717', '-5.56271933417962856348e+318', 63, 6);
  t('2.697889121257369578794302804713840199125414782E+722936866339985', '4.751392590949911333231539875346676790E+62614526', '2.697889121257369578794302804713840199125414782e+722936866339985', 80, 4);
  t('-8.4159737886565185959840386901023E+27998158934', '-0.0006910115130926718132200726', '-8.4e+27998158934', 2, 4);
  t('-3E+716433394', '3.7578484576584432474906E+2055714482908573', '3.75784845765844324e+2055714482908573', 18, 3);
  t('4.1991889000120954623433E+120229080204960', '-3.3546959E+9', '4.19918890001209546234329999999999999999999999e+120229080204960', 45, 3);
  t('-6.08543E+593', '-4.2796104838229607523910876902809840004902562E+34653525253701', '-4e+34653525253701', 1, 4);
  t('2.097885E+2218000078041', '4.0901314719937469579558216315076E+7927180549480', '4.0901314719937469579558216315076e+7927180549480', 59, 6);
  t('-6.84838207210244388737666E+751865040904331', '5.6342237604877455194367788E+3566137', '-6.848382072102444e+751865040904331', 16, 0);
  t('-6.8451100837629162348231001524006244741E-755406185', '2.81427E+40', '2.81427e+40', 22, 2);
  t('-8.10074412915767163954342093963E-879027289', '6.625725742052367170881318E+42326835776941', '6.6257257421e+42326835776941', 11, 5);
  t('-1.5468294728676E+95900', '-5.367279217862540884E+306281312532061', '-5.367279217862540884e+306281312532061', 51, 4);
  t('1.264071719E+321503958769375', '-6.1691460640507553006E+342', '1.264071719e+321503958769375', 19, 4);
  t('-2.0190695742646710403319E+66234', '3.427522354645614160599771916552730553886396746E+1672076199400898', '3.427522354645614160599771916552730553886396746e+1672076199400898', 68, 0);
  t('-1.27950222977734E+3618101661471', '-6.038794398E+18', '-1.27950222977734000000000000000000000001e+3618101661471', 39, 0);
  t('-2.89312941853532013161583001147737377090976778027249679E+941233067914692', '4.253036163217764443378E-507244721095', '-2.89312941853532013161583001147737377090976778027249679e+941233067914692', 80, 5);
  t('-1.3536812468636840678594E-236803', '4.36418509785519617E+73502037', '4.36418509785519617e+73502037', 32, 5);
  t('7.9674746447477270E+32024230', '9.378127937107139691153209079100915E+74421', '7.967474644747727e+32024230', 53, 1);
  t('8.6526468560473890123167324816104194119468735E-45492', '-9.09989833699618989418290584169989921293754881009537E+328', '-9.0998983369961898941829058416998992129375488100953699e+328', 53, 1);
  t('-2.017312910445149289870836785763240127750931575919499156E+650307495816', '-5.190143261E-8152', '-2.017312910445149289870836786e+650307495816', 28, 4);
  t('-5.6E-269576', '6.2177568119546215E+88218225511', '6.2177568119546214999999999e+88218225511', 26, 3);
  t('-68.5923815861262', '-3.811396801043640697131493382343876010462109E+70269300763209', '-3.811396801043640697131493382343876010462109e+70269300763209', 69, 5);
  t('1.34183756713717E-74940093', '-2.175904333624324764545701179292E-7612659894530876', '1.342e-74940093', 4, 5);
  t('7.3823394401894199541813669526973321281009006E+342964619755', '-2.7835169984777076297255968060492117442353849489382980019E-3932414405016', '7.382339440189419954181366952697332128100900599999999999999999999999999e+342964619755', 70, 1);
  t('3.50120867311240338577225E+98746437806395', '4.634063E+694234208', '3.5012086731124033857722500000000000000000000000001e+98746437806395', 50, 2);
  t('-8.8828437E+56', '-7.25397665E+8727060744642966', '-7.25397665000001e+8727060744642966', 15, 0);
  t('9.74635382016114470156547448407257344E+49850', '-1.1076849722027014065336890857136074823222610810E-9548340671471', '9.746353820161e+49850', 13, 6);
  t('8.897670839087223245089442771565E-378988393824', '2.3526883180071781814641756178335866180190260975904818586E+635805332', '2.3526883180071781814641756178335866181e+635805332', 38, 2);
  t('608.55', '77150540721455106827873201970.2593264916906504211808899164', '7.71506e+28', 6, 0);
  t('6.4067091417754590779085450081350996049285898291514993E+694340', '-8.5034905E+494', '6.4067091417754590779085450081350996049285898291514993e+694340', 67, 6);
  t('4.36091111444062354E+999893190041790', '-8.558549934089025241421814657431478803E+540863327424681', '4.36091111444062354e+999893190041790', 23, 4);
  t('2.02780522531675E+136520669', '-9.0788306407510641081565940668E+8724564', '2.02780522531675e+136520669', 67, 5);
  t('5.93767365838216619745732348758017158E+955', '8.9721E+431904020646180', '8.9e+431904020646180', 2, 1);
  t('2.8711643148116200158949330849885047181E-4600476362', '9.75359230961676936389552548160010E+48898513453', '9.7535923096167693638955254816001e+48898513453', 55, 4);
  t('-6.030923423855781598475618709786333713169903670511765E+2251933072', '-6.13420556973715923255349754248653E-3225321584897', '-6e+2251933072', 2, 5);
  t('-5.0643343191104039786855760123159419069764106624E-98164034180568', '3.26E+831244', '3.26e+831244', 9, 5);
  t('7.9721847193575692498E-6862754', '9.7863576467540E-2650985287153', '7.9721847193575692498000000000000000000000000000000001e-6862754', 53, 2);
  t('1.573624345267447620812635567020744272980281482501E-221456', '2.8539739189E+650', '2.8539739189e+650', 12, 5);
  t('9.637986740398752374655875060E-3225576540261', '9.922432863495047234277553E+488324', '9.922432863495047234277553e+488324', 64, 5);
  t('-1.97125828493768337239967389442233708197656129981818E-7773', '-3.96013273553579497952556645107287914498537E+749953246360250', '-3.9601327355357949795255664510728791449e+749953246360250', 38, 2);
  t('8.82633392266893780638594971747302E+61229617', '9.79196617176522263400962907E+69216771', '9.79196617176522263400962907e+69216771', 27, 6);
  t('-9.302E+58430888245', '2.6636053535239380639211536504878285554494542504889E+79', '-9.302e+58430888245', 39, 3);
  t('-2.535839780522381110942953914534457002360765692E+6526170967', '2.62901893438991069611578793857879593638283E+788506702', '-2.535839780522381110942954e+6526170967', 25, 3);
  t('-3.9345030935445477843107831E+64762840931', '-3.1095283664E+7659727206401166', '-3.1095283664e+7659727206401166', 31, 6);
  t('-5.5484066781366337113E+2491043234', '-8.550643596361753733664982399991E+5499636640', '-8.55064359636e+5499636640', 12, 5);
  t('-3.976379752653358042097409682967108553805676770208E-40967289670', '9.1059206981657962655E-61061817374', '-3.976379752653358042097409682967108553805676770208e-40967289670', 53, 6);
  t('5.9462859572100716863693502905346778635821306287894865E-832187', '7.8737352420E+68285205095552', '7.8737352420000000000000000000000000000000000000000000000000000000000000001e+68285205095552', 74, 2);
  t('-9.11734E+18', '3.2853037519476420901581776332264959408326665988606202841463E+6304', '3.28530375194764209015817763322649594083266659886063e+6304', 51, 0);
  t('5.408E-422768', '-4.8335290132E+201375094', '-4.83352901319999999999999999999e+201375094', 30, 1);
  t('6.6092880346019434575388497651410286421289864004619E+4656', '3.9295689345852656614938567253540505E+508', '6.6092880346019434575388497651410286421289864004619e+4656', 80, 4);
  t('-4.91401093941520751075270E+4185533155', '-6.7381334573014505967256547E+42772145075452', '-6.73813345730145059672565470000000000000000000000000000000000000000000000001e+42772145075452', 75, 3);
  t('-5.8801E+99602', '5.6576405606427550227433851380E+128', '-5.8801e+99602', 65, 4);
  t('5.0710745757516767273108615078180531823602028727499202502506E-610041', '-9.0852256033047655667019322791175377236422690704992E+9179', '-9.0852256033047655667019322791175377236422690704992e+9179', 69, 3);
  t('-5255068.0825602021074390953683541500', '-4.592648796456153745297889532992250988472527208E+7065130', '-4.592648796456153745297889532992250988472527208e+7065130', 57, 1);
  t('-8.61396448876019522705889E-41106369062753', '-4.41483E-492481', '-4.41483e-492481', 26, 6);
  t('1.14016605836177006720482326997467845E+14993503', '-6.177069655310505213411310366510839650686664264E+2157211', '1.14016605836177006720482326997467845e+14993503', 44, 2);
  t('9.300645020005693E+3504806', '-5146273.1057618938602833714714181477791932845153454810', '9.300645020005693e+3504806', 53, 4);
  t('-5.2599E-3314798970191587', '5.4855558375519149825404452047974310885988201117831335654E+26514', '5.4855558375519149825404452047974310885988201117831335654e+26514', 67, 2);
  t('6.59649576007342568868449989E+538872427755185', '891005.33873644936728042736', '6.59649576e+538872427755185', 10, 1);
  t('2.4554170563505956971541656717107582335E+34170418740515', '5.83837E-7653573454209795', '2.455417056350595697154165671710758233501e+34170418740515', 40, 0);
  t('3.41080223464725380344854090346498493563054230776E+7037668', '3.14842150141E+41335484610639', '3.14842150141e+41335484610639', 77, 1);
  t('8.681311157248225923748470344192829943E+231223026', '-1.07602358980463083281256556022721406134270139605003141390391E+14272011657878', '-1.0760235898e+14272011657878', 11, 2);
  t('8.225279625048415710E-9666971511366', '36496651859919549044121555436.37268263369', '4e+28', 1, 0);
  t('-8.614413777937082274727372801222121623313883147740278E-61723045895580', '5.28378902145634261235395340604401595937012799E+7419410', '5.28378902145634261235395340604401595937012799e+7419410', 73, 6);
  t('-7.077924E-1266241', '-3.913E+834901020942', '-3.91300000000000000000000000001e+834901020942', 30, 3);
  t('-3.732216167925363337458992550E+4177548766584500', '-8.7934860531224834587594863442634283E+1462688214811066', '-3.73221616792536333745899255e+4177548766584500', 52, 4);
  t('6.97755E+6', '2.2877809304854828E+4407', '2.2877e+4407', 5, 1);
  t('-45.4067286479389782840054', '8.91218613981377815287835E-188925528640', '-4.54067286479389782840054e+1', 62, 0);
  t('-4.800831476904E-85082047472366', '-4.379291E-93', '-4.379291e-93', 42, 2);
  t('-1.194105902974498122551096123E-94894098890', '1.5568136322038996483E-354014309250', '-1.194105902974498122551096122999999999999999999999999999999999999999999999999999e-94894098890', 79, 1);
  t('6.75696439265896E-86458437937', '-6.18815899276E-2388', '-6.189e-2388', 4, 3);
  t('-3.409024632670161023496280178531339457629E+27112316', '4.46068565317202874282395165E-495827551', '-3.409024632670161023496280178531339457629e+27112316', 53, 5);
  t('-8.813022012056952980968848339036938602362901545401799834976E+391789', '4.731941223371247463346388871E-23952433', '-8.813022012056952980968848339036938602e+391789', 37, 4);
  t('2.923027E+7530', '-4.419058716377620251E+4887595742770', '-4.41905871637762025099999999999999999999999999999999999999999e+4887595742770', 60, 2);
  t('-6.664622043984176E+4712094542244', '-9.32102112282264088803042213727564362E+66', '-6.66462204398417600001e+4712094542244', 21, 3);
  t('9.02650078E-467498', '4.9512729715812779516286425980E-549954546', '9.026500780000000000000000000000000000000000000000000000001e-467498', 58, 2);
  t('6.689E+85003676', '-6.638166578074876738622496356890159228155342785776E+60234', '6.689e+85003676', 15, 4);
  t('-2.05652510811513048748544693966238135E+818235279987367', '4.1709344738E+74574013379', '-2.05652510811513048748544693966238134999999999999999e+818235279987367', 51, 2);
  t('6.398711908004381554802283360719752E+4730095275933669', '-4.31188257820938081843719397433E-920870554', '6.39872e+4730095275933669', 6, 2);
  t('8.73156173934528358105E-953382092198', '1.340875238602177021401678575690530646009809571023927E-271', '1.3408752386021770214016786e-271', 26, 4);
  t('-5.95550E+205', '2.6250371757678874307877154206659878727065719650450089880E+41229666', '2.6250371757678875e+41229666', 17, 2);
  t('7E+4830388', '-9.8714354879284623160331871741477171205989649E+4186330687', '-9.871435487928462316033187174147717120598965e+4186330687', 43, 6);
  t('-4.3582926017193518533450323833708928536970E+625928309325', '-7.78629536748566236211756044018672869112292055163649E+3878224580', '-4.35829260171935185e+625928309325', 18, 1);
  t('-6.609279764226070813913560543984805556E+5187974', '3.905435051270559004992988312452E+230155836810', '3.905435051270559004992988312452e+230155836810', 68, 0);
  t('9.45887186206705E+73502', '5.96429970044883173203099491097390061980598800252174E-515181547125762', '9.4588718620671e+73502', 14, 4);
  t('8.89591337480739908E-3743', '-3.8735632122461E+397767219633', '-3.8735632122461e+397767219633', 72, 6);
  t('-3.153382278776414362284133112135439578116973502546065164648E+74851409249', '-3.48681230697632258727902322960960915324023093E+766', '-3.1533822787764143622841331121354395781169735025460651646480000000000000001e+74851409249', 74, 3);
  t('3.9336988381067397011905489E+527055189766', '8.64378899166797231243984509974017E+117148', '3.9336988381067397011905489e+527055189766', 47, 3);
  t('-6.328346210870118641610455940563281499E-7093', '4.452E+232368123', '4.452e+232368123', 6, 6);
  t('-5.41149301394611E+3527860638', '-5.5223448959516291E+420253319108', '-5.5223448959516291e+420253319108', 48, 6);
  t('2.574581610771706413056866071726985676312804627268933154E+94644971150599', '4.19564452200065037E+5871645', '2.574581610771706413056866071726985676312804627268933155e+94644971150599', 55, 0);
  t('4.338817746880378436738495919632651616263614596719991E-16853', '2E+5897995878425272', '2e+5897995878425272', 69, 1);
  t('9.71853832510229703370187576961518504499101205577E+9111269390', '3.2496E+50', '9.71853832510229703370187576961518504499101205577e+9111269390', 48, 1);
  t('-6.0218320829566386022648890733157200033176949278160808E+82557730169', '-5.361170016575593681E+143113', '-6e+82557730169', 1, 6);
  t('4.6490451654393844446651609606687868475756153576312E-376341', '2.0912257288342702429330869412758769521111E+78', '2.09122572883427e+78', 15, 6);
  t('3.66994928396252359649930732470531579201351871187E+99703133411809', '5.529480595445504035611119719527454873014652197E+812902246788', '3.669949283962523596499307324705e+99703133411809', 31, 5);
  t('-8.74517397956014322789450002358854144583299779882180502E-87701728', '27389405822313039778073140591625089949761116819676335.6', '2.7389405822313039778073140591626e+52', 32, 2);
  t('-5.021020055608114534378724810450139799833E+767625484', '-8.884576767698250355498196340E+7492833393', '-8.8845767676982503554981963400000000000000000000000000000000000001e+7492833393', 65, 3);
  t('7.62247750609277312122360492978483581579463629778131427161448E+142155730', '-1.366156567618910E+846', '7.622477506092773121223604929784835815794636298e+142155730', 46, 6);
  t('3.80236397982391857012210459271537523992857680345840985E+66118241', '-1.884520251072700423391655E+6217874997702719', '-1.884520251072700423391655e+6217874997702719', 56, 4);
  t('-9.2321759503981412397425795826926029942098E+8733527', '-8.7062475124E+556924354520332', '-8.7062475124e+556924354520332', 72, 1);
  t('-1E+2739', '1.07504967942503E+72399100', '1.07504968e+72399100', 9, 2);
  t('-4.501139811060428663086503396605E+7512952841277', '9.74754E+612863833268', '-4.501139811060428663086503396605e+7512952841277', 61, 4);
  t('-2.791920E-6979589', '4.583273832236155230494571397462096592220280771098596264E+469', '4.583273832236155230494571397462096592220280771099e+469', 49, 6);
  t('7.0187499136E+7492068713516986', '3.7142E+2095489320006', '7.01875e+7492068713516986', 7, 4);
  t('1.9019789382830376195918378941236968514484E+71410', '-7.99725347286638007091478935373392E+7365295302', '-7.997253472866380070914789353733919999999999999999e+7365295302', 49, 1);
  t('3338.5172841353250437158190589780380385578461660', '6.77505835256019499058101854770781E+7539', '6.77505835256019499058101854770781e+7539', 59, 3);
  t('-4.9444824029891309040139610391966744050392458E+8876674546259', '4.89404922143193474090165342957850665932E-6025919172081577', '-4.944482402989e+8876674546259', 13, 2);
  t('4.6936762453E+233633', '2.0298247403278282426503226083480322850955621952E-8131464822976878', '4.6936762453e+233633', 45, 6);
  t('-9.32605858974123003945425542726428573E+68805', '2.264033411227266600823672868731041126239E+6324382', '2.2640334112272666008236729e+6324382', 26, 6);
  t('9.1539648386573763709509451330207457217916752551140E+23040194', '1.627941876070761636770835706311844E+828318811692', '1.6279418760707e+828318811692', 14, 3);
  t('-6.4390826883656826596492556633348105286077347172155E+63033524473414', '-7.956411056870579819719891052088596785E+153011', '-6.4390826883657e+63033524473414', 14, 0);
  t('2.7064812231096664184153270330230630627E+8960', '4.44890E+14', '2.7064812e+8960', 8, 1);
  t('-7.327642780E+587240674278', '2.07656E-985537910254605', '-7.32764278e+587240674278', 69, 6);
  t('-1.42E+96055746', '-8.5604850422145679122796946005700E-4653494', '-1.42e+96055746', 7, 5);
  t('4992508.301645767523155459894678718123351604802445323895619', '3.72990E+13842745125710', '3.7299e+13842745125710', 80, 5);
  t('-1.256953594E+7236118344136', '-4.09465177284425514185539130054401074055459986086339943540E-83546135946', '-1.2569535940000000000000000000000000000000000000000000000000000000000000000001e+7236118344136', 77, 3);
  t('9.41029318105473329613253011839087290833275595995122371849056E-29625', '1.9307258474166719484463724285178525232866288768E+70920510546', '1.9307258474166719484463724285178525232866288768e+70920510546', 76, 5);
  t('9.813278809076871149704216239300830845772E+29235417875422', '7.726439068E-988962', '9.813278809076871149704216239300830845772000000000001e+29235417875422', 52, 0);
  t('-8.463639551664145703187886E-156', '279081667.59356802248980941113608257729512599', '2.7908166759356802e+8', 17, 3);
  t('4.47214184653505223492E+883515139', '3.309873456105815E+64313779158', '3.309873456105815e+64313779158', 32, 5);
  t('3.491602376219586080929052381565390173E+7489419310036271', '8.10306339952872193677742862466525541949283E+7626', '3.4916023762195860809290524e+7489419310036271', 26, 6);
  t('6.46583186092292730436357802632474725780631273259822452268E+2939004825', '-6.65941847768578026064519388121523E-28546746064', '6.465831860922927304363578026324e+2939004825', 31, 1);
  t('1.2710796647887570986885389E+640281', '5.9877E+5313', '1.2710796647887570986885389e+640281', 70, 6);
  t('2.771760951631352296263177504506321685499956876587860325415E+343435941', '-6.792699861338523228756017166452E+7780345', '2.771760951631352296263177504506321685499956876587860325415e+343435941', 74, 6);
  t('-6.20173382474746097657595687E+1448469961988', '-1.657862720655925862E-2011442610356', '-6.201733824747460976576e+1448469961988', 22, 0);
  t('7.19E-428778666962', '9.28150017607E-3474', '9.28150017607e-3474', 79, 5);
  t('-747.551680232156958777701799199491737550380489867119185020', '-9.73709891915652547616933631936419903292094E+651', '-9.73709891915652547616933631936419903292094e+651', 53, 6);
  t('-8.078E+3315302', '2.15169943796281232304472863921988890E+2567', '-8.07799999e+3315302', 9, 1);
  t('5.8E+39461267787915', '2.5148686588857181692588619317395E-31', '5.8e+39461267787915', 15, 1);
  t('-2.22064998995261039631383890501518945859422759945176340341E-635060', '-1.39843508580856562818161E+64778849', '-1.39843508580856562818161e+64778849', 73, 2);
  t('5.0034947083288E+363060935274072', '-4.66561393145727043E+619806221', '5.0034947083288e+363060935274072', 73, 6);
  t('24.963545620543750269937636534', '1.66087688E-6943765783047602', '2.4963545620543750269937636534e+1', 63, 5);
  t('2.92620971559898998370677002811838966620180373404214E+765499459035', '-9.423154983933572852327650800018652947E+6448557538461', '-9.4231549839335728523277e+6448557538461', 23, 6);
  t('2.5036763992E+495898874868823', '-7.56061998765E+82514018', '2.503676399199999999999999e+495898874868823', 25, 3);
  t('9.96718458512404477043E-1710123367716', '8.104905016458678631054282511661E+33', '8.104905e+33', 7, 6);
  t('-1.1900511840900E+5036591923240275', '8.8806031666267441124095697286490841782E+2241911599', '-1.19005118409e+5036591923240275', 38, 6);
  t('-9.7E-34', '5.7452E-54', '-1e-33', 1, 3);
  t('-2.310978793269215902937528986527468629E+99530314910', '-7.923E-1175647952319965', '-2.3109787932692159029375289e+99530314910', 26, 1);
  t('6.9518585698785793563297313E-945937', '6.932072394145325883412710200096049709866325251010349E+68', '6.9320723941453258834127102000960497098663e+68', 41, 1);
  t('4.271660437098185580170228994794504852515473072113049511E+6867343324461', '-9.619475429747231385107E-874917830245325', '4.2716604370981855801702289947946e+6867343324461', 32, 2);
  t('2.8244742854519608761599348370226032496180257E+797560705822049', '-7.27641103678856E+159287', '2.8244742854519608761599348370226032496180257e+797560705822049', 63, 2);
  t('-5.079803926450090090557901718539991325403110917548785208E+6562216', '1.271E+9', '-5.0798039264500900905579018e+6562216', 26, 3);
  t('-49607.52920242', '-3.0423605762260014205798310416784E+687914', '-3.042360576226001420579831041e+687914', 28, 2);
  t('8.319E+65', '3.450099689207871250018932572724713674993594305684395175E+1105524162723988', '3.45009968920787125001893257272471367499359430568439517e+1105524162723988', 54, 1);
  t('-4.155002715998248749260902E+351917', '-8.85351844209282695711327416305205905957026104336372532549E+918163', '-8.853518442092826957e+918163', 19, 6);
  t('6.677529483504817634992289924634252287435032024E-1398276710', '-8.02352672161419209922046679721242087229537998864E+5636072179776', '-8.02352672161419209922046679721242087229537998864e+5636072179776', 60, 3);
  t('1.00324334090313821425119773E+18162716', '-9.8311587249496686218477917338573149891E+4389576', '1.00324334090313821425119773e+18162716', 66, 4);
  t('-1.80675212281025983502183816255800130855681677499017073164E+38154248585736', '3.839745515139E+163721165580865', '3.839745515139e+163721165580865', 73, 4);
  t('-6.467118897564271943317E+31710530634', '-4.331781383544615507962118E+933', '-6.467118897564271943317e+31710530634', 54, 4);
  t('-2.00021771512644929873355287776353191315467611434075665555E-1268699717413292', '4.8989E+30374843', '4.8989e+30374843', 26, 6);
  t('4.7593360865E+590704292', '4.39E-602', '4.7593360865e+590704292', 14, 1);
  t('-2.5938885083523496740704149254713151E+49220864964', '1.811700261417346237315E+73185', '-2.5938885083523496740704149254713151e+49220864964', 35, 4);
  t('-7.4285E-5714', '-6.532075721513261186334596133875081264260077128E-2890085', '-7.4285e-5714', 5, 1);
  t('5.57351254343543582636699210879757142834691520880008862E-253659', '1.090332810401880412785395916897131E+7634050', '1.0903328104018804e+7634050', 17, 4);
  t('1.34759310826671526469601E-8181', '-72.38464556646799', '-7.238464556646799e+1', 48, 3);
  t('1.5682248146079441047586601742341487542547034070484E+945939911', '4.79066844416290593239787E+824630834216', '4.79066844416290593239787000001e+824630834216', 30, 2);
  t('4.2383985061254134405021602500526708871317E+4881', '4.526045039605043193476', '4.23839850612541344050216025e+4881', 27, 6);
  t('1.599E-969990', '1.5975593E+4022964489627609', '1.59755930000000000000000000000000000000000000000000001e+4022964489627609', 54, 2);
  t('-7142119.720911165874810462516269453305725667289912712701', '-1.848408768057294511758173408760522534264253031503300E+8459', '-1.8484e+8459', 5, 5);
  t('7.037419557287745E+313', '7.05860865497331E-10810', '7.037419557287745e+313', 26, 6);
  t('4.56495E-5808318263076', '-2.91194632540987269142712896E+6924', '-2.91194632540987269142712896e+6924', 32, 3);
  t('-9.8244430339083147076468477082627178751690687E+4254083643929', '-38621.038110350129118309575926520846538317922338', '-9.8244430339083147076468477082627178751690687e+4254083643929', 49, 2);
  t('-4.584909897360563298559054720592232664161817685E+628120', '-5.1917450941716E-190914', '-4.5849098973605632985590547205922326641618176850000000000000001e+628120', 62, 3);
  t('8.03243446143872757E+211483377', '5.61727436301333E-260', '8.03243446143872757000000000000000000000000000000000000000001e+211483377', 60, 2);
  t('-4.827225749886953538495076813742475360224238189712524E-3303158225', '-6.71729542E-978132542253', '-4.82722574988695353849507681374247536022423818971e-3303158225', 48, 2);
  t('-5.649122235004848E+536', '2.68181547624628851020596865E+3343', '2.68181547624628851020596865e+3343', 67, 6);
  t('-2.6380E-4507628', '8.99487E-73271464863', '-2.638e-4507628', 61, 6);
  t('-4.929E+70234249851318', '5.627635992026919817804603195821345E-5786080113', '-4.929e+70234249851318', 20, 6);
  t('8.64202393453221450512127541149945167949410714E+565440368128', '6.33233383642357448522006484378656939213809E-1610829306794', '8.642023934532e+565440368128', 13, 3);
  t('-5.92855144169776069339856502548175905921889095165203E-8082006', '-7.316449962573146326313114898373E+7824299', '-7.316449962573146326313114898373e+7824299', 43, 4);
  t('-5.85651314448476344633017015538879112075882122776671257E+6013591480', '4.26163133590791573434721691328064198300379701792739380E+6595493', '-5.85651314448476344633017015538879112075882122776671257e+6013591480', 62, 0);
  t('-3.164150183623036654534E+51875', '1.0331715804445681688E+698957', '1.0331715804445681688e+698957', 42, 5);
  t('9E+48483', '-7.214794132691567671310611996E+247698569870', '-7.21479413269156767131061199e+247698569870', 27, 2);
  t('-2.4216039528322470122855038816218344802073120714E+7506689889966981', '-9.0251413964597101E+352784708996694', '-2.4216039528322470122855038816218344802073120714e+7506689889966981', 56, 5);
  t('-0.000273294', '-2.455139510312E+6923306', '-2.455139510312e+6923306', 74, 4);
  t('-8.302800118630199459723591606687249E+98774131', '-95.478225292506152790138888725617720152620824243', '-8.302800118630199459723591606687249e+98774131', 74, 5);
  t('8.5388207560433728E-487531241199571', '4.6189690903892583097304455452838826E+122511151', '4.6189690903892583097304455452838826e+122511151', 47, 1);
  t('6.56365772838268429E+80232692861310', '9.64896627700934397894E+7824211', '6.563657728382684290000000000000000000000000000000000000001e+80232692861310', 58, 0);
  t('-2E+291838639227', '-8.0532964996850802E+8656261582854', '-8.0532964996850803e+8656261582854', 17, 0);
  t('-8.90E+4686', '-1E-7', '-8.900000000000000000000000000000000000000000000000001e+4686', 52, 0);
  t('4.402131638E-6217', '6146491945.8', '6.1464919458e+9', 36, 1);
  t('5.12097706894147087504468951401E+256864997288296', '-1.777884422157E+77650', '5.12097706894147087504468951401e+256864997288296', 44, 6);
  t('2.5811533962478340838808059474995493626E+982498068', '-5.6831181893984052911450512752979067947489E+8631', '2.5812e+982498068', 5, 2);
  t('-2.8734291715863002559123E+147', '6.25942844695424081412114007619833760502265344E+399756305', '6.259428446954240814121140076198337605022653439999999e+399756305', 52, 3);
  t('6.9289577407962086513411E+107322', '4.8537743672675427493798440456781101916796E-99502', '6.9289577407962086513411e+107322', 35, 6);
  t('-5.061791767042498992573E+2309319519375', '7.642114490594556355251068470707459434284820307467411805810E+884217491', '-5.061791767042498992573e+2309319519375', 68, 0);
  t('4.8182518E-2515', '-4.83640399353409456275E+697526474723855', '-4.83640399353409456275e+697526474723855', 33, 0);
  t('-1.177038E-78878', '7.137322321377794375953981899619219081E+93515792', '7.13732232137779438e+93515792', 18, 5);
  t('8.0072790197256363087348481654E-886', '9.659444066130731326587575E+6881660627', '9.659444066130731326587575e+6881660627', 62, 4);
  t('-7.17670797285406422768223041889654735567494905100658013956648E+638026293', '-1.7E+60214', '-7.1767079728540642276822304188965473556749490510065801395664800000000000000000001e+638026293', 80, 0);
  t('-4.03388180749058619699003510605847924E+78545958', '4.3625456613499683721958241204896001040021717005325752328E+39364297178981', '4.3625456613499683721958241204896001040021717005325752328e+39364297178981', 58, 4);
  t('-2.972372382721668939819639679916327411826264121880723457E+1781010824756504', '9.71153144441156540627678120156141000907940917403794829825976E-573', '-2.972372382721668939819639679916327411826264121880723e+1781010824756504', 52, 5);
  t('1.862475926914447653233418227051858E+740720742', '6.8493448181126371550788106130738344303E-645359978319030', '1.862475926914447653233418227051858e+740720742', 40, 6);
  t('-5.578567892243038503277319633407036215243259262150063121E+594731667695', '-6.3517297363775545121884948155101764722841E+368781436541', '-5.57856789224303850327731963340703621524325927e+594731667695', 45, 3);
  t('291.124807017175824801547138875810139596516418378', '-89.92102721555465498', '2.01203779801621169821547138875810139596516418378e+2', 77, 5);
  t('-8.509683427852276646670570409266557284828335318865907116E-39839', '1.557861459370794009408683571127872493667E-1255779494263298', '-9e-39839', 1, 3);
  t('6.422043355924505799317111081083971466157E+3349989', '0.0033184335419253', '6.422043355924505799317111081083971466157000000000000000000000000000000000000001e+3349989', 79, 2);
  t('-84.367077976340', '-4.16576872870125019268679945E+84305', '-4.16576872870125019268679945001e+84305', 30, 3);
  t('6.419442715248065526915355491681740863421E+6495050058826785', '-5.206680E+9281303', '6.419442715248065526915355491681740863421e+6495050058826785', 61, 5);
  t('-8.4088228473734913640627E+327', '-2.09574304941817920598E+6913681643', '-2.0957430494182e+6913681643', 14, 4);
  t('5.99446087297375E-7932', '-5.772420691411103390267667347783985689260171E+2767252', '-5.772420691411103390267667347783985689260171e+2767252', 75, 4);
  t('-6.41203656665513688375399802402278705152875299525431086826233E+979666858266955', '-4.27219E-380100827', '-6.4120365666551368837539980240227870515287529952543108682623e+979666858266955', 59, 2);
  t('-6.25902119991670457733749303467418929679599586332398460842E-174878', '-8.2780689664804985927E-64480593', '-6.25902119991670457733749303467418929679599586332398460842e-174878', 59, 5);
  t('9.8431309607901348197595528724708873E+49050', '1.15500901869589638010747292352285E+3911847', '1.15500901869589638010747292352285e+3911847', 56, 3);
  t('-2.12106081580260210255945848745137201793581158315964E+25131294640592', '-7.85165105852117376421049993408929860044848E+444126704454', '-2.1210608158026021025594584874513720179e+25131294640592', 38, 6);
  t('2.226189547517E+773466371451', '7.0003089470039364E+4621403733677585', '7.00030894700393640000000000000000000000000000000000000000000000000000000000001e+4621403733677585', 78, 0);
  t('6.61443100729216275764908834386016575464653E+312916490935810', '9.43649732874441825128E-676693579', '6.614431007292162757649088343e+312916490935810', 28, 1);
  t('2.00316367455750763441863132393624952003E+1664776295953', '8.9041429682881927604452761496890904865503241514246712651051E-41', '2.00316367e+1664776295953', 9, 4);
  t('6.16310007323656E+923827938105', '-5.237117564730560853952E+2717', '6.163100073236559999999999999e+923827938105', 28, 1);
  t('7.973804113785698272010640978103310705492180197703029625E+426', '-1.616596006582394668283412445265827926465E+181546784669', '-1.616596006582394668283412445265827926465e+181546784669', 48, 6);
  t('4.4E+57', '3.99E+6199817', '3.9900000000000000000000000000000000000001e+6199817', 41, 2);
  t('6.407773391879524E-624218926406576', '5.864631545659357543848307123243933897691521705569062751724E-51503787491', '5.8646e-51503787491', 5, 1);
  t('-2.364965462745980617994777296333455262151884322083226414E+1213629023', '-3.79453064026732668425808079905E+2673033', '-2.365e+1213629023', 4, 4);
  t('-5.62345042766371782919812370405114004564152137E-141', '-1.8069303919949469931013818326394724624E-63', '-1.8069303919949469931013818326394724624e-63', 69, 6);
  t('7.19796265933963494574706299811E+83107792', '5.37310896774246898951619110285502982706E+68306410255', '5.3731089677424689895161911028550298270600000000000000000000001e+68306410255', 62, 2);
  t('6.60297728E+9963', '9.4213714886539761623977E+394199349', '9.421371488e+394199349', 10, 3);
  t('-8.47595362473254148933622419130878331368251361E+2400722106', '-9.52465269051174472096029260815165200339158227E+8041815834545', '-9.5246526905118e+8041815834545', 14, 3);
  t('4.4656488642942594963839E+373796430679938', '-1.056640657382110218E+3128091', '4.4656488642942594963839e+373796430679938', 63, 6);
  t('6.7297112E-73453', '5.0386353730752848497832812296405428989277853279570314153E-394223256223534', '6.72971120000000000000000000000000000000000000000000001e-73453', 54, 2);
  t('2.84839799575E+623615817625', '5.50889327941821228297829749394910904479858259360828E+7798477', '2.84839799575e+623615817625', 30, 5);
  t('2.07068453940E+4906058175651', '-4.508495233170590884E+8233', '2.07e+4906058175651', 3, 4);
  t('6.76396620108847357118847693333451331E+491605', '2.791082362780758603053178517336857867402793002307167849E+3596', '6.76396620108847357118847693334e+491605', 30, 2);
  t('-5.0645434564003444438468E+6344', '1.984E+5910981958', '1.98399999999999999999999e+5910981958', 24, 3);
  t('-1.651695591500904E+118915709719', '2.93926765163691371499680590478E+492564945217', '2.93926765163691371499680590478e+492564945217', 40, 4);
  t('7E-836803474', '4.791582739946E+366882796783', '4.791582739946e+366882796783', 60, 5);
  t('-4.72160666280E+23484590', '7.33136321653969173774205876750E+90080', '-4.7216066628e+23484590', 27, 4);
  t('5.57904842681446727324E-69190592808', '-17.918415343989609375257262175893344195530635918723369165', '-1.791841534398960937525726217589334419553064e+1', 43, 5);
  t('4.563266233542410668424258176720105070E+7046704441902', '-5.700E+82397532290', '4.56326623354241066842425817672010507e+7046704441902', 60, 6);
  t('7.698906499359180629654244E+34137', '1.20483887475046673151612268808663688597266E+8611324064', '1.204838874750466731516122688086636886e+8611324064', 37, 6);
  t('8.44580825136855356627150417912859140889386281884686612892931E+4770', '-6.744664458952416199794371843229337722429516E+4541615', '-6.7e+4541615', 2, 5);
  t('3.1569652506796065351705526396160648867426528048027430959087E+26547', '-4.989994E+61342', '-4.989994e+61342', 36, 5);
  t('9.62508871777897399988555008669904E-5128244', '-9.3321340E+2418196670', '-9.33213399999999999999999999999999999999999999999999999999999999999999999999e+2418196670', 75, 2);
  t('1.05049542597264733425411022852426765E+85258', '-5.0188301016972641281185242503012E+7140229282', '-5.0188301016972641281185242503012e+7140229282', 76, 5);
  t('-0.23155200325363154603083821560839407666', '-1.191983215485E+86367116961620', '-1.1919832154850000000000000000000000000000000000000000000000000000000000000001e+86367116961620', 77, 0);
  t('6.577296990559E-3569926217661', '1.89097325994409302057883856628223469536E+6101204', '1.890973259944093020578838566282234695360000000000000000000000000001e+6101204', 67, 2);
  t('7E+8031229', '4.05349021928512412448845923836308834593E+297765528415', '4.053490219285124124488e+297765528415', 22, 1);
  t('-6.20526072758893684201403135481280705447918196549080710989E+721750762', '6.468211363736011041528394819884E+7049631408', '6.468211363736011041528394819884e+7049631408', 70, 0);
  t('5.77864067417853822250311418757717518211106638198899844E+75405982757', '4.458E+9331399499959', '4.458e+9331399499959', 73, 4);
  t('-3.5890615418212225883486026221245704777866326281771080336057E+736', '-3.10899334878902412E-451', '-3.589061541821222588348602622124570477e+736', 37, 2);
  t('3.80504878191229324354011932160391143603E+2032224', '6.57240456626171385593681865143840308158898900925449E+4774791267621', '6.57240456626171385593681865143840308158898900925449e+4774791267621', 65, 4);
  t('9.6610172921069549979395883660358763791806249868652E+4869', '-4.044922249E+27423985', '-4.044922248e+27423985', 10, 2);
  t('-3.23441338692447615E+731667319946640', '1.6564004817431066315512571583E+8737112727', '-3.234413e+731667319946640', 7, 1);
  t('5.32E+687286785614', '3.84454572069275180741366546137713388314265322533246334048E+20663709', '5.32e+687286785614', 49, 4);
  t('8.8207743377180485184933796865273E+2818', '156708.35111610655960356251113', '8.8207743377180485185e+2818', 21, 2);
  t('8.07600E-3167114890609796', '4.49379785612193203546441944699154417651288200E-67617878473229', '4.493797856121932035464419446991544176512882e-67617878473229', 55, 1);
  t('-9.66660328901624012717348158441891393E-1614', '7.31E-629961403', '-9.6666032890162401271734815844189139e-1614', 35, 6);
  t('2.12254666498470963829759595300436242711E+966104378854970', '9.071274948263947819068114449605E+6819759', '2.122546664984709638297596e+966104378854970', 25, 2);
  t('8.496919048E-31325', '-2.566E+709415962476', '-2.566e+709415962476', 69, 4);
  t('8.306712361416869109630518877958672E+323737', '-7.9951255548188953895539E+783959319', '-7.9951255548188953895539e+783959319', 54, 3);
  t('-6.681886785165379127381E+9275', '1.0726545292422254866258191427482759408534602693264821059E+1989743', '1.072654529242225486625819142748275940853460269326482105899999999999e+1989743', 67, 3);
  t('-6.728802075E+876107948', '7.3512384720247429540931554335845784511696731313599649E+150584118891', '7.35123847202474295409315543358457845116967313136e+150584118891', 50, 6);
  t('-28.81037133826276750', '5.1936556447492191735350890130856026298360983970431E+2249', '5.19365564474921917353508901308560262983609839e+2249', 45, 3);
  t('7.10099910618603162602913895464216183877374670763177E-664701', '2.9497134176736888244719710189373062001792229363001946E-7040302304129787', '7.10099910618603162602913895464216183877374670763177e-664701', 69, 1);
  t('1.64175944015415812140933751787E+444', '1.2954473668084313244420E+223426141', '1.295447366808431324442e+223426141', 62, 5);
  t('6.581535322866298020044368782141255E+70', '-5.62744989463671461270153048386765E+284644000508', '-5.62744989e+284644000508', 9, 6);
  t('6.0512562041716870055332392106040106E+2357992', '63003817715628643691298785558004.2909447940061335543016897566', '6.0512562041716870055332392106040106e+2357992', 38, 3);
  t('1.06688046725342509E+2896110354804', '1.716332293516042191413285237950474221E+61590174884072', '1.716332294e+61590174884072', 10, 2);
  t('-1.512451018475257876277127821968523125838297056202426270213E+3144717232', '-18738205.4490511337042595567871167004228561726499011786', '-1.512451018475257876277127821968523125838297056202426270213e+3144717232', 60, 2);
  t('4886.4347871', '9.151247525058993769771469427723658911334614944817E+134177', '9.151247525059e+134177', 14, 4);
  t('-3.094340E+554283948', '5.0768121656851995197401968847961083039099467E+8108', '-3.09434e+554283948', 15, 3);
  t('5.6313414235294575732096707798E-31287', '8.8E+79835', '8.8e+79835', 31, 3);
  t('-5.3E+574998754356951', '-1.69638098336E-58755671610', '-5.3000000000000000000000000000000000000000000001e+574998754356951', 47, 0);
  t('-8.792120704243169221963458368538849883234893489905922752388E+587', '7.49572234668198069505142815588526899353E+567570708', '7.5e+567570708', 3, 4);
  t('-2.33920259118167E+50114', '6.78548455924243699957687591127497874706583278840772902107E+634851546479226', '6.7854845592424369995e+634851546479226', 20, 3);
  t('-3.646645421734242504698443629037473098514801281E+650392548649896', '9.96492526364441139607916306374819748246849068534981018489926E-62440', '-3.64664542173424250469844362903747309851480128099999999999e+650392548649896', 57, 2);
  t('9.5102722654268157800697653108392846930298E-47263', '-9.0610824786E+91453766140531', '-9.0610824786e+91453766140531', 78, 4);
  t('-5.831689E+777404', '7.622159514755800E-3478851665170356', '-5.831689e+777404', 70, 0);
  t('-39.80065889251010300744667', '1.514895557005345609827084129E+713371460278', '1.514896e+713371460278', 7, 4);
  t('-7.149679568403758452881575701400163658449105777641515E+127502583928', '3.822698618550147519252184348622514773822087862894E+4545853236', '-7.149679568403758452881575701400163658449105777641515e+127502583928', 72, 6);
  t('-3.745E+24544', '3.375497742775844691015092354361010E+120', '-3.745e+24544', 49, 5);
  t('-3.60348795228017780049569753533186294E+499', '8.80786821E+82223318', '8.80786821e+82223318', 28, 0);
  t('-9.8967145096522848735804634E+98677519', '-8.39861E+5784323495519916', '-8.39e+5784323495519916', 3, 1);
  t('-5.54480709948277E-718373854', '-7.119627881088288640389987E-896127305535', '-5.54480709948277e-718373854', 56, 6);
});
