# -*- coding: utf-8 -*-
#
# Configuration file for the Sphinx documentation builder.
#
# For the full list of built-in configuration values, see the documentation:
# https://www.sphinx-doc.org/en/master/usage/configuration.html

# -- Path setup --------------------------------------------------------------

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
#
import os
import sys
sys.path.insert(0, os.path.abspath('.'))
sys.path.append(os.path.abspath("./_ext"))

# -- Project information -----------------------------------------------------
# https://www.sphinx-doc.org/en/master/usage/configuration.html#project-information

project = 'FAQ LaTeX GUTenberg'
copyright = '2020-2024, Association GUTenberg'
author = 'Association GUTenberg'

# The full version, including alpha/beta/rc tags
release = '0.0.1'

# -- General configuration ---------------------------------------------------
# https://www.sphinx-doc.org/en/master/usage/configuration.html#general-configuration

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = [
    'sphinx_comments',
    'sphinx.ext.todo',
    'sphinx.ext.mathjax',
    'sphinx.ext.extlinks',
    'sphinx_design',
    'sphinxext.opengraph',
    'myst_parser',
    # 'sphinx_tippy',
    'sphinx_sitemap',
    'sphinx_copybutton',
    'sphinx_examples',
    'sphinx_togglebutton',
    'sphinx_last_updated_by_git',
]

suppress_warnings = ["myst.strikethrough"]

git_last_updated_timezone = 'Europe/Paris'

copybutton_selector = 'div.noedit > div.highlight > pre,div[class*="highlight"]:not([class*="acetex"]) > div.highlight > pre'

# Add any paths that contain templates here, relative to this directory.
templates_path = ['_templates']

# The suffix(es) of source filenames.

source_suffix = {
    '.rst': 'restructuredtext',
    '.md': 'markdown',
}

# The master toctree document.
master_doc = 'index'

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
language = 'fr'

locale_dirs = ['_static/locale']

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path.
exclude_patterns = ['build', 'Thumbs.db', '.DS_Store']

# html_extra_path pour ajouter des fichiers statics à la production
html_extra_path = ['../robots.txt','../BingSiteAuth.xml']

# The name of the Pygments (syntax highlighting) style to use.
# (voir plus bas après la config pydata)
# pygments_style = 'monokai'
# pygments_dark_style = "monokai"

highlight_language = 'acetex'

comments_config = {
    "hypothesis": True
}

# https://www.sphinx-doc.org/en/master/usage/configuration.html#confval-today_fmt
today_fmt='%A %d %B %Y'  # donne par exemple lundi 16 septembre 2024
# utilisé en tout cas pour la date au bas de la page de titre du PDF

# -- Options for HTML output -------------------------------------------------
# https://www.sphinx-doc.org/en/master/usage/configuration.html#options-for-html-output

# html_file_suffix = ""
#
# # Suffix for generated links to HTML files. The default is whatever
# # html_file_suffix is set to; it can be set differently (e.g. to support
# # different web server setups).
# html_link_suffix = ""

linkcheck_timeout = 15

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
#
# html_theme = "sphinx_book_theme"
html_theme = "pydata_sphinx_theme"

# Theme options are theme-specific and customize the look and feel of a theme
# further.  For a list of options available for each theme, see the
# documentation.

# Options for pydata_sphinx_theme options -------------------------------------------------
from sphinx.locale import get_translation
catalog = "theme"
_ = get_translation(catalog)

html_theme_options = {
    "search_bar_text": _("Recherche dans la FAQ"),
    "use_edit_page_button": True,
    "header_links_before_dropdown": 8,
    "content_footer_items": ["last-updated"],
    "gitlab_url": "https://gitlab.gutenberg-asso.fr/gutenberg/faq-gut",
    "icon_links": [
        {
            "name": "GUTenberg",
            "url": "https://www.gutenberg-asso.fr/",
            "icon": "_static/images/svg/gutlogo-mini.svg",
            "type": "local",
            "attributes": {"target": "_blank"},
        },
    ],
    # "announcement": "Site <em>expérimental</em>, encore en construction !",
    #"announcement": "La FAQ LaTeX GUTenberg, c'est la FAQ de la communauté francophone ! Mises à jour quotidiennes !",
    "pygment_light_style": "dawn",  # horriblement inadapté à LaTeX (array!), BibTeX, ...
    "pygment_dark_style": "acedracula",
    "logo": {
        "image_light": "_static/images/svg/logoFAQ-light-theme.svg",
        "image_dark":  "_static/images/svg/logoFAQ-dark-theme.svg",
    },
    "navbar_persistent": ["search-button"],
    "navbar_align": "left",
}

# ces variables sont ignorées par pydata_sphinx_theme mais peuvent servir
# à d'autres builders que html
pygments_style = "autumn"
# Memo: ici on pourrait mettre acedracula.AceDracula, à partir du moment
# ou un module acedracula existe que Python peut voir, mais pydata lui
# ne reconnaît pas cette syntaxe, cf
#    https://github.com/pydata/pydata-sphinx-theme/issues/1593
# (consulté le 3 juin 2024).
# Pour que la syntaxe ci-dessous fonctionne il faut avoir ajouté
# acedracula aux styles de Pygments voir
#    https://pygments.org/docs/styledevelopment/
# ou que acedracula soit connu de Pygments par un mécanisme
# de "plug-in" cf https://github.com/pygments/pygments-plugin-scaffolding
pygments_dark_style = "acedracula"

html_context = {
    "gitlab_url": "https://gitlab.gutenberg-asso.fr/-/ide/project/", # or your self-hosted GitLab
    "gitlab_user": "gutenberg/",
    "gitlab_repo": "faq-gut/",
    "gitlab_version": "main/",
    "gitlab_edit": "edit/",
    "doc_path": "-/source/",
    "edit_page_url_template": "{{ gitlab_url }}{{ gitlab_user }}{{ gitlab_repo }}{{ gitlab_edit }}{{ gitlab_version }}{{ doc_path }}{{ file_name }}",
}

html_title = 'FAQ LaTeX GUTenberg'

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = ['_static']

# These paths are either relative to html_static_path
# or fully qualified paths (eg. https://...)
html_css_files = [
    'css/custom.css',
    # 'css/tippy.css',
]
html_js_files = [
    'js/runlatex.js',
    'js/translations.js',
    'js/preloadPDF.js',
    'https://cdnjs.cloudflare.com/ajax/libs/ace/1.35.2/ace.min.js'
]

# Custom sidebar templates, must be a dictionary that maps document names
# to template names.
#
# The default sidebars (for documents that don't match any pattern) are
# defined by theme itself.  Builtin themes are using these templates by
# default: ``['localtoc.html', 'relations.html', 'sourcelink.html',
# 'searchbox.html']``.
#
# html_sidebars = {}

html_favicon = '_static/images/favicon.svg'

# -- Options for HTMLHelp output ---------------------------------------------

# Output file base name for HTML help builder.
htmlhelp_basename = 'FAQLaTeXGUTenbergdoc'


# -- Options for LaTeX output ------------------------------------------------

latex_engine = 'lualatex'

# Configuration de latex_elements['sphinxsetup'] pour
# le style des code-blocks et des admonitions
#
# MEMO: il n'est actuellement pas possible pour le code
# dans _ext/_GUT_docs_to_PDF.py de mettre de l'information
# dans le .tex avant les sphinxVerbatim pour les informer
# qu'ils sont suivis, ou non, par l'inclusion de leur
# output.  On utilise ici une configuration qui donne au
# cadre pour les blocs de codes des coins carrés en bas,
# pour faire la jonction avec le cadre pour l'output de
# ce bloc de code (qui est un PDF inclus).
#
# Pour un peu de variété pour les admonitions, on s'est inspiré
# de la configuration indiquée dans la documentation de Sphinx
# en PDF, chapitre sur LaTeX.
# https://www.sphinx-doc.org/en/master/latex.html#the-sphinxsetup-configuration-setting
# Pour la FAQ les types les plus fréquents sont d'abord en premier
# lieu et de loin note, puis en ordre de fréquence décroissante
# warning, tip, important, attention, danger.
#
# À faire (ou pas !): donner des noms à toutes les couleurs utilisées ?
# Problème: il faut injecter avant \usepackage{sphinx}, donc pas
# par 'preamble'.  On pourrait utiliser 'passoptionstopackages'
# sauf que \definecolor ou \colorlet pas encore disponibles...
# car sphinx qui fait \RequirePackage{xcolor}. Il faudra donc
# mettre \usepackage{xcolor} dans le  'passoptionstopackages'
# plus haut *après* le \PassOptions... dont on ne peut pas faire
# l'économie et alors on peut définir des couleurs dans cette clé.
# Bof, bien compliqué tout cela...
#
# pre_background-TeXcolor est un alias de VerbatimColor, et sa
# valeur a été récupérée sur le rendu HTML de pydata_sphinx_theme
# en mode clair
FAQ_LATEXSPHINXSETUP = """
    pre_border-TeXcolor=cyan,
    pre_background-TeXcolor={RGB}{249,249,249},% aka VerbatimColor
    pre_border-radius=0pt,
    pre_border-width=1pt,
    pre_padding=3pt,
    hmargin={3.5cm,3cm},
    vmargin=3cm,
    marginpar=2.5cm,
"""

latex_elements = {
    # The paper size ('letterpaper' or 'a4paper').
    #
    'papersize': 'a4paper',

    # The font size ('10pt', '11pt' or '12pt').
    #
    'pointsize': '11pt',

    'passoptionstopackages': ('\\PassOptionsToPackage{svgnames}{xcolor}\n'
                              '\\PassOptionsToPackage{marginparsep=12pt}'
                              '{geometry}'),

    # Utilisation de babel-french
    # cf https://github.com/sphinx-doc/sphinx/issues/12410
    'polyglossia':  '',
    'babel':        r'\usepackage{babel}\frenchsetup{og=«,fg=»}',

    # font packages,
    'fontpkg': r'\usepackage{FAQfonts}',

    # Avec Inconsolatazi4 et Scale=MatchLowercase en relation avec XCharter,
    # et hmargin={3.5cm,3cm} on observe environ de l'ordre de 70 caractères par
    # ligne.  C'est un peu insuffisant pour du code source.  Donc \small.
    'fvset': r'\fvset{fontsize=\small}',

    # Additional stuff for the LaTeX preamble.
    #
    # 'preamble': '',
    'preamble': r'\usepackage{FAQpreamble}',

    # Latex figure (float) alignment
    #
    # 'figure_align': 'htbp'

    'sphinxsetup': FAQ_LATEXSPHINXSETUP,

    # On supprime tout effet dans PDF (voir preamble.tex)
    # car utilisée dans les sources uniquement via directive sources (sic).
    # Mais pour que les demos de Markdown fonctionnent on met ici une macro
    # customisable.
    'transition': "\n\n\\faqGUTtransition\n\n",
}

# Grouping the document tree into LaTeX files. List of tuples
# (source start file, target name, title,
#  author, documentclass [howto, manual, or own class]).
latex_documents = [
    (master_doc, 'FAQLaTeXGUTenberg.tex', 'FAQ LaTeX GUTenberg',
     'Association GUTenberg', 'manual'),
]

latex_toplevel_sectioning = 'part'
latex_logo = '_static/logoFAQ-light-theme.png'
latex_additional_files = ['latex_extras/Makefile',
                          'latex_extras/FAQpreamble.sty',
                          'latex_extras/FAQfonts.sty',
                          'latex_extras/FAQoutputbox.sty',
                          'latex_extras/FAQsourcesbox.sty',
                          'latex_extras/FAQtochack.sty',
                          'latex_extras/FAQtitleshack.sty',
                          'latex_extras/hologo-suppl.sty',
                          'latex_extras/tex-logos-in-tex_gutfr.bin',
                          'latex_extras/fais-les-pdf-et-crop-les.sh',
                          'latex_extras/latexmkrc_pour_codedir',
                          ]


# -- Options for manual page output ------------------------------------------

# One entry per manual page. List of tuples
# (source start file, name, description, authors, manual section).
man_pages = [
    (master_doc, 'faqlatexgutenberg', 'FAQ LaTeX GUTenberg Documentation',
     [author], 1)
]


# -- Options for Texinfo output ----------------------------------------------

# Grouping the document tree into Texinfo files. List of tuples
# (source start file, target name, title, author,
#  dir menu entry, description, category)
texinfo_documents = [
    (master_doc, 'FAQLaTeXGUTenberg', 'FAQ LaTeX GUTenberg Documentation',
     author, 'FAQLaTeXGUTenberg', 'One line description of project.',
     'Miscellaneous'),
]


# -- Options for Epub output -------------------------------------------------

# Bibliographic Dublin Core info.
epub_title = project

# The unique identifier of the text. This can be a ISBN number
# or the project homepage.
#
# epub_identifier = ''

# A unique identification for the text.
#
# epub_uid = ''

# A list of files that should not be packed into the epub file.
epub_exclude_files = ['search.html']

# html_show_search_summary = True

# -- Extension configuration -------------------------------------------------

# -- Options for intersphinx extension ---------------------------------------

# Example configuration for intersphinx: refer to the Python standard library.
intersphinx_mapping = {'https://docs.python.org/': None}

togglebutton_hint = "Cliquer pour déplier"

# -- Options for todo extension ----------------------------------------------

# If true, `todo` and `todoList` produce output, else they produce nothing.
todo_include_todos = True
todo_link_only = True

html_last_updated_fmt = '%d %b %Y à %H:%M'

# extlinks
extlinks = {
    # 'ctanpkg': ('https://ctan.org/pkg/%s','%s'), # :ctanpkg:`memoir` (par exemple)
    # 'texdoc': ('https://texdoc.net/pkg/%s','%s'), # :texdoc:`bib2gls` (par exemple)
    # 'faquk': ('https://texfaq.org/%s','%s'), # :faquk:`FAQ-LaTeX3` (par exemple)
    # 'isbn': ('https://isbndb.com/book/%s','%s'), # :isbn:`978-2744071331` (par exemple)
    # 'doi': ('https://doi.org/%s','%s'), # :doi:`10.1145/309739.309744` (par exemple)
}

# myst_all_links_external = True

myst_url_schemes = {
    "doi": "https://doi.org/{{path}}", # <doi:10.1145/309739.309744> (par exemple)
    "http": None,
    "https": None,
    "ftp": None,
    "mailto": None,
    "ctanpkg": {
        "url": "https://ctan.org/pkg/{{path}}",
        "title": "{{path}}",
        "classes": ["ctanpkg"],
    },
    "texdoc": {
        "url": "https://texdoc.org/serve/{{path}}/0",
        "title": "{{path}}",
        "classes": ["texdoc"],
    },
    "faquk": {
        "url": "https://texfaq.org/{{path}}",
        "title": "{{path}}",
        "classes": ["faquk"],
    },
    "wpfr": {
        "url": "https://fr.wikipedia.org/wiki/{{path}}#{{fragment}}",
        "title": "{{path}}",
        "classes": ["wp"],
    },
    "wp": {
        "url": "https://en.wikipedia.org/wiki/{{path}}#{{fragment}}",
        "title": "{{path}}",
        "classes": ["wp"],
    },
    "isbn": {
        "url": "https://isbndb.com/book/{{path}}",
        "title": "{{path}}",
        "classes": ["isbn"],
    },
}

myst_heading_anchors = 6

# metatags (cf. https://pypi.org/project/sphinxext-opengraph/)
ogp_site_name = "FAQ LaTeX francophone"
ogp_site_url = "https://faq.gutenberg-asso.fr/start"
ogp_image = "https://faq.gutenberg-asso.fr/_images/logoFAQ-light-theme.png"
ogp_description_length = 300
ogp_type = "website"

myst_enable_extensions = [
    "amsmath",
    "attrs_inline",
    "attrs_block",
    "colon_fence",
    "deflist",
    "dollarmath",
    "fieldlist",
    "html_admonition",
    "html_image",
    "linkify",
    "replacements",
    # "smartquotes",
    "strikethrough",
    "substitution",
    "tasklist",
]

html_baseurl = 'https://faq.gutenberg-asso.fr/'
sitemap_url_scheme = "{link}"

# -- Local Sphinx extensions -------------------------------------------------


def setup(app: Sphinx):
    """Add functions to the Sphinx setup."""
    from myst_parser._docs import (
        DirectiveDoc,
        DocutilsCliHelpDirective,
        MystAdmonitionDirective,
        MystConfigDirective,
        MystExampleDirective,
        MystLexer,
        MystToHTMLDirective,
        MystWarningsDirective,
        NumberSections,
        StripUnsupportedLatex,
    )

    from _GUT_docs import (
        GutAdmonitionDirective,
        GutSourcesDirective,
        GutLaTeXNoEditDirective,
    )

    from _GUT_docs_to_PDF import FAQLaTeXGUTranslator

    app.add_directive("myst-config", MystConfigDirective)
    app.add_directive("docutils-cli-help", DocutilsCliHelpDirective)
    app.add_directive("doc-directive", DirectiveDoc)
    app.add_directive("myst-warnings", MystWarningsDirective)
    app.add_directive("myst-example", MystExampleDirective)
    app.add_directive("myst-admonitions", MystAdmonitionDirective)
    app.add_directive("myst-to-html", MystToHTMLDirective)
    app.add_post_transform(StripUnsupportedLatex)
    app.add_post_transform(NumberSections)
    app.add_lexer("myst", MystLexer)

    app.add_directive("gut-admonitions", GutAdmonitionDirective)
    app.add_directive("sources", GutSourcesDirective)
    app.add_directive("noedit", GutLaTeXNoEditDirective)

    app.set_translator('latex', FAQLaTeXGUTranslator)
